/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE skrooge@miraks.com    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
#ifndef SKGTREEVIEW_H
#define SKGTREEVIEW_H
/** @file
 * A tree view for skrooge.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgservices.h"
#include "skgobjectbase.h"

#include <kglobal.h>
#include <QStringList>
#include <QTreeView>
#include <QtDesigner/QDesignerExportWidget>
#include <QBasicTimer>

#include <kmenu.h>

class SKGDocument;

/**
 * This file is Skrooge tab widget used by plugins
 */
class KDE_EXPORT SKGTreeView : public QTreeView
{
        Q_OBJECT
        Q_PROPERTY( QString state READ getState WRITE setState USER true)
        Q_PROPERTY( bool autoResized READ isAutoResized)
        Q_PROPERTY( bool smoothScrolling READ isSmoothScrolling)
public:
        /**
         * Default Constructor
         * @param parent the parent
         */
        SKGTreeView(QWidget *parent = 0);

        /**
         * Default Destructor
         */
        virtual ~SKGTreeView();

        /**
         * Get the current state
         * @return a string containing all information needed to set the same state.
         * Could be an XML stream
         */
        virtual QString getState();

        /**
         * Set the current state
         * MUST BE OVERWRITTEN
         * @param iState must be interpreted to set the state of the widget
         */
        virtual void setState(const QString& iState);

        /**
         * To know if the autoresized mode is enable
         */
        virtual bool isAutoResized();

        /**
         * To know if the smoothscrolling mode is enable
         */
        virtual bool isSmoothScrolling();

        /**
        * Set parameter to activate and save default state of this table
        * @param iDocument document pointer
        * @param iParameterName parameter name in this document
        */
        virtual void setDefaultSaveParameters(const SKGDocument* iDocument, const QString& iParameterName);

        /**
         * Get the table content
         * @return the table content
         */
        virtual SKGStringListList getTable();

        /**
         * Get the current selection
         * @return selected objects
         */
        virtual SKGObjectBase::SKGListSKGObjectBase getSelectedObjects();

        /**
         * Get the number of seleted object
         * @return number of selected objects
         */
        virtual int getNbSelectedObjects();

protected:

        /**
          * This function is called with the given event when a mouse button is pressed while the cursor is inside the widget.
         * If a valid item is pressed on it is made into the current item. This function emits the pressed() signal.
         * @param event the event
         */
        virtual void mousePressEvent ( QMouseEvent * event );

public slots:
        /**
         * Save the selection
         */
        virtual void saveSelection();

        /**
         * Reset the selection
         */
        virtual void resetSelection();

        /**
         * This property holds whether to draw the background using alternating colors.
         * @param enable true of false
         */
        virtual void setAlternatingRowColors(bool enable);

        /**
         * Resizes all columns based on the size hints of the delegate used to render each item in the columns.
         */
        virtual void resizeColumnsToContents();

        /**
         * Swith the smooth Scrolling mode
         */
        virtual void switchSmoothScrolling();

private slots:
        void onExportPDF();
        void onExportCSV();
        void onExportTXT();

        void moveSection();
        void setupHeaderMenu(bool iRefreshColumnSize=true);
        void showHeaderMenu(const QPoint& pos);
        void showHideColumn();
        void switchAutoResize();
        void changeSchema();
        void saveDefaultClicked();

private:
        Q_DISABLE_COPY(SKGTreeView);

        void timerEvent(QTimerEvent *event);
        bool eventFilter(QObject *object, QEvent *event);

        KMenu* header_menu;
        bool autoResize;
        QAction* actAutoResize;
        bool smoothScrolling;
        QAction* actSmoothScrolling;
        SKGDocument* document;
        QString parameterName;
        QStringList selection;


        typedef enum { Steady, Pressed, ManualScroll, AutoScroll, Stop } State;
        State state;
        QPoint pressPos;
        QPoint offset;
        QPoint dragPos;
        QPoint speed;
        QList<QEvent*> ignored;
        QBasicTimer ticker;
};

#endif // SKGTREEVIEW_H
