/* $Id: dhcp.c,v 1.5 2003/05/30 00:42:39 cgd Exp $ */

/*
 * Copyright 2001, 2003
 * Broadcom Corporation. All rights reserved.
 *
 * This software is furnished under license and may be used and copied only
 * in accordance with the following terms and conditions.  Subject to these
 * conditions, you may download, copy, install, use, modify and distribute
 * modified or unmodified copies of this software in source and/or binary
 * form. No title or ownership is transferred hereby.
 *
 * 1) Any source code used, modified or distributed must reproduce and
 *    retain this copyright notice and list of conditions as they appear in
 *    the source file.
 *
 * 2) No right is granted to use any trade name, trademark, or logo of
 *    Broadcom Corporation.  The "Broadcom Corporation" name may not be
 *    used to endorse or promote products derived from this software
 *    without the prior written permission of Broadcom Corporation.
 *
 * 3) THIS SOFTWARE IS PROVIDED "AS-IS" AND ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING BUT NOT LIMITED TO, ANY IMPLIED WARRANTIES OF
 *    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 *    NON-INFRINGEMENT ARE DISCLAIMED. IN NO EVENT SHALL BROADCOM BE LIABLE
 *    FOR ANY DAMAGES WHATSOEVER, AND IN PARTICULAR, BROADCOM SHALL NOT BE
 *    LIABLE FOR DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 *    OR OTHERWISE), EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "port.h"
#include "dhcp.h"
#include "global.h"
#include "udp.h"
#include "eth.h"
#include "ip.h"
#include "misc.h"

#define DHCP_OP_REQUEST    1
#define DHCP_OP_REPLY      2


#define DHCP_HW_ETH        1

#define DHCP_HW_LENGTH_ETH 6


#define DHCP_OPT_PAD         0
#define DHCP_OPT_SUBNET      1
#define DHCP_OPT_NAMESERVERS 6
#define DHCP_OPT_MESSAGE    53
#define DHCP_OPT_SERVER     54
#define DHCP_OPT_PARAMS     55
#define DHCP_OPT_END       255

#define DHCP_MAGIC   0x63825363

#define DHCP_OPT_MSG_DISCOVER 1
#define DHCP_OPT_MSG_OFFER    2


/* 
 * Data Structures
 */

typedef struct {
	uint8_t opcode             ;
	uint8_t hw_type            ;
	uint8_t hw_addr_length     ;
	uint8_t hop_count          ;
	uint8_t trans_id[4]        ;
	uint8_t seconds[2]         ;
	uint8_t flags[2]           ;
	uint8_t client_ip[4]       ;
	uint8_t your_ip[4]         ;
	uint8_t server_ip[4]       ;
	uint8_t gateway_ip[4]      ;
	uint8_t client_hw_addr[16] ;
	uint8_t server_name[64]    ;
	uint8_t boot_filename[128] ;
	uint8_t vendor_specific[64];
} dhcp_packet_t;

typedef struct {
	uint32_t subnet_mask;
	uint32_t server;
	uint32_t nameservers[DHCP_MAX_NAMESERVERS];
	uint8_t message;
} dhcp_vendor_options_t;

typedef struct {
	uint32_t trans_id;
	dhcp_info_t *info;
	unsigned int *flag;
} dhcp_packet_handler_data_t;



/*
 * Static functions
 */


/*
 * Parse out the vendor options field of the DHCP packet.  This
 * is where the subnet mask, nameservers, and several other
 * tidbits of information lie
 */
static void parse_vendor_options(dhcp_packet_t *packet, dhcp_vendor_options_t *options) 
{
	uint8_t *vendor_ptr, *end_ptr;
	vendor_ptr = packet->vendor_specific + 4;
	end_ptr = packet->vendor_specific + 64;
	lib_bzero(options, sizeof(*options));
	while (vendor_ptr < (end_ptr + 1)) {
		int i, tmp;
		/* These are defined in RFCs 1533 & 2132 */
		/* Going into this switch, we only know that vendor_ptr and vendor_ptr + 1
		   point to valid locations.  For accesses beyond that, we'll have to
		   do bounds checking. */
		switch(*vendor_ptr) {
		case DHCP_OPT_PAD:
			vendor_ptr++;
			break;
		case DHCP_OPT_SUBNET:
			if (vendor_ptr[1] != 4) {
				break;
			}
			vendor_ptr += 2;
			if ((vendor_ptr + 4) < end_ptr) {
				options->subnet_mask = get_uint32_field(vendor_ptr);
			}
			vendor_ptr += 4;
			break;
		case DHCP_OPT_MESSAGE:
			if (vendor_ptr[1] != 1) {
				break;
			}
			vendor_ptr += 2;
			if (vendor_ptr < end_ptr) {
				options->message = *vendor_ptr;
			}
			vendor_ptr++;
			break;
		case DHCP_OPT_NAMESERVERS:
			if (vendor_ptr[1] & 0x3) {
				break;
			}
			tmp = vendor_ptr[1];
			vendor_ptr += 2;
			if ((vendor_ptr + tmp) < end_ptr) {
				for (i = 0; (i<<2) < tmp; i ++) {
					if (i < DHCP_MAX_NAMESERVERS) {
						options->nameservers[i] = get_uint32_field(vendor_ptr + (i<<2));
					} else {
						break;
					}
				}
			}
			vendor_ptr += tmp;
			break;
		case DHCP_OPT_SERVER:
			if (vendor_ptr[1] != 4) {
				break;
			}
			vendor_ptr += 2;
			if ((vendor_ptr + 4) < end_ptr) {
				options->server = get_uint32_field(vendor_ptr);
			}
			vendor_ptr += 4;
			break;
		case DHCP_OPT_END:
			vendor_ptr = end_ptr;
			break;
		default:
			/* Unhandled option.  Use the length field to skip it */
			vendor_ptr += 2 + vendor_ptr[1];
			break;
		}
	}
}


/*
 * Handler for dhcp offers.  Check if this is an offer with
 * the proper tid.  If it is, grab the info from it.
 */
static int dhcp_packet_handler(mbuf_t *mbuf, void *ptr, uint32_t ignore1, uint32_t ignore2, uint16_t ignore3, uint16_t ignore4)
{ 
	dhcp_packet_t *packet;
	dhcp_packet_handler_data_t *data = (dhcp_packet_handler_data_t *)ptr;
	
	if (mbuf->size < sizeof(dhcp_packet_t)) {
		return 0;
	}

	packet = (dhcp_packet_t *)mbuf->front_ptr;
	if ((packet->opcode == DHCP_OP_REPLY)
	    && (packet->hw_type == DHCP_HW_ETH)
	    && (packet->hw_addr_length == DHCP_HW_LENGTH_ETH)
	    && (get_uint32_field(packet->vendor_specific) == DHCP_MAGIC)
	    && (data->trans_id == get_uint32_field(packet->trans_id))) {
		int i;

		dhcp_vendor_options_t options;
		
		parse_vendor_options(packet, &options);
		
		if (options.message != DHCP_OPT_MSG_OFFER) {
			return 0;
		}

		/* It's an offer.  Cool */

		(*(data->flag)) = 1;
		
		data->info->client_ip = get_uint32_field(packet->your_ip);
		lib_strncpy(data->info->server_name, packet->server_name, 64);
		data->info->server_name[63] = 0;
		lib_strncpy(data->info->boot_filename, packet->boot_filename, 128);
		data->info->boot_filename[127] = 0;
		data->info->gateway = get_uint32_field(packet->gateway_ip);
		data->info->subnet_mask = options.subnet_mask;
		data->info->server = options.server;
		for (i = 0; i < DHCP_MAX_NAMESERVERS; i++) {
			data->info->nameservers[i] = options.nameservers[i];
		}
	}
	return 1;
}

/*
 * Exported interface.  These are described in the headers
 */

int dhcp_get_boot_info(dhcp_info_t *info)
{
	uint8_t *dhcp_fields;
	dhcp_packet_t *packet;
	unsigned int done_flag = 0;
	dhcp_packet_handler_data_t data;
	mbuf_t *mbuf;
	int retval = -1;


	mbuf = lib_malloc(sizeof(mbuf_t));
	
	if (!mbuf) {
		lib_die("Out of memory");
	}

	data.flag = &done_flag;
	data.trans_id = 0x31415926;
	data.info = info;
	udp_add_handler(dhcp_packet_handler, &data, 0, 0, 67, 68);

	mbuf_init(mbuf, 0);
	mbuf_append_space(mbuf, sizeof(dhcp_packet_t));
	packet = (dhcp_packet_t *)(mbuf->front_ptr);
	
	lib_bzero(packet, sizeof(dhcp_packet_t));

	packet->opcode         = DHCP_OP_REQUEST;
	packet->hw_type        = DHCP_HW_ETH;
	packet->hw_addr_length = DHCP_HW_LENGTH_ETH;
	packet->hop_count      = 0;

	set_uint32_field(packet->trans_id, 0x31415926); /* FIXME - This is supposed to be pseudorandom */
	set_uint48_field(packet->client_hw_addr, net_cfg.mac_addr);
	dhcp_fields = packet->vendor_specific;
	set_uint32_field(dhcp_fields, DHCP_MAGIC);
	dhcp_fields += 4;
	(*dhcp_fields) = DHCP_OPT_MESSAGE;
	dhcp_fields++;
	(*dhcp_fields) = 1;  /* Length of request */
	dhcp_fields++;
	(*dhcp_fields) = DHCP_OPT_MSG_DISCOVER;
	dhcp_fields++;
	(*dhcp_fields) = DHCP_OPT_PARAMS;
	dhcp_fields++;
	(*dhcp_fields) = 4;
	dhcp_fields++;
	(*dhcp_fields) = DHCP_OPT_SUBNET;
	dhcp_fields++;
	(*dhcp_fields) = DHCP_OPT_NAMESERVERS;
	dhcp_fields++;
	(*dhcp_fields) = DHCP_OPT_MESSAGE;
	dhcp_fields++;
	(*dhcp_fields) = DHCP_OPT_SERVER;
	dhcp_fields++;
	(*dhcp_fields) = DHCP_OPT_END;
	udp_send_packet(mbuf, 0, 0xffffffff, IP_FLAGS_NO_FRAGMENT, 68, 67);
	eth_recv_loop(500000, &done_flag);
	udp_del_handler(dhcp_packet_handler, &data, 0, 0, 67, 68);
	if (done_flag) {
		retval = 0;
	}
	lib_free(mbuf);
	return retval;
}
