/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/
/** $Revision: 1.01 $
 * Libdvd common header
 * @file dvd-types.h global types
 * @ingroup core
 *
 * libdvd
 */

#if !defined (__DVD_H_INSIDE__) && !defined (LIBDVD_COMPILATION)
#error "Only <dvd.h> can be included directly."
#endif

#ifndef __DVD_TYPES_H__
#define __DVD_TYPES_H__


G_BEGIN_DECLS


/**
 * @ingroup core
 * Common Types used in libdvd
 */

/** GQuark error domain for libdvd */
GQuark	 dvd_error_quark (void);

/**
 * Error domain for libdvd that include errors from the
 * #DvdError enumeration. Some functions may also
 * return (glib) G_FILE_ERROR domain errors
 * which will be errors from the GFileError enumeration.
*/
#define DVD_ERROR dvd_error_quark ()

/** DVD GError errors */
typedef enum _DvdError {
	/* general errors */
	DVD_ERROR_CANCELLED,		/**< Operation was cancelled by calling stop function */
	
	/* dvd errors */
	DVD_ERROR_INVALID_ANGLE,	/**< Specified angle is invalid */
	DVD_ERROR_INVALID_CHAPTER,	/**< Specified chapter is invalid */
	DVD_ERROR_INVALID_TITLE,	/**< Specified title is invalid */
	
	/* disk errors */
	DVD_ERROR_DISK_OPEN,		/**< dvdread ifoOpen or DVDOpenFile failed */
	DVD_ERROR_DISK_READ,		/**< dvdread block read operation failed */
	
	/* directory errors */
	DVD_ERROR_INVALID_DIR		/**< Directory is a regular file or does not exist */
} DvdError;

/**
 * Time representation with broken down time
 *
 * DVD times are specified h:m:s and leftover frames.
 * MPEG streams express time as a time stamp in a value of 1/90000 of a second.
 * There are 30000/1001 (close to 29.97) NTSC frames per second and
 * 25 PAL frames per second which is
 * 3003 MPEG clock cycles for each NTSC frame and
 * 3600 MPEG clock cycles for each PAL frame.
 *
 * Humanly readable (broken down time) must not be used
 * for calculations because bk_milliseconds has no
 * fraction and may be rounded down for NTSC video.
 */

typedef struct _DvdTime {
	guint32 ts;		/**< total mpeg time stamp (90kHz) */
	guint8	bk_hours;	/**< full hours */
	guint8	bk_minutes;	/**< full minutes */
	guint8	bk_seconds;	/**< full seconds */
	guint16	bk_milliseconds;/**< full milliseconds only */
} DvdTime;

/** Number of MPEG clock cyces per PAL frame */
#define	PAL_CLOCK_CYCLES_PER_FRAME  3600

/** Number of MPEG clock cyces per NTSC frame */
#define NTSC_CLOCK_CYCLES_PER_FRAME 3003

/** Source of dvd data */
typedef enum _DvdSource {
	DVD_SOURCE_DISK,	/**< The data source is a optical disk drive */
	DVD_SOURCE_FILE		/**< The data source is on local hard disk */
} DvdSource;

/** Dvd output type from object signal */
typedef enum _DvdStream {
	DVD_STREAM_VOB,		/**< Full vob file */
	DVD_STREAM_MPEG1_VIDEO,	/**< MPEG 1 video stream - Not tested at all */
	DVD_STREAM_MPEG2_VIDEO,	/**< MPEG 2 video stream */
	DVD_STREAM_MPEG1_AUDIO,	/**< MPEG 1 audio stream - Not tested at all */
	DVD_STREAM_MPEG2_AUDIO,	/**< MPEG 2 audio stream - Not tested at all */
	/* rest are mpeg private stream */
	DVD_STREAM_AC3_AUDIO,	/**< AC3 audio stream */
	DVD_STREAM_LPCM_AUDIO,	/**< LPCM audio stream - Not tested at all */
	DVD_STREAM_SDDS_AUDIO,	/**< Currently unused in DVD's - Not implimented */
	DVD_STREAM_DTS_AUDIO,	/**< DTS audio stream */
	DVD_STREAM_SUB_PICTURE	/**< SUB Picture stream - Not implimented */
} DvdStream;

/** Dvd cell definition specifies a cell on an optical drive */
typedef struct _DvdCell {
	guint	 cell_id;	  /**< Cell ID starting from 0 */
	guint	 first_cell_block;/**< First cell block on optical disk */
	guint	 last_cell_block; /**< Last cell block on optical disk */
	guint	 blocks;	  /**< Number of blocks on optical disk */
	DvdTime	*time;		  /**< Full cell time @see _DvdTime */
} DvdCell;


/** Dvd media types */
#define DVD_MEDIA_SL_BLOCKS	2295104
#define DVD_MEDIA_DL_BLOCKS	4590208

typedef enum _DvdMedia {
	DVD_MEDIA_SL = 0,	/**< Single layer DVD disk (4.7Gb) */
	DVD_MEDIA_DL = 1	/**< Duel layer DVD disk (8.5Gb) */
} DvdMedia;
G_END_DECLS

#endif /*__DVD_TYPES_H__*/
