//----------------------------------------------------------------------------
//
//  This file is part of seq24.
//
//  seq24 is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  seq24 is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with seq24; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//-----------------------------------------------------------------------------
#include "seqkeys.h"


seqkeys::seqkeys(sequence *a_seq): DrawingArea() 
{     
    m_seq = a_seq;

    add_events( GDK_BUTTON_PRESS_MASK | 
		GDK_BUTTON_RELEASE_MASK |
		GDK_LEAVE_NOTIFY_MASK |
		GDK_BUTTON_MOTION_MASK );

    /* set default size */
    size( c_keyarea_x, c_keyarea_y );	

    // in the construor you can only allocate colors, 
    // get_window() returns 0 because we have not be realized
    Gdk_Colormap colormap = get_default_colormap();

    m_black = Gdk_Color( "black" );
    m_white = Gdk_Color( "white" );
    m_grey = Gdk_Color( "grey" );
        
    colormap.alloc( m_black );
    colormap.alloc( m_white );
    colormap.alloc( m_grey );
    
    m_keying = false;
    m_hint_state = false;
}

void 
seqkeys::realize_impl()
{
    // we need to do the default realize
    Gtk::DrawingArea::realize_impl();

    // Now we can allocate any additional resources we need
    m_window = get_window();
    m_gc.create( m_window );
    m_window.clear();

    m_pixmap = Gdk_Pixmap(m_window,
			  c_keyarea_x,
			  c_keyarea_y);
  
    update_pixmap();
}

void 
seqkeys::update_pixmap()
{
    m_gc.set_foreground(m_black);
    m_pixmap.draw_rectangle(m_gc,true,
			    0,
			    0, 
			    c_keyarea_x, 
			    c_keyarea_y  );

    m_gc.set_foreground(m_white);
    m_pixmap.draw_rectangle(m_gc,true,
			    1,
			    1, 
			    c_keyoffset_x - 1, 
			    c_keyarea_y - 2  );

  
    for ( int i=0; i<c_num_keys; i++ )
		{
			m_gc.set_foreground(m_white);
			m_pixmap.draw_rectangle(m_gc,true,
									c_keyoffset_x + 1,
									(c_key_y * i) + 1, 
									c_key_x - 2, 
									c_key_y - 1 );
			
			/* the the key in the octave */
			int key = (c_num_keys - i - 1) % 12;
			
			if ( key == 1 || 
				 key == 3 || 
				 key == 6 || 
				 key == 8 || 
				 key == 10 ){
				
				m_gc.set_foreground(m_black);
				m_pixmap.draw_rectangle(m_gc,true,
										c_keyoffset_x + 1,
										(c_key_y * i) + 2, 
										c_key_x - 3, 
										c_key_y - 3 );
			}
			
			if ( key == 0 /* ||    
				 key == 2 ||    
				 key == 4 ||    
				 key == 5 ||    
				 key == 7 ||    
				 key == 9 ||    
				 key == 11 */ ){
				
				char note= ' ';


				if ( key == 0  ) note = 'c';
				if ( key == 2  ) note = 'd';
				if ( key == 4  ) note = 'e';
				if ( key == 5  ) note = 'f';
				if ( key == 7  ) note = 'g';
				if ( key == 9  ) note = 'a';
				if ( key == 11 ) note = 'b';
				
				/* notes */
				char notes[5];
				sprintf( notes, "%c%1d", note, ((c_num_keys - i - 1) / 12) - 1 ); 
				
				
				m_gc.set_foreground(m_black);
				m_pixmap.draw_string(c_text_font_6_12, m_gc, 
									 4, 
									 c_key_y * i + c_key_y - 1,
									 notes );
			}
		}
}

void 
seqkeys::draw_area()
{
      update_pixmap();
      m_window.draw_pixmap(m_gc, 
			   m_pixmap, 
			   0,0,
			   0,0,
			   c_keyarea_x,
			   c_keyarea_y );
}


int 
seqkeys::expose_event_impl(GdkEventExpose* a_e)
{
    m_window.draw_pixmap(m_gc, 
      			 m_pixmap, 
      			 a_e->area.x,
  			 a_e->area.y,
  			 a_e->area.x,
  			 a_e->area.y,
  			 a_e->area.width,
  			 a_e->area.height );
    return true;
}



/* takes screen corrdinates, give us notes and ticks */
void 
seqkeys::convert_y( int a_y, int *a_note)
{
    *a_note = (c_rollarea_y - a_y - 2) / c_key_y; 
}


int 
seqkeys::button_press_event_impl(GdkEventButton *a_e)
{
    int y,note;
   
    if ( a_e->type == GDK_BUTTON_PRESS ){

	y = (int) a_e->y;

	if ( a_e->button == 1 ){
	    
	    m_keying = true;

	    convert_y( y,&note );
	    m_seq->play_note_on(  note );

	    m_keying_note = note;
	}
    }
    return true;
}


int 
seqkeys::button_release_event_impl(GdkEventButton* a_e)
{   
    if ( a_e->type == GDK_BUTTON_RELEASE ){

	if ( a_e->button == 1 && m_keying ){
	    
	    m_keying = false;
	    m_seq->play_note_off( m_keying_note );
	}
    }
    return true;
}


int 
seqkeys::motion_notify_event_impl(GdkEventMotion* a_p0)
{

    int y, note;
 
    if ( m_keying ){

	y = (int) a_p0->y;
	convert_y( y,&note );

	if ( note != m_keying_note ){

	    m_seq->play_note_off( m_keying_note );
	    m_seq->play_note_on(  note );
	    m_keying_note = note;

	}
    }

    return true;
}

int 
seqkeys::leave_notify_event_impl(GdkEventCrossing* p0)
{
    if ( m_keying ){

	m_keying = false;
	m_seq->play_note_off( m_keying_note );

    }

    return true;
}

/* sets key to grey */
void 
seqkeys::set_hint_key( int a_key )
{

  draw_key( m_hint_key, false );

  m_hint_key = a_key;
  
  if ( m_hint_state )
    draw_key( a_key, true );
}

/* true == on, false == off */
void 
seqkeys::set_hint_state( bool a_state )
{
  m_hint_state = a_state;

  if ( !a_state )
    draw_key( m_hint_key, false );
}

/* a_state, false = normal, true = grayed */
void 
seqkeys::draw_key( int a_key, bool a_state )
{

  /* the the key in the octave */
  int key = a_key % 12;

  a_key = c_num_keys - a_key - 1; 

  if ( key == 1 || 
       key == 3 || 
       key == 6 || 
       key == 8 || 
       key == 10 ){
    
    m_gc.set_foreground(m_black);
  }
  else
    m_gc.set_foreground(m_white);


  m_window.draw_rectangle(m_gc,true,
			  c_keyoffset_x + 1,
			  (c_key_y * a_key) + 2, 
			  c_key_x - 3, 
			  c_key_y - 3 );

  if ( a_state ){

    m_gc.set_foreground(m_grey);
 
    m_window.draw_rectangle(m_gc,true,
			    c_keyoffset_x + 1,
			    (c_key_y * a_key) + 2, 
			    c_key_x - 3, 
			    c_key_y - 3 );

  }
}

