//----------------------------------------------------------------------------
//
//  This file is part of seq24.
//
//  seq24 is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  seq24 is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with seq24; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//-----------------------------------------------------------------------------
#include "event.h"
#include "seqevent.h"


seqevent::seqevent(sequence *a_seq, int a_zoom, int a_snap, seqdata *a_seqdata_wid ): DrawingArea() 
{    
    using namespace Menu_Helpers;

    Gdk_Colormap colormap = get_default_colormap();

    m_black = Gdk_Color( "black" );
    m_white = Gdk_Color( "white" );
    m_grey  = Gdk_Color( "grey" );

    colormap.alloc( m_black );
    colormap.alloc( m_white );
    colormap.alloc( m_grey );

    m_seq =   a_seq;
    m_zoom = a_zoom;
    m_snap =  a_snap;
    m_seqdata_wid = a_seqdata_wid;

    add_events( GDK_BUTTON_PRESS_MASK | 
		GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK |
		GDK_KEY_PRESS_MASK |
		GDK_KEY_RELEASE_MASK |
		GDK_FOCUS_CHANGE_MASK );

    m_selecting = false;
    m_moving    = false;
    m_growing   = false;
    m_adding    = false;
    m_paste     = false;

    m_status = EVENT_NOTE_ON;
} 





void 
seqevent::realize_impl()
{
    // we need to do the default realize
    Gtk::DrawingArea::realize_impl();

    set_flags( GTK_CAN_FOCUS );

    // Now we can allocate any additional resources we need
    m_window = get_window();
    m_gc.create( m_window );
    m_window.clear();

}

int 
seqevent::idle_redraw()
{
    draw_events_on( &m_window );
    draw_events_on( &m_pixmap );
    return true;
}

void 
seqevent::update_sizes()
{
    /* set default size */
    size( m_seq->get_length() / m_zoom , c_eventarea_y );

    /* window size */
    m_window_x = m_seq->get_length() / m_zoom;
    m_window_y =  c_eventarea_y;

    /* create pixmaps with window dimentions */
    m_pixmap = Gdk_Pixmap( m_window,
			   m_window_x,
			   m_window_y );

    m_background = Gdk_Pixmap( m_window,
			       m_window_x,
			       m_window_y );

    /* and fill the background ( dotted lines n' such ) */
    fill_background_pixmap();
}

/* basically resets the whole widget as if it was realized again */
void 
seqevent::reset()
{
    update_sizes();
    update_pixmap();
    draw_pixmap_on_window();
}

/* updates background */
void 
seqevent::fill_background_pixmap()
{
    /* clear background */
    m_gc.set_foreground(m_white);
    m_background.draw_rectangle(m_gc,true,
				0,
				0, 
				m_window_x, 
				m_window_y );

    /* draw horz grey lines */
    m_gc.set_foreground(m_grey);
    m_gc.set_line_style( GDK_LINE_ON_OFF_DASH );
    m_gc.set_dashes( 1 );

    int numberLines = 128 / m_seq->get_bw() / m_zoom; 
    int distance = c_ppqn / 32;
   
    /* draw vert lines */
    for ( int i=0; i< c_maxbeats; i++ ){
	
	int base_line = i * (c_ppqn * 4) / m_seq->get_bw() / m_zoom;
	
	/* bars are darker */
	if ( i % m_seq->get_bpm() == 0 )
	    m_gc.set_foreground(m_black);
	else
	    m_gc.set_foreground(m_grey);

	/* solid line on every beat */
	m_gc.set_line_style( GDK_LINE_SOLID );
	m_background.draw_line(m_gc,
			       base_line,
			       0,
			       base_line,
			       m_window_y);
	
	/* in betweens */
	for ( int j=1; j < numberLines; j++ ){
	    
	    m_gc.set_foreground(m_grey);
	    m_gc.set_line_style( GDK_LINE_ON_OFF_DASH );
	    m_gc.set_dashes( 1 );
	    m_background.draw_line(m_gc,
				   base_line + j * distance,
				   0,
				   base_line + j * distance,
				   m_window_y);
	}
    }

    /* reset line style */
    m_gc.set_line_style( GDK_LINE_SOLID );
    m_gc.set_foreground(m_black);
    m_background.draw_line(m_gc,
			   0,
			   0,
			   m_window_x,
			   0 );

}

/* sets zoom, resets */
void 
seqevent::set_zoom( int a_zoom )
{
    if ( m_zoom != a_zoom ){

	m_zoom = a_zoom;
	reset();
    }
}

/* simply sets the snap member */
void 
seqevent::set_snap( int a_snap )
{
    m_snap = a_snap;
}


void 
seqevent::set_data_type( unsigned char a_status, unsigned char a_control = 0 )
{
    m_status = a_status;
    m_cc = a_control;

    this->reset();
}
   
 


/* draws background pixmap on main pixmap,
   then puts the events on */
void 
seqevent::update_pixmap()
{
    m_pixmap.draw_pixmap(m_gc, 
			 m_background, 
			 0,
			 0,
			 0,
			 0,
			 m_window_x,
			 m_window_y);

    draw_events_on_pixmap();
}


void
seqevent::draw_events_on( Gdk_Drawable *a_draw )
{
    long tick;

    int x;

    unsigned char d0,d1;

    bool selected;


    /* draw boxes from sequence */
    m_gc.set_foreground( m_black );

    m_seq->reset_draw_marker();
    while ( m_seq->get_next_event( m_status,
				   m_cc,
				   &tick, &d0, &d1, 
				   &selected ) == true ){
	
	/* turn into screen corrids */
	x = tick / m_zoom;

	m_gc.set_foreground(m_black);

	//  0   
	//  |    
	//  |   1  
	//  |   
	//  2   

	GdkPoint triangle[3];

	triangle[0].x = x;
	triangle[1].x = x + c_eventevent_x;
	triangle[2].x = x;

	triangle[0].y = (c_eventarea_y - c_eventevent_y)/2;
	triangle[1].y = c_eventarea_y /2;
	triangle[2].y = triangle[0].y + c_eventevent_y;

	m_gc.set_foreground(m_black);
	a_draw->draw_polygon(m_gc,true,
			     triangle,
			     3 );

	if ( selected )
	    m_gc.set_foreground(m_grey);
	else
	    m_gc.set_foreground(m_white);

	triangle[0].x++;
	triangle[1].x--;
	triangle[2].x++;

	triangle[0].y+=2;
	triangle[2].y-=2;

	a_draw->draw_polygon(m_gc,true,
			     triangle,
			     3 );




    }

}

/* fills main pixmap with events */
void 
seqevent::draw_events_on_pixmap()
{
    draw_events_on( &m_pixmap );
}

/* draws pixmap, tells event to do the same */
void 
seqevent::draw_pixmap_on_window()
{
    /* we changed something on this window, and chances are we
       need to update the event widget as well */
   
    queue_draw(); 

    // and update our velocity window
    m_seqdata_wid->update_pixmap();
    m_seqdata_wid->draw_pixmap_on_window();
}

/* checks mins / maxes..  the fills in x,y
   and width and height */
void 
seqevent::x_to_w( int a_x1, int a_x2,
		int *a_x, int *a_w  )
{
    if ( a_x1 < a_x2 ){
	*a_x = a_x1; 
	*a_w = a_x2 - a_x1;
    } else {
	*a_x = a_x2; 
	*a_w = a_x1 - a_x2;
    }
}

void 
seqevent::draw_selection_on_window()
{
    int x,w;

    int y = (c_eventarea_y - c_eventevent_y)/2;
    int h =  c_eventevent_y;  

    m_gc.set_line_style( GDK_LINE_SOLID );

    /* replace old */
    m_window.draw_pixmap(m_gc, 
			 m_pixmap, 
			 m_old.x,
			 y,
			 m_old.x,
			 y,
			 m_old.width + 1,
			 h + 1 );

    if ( m_selecting ){
	
	x_to_w( m_drop_x, m_current_x, &x,&w );

	m_old.x = x;
	m_old.width = w;

	m_gc.set_foreground(m_black);
	m_window.draw_rectangle(m_gc,false,
				x,
				y, 
				w, 
				h );
    }
    
    if ( m_moving || m_paste ){

	int delta_x = m_current_x - m_drop_x;

	x = m_selected.x + delta_x;

	m_gc.set_foreground(m_black);
	m_window.draw_rectangle(m_gc,false,
				x,
				y, 
				m_selected.width, 
				h );
	m_old.x = x;
	m_old.width = m_selected.width;
    }
}

int 
seqevent::expose_event_impl(GdkEventExpose* e)
{
    m_window.draw_pixmap(m_gc, 
    			 m_pixmap, 
    			 e->area.x,
			 e->area.y,
			 e->area.x,
			 e->area.y,
			 e->area.width,
			 e->area.height );

    draw_selection_on_window();
    return true;
}



void
seqevent::start_paste( )
{
     long tick_s;
     long tick_f;
     int note_h;
     int note_l;
     int x, w;

     snap_x( &m_current_x );
     snap_y( &m_current_x );

     m_drop_x = m_current_x;
     m_drop_y = m_current_y;

     m_paste = true;

     /* get the box that selected elements are in */
     m_seq->get_clipboard_box( &tick_s, &note_h, 
			       &tick_f, &note_l );

     /* convert box to X,Y values */
     convert_t( tick_s, &x );
     convert_t( tick_f, &w );

     /* w is actually corrids now, so we have to change */
     w = w-x; 

     /* set the m_selected rectangle to hold the
	x,y,w,h of our selected events */

     m_selected.x = x;                  
     m_selected.width=w;
     m_selected.y = (c_eventarea_y - c_eventevent_y)/2;
     m_selected.height = c_eventevent_y;  

     /* adjust for clipboard being shifted to tick 0 */
     m_selected.x += m_drop_x;
}





/* takes screen corrdinates, give us notes and ticks */
void 
seqevent::convert_x( int a_x, long *a_tick )
{
    *a_tick = a_x * m_zoom; 
}


/* notes and ticks to screen corridinates */
void 
seqevent::convert_t( long a_ticks, int *a_x )
{
    *a_x = a_ticks /  m_zoom;
}


/* popup menu calls this */
void
seqevent::set_adding( bool a_adding )
{
    if ( a_adding ){
 
	get_window().set_cursor(  Gdk_Cursor( GDK_PENCIL ));
	m_adding = true;
    }
    else {

	get_window().set_cursor( Gdk_Cursor( GDK_LEFT_PTR ));
	m_adding = false;
    }
}


int 
seqevent::button_press_event_impl(GdkEventButton* a_ev)
{
    int x,w,numsel;

    long tick_s;
    long tick_f;
    long tick_w;

    convert_x( c_eventevent_x, &tick_w  );

    /* if it was a button press */
    if ( m_status != EVENT_NOTE_OFF &&
	 m_status != EVENT_NOTE_ON  ){

	/* set values for dragging */
	m_drop_x = m_current_x = (int) a_ev->x;

	/* reset box that holds dirty redraw spot */
	m_old.x = 0;
	m_old.y = 0;
	m_old.width = 0;
	m_old.height = 0;

	if ( m_paste ){
    
	    snap_x( &m_current_x );
	    convert_x( m_current_x, &tick_s );
	    m_paste = false;
	    m_seq->paste_selected( tick_s, 0 );
	}

	/*      left mouse button     */
	if ( a_ev->button == 1 ){ 

	    /* turn x,y in to tick/note */
	    convert_x( m_drop_x, &tick_s );

	    /* shift back a few ticks */
	    tick_f = tick_s + m_zoom;
	    tick_s -= (tick_w - 2);

	    if ( tick_s < 0 ) tick_s = 0;

	    numsel = m_seq->select_events( tick_s, tick_f,
					   m_status,
					   m_cc );
	    
	    /* if we didnt select anyhing (user clicked empty space)
	       unselect all notes, and start selecting */
	    
	    /* none selected, start selection box */
	    if ( numsel == 0 && !m_adding ){ 
		
		m_seq->unselect();	    
		m_selecting = true;
	    }
	    
	    /* add a new note if we didnt select anything */
	    else if ( numsel == 0 && m_adding ){

		snap_x( &m_drop_x );
		/* turn x,y in to tick/note */
		convert_x( m_drop_x, &tick_s );
		/* add note, length = little less than snap */
		drop_event( tick_s );
	    }

	    /* if we clicked on an event, we can start moving all selected
	       notes */
	    else if ( numsel > 0 ){

		m_moving = true;
		int note;

		/* get the box that selected elements are in */
		m_seq->get_selected_box( &tick_s, &note, 
					 &tick_f, &note );

		tick_f += tick_w;

		/* convert box to X,Y values */
		convert_t( tick_s, &x );
		convert_t( tick_f, &w );

		/* w is actually corrids now, so we have to change */
		w = w-x; 

		/* like we clicked at the very begining, for snapping */
		m_drop_x = x;

		/* set the m_selected rectangle to hold the
		   x,y,w,h of our selected events */

		m_selected.x = x;                  
		m_selected.width=w;

		m_selected.y = (c_eventarea_y - c_eventevent_y)/2;
		m_selected.height = c_eventevent_y;  
	    } 
	    
	} /* end if button == 1 */
    
	if ( a_ev->button == 3 ){ 

	    set_adding( true );
	}

	/* if they clicked, something changed */
	update_pixmap();
	draw_pixmap_on_window();  

	return true;
    
    } 

    if ( m_status == EVENT_NOTE_OFF ||
	 m_status == EVENT_NOTE_ON ) {

	get_window().set_cursor( Gdk_Cursor( GDK_PIRATE ));
	return true;
    }

    return false;
}


void
seqevent::drop_event( long a_tick )
{
    
    unsigned char status = m_status;
    unsigned char d0 = m_cc;
    unsigned char d1 = 0x40;

    if ( m_status == EVENT_AFTERTOUCH )
	d0 = 0;
    
    if ( m_status == EVENT_PROGRAM_CHANGE )
	d0 = 0; /* d0 == new patch */

    if ( m_status == EVENT_CHANNEL_PRESSURE )
	d0 = 0x40; /* d0 == pressure */

    if ( m_status == EVENT_PITCH_WHEEL )
	d0 = 0;

    if ( m_status != EVENT_NOTE_OFF &&
	 m_status != EVENT_NOTE_ON ) {

	m_seq->add_event( a_tick, 
			  status,
			  d0,
			  d1 );
    }
}


int 
seqevent::button_release_event_impl(GdkEventButton* a_ev)
{
    long tick_s;
    long tick_f;
  
    int x,w;
    int numsel;

    grab_focus();

    m_current_x = (int) a_ev->x;

    if ( m_moving )
	snap_x( &m_current_x );

    int delta_x = m_current_x - m_drop_x;

    long delta_tick;

    if ( a_ev->button == 1 ){

	if ( m_selecting ){

	    x_to_w( m_drop_x, m_current_x, &x, &w );
	    
	    convert_x( x,   &tick_s );
	    convert_x( x+w, &tick_f );
	    
	    numsel = m_seq->select_events( tick_s, tick_f,
					   m_status,
					   m_cc );
	}

	if ( m_moving ){
	    
	    /* convert deltas into screen corridinates */
	    convert_x( delta_x, &delta_tick );
	    
	    /* not really notes, but still moves events */
	    m_seq->move_selected_notes( delta_tick, 0 );
	}

	set_adding( m_adding );
    }

    if ( a_ev->button == 3 ){

	set_adding( false );

    }

    /* turn off */
    m_selecting = false;
    m_moving = false;
    m_growing = false;

    /* if they clicked, something changed */
    update_pixmap();
    draw_pixmap_on_window(); 

    return true;
}



int 
seqevent::motion_notify_event_impl(GdkEventMotion* a_ev)
{
    if ( m_selecting || m_moving || m_paste  ){

	m_current_x = (int) a_ev->x;

	if ( m_moving || m_paste )
	    snap_x( &m_current_x );
 
	draw_selection_on_window();
    }
    return true;
}



/* performs a 'snap' on y */
void 
seqevent::snap_y( int *a_y )
{
    *a_y = *a_y - (*a_y % c_key_y);
}

/* performs a 'snap' on x */
void 
seqevent::snap_x( int *a_x )
{
    //snap = number pulses to snap to
    //m_zoom = number of pulses per pixel
    //so snap / m_zoom  = number pixels to snap to

    *a_x = *a_x - (*a_x % (m_snap / m_zoom));
}


int 
seqevent::focus_in_event_impl(GdkEventFocus*)
{
    set_flags(GTK_HAS_FOCUS);

    m_seq->clear_clipboard();

    draw_focus( );
    return false;
}

int 
seqevent::focus_out_event_impl(GdkEventFocus*)
{
    unset_flags(GTK_HAS_FOCUS);
    draw_focus();


    return false;
}

int 
seqevent::key_press_event_impl(GdkEventKey* a_p0)
{
    bool ret = false;

    if ( a_p0->type == GDK_KEY_PRESS ){

	if ( a_p0->keyval ==  GDK_Delete ){

	    m_seq->remove_selected();
	    ret = true;
	}

	if ( a_p0->state & GDK_CONTROL_MASK ){

	    /* cut */
	    if ( a_p0->keyval == GDK_x || a_p0->keyval == GDK_X ){
		
		m_seq->copy_selected();
		m_seq->remove_selected();

		ret = true;
	    }
	    /* copy */
	    if ( a_p0->keyval == GDK_c || a_p0->keyval == GDK_C ){
		
		m_seq->copy_selected();
		m_seq->unselect();
		ret = true;
	    }
	    /* paste */
	    if ( a_p0->keyval == GDK_v || a_p0->keyval == GDK_V ){
		
		start_paste();
		ret = true;
	    }
	}
    }

    if ( ret == true ){
	
	reset(); 
	m_seqdata_wid->reset();
	return true;
    }
    else

	return false;
}
