/***************************************************************************
                          main.cpp  -  description
                             -------------------
    begin                : Сбт Дек 13 MSK 2003
    copyright            : (C) 2003 by Evgeney
    email                : dushistov@mail.ru
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <cstdlib>
#include <cstdio>
#include <clocale>
#include <glib.h>
#include <glib/gi18n.h>
#include <getopt.h>
#include <string>
#include <vector>
#include <map>
#include <sys/stat.h>
#include <sys/types.h>

#ifdef WITH_READLINE
#  include <readline/readline.h>
#  include <readline/history.h>
#endif

#include "lib.h"

using std::string;
using std::vector;

//#define SHOW_STATS

//this structure is wrapper and it need for unification
//results of search whith return Libs class
struct TSearchResult{
  string bookname;
  string def;
  string exp;

  TSearchResult(const string& bookname_, const string& def_, const string& exp_)
    : bookname(bookname_), def(def_), exp(exp_)
  {
  }
};

typedef vector<TSearchResult> TSearchResultList;
typedef TSearchResultList::iterator PSearchResult;

//this class is wrapper around Libs class for easy use
//of it
class Library {
public:
  Library(Libs& libs) : oLibs(libs) {}

  void SimpleSearch(const char *str, TSearchResultList& res_list);
  void LookupWithFuzzy(const char *str, TSearchResultList& res_list);
  void LookupWithRule(const char *str, TSearchResultList& res_lsit);
private:
  Libs &oLibs;
};

char gVersion[] = VERSION;

static gchar *utf8_to_locale_ign_err(const gchar *utf8_str)
{
  gsize bytes_read, bytes_written;
  GError *err=NULL;
  gchar *res;
  
  const char * charset;
  if(g_get_charset(&charset))
    res=g_strdup(utf8_str);
  else{
    res=g_convert_with_fallback(utf8_str, -1, charset, "UTF-8", NULL, 
				&bytes_read, &bytes_written, &err);
    if(NULL==res){
      fprintf(stderr, _("Can not convert %s to current locale.\n"), utf8_str);
      fprintf(stderr, "%s\n", err->message);
      g_error_free(err);
      exit(EXIT_FAILURE);
    }
  }

  return res;
}

static gchar *locale_to_utf8(const gchar *loc_str)
{
  if(NULL==loc_str)
    return NULL;
  gsize bytes_read;
  gsize bytes_written;
  GError *err=NULL;
  gchar *str=NULL;
  str=g_locale_to_utf8(loc_str, -1, &bytes_read, &bytes_written, &err);
  if(NULL==str){
    fprintf(stderr, _("Can not convert %s to utf8.\n"), loc_str);
    fprintf(stderr, "%s\n", err->message);
    g_error_free(err);
    return str;
  }

  return str;
}

static void print_search_result(FILE *out, const TSearchResult & res, bool utf8_output)
{
  gchar *loc_bookname=NULL, *loc_def=NULL, *loc_exp=NULL;
  if(!utf8_output){
    loc_bookname=utf8_to_locale_ign_err(res.bookname.c_str());
    loc_def=utf8_to_locale_ign_err(res.def.c_str());
    loc_exp=utf8_to_locale_ign_err(res.exp.c_str());
  }
  
  printf("-->%s\n-->%s\n%s\n\n",
	 utf8_output ? res.bookname.c_str() : loc_bookname, 
	 utf8_output ? res.def.c_str() : loc_def, 
	 utf8_output ? res.exp.c_str() : loc_exp);
 
  g_free(loc_bookname);
  g_free(loc_def);
  g_free(loc_exp);
}

static string parse_data(const gchar *data)
{
  if (!data)
    return "";

  string res;
  guint32 data_size, sec_size=0;
  gchar *m_str;
  const gchar *p=data;
  data_size=*((guint32 *)p);
  p+=sizeof(guint32);
  while (p - (data + sizeof(guint32))<data_size) {
    switch (*p++) {
    case 'm':
    case 'l': //need more work...
    case 'g':
      sec_size = strlen(p);
      if (sec_size) {
	res+="\n";
	m_str = g_strndup(p, sec_size);
	res += m_str;
	g_free(m_str);							
      }
      sec_size++;
      break;
    case 't':
      sec_size = strlen(p);
      if(sec_size){
	res+="\n";
	m_str = g_strndup(p, sec_size);
	res += "["+string(m_str)+"]";
	g_free(m_str);
      }
      sec_size++;
      break;
    case 'y':
      sec_size = strlen(p);
      sec_size++;						
      break;
    case 'W':
    case 'P':
      sec_size=*((guint32 *)p);
      sec_size+=sizeof(guint32);
      break;
    }								
    p += sec_size;
  }

  
  return res;
}

static bool getline(FILE *in, string & str)
{
  str.clear();
  int ch;
  while((ch=fgetc(in))!=EOF && ch!='\n')
    str+=ch;
  if(EOF==ch)
    return false;
  return true;
}


void Library::SimpleSearch(const char *str, TSearchResultList& res_list)
{
  glong iWordIndex;
  for (int i=0; i<oLibs.total_libs(); ++i)
    if (oLibs.SimpleLookupWord(str, iWordIndex, i))
      res_list.push_back(TSearchResult(oLibs.GetBookname(i), str, parse_data(oLibs.poGetWordData(iWordIndex, i))));      
}

void Library::LookupWithFuzzy(const char *str, TSearchResultList& res_list)
{
#define MAX_FUZZY_MATCH_ITEM 4
  Libs::Fuzzystruct oFuzzystruct[MAX_FUZZY_MATCH_ITEM];
  if (oLibs.LookupWithFuzzy(str, oFuzzystruct,
			      MAX_FUZZY_MATCH_ITEM, NULL))
    for (int i=0; i<MAX_FUZZY_MATCH_ITEM; ++i) 
      if (oFuzzystruct[i].pMatchWord)
	SimpleSearch(oFuzzystruct[i].pMatchWord, res_list);
}

void Library::LookupWithRule(const char *str, TSearchResultList& res_list)
{
  gchar **ppMatchWord = 
    (gchar **)g_malloc(sizeof(gchar *) * MAX_MATCH_ITEM_PER_LIB * oLibs.total_libs());
  gint count = oLibs.LookupWithRule(str, NULL, ppMatchWord);
  

  for (gint i=0; i<count; i++)
    if (ppMatchWord[i]) {
      SimpleSearch(ppMatchWord[i], res_list);
      g_free(ppMatchWord[i]);
    }

  g_free(ppMatchWord);
}


static void process_phrase(const char *loc_str, Library& lib, bool utf8_input,
			   bool utf8_output, bool force=false)
{
  if(NULL==loc_str)
    return;

#ifdef WITH_READLINE
{
  const char *hist=loc_str;
  if (hist && hist[0]=='/')
    ++hist;
#if 0
  int pos=-1;
  if ((pos=history_search(hist, -1))!=-1 || (pos=history_search(hist, 0))!=-1) {
    HIST_ENTRY *he=remove_history(pos);
    free_history_entry(he);
  }
#endif
  if (hist && *hist!='\0')
    add_history(hist);
}
#endif

#ifdef SHOW_STATS
  clock_t t=clock();
#endif
  gsize bytes_read;
  gsize bytes_written;
  GError *err=NULL;
  char *str=NULL;
  if (!utf8_input)
    str=g_locale_to_utf8(loc_str, -1, &bytes_read, &bytes_written, &err);
  else
    str=g_strdup(loc_str);
  if (NULL==str) {
    fprintf(stderr, _("Can not convert %s to utf8.\n"), loc_str);
    fprintf(stderr, "%s\n", err->message);
    g_error_free(err);
    return;
  }

  if(str[0]=='\0')
    return;

  
  TSearchResultList res_list;

  if (str[0]=='/')
    lib.LookupWithFuzzy(str+1, res_list);
  else if (bContainRule(str))
    lib.LookupWithRule(str, res_list);
  else {
    lib.SimpleSearch(str, res_list);
    if (res_list.empty())
      lib.LookupWithFuzzy(str, res_list);
  }
#ifdef SHOW_STATS
  t=clock()-t;
  printf("Time of search: %lf\n", double(t)/CLOCKS_PER_SEC);
#endif
  if (!res_list.empty()) {
    gchar *loc_str=NULL;
    if(!utf8_output)
      loc_str=utf8_to_locale_ign_err(str);
    printf(_("Found %d items, similar to %s.\n"), res_list.size(), utf8_output ? str : loc_str);

    g_free(loc_str);
    /* try to be more clever, if there are
       one or zero results per dictionary show all
    */
    bool show_all_results=true;
    typedef std::map< string, int, std::less<string> > DictResMap;
    if (!force) {
      DictResMap res_per_dict;
      for(TSearchResultList::iterator ptr=res_list.begin(); ptr!=res_list.end(); ++ptr){
	std::pair<DictResMap::iterator, DictResMap::iterator> r=res_per_dict.equal_range(ptr->bookname);
	DictResMap tmp(r.first, r.second);
	if (tmp.empty()) //there are no yet such bookname in map
	  res_per_dict.insert(DictResMap::value_type(ptr->bookname, 1));
	else {
	  ++((tmp.begin())->second);
	  if (tmp.begin()->second>1) {
	    show_all_results=false;
	    break;
	  }
	}
      }
    }//if (!force)

    if (!show_all_results) {
      if (force) {
        PSearchResult ptr;
	for(ptr=res_list.begin(); ptr!=res_list.end(); ++ptr)
	  print_search_result(stdout, *ptr, utf8_output);
      } else{
	for (size_t i=0; i<res_list.size(); ++i) {
	  gchar *loc_bookname=NULL, *loc_def=NULL;
	  loc_bookname=utf8_to_locale_ign_err(res_list[i].bookname.c_str());
	  loc_def=utf8_to_locale_ign_err(res_list[i].def.c_str());
	  printf("%d)%s-->%s\n", i,
		 utf8_output ?  res_list[i].bookname.c_str() : loc_bookname,
		 utf8_output ? res_list[i].def.c_str() : loc_def);					 
	}
	int choise;
	for (;;) {
	  string str_choise;
	  printf(_("Your choice: "));

	  if(!getline(stdin, str_choise)){
	    putchar('\n');
	    exit(EXIT_SUCCESS);
	  }
	  sscanf(str_choise.c_str(), "%d", &choise);
	  if (choise>=0 && choise<int(res_list.size())) { 
	    print_search_result(stdout, res_list[choise], utf8_output);
	    break;
	  } else 
	    printf(_("Invalid choise.\nIt must be from 0 to %d.\n"), res_list.size()-1);	  
	}
      }
    } else 
      for(PSearchResult ptr=res_list.begin(); ptr!=res_list.end(); ++ptr)
	print_search_result(stdout, *ptr, utf8_output);
    
  } else {
    gchar *loc_str=NULL;
    if (!utf8_output)
      loc_str=utf8_to_locale_ign_err(str);
    
    printf(_("Nothing similar to %s, sorry :(\n"), utf8_output ? str : loc_str);
    g_free(loc_str);
  }
  g_free(str);
}


struct option longopts[] ={
  {"version", no_argument, NULL, 'v' },
  {"help", no_argument, NULL, 'h' },
  {"list-dicts", no_argument, NULL, 'l'},
  {"use-dict", required_argument, NULL, 'u'},
  {"non-interactive", no_argument, NULL, 'n'},
  {"utf8-output", no_argument, NULL, 0},
  {"utf8-input", no_argument, NULL, 1},
  {"data-dir", required_argument, NULL, 2},
  { NULL, 0, NULL, 0 }
};


int main(int argc, char *argv[])
{
#ifdef SHOW_STATS
  clock_t t=clock();
#endif
  setlocale(LC_ALL, "");
#if ENABLE_NLS
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);
#endif	 
  int optc;
  bool h = false, v = false, show_list_dicts=false, 
    use_book_name=false, non_interactive=false, 
    utf8_output=false, utf8_input=false;
  GSList *enable_list=NULL;
  string data_dir;
  int option_index = 0;
  while((optc = getopt_long (argc, argv, "hvu:ln", longopts, 
                             &option_index))!=-1)
    switch (optc){
    case 0:
      utf8_output=true;
      break;
    case 1:   
      utf8_input=true;
      break;
    case 2:
      data_dir=optarg;
      break;
    case 'v':
      v = true;
      break;
    case 'h':
      h = true;
      break;
    case 'l':
      show_list_dicts=true;
      break;
    case 'u':
      use_book_name=true;
      enable_list=g_slist_append(enable_list, locale_to_utf8(optarg));
      break;
    case 'n':
      non_interactive=true;
      break;
    case '?':
      fprintf(stderr, _("Unknown option.\nTry '%s --help' for more information.\n"), argv[0]);
      return EXIT_FAILURE;
    }
  
  if (h) {
    printf("sdcv - console version of StarDict.\n");
    printf(_("Usage: %s [OPTIONS] words\n"), argv[0]);
    printf(_("-h, --help               display this help and exit\n"));
    printf(_("-v, --version            display version information and exit\n"));
    printf(_("-l, --list-dicts         display list of available dictionaries and exit\n"));
    printf(_("-u, --use-dict bookname  for search use only dictionary with this bookname\n"));
    printf(_("-n, --non-interactive    for use in scripts\n"));
    printf(_("--utf8-output            output must be in utf8\n"));
    printf(_("--utf8-input             input of sdcv in utf8\n"));
    printf(_("--data-dir path/to/dir   use this directory as path to stardict data directory\n"));

    return EXIT_SUCCESS;
  }

  if (v) {
    printf(_("Console version of Stardict, version %s\n"), gVersion);
    return EXIT_SUCCESS;
  }

  const gchar *stardict_data_dir=g_getenv("STARDICT_DATA_DIR");
  if (data_dir.empty() && stardict_data_dir)
    data_dir=stardict_data_dir;

  if (data_dir.empty())
    data_dir="/usr/share/stardict";

  if (show_list_dicts) {
    DictInfoList  dict_info_list;
    Libs::GetDictInfoList(data_dir+G_DIR_SEPARATOR+"dic", NULL, NULL, dict_info_list);
    printf(_("bookname     wordcount\n"));
    for(size_t i=0; i<dict_info_list.size(); ++i){
      gchar *bookname=utf8_to_locale_ign_err(dict_info_list[i].bookname.c_str());
      printf("%s    %d\n", bookname, dict_info_list[i].wordcount);
      g_free(bookname);
    }
    
    return EXIT_SUCCESS;
  }

  GSList *disable_list=NULL;
  DictInfoList  dict_info_list;
  
  if (use_book_name) {
    Libs::GetDictInfoList(data_dir+G_DIR_SEPARATOR+"dic", NULL, NULL, dict_info_list);
    
    GSList *not_disable_list=NULL;
    for (GSList *ptr=enable_list; ptr; ptr=g_slist_next(ptr)) {
      bool there_is_dict_with_this_name=false;
      for(size_t i=0; i<dict_info_list.size(); ++i){
	if(strcmp((char *)(ptr->data), dict_info_list[i].bookname.c_str())==0){
	  there_is_dict_with_this_name=true;
	  not_disable_list=g_slist_append(not_disable_list, 
			                  g_strdup(dict_info_list[i].ifo_file_name.c_str()));
	}
      }
      if(!there_is_dict_with_this_name){
	fprintf(stderr, _("There is no dictionary with this bookname: %s.\n"), ptr->data);
	exit(EXIT_FAILURE);
      }
    }

    for (size_t i=0; i<dict_info_list.size(); ++i) {
      bool disable=true;
      string ifofilename=dict_info_list[i].ifo_file_name;
      for(GSList *ptr=not_disable_list; ptr; ptr=g_slist_next(ptr))
	if(strcmp(ifofilename.c_str(), (char *)(ptr->data))==0)
	  disable=false;
      if(disable)
	disable_list=g_slist_append(disable_list,
		                    g_strdup(ifofilename.c_str()));
    }
    g_slist_foreach(not_disable_list, (GFunc)g_free, NULL);
    g_slist_free(not_disable_list);
  }

    
  string conf_dir = string(g_get_home_dir())+G_DIR_SEPARATOR+".stardict";
  mkdir(conf_dir.c_str(), S_IRWXU);

  
  Libs lib;
  lib.Load(data_dir+G_DIR_SEPARATOR+"dic", NULL, disable_list);
#ifdef SHOW_STATS
  t=clock()-t;
  printf("Time of start: %lf\n", double(t)/CLOCKS_PER_SEC);
#endif
  Library library(lib);

#ifdef WITH_READLINE
  using_history();
  string histname=(string(g_get_home_dir())+G_DIR_SEPARATOR+".sdcv_history");
  read_history(histname.c_str());;
#endif

  if (optind < argc) {
    for(int i=optind; i<argc; ++i)
      process_phrase(argv[i], library, utf8_input, utf8_output, non_interactive);
  } else if(!non_interactive) {    
#ifdef WITH_READLINE
    char *phrase=NULL;
    do {
      phrase=readline(_("Enter word or phrase: "));
      if (phrase) {
	process_phrase(phrase, library, utf8_input, utf8_output);
	free(phrase);
      }
    } while (phrase);
#else
    string phrase;
    printf(_("Enter word or phrase: "));
    while (getline(stdin, phrase)) {
      process_phrase(phrase.c_str(), library, utf8_input, utf8_output);
      printf(_("Enter word or phrase: "));
    }
#endif
    putchar('\n');
  } else {
    fprintf(stderr, _("There are no words/phrases to translate.\n"));
  }

#ifdef WITH_READLINE
  write_history(histname.c_str());
  const gchar *hist_size_str=g_getenv("SDCV_HISTSIZE");
  int hist_size;
  if (!hist_size_str || sscanf(hist_size_str, "%d", &hist_size)<1)
    hist_size=2000;
  history_truncate_file(histname.c_str(), hist_size);
#endif 

  g_slist_foreach(disable_list, (GFunc)g_free, NULL); 
  g_slist_free(disable_list);
  
  return EXIT_SUCCESS;
}

