/*
 * $Id: streamengine.c,v 1.19 2004/01/22 15:10:01 rohde Exp $
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * Acknowledgement
 * This work was partially funded by the Bundesministerium fr Bildung und
 * Forschung (BMBF) of the Federal Republic of Germany (Frderkennzeichen 01AK045).
 * The authors alone are responsible for the contents.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 *
 * Purpose: This modules implements the interface defined in streamengine.h and
 *          holds the private list of streams to process them for reasons of
 *          sending and receiving datachunks.
 *
 */

#include <assert.h>
#include <errno.h>
#include <glib.h>

#include "globals.h"
#include "sctp.h"
#include "distribution.h"
#include "adaptation.h"


#include "flowcontrol.h"
#include "bundling.h"
#include "errorhandler.h"

#include "recvctrl.h"

#include "streamengine.h"




/******************** Structure Definitions ****************************************/

typedef struct
{
    GList *pduList;          /* list of PDUs waiting for pickup (after notification has been called) */
    GList *prePduList;       /* list of PDUs waiting for transfer to pduList and doing mdi arrive notification */
    guint16 nextSSN;
    guint16 highestSSN;      /* used to detect Protocol violations in se_searchReadyPdu */
    gboolean highestSSNused;
    int index;
}ReceiveStream;

typedef struct
{
    unsigned int nextSSN;
    GList *messageList;
    unsigned int messagesQueued;
    unsigned int chunksQueued;
    unsigned int bytesQueued;
}SendStream;


typedef struct
{
    Association*    myAssociation;
    unsigned int    numSendStreams;
    unsigned int    numReceiveStreams;
    gboolean        unreliable;

    ReceiveStream*  RecvStreams;
    gboolean*       recvStreamActivated;
    unsigned int    queuedBytesRecvQueue;

    SendStream*     SendStreams;
    gboolean*       sendStreamActivated;
    unsigned int    queuedChunksSendQueue;
    unsigned int    queuedBytesSendQueue;
    unsigned int    lastSendStream;
    
    gboolean        messageIsScheduled;
    message_vector* scheduledMessage;
    int             scheduledStream;
    gboolean        shutdown_received;
    
    GList           *List;	 /* list for all packets */
}StreamEngine;

/*
 * this stores all the data need to be delivered to the user
 */
typedef struct _delivery_data
{
    guint8 chunk_flags;
    guint16 data_length;
    guint32 tsn;
    guint16 stream_id;
    guint16 stream_sn;
    guint32 protocolId;
    guchar data[MAX_DATACHUNK_PDU_LENGTH];
}
delivery_data;


/*
 * this struct stores several chunks that can be delivered to
 * the user as one message.
 */
typedef struct _delivery_pdu
{
    guint32  number_of_chunks;
    guint32  read_position;
    guint32  read_chunk;
    guint32  chunk_position;
    guint32  total_length;
    /* one chunk pointer or an array of these */
    delivery_data** ddata;
}delivery_pdu;


/*
typedef struct extended_message_vector
{
    chunk_data*   cdat[SCTP_MAX_NUM_CHUNKS];
    unsigned int  numberOfChunks;
    unsigned int  totalLength;
}
message_vector;
*/

/******************** Declarations and Helper Functions ****************************************/
int se_searchReadyPdu(StreamEngine* se);
int se_deliverWaiting(StreamEngine* se, unsigned short sid);

void print_element(gpointer list_element, gpointer user_data)
{
    delivery_data * one = (delivery_data *)list_element;
    if (one) {
        event_logii (VERBOSE, "chunklist: tsn %u, SID %u",one->tsn, one->stream_id);
    } else {
        event_log (VERBOSE, "chunklist: NULL element ");
    }
}

int sort_tsn_se(delivery_data * one, delivery_data * two)
{
    if (before(one->tsn, two->tsn)) {
        return -1;
    } else if (after(one->tsn, two->tsn)) {
        return 1;
    } else                      /* one==two */
        return 0;
}

/******************** Function Definitions *****************************************/

/**
 *  This function is called to instanciate one Stream Engine for an association.
 *  It creates and initializes the Lists for Sending and Receiving Data.
 *  It is called by Message Distribution.
 *  @return  pointer to the Stream Engine
 */
void* se_new_stream_engine (Association* asok,


                            unsigned int numberReceiveStreams,        /* max of streams to receive */
                            unsigned int numberSendStreams,           /* max of streams to send */
                            gboolean assocSupportsPRSCTP)
{
    unsigned int i;
    StreamEngine* se;

    event_logiii (EXTERNAL_EVENT, "new_stream_engine: #inStreams=%d, #outStreams=%d, unreliable == %s",
            numberReceiveStreams,	numberSendStreams, (assocSupportsPRSCTP==TRUE)?"TRUE":"FALSE");

    /* Allocate all the memory */
    se = (StreamEngine*) malloc(sizeof(StreamEngine));
    if (se == NULL) {
        error_log(ERROR_FATAL,"Out of Memory in se_new_stream_engine()");
        return NULL;
    }
    se->RecvStreams = (ReceiveStream*)malloc(numberReceiveStreams*sizeof(ReceiveStream));
    if (se->RecvStreams == NULL) {
        free(se);
        error_log(ERROR_FATAL,"Out of Memory in se_new_stream_engine()");
        return NULL;
    }
    se->recvStreamActivated = (gboolean*)malloc(numberReceiveStreams*sizeof(gboolean));
    if (se->recvStreamActivated == NULL) {
        free(se->RecvStreams);
        free(se);
        error_log(ERROR_FATAL,"Out of Memory in se_new_stream_engine()");
        return NULL;
    }

    se->SendStreams = (SendStream*)malloc(numberSendStreams*sizeof(SendStream));
    if (se->SendStreams == NULL) {
        free(se->recvStreamActivated);
        free(se->RecvStreams);
        free(se);
        error_log(ERROR_FATAL,"Out of Memory in se_new_stream_engine()");
        return NULL;
    }
    se->sendStreamActivated = (gboolean*)malloc(numberSendStreams*sizeof(gboolean));
    if (se->sendStreamActivated == NULL) {
        free(se->recvStreamActivated);
        free(se->RecvStreams);
        free(se->SendStreams);
        free(se);
        error_log(ERROR_FATAL,"Out of Memory in se_new_stream_engine()");
        return NULL;
    }


    /* initialize the structures */
    for (i=0; i < numberReceiveStreams; i++) se->recvStreamActivated[i] = FALSE;
    for (i=0; i < numberSendStreams; i++)    se->sendStreamActivated[i] = FALSE;
    se->numSendStreams      = numberSendStreams;
    se->numReceiveStreams   = numberReceiveStreams;
    se->unreliable          = assocSupportsPRSCTP;

    for (i = 0; i < numberReceiveStreams; i++) {
      (se->RecvStreams)[i].nextSSN = 0;
      (se->RecvStreams)[i].pduList = NULL;
      (se->RecvStreams)[i].prePduList = NULL;
      (se->RecvStreams)[i].index = 0; /* for ordered chunks, next ssn */
    }

    for (i = 0; i < numberSendStreams; i++)
    {
      (se->SendStreams)[i].nextSSN = 0;
      (se->SendStreams)[i].messagesQueued = 0;
      (se->SendStreams)[i].bytesQueued = 0;
      (se->SendStreams)[i].messageList = 0;
    }
    se->queuedBytesRecvQueue    = 0;
    se->myAssociation           = asok;
    se->lastSendStream          = numberSendStreams;
    se->queuedChunksSendQueue   = 0;
    se->queuedBytesSendQueue    = 0;
    se->scheduledMessage          = NULL;
    se->scheduledStream         = -1;
    se->messageIsScheduled        = FALSE;
    se->shutdown_received       = FALSE;
    se->List          = NULL;
    return (se);
}



/**
 * Deletes the streamengine instance of this association.
 */
void se_delete_stream_engine (Association* asok)
{
    unsigned int i;
    StreamEngine* se = (StreamEngine*)mdi_readStreamEngine (asok);


    assert(se);
    event_log (INTERNAL_EVENT_0, "delete streamengine: freeing send streams");
    for (i = 0; i < se->numSendStreams; i++) {
        g_list_foreach(se->SendStreams[i].messageList, &free_list_element, NULL);
        g_list_free(se->SendStreams[i].messageList);
    }
    free(se->SendStreams);
    free(se->sendStreamActivated);
    for (i = 0; i < se->numReceiveStreams; i++) {
        event_logi (VERBOSE, "delete streamengine: freeing data for receive stream %d",i);
        /* whatever is still in these lists, delete it before freeing the lists */
        g_list_foreach(se->RecvStreams[i].pduList, &free_list_element, NULL);
        g_list_foreach(se->RecvStreams[i].prePduList, &free_list_element, NULL);
        g_list_free(se->RecvStreams[i].pduList);
        g_list_free(se->RecvStreams[i].prePduList);
    }
    event_log (INTERNAL_EVENT_0, "delete streamengine: freeing receive streams");
    free(se->RecvStreams);
    free(se->recvStreamActivated);
    free (se);
    event_log (EXTERNAL_EVENT, "deleted streamengine");
}



int se_readNumberOfStreams (Association* asok,
                            unsigned short *inStreams,
                            unsigned short *outStreams)
{
    StreamEngine* se = (StreamEngine*) mdi_readStreamEngine (asok);

    if (se == NULL) {
        error_log(ERROR_MAJOR, "Called se_readNumberOfStreams, but no Streamengine is there !");
        *inStreams = 0;
        *outStreams = 0;
        return SCTP_MODULE_NOT_FOUND;
    }
    *inStreams = se->numReceiveStreams;
    *outStreams = se->numSendStreams;
    return SCTP_SUCCESS;
}


/******************** Functions for Sending *****************************************/

int se_enqueue_message(StreamEngine* se, message_vector* messageVec,  unsigned short streamId)
{
    se->SendStreams[streamId].messageList = g_list_append(se->SendStreams[streamId].messageList,
                                                            messageVec);
    se->SendStreams[streamId].bytesQueued  += messageVec->totalLength;
    se->SendStreams[streamId].messagesQueued++;
    se->SendStreams[streamId].chunksQueued += messageVec->numberOfChunks;
    se->queuedChunksSendQueue   += messageVec->numberOfChunks;
    se->queuedBytesSendQueue    += messageVec->totalLength;
    se->sendStreamActivated[streamId] = TRUE;

    return SCTP_SUCCESS;
}


message_vector* se_schedule_next_message (Association* asok)
{
    int nextStream;
    StreamEngine* se = (StreamEngine*) mdi_readStreamEngine (asok);
    assert(se);
    if (se->messageIsScheduled && se->scheduledMessage != NULL)
        return se->scheduledMessage;
    /* okay, we are still here: choose one (activated) stream   */
    /* and store it, store pointer to first chunk, and set flag */
    /* return that data */

    assert(se->numSendStreams != 0);
    
    nextStream = (se->lastSendStream + 1) % se->numSendStreams;
    while (nextStream != se->lastSendStream && !se->sendStreamActivated[nextStream]) {
        nextStream = (nextStream + 1) % se->numSendStreams;
    }

    if (se->sendStreamActivated[nextStream] == FALSE) return NULL;

    /* okay: we found an activated stream, do bookkeeping, and return message */
    se->messageIsScheduled = TRUE;
    se->scheduledMessage = g_list_nth_data(se->SendStreams[nextStream].messageList, 0);
    se->scheduledStream = nextStream;
        
    return se->scheduledMessage;
}


int se_dequeue_message(Association* asok, message_vector* mVec)
{
    StreamEngine* se = (StreamEngine*) mdi_readStreamEngine (asok);
    assert(se);
    assert(mVec);
    
    se->SendStreams[se->scheduledStream].messageList =
        g_list_remove(se->SendStreams[se->scheduledStream].messageList, mVec);

    se->SendStreams[se->scheduledStream].messagesQueued--;
    se->SendStreams[se->scheduledStream].chunksQueued -= mVec->numberOfChunks;
    se->SendStreams[se->scheduledStream].bytesQueued -= mVec->totalLength;
    se->queuedChunksSendQueue -= mVec->numberOfChunks;
    se->queuedBytesSendQueue -= mVec->totalLength;
    se->lastSendStream = se->scheduledStream;
    free(mVec);
    
    se->messageIsScheduled = FALSE;
    se->scheduledMessage = NULL;
    se->scheduledStream = -1;

    return SCTP_SUCCESS;
}


/**
 * This function is called to send a chunk.
 * It is  called from message distribution.
 * @return SCTP_SUCCESS for success, -1 for error (e.g. data sent in shutdown state etc.)
*/
int se_ulpsend(Association* asok,
               unsigned short streamId,
               unsigned char *buffer,
               unsigned int byteCount, unsigned int protocolId,
               unsigned int destAddressKey,  unsigned int lifetime,void* context,
               gboolean unorderedDelivery,  /* optional (=FALSE if none) */
               gboolean dontBundle)         /* optional (=null if none)  */
{
    StreamEngine* se = NULL;
    guint32 state;
    chunk_data*  cdata = NULL;
    SCTP_data_chunk* dchunk = NULL;
    unsigned char* bufPosition = buffer;
    message_vector* mVec = NULL;

    unsigned int bCount = 0, maxQueueLen = 0;
    int numberOfSegments, residual;

    int i = 0;
    int result = 0, retVal;
    int maxDataLength;
    


    if (sci_shutdown_procedure_started(asok) == TRUE)
    {
        state = sci_getState(asok);
        event_logi (EXTERNAL_EVENT,
        "se_ulpsend: Cannot send Chunk, Association (state==%u) in SHUTDOWN-phase", state);
        return SCTP_SPECIFIC_FUNCTION_ERROR;
    }

    event_logii (EXTERNAL_EVENT, "se_ulpsend : %u bytes for stream %u", byteCount,streamId);

    se = (StreamEngine*) mdi_readStreamEngine (asok);
    if (se == NULL)
    {
        error_log (ERROR_MAJOR, "se_ulpsend: StreamEngine Instance doesn't exist....Returning !");
        return SCTP_MODULE_NOT_FOUND;
    }
    if (se->shutdown_received == TRUE)
    {
        event_log(EXTERNAL_EVENT, "se_ulpsend: SHUTDOWN already received");
        return SCTP_SPECIFIC_FUNCTION_ERROR;
    }
    
    if (streamId >= se->numSendStreams)
    {
        error_logii (ERROR_MAJOR, "STREAM ID OVERFLOW in se_ulpsend: wanted %u, got only %u",
            streamId, se->numSendStreams);
        mdi_sendFailureNotif (asok, buffer, byteCount, context);
        return SCTP_PARAMETER_PROBLEM;
    }

    if (byteCount > SCTP_MAXIMUM_MESSAGE_LENGTH) {
        return SCTP_MESSAGE_TOO_LONG;        
    }
    result = fc_get_maxSendQueue(asok, &maxQueueLen);
    if (result != SCTP_SUCCESS) return SCTP_UNSPECIFIED_ERROR;

    retVal = SCTP_SUCCESS;
    mVec = malloc(sizeof(message_vector));
    if (mVec == NULL) return SCTP_OUT_OF_RESOURCES;
    mVec->numberOfChunks = 0;

    mVec->totalLength = 0;

    //! \todo think about a more elegant solution
    //! \todo Currently msxDataLength is mtu - DataChunkHeader - SACK-Header
    //! \todo - assumed MaxSize of variable fields in SackChunk
    maxDataLength = pm_readAsocMTU(asok) - sizeof(SCTP_common_header) - sizeof(SCTP_chunk_header) - sizeof(SCTP_data_chunk_header) ;
    if (byteCount <= maxDataLength)
    {
        if (maxQueueLen > 0) {
            if ((1 + se->queuedChunksSendQueue + fc_readNumberOfQueuedChunks(asok)) > maxQueueLen)
                return SCTP_QUEUE_EXCEEDED;
        }

        cdata = malloc(sizeof(chunk_data));
        if (cdata == NULL) return SCTP_OUT_OF_RESOURCES;

        dchunk = (SCTP_data_chunk*)cdata->data;
        dchunk->chunk_id      = CHUNK_DATA;
        dchunk->chunk_flags   = SCTP_DATA_BEGIN_SEGMENT + SCTP_DATA_END_SEGMENT;
        dchunk->chunk_length  = htons (byteCount + FIXED_DATA_CHUNK_SIZE);
        dchunk->tsn = htonl(0);        /* gets assigned in the flowcontrol module */
        dchunk->stream_id     = htons (streamId);
        dchunk->protocolId    = htonl (protocolId);

        if (unorderedDelivery)
        {
            dchunk->stream_sn = htons (0);
            dchunk->chunk_flags += SCTP_DATA_UNORDERED;
        }
        else
        {       /* unordered flag not put */
            dchunk->stream_sn = htons (se->SendStreams[streamId].nextSSN);
            se->SendStreams[streamId].nextSSN++;
        }
        /* copy the data, but only once ! */
        memcpy (dchunk->data, buffer, byteCount);


        if (!se->unreliable) lifetime = SCTP_INFINITE_LIFETIME;
        /* NEW: append chunk to the stream's list */
        cdata->chunk_len   = byteCount + FIXED_DATA_CHUNK_SIZE;
        cdata->chunk_tsn = 0;
        cdata->gap_reports = 0;
        cdata->ack_time    = 0;
        cdata->context     = context;
        cdata->dontBundle  = dontBundle;
        cdata->num_of_transmissions = 0;
        cdata->hasBeenAcked = FALSE;
        cdata->hasBeenDropped = FALSE;
        cdata->initialDestinationKey = destAddressKey;
        /* check these two next statements..***************** */
        cdata->lastDestinationKey = 0;
        timerclear(&(cdata->transmission_time));
        /* up to here *************************************** */
        if (lifetime == SCTP_INFINITE_LIFETIME) {
            timerclear(&(cdata->expiry_time));
        } else if (lifetime == 0) {
            adl_gettime(&(cdata->expiry_time));
        } else {
            adl_gettime(&(cdata->expiry_time));
            adl_add_msecs_totime(&(cdata->expiry_time), lifetime);
        }
        mVec->cdat[0] = cdata;
        mVec->numberOfChunks = 1;
        /* FIXME: we add header to chunk len, and also count it in our congestion/flow control... */
        mVec->totalLength = cdata->chunk_len;
        event_logiii(EXTERNAL_EVENT, "=========> SE queues one chunk from ULP (SSN=%u, SID=%u, Len=%u) <=======",
                      ntohs (dchunk->stream_sn),ntohs (dchunk->stream_id), cdata->chunk_len);
       
    }
    else
    {
        /* calculate nr. of necessary chunks -> use fc_getMTU() later !!! */
        numberOfSegments = byteCount / maxDataLength;	
        residual = byteCount % maxDataLength;
        if (residual != 0) {
            numberOfSegments++;
        } else {
            residual = maxDataLength;
        }


        if (maxQueueLen > 0) {
            if ((numberOfSegments + se->queuedChunksSendQueue + fc_readNumberOfQueuedChunks(asok)) > maxQueueLen)
                return SCTP_QUEUE_EXCEEDED;
        }

        for (i = 1; i <= numberOfSegments; i++) {
            cdata = malloc(sizeof(chunk_data));
            assert(cdata);
            /* if we run out of memory here, it will be a mess */

            dchunk = (SCTP_data_chunk*)cdata->data;




            if ((i != 1) && (i != numberOfSegments)) {
                dchunk->chunk_flags = 0;
                bCount = maxDataLength;
                event_log (VERBOSE, "NEXT FRAGMENTED CHUNK -> MIDDLE");
            }
            else if (i == 1) {
                dchunk->chunk_flags = SCTP_DATA_BEGIN_SEGMENT;
                event_log (VERBOSE, "NEXT FRAGMENTED CHUNK -> BEGIN");
                bCount = maxDataLength;
            }
            else if (i == numberOfSegments) {
                dchunk->chunk_flags = SCTP_DATA_END_SEGMENT;
                event_log (EXTERNAL_EVENT, "NEXT FRAGMENTED CHUNK -> END");
                bCount = residual;
            }
            dchunk->chunk_id = CHUNK_DATA;
            dchunk->chunk_length = htons (bCount + FIXED_DATA_CHUNK_SIZE);
            dchunk->tsn = htonl (0);
            dchunk->stream_id  = htons (streamId);
            dchunk->protocolId = htonl (protocolId);

            if (unorderedDelivery) {
                dchunk->stream_sn = 0;
                dchunk->chunk_flags += SCTP_DATA_UNORDERED;
            } else {
                /* unordered flag not put */
                dchunk->stream_sn = htons (se->SendStreams[streamId].nextSSN);
                /* only after the last segment we increase the SSN */
                if (i == numberOfSegments) {
                    se->SendStreams[streamId].nextSSN++;
                }
            }
            memcpy (dchunk->data, bufPosition, bCount);
            bufPosition += bCount * sizeof(unsigned char);


            if (!se->unreliable) lifetime = SCTP_INFINITE_LIFETIME;

            cdata->chunk_len   = bCount + FIXED_DATA_CHUNK_SIZE;
            cdata->chunk_tsn = 0;
            cdata->gap_reports = 0;
            cdata->ack_time    = 0;
            cdata->context     = context;
            cdata->dontBundle  = dontBundle;
            cdata->num_of_transmissions = 0;
            cdata->hasBeenAcked = FALSE;
            cdata->hasBeenDropped = FALSE;
            cdata->initialDestinationKey = destAddressKey;
            /* check these two next statements..***************** */
            cdata->lastDestinationKey = 0;
            timerclear(&(cdata->transmission_time));
            /* up to here *************************************** */
            if (lifetime == SCTP_INFINITE_LIFETIME) {
                timerclear(&(cdata->expiry_time));
            } else if (lifetime == 0) {
                adl_gettime(&(cdata->expiry_time));
            } else {
                adl_gettime(&(cdata->expiry_time));
                adl_add_msecs_totime(&(cdata->expiry_time), lifetime);
            }
            mVec->cdat[i-1] = cdata;
            mVec->numberOfChunks++;
            mVec->totalLength += cdata->chunk_len;

            event_logiiii(EXTERNAL_EVENT, "======> SE queues fragment %d of chunk (SSN=%u, SID=%u, Len=%u)  <======",
                            i, ntohs(dchunk->stream_sn), ntohs(dchunk->stream_id), cdata->chunk_len);



        } /* END: for (i = 1; i <= numberOfSegments; i++) */
    }
    se_enqueue_message(se, mVec, streamId);
    /* result = fc_send_data_chunk (asok, cdata, destAddressKey, lifetime, dontBundle, context); */
    result = fc_send_all_you_can(asok, FALSE);
    return retVal;
}


/******************** Functions for Receiving Data ***********************************/

/**
 * This function is called from distribution layer, when the ULP wants to retrieve
 * normal data chunks from the queue
 */
int se_ulpreceive(Association* asok,
                  unsigned char *buffer, unsigned int *byteCount,
                  unsigned short streamId, unsigned short* streamSN,
                  unsigned int * tsn, unsigned int flags)
{

    delivery_pdu  *d_pdu = NULL;
    unsigned int copiedBytes, residual, i;
    guint32 r_pos, r_chunk, chunk_pos, oldQueueLen;

    StreamEngine* se = (StreamEngine *) mdi_readStreamEngine (asok);

    if (se == NULL) {
        error_log (ERROR_MAJOR, "Could not retrieve SE instance ");
        return SCTP_MODULE_NOT_FOUND;
    }
    if (buffer == NULL || byteCount == NULL) {
        error_log (ERROR_MAJOR, "Wrong Arguments : Pointers are NULL");
        return SCTP_PARAMETER_PROBLEM;
    }

    if (streamId >= se->numReceiveStreams) {
        error_log (ERROR_MINOR, "STREAM ID OVERFLOW");
        return (STREAM_ID_OVERFLOW);
    } else {
        event_logii (EXTERNAL_EVENT, "SE_ULPRECEIVE (sid: %u, numBytes: %u) CALLED",streamId,*byteCount);

        if (se->RecvStreams[streamId].pduList == NULL) {
            event_log (EXTERNAL_EVENT, "NO DATA AVAILABLE");
            return (NO_DATA_AVAILABLE);
        } else {
            oldQueueLen = se->queuedBytesRecvQueue;
            
            copiedBytes = 0;
            d_pdu = g_list_nth_data (se->RecvStreams[streamId].pduList, 0);

            r_pos       = d_pdu->read_position;
            r_chunk     = d_pdu->read_chunk;
            chunk_pos   = d_pdu->chunk_position;

            *streamSN   = d_pdu->ddata[d_pdu->read_chunk]->stream_sn;
            *tsn        = d_pdu->ddata[d_pdu->read_chunk]->tsn;

            event_logiiii (VVERBOSE, "SE_ULPRECEIVE (read_position: %u, read_chunk: %u, chunk_position: %u, total_length: %u)",
                    r_pos,  r_chunk, chunk_pos, d_pdu->total_length);

            if (d_pdu->total_length - d_pdu->read_position < *byteCount) {
                *byteCount = d_pdu->total_length-d_pdu->read_position;
            }

            residual = *byteCount;

            while (copiedBytes < *byteCount) {


                if (d_pdu->ddata[d_pdu->read_chunk]->data_length - d_pdu->chunk_position > residual) {
                    event_logiii (VVERBOSE, "Copy in SE_ULPRECEIVE (residual: %u, copied bytes: %u, byteCount: %u)",
                        residual, copiedBytes,*byteCount);

                    memcpy (&buffer[copiedBytes],
                            &(d_pdu->ddata[d_pdu->read_chunk]->data)[d_pdu->chunk_position],
                            residual);

                    d_pdu->chunk_position += residual;
                    d_pdu->read_position  += residual;
                    copiedBytes           += residual;
                    residual = 0;
                } else {
                    event_logi (VVERBOSE, "Copy in SE_ULPRECEIVE (num: %u)",d_pdu->ddata[d_pdu->read_chunk]->data_length - d_pdu->chunk_position);

                    memcpy (&buffer[copiedBytes],
                            &(d_pdu->ddata[d_pdu->read_chunk]->data)[d_pdu->chunk_position],
                            d_pdu->ddata[d_pdu->read_chunk]->data_length - d_pdu->chunk_position);

                    d_pdu->read_position += (d_pdu->ddata[d_pdu->read_chunk]->data_length - d_pdu->chunk_position);
                    copiedBytes          += (d_pdu->ddata[d_pdu->read_chunk]->data_length - d_pdu->chunk_position);
                    residual             -= (d_pdu->ddata[d_pdu->read_chunk]->data_length - d_pdu->chunk_position);
                    d_pdu->chunk_position = 0;
                    d_pdu->read_chunk++;
                }
            }

            if (flags == SCTP_MSG_PEEK) {
                d_pdu->chunk_position   = chunk_pos;
                d_pdu->read_position    = r_pos;
                d_pdu->read_chunk       = r_chunk;
            } else {

               if (d_pdu->read_position >= d_pdu->total_length) {

                    se->queuedBytesRecvQueue -= d_pdu->total_length;

                    se->RecvStreams[streamId].pduList =
                        g_list_remove (se->RecvStreams[streamId].pduList,
                                       g_list_nth_data (se->RecvStreams[streamId].pduList, 0));
                    event_log (VERBOSE, "Remove PDU element from the SE list, and free associated memory");
                    for (i=0; i < d_pdu->number_of_chunks; i++) free(d_pdu->ddata[i]);
                    free(d_pdu->ddata);
                    free(d_pdu);
                    rxc_start_sack_timer(asok, oldQueueLen);
                }
            }

        }

    }
    event_logi (EXTERNAL_EVENT, "ulp receives %u bytes from se", *byteCount);
    return (RECEIVE_DATA);
}

/*
 * function that gets chunks from the Lists, transforms them to PDUs, puts them
 * to the pduList, and calls DataArrive-Notification
 */
int se_doNotifications(Association* asok)
{
    int retVal;
    unsigned int i;
    
    StreamEngine* se = (StreamEngine *) mdi_readStreamEngine (asok);

    if (se == NULL) {
        error_log (ERROR_MAJOR, "Could not retrieve SE instance ");
        return SCTP_MODULE_NOT_FOUND;
    }

    event_log (INTERNAL_EVENT_0, " ================> se_doNotifications <=============== ");

    retVal = SCTP_SUCCESS;
    retVal = se_searchReadyPdu(se);

    for (i = 0; i < se->numReceiveStreams; i++)
    {
        if(se->RecvStreams[i].prePduList != NULL)
        {
            retVal = se_deliverWaiting(se, i);
        }
    }
    event_log (INTERNAL_EVENT_0, " ================> se_doNotifications: DONE <=============== ");
    return retVal;
}


/*
 * This function is called from Receive Control to forward received chunks to Stream Engine.
 * returns an error chunk to the peer, when the maximum stream id is exceeded !
 */
int se_recvDataChunk (Association* asok, SCTP_data_chunk * dataChunk, unsigned int byteCount)
{
    guint16 datalength;
    SCTP_InvalidStreamIdError error_info;
    delivery_data* d_chunk;
    StreamEngine* se = (StreamEngine *) mdi_readStreamEngine (asok);
    assert(se);

    event_log (INTERNAL_EVENT_0, "SE_RECVDATACHUNK CALLED");

    d_chunk = malloc (sizeof (delivery_data));
    if (d_chunk == NULL) return SCTP_OUT_OF_RESOURCES;

    datalength =  byteCount - FIXED_DATA_CHUNK_SIZE;
    d_chunk->stream_id =    ntohs (dataChunk->stream_id);

    if (d_chunk->stream_id >= se->numReceiveStreams) {
        /* return error, when numReceiveStreams is exceeded */
        error_info.stream_id = htons(d_chunk->stream_id);
        error_info.reserved = htons(0);

        scu_abort(asok, ECC_INVALID_STREAM_ID, sizeof(error_info), (void*)&error_info);
        free(d_chunk);
        return SCTP_UNSPECIFIED_ERROR;
    }

    d_chunk->tsn = ntohl (dataChunk->tsn);     /* for efficiency */

    if (datalength <= 0) {
        scu_abort(asok, ECC_NO_USER_DATA, sizeof(unsigned int), (void*)&(dataChunk->tsn));
        free(d_chunk);
        return SCTP_UNSPECIFIED_ERROR;
    }

    memcpy (d_chunk->data, dataChunk->data, datalength);
    d_chunk->data_length = datalength;
    d_chunk->chunk_flags = dataChunk->chunk_flags;
    d_chunk->stream_sn =    ntohs (dataChunk->stream_sn);
    d_chunk->protocolId =   ntohl (dataChunk->protocolId);


    se->List = g_list_insert_sorted(se->List, d_chunk, (GCompareFunc) sort_tsn_se);

    se->queuedBytesRecvQueue += datalength;
    se->recvStreamActivated[d_chunk->stream_id] = TRUE;
    return 0;
}


int se_searchReadyPdu(StreamEngine* se)
{
    GList* tmp = g_list_first(se->List);
    GList* firstItem = NULL;
    delivery_data* d_chunk;
    delivery_pdu* d_pdu;
    guint32 firstTSN = 0;
    guint16 currentSID = 0;
    guint16 currentSSN = 0;
    guint16 nrOfChunks = 0;
    gboolean complete = FALSE;
    gboolean unordered = FALSE;
    int i = 0;
    guint32 itemPosition = 0;
    event_log (INTERNAL_EVENT_0, " ================> se_searchReadyPdu <=============== ");
    event_logi (VVERBOSE, "List has %u elements", g_list_length(se->List));
    for (i = 0; i < se->numReceiveStreams; i++) {
        se->RecvStreams[i].highestSSN = 0;
        se->RecvStreams[i].highestSSNused = FALSE; 
    }

    while(tmp != NULL)
    {
        d_chunk = (delivery_data*)(tmp->data);
        event_logiii(VVERBOSE, "Handling Packet with TSN: %u, SSN: %u, SID: %u", d_chunk->tsn, d_chunk->stream_sn, d_chunk->stream_id);

        currentSID = d_chunk->stream_id;
        currentSSN = d_chunk->stream_sn;
        unordered = (d_chunk->chunk_flags & SCTP_DATA_UNORDERED);

        if((se->RecvStreams[currentSID].highestSSNused) && (sAfter(se->RecvStreams[currentSID].highestSSN, currentSSN)))
        {
            error_logi(VERBOSE, "Wrong ssn and tsn order", d_chunk->stream_sn);
            scu_abort(se->myAssociation, ECC_PROTOCOL_VIOLATION, 0, NULL);
            return SCTP_UNSPECIFIED_ERROR;
        }
        if(!unordered)
        {
            se->RecvStreams[currentSID].highestSSN = currentSSN;
            se->RecvStreams[currentSID].highestSSNused = TRUE;
        }

                
        if(d_chunk->chunk_flags & SCTP_DATA_BEGIN_SEGMENT)
        {
            event_log (VVERBOSE, "Found Begin Segment");

            nrOfChunks = 1;
            firstItem = tmp;
            firstTSN = d_chunk->tsn;
            
            if((sBefore(currentSSN, se->RecvStreams[currentSID].nextSSN)) || (currentSSN == se->RecvStreams[currentSID].nextSSN) || (d_chunk->chunk_flags & SCTP_DATA_UNORDERED))
            {
                
                if(d_chunk->chunk_flags & SCTP_DATA_END_SEGMENT)
                {
                    event_log (VVERBOSE, "Complete PDU found");
                    complete = TRUE;
                }

                        
                while((tmp != NULL) && (!complete))
                {
                    nrOfChunks++;
                    event_logi (VVERBOSE, "Handling chunk nr: %u", nrOfChunks);
                    
                    tmp = g_list_next(tmp);
                    if(tmp == NULL) break;
                    d_chunk = (delivery_data*)(tmp->data);
                    event_logiii(VVERBOSE, "Handling Packet with TSN: %u, SSN: %u, SID: %u", d_chunk->tsn, d_chunk->stream_sn, d_chunk->stream_id);
                    
                    
                    if((d_chunk->stream_id == currentSID)
                        && ((d_chunk->stream_sn == currentSSN) || unordered)
                        && (firstTSN + nrOfChunks - 1 == d_chunk->tsn))
                    {
                        if(d_chunk->chunk_flags & SCTP_DATA_BEGIN_SEGMENT)
                        {
                            error_logi(VERBOSE, "Multiple Begins found with SSN: %u", d_chunk->stream_sn);
                            scu_abort(se->myAssociation, ECC_PROTOCOL_VIOLATION, 0, NULL);
                            return SCTP_UNSPECIFIED_ERROR;
                        }
                        else if((d_chunk->chunk_flags & SCTP_DATA_UNORDERED) != unordered)
                        {
                            error_logi(VERBOSE, "Mix Ordered and unordered Segments found with SSN: %u", d_chunk->stream_sn);
                            scu_abort(se->myAssociation, ECC_PROTOCOL_VIOLATION, 0, NULL);
                            return SCTP_UNSPECIFIED_ERROR;
                        }
                        else if(d_chunk->chunk_flags & SCTP_DATA_END_SEGMENT)
                        {
                            event_log (VVERBOSE, "Complete PDU found");
                            complete = TRUE;
                        }
                    }
                    else
                    {
                        if(firstTSN + nrOfChunks - 1 == d_chunk->tsn)
                        {
                            error_logi(VERBOSE, "Data without end segment found", d_chunk->stream_sn);
                            scu_abort(se->myAssociation, ECC_PROTOCOL_VIOLATION, 0, NULL);
                            return SCTP_UNSPECIFIED_ERROR;
                        }
                        event_log (VVERBOSE, "Abort current ssn search - Incomplete!");    
                        break;
                    }
                    
                     
                }
                if(complete)
                {
                      event_log (VVERBOSE, "handling complete PDU");

                      d_pdu = malloc(sizeof(delivery_pdu));
                      if (d_pdu == NULL) {
                          return SCTP_OUT_OF_RESOURCES;
                      }
                      d_pdu->number_of_chunks = nrOfChunks;
                      d_pdu->read_position = 0;

                      d_pdu->read_chunk = 0;
                      d_pdu->chunk_position = 0;
                      d_pdu->total_length = 0;

                      d_pdu->ddata = malloc(nrOfChunks*sizeof(delivery_data*));
                      if (d_pdu->ddata == NULL) {
                          free(d_pdu);
                          return SCTP_OUT_OF_RESOURCES;
                      }

                      tmp = firstItem;
                      itemPosition = g_list_position(se->List, tmp);

                      /* get pointers to the first chunks and put them into the pduList */
                      for (i = 0; i < nrOfChunks; ++i) {
                          d_pdu->ddata[i] = (delivery_data*)(tmp->data);
                          d_pdu->total_length += d_pdu->ddata[i]->data_length;
                          tmp = g_list_next(tmp);
                      }
                      if(!unordered && (se->RecvStreams[d_pdu[0].ddata[0]->stream_id].nextSSN == currentSSN))
                           se->RecvStreams[d_pdu[0].ddata[0]->stream_id].nextSSN++;

                      se->RecvStreams[d_pdu[0].ddata[0]->stream_id].prePduList = g_list_append(se->RecvStreams[d_pdu[0].ddata[0]->stream_id].prePduList, d_pdu);
                      /* remove chunks from the list and return */
                      for (i = 1; i <= nrOfChunks; i++)
                      {
                           se->List = g_list_remove(se->List, g_list_nth_data(se->List, itemPosition ));
                           event_logiii(VERBOSE, "Removing chunk nr: %u(%u) list size after remove: %u", i,itemPosition, g_list_length(se->List));
                    
                      }
                      tmp = g_list_nth(se->List, itemPosition);
                      nrOfChunks = 0;
                      firstItem = NULL;
                      firstTSN = 0;
                      currentSID = 0;
                      currentSSN = 0;
                      complete = FALSE;
                      continue;
                }
                else
                {
                    nrOfChunks = 0;
                    firstItem = NULL;
                    firstTSN = 0;
                    currentSID = 0;
                    currentSSN = 0;
                    continue;
                }
            }
            else
            {
                event_log (VVERBOSE, "No begin chunk!");
                            
                nrOfChunks = 0;
                firstItem = NULL;
                firstTSN = 0;
                currentSID = 0;
                currentSSN = 0;
            }
                
        }
        if(tmp != NULL)
            tmp = g_list_next(tmp);
    }
    event_log (INTERNAL_EVENT_0, " ================> se_searchReadyPdu Finished <=============== ");

    return SCTP_SUCCESS;
}


int se_deliverWaiting(StreamEngine* se, unsigned short sid)
{
    GList* waitingListItem = g_list_first(se->RecvStreams[sid].prePduList);
    delivery_pdu* d_pdu;
    
    while(waitingListItem != NULL)
    {
        d_pdu = (delivery_pdu*)waitingListItem->data;   
        se->RecvStreams[sid].pduList = g_list_append(se->RecvStreams[sid].pduList, d_pdu);
        mdi_dataArriveNotif(se->myAssociation, sid, d_pdu->total_length, d_pdu->ddata[0]->stream_sn, d_pdu->ddata[0]->tsn,
                                d_pdu->ddata[0]->protocolId, (d_pdu->ddata[0]->chunk_flags & SCTP_DATA_UNORDERED) ? 1 : 0);        
        if(waitingListItem != NULL)
            waitingListItem = g_list_next(waitingListItem);
    }
    g_list_free(se->RecvStreams[sid].prePduList);
    se->RecvStreams[sid].prePduList = NULL;
    return SCTP_SUCCESS;
}



/**
 * function to return the number of chunks that can be retrieved
 * by the ULP - this function may need to be refined !!!!!!
 */
guint32 se_numOfQueuedChunks (Association* asok)
{

    guint32 i, num_of_chunks = 0;
    StreamEngine* se = (StreamEngine *) mdi_readStreamEngine (asok);

    if (se == NULL) {
        error_log (ERROR_MAJOR, "Could not read StreamEngine Instance !");
        return 0;
    }

    for (i = 0; i < se->numReceiveStreams; i++) {

        /* Add number of all chunks (i.e. lengths of all pduList lists of all streams */
        num_of_chunks += g_list_length (se->RecvStreams[i].pduList);
    }
    return num_of_chunks;
}



/**
 * function to return the number of streams that we may
 * send on
 */
inline guint16 se_numOfSendStreams (Association* asok)
{
    StreamEngine* se = (StreamEngine *) mdi_readStreamEngine(asok);
    assert(se);
    return (guint16) (se->numSendStreams);
}

/**


 * function to return the number of streams that we are allowed to
 * receive data on
 */
inline guint16 se_numOfRecvStreams (Association* asok)
{
    StreamEngine* se = (StreamEngine *) mdi_readStreamEngine(asok);
    assert(se);
    return (guint16) (se->numReceiveStreams);
}


int se_deliver_unreliably(Association* asok, unsigned int up_to_tsn, SCTP_forward_tsn_chunk* chk)
{
    int i;
    int numOfSkippedStreams;
    unsigned short skippedStream, skippedSSN;
    pr_stream_data* psd;
    GList* tmp;
    delivery_data  *d_chunk = NULL;

    StreamEngine* se = (StreamEngine *) mdi_readStreamEngine(asok);
    if (se == NULL) {
        error_log (ERROR_MAJOR, "Could not read StreamEngine Instance !");
        return SCTP_MODULE_NOT_FOUND;
    }

    numOfSkippedStreams = (ntohs(chk->chunk_header.chunk_length) -
                          sizeof(unsigned int) - sizeof(SCTP_chunk_header)) / sizeof(pr_stream_data);

    if (se->unreliable == TRUE) {
        /* TODO: optimization !!!! loop through all streams */
        for (i = 0; i < numOfSkippedStreams; i++)
        {
            psd = (pr_stream_data*) &chk->variableParams[sizeof(pr_stream_data)*i];
            skippedStream = ntohs(psd->stream_id);
            skippedSSN = ntohs(psd->stream_sn);
            event_logiii (VERBOSE, "delivering dangling messages in stream %d for forward_tsn=%u, SSN=%u",
                        skippedStream, up_to_tsn, skippedSSN);
            /* if unreliable, check if messages can be  delivered */
            se->RecvStreams[skippedStream].nextSSN = skippedSSN + 1;
        }
        se_doNotifications(asok);

        tmp = g_list_first(se->List);
        while((tmp != NULL) && (((delivery_data*)(tmp->data))->tsn <= up_to_tsn))
        {
             d_chunk = (delivery_data*)(tmp->data);
      
             g_list_remove (se->List, d_chunk);
             free(d_chunk);
             tmp = g_list_first(se->List);
        }
    }
    return SCTP_SUCCESS;
}


int se_getQueuedBytesRecvQueue(Association* asok)
{
    StreamEngine* se = (StreamEngine *) mdi_readStreamEngine(asok);
    if (se == NULL) {
        error_log (ERROR_MAJOR, "Could not read StreamEngine Instance !");
        return SCTP_MODULE_NOT_FOUND;
    }
    return (int)se->queuedBytesRecvQueue;
}

int se_getQueuedBytesSendQueue(Association* asok)
{
    StreamEngine* se = (StreamEngine *) mdi_readStreamEngine(asok);
    if (se == NULL) {
        error_log (ERROR_MAJOR, "Could not read StreamEngine Instance !");
        return SCTP_MODULE_NOT_FOUND;
    }
    return (int)se->queuedBytesSendQueue;
}

int se_getQueuedChunksSendQueue(Association* asok)
{
    StreamEngine* se = (StreamEngine *) mdi_readStreamEngine(asok);
    if (se == NULL) {
        error_log (ERROR_MAJOR, "Could not read StreamEngine Instance !");
        return SCTP_MODULE_NOT_FOUND;
    }
    return (int)se->queuedChunksSendQueue;
}


/**
 * this function should be called to signal to stream engine that our ULP
 * has initiated a shutdown procedure. We must not accept new data for
 * sending from now on ! The association is about to terminate !
 */
int se_shutdown(Association* asok)
{
    StreamEngine* se = (StreamEngine *) mdi_readStreamEngine(asok);
    if (se == NULL) {
        error_log (ERROR_MAJOR, "Could not read StreamEngine Instance !");
        return SCTP_MODULE_NOT_FOUND;
    }
    se->shutdown_received = TRUE;
    return SCTP_SUCCESS;
}

