/*
 *  $Id: pathmanagement.c,v 1.16 2004/01/13 16:13:14 ajung Exp $
 *
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * Acknowledgement
 * This work was partially funded by the Bundesministerium fr Bildung und
 * Forschung (BMBF) of the Federal Republic of Germany (Frderkennzeichen 01AK045).
 * The authors alone are responsible for the contents.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * Purpose: This module does faultmanagement for paths.
 *
 * function prefix: pm_
 * TODO : make sure that when an address is removed, all HB timers are stopped, too,
 *        since these may reference the structures when callback is executed...
 */

#include <assert.h>
#include <stdio.h>

#include "globals.h"
#include "adaptation.h"
#include "chunkHandler.h"
#include "sctp-control.h"
#include "bundling.h"
#include "pathmanagement.h"


/*------------------------ defines -----------------------------------------------------------*/
#define RTO_ALPHA            0.125
#define RTO_BETA              0.25

/*----------------------- Typedefs ------------------------------------------------------------*/

/**
 * this struct contains the necessary data per (destination or) path.
 * There may be more than one within an association
 */
typedef struct pm_PATHDATA
{
    /* @{ */
    /** operational state of pathmanagement for one path */
    int state;
    /** ==1 if path has been confirmed, else == 0*/
    int confirmationState;
    /** true if heartbeat is enabled */
    boolean heartbeatEnabled;
    /** true as long as RTO-Calc. has been done */
    boolean firstRTO;
    /** Only once per HB-intervall */
    boolean timerBackoff;
    /** set to true when data chunks are acknowledged */
    boolean chunksAcked;
    /** TRUE, if chunks have been sent over that path within last RTO */
    boolean chunksSent;
    /** set to true when a heartbeat is sent. */
    boolean heartbeatSent;
    /** set to true when a hearbeat is acknowledged and to false when a
       heartbeat is sent when the heartbeat timer expires. */
    boolean heartbeatAcked;
    /** Counter for retransmissions on a single path */
    unsigned int pathRetranscount;
    /** Retransmission time out used for all retrans. timers */
    unsigned int rto;
    /** smoothed round trip time */
    unsigned int srtt;
    /** round trip time variation */
    unsigned int rttvar;
    /** defines the rate at which heartbeats are sent */
    unsigned int heartbeatIntervall;
    /** ID of the heartbeat timer */
    TimerID hearbeatTimer;
    /** time of last rto update */
    struct timeval rto_update;
    /** ID of path */
    unsigned int pathKey;
    /** pointer to ID of path - for timer callbacks */
    unsigned int* pathKeyPtr;
    /** Path MTU */
    unsigned int pathSCTPMTU;
    /** DO PMTUD immediately */
    gboolean doPMTUD;
    /** */
    unsigned int ipHeaderLength;
#ifdef DO_PMTUD
    /** Path MTU DiscoveryTimer */
    TimerID pathMTUDTimer;
    /** Path MTU DiscoveryIncreaseTimer */
    TimerID pathMTUDIncreaseTimer;
#endif /* DO_PMTUD */
    /* @} */
} pm_pathData;


/**
 * this struct contains all necessary data for one instance of the path management
 * module. There is one such module per existing association.
 */
typedef struct pm_PATHMANDATA
{
    /* @{ */
    /** stores the current primary path */
    unsigned int primaryPathKey;
    /** stores the current primary path index into the array */
    int primaryPathIndex;
    /** the number of paths used by this assoc. */
    int numberOfPaths;
    /** Counter for all retransmissions over all paths */
    unsigned int peerRetranscount;
    /** pointer to path-specific data */
    GArray*  pathData;
    /** association-ID */
    Association* myAsok;
    /** maximum retransmissions per path parameter */
    int maxPathRetransmissions;
    /** initial RTO, a configurable parameter */
    int rto_initial;
    /** minimum RTO, a configurable parameter */
    int rto_min;
    /** maximum RTO, a configurable parameter */
    int rto_max;
    /* change by srohde in progress */
    /** MTU for all paths */
    int mtu;
    /* change by srohde in progress */
    /* @} */
} pm_pathMan;


/*----------------------Declares ----------------------------------------------------------------*/

static gboolean handleChunksRetransmitted(Association* asok, pm_pathMan* pm, int pathIndex, unsigned int pathKey);

/*-------------------------- Function Implementations -------------------------------------------*/

/*------------------- Functions called to create, init and delete pathman-instances --------------*/



/**
 * pm_newPathman creates a new instance of path-management. Path management does heartbeating,
 * and path monitoring as well as RTO management
 * @param asok               pointer to the new association
 * @param numberOfPaths      number of paths of the association
 * @param primaryPathKey     initial primary path key
 * @return pointer to the newly created path-management instance
 */
void *pm_newPathman(Association* asok, int numberOfPaths, unsigned int primaryPathKey)
{
    pm_pathMan *pmData;

    pmData = (pm_pathMan *) malloc(sizeof(pm_pathMan));
    if (!pmData) {
        error_log(ERROR_FATAL, "pm_newPathman: out of memory");
        return NULL;
    }
    pmData->pathData = NULL;

    pmData->primaryPathKey   = primaryPathKey;
    pmData->primaryPathIndex = mdi_getIndexForKey(asok, primaryPathKey);
    pmData->numberOfPaths    = numberOfPaths;
    pmData->myAsok           = asok;
    pmData->maxPathRetransmissions = mdi_getDefaultPathMaxRetransmits(asok->sctpInstance);
    pmData->rto_initial = mdi_getDefaultRtoInitial(asok->sctpInstance);
    pmData->rto_min     = mdi_getDefaultRtoMin(asok->sctpInstance);
    pmData->rto_max     = mdi_getDefaultRtoMax(asok->sctpInstance);
    /* change by srohde in progress */
    pmData->mtu         = mdi_getCurrentMinAdapterMTU() - IP_HEADERLENGTH;
    /* change by srohde in progress */
    event_logii(VERBOSE, "pm_newPathman: created new pathMan module with %d adresses, primary: %u",
                    numberOfPaths, primaryPathKey);
    return pmData;
}                               /* end: pm_newPathman */


/**
 * Deletes the instance pointed to by pathmanPtr.
 * @param   pathmanPtr pointer to the instance that is to be deleted
 */
void pm_deletePathman(Association* asok)
{
    int i = 0;
    pm_pathMan *pm = (pm_pathMan *) asok->pathMan;
    pm_pathData* pd = NULL;

    event_log(INTERNAL_EVENT_0, "deleting pathmanagement");

    if (pm != NULL && pm->pathData != NULL) {
        for (i = 0; i < pm->numberOfPaths; i++) {
            pd = &g_array_index(pm->pathData, pm_pathData, i);
            free(pd->pathKeyPtr);
            if (pd->heartbeatEnabled ==TRUE || pd->hearbeatTimer != 0) {
                adl_stopTimer(pd->hearbeatTimer);
            }
#ifdef DO_PMTUD
            adl_stopTimer(pd->pathMTUDTimer);
            if(pd->pathMTUDIncreaseTimer != 0)
                adl_stopTimer(pd->pathMTUDIncreaseTimer);
#endif /* DO_PMTUD */
        }
    }
    event_log(VVERBOSE, "stopped timers");
    if (pm->pathData) {
        g_array_free(pm->pathData, TRUE);
    }
    if (pm) free(pm);
}                               /* end: pm_deletePathman */


/**
  pm_heartbeatTimer is called when the heartbeat timer expires.
  It may set the path to inactive, or restart timer, or even cause COMM LOST
  As all timer callbacks, it takes three arguments  (two pointers to necessary data)
  @param timerID  ID of the HB timer that expired.
  @param associationIDvoid  pointer to the association-ID
  @param pathIDvoid         pointer to the path-ID
*/
void pm_heartbeatTimer(TimerID timerID, void *assocVoid, void *pathKeyVoid)
{
    int pathIndex;

    ChunkID heartbeatCID;
    gboolean removed_association = FALSE;
    Association* asok = (Association*)assocVoid;
    unsigned int pKey = *((unsigned int*)pathKeyVoid);
    pm_pathMan*  pm = NULL;
    pm_pathData* pd = NULL;

    pm = (pm_pathMan *) mdi_readPathMan(asok);
    assert(pm);
    event_logi(INTERNAL_EVENT_0, "Heartbeat timer expired for path %u", pKey);

    pathIndex = mdi_getIndexForKey(asok, pKey);
    pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
    assert(pd);

    if (pd->heartbeatSent && !pd->heartbeatAcked) {
        /* Heartbeat has been sent and not acknowledged: handle as retransmission */

        if (pd->state == PM_ACTIVE) {
            /* Handling of unacked heartbeats is the same as that of unacked data chunks.
               The state after calling pm_chunksRetransmitted may have changed to inactive. */
           removed_association = handleChunksRetransmitted(asok, pm, pathIndex, pd->pathKey);
           if (removed_association) {
                event_logi(INTERNAL_EVENT_0, "Association was removed by handleChunksRetransmitted(PathIndex %u)!!!!",pathIndex);
           }
        } /* else if (pd->state == PM_INACTIVE) { */
          /* path already inactive, dont increase counter etc. */
          /*  ; */
        /* }    */

        if (!removed_association) {
            if (!pd->timerBackoff) {
                /* Timer backoff */
                pd->rto = min(2 * pd->rto, pm->rto_max);
                event_logiii(VERBOSE, "Backing off timer : Path Index %d Key %u, RTO= %u", pathIndex, pd->pathKey, pd->rto);
            }
        }
    }

    if (!removed_association && !pd->chunksAcked && pd->heartbeatEnabled && !pd->chunksSent) {
        /* Remark: If commLost is detected in handleChunksRetransmitted, the current association
           is marked for deletetion. Doing so, all timers are stopped. The HB-timers are
           stopped by calling pm_disableHB in mdi_deleteCurrentAssociation. This is why
           heartBeatEnabled is checked above.
         */
        /* send heartbeat if no chunks have been acked in the last HB-intervall (path is idle). */
        event_log(VERBOSE, "--------------> Sending HB <---------------------");
        heartbeatCID = ch_makeHeartbeat(pm_getTime(), pd->pathKey, asok);
        bu_put_Ctrl_Chunk(asok, ch_chunkString(heartbeatCID), pd->pathKey);
        bu_sendAllChunks(asok, pd->pathKey);
        ch_deleteChunk(heartbeatCID);
        pd->heartbeatSent = TRUE;
    } else if (!removed_association) {
        pd->heartbeatSent = FALSE;
    }

    if (!removed_association && pd->heartbeatEnabled) {
        /* heartbeat could have been disabled when the association went down after commLost
           detected in handleChunksRetransmitted */
        pd->hearbeatTimer = adl_startTimer(pd->heartbeatIntervall + pd->rto,
                                           &pm_heartbeatTimer,
                                           TIMER_TYPE_HEARTBEAT,
                                           asok,
                                           pathKeyVoid);

        /* reset this flag, so we can check, whether the path was idle */
        pd->chunksSent = FALSE;

        event_logiii(INTERNAL_EVENT_0,
                    "Heartbeat timer started with %u msecs for path index %u, RTO=%u msecs",
                    (pd->heartbeatIntervall + pd->rto), pathIndex, pd->rto);
    }

    if (!removed_association) {
        pd->heartbeatAcked  = FALSE;
        pd->timerBackoff    = FALSE;
        pd->chunksAcked     = FALSE;
    }
}                               /* end: pm_heartbeatTimer */

#ifdef DO_PMTUD
void pm_PMTUDTimer(TimerID timerID, void *assocVoid, void *pathKeyVoid)
{
    int pathIndex;

    Association* asok = (Association*)assocVoid;
    unsigned int pKey = *((unsigned int*)pathKeyVoid);
    pm_pathMan*  pm = NULL;
    pm_pathData* pd = NULL;


    pm = (pm_pathMan *) mdi_readPathMan(asok);
    assert(pm);
    event_logi(VERBOSE, "PMTUD timer expired for path %u", pKey);

    pathIndex = mdi_getIndexForKey(asok, pKey);
    pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
    assert(pd);
    pd->doPMTUD = TRUE;
}
#endif /* DO_PMTUD */

#ifdef DO_PMTUD
void pm_PMTUDIncreaseTimer(TimerID timerID, void *assocVoid, void *pathKeyVoid)
{
    int pathIndex;
    int i = 0;
    Association* asok = (Association*)assocVoid;
    unsigned int pKey = *((unsigned int*)pathKeyVoid);
    pm_pathMan*  pm = NULL;
    pm_pathData* pd = NULL;
    pm_pathData* currentPathData = NULL;

    pm = (pm_pathMan *) mdi_readPathMan(asok);
    assert(pm);
    event_logi(VERBOSE, "PMTUD increase timer expired for path %u", pKey);

    pathIndex = mdi_getIndexForKey(asok, pKey);
    pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
    assert(pd);

    pm->mtu = mdi_getCurrentMinAdapterMTU() - IP_HEADERLENGTH;
    pd->pathSCTPMTU = mdi_getCurrentMinAdapterMTU() - IP_HEADERLENGTH - sizeof(SCTP_common_header);
    for (i = 0; i < pm->numberOfPaths; i++)
    {
        currentPathData = &g_array_index(pm->pathData, pm_pathData, i);
        if(pm->mtu > currentPathData->pathSCTPMTU + sizeof(SCTP_common_header))
        {
            pm->mtu = currentPathData->pathSCTPMTU + sizeof(SCTP_common_header);
        }
    }
}
#endif /* DO_PMTUD */


#ifdef DO_PMTUD
gboolean pm_pathMTUDMustBeDone(Association *asok, int pathKey)
{   
    pm_pathMan*  pm = NULL;
    pm_pathData* pd = NULL;
    int pathIndex;

    if(asok == NULL)
      return FALSE;

    pm = (pm_pathMan *) mdi_readPathMan(asok);
    assert(pm);

    pathIndex = mdi_getIndexForKey(asok, pathKey);
    if(pm->pathData != 0)
    {
      pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
      assert(pd);
      return pd->doPMTUD;
    }
    else
      return FALSE;
} 
#endif


void pm_changeMTU(Association* asoc, unsigned int key, unsigned int newMTU)
{
    pm_pathMan*  pm = NULL;
    pm_pathData* pd = NULL;
    int pathIndex;

    pm = (pm_pathMan *) mdi_readPathMan(asoc);
    assert(pm);

    pathIndex = mdi_getIndexForKey(asoc, key);
    pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
    assert(pd);
    
    if(pd->pathSCTPMTU + pd->ipHeaderLength + sizeof(SCTP_common_header) > newMTU)
    {
#ifdef DO_PMTUD
        if(pd->pathMTUDIncreaseTimer != 0)
        {
            sctp_stopTimer(pd->pathMTUDIncreaseTimer);
            pd->pathMTUDIncreaseTimer = 0;
        }
        pd->pathMTUDIncreaseTimer = adl_startTimer(PM_PMTUD_INCREASE_INTERVAL,
                                              &pm_PMTUDIncreaseTimer,
                                              TIMER_TYPE_PMTUD_INCREASE,
                                              asoc,
                                              (void *) pd->pathKeyPtr);
#endif /* DO_PMTUD */
        pd->pathSCTPMTU = newMTU - pd->ipHeaderLength - sizeof(SCTP_common_header);       
        if(pm->mtu + pd->ipHeaderLength > newMTU)
        {
            pm->mtu = newMTU - pd->ipHeaderLength;
        }
    }
}


#ifdef DO_PMTUD
void pm_pathMTUDWasDone(Association *asok, int pathKey)
{ 
    pm_pathMan*  pm = NULL;
    pm_pathData* pd = NULL;
    int pathIndex;

    pm = (pm_pathMan *) mdi_readPathMan(asok);
    assert(pm);

    pathIndex = mdi_getIndexForKey(asok, pathKey);
    pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
    assert(pd);
    pd->doPMTUD = FALSE;
    event_logi(VERBOSE, "PMTUD timer restarted for path %u", pathKey);

    pd->pathMTUDTimer = adl_startTimer(PM_PMTUD_INTERVAL,
                                              &pm_PMTUDTimer,
                                              TIMER_TYPE_PMTUD,
                                              asok,
                                              (void *) pd->pathKeyPtr);
}
#endif


#ifdef DO_PMTUD

void pm_adjustPMTU(union sockunion * address, unsigned short remotePort, unsigned short localPort, unsigned short newMTU)
{
    Association *asoc = mdi_retrieveAssociationByTransportAddress(address, remotePort, localPort);
    unsigned int key = mdi_getKeyForAddress(asoc, address);

    pm_changeMTU(asoc, key, newMTU);
}
#endif /* DO_PMTUD */


/**
 * pm_setPaths modufies number of paths and sets the primary path.
 * This is required for association setup, where the local ULP provides
 * only one path and the peer may provide additional paths.
 * This function also initializes the path structures and starts the heartbeat timer for each
 * path. For this reason it is recommended to call this function when communication up is called.
 * @params asok             pointer to the current association
 * @params noOfPaths        number of paths to the destination endpoint
 * @param  primaryPathKey   key to the address that is to be used as primary address
 */
int pm_setPaths(Association* asok, int noOfPaths, unsigned int primaryPathKey)
{
    pm_pathMan *pm = NULL;
    pm_pathData pd;
    pm_pathData* pdPtr = NULL;
    union sockunion *addr_ptr;
    int i;

    pm = (pm_pathMan *) mdi_readPathMan(asok);

    if (pm == NULL) {
        error_log(ERROR_MAJOR, "pm_setPrimaryPath: mdi_readPathMan failed");
        return SCTP_MODULE_NOT_FOUND;
    }
    if (pm_checkValidPathKey(asok, primaryPathKey) == SCTP_SUCCESS) {

        pm->pathData = g_array_new(FALSE, TRUE, sizeof(pm_pathData));

        pm->primaryPathKey   = primaryPathKey;
        pm->primaryPathIndex = mdi_getIndexForKey(asok, primaryPathKey);
        pm->numberOfPaths    = noOfPaths;
        pm->peerRetranscount = 0;

        for (i = 0; i < noOfPaths; i++)
        {
            addr_ptr = &g_array_index(asok->destAddresses, union sockunion, i);
            switch (sockunion_family(addr_ptr)) {
            case AF_INET:
                pd.ipHeaderLength = 20;
            break;
#ifdef HAVE_IPV6
            case AF_INET6:
                pd.ipHeaderLength = 40;
                if(pm->mtu > mdi_getCurrentMinAdapterMTU() - pd.ipHeaderLength)
                    pm->mtu = mdi_getCurrentMinAdapterMTU() - pd.ipHeaderLength;
                break;
#endif
            default:
                error_logi(ERROR_MAJOR, "Address family %d not supported", sockunion_family(addr_ptr));
            break;
            } 
            pd.state            = PM_ACTIVE;
            pd.heartbeatEnabled = TRUE;
            pd.firstRTO         = TRUE;
            pd.pathRetranscount = 0;
            pd.rto              = pm->rto_initial;
            pd.srtt             = pm->rto_initial;
            pd.rttvar           = 0;

            if (i != pm->primaryPathIndex) {
                pd.confirmationState = PM_PATH_UNCONFIRMED;
            } else {
                pd.confirmationState = PM_PATH_CONFIRMED;
            }

            pd.heartbeatSent    = FALSE;
            pd.heartbeatAcked   = FALSE;
            pd.timerBackoff     = FALSE;
            pd.chunksAcked      = FALSE;
            pd.chunksSent       = FALSE;

            pd.heartbeatIntervall = PM_INITIAL_HB_INTERVAL;
            pd.hearbeatTimer    = 0;
            /* assign proper keys based on the indexes used in distribution */
            pd.pathKey          = mdi_getKeyForIndex(asok, i);
            pd.pathKeyPtr       = malloc(sizeof(unsigned int));
            pd.pathSCTPMTU      = mdi_getCurrentMinAdapterMTU() - pd.ipHeaderLength - sizeof(SCTP_common_header);
#ifdef DO_PMTUD
            pd.pathMTUDTimer  = 0;
            pd.pathMTUDIncreaseTimer = 0;
#endif /* DO_PMTUD */
            pd.doPMTUD          = FALSE;
            memcpy(pd.pathKeyPtr, &pd.pathKey, sizeof(unsigned int));

            event_logii(VERBOSE, "pm_setPaths: created path data structure for path (key=%u, index=%d)", pd.pathKey,i);

            /* after RTO we can do next RTO update */
            adl_gettime(&(pd.rto_update));
            adl_add_msecs_totime(&(pd.rto_update), pd.rto);
            pm->pathData = g_array_append_val(pm->pathData, pd);
            pdPtr = &g_array_index(pm->pathData, pm_pathData, i);
            event_logiii(VERBOSE, "pm_setPaths: test: key from pointer %x: (key=%u, state=%d)", pdPtr, pdPtr->pathKey, pdPtr->state);

        }

        for (i = 0; i < noOfPaths; i++) {
            pdPtr = &g_array_index(pm->pathData, pm_pathData, i);
            event_logiii(VERBOSE, "pm_setPaths: loop: pd pointer %x: key=%u, state=%d)", pdPtr, pdPtr->pathKey, pdPtr->state);

            if (i != pm->primaryPathIndex)
            {
                /* send HB fairly quickly to all unconfirmed paths */
                pdPtr->hearbeatTimer = adl_startTimer((i*pdPtr->rto + 100),
                                                  &pm_heartbeatTimer,
                                                  TIMER_TYPE_HEARTBEAT,
                                                  asok,
                                                  (void *) pdPtr->pathKeyPtr);
            } else {
                /* primary path is CONFIRMED : regular HB interval */
                pdPtr->hearbeatTimer = adl_startTimer(pdPtr->heartbeatIntervall+pdPtr->rto,
                                                  &pm_heartbeatTimer,
                                                  TIMER_TYPE_HEARTBEAT,
                                                  asok,
                                                  (void *) pdPtr->pathKeyPtr);
            }

#ifdef DO_PMTUD
            addr_ptr = &g_array_index(asok->destAddresses, union sockunion, i);
            switch (sockunion_family(addr_ptr)) {
            case AF_INET:
                pdPtr->pathMTUDTimer = adl_startTimer(PM_PMTUD_INTERVAL,
                                              &pm_PMTUDTimer,
                                              TIMER_TYPE_PMTUD,
                                              asok,
                                              (void *) pdPtr->pathKeyPtr);
                break;
            default:
                event_logi(VERBOSE, "Large Heartbeats disabled for ipv6! pathindex: %u", i);
                break;
            } 
            
#endif /* DO_PMTUD */
            
        }
        event_log(INTERNAL_EVENT_0, "pm_setPaths called : return SUCCESS");
        return SCTP_SUCCESS;
    } else {
        error_log(ERROR_MAJOR, "pm_setPaths: invalid path ID");
        return SCTP_PARAMETER_PROBLEM;
    }
}                               /* end: pm_setPaths */

/*------------------- Internal Functions --------------------------------------------------------*/

/**
  return the current system time converted to a value of  milliseconds.
  MSB of tv_sec field are removed in order
  to make representation in millisecs possible. This done by taking the remainder of
  a division by 1728000 = 20x24x60x60, restarting millisecs count every 20 days.
  @return unsigned 32 bit value representing system time in milliseconds. Hmmmh.
*/
unsigned int pm_getTime(void)
{
    unsigned int curTimeMilli;
    struct timeval curTime;

    adl_gettime(&curTime);

    /* modulo operation overlfows every 20 days */
    curTimeMilli = (curTime.tv_sec % 1728000) * 1000 + curTime.tv_usec / 1000;

    return curTimeMilli;
}                               /* end: pm_ sctp_getTime */


/**
 *  handleChunksRetransmitted is called whenever datachunks are retransmitted or a hearbeat-request
 *  has not been acknowledged within the current heartbeat-intervall. It increases path- and peer-
 *  retransmission counters and compares these counters to the corresonding thresholds.
 *  @param  pathID index to the path that CAUSED retransmission
 *  @return TRUE if association was deleted, FALSE if not
 */
static gboolean handleChunksRetransmitted(Association* asok, pm_pathMan* pm, int pathIndex, unsigned int pathKey)
{
    int pID;
    boolean allPathsInactive;
    pm_pathData *pd = NULL, *tmp_pd = NULL;

    assert(pm);
    assert(pm->pathData);

    pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
    assert(pd);

    event_logiii(INTERNAL_EVENT_0,
                 "handleChunksRetransmitted(%d) : path-rtx-count==%u, peer-rtx-count==%u",
                 pathIndex, pd->pathRetranscount, pm->peerRetranscount);

    if (pd->state == PM_ACTIVE) {
        pd->pathRetranscount++;
        pm->peerRetranscount++;
    } else {
        event_log(INTERNAL_EVENT_0,
                  "handleChunksRetransmitted: ignored, because already inactive");
        return FALSE;
    }

    if (pm->peerRetranscount >= sci_getMaxAssocRetransmissions(asok)) {
        mdi_deleteAssociation(asok);
        mdi_communicationLostNotif(asok, SCTP_COMM_LOST_EXCEEDED_RETRANSMISSIONS);
        event_log(INTERNAL_EVENT_0, "handleChunksRetransmitted: communication lost");
        return TRUE;
    }

    if (pd->pathRetranscount >= pm->maxPathRetransmissions) {
        /* Set state of this path to inactive and notify change of state to ULP */
        pd->state = PM_INACTIVE;
        event_logi(INTERNAL_EVENT_0, "handleChunksRetransmitted: path %d to INACTIVE ", pathIndex);
        /* check if an active path is left */
        allPathsInactive = TRUE;
        for (pID = 0; pID < pm->numberOfPaths; pID++) {
            tmp_pd = &g_array_index(pm->pathData, pm_pathData, pID);
            assert(tmp_pd);

            if (tmp_pd->state == PM_ACTIVE) {
                allPathsInactive = FALSE;
            }
        }

        if (allPathsInactive) {
            /* No active parts are left, communication lost to ULP */
            mdi_deleteAssociation(asok);
            mdi_communicationLostNotif(asok, SCTP_COMM_LOST_ENDPOINT_UNREACHABLE);

            event_log(INTERNAL_EVENT_0,
                      "handleChunksRetransmitted: communication lost (all paths are INACTIVE)");
            return TRUE;
        } else {
            mdi_networkStatusChangeNotif(asok, pathKey, PM_INACTIVE);
        }
    }

    return FALSE;
}                               /* end: handleChunksRetransmitted */


/**
 * Function is used to update RTT, SRTT, RTO values after chunks have been acked.
 * @param  pathIndex index of the path where data was acked
 * @param  newRTT    new RTT measured, when new data was acked, or zero if retransmitted
 *                   data was acked
 */
static void handleChunksAcked(pm_pathMan *pm, int pathIndex, unsigned int newRTT)
{
    pm_pathData* pd = NULL;

    assert(pm);
    assert(pm->pathData);

    pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);

    assert(pd);

    event_logii(INTERNAL_EVENT_0, "handleChunksAcked: pathIndex: %d, new RTT: %u msecs", pathIndex, newRTT);

    if (newRTT > 0) {
        /* RTO measurement done */
        /* calculate new RTO, SRTT and RTTVAR */
        if (pd->firstRTO) {
            pd->srtt    = newRTT;
            pd->rttvar  = max(newRTT / 2, GRANULARITY);     /* minimum rttvar is 1 msec !!! */
            pd->rto     = max(min(newRTT * 3, (unsigned int)pm->rto_max), (unsigned int)pm->rto_min);
            pd->firstRTO = FALSE;
        } else {
            pd->rttvar  = (unsigned int)(1. - RTO_BETA) * pd->rttvar + RTO_BETA * abs(pd->srtt - newRTT);
            pd->rttvar  = max(pd->rttvar, GRANULARITY);  /* minimum rttvar is 1 msec !!! */
            pd->srtt    = (unsigned int)(1. - RTO_ALPHA) * pd->srtt + RTO_ALPHA * newRTT;
            pd->rto     = pd->srtt + 4 * pd->rttvar;
            pd->rto     = max(min(pd->rto, (unsigned int)pm->rto_max), (unsigned int)pm->rto_min);
        }
        event_logiii(INTERNAL_EVENT_0,
                     "handleChunksAcked: RTO update done: RTTVAR: %u msecs, SRTT: %u msecs, RTO: %u msecs",
                     pd->rttvar, pd->srtt, pd->rto);
    } else {
        event_log(INTERNAL_EVENT_0, "handleChunksAcked: chunks acked without RTO-update");
    }
    /* reset counters */
    pd->pathRetranscount = 0;
    pm->peerRetranscount = 0;
}                               /* end: handleChunksAcked */



/*----------------- Functions to answer peer HB requests -----------------------------------------*/

/**
 * pm_heartbeat is called when a heartbeat was received from the peer.
 * This function just takes that chunk, and sends it back.
 * @param heartbeatChunk pointer to the heartbeat chunk
 * @param source_address address we received the HB chunk from (and where it is echoed)
 */
int pm_heartbeat(Association* asok, SCTP_heartbeat * heartbeatChunk, unsigned int addressKey)
{
    heartbeatChunk->chunk_header.chunk_id = CHUNK_HBACK;

    bu_put_Ctrl_Chunk(asok, (SCTP_simple_chunk *) heartbeatChunk, addressKey);

    bu_sendAllChunks(asok, addressKey);

    return SCTP_SUCCESS;
}                               /* end: pm_heartbeat */



/*------------------- Signals --------------------------------------------------------------------*/

/*------------------- Signals from the Unix-Interface --------------------------------------------*/


/**
 * simple function that sends a heartbeat chunk to the indicated address
 * @param  pathID index to the address, where HB is to be sent to
 */
int pm_doHB(Association* asok, unsigned int pathKey)
{
    ChunkID heartbeatCID;
    pm_pathMan*  pm = NULL;
    pm_pathData* pd = NULL;
    int pathIndex;

    pm = (pm_pathMan *) mdi_readPathMan(asok);

    if (pm == NULL) {
        error_log(ERROR_MAJOR, "pm_doHB: mdi_readPathMan failed");
        return SCTP_MODULE_NOT_FOUND;
    }
    if (!pm->pathData) {
        error_logi(ERROR_MAJOR, "pm_doHB(%u): Path Data Structures not initialized yet, returning !", pathKey);
        return SCTP_UNSPECIFIED_ERROR;
    }
    /* validity of path key has already been checked in distribution.c */
    pathIndex =  mdi_getIndexForKey(asok, pathKey);
    pd =  &g_array_index(pm->pathData, pm_pathData, pathIndex);
    assert(pd);

    heartbeatCID = ch_makeHeartbeat(pm_getTime(), pathKey, asok);
    bu_put_Ctrl_Chunk(asok, ch_chunkString(heartbeatCID), pathKey);
    bu_sendAllChunks(asok, pathKey);
    ch_deleteChunk(heartbeatCID);
    pd->heartbeatSent = TRUE;

    return SCTP_SUCCESS;
}


/**
 * pm_heartbeatAck is called when a heartbeat acknowledgement was received from the peer.
 * checks RTTs, normally resets error counters, may set path back to ACTIVE state
 * @param heartbeatChunk pointer to the received heartbeat ack chunk
 */
int pm_heartbeatAck(Association* asok, SCTP_heartbeat * heartbeatChunk)
{
    pm_pathMan*  pm = NULL;
    pm_pathData* pd = NULL;
    int roundtripTime;
    unsigned int sendingTime;
    unsigned int pathKey;
    gboolean hbSignatureOkay = FALSE;
    int pathIndex;
    ChunkID heartbeatCID;


    pm = (pm_pathMan *) mdi_readPathMan(asok);

    if (pm == NULL) {
        error_log(ERROR_MAJOR, "pm_heartbeatAck: mdi_readPathMan failed");
        return SCTP_MODULE_NOT_FOUND;
    }
    if (!pm->pathData) {
        error_log(ERROR_MAJOR, "pm_heartbeatAck: Path Data Structures not initialized yet, returning !");
        return SCTP_UNSPECIFIED_ERROR;
    }

    heartbeatCID  = ch_makeChunk((SCTP_simple_chunk *) heartbeatChunk);
    pathKey       = ch_HBpathID(heartbeatCID);
    sendingTime   = ch_HBsendingTime(heartbeatCID);
    roundtripTime = pm_getTime() - sendingTime;
    /* take care of wrapround */
    if (roundtripTime < 0) roundtripTime += 1728000;

    event_logii(INTERNAL_EVENT_0, "HBAck for path %u, RTT = %d msecs", pathKey, roundtripTime);

    hbSignatureOkay = ch_verifyHeartbeat(heartbeatCID);
    event_logi(EXTERNAL_EVENT, "HB Signature is %s", (hbSignatureOkay == TRUE)?"correct":"FALSE");

    if (hbSignatureOkay == FALSE) {
        error_log(ERROR_MAJOR, "pm_heartbeatAck: FALSE SIGNATURE...ignore HB chunk...");
        ch_forgetChunk(heartbeatCID);
        return SCTP_FALSE_SIGNATURE;
    }
    ch_forgetChunk(heartbeatCID);

    if (pm_checkValidPathKey(asok, pathKey) != SCTP_SUCCESS ) {
        error_logi(ERROR_MAJOR, "pm_heartbeatAck: invalid (old ?) path key %u received", pathKey);
        return SCTP_SPECIFIC_FUNCTION_ERROR;
    }

    pathIndex = mdi_getIndexForKey(asok, pathKey);
    pd =  &g_array_index(pm->pathData, pm_pathData, pathIndex);
    assert(pd);
        
    if (pd->confirmationState == PM_PATH_UNCONFIRMED)
    {
        mdi_networkStatusChangeNotif(asok, pathKey, PM_PATH_CONFIRMED);
        pd->confirmationState = PM_PATH_CONFIRMED;
        event_logii(EXTERNAL_EVENT, "Path Status of PATH (index %d, key %u) is CONFIRMED", pathIndex, pathKey);
    }

    /* this also resets error counters */
    handleChunksAcked(pm, pathIndex, (unsigned int)roundtripTime);

    if (pd->state == PM_INACTIVE) {
        /* Handling of acked heartbeats is the simular that that of acked data chunks. */
        /* change to the active state */
        pd->state = PM_ACTIVE;
        event_logi(INTERNAL_EVENT_0, "path with key %u changed to ACTIVE", pathKey);
        mdi_networkStatusChangeNotif(asok, pathKey, PM_ACTIVE);

        /* restart timer with new RTO */
        sctp_stopTimer(pd->hearbeatTimer);
        pd->hearbeatTimer = adl_startTimer((pd->heartbeatIntervall + pd->rto),
                                            &pm_heartbeatTimer, TIMER_TYPE_HEARTBEAT,
                                            (void *) asok,
                                            (void *) pd->pathKeyPtr);
    }
    pd->heartbeatAcked  = TRUE;
    pd->timerBackoff    = FALSE;

    return SCTP_SUCCESS;
}                               /* end: pm_heartbeatAck */



/*------------------- Signals from SCTP internal modules -----------------------------------------*/

/**
 * pm_chunksAcked is called by reliable transfer whenever chunks have been acknowledged.
 * @param pathKey  key of path where chunks were originally sent to (and thus probably acked from)
 * @param newRTO   the newly determined RTT in milliseconds, null if chunks had been retransmitted
 */
int pm_chunksAcked(Association* asok, unsigned int  pathKey, unsigned int newRTO)
{
    struct timeval now;
    pm_pathData* pd = NULL;
    pm_pathMan*  pm = NULL;
    unsigned int newRTT;
    int pathIndex;

    pm = (pm_pathMan *) mdi_readPathMan(asok);

    if (pm == NULL) {
        error_log(ERROR_MAJOR, "pm_chunksAcked: mdi_readPathMan failed");
        return SCTP_MODULE_NOT_FOUND;
    }
    if (!pm->pathData) {
        error_logii(ERROR_MAJOR, "pm_chunksAcked(%u, %u): Path Data Structures not initialized yet, returning !",
                    pathKey, newRTO);
        return SCTP_UNSPECIFIED_ERROR;
    }
    if (pm_checkValidPathKey(asok, pathKey) != SCTP_SUCCESS) {
        error_log(ERROR_MAJOR, "pm_chunksAcked: got wrong address key !!");
        return  SCTP_PARAMETER_PROBLEM;
    }

    newRTT = min(newRTO, pm->rto_max);
    pathIndex = mdi_getIndexForKey(asok, pathKey);
    pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
    assert(pd);

    if (pd->state == PM_ACTIVE) {
        /* Update RTO only if is the first data chunk acknowldged in this RTO intervall. */
        adl_gettime(&now);
        if (timercmp(&now, &(pd->rto_update), < )) {
            event_logiiii(VERBOSE, "pm_chunksAcked: now %lu sec, %lu usec - no update before %lu sec, %lu usec",
                        now.tv_sec, now.tv_usec,
                        pd->rto_update.tv_sec,
                        pd->rto_update.tv_usec);
            newRTT = 0;
        } else {
            if (newRTT != 0) {
                /* only if actually new valid RTT measurement is taking place, do update the time */
                pd->rto_update = now;
                adl_add_msecs_totime(&(pd->rto_update), pd->srtt);
            }
        }
        handleChunksAcked(pm, pathIndex, newRTT);
        pd->chunksAcked = TRUE;
    } else {
        /* FIX :::::::
            we got an ACK possibly from on an inactive path */
        /* immediately send out a Heartbeat on that path, then when we get */
        /* a HB-ACK, we can set the path back to ACTIVE */
        /* when original newRTT is 0 then we got a RTX-SACK, else if we are */
        /* inactive, get ACTIVE */
        /* Nay, nay nay !   stale acknowledgement, silently discard */
        return SCTP_SUCCESS;
    }
    return SCTP_SUCCESS;
}                               /* end: pm_chunksAcked */


/**
 * helper function, that simply sets the chunksSent flag of this path management instance to TRUE
 * @param pathID  index of the address, where flag is set
 */
void pm_chunksSentOn(Association* asok, unsigned int pathKey)
{
    pm_pathData* pd = NULL;
    pm_pathMan*  pm = NULL;
    int pathIndex;

    pm = (pm_pathMan *) mdi_readPathMan(asok);
    if (pm == NULL) {
        error_log(ERROR_MAJOR, "pm_chunksSentOn: mdi_readPathMan failed");
        return;
    }
    if (pm->pathData == NULL) {
        error_logi(ERROR_MAJOR, "pm_chunksSentOn(%u): Path Data Structures not initialized yet, returning !", pathKey);
        return;
    }
    if (pm_checkValidPathKey(asok, pathKey) != SCTP_SUCCESS) {
        error_log(ERROR_MAJOR, "pm_chunksSentOn: got wrong address key !!");
        return;
    }
    event_logi(VERBOSE, "Calling pm_chunksSentOn(%u)", pathKey);
    pathIndex = mdi_getIndexForKey(asok, pathKey);
    pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
    assert(pd);
    pd->chunksSent = TRUE;
}


/**
 * pm_chunksRetransmitted is called by reliable transfer whenever chunks have been retransmitted.
 * @param  pathKey  address key, where timeout has occurred (i.e. which caused retransmission)
 * @return true if association was removed (limit exceeded), false it assoc still exists
 */
gboolean pm_chunksRetransmitted(Association* asok, unsigned int pathKey)
{
    gboolean removed_association = FALSE;
    pm_pathData* pd = NULL;
    pm_pathMan*  pm = NULL;
    int pathIndex;

    pm = (pm_pathMan *) mdi_readPathMan(asok);

    if (pm == NULL) {
        error_log(ERROR_MAJOR, "pm_chunksRetransmitted: mdi_readPathMan failed");
        return removed_association;
    }
    if (pm->pathData == NULL) {
        error_logi(ERROR_MAJOR, "pm_chunksRetransmitted(%u): Path Data Structures not initialized yet, returning !", pathKey);
        return removed_association;
    }

    if (pm_checkValidPathKey(asok, pathKey) != SCTP_SUCCESS) {
        error_logi(ERROR_MAJOR, "pm_chunksRetransmitted: invalid path key %u", pathKey);
        return removed_association;
    }

    pathIndex = mdi_getIndexForKey(asok, pathKey);
    pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
    assert(pd);

    if (pd->state == PM_INACTIVE) {
        error_logi(ERROR_MAJOR, "pm_chunksRetransmitted: retransmissions over inactive path %u", pathKey);
        return removed_association;
    } else {
        removed_association = handleChunksRetransmitted(asok, pm, pathIndex, pathKey);
    }
    return removed_association;
}                               /* end: pm_chunksRetransmitted */



/**
  pm_rto_backoff is called by reliable transfer when the T3 retransmission timer expires.
  Each call of this function may double the RTO (timer back off).
  @param pathID  index of the address where the timeout occurred
*/
void pm_rto_backoff(Association* asok, unsigned int pathKey)
{
    pm_pathData* pd = NULL;
    pm_pathMan*  pm = NULL;
    int pathIndex;

    pm = (pm_pathMan *) mdi_readPathMan(asok);

    if (pm == NULL) {
        error_log(ERROR_MAJOR, "pm_rto_backoff: mdi_readPathMan failed");
        return;
    }
    if (pm->pathData == NULL) {
        error_logi(ERROR_MAJOR, "pm_rto_backoff(%u): Path Data Structures not initialized yet!", pathKey);
        return;
    }
    if (pm_checkValidPathKey(asok, pathKey) != SCTP_SUCCESS) {
        error_logi(ERROR_MAJOR, "pm_rto_backoff: invalid path key %u", pathKey);
        return;
    }
    pathIndex = mdi_getIndexForKey(asok, pathKey);
    pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
    assert(pd);

    if (pd->state == PM_ACTIVE) {
        /* Backoff timer anyway ! */
        pd->rto = min(2 * pd->rto, pm->rto_max);

        event_logiii(INTERNAL_EVENT_0, "pm_rto_backoff called for path %u, index %d: new RTO =%d",
                        pathKey, pathIndex, pd->rto);
        pd->timerBackoff = TRUE;

    } else {
        error_logi(ERROR_MINOR, "pm_rto_backoff: timer backoff for an inactive path %u", pathKey);
    }
}                               /* end pm_rto_backoff */



/*------------------- Functions called by the ULP ------------------------------------------------*/

/**
  pm_enableHB is called when ULP wants to enable heartbeat.
  @param  pathID index of address, where we sent the HBs to
  @param  hearbeatIntervall time in msecs, that is to be added to the RTT, before sending HB
  @return error code, 0 for success, 1 for error (i.e. address index too large)
*/
int pm_enableHB(Association* asok, unsigned int pathKey, unsigned int hearbeatIntervall)
{
    pm_pathData* pd = NULL;
    pm_pathMan*  pm = NULL;
    int pathIndex;

    pm = (pm_pathMan *) mdi_readPathMan(asok);

    if (pm == NULL) {
        error_log(ERROR_MAJOR, "pm_enableHB: mdi_readPathMan failed");
        return SCTP_MODULE_NOT_FOUND;
    }
    if (pm->pathData == NULL) {
        error_logii(ERROR_MAJOR, "pm_enableHB(%u,%u): Path Data Structures not initialized yet !",
            pathKey, hearbeatIntervall);
        return SCTP_MODULE_NOT_FOUND;
    }
    if (pm_checkValidPathKey(asok, pathKey) != SCTP_SUCCESS) {
        error_logi(ERROR_MAJOR, "pm_enableHB: invalid path ID %d", pathKey);
        return SCTP_PARAMETER_PROBLEM;
    }

    pathIndex = mdi_getIndexForKey(asok, pathKey);
    pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
    assert(pd);

    if (hearbeatIntervall <= pm->rto_max) {
        pd->heartbeatIntervall = hearbeatIntervall;
    } else {
        pd->heartbeatIntervall = pm->rto_max;
    }

    event_logii(VERBOSE, "pm_enableHB(%u): set interval %u msecs",pathKey,hearbeatIntervall);


    if (!pd->heartbeatEnabled) {
        pd->heartbeatEnabled = TRUE;

        pd->firstRTO = TRUE;
        pd->pathRetranscount = 0;
        pm->peerRetranscount = 0;

        pd->heartbeatSent   = FALSE;
        pd->heartbeatAcked  = FALSE;
        pd->timerBackoff    = FALSE;
        pd->chunksAcked     = FALSE;
        pd->chunksSent      = FALSE;

        pd->rto  = pm->rto_initial;
        pd->srtt = pm->rto_initial;
        pd->rttvar = 0;
        pd->hearbeatTimer = adl_startTimer((pd->heartbeatIntervall+pd->rto), &pm_heartbeatTimer,
                            TIMER_TYPE_HEARTBEAT, (void *) asok, (void *) pd->pathKeyPtr);
        event_logi(VERBOSE, "pm_enableHB: started timer - going off in %u msecs",
                   pd->heartbeatIntervall + pd->rto);
    } else {
        pd->hearbeatTimer   = adl_restartTimer(pd->hearbeatTimer, (pd->heartbeatIntervall+pd->rto));
        pd->chunksSent      = FALSE;
        event_logi(VERBOSE, "pm_enableHB: restarted timer - going off in %u msecs", pd->heartbeatIntervall+pd->rto);
    }
    return SCTP_SUCCESS;
}                               /* end: pm_enableHB */



/**
  pm_disableAllHB is usually called on shutdown to disable all heartbeats.
*/
int pm_disableAllHB(Association* asok)
{
    int pathIndex;
    pm_pathData* pd = NULL;
    pm_pathMan*  pm = NULL;

    pm = (pm_pathMan *) mdi_readPathMan(asok);

    if (pm == NULL) {
        error_log(ERROR_MAJOR, "pm_disableAllHB: mdi_readPathMan failed");
        return SCTP_MODULE_NOT_FOUND;
    }

    if (pm->pathData == NULL) {
        error_log(ERROR_MAJOR, "pm_disableAllHB: no paths set");
        return SCTP_UNSPECIFIED_ERROR;
    }

    for (pathIndex = 0; pathIndex < pm->numberOfPaths; pathIndex++) {

        pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
        assert(pd);

        if (pd->heartbeatEnabled) {
            sctp_stopTimer(pd->hearbeatTimer);
            pd->heartbeatEnabled = FALSE;
            pd->hearbeatTimer = 0;
            event_logi(INTERNAL_EVENT_0, "pm_disableAllHB: path index %d disabled", pathIndex);
        }
    }
    return SCTP_SUCCESS;
}                               /* end: pm_disableAllHB */



/**
 * pm_disableHB is called to disable heartbeat for one specific path id.
 * @param  pathKey index of  address, where HBs should not be sent anymore
 * @return error code: SCTP_SUCCESS for success, SCTP_MODULE_NOT_FOUND, SCTP_PARAMETER_PROBLEM
 */
int pm_disableHB(Association * asok, unsigned int pathKey)
{
    int pathIndex;
    pm_pathData* pd = NULL;
    pm_pathMan*  pm = NULL;

    pm = (pm_pathMan *) mdi_readPathMan(asok);

    if (pm == NULL) {
        error_log(ERROR_MAJOR, "pm_disableHB: mdi_readPathMan failed");
        return SCTP_MODULE_NOT_FOUND;
    }

    if (pm->pathData == NULL) {
        error_logi(ERROR_MAJOR, "pm_disableHB(%d): no paths set", pathKey);
        return SCTP_MODULE_NOT_FOUND;
    }

    if (pm_checkValidPathKey(asok, pathKey) != SCTP_SUCCESS) {
        error_logi(ERROR_MAJOR, "pm_disableHB: invalid path ID %d", pathKey);
        return SCTP_PARAMETER_PROBLEM;
    }

    pathIndex = mdi_getIndexForKey(asok, pathKey);
    pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
    assert(pd);

    if (pd->heartbeatEnabled) {
        sctp_stopTimer(pd->hearbeatTimer);
        pd->heartbeatEnabled = FALSE;
        event_logi(INTERNAL_EVENT_0, "pm_disableHB: path %u disabled", pathKey);
    }
    return SCTP_SUCCESS;
}                               /* end: pm_disableHB */



/**
 * pm_setPrimaryPathKey sets the primary path.
 * @param pathID     index of the address that is to become primary path
 * @return SCTP_SUCCESS if okay, else SCTP_MODULE_NOT_FOUND, SCTP_UNSPECIFIED_ERROR,
 *         SCTP_SPECIFIC_FUNCTION_ERROR or SCTP_PARAMETER_PROBLEM
 */
int pm_setPrimaryPathKey(Association* asok, unsigned int pathKey)
{
    int pathIndex;
    pm_pathData* pd = NULL;
    pm_pathMan*  pm = NULL;

    pm = (pm_pathMan *) mdi_readPathMan(asok);

    if (pm == NULL) {
        error_log(ERROR_MAJOR, "pm_setPrimaryPath: mdi_readPathMan failed");
        return SCTP_MODULE_NOT_FOUND;
    }
    if (pm->pathData == NULL) {
        error_logi(ERROR_MAJOR, "pm_setPrimaryPath(%u): no paths set", pathKey);
        return SCTP_UNSPECIFIED_ERROR;
    }

    if (pm_checkValidPathKey(asok, pathKey) == SCTP_SUCCESS) {
        pathIndex = mdi_getIndexForKey(asok, pathKey);
        pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
        assert(pd);

        if (pd->state == PM_ACTIVE) {
            pm->primaryPathKey = pathKey;
            pd->chunksSent = FALSE;
            event_logi(INTERNAL_EVENT_0, "pm_setPrimaryPath: path %u is primary", pathKey);
            return SCTP_SUCCESS;
        } else {
            event_logi(INTERNAL_EVENT_0, "pm_setPrimaryPath: path %u not ACTIVE", pathKey);
            return SCTP_SPECIFIC_FUNCTION_ERROR;
        }
    } else {
        error_logi(ERROR_MAJOR, "pm_setPrimaryPath: invalid path %u", pathKey);
        return SCTP_PARAMETER_PROBLEM;
    }
}                               /* end: pm_setPrimaryPath */



/*------------------- Functions called by ULP to read pathmanagement state info ------------------*/

/**
 * pm_readRTO returns the currently set RTO value in msecs for a certain path.
 * @param pathKey   key of the address/path
 * @return  path's current RTO
*/
int pm_readRTO(Association* asok, unsigned int pathKey)
{
    int pathIndex;
    pm_pathData* pd = NULL;
    pm_pathMan*  pm = NULL;

    pm = (pm_pathMan *) mdi_readPathMan(asok);
    assert(pm);

    if (pm_checkValidPathKey(asok, pathKey) == SCTP_SUCCESS) {
        if (pm->pathData == NULL) {
            return (int)pm->rto_initial;
        } else {
            pathIndex = mdi_getIndexForKey(asok, pathKey);
            pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
            assert(pd);
            return (int)pd->rto;
        }
    } else {
        error_logi(ERROR_MAJOR, "pm_readRTO(%u): invalid path ID", pathKey);
        return SCTP_PARAMETER_PROBLEM;
    }
}                               /* end: pm_readRTO */

/**
 * pm_readRttVar returns the currently measured value for Round-Trip
 * time variation of a certain path.
 * @param pathKey    key of the address/path
 * @return  path's current RTTvar in msecs, SCTP_UNSPECIFIED_ERROR or SCTP_PARAMETER_PROBLEM
*/
int pm_readRttVar(Association* asok, unsigned int pathKey)
{
    int pathIndex;
    pm_pathData* pd = NULL;
    pm_pathMan*  pm = NULL;

    pm = (pm_pathMan *) mdi_readPathMan(asok);
    assert(pm);

    if (!pm->pathData) {
        error_logi(ERROR_MAJOR, "pm_readRttVAr(%u): Path Data Structures are NULL !", pathKey);
        return SCTP_UNSPECIFIED_ERROR;
    }

    if (pm_checkValidPathKey(asok, pathKey) == SCTP_SUCCESS) {
        pathIndex = mdi_getIndexForKey(asok, pathKey);
        pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
        assert(pd);
        return (int)pd->rttvar;
    } else {
        error_logi(ERROR_MAJOR, "pm_readRttVar(%u): invalid path ID", pathKey);
        return SCTP_PARAMETER_PROBLEM;
    }
}                               /* end: pm_readRttVar */


/**
 * pm_readSRTT returns the currently set SRTT value for a certain path.
 * @param pathKey  key of the address/path
 * @return  path's current smoothed round trip time, or SCTP_UNSPECIFIED_ERROR, SCTP_PARAMETER_PROBLEM
 */
int pm_readSRTT(Association* asok, unsigned int pathKey)
{
    int pathIndex;
    pm_pathData* pd = NULL;
    pm_pathMan*  pm = NULL;

    pm = (pm_pathMan *) mdi_readPathMan(asok);
    assert(pm);

    if (!pm->pathData) {
        event_logi(VERBOSE, "pm_readSRTT(%d): Path Data Structures null", pathKey);
        return SCTP_UNSPECIFIED_ERROR;
    }
    if (pm_checkValidPathKey(asok, pathKey) == SCTP_SUCCESS) {
        pathIndex = mdi_getIndexForKey(asok, pathKey);
        pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
        assert(pd);

        return (int)(pd->srtt);
    } else {
        error_logi(ERROR_MAJOR, "pm_readSRTT: invalid path ID %d", pathKey);
        return SCTP_PARAMETER_PROBLEM;
    }
}                               /* end: pm_readSRTT */

/**
 * pm_pathConfirmed returns the current confirmation state of the path.
 * @param pathID  path-ID
 * @return TRUE if path has been confirmed, FALSE if path has not been confirmed
*/
gboolean pm_pathConfirmed(Association* asok, unsigned int pathKey)
{
    int pathIndex;
    pm_pathData* pd = NULL;
    pm_pathMan*  pm = NULL;

    pm = (pm_pathMan *) mdi_readPathMan(asok);
    assert(pm);
    
    if (!pm->pathData) {
        error_log(ERROR_MAJOR, "pm_pathConfirmed: pathData==NULL failed");
        return FALSE;
    }
    if (pm_checkValidPathKey(asok, pathKey) == SCTP_SUCCESS) {
        pathIndex = mdi_getIndexForKey(asok, pathKey);
        pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
        assert(pd);
         return (pd->confirmationState == PM_PATH_CONFIRMED);
    } else {
        error_logi(ERROR_MAJOR, "pm_pathConfirmed: invalid path key %u", pathKey);
        return FALSE;
    }
}


/**
 * pm_readState returns the current state of the path.
 * @param pathKey  key of the questioned address
 * @return state of path (PM_INACTIVE / PM_ACTIVE), or  SCTP_PARAMETER_PROBLEM
 */
int pm_readState(Association* asok, unsigned int pathKey)
{
    int pathIndex;
    pm_pathData* pd = NULL;
    pm_pathMan*  pm = NULL;

    pm = (pm_pathMan *) mdi_readPathMan(asok);
    assert(pm);

    if (pm->pathData == NULL) return PM_INACTIVE;

    if (pm_checkValidPathKey(asok, pathKey) == SCTP_SUCCESS) {
        pathIndex = mdi_getIndexForKey(asok, pathKey);
        pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
        assert(pd);
        return pd->state;
    } else {
        error_logi(ERROR_MAJOR, "pm_readState: invalid path ID %u", pathKey);
        return SCTP_PARAMETER_PROBLEM;
    }
}                               /* end: pm_readState */


/**
 * pm_readAsocMTU returns the current MTU of the asociation
 * which is the lowest path mtu
 * @return Current MTU or SCTP_PARAMETER_PROBLEM
 */
int pm_readAsocMTU(Association* asok)
{
    pm_pathMan*  pm = NULL;

    if(asok == NULL)
      return MAX_SCTP_PDU;
    
    pm = (pm_pathMan *) mdi_readPathMan(asok);
    assert(pm);

    return pm->mtu;
}                               /* end: pm_readAsocMTU */


/**
 * pm_readPathSCTPMTU returns the current pathMTU of the path.
 * That means: the length that is left for chunks (without SCTP common header)
 * @param pathKey  key of the questioned address
 * @return Current Path MTU or SCTP_PARAMETER_PROBLEM
 */
int pm_readPathSCTPMTU(Association* asok, unsigned int pathKey)
{
    int pathIndex;
    pm_pathData* pd = NULL;
    pm_pathMan*  pm = NULL;

    if(asok == NULL)
      return MAX_SCTP_PDU;

    pm = (pm_pathMan *) mdi_readPathMan(asok);
    assert(pm);

    if (pm_checkValidPathKey(asok, pathKey) == SCTP_SUCCESS) {
        pathIndex = mdi_getIndexForKey(asok, pathKey);
        pd = &g_array_index(pm->pathData, pm_pathData, pathIndex);
        assert(pd);
        return pd->pathSCTPMTU;
    } else {
        error_logi(ERROR_MAJOR, "pm_readPathSCTPMTU: invalid path ID %u", pathKey);
        return SCTP_PARAMETER_PROBLEM;
    }
}                               /* end: pm_readPathSCTPMTU */


int pm_checkValidPathKey(Association* asok, unsigned int pathKey)
{
    int idx = 0;
    assert(asok);

    idx = mdi_getIndexForKey(asok, pathKey);

    if (idx == SCTP_PARAMETER_PROBLEM) return  SCTP_SPECIFIC_FUNCTION_ERROR;


    return SCTP_SUCCESS;
}


/**
 * pm_getPrimaryPathKey is used to determine the current primary path
 * @return key to the primary path, or 0 if error (pathmanagement is NULL)
 */
unsigned int pm_getPrimaryPathKey(Association* asok)
{
    pm_pathMan * pmData = (pm_pathMan *) mdi_readPathMan(asok);

    if (pmData == NULL) {
        event_log(ERROR_MAJOR, "pm_readPrimaryPathKey: pathmanagement is NULL");
        return 0;
    } else {
        return pmData->primaryPathKey;
    }
}                               /* end: pm_readPrimaryPath */

/**
 * pm_getMaxPathRetransmisions is used to get the current  maxPathRetransmissions
 * parameter value
 * @return   maxPathRetransmissions of the current instance, or SCTP_MODULE_NOT_FOUND
 */
int pm_getMaxPathRetransmisions(Association* asok)
{
    pm_pathMan * pmData = (pm_pathMan *) mdi_readPathMan(asok);

    if (pmData == NULL) {
        event_log(ERROR_MAJOR, "pm_getMaxPathRetransmisions(): pathmanagement is NULL");
        return SCTP_MODULE_NOT_FOUND;
    } else {
        return pmData->maxPathRetransmissions;
    }
}                               /* end: pm_getMaxPathRetransmisions(void) */

/**
 * pm_setMaxPathRetransmisions is used to get the current  maxPathRetransmissions
 * parameter value
 * @param   new_max  new value for  maxPathRetransmissions parameter
 * @return   SCTP_SUCCESS for success, SCTP_MODULE_NOT_FOUND for error
 */
int pm_setMaxPathRetransmisions(Association* asok, int new_max)
{
    pm_pathMan * pmData = (pm_pathMan *) mdi_readPathMan(asok);

    if (pmData == NULL) {
        event_log(ERROR_MAJOR, "pm_setMaxPathRetransmisions(): pathmanagement-instance does not exist");
        return SCTP_MODULE_NOT_FOUND;
    } else {
        pmData->maxPathRetransmissions = new_max;
    }
    return SCTP_SUCCESS;
}                               /* end: pm_setMaxPathRetransmisions(void) */

/**
 * pm_getRtoInitial is used to get the current  rto_initial parameter value
 * @return rto_initial on success, SCTP_MODULE_NOT_FOUND for error
 */
int  pm_getRtoInitial(Association* asok)
{
    pm_pathMan * pmData = (pm_pathMan *) mdi_readPathMan(asok);

    if (pmData == NULL) {
        event_log(ERROR_MAJOR, "pm_getRtoInitial(): pathmanagement-instance does not exist");
        return SCTP_MODULE_NOT_FOUND;
    }
    return pmData->rto_initial;
}                               /* end: pm_getRtoInitial() */

/**
 * pm_setRtoInitial is used to set the current  rto_initial parameter value
 * @return  SCTP_SUCCESS on success, SCTP_MODULE_NOT_FOUND for error
 */
int  pm_setRtoInitial(Association* asok, int new_rto_initial)
{
    pm_pathMan * pmData = (pm_pathMan *) mdi_readPathMan(asok);

    if (pmData == NULL) {
        event_log(ERROR_MAJOR, "pm_setRtoInitial(): pathmanagement-instance does not exist");
        return SCTP_MODULE_NOT_FOUND;
    }
    pmData->rto_initial = new_rto_initial;
    return SCTP_SUCCESS;
}                               /* end: pm_setRtoInitial() */


int  pm_setRtoMin(Association* asok, int new_rto_min)
{
    pm_pathMan * pmData = (pm_pathMan *) mdi_readPathMan(asok);

    if (pmData == NULL) {
        event_log(ERROR_MAJOR, "pm_setRtoMin(): pathmanagement-instance does not exist");
        return SCTP_MODULE_NOT_FOUND;
    }
    pmData->rto_min = new_rto_min;
    return SCTP_SUCCESS;
}                               /* end: pm_setRtoMin() */

int  pm_getRtoMin(Association* asok)
{
    pm_pathMan * pmData = (pm_pathMan *) mdi_readPathMan(asok);

    if (pmData == NULL) {
        event_log(ERROR_MAJOR, "pm_getRtoMin(): pathmanagement-instance does not exist");
        return SCTP_MODULE_NOT_FOUND;
    }
    return pmData->rto_min;
}                               /* end: pm_getRtoMin() */

int  pm_setRtoMax(Association* asok, int new_rto_max)
{
    pm_pathMan * pmData = (pm_pathMan *) mdi_readPathMan(asok);

    if (pmData == NULL) {
        event_log(ERROR_MAJOR, "pm_setRtoMax(): pathmanagement-instance does not exist");
        return SCTP_MODULE_NOT_FOUND;
    }
    pmData->rto_max = new_rto_max;
    return SCTP_SUCCESS;
}                               /* end: pm_setRtoMax() */

int  pm_getRtoMax(Association* asok)
{
    pm_pathMan * pmData = (pm_pathMan *) mdi_readPathMan(asok);

    if (pmData == NULL) {
        event_log(ERROR_MAJOR, "pm_getRtoMax(): pathmanagement-instance does not exist");
        return SCTP_MODULE_NOT_FOUND;
    }
    return pmData->rto_max;
}                               /* end: pm_getRtoMax() */


int pm_addDestinationAddress(Association* asok,
                             unsigned int newKey,
                             int newIndex)
{
    pm_pathMan*  pm = NULL;
    pm_pathData pd;
    pm_pathData* pdPtr = NULL;

    event_logiii(INTERNAL_EVENT_0, "pm_addDestinationAddress: asokID=%u, newKey=%u, newIndex=%d)",
        asok->assocId, newKey, newIndex);
            
    pm = (pm_pathMan *) mdi_readPathMan(asok);
    
    if (pm == NULL) {
        error_log(ERROR_MAJOR, "pm_addDestinationAddress: mdi_readPathMan failed");
        return SCTP_MODULE_NOT_FOUND;
    }
        
    pd.state            = PM_ACTIVE;
    pd.heartbeatEnabled = TRUE;
    pd.firstRTO         = TRUE;
    pd.pathRetranscount = 0;
    pd.rto              = pm->rto_initial;
    pd.srtt             = pm->rto_initial;
    pd.rttvar           = 0;

    pd.heartbeatSent    = FALSE;
    pd.heartbeatAcked   = FALSE;
    pd.timerBackoff     = FALSE;
    pd.chunksAcked      = FALSE;
    pd.chunksSent       = FALSE;

    pd.heartbeatIntervall = PM_INITIAL_HB_INTERVAL;
    pd.hearbeatTimer    = 0;
    /* assign proper keys based on the indexes used in distribution */
    pd.pathKey          = newKey;
    pd.pathKeyPtr       = malloc(sizeof(unsigned int));
    memcpy(pd.pathKeyPtr, &pd.pathKey, sizeof(unsigned int));
     
    event_logi(VERBOSE, "pm_addDestinationAddress: created path data structure for path (key=%u)", pd.pathKey);
    event_logi(VERBOSE, "pm_addDestinationAddress: created path data structure for path (index=%d)", newIndex);

    /* after RTO we can do next RTO update */
    adl_gettime(&(pd.rto_update));
    adl_add_msecs_totime(&(pd.rto_update), pd.rto);
    pm->pathData = g_array_append_val(pm->pathData, pd);
    
    pdPtr = &g_array_index(pm->pathData, pm_pathData, newIndex);
    
    event_logi(VERBOSE, "pm_addDestinationAddress: test: key from pointer %x", pdPtr);
    event_logii(VERBOSE, "pm_addDestinationAddress: test: (key=%u, state=%d)", pdPtr->pathKey, pdPtr->state);
           
    pdPtr->hearbeatTimer = adl_startTimer(pdPtr->heartbeatIntervall+pdPtr->rto,
                                          &pm_heartbeatTimer,
                                          TIMER_TYPE_HEARTBEAT,
                                          asok,
                                          (void *) pdPtr->pathKeyPtr);
    pm->numberOfPaths++;
   
    return SCTP_SUCCESS;
}

int pm_delDestinationAddress(Association* asok,
                             unsigned int delKey,
                             int delIndex)
{
    pm_pathData* pd = NULL;
    pm_pathMan*  pm = NULL;
    int newIndex = 0;

    pm = (pm_pathMan *) mdi_readPathMan(asok);
    if (pm == NULL) {
        error_log(ERROR_MAJOR, "pm_delDestinationAddress: mdi_readPathMan failed");
        return SCTP_MODULE_NOT_FOUND;
    }
    
    pd = &g_array_index(pm->pathData, pm_pathData, delIndex);
    assert(pd);
    adl_stopTimer(pd->hearbeatTimer);
    free(pd->pathKeyPtr);
    
    pm->pathData = g_array_remove_index(pm->pathData, delIndex);
    pm->numberOfPaths--;

    if (delKey == pm->primaryPathKey) {
        newIndex = (delIndex)%pm->numberOfPaths;
        pm->primaryPathKey = mdi_getKeyForIndex(asok, newIndex);
        pm->primaryPathIndex = newIndex;
        event_logii(VERBOSE, "pm_delDestinationAddress: setting new PRIMARY KEY: %u, index: %d",
                pm->primaryPathKey, pm->primaryPathIndex);
    }
    return SCTP_SUCCESS;
}




























