/*
 * $Id: distribution.h,v 1.9 2003/10/10 15:26:20 rohde Exp $
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * Acknowledgement
 * This work was partially funded by the Bundesministerium fr Bildung und
 * Forschung (BMBF) of the Federal Republic of Germany (Frderkennzeichen 01AK045).
 * The authors alone are responsible for the contents.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * Purpose: This header-file defines the SCTP-internal interface of message
 *          distribution that uses SCTP.
 *          The function receive_msg called by the unix-interface when a new datagramm
 *          is received, is not defined here but in distribution.c, because it is
 *          called via a function-pointer that is registered at the UNIX-interface
 *          when SCTP is initialized. In this way no other module can call receive_msg
 *          directly.
 *          Further comments on message distribution can be found in the implemen-
 *          tation of this interface distribution.c.
 *
 * function prefix: mdi_
 */

#ifndef DISTRIBUTION_H
#define DISTRIBUTION_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include  "globals.h"           /* for public association data structure */
#include  "messages.h"

/* define some important constants */
#define ASSOCIATION_MAX_RETRANS 10
#define MAX_INIT_RETRANSMITS    8
#define MAX_PATH_RETRANSMITS    5
#define VALID_COOKIE_LIFE_TIME  10000
/* #define RWND_CONST              64000 */
#define SACK_DELAY              200
#define RTO_INITIAL             3000
#define IPTOS_DEFAULT           0x10    /* IPTOS_LOWDELAY */
#define RTO_MIN                 1000
#define DEFAULT_MAX_SENDQUEUE   0       /* unlimited send queue */
#define RTO_MAX                 60000


/******************** Function Definitions ********************************************************/


/**
 * This struct stores data of SCTP-instances. Each SCTP-instances is associated to
 * one port. I.E. Server==one instance, multiple associations....
 */
typedef struct SCTPINSTANCE
{
    /* @{ */ 
    /** The name of this SCTP-instance, used as key. */
    int sctpInstanceName;
    /** The local port of this instance, or zero for don't cares.
        Once assigned this should not be changed !   */
    unsigned short localPort;
    unsigned int numberOfLocalAddresses;
    /*  union sockunion *localAddressList; */
    GArray*   localAddresses;
    GArray*   localAddressKeys;
    gboolean    has_INADDR_ANY_set;
    gboolean    has_IN6ADDR_ANY_set;
    gboolean    uses_IPv4;
    gboolean    uses_IPv6;
    /** set of callback functions that were registered by the ULP */
    SCTP_ulpCallbacks ULPcallbackFunctions;
    /** maximum number of incoming streams that this instance will take */
    unsigned short noOfInStreams;
    /** maximum number of outgoingng streams that this instance will take */
    unsigned short noOfOutStreams;
    /* if this is 0 or 1 we use reliable or unreliable streams, -1 if we do not support it */
    int supportsPRSCTP;
    /* if this is 0 or 1 we use reliable or unreliable streams, -1 if we do not support it */
    int supportsADDIP;
    /** here follow default parameters for instance initialization */
    unsigned int default_rtoInitial;
    unsigned int default_validCookieLife;
    unsigned int default_assocMaxRetransmits;
    unsigned int default_pathMaxRetransmits;
    unsigned int default_maxInitRetransmits;
    unsigned int default_myRwnd;
    unsigned int default_delay;
    unsigned char default_ipTos;
    unsigned int default_rtoMin;
    unsigned int default_rtoMax;
    unsigned int default_maxSendQueue;
    unsigned int default_maxRecvQueue;
    unsigned int supportedAddressTypes;
   /* @} */
}
SCTP_instance;


/**
 * This struct contains all data of an association. As far as other modules must know elements
 * of this struct, read functions are provided. No other module has write access to this structure.
 */
typedef struct ASSOCIATION
{
    /* @{ */
    /** The current ID of this association,
        it is used as a key to find a association in the list,
        and never changes in the  live of the association  */
    unsigned int assocId;
    /** The local tag of this association. */
    unsigned int tagLocal;
    /** The tag of remote side of this association */
    unsigned int tagRemote;
    /** Pointer to the SCTP-instance this association
        belongs to. It is equal to the wellknown port
        number of the ULP that uses this instance. */
    SCTP_instance*  sctpInstance;
    /** the local port number of this association. */
    unsigned short localPort;
    /** the remote port number of this association. */
    unsigned short remotePort;
    /** number of destination networks (paths) */
    unsigned int noOfDestAddresses;
    /** array of destination addresses */
    /*  union sockunion *destinationAddresses; */
    GArray*  destAddresses;
    GArray*  destAddressKeys;
    unsigned int lastDestKey;
    /** number of own addresses */
    unsigned int noOfLocalAddresses;
    /** array of local addresses */
    GArray*   localAddresses;
    GArray*   localAddressKeys;
    unsigned int lastLocalKey;
    /*    union sockunion *localAddresses; */
    /** pointer to flowcontrol structure */
    void *flowControl;
    /** pointer to reliable-transfer structure */
    void *reliableTransfer;
    /** pointer to receive-control structure */
    void *rx_control;
    /** pointer to stream structure */
    void *streamengine;
    /** pointer to pathmanagement structure */
    void *pathMan;
    /** pointer to bundling structure */
    void *bundling;
    /** pointer to SCTP-control */
    void *sctp_control;
    /** pointer to module for deleting/adding addresses and per stream based flow control */
    void *sctp_asconf;
    /** marks an association for deletion */
    boolean deleted;
    /** transparent pointer to some upper layer data */
    void * ulp_dataptr;
    /** IP TOS value per association */
    unsigned char ipTos;
    unsigned int supportedAddressTypes;
    unsigned int maxSendQueue;
    unsigned int maxRecvQueue;
    gboolean    had_INADDR_ANY_set;
    gboolean    had_IN6ADDR_ANY_set;
    /* if this is 0 we use only  reliable streams */
    int   supportsPRSCTP;
    /* if this is 0 peer does not use unreliable streams, -1 if it does not support them */
    int   peerSupportsPRSCTP;
    /* if this is 0 we use only  reliable streams */
    int   supportsADDIP;
    /* if this is 0 peer does not use unreliable streams, -1 if it does not support them */
    int   peerSupportsADDIP;
    /* if this is 1 if peer autoselected a port, 0 if it is was preselected by the user */
    int   selectedPort;
    /* if this is 0 if peer did not receive adaptation layer indication, else it is len of this memory area*/
    int   adaptationLayerIndicationLen;                                                                   
    /* this contains the pointer to the adaptation layer indication data */
    void* adaptationLayerIndication;

    /* @} */
} Association;



/*------------------- Functions called by the ULP ------------------------------------------------*/

/**
 * These functions are defined in a separate header file sctp.h in order to separate the interface to
 *  the ULP and the interface to other modules within SCTP.
 */

int mdi_getMyNumberOfAddresses(void);

/*------------------- Functions called by the Unix-Interface -------------------------------------*/

/**
 * \fn  mdi_receiveMessage
 *  mdi_receiveMessage is the callback function of the SCTP-message distribution.
 *  It is called by the Unix-interface module when a new datagramm is received.
 *  This function also performs OOTB handling, tag verification etc.
 *  (see also RFC 2960, section 8.5.1.B)  and sends data to the bundling module of
 *  the right association
 *
 *  @param socket_fd          the socket file discriptor
 *  @param buffer             pointer to arrived datagram
 *  @param bufferlength       length of datagramm
 *  @param fromAddress        source address of DG
 *  @param portnum            bogus port number
 */
void mdi_receiveMessage(gint socket_fd, unsigned char *buffer,
                        int bufferLength, union sockunion * source_addr,
                        union sockunion * dest_addr);

/*------------------- Functions called by the SCTP bundling --------------------------------------*/

/**
 * \fn   mdi_send_message
 * Used by bundling to send a SCTP-daatagramm.
 *  Before calling send_message at the adaption-layer, this function does:
 *  - add the SCTP common header to the message
 *  - convert the SCTP message to a byte string
 *  - retrieve the socket-file descriptor of the SCTP-instance
 *  - retrieve the destination address
 *  @param SCTP_message     SCTP message (UDP-datapart) as a struct
 *  @param length           length of SCTP message.
 *  @param destAddresIndex  Index of address in the destination address list.
 *  @return                 Errorcode.
 */
int mdi_send_message(Association* asoc,
                     SCTP_message * message,
                     unsigned int length,
                     unsigned int destAddressKey);
                     

/*------------------- Functions called by the Pathmanagement --------------------------------------*/

/**
 *   retrieveAssociation retrieves a association from the list using the transport address as key.
 *   Returns NULL also if the association is marked "deleted" !
 *   CHECKME : Must return NULL, if no Address-Port combination does not occur in ANY existing assoc.
 *             If it occurs in one of these -> return it
 *
 *   @param  fromAddress address from which data arrived
 *   @param  fromPort SCTP port from which data arrived
 *   @return pointer to the retrieved association, or NULL
 */
Association *mdi_retrieveAssociationByTransportAddress(union sockunion * fromAddress,
                                                   unsigned short fromPort,
                                                   unsigned short toPort);



unsigned int mdi_getCurrentMinAdapterMTU();

/*------------------- Functions called by the SCTP to forward primitives to ULP ------------------*/


/**
 * \fn  mdi_dataArriveNotif
 *  indicates new data (i.e. length bytes) have arrived from peer (chapter 10.2.A).
 *  @param streamID    data has arrived for this stream
 *  @param length       number of bytes thet have arrived
 *  @param  unordered  unordered flag (TRUE==1==unordered, FALSE==0==normal,numbered chunk)
 *  @param   asok  the association where this happened
 */
void mdi_dataArriveNotif(Association* asoc,
                         unsigned int streamID,
                         unsigned int length,
                         unsigned short streamSN,
                         unsigned int tsn,
                         unsigned int protoID,
                         unsigned int unordered);


/**
 * \fn mdi_networkStatusChangeNotif
 *  indicates a change of network status (chapter 10.2.C).
 *  @param  destinationAddress   index of the destination addresses that has changed state
 *  @param  newState             variable indicating the new state
 *  @param   asok  the association where this happened
 */
void mdi_networkStatusChangeNotif(Association* asoc, unsigned int destinationAddress, int newState);

/**
 * indicates a send failure (chapter 10.2.B) Maybe because data's lifetime has expired.
 * @param  data         pointer to data not sent
 * @param  dataLength   length of the data buffer
 * @param  context      a pointer to a context (unsused ?)
 *  @param   asok  the association where this happened
 */
void mdi_sendFailureNotif(Association* asoc, unsigned char *data, unsigned int dataLength, unsigned int *context);

/**
 *  indicates that communication was lost to peer (chapter 10.2.E).
 *  @param  status  identifier for the type of event (i.e. how it happened)
 *  @param   asok  the association where this happened
 */
void mdi_communicationLostNotif(Association* asoc, int status);

/**
 *  indicates that communication has been gracefully terminated (10.2.H).
 *  @param   asok  the association where this happened
 */
void mdi_shutdownCompleteNotif(Association* asoc);

/**
 *  indicates that a restart has occured (10.2.G.)
 *  @param   asok  the association where this happened
 */
void mdi_restartNotif(Association* asoc);

/**
 *  indicates that a association was established (chapter 10.2.D).
 *  @param   status  variable indicating the type of event
 *  @param   asok  the association where this happened
 */
void mdi_communicationUpNotif(Association* asoc, unsigned short status);

/**
 *  indicates that a association was established (chapter 10.2.D).
 *  @param   asok  the association where this happened
 */
void mdi_peerShutdownReceivedNotif(Association* asoc);

/**
 * Function that notifies the ULP of a change in the queue status.
 * I.e. a limit may be exceeded, and therefore subsequent send-primitives will
 * fail, OR the queue length has dropped below a previously set queue length
 *
 * @param  queueType i.e. an outbound queue, stream-engine queue, per stream queue (?)
 * @param  queueId   i.e. i.e. stream id for a per stream queue
 * @param  queueLen  in bytes or in messages, depending on the queue type
 */
void mdi_queueStatusChangeNotif(Association* asoc, int queueType, int queueId, int queueLen);


/**
 * function called by the adaptation layer, when an address is added
 * or removed (as indicated by a routing socket) _locally_
 *
 * This functions needs to trigger sending of ADD-IP, DEL-IP ASCONF
 * chunks for all associations that had INADDR_ANY specified.
 */
int mdi_addressChange(union sockunion* address, gboolean added);



/*------------------- Functions called by the SCTP to get current association data----------------*/
/**
 * When processing external events from outside the SCTP (socket events, timer events and
 * function calls from the ULP), first the data of the addressed association are read
 * from the list of associations and stored in a private but static datastructure.
 * Elements of this association data can be read by the following functions.
 *
 * The following functions return pointer to data of modules of the SCTP. As only these
 * modules know the exact type of these data structures, so the returned pointer are
 * of type void.
 */

/*
 * returns: pointer to the flow control data structure.
 *          null in case of error.
 */
extern inline void *mdi_readFlowControl(Association* currentAssociation){
    if (currentAssociation == NULL) return NULL;
    return currentAssociation->flowControl;
}

/*
 * returns: pointer to the reliable transfer data structure.
 *          null in case of error.
 */
extern inline void *mdi_readReliableTransfer(Association* currentAssociation){
    if (currentAssociation == NULL) return NULL;
    return currentAssociation->reliableTransfer;
}

/*
 * returns: pointer to the RX-control data structure.
 *          null in case of error.
 */
extern inline void *mdi_readRX_control(Association* currentAssociation){
    if (currentAssociation == NULL) return NULL;
    return currentAssociation->rx_control;
}

/*
 * returns: pointer to the stream engine data structure.
 *          null in case of error.
 */
extern inline void *mdi_readStreamEngine(Association* currentAssociation){
    if (currentAssociation == NULL) return NULL;
    return currentAssociation->streamengine;
}

/*
 * returns: pointer to the pathmanagement data structure.
 *          null in case of error.
 */
extern inline void *mdi_readPathMan(Association* currentAssociation){
    if (currentAssociation == NULL) return NULL;
    return currentAssociation->pathMan;
}

/*
 * returns: pointer to the bundling data structure.
 *          null in case of error.
 */
extern inline void *mdi_readBundling(Association* currentAssociation){
    if (currentAssociation == NULL) return NULL;
    return currentAssociation->bundling;
}

/*
 * returns: pointer to the SCTP-control data structure.
 *          null in case of error.
 */
extern inline void *mdi_readSCTP_control(Association* currentAssociation){
    if (currentAssociation == NULL) return NULL;
    return currentAssociation->sctp_control;
}

extern inline void *mdi_readASCONF(Association* currentAssociation){
    if (currentAssociation == NULL) return NULL;
    return currentAssociation->sctp_asconf;
}


/*
 * returns: association-ID of the current association
 *          0 means the association is not set (an error).
 */

/*inline unsigned int mdi_readAssociationID(Association* currentAssociation){
    if (currentAssociation == NULL) return 0;
    return currentAssociation->assocId;
} */

extern inline unsigned int mdi_readTagRemote(Association* currentAssociation){
    if (currentAssociation == NULL) return 0; 
    return currentAssociation->tagRemote;
}

extern inline unsigned int mdi_readLocalTag(Association* currentAssociation){
    if (currentAssociation == NULL) return 0; 
    return currentAssociation->tagLocal;
}

gboolean mdi_librarySupportsPRSCTP();

gboolean mdi_supportsPRSCTP(Association* currentAssociation);

unsigned int mdi_readLastFromPathKey(void);

/* returns: a ID  for new association */
unsigned int mdi_generateTag(void);
/* returns: a the start TSN for new association */
unsigned int mdi_generateStartTSN(void);

/*------------- functions to for the cookie mechanism --------------------------------------------*/
/* each time a datagram is received, its source address, source port and destination port
   is saved. This address-info is used in the following cases:
   - z-side: for sending the initAck (no association exists from which the addresses could be read).
   - a-side: if the initAck  ??
   - z-side: if the cookie-chunk does not contain addresses of the a-side, lastFromAddress_in
             is used as the only destination address.
*/


/**
 *  sets the address from which the last datagramm was received (host byte order).
 *   Returns 0 if successful, 1 if address could not be set !
 */
int mdi_readLastFromAddress(union sockunion* fromAddress);

/**
 * reads the path from which the last DG was received. -1 is returned if no DG was received.
 */
short mdi_readLastFromPath(void);

/**
 * returns the port of the sender of the last received DG.
 */
unsigned short mdi_readLastFromPort(void);

/**
 * returns the port of the destination of the last received DG.
 */
unsigned short mdi_readLastDestPort(void);

/* return the last initiate tag  */
unsigned int mdi_readLastInitiateTag(void);

/**
 * write the initiate tag of a-side to be used as verification tag for the initAck
 */
void mdi_writeLastInitiateTag(unsigned int initiateTag);

/**
 * rewrite the initiate tag of peer in case of a peer restart.
 */
void mdi_rewriteTagRemote(Association* asoc, unsigned int newInitiateTag);

/**
 * rewrite my local tag in case of a restart.
 */
void mdi_rewriteLocalTag(Association* asoc, unsigned int newTag);


extern inline int mdi_getDefaultRtoInitial(SCTP_instance* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else return sctpInstance->default_rtoInitial;
}

extern inline int mdi_getDefaultValidCookieLife(SCTP_instance* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else return sctpInstance->default_validCookieLife;
}

extern inline int mdi_getDefaultAssocMaxRetransmits(SCTP_instance* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else return sctpInstance->default_assocMaxRetransmits;
}

extern inline int mdi_getDefaultPathMaxRetransmits(SCTP_instance* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else return sctpInstance->default_pathMaxRetransmits;
}

extern inline int mdi_getDefaultMaxInitRetransmits(SCTP_instance* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else return sctpInstance->default_maxInitRetransmits;
}

extern inline int mdi_getDefaultMyRwnd(SCTP_instance* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else return sctpInstance->default_myRwnd;
}

extern inline int mdi_getDefaultRtoMin(SCTP_instance* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else return sctpInstance->default_rtoMin;
}

extern inline int mdi_getDefaultRtoMax(SCTP_instance* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else return sctpInstance->default_rtoMax;
}

extern inline int mdi_getDefaultDelay(SCTP_instance* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else return sctpInstance->default_delay;
}

extern inline int mdi_getDefaultIpTos(SCTP_instance* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else return sctpInstance->default_ipTos;
}

extern inline int mdi_getDefaultMaxSendQueue(SCTP_instance*  sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else return sctpInstance->default_maxSendQueue;
}

extern inline int mdi_getDefaultMaxRecvQueue(SCTP_instance*  sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else return sctpInstance->default_maxRecvQueue;
}

extern inline unsigned int mdi_getSupportedAddressTypes(SCTP_instance* sctpInstance)
{
    if (sctpInstance == NULL) return -1;
    else return sctpInstance->supportedAddressTypes;
}


/*------------- functions to write and read addresses --------------------------------------------*/

int mdi_writeDestinationAddresses(Association* asoc,
                                  union sockunion addresses[MAX_NUM_ADDRESSES],
                                  int noOfAddresses);


gboolean mdi_addressContainsLocalhost(SCTP_instance *sI, union sockunion* anAddress);


void mdi_readLocalAddresses(Association* asok,
                            SCTP_instance* sctpInstance,
                            union sockunion laddresses[MAX_NUM_ADDRESSES],
                            unsigned int * noOfAddresses,
                            union sockunion *paddress,
                            unsigned int numPeerAddresses,
                            unsigned int addressTypes,
                            gboolean receivedFromPeer);


/*
 * function returns TRUE, if address passed is a valid address for this association
 */
gboolean mdi_checkForValidLocalAddressInAssociation(Association* asok, union sockunion* su);

/**
 * get an index into the current array structure associated with addresses
 * address is associated with the index (the return value) for all modules
 */
int mdi_getIndexForAddress(Association* asok, union sockunion* address);

/**
 * get the key corresponding to a destination path address
 * address is associated with the key (the return value) for all modules
 */
unsigned int mdi_getKeyForAddress(Association* asok,union sockunion* address);

/**
 * given a key of a path identifier, returns the corresponding, currently valid
 * index into the array - valid for all modules
 */
int mdi_getIndexForKey(Association* asok, unsigned int pathKey);

/**
 * given a key of a path identifier, returns the corresponding, currently valid
 * index into the array - valid for all modules
 */
unsigned int mdi_getKeyForIndex(Association* asok, int pathIndex);


/*------------- Functions for the ADDIP-DELIP thing --------------------------------------------*/

/**
 * adds/removes local address to the array within the current association
 *
 * @param newAddress    the address that has been locally added
 * @aram asoc          the association where the address is added
 */
int mdi_addLocalAddressToAssoc(Association* asoc, union sockunion* newAddress);
int mdi_delLocalAddressFromAssoc(Association* asoc,
                                       union sockunion* delAddress);

/**
 * adds/removes destination address to the array passed as parameter to
 * the current association. Also needs to add appropriate structures in
 * pathmanagement, reliableTransfer, flowControl.
 *
 * @param newAddress    the address that has been added by an AS-CONF
 * @aram asoc          the association where the address is added
 * @return              the new Key of the added address
 */
unsigned int mdi_addDestinationAddressToAssoc(Association* asoc,
                                     union sockunion* newAddress);
int mdi_delDestinationAddressFromAssoc(Association* asoc,
                                       union sockunion* newAddress);

/**
 * adds address to  an sctp instance structure, if IN(6)ADDR_ANY had been specified
 * @param newAddress    the address that has been added
 * @aram ins           the instance where the address is added
 */
int mdi_addressUpdate(void);

/**
 * adds new address to  an sctp instance structure
 * @param newAddress    the address that has been added
 * @aram ins           the instance where the address is added
 */
int mdi_addAddressToInstanceForced(SCTP_instance* ins, union sockunion* addAddress);

/**
 * takes away an address from an sctp instance structure
 * @param delAddress    the address that is being deleted
 * @aram asoc          the instance where the address to be removed
 */
int mdi_delAddressFromInstance(SCTP_instance* ins, union sockunion* delAddress);


                                       
/*------------------- Functions to create and delete associations --------------------------------*/

/**
 * These functions allocate memory for  new associations.
 *
 *  For the active side of an association, mdi_newAssociation is called when the associate is called from
 *  the ULP. For the passive side this function is called when a cookie message is received.
 *  It also creates all the modules of an association supplying to them the data needed as far they
 *  they are determined at time the this module is called.
 *
 */
Association*   mdi_newAssociation(SCTP_instance* sInstance,
                                  unsigned short local_port,
                                  unsigned short remote_port,
                                  unsigned int tagLocal,
                                  unsigned int primaryDestinationAddress,
                                  unsigned int noOfDestinationAddresses,
                                  union sockunion *destinationAddressList);

int mdi_initAssociation(Association* asok,
                        unsigned int remoteSideReceiverWindow,
                        unsigned short noOfInStreams,
                        unsigned short noOfOutStreams,
                        unsigned int remoteInitialTSN,
                        unsigned int tagRemote, unsigned int localInitialTSN,
                        gboolean peerSupportsPRSCTP, gboolean peerSupportsADDIP,
                        int adaptationIndicationLen, void* adaptationIndication);


int mdi_restartAssociation(Association* asok,
                           unsigned short noOfInStreams,
                           unsigned short noOfOutStreams,
                           unsigned int new_rwnd,
                           unsigned int remoteInitialTSN,
                           unsigned int localInitialTSN,
                           unsigned int noOfDestinationAddresses,
                           unsigned int primaryAddress,
                           union sockunion *destinationAddressList,
                           gboolean peerSupportsPRSCTP, gboolean peerSupportsADDIP,
                           int adaptationIndicationLen, void* adaptationIndication);


void mdi_deleteAssociation(Association* currentAssociation);



#endif
