/*
 *  $Id: distribution.c,v 1.24 2004/01/13 16:13:14 ajung Exp $
 *
 * SCTP implementation according to RFC 2960.
 * Copyright (C) 2000 by Siemens AG, Munich, Germany.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * Acknowledgement
 * This work was partially funded by the Bundesministerium fr Bildung und
 * Forschung (BMBF) of the Federal Republic of Germany (Frderkennzeichen 01AK045).
 * The authors alone are responsible for the contents.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * There are two mailinglists available at http://www.sctp.de which should be
 * used for any discussion related to this implementation.
 *
 * Contact: discussion@sctp.de
 *          Michael.Tuexen@icn.siemens.de
 *          ajung@exp-math.uni-essen.de
 *
 * Purpose: This modules implements the interface defined distribution.h and sctp.h
 *          and holds a private list of associations.
 *          As an SCTP-instance has usually more than one associations, the main purpose
 *          of this module is the  distribution of signals from the ULP and from the 
 *          peer (via the socket and unix-interface) to the addressed association.
 *          Signals from the UNIX-interface are always forwarded to 
 *          the bundling module of the addressed association.
 *          The corresponding function receive_msg is not defined here but in distribution.c,
 *          because it is called via a function-pointer that is registered at the
 *          UNIX-interface when SCTP is initialized.
 *          Signals from the ULP are forwarded to SCTP-Control, Pathmanagement or
 *          Streamengine of the addressed association depending on the chunk type.
 *          All signals from the ULP contain the association ID, which is used to
 *          identify the association.
 *
 * Remarks: Host and network byte order (HBO and NBO):
 *          In this module all data are data are HBO, except the IP-Addresses and message strings.
 *          Thus everything entered into a message string must be converted to NBO and
 *          everthing extracted from a message string must be converted to HBO (except the
 *          IP-addresses).
 *
 *  Usage of Keys and Indexes:
 *  All modules must internally use Keys to reference paths and addresses.
 *  When these are actually referenced in the modules, the functions for
 *  converting keys into indexes must be used.
 *
 * function prefixes: sctp_    for interface-defs imported from sctp.h
 *                    mdi_     for interface-defs imported from distribution.h
 */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include  "sctp.h"              /* ULP-interface definitions */
#include  "distribution.h"      /* SCTP-internal interfaces to message distribution */

#include  "adaptation.h"        /* interfaces to adaptation layer */
#include  "bundling.h"          /* interfaces to bundling */
#include  "sctp-control.h"      /* interfaces to sctp-control */
#include  "auxiliary.h"
#include  "streamengine.h"      /* interfaces to streamengine */
#include  "flowcontrol.h"       /* interfaces to flowcontrol */
#include  "recvctrl.h"          /* interfaces to receive-controller */
#include  "chunkHandler.h"
#include  "sctp_asconf.h"

#include  <sys/types.h>
#include  <errno.h>
#ifdef WIN32
    #include <winsock2.h>
#else
    #include  <arpa/inet.h>         /* for inet_ntoa() under both SOLARIS/LINUX */
#endif

#ifndef IN_EXPERIMENTAL
    #define       IN_EXPERIMENTAL(a)      ((((long int) (a)) & 0xf0000000) == 0xf0000000)
#endif

#ifndef IN_BADCLASS
    #define       IN_BADCLASS(a)          IN_EXPERIMENTAL((a))
#endif

#include  <assert.h>


/*------------------------ Default Definitions --------------------------------------------------*/

/*------------------------ more or less global variables ----------------------------------------*/
static int myRWND = 0x7FFF;

static union sockunion *myAddressList = NULL;
static unsigned int myNumberOfAddresses = 0;

static gboolean sendAbortForOOTB        = TRUE;
static gboolean librarySupportsPRSCTP   = TRUE;
static gboolean librarySupportsADDIP    = TRUE;

static int      checksumAlgorithm   = SCTP_CHECKSUM_ALGORITHM_CRC32C;
static gboolean sctpLibraryInitialized = FALSE;

/******************** Declarations ****************************************************************/
/**
 * Descriptor of socket used by all associations and SCTP-instances.
 */
static gint sctp_socket;

#ifdef HAVE_IPV6
 static gint ipv6_sctp_socket;
#endif

/**
 * Keyed list of SCTP instances with the instance name as key
 * this should become a hashed list for more optimal performance
 */
static GList* InstanceList = NULL;

/**
 * Keyed list of associations with the association-ID as key
 * this should become a hashed list for more optimal performance
 */
static GList* AssociationList = NULL;


static unsigned int min_MTU = DEFAULT_MTU_CEILING;


static unsigned int  ipv4_users = 0;
#ifdef HAVE_IPV6
 static unsigned int ipv6_users = 0;
#endif

/* variables defined for efficiency */
static Association      tmpAssoc;

/**
 * contains the next  SctpInstanceName
 */
static int nextSctpInstanceName = 1;

/**
 * AssociationIDs are counted up, and if a new one is needed, we get the next free one
 */
static unsigned int nextAssocId = 1;

/**
 * these are pointers to the addresses of the last received IP packet, and values belonging
 * to other data associated with the last packet. After processing of an arrived packet
 * these may (and probably are) become void
 */
static union sockunion *lastFromAddress;
static union sockunion *lastDestAddress;
static unsigned short   lastFromPort;
static unsigned short   lastDestPort;
static unsigned int     lastFromPathIndex;
static unsigned int     lastFromPathKey;
static unsigned int     lastInitiateTag;


/* port management array */
static unsigned char    portsSeized[0x10000];
static unsigned int     numberOfSeizedPorts;

#ifdef HAVE_RANDOM
    static long rstate[2];
#endif


/* ---------------------- Internal Function Prototypes ------------------------------------------- */

/* ------------------------- Function Implementations --------------------------------------------- */

/*------------------- Internal Functions ---------------------------------------------------------*/

#define CHECK_LIBRARY           if(sctpLibraryInitialized == FALSE) return SCTP_LIBRARY_NOT_INITIALIZED
#define ZERO_CHECK_LIBRARY      if(sctpLibraryInitialized == FALSE) return 0

#ifdef LIBRARY_DEBUG
 #define ENTER_LIBRARY(fname)	printf("Entering sctplib  (%s)\n", fname); fflush(stdout);
 #define LEAVE_LIBRARY(fname)	printf("Leaving  sctplib  (%s)\n", fname); fflush(stdout);
 #define ENTER_CALLBACK(fname)	printf("Entering callback (%s)\n", fname); fflush(stdout);
 #define LEAVE_CALLBACK(fname)	printf("Leaving  callback (%s)\n", fname); fflush(stdout);
#else
 #define ENTER_LIBRARY(fname)	
 #define LEAVE_LIBRARY(fname)	
 #define ENTER_CALLBACK(fname)	
 #define LEAVE_CALLBACK(fname)	
#endif

/*------------------- Internal LIST Functions ----------------------------------------------------*/

/*
 * return 1 or -1 if instances have different port,
 * return 0 if same ports and one address is in set of second instances addresses
 */
gint checkForAddressInInstance(gconstpointer a, gconstpointer b)
{
    int acount,bcount;
    gboolean found;
    SCTP_instance* ai = (SCTP_instance*)a;
    SCTP_instance* bi = (SCTP_instance*)b;

    event_logii(VVERBOSE, "DEBUG: CheckForAddressInInstance, comparing instance a port %u, instance b port %u",
        ai->localPort, bi->localPort);

    if (ai->localPort < bi->localPort) return -1;
    else if (ai->localPort > bi->localPort) return 1;

    else {
        /* one has IN(6)ADDR_ANY : return equal ! */
        if (ai->has_IN6ADDR_ANY_set && bi->has_IN6ADDR_ANY_set) return 0;
        if (ai->has_INADDR_ANY_set  && bi->has_INADDR_ANY_set) return 0;
        if (ai->has_INADDR_ANY_set  && bi->has_IN6ADDR_ANY_set) return 0;
        if (ai->has_IN6ADDR_ANY_set && bi->has_INADDR_ANY_set) return 0;
        if ( (ai->has_IN6ADDR_ANY_set || ai->has_INADDR_ANY_set) &&
            !(bi->has_IN6ADDR_ANY_set || bi->has_INADDR_ANY_set)) return 0;
        if (!(ai->has_IN6ADDR_ANY_set || ai->has_INADDR_ANY_set) &&
             (bi->has_IN6ADDR_ANY_set || bi->has_INADDR_ANY_set)) return 0;

        /* both do not have an INADDR_ANY : use code below */
        found = FALSE;

        for (acount = 0; acount < ai->numberOfLocalAddresses; acount++) {
            for (bcount = 0; bcount < bi->numberOfLocalAddresses; bcount++) {
                /* if addresses are equal: set found TRUE and break; */
                if (
                adl_equal_address(&g_array_index(ai->localAddresses, union sockunion, acount),
                                  &g_array_index(bi->localAddresses, union sockunion, bcount)) == TRUE) {
                        found = TRUE;
                }

                event_logiii(VVERBOSE, "DEBUG: CheckForAddressInInstance, acount %u, bcount %u, found = %s",
                    acount, bcount, (found==TRUE)?"TRUE":"FALSE");

                if (found == TRUE) break;
            }
            if (found == TRUE) break;
        }
        /* if address was not found, it is not in this instance */
        if (found == FALSE) return -1; /* to continue search */
    }
    return 0;

}


gint compareInstanceNames(gconstpointer a, gconstpointer b)
{
    if ((((SCTP_instance*)a)->sctpInstanceName) < ((SCTP_instance*)b)->sctpInstanceName) return -1;
    else if ((((SCTP_instance*)a)->sctpInstanceName) > ((SCTP_instance*)b)->sctpInstanceName) return 1;
    else return 0;
}


/**
 *  compareAssociationIDs compares the association ID's of two associations and returns 0
 *  if they are equal. This is a call back function called by List Functions whenever two
 *  association need to be compared.
 *  @param a  pointer to association struct 1
 *  @param b  pointer to association struct 2
 *  @return    0 if a->assocId equals b->assocId, 1 if bigger, -1 if smaller
 */
gint compareAssociationIDs(gconstpointer a, gconstpointer b)
{
    /* two associations are equal if there local tags (in this implementation also used as
       association ID) are equal. */
    if (((Association*)a)->assocId < ((Association*)b)->assocId)        return -1;
    else if (((Association*)a)->assocId == ((Association*)b)->assocId)  return 0;
    else return 1;
}


/**
 *  equalAssociations compares two associations and returns 0 if they are equal. In contrast to
 *  function compareAssociationIDs, equal here means the two associations belong to the same
 *  SCTP-instance and have at least one destinationaddress in common.
 *  This is a call back function called by GList-functions whenever two association need to be compared.
 *  @param i1  association data 1
 *  @param i2  association data 2
 *  @return 0 if il1 and il2 are equal according to above definition, 1 else
 */
gint equalAssociations(gconstpointer a, gconstpointer b)
{
    int i,j;
    event_logiiii(VVERBOSE, "equalAssociations: checking assoc A[lport %u, rport %u] and assoc B[lport %u, rport %u]",
        ((Association*)a)->localPort ,((Association*)a)->remotePort, ((Association*)b)->localPort ,((Association*)b)->remotePort);

    /* two associations are equal if their remote and local ports are equal and at least
       one of their remote addresses are equal. This is like in TCP, where a connection
       is identified by the transport address, i.e. the IP-address and port of the peer. */

    if ( (((Association *)a)->remotePort == ((Association *)b)->remotePort) &&
         (((Association *)a)->localPort == ((Association *)b)->localPort) ){
        for (i = 0; i < ((Association *)a)->noOfDestAddresses; i++)
            for (j = 0; j < ((Association *)b)->noOfDestAddresses; j++) {
                event_logii(VVERBOSE, "equalAssociations: checking address A[%d] address B[%d]",i,j);
                if (adl_equal_address
                    (&g_array_index( ((Association *)a)->destAddresses,union sockunion, i),
                     &g_array_index( ((Association *)b)->destAddresses,union sockunion, j)) == TRUE) {

                    if ( (((Association *)b)->deleted == FALSE) && (((Association *)a)->deleted == FALSE))
                        return 0;
                    else
                        return 1;
                }
            }
        return 1;
    }
    return 1;
}

int mdi_getMyNumberOfAddresses(void){
    return myNumberOfAddresses;
}

/**
 * mdi_getAssociationByID retrieves a association from the list using the id as key.
 * Returns NULL also if the association is marked "deleted", unless ignore_delete is TRUE !
 * @param assocID  association ID
 * @return  pointer to the retrieved association, or NULL
 */
Association *mdi_getAssociationByID(unsigned int assocID, gboolean ignore_delete)
{
    Association *assoc;
    Association *assocFindP;
    GList* result = NULL;

    event_logi(VERBOSE, "retrieving association %u from list", assocID);

    tmpAssoc.assocId = assocID;
    tmpAssoc.deleted = FALSE;
    assocFindP = &tmpAssoc;
    assoc = NULL;

    result = g_list_find_custom(AssociationList, assocFindP, &compareAssociationIDs);
    if (result != NULL) {

        assoc = (Association *)result->data;

        if (assoc->deleted && ignore_delete == FALSE) {
            assoc = NULL;
        }
    } else {
        event_logi(VERBOSE, "association %u not in list", assocID);
        assoc = NULL;
    }
    return assoc;
}


/**
 *   retrieveAssociation retrieves a association from the list using the transport address as key.
 *   Returns NULL also if the association is marked "deleted" !
 *   CHECKME : Must return NULL, if no Address-Port combination does not occur in ANY existing assoc.
 *             If it occurs in one of these -> return it
 *
 *   @param  fromAddress address from which data arrived
 *   @param  fromPort SCTP port from which data arrived
 *   @return pointer to the retrieved association, or NULL
 */
Association *mdi_retrieveAssociationByTransportAddress(union sockunion * fromAddress,
                                                   unsigned short fromPort,
                                                   unsigned short toPort)
{
    Association *assocr = NULL;
    Association *assocp = NULL;
    GList* result = NULL;

    assert(fromAddress);

    tmpAssoc.noOfDestAddresses = 1;
    tmpAssoc.destAddresses = g_array_new(FALSE, FALSE, sizeof(union sockunion));
    tmpAssoc.remotePort = fromPort;
    tmpAssoc.localPort = toPort;
    tmpAssoc.deleted = FALSE;

    switch (sockunion_family(fromAddress)) {
    case AF_INET:
        event_logi(INTERNAL_EVENT_0, "Looking for IPv4 Address %x (in NBO)", sock2ip(fromAddress));
        /* fromAddress->sa.sa_family = AF_INET; */
        break;
#ifdef HAVE_IPV6
    case AF_INET6:
        /* fromAddress->sa.sa_family = AF_INET6; */
        event_log(INTERNAL_EVENT_0, "Looking for IPv6 Address %x, check NTOHX() ! ");
        break;
#endif
    default:
        error_logi(ERROR_FATAL, "Unsupported Address Type %d in retrieveAssociationByTransportAddress()",
                   sockunion_family(fromAddress));
        return NULL;
        break;
    }

    tmpAssoc.destAddresses  = g_array_append_val(tmpAssoc.destAddresses, (*fromAddress));
    assocp = &tmpAssoc;
    event_log(INTERNAL_EVENT_0, "retrieving association by transport address from list");
    result = g_list_find_custom(AssociationList, assocp, equalAssociations);
    g_array_free(tmpAssoc.destAddresses, TRUE);

    if (result != NULL){
        assocr = (Association *)result->data;
        if (assocr->deleted) {
            event_logi(VERBOSE, "Found assoc that should be deleted, with id %u",assocr->assocId);
            assocr= NULL;
        }
        if (assocr != NULL) event_logi(VERBOSE, "Found valid assoc assoc with id %u",assocr->assocId);
        return assocr;
    } else {
        event_log(INTERNAL_EVENT_0, "association indexed by transport address not in list");
    }
    event_log(VVERBOSE, "return NULL in retrieveAssociationByTransportAddress");
    return NULL;
}

/**
 *  checkForExistingAssociations checks wether a given association is already in the list using
 *  the equality condition given by function equalAssociations.
 *  TODO : this must still be implemented. Where is it used ??????????????
 *
 *  @param assoc_new the association to be compared with the association in the list.
 *  @return      1 if was association found, else  0
 */
static int checkForExistingAssociations(Association * assoc_new)
{
    GList* result = NULL;

    if (AssociationList == NULL) {
        event_logi(VERBOSE, "checkForExistingAssociations(new_assoc = %u) AssocList not set",
            assoc_new->assocId);

        return 0;
    }
    result = g_list_find_custom(AssociationList, assoc_new, equalAssociations);

    if (result) /* then one of addresses of assoc A was in set of addresses of B */
        return 1;
    else
        return 0;
}

gboolean mdi_addressContainsLocalhost(SCTP_instance *sI, union sockunion* anAddress)
{
    gboolean result = FALSE;
    unsigned int counter;

    switch(sockunion_family(anAddress)) {
        case AF_INET:
            if (ntohl(sock2ip(anAddress)) == INADDR_LOOPBACK) {
                event_log(VVERBOSE, "Found IPv4 loopback address ! ");
                result = TRUE;
             }
             break;
#ifdef HAVE_IPV6
        case AF_INET6:
  #if defined (LINUX)
            if ( IN6_IS_ADDR_LOOPBACK( sock2ip6(anAddress))) {
  #else
            if ( IN6_IS_ADDR_LOOPBACK(&sock2ip6addr(anAddress)) ) {
  #endif
                event_log(VVERBOSE, "Found IPv6 loopback address ! ");
                result = TRUE;
            }
            break;
#endif
        default:
            break;
    }

    if (sI) {
        if (sI->numberOfLocalAddresses > 0){
            for (counter = 0; counter < sI->numberOfLocalAddresses; counter++) {
                if (adl_equal_address(anAddress,
                                      &g_array_index(sI->localAddresses,union sockunion, counter))) result = TRUE;
            }
        } else {
            if (sI->has_INADDR_ANY_set) {
                for (counter = 0; counter < myNumberOfAddresses; counter++) {
                    if (sockunion_family(&myAddressList[counter]) == AF_INET) {
                        if (adl_equal_address(anAddress, &(myAddressList[counter])) == TRUE) result = TRUE;
                    }
                }
            }
            if (sI->has_IN6ADDR_ANY_set) {
                for (counter = 0; counter < myNumberOfAddresses; counter++) {
                    if (adl_equal_address(anAddress, &(myAddressList[counter])) == TRUE) result = TRUE;
                }
            }
        }
   }

    event_logi(VVERBOSE, "Found loopback address returns %s", (result == TRUE)?"TRUE":"FALSE");

    return result;
}

gboolean mdi_checkForValidLocalAddress(union sockunion* su)
{
    gboolean found = FALSE;
    unsigned int counter;
    /* make sure, if IN(6)ADDR_ANY is specified, it is the only specified address */
    switch(sockunion_family(su)) {
        case AF_INET:
            if (sock2ip(su) == INADDR_ANY) return FALSE;
            break;
#ifdef HAVE_IPV6
        case AF_INET6:
  #if defined (LINUX)
            if (IN6_IS_ADDR_UNSPECIFIED(sock2ip6(su))) return FALSE;
  #else
            if (IN6_IS_ADDR_UNSPECIFIED(&sock2ip6addr(su))) return FALSE;
  #endif
            break;
#endif
        default:
            return FALSE;
            break;
    }
    for (counter = 0; counter < myNumberOfAddresses; counter++) {
        if (adl_equal_address(su, &(myAddressList[counter])) == TRUE) found = TRUE;
    }
    return found;
}

gboolean mdi_checkForValidLocalAddressInAssociation(Association* asok, union sockunion* su)
{
    gboolean found = FALSE;
    unsigned int counter;
    /* make sure, if IN(6)ADDR_ANY is specified, it is the only specified address */
    switch(sockunion_family(su)) {
        case AF_INET:
            if (sock2ip(su) == INADDR_ANY) return FALSE;
            break;
#ifdef HAVE_IPV6
        case AF_INET6:
  #if defined (LINUX)
            if (IN6_IS_ADDR_UNSPECIFIED(sock2ip6(su))) return FALSE;
  #else
            if (IN6_IS_ADDR_UNSPECIFIED(&sock2ip6addr(su))) return FALSE;
  #endif
            break;
#endif
        default:
            return FALSE;
            break;
    }
    if (asok->had_IN6ADDR_ANY_set) {
        for (counter = 0; counter < myNumberOfAddresses; counter++) {
            if (adl_equal_address(su, &(myAddressList[counter])) == TRUE) found = TRUE;
        }
    } else if (asok->had_INADDR_ANY_set && sockunion_family(su) == AF_INET) {
        for (counter = 0; counter < myNumberOfAddresses; counter++) {
            if (adl_equal_address(su, &(myAddressList[counter])) == TRUE) found = TRUE;
        }
    } else {
        for (counter = 0; counter < asok->noOfLocalAddresses; counter++) {
            if (adl_equal_address(su, &g_array_index(asok->localAddresses ,union sockunion, counter)) == TRUE) {
                found = TRUE;
            }
        }
    }
    return found;
}


gboolean mdi_supportsPRSCTP(Association* currentAssociation){
    if (currentAssociation == NULL) return FALSE;
    return (gboolean)(currentAssociation->supportsPRSCTP && currentAssociation->peerSupportsPRSCTP);
}


gboolean mdi_librarySupportsPRSCTP()
{
    event_logi(VVERBOSE, "this library %s support FW-TSN/PR-SCTP",  librarySupportsPRSCTP==TRUE?"does":"does not");
    return librarySupportsPRSCTP;
}

/*------------------- Internal port management Functions -----------------------------------------*/

/**
 * seizePort return a free port number, which is larger than 1024.
 * return 0 if all ports are used !
 * @return free port.
 */
static unsigned short seizePort(void)
{
    unsigned short nextPort = 0;

    if (numberOfSeizedPorts >= 0xFBFF) return 0x0000;

    nextPort = (unsigned short) (adl_random() % 0xFFFF);

    while ((portsSeized[nextPort]) || (nextPort < 0x0400)) {
        nextPort = (unsigned short) (adl_random() % 0xFFFF);
    }

    numberOfSeizedPorts++;
    portsSeized[nextPort] = 1;

    return nextPort;
}



/**
 * releasePort frees a previously used port.
 * @param portSeized port that is to be freed.
 */
static void releasePort(unsigned short releasePort)
{
    if (portsSeized[releasePort] == 0) {
        error_log(ERROR_MINOR, "Warning: release of port that is not seized");
        return;
    }
    numberOfSeizedPorts--;
    portsSeized[releasePort] = 0;
}



/*------------------- Other Internal Functions ---------------------------------------------------*/

/**
 *  mdi_removeAssociationData removes the association from the list of associations,
 *  frees all data allocated for it and calls <moduleprefix>_delete*(...) function for
 *  all modules.
 *  @param assoc  pointer to the association to be deleted.
 */
static void mdi_removeAssociationData(Association * assoc)
{
    if (assoc != NULL) {
        event_logiii(INTERNAL_EVENT_0, "Deleting association %x (pointer: %x, rtag=%u) ",
            assoc->assocId, assoc, assoc->tagRemote);

        /* free module data */
        if (assoc->tagRemote != 0) {
            /* association init was already completed */
            fc_delete_flowcontrol(assoc);
            rtx_delete_reltransfer(assoc);
            rxc_delete_recvctrl(assoc);
            se_delete_stream_engine(assoc);
            asc_delete(assoc);
        }

        pm_deletePathman(assoc);
        bu_delete(assoc);
        sci_deleteSCTP_control(assoc);
        if (assoc->selectedPort == 1) {
            releasePort(assoc->localPort);
        }
        /* free association data */
        if (assoc->destAddresses)    g_array_free(assoc->destAddresses, TRUE);
        if (assoc->destAddressKeys)  g_array_free(assoc->destAddressKeys, TRUE);
        if (assoc->localAddresses)   g_array_free(assoc->localAddresses, TRUE);
        if (assoc->localAddressKeys) g_array_free(assoc->localAddressKeys, TRUE);
        free(assoc);
    } else {
        error_log(ERROR_FATAL, "mdi_removeAssociationData: Null Pointer passed !");
    }
    return;
}                               /* end: mdi_removeAssociationData */


/*
 * after sctpInstance and currentAssociation have been set for an incoming packet,
 * this function will return TRUE, if a packet is destined for this association or
 * instance and may subsequently be processed, or FALSE  if it is not destined for
 * this instance.
 */
boolean mdi_destination_address_okay(SCTP_instance* sctpInstance,
                                     Association* currentAssociation,
                                     union sockunion * dest_addr)
{
    unsigned int i;
    gboolean found = FALSE;
    gboolean any_set = FALSE;

    /* this case will be specially treated after the call to mdi_destination_address_okay() */
    if (sctpInstance == NULL && currentAssociation == NULL) return TRUE;

    /* if (sctpInstance == NULL && currentAssociation == NULL) return FALSE; */
    if (currentAssociation != NULL) {
        /* search through the _association_ address list */
        /* and accept or decline */
        if (currentAssociation->had_IN6ADDR_ANY_set) {
            for (i = 0; i < myNumberOfAddresses; i++) {
                if (adl_equal_address(dest_addr, &(myAddressList[i])) == TRUE) {
                    found = TRUE;
                    break;
                }
            }
        } else if (currentAssociation->had_INADDR_ANY_set && sockunion_family(dest_addr) == AF_INET) {
            for (i = 0; i < myNumberOfAddresses; i++) {
                if (adl_equal_address(dest_addr, &(myAddressList[i])) == TRUE) {
                    found = TRUE;
                    break;
                }
            }
        } else {
            for (i=0; i< currentAssociation->noOfLocalAddresses; i++) {
                if(adl_equal_address(dest_addr,
                                     &g_array_index(currentAssociation->localAddresses ,union sockunion, i) ) == TRUE){
                    found = TRUE;
                    break;
                }
            }
        }
    } else {
        /* check whether _instance_ has INADDR_ANY */
        if (sctpInstance->has_INADDR_ANY_set == TRUE) {
            any_set = TRUE;
            /* if so, accept */
            switch(sockunion_family(dest_addr)) {
                case AF_INET:
                    return TRUE;
                    break;
                case AF_INET6:
                    return FALSE;
                    break;
                default:
                    break;

            }
        }
        if (sctpInstance->has_IN6ADDR_ANY_set == TRUE) {
            any_set = TRUE;
            /* if so, accept */
            switch(sockunion_family(dest_addr)) {
                case AF_INET:
                    return TRUE;
                    break;
                case AF_INET6:
                    return TRUE;
                    break;
                default:
                    break;

            }
        }
        if (any_set == TRUE) return FALSE;
        /* if not, search through the list */
        for (i=0; i< sctpInstance->numberOfLocalAddresses; i++) {
            if(adl_equal_address(dest_addr,
                                 &g_array_index(sctpInstance->localAddresses ,union sockunion, i) ) == TRUE){
                found = TRUE;
                break;
            }
        }
        /* and accept or decline */
    }
    event_logi(VVERBOSE, "mdi_destination_address_okay: returning FOUND == %s",(found==TRUE)?"TRUE":"FALSE");
    return found;
}


/*------------------- Functions called by the Unix-Interface (Adapation) ------------------------------*/
void
mdi_dummy_callback(gint socket_fd,
                   unsigned char *buffer,
                   int bufferLength,
                   unsigned char *hoststring,
                   unsigned short fromAddressLength)
{
    error_log(ERROR_FATAL, "DUMMY CALLBACK should never be EXECUTED !");
}

inline void mdi_reset_last(void)
{
    lastFromAddress = NULL;
    lastDestAddress = NULL;
    lastFromPort = 0;
    lastDestPort = 0;
    lastFromPathIndex = 0;
    lastFromPathKey = 0;
    lastInitiateTag = 0;
}


/**
 *  mdi_receiveMessage is the callback function of the SCTP-message distribution.
 *  It is called by the Unix-interface module when a new datagramm is received.
 *  This function also performs OOTB handling, tag verification etc.
 *  (see also RFC 2960, section 8.5.1.B)  and sends data to the bundling module of
 *  the right association
 *
 *  @param socket_fd          the socket file discriptor
 *  @param buffer             pointer to arrived datagram
 *  @param bufferlength       length of datagramm
 *  @param fromAddress        source address of DG
 *  @param portnum            bogus port number
 */
void
mdi_receiveMessage(gint socket_fd,
                   unsigned char *buffer,
                   int bufferLength,
                   union sockunion * source_addr,
                   union sockunion * dest_addr)
{
    SCTP_instance *ins = NULL;
    SCTP_instance temporary;
    Association *asok = NULL;
    GList* result = NULL;

    SCTP_message *message;
    SCTP_simple_chunk *chunk = NULL;
    SCTP_init_fixed *initChunk = NULL;
    guchar* initPtr = NULL;
    guchar* chPtr = NULL;
    guchar source_addr_string[SCTP_MAX_IP_LEN];
    guchar dest_addr_string[SCTP_MAX_IP_LEN];
    SCTP_vlparam_header* vlptr = NULL;
    
    union sockunion alternateFromAddress;
    unsigned int i=0, len = 0, chunkArray = 0, state;
    gboolean sourceAddressExists = FALSE;
    gboolean sendAbort = FALSE;
    gboolean discard = FALSE;
    gboolean initFound = FALSE, cookieEchoFound = FALSE, abortFound = FALSE;

    unsigned int addressType = 0;
    int retval = 0, supportedAddressTypes = 0;
    short shutdownCompleteCID;
    short abortCID;

    lastFromAddress = source_addr;
    lastDestAddress = dest_addr;
    lastFromPathIndex = 0;
    lastFromPathKey = 0;

    message = (SCTP_message *) buffer;

    if (!validate_datagram(buffer, bufferLength)) {
        event_log(EXTERNAL_EVENT, "received corrupted datagramm");
        mdi_reset_last();
        return;
    }
    len = bufferLength - sizeof(SCTP_common_header);

    /* save from address for response if a remote address is not available otherwise.
       For instance initAck or cookieAck. */
    lastFromPort = ntohs(message->common_header.src_port);
    lastDestPort = ntohs(message->common_header.dest_port);

    if (lastFromPort == 0 || lastDestPort == 0) {
        event_log(EXTERNAL_EVENT, "received DG with invalid (i.e. 0) ports");
        mdi_reset_last();
        return;
    }

    if (sockunion_family(dest_addr) == AF_INET) {
        addressType = SUPPORT_ADDRESS_TYPE_IPV4;
        event_log(VERBOSE, "mdi_receiveMessage: checking for correct IPV4 addresses");
        if (IN_CLASSD(ntohl(dest_addr->sin.sin_addr.s_addr)))            discard = TRUE;
        if (IN_EXPERIMENTAL(ntohl(dest_addr->sin.sin_addr.s_addr)))      discard = TRUE;
        if (IN_BADCLASS(ntohl(dest_addr->sin.sin_addr.s_addr)))          discard = TRUE;
        if (INADDR_ANY == ntohl(dest_addr->sin.sin_addr.s_addr))         discard = TRUE;
        if (INADDR_BROADCAST == ntohl(dest_addr->sin.sin_addr.s_addr))   discard = TRUE;

        if (IN_CLASSD(ntohl(source_addr->sin.sin_addr.s_addr)))          discard = TRUE;
        if (IN_EXPERIMENTAL(ntohl(source_addr->sin.sin_addr.s_addr)))    discard = TRUE;
        if (IN_BADCLASS(ntohl(source_addr->sin.sin_addr.s_addr)))        discard = TRUE;
        if (INADDR_ANY == ntohl(source_addr->sin.sin_addr.s_addr))       discard = TRUE;
        if (INADDR_BROADCAST == ntohl(source_addr->sin.sin_addr.s_addr)) discard = TRUE;
    } else
#ifdef HAVE_IPV6
            if (sockunion_family(dest_addr) == AF_INET6) {
        addressType = SUPPORT_ADDRESS_TYPE_IPV6;
        event_log(VERBOSE, "mdi_receiveMessage: checking for correct IPV6 addresses");
#if defined (LINUX)
        if (IN6_IS_ADDR_UNSPECIFIED(&(dest_addr->sin6.sin6_addr.s6_addr)))  discard = TRUE;
        if (IN6_IS_ADDR_MULTICAST(&(dest_addr->sin6.sin6_addr.s6_addr)))    discard = TRUE;
        if (IN6_IS_ADDR_LINKLOCAL(&(dest_addr->sin6.sin6_addr.s6_addr)))    discard = TRUE;

        if (IN6_IS_ADDR_UNSPECIFIED(&(source_addr->sin6.sin6_addr.s6_addr)))discard = TRUE;
        if (IN6_IS_ADDR_MULTICAST(&(source_addr->sin6.sin6_addr.s6_addr)))  discard = TRUE;
        if (IN6_IS_ADDR_LINKLOCAL(&(source_addr->sin6.sin6_addr.s6_addr)))  discard = TRUE;
        if (
            (!IN6_IS_ADDR_LOOPBACK(&(source_addr->sin6.sin6_addr.s6_addr))) &&
            IN6_ARE_ADDR_EQUAL(&(source_addr->sin6.sin6_addr.s6_addr),
                                &(dest_addr->sin6.sin6_addr.s6_addr)))      discard = TRUE;
#else
        if (IN6_IS_ADDR_UNSPECIFIED(&(dest_addr->sin6.sin6_addr)))          discard = TRUE;
        if (IN6_IS_ADDR_MULTICAST(&(dest_addr->sin6.sin6_addr)))            discard = TRUE;
        if (IN6_IS_ADDR_LINKLOCAL(&(dest_addr->sin6.sin6_addr)))            discard = TRUE;

        if (IN6_IS_ADDR_UNSPECIFIED(&(source_addr->sin6.sin6_addr)))        discard = TRUE;
        if (IN6_IS_ADDR_MULTICAST(&(source_addr->sin6.sin6_addr)))          discard = TRUE;
        if (IN6_IS_ADDR_LINKLOCAL(&(source_addr->sin6.sin6_addr)))          discard = TRUE;
        if (
            (!IN6_IS_ADDR_LOOPBACK(&(source_addr->sin6.sin6_addr))) &&
            IN6_ARE_ADDR_EQUAL(&(source_addr->sin6.sin6_addr),
                                &(dest_addr->sin6.sin6_addr)))              discard = TRUE;

#endif
    } else
#endif
            {
        error_log(ERROR_FATAL, "mdi_receiveMessage: Unsupported AddressType Received !");
        discard = TRUE;
    }
    adl_sockunion2str(source_addr, source_addr_string, SCTP_MAX_IP_LEN);
    adl_sockunion2str(dest_addr, dest_addr_string, SCTP_MAX_IP_LEN);
    event_logiiiii(EXTERNAL_EVENT,
                  "mdi_receiveMessage : len %d, sourceaddress : %s, src_port %u,dest: %s, dest_port %u",
                  bufferLength, source_addr_string, lastFromPort, dest_addr_string,lastDestPort);

    if (discard == TRUE) {
        event_log(INTERNAL_EVENT_0, "mdi_receiveMessage: discarding packet for incorrect addresses");
        mdi_reset_last();
        return;
    }

    /* Retrieve association from list  */
    asok = mdi_retrieveAssociationByTransportAddress(lastFromAddress, lastFromPort,lastDestPort);

    if (asok != NULL) {
        /* meaning we MUST have an instance */
        ins = asok->sctpInstance;
    } else {
        /* OK - if this packet is for a server, we will find an SCTP instance, that shall
           handle it (i.e. we have the SCTP instance's localPort set and it matches the
           packet's destination port */
        temporary.localPort = lastDestPort;
        temporary.numberOfLocalAddresses = 1;
        temporary.has_INADDR_ANY_set = FALSE;
        temporary.has_IN6ADDR_ANY_set = FALSE;
        temporary.localAddresses  = g_array_new(FALSE, FALSE, sizeof(union sockunion));
        temporary.localAddresses  = g_array_append_val(temporary.localAddresses, (*dest_addr));
        temporary.supportedAddressTypes = addressType;
                                                                                    
        event_logii(VERBOSE, "Searching for instance with address %x, port %u in instance-list",
            dest_addr->sin.sin_addr, lastDestPort);
        result = g_list_find_custom(InstanceList, &temporary, &checkForAddressInInstance);

        g_array_free(temporary.localAddresses, TRUE);

        if (result == NULL) {
            event_logi(VERBOSE, "Couldn't find SCTP Instance for Port %u and Address in List !",lastDestPort);
            /* may be an an association that is a client (with instance port 0) */
            ins = NULL;
#ifdef HAVE_IPV6
            supportedAddressTypes = SUPPORT_ADDRESS_TYPE_IPV6 | SUPPORT_ADDRESS_TYPE_IPV4;
#else
            supportedAddressTypes = SUPPORT_ADDRESS_TYPE_IPV4;
#endif
        } else {
            ins = result->data;
            supportedAddressTypes = ins->supportedAddressTypes;
        }
    }

    if (mdi_destination_address_okay(ins, asok, dest_addr) == FALSE) {
         event_log(VERBOSE, "mdi_receiveMsg: this packet is not for me, DISCARDING !!!");
         mdi_reset_last();
         return;
    }

    lastInitiateTag =   ntohl(message->common_header.verification_tag);
    chunk =             (SCTP_simple_chunk *) &message->sctp_pdu[0];
    chunkArray = rbu_scanPDU(message->sctp_pdu, len);
    
    /* now check possible other addresses in INIT or INIT ACK chunks, that could match */
    if (asok == NULL) {
        if ((initPtr = rbu_findChunk(message->sctp_pdu, len, CHUNK_INIT)) != NULL) {
            retval = 0; i = 1;
            do {
                retval = rbu_findAddress(initPtr, i, &alternateFromAddress, supportedAddressTypes);
                if (retval == 0) {
                    asok = mdi_retrieveAssociationByTransportAddress(&alternateFromAddress,
                                                                 lastFromPort,lastDestPort);
                }
                i++;
            } while (asok == NULL && retval == 0);
        }
        if ((initPtr = rbu_findChunk(message->sctp_pdu, len, CHUNK_INIT_ACK)) != NULL) {
            retval = 0; i = 1;
            do {
                retval = rbu_findAddress(initPtr, i, &alternateFromAddress, supportedAddressTypes);
                if (retval == 0) {
                    asok = mdi_retrieveAssociationByTransportAddress(&alternateFromAddress,
                                                                 lastFromPort,lastDestPort);
                }
                i++;
            } while (asok == NULL && retval == 0);
        }
        if (asok != NULL) {
            event_log(VERBOSE, "mdi_receiveMsg: found association from INIT (ACK) CHUNK");
            sourceAddressExists = TRUE;
        } else {
            event_log(VERBOSE, "mdi_receiveMsg: found NO association from INIT (ACK) CHUNK");
        }

    }

    /* check whether chunk is illegal or not (see section 3.1 of RFC 2960) */
    if ( ((rbu_datagramContains(CHUNK_INIT, chunkArray) == TRUE) && (chunkArray != (1 << CHUNK_INIT))) ||
         ((rbu_datagramContains(CHUNK_INIT_ACK, chunkArray) == TRUE) && (chunkArray != (1 << CHUNK_INIT_ACK))) ||
         ((rbu_datagramContains(CHUNK_SHUTDOWN_COMPLETE, chunkArray) == TRUE) && (chunkArray != (1 << CHUNK_SHUTDOWN_COMPLETE)))
       ){

        event_log(EXTERNAL_EVENT, "mdi_receiveMsg: discarding illegal packet (chunks were not alone....... :-)");
        /* silently discard, chunks */
        mdi_reset_last();
        return;
    }

    /* check if sctp-message belongs to an existing association */
    if (asok == NULL) {
        event_log(VVERBOSE, "mdi_receiveMsg: Association==NULL, start scanning !");
        /* This is not very elegant, but....only used when assoc is being build up, so :-D */
        if (rbu_datagramContains(CHUNK_ABORT, chunkArray) == TRUE) {
            event_log(INTERNAL_EVENT_0, "mdi_receiveMsg: Found ABORT chunk, discarding it !");
            mdi_reset_last();
            return;
        }
        if (rbu_datagramContains(CHUNK_SHUTDOWN_ACK,chunkArray) == TRUE) {
            event_log(INTERNAL_EVENT_0, "mdi_receiveMsg: Got SHUTDOWN_ACK chunk, send SHUTDOWN_COMPLETE !");
            /* section 8.4.5 : return SHUTDOWN_COMPLETE with peers veri-tag and T-Bit set */
            shutdownCompleteCID = ch_makeSimpleChunk(CHUNK_SHUTDOWN_COMPLETE, FLAG_NO_TCB);
            bu_put_Ctrl_Chunk(asok, ch_chunkString(shutdownCompleteCID), 0);
            bu_unlock_sender(asok, 0);
            /* should send it to last address */
            bu_sendAllChunks(asok, 0);
            /* free abort chunk */
            ch_deleteChunk(shutdownCompleteCID);
            /* send an ABORT with peers veri-tag, set T-Bit */
            event_log(VERBOSE, "mdi_receiveMsg: sending CHUNK_SHUTDOWN_COMPLETE  ");
            mdi_reset_last();
            return;
        }
        if (rbu_datagramContains(CHUNK_SHUTDOWN_COMPLETE, chunkArray) == TRUE) {
            event_log(INTERNAL_EVENT_0, "mdi_receiveMsg: Got SHUTDOWN_COMPLETE, discarding it !");
            mdi_reset_last();
            return;
        }
        if (rbu_datagramContains(CHUNK_COOKIE_ACK, chunkArray) == TRUE) {
            event_log(INTERNAL_EVENT_0, "mdi_receiveMsg: Found COOKIE_ACK chunk, discarding it !");
            mdi_reset_last();
            return;
        }
        /* section 8.4.7) : Discard the datagram, if it contains a STALE-COOKIE ERROR */
        if (rbu_scanDatagramForError(message->sctp_pdu, len, ECC_STALE_COOKIE_ERROR) == TRUE) {
            event_log(INTERNAL_EVENT_0, "mdi_receiveMsg: Got STALE COOKIE ERROR, discarding it !");
            mdi_reset_last();
            return;
        }

        if ((initPtr = rbu_findChunk(message->sctp_pdu, len, CHUNK_INIT)) != NULL) {
            if (ins != NULL) {
                if (lastDestPort != ins->localPort || ins->localPort == 0) {
                    /* destination port is not the listening port of this this SCTP-instance. */
                    event_log(INTERNAL_EVENT_0, "mdi_receiveMsg: got INIT Message, but dest. port does not fit -> ABORT");
                    sendAbort = TRUE;
                } else {
                     event_log(INTERNAL_EVENT_0, "mdi_receiveMsg: INIT Message - processing it !");
                }
                event_logi(VERBOSE, "setting lastInitiateTag to %x ", lastInitiateTag);
                if ((vlptr = (SCTP_vlparam_header*)rbu_scanInitChunkForParameter(initPtr, VLPARAM_HOST_NAME_ADDR)) != NULL) {
                        sendAbort = TRUE;
                }

            } else {    /* we do not have an instance up listening on that port-> ABORT him */
                event_log(INTERNAL_EVENT_0, "mdi_receiveMsg: got INIT Message, but no instance found -> ABORT");
                sendAbort = TRUE;
                event_logi(VERBOSE, "setting lastInitiateTag to %x ", lastInitiateTag);
            }
            initChunk = ((SCTP_init_fixed *) & ((SCTP_init *) message->sctp_pdu)->init_fixed);
            lastInitiateTag = ntohl(initChunk->init_tag);

        } else if (rbu_datagramContains(CHUNK_COOKIE_ECHO, chunkArray) == TRUE) {
            if (ins != NULL) {
                if (lastDestPort != ins->localPort || ins->localPort == 0) {
                    /* destination port is not the listening port of this this SCTP-instance. */
                    event_log(INTERNAL_EVENT_0,
                              "mdi_receiveMsg: got COOKIE_ECHO, but dest. port does not fit -> ABORT");
                    sendAbort = TRUE;
                } else {
                    event_log(INTERNAL_EVENT_0, "mdi_receiveMsg: COOKIE_ECHO Message - processing it !");
                }
            } else { /* Instance == NULL */
                event_log(INTERNAL_EVENT_0,
                          "mdi_receiveMsg: got COOKIE ECHO Message, but no instance found -> IGNORE");
                mdi_reset_last();
                return;
            }
        } else if ((chPtr = rbu_findChunk(message->sctp_pdu, len, CHUNK_ASCONF)) != NULL) {
            /* get address from ASCONF chunk */
            retval = asc_getAdressFromChunk((SCTP_simple_chunk*) chPtr, &alternateFromAddress);
                        
            event_log(VERBOSE, "mdi_receiveMsg: searching for alternate address in ASCONF chunk ");

            asok = mdi_retrieveAssociationByTransportAddress(&alternateFromAddress,lastFromPort,lastDestPort);
            
            if (asok == NULL) {
                event_log(INTERNAL_EVENT_0,
                          "mdi_receiveMsg: address NOT FOUND: got ASCONF chunk as OOTB -> ABORT (OOTB - see section 8.4.8) ");
                sendAbort = TRUE;
            }
        } else {
            /* section 8.4.8) send an ABORT with peers veri-tag, set T-Bit */
                event_log(INTERNAL_EVENT_0,
                          "mdi_receiveMsg: send ABORT -> message ignored (OOTB - see section 8.4.8) ");
                sendAbort = TRUE;
        }

    } else { /* i.e. if (asok != NULL) */

        /**
         * If the association exists, both ports of the message must be equal to the ports
         * of the association and the source address must be in the addresslist of the peer
         * check src- and dest-port and source address
         */
        if (lastFromPort != asok->remotePort || lastDestPort != asok->localPort) {
            event_logiiii(INTERNAL_EVENT_0,
                          "port mismatch in received DG (lastFromPort=%u, assoc->remotePort=%u, lastDestPort=%u, assoc->localPort=%u ",
                          lastFromPort, asok->remotePort, lastDestPort, asok->localPort);
            mdi_reset_last();
            return;
        }

        if (ins == NULL) {
            ins = asok->sctpInstance;
            if (ins == NULL) {
                error_log(ERROR_FATAL, "We have an Association, but no Instance, FIXME !");
                mdi_reset_last();
                return;
            }
        }
        /* check if source address is in address list of this association. */
        if (sourceAddressExists == FALSE) {
            for (i = 0; i < asok->noOfDestAddresses; i++) {
                if (adl_equal_address
                    (&g_array_index(asok->destAddresses,union sockunion, i), lastFromAddress) == TRUE) {
                    sourceAddressExists = TRUE;
                    break;
                }
            }
        }
        if (!sourceAddressExists) {
            event_log(INTERNAL_EVENT_0,
                      "source address of received DG is not in the destination addresslist");
            mdi_reset_last();
            return;
        }
        lastFromPathIndex = i;
        lastFromPathKey = g_array_index(asok->destAddressKeys, unsigned int, i);
 
        /* check for verification tag rules --> see section 8.5 */
        if ((initPtr = rbu_findChunk(message->sctp_pdu, len, CHUNK_INIT)) != NULL) {
            initFound = TRUE;
            if (lastInitiateTag != 0) {
                event_log(VERBOSE, "mdi_receiveMsg: scan found INIT, lastInitiateTag!=0, returning");
                mdi_reset_last();
                return;
            }
            initChunk = ((SCTP_init_fixed *) & ((SCTP_init *) message->sctp_pdu)->init_fixed);
            /* make sure, if you send an ABORT later on (i.e. when peer requests 0 streams),
             * you pick the right tag */
            lastInitiateTag = ntohl(initChunk->init_tag);
            event_logi(VVERBOSE, "Got an INIT CHUNK with initiation-tag %u", lastInitiateTag);
            if ((vlptr = (SCTP_vlparam_header*)rbu_scanInitChunkForParameter(initPtr, VLPARAM_HOST_NAME_ADDR)) != NULL) {
                sendAbort = TRUE;
            }
        }

        if (rbu_datagramContains(CHUNK_ABORT, chunkArray) == TRUE) {
            /* accept my-tag or peers tag, else drop packet */
            event_log(VERBOSE, "mdi_receiveMsg: scan found ABORT");
            if ((lastInitiateTag != asok->tagLocal &&
                 lastInitiateTag != asok->tagRemote) || initFound == TRUE) {
                mdi_reset_last();
                return;
            }
            abortFound = TRUE;
        }
        if (rbu_datagramContains(CHUNK_SHUTDOWN_COMPLETE, chunkArray) == TRUE) {
            /* accept my-tag or peers tag, else drop packet */
            /* TODO : make sure that if it is the peer's tag also T-Bit is set */
            event_log(VERBOSE, "mdi_receiveMsg: scan found SHUTDOWN_COMPLETE");
            if ((lastInitiateTag != asok->tagLocal &&
                 lastInitiateTag != asok->tagRemote) || initFound == TRUE) {
                mdi_reset_last();
                return;
            }
        }
        if (rbu_datagramContains(CHUNK_SHUTDOWN_ACK, chunkArray) == TRUE) {
            event_log(VERBOSE, "mdi_receiveMsg: scan found SHUTDOWN_ACK");
            if (initFound == TRUE) {
                mdi_reset_last();
                return;
            }
            state = sci_getState(asok);
            if (state == COOKIE_ECHOED || state == COOKIE_WAIT) {
                /* see also section 8.5.E.) treat this like OOTB packet */
                event_logi(EXTERNAL_EVENT,
                           "mdi_receive_message: shutdownAck in state %u, send SHUTDOWN_COMPLETE ! ",
                           state);
                shutdownCompleteCID = ch_makeSimpleChunk(CHUNK_SHUTDOWN_COMPLETE, FLAG_NO_TCB);
                bu_put_Ctrl_Chunk(asok, ch_chunkString(shutdownCompleteCID), 0);
                bu_sendAllChunks(asok, 0);
                ch_deleteChunk(shutdownCompleteCID);
                mdi_reset_last();
                return;
            }
        }
        if (rbu_datagramContains(CHUNK_COOKIE_ECHO, chunkArray) == TRUE) {
           event_log(VERBOSE, "mdi_receiveMsg: scan found COOKIE_ECHO");
           cookieEchoFound = TRUE;
        }
        if ((initPtr = rbu_findChunk(message->sctp_pdu, len, CHUNK_INIT_ACK)) != NULL) {

            if ((vlptr = (SCTP_vlparam_header*)rbu_scanInitChunkForParameter(initPtr, VLPARAM_HOST_NAME_ADDR)) != NULL) {
                /* actually, this does not make sense...anyway: kill assoc, and notify user */
                scu_abort(asok, ECC_UNRECOGNIZED_PARAMS, ntohs(vlptr->param_length), (guchar*)vlptr);
                mdi_reset_last();
                return;
            }
        }

        if (!cookieEchoFound && !initFound && !abortFound && lastInitiateTag != asok->tagLocal) {
            event_logii(VERBOSE,
                        "Tag mismatch in receive DG, received Tag = %u, local Tag = %u -> discarding",
                        lastInitiateTag, asok->tagLocal);
            mdi_reset_last();
            return;

        }
    }

    if (sendAbort == TRUE) {
        if (sendAbortForOOTB == FALSE) {
            event_log(VERBOSE, "mdi_receiveMsg: sendAbortForOOTB==FALSE -> Discarding MESSAGE: not sending ABORT");
            /* and discard that packet */
            mdi_reset_last();
            return;
        }
        /* make and send abort message */
        abortCID = ch_makeSimpleChunk(CHUNK_ABORT, FLAG_NO_TCB);
        bu_put_Ctrl_Chunk(asok, ch_chunkString(abortCID), 0);
        /* should send it to last address */
        bu_unlock_sender(asok, 0);
        bu_sendAllChunks(asok, 0);
        /* free abort chunk */
        ch_deleteChunk(abortCID);
        /* send an ABORT with peers veri-tag, set T-Bit */
        event_log(VERBOSE, "mdi_receiveMsg: sending ABORT with T-Bit");
        mdi_reset_last();
        /* and discard that packet */
        return;
    }

    event_logi(INTERNAL_EVENT_0, "mdi_receive_message: Processing received SCTP datagram (len: %u)",bufferLength);

    /* forward DG to bundling */
    rbu_rcvDatagram(ins, asok, lastFromPathIndex, lastFromPathKey, message->sctp_pdu, bufferLength - sizeof(SCTP_common_header));

    mdi_reset_last();

}                               /* end: mdi_receiveMessage */


int mdi_addressChange(union sockunion* address, gboolean added)
{
    int count, result;
    unsigned int listSize = g_list_length(InstanceList);
    unsigned int correlationId;
    gboolean updateNeeded = FALSE;
    GList* tmp;
    SCTP_instance* ins;
    Association* asoc;

    event_logi(INTERNAL_EVENT_0, "mdi_addressChange: %s an address",(added==TRUE)?"Add":"Delete");

    /* search for an Instance with INADDR_ANY - if one is found we must update addresses */
    for (count = 0; count < listSize; count++) {
        tmp = g_list_nth(InstanceList, count);
        ins = (SCTP_instance*)tmp->data;
        if (ins) {
            if (ins->numberOfLocalAddresses == 0) {
                updateNeeded =  TRUE;
                break;
            }
        }
    }
    event_logi(VVERBOSE, "mdi_addressChange: instance %s an address update", (updateNeeded==TRUE)?"needs":"does NOT need");
    if (updateNeeded) result = mdi_addressUpdate();
    /* trigger ASCONF ADD/DEL IP for all associations that stem from
       an INADDR_ANY.... */
    listSize = g_list_length(AssociationList);
    for (count = 0; count < listSize; count++) {
        event_logii(VVERBOSE, "mdi_addressChange: Starting with Association %d from %u", count, listSize);
        tmp = g_list_nth(AssociationList, count);
        asoc = (Association*)tmp->data;
        if (asoc) {
            if (asoc->had_IN6ADDR_ANY_set || (asoc->had_INADDR_ANY_set && sockunion_family(address)==AF_INET)) {
                event_logi(VERBOSE, "Adding/deleting Address to association %u", asoc->assocId);
                if (added == TRUE) {
                    result = asc_addIP(asoc, address, &correlationId);
                    event_logii(VERBOSE, "Adding Address to association %u, corrId: %u", asoc->assocId, correlationId);
                } else {
                    result = asc_deleteIP(asoc, address, &correlationId);
                    event_logii(VERBOSE, "Deleting Address from association %u, corrId: %u", asoc->assocId, correlationId);
                }

            }    
        }
    }
    return SCTP_SUCCESS;
}

/*------------------- Functions called by the ULP ------------------------------------------------*/
/*------------------- Prototypes are defined in sctp.h -------------------------------------------*/


/**
 * Function returns coded library version as result. This unsigned integer
 * contains the major version in the upper 16 bits, and the minor version in
 * the lower 16 bits.
 * @return library version, or 0 (i.e. zero) as error !
 */
unsigned int sctp_getLibraryVersion(void)
{
    event_logii(VVERBOSE, "You are using SCTPLIB version %u.%u ",SCTP_MAJOR_VERSION, SCTP_MINOR_VERSION);
    return (unsigned int)(SCTP_MAJOR_VERSION << 16 | SCTP_MINOR_VERSION);
}




/**
 * Function that needs to be called in advance to all library calls.
 * It initializes all file descriptors etc. and sets up some variables
 * @return 0 for success, 1 for adaptation level error, -9 for already called
 * (i.e. the function has already been called), -2 for insufficient rights.
 */
int sctp_initLibrary(void)
{
    unsigned int i, result, sfd = -1, maxMTU=0;
    /* initialize the output of event/error-log functions */
    ENTER_LIBRARY("sctp_initLibrary");

    if (sctpLibraryInitialized == TRUE) {
        LEAVE_LIBRARY("sctp_initLibrary");
        return SCTP_LIBRARY_ALREADY_INITIALIZED;
    }
    read_tracelevels();

#if defined(HAVE_GETEUID)
    /* check privileges. Must be root or setuid-root for now ! */
     if (geteuid() != 0) {
        error_log(ERROR_MAJOR, "You must be root to use the SCTPLIB-functions (or make your program SETUID-root !).");
        LEAVE_LIBRARY("sctp_initLibrary");
        return SCTP_INSUFFICIENT_PRIVILEGES;
    }
#endif

    event_log(EXTERNAL_EVENT, "sctp_initLibrary called");
    result = adl_init_adaptation_layer(&myRWND);

    if (result != 0) {
        LEAVE_LIBRARY("sctp_initLibrary");
        return SCTP_SPECIFIC_FUNCTION_ERROR;
    }

    /* initialize ports seized -- see comments above !!! */
    for (i = 0; i < 0x10000; i++) portsSeized[i] = 0;
    numberOfSeizedPorts = 0;

    /* initialize bundling, i.e. the common buffer for sending chunks
       when no association exists. */
    bu_init_bundling();

    /* this block is to be executed only once for the lifetime of the library */
    key_operation(KEY_INIT);

    /* we might need to replace this socket !*/
    sfd = adl_get_sctpv4_socket();

    if (adl_gatherLocalAddresses(&myAddressList, (int*)&myNumberOfAddresses, sfd, TRUE, &maxMTU, &min_MTU, flag_Default) == FALSE) {
        LEAVE_LIBRARY("sctp_initLibrary");
        return SCTP_SPECIFIC_FUNCTION_ERROR;
    }
    event_logi(VERBOSE, "min mtu is %u", min_MTU);
    
    sctpLibraryInitialized = TRUE;
    LEAVE_LIBRARY("sctp_initLibrary");
    return SCTP_SUCCESS;
}

/**
 *  sctp_registerInstance is called to initialize one SCTP-instance.
 *  Each Adaption-Layer of the ULP must create its own SCTP-instance, and
 *  define and register appropriate callback functions.
 *  An SCTP-instance may define an own port, or zero here ! Servers and clients
 *  that care for their source port must chose a port, clients that do not really
 *  care which source port they use, chose ZERO, and have the implementation chose
 *  a free source port.
 *
 *  @param port                   wellknown port of this sctp-instance
 *  @param noOfLocalAddresses     number of local addresses
 *  @param localAddressList       local address list (pointer to a string-array)
 *  @param ULPcallbackFunctions   call back functions for primitives passed from sctp to ULP
 *  @return     instance name of this SCTP-instance or 0 in case of errors, or error code
 */

int sctp_registerInstance(unsigned short localPort,
                            unsigned short noOfInStreams,
                            unsigned short noOfOutStreams,
                            unsigned int   noOfLocalAddresses,
                            unsigned char  localAddressList[][SCTP_MAX_IP_LEN],
                            SCTP_ulpCallbacks ULPcallbackFunctions)
{
    SCTP_instance* sctpInstance;
    int adl_rscb_code, result;
    union sockunion su;
    int i;
    unsigned int initialKey = 1;
    GList* list_result = NULL;

    gboolean with_ipv4 = FALSE;
#ifdef HAVE_IPV6
    gboolean with_ipv6 = FALSE;
#endif

    ENTER_LIBRARY("sctp_registerInstance");
    CHECK_LIBRARY;

    event_log(EXTERNAL_EVENT, "sctp_registerInstance called");

/* check user input */
    if ((noOfInStreams==0) || (noOfOutStreams == 0) || (noOfLocalAddresses == 0) || (localAddressList == NULL)) {
            error_log(ERROR_MAJOR, "Parameter Problem in sctp_registerInstance - Error !");
            LEAVE_LIBRARY("sctp_registerInstance");
            return SCTP_PARAMETER_PROBLEM;
    }

    for (i=0; i< noOfLocalAddresses; i++) {
        if (adl_str2sockunion((localAddressList[i]), &su) < 0) {
            error_logi(ERROR_MAJOR, "Address Error in sctp_registerInstance(%s)", (localAddressList[i]));
            LEAVE_LIBRARY("sctp_registerInstance");
            return SCTP_PARAMETER_PROBLEM;
        } else {
            if (su.sa.sa_family == AF_INET) with_ipv4 = TRUE;
#ifdef HAVE_IPV6
            if (su.sa.sa_family == AF_INET6) with_ipv6 = TRUE;
#endif
        }
    }
    event_logi(VERBOSE, "sctp_registerInstance : with_ipv4 : %s ",(with_ipv4==TRUE)?"TRUE":"FALSE" );
#ifdef HAVE_IPV6
    /* if not IPv6 callback must be registered too ! */
    event_logi(VERBOSE, "sctp_registerInstance : with_ipv6: %s ",(with_ipv6==TRUE)?"TRUE":"FALSE" );
#endif


    if ((with_ipv4 != TRUE)
#ifdef HAVE_IPV6
            && (with_ipv6 != TRUE)
#endif
                                 ) {
            error_log(ERROR_MAJOR, "No valid address in sctp_registerInstance()");
            LEAVE_LIBRARY("sctp_registerInstance");
            return SCTP_PARAMETER_PROBLEM;
    }

/* create and populate new instance */
    sctpInstance = (SCTP_instance *) malloc(sizeof(SCTP_instance));
    if (!sctpInstance) {
        error_log_sys(ERROR_MAJOR, errno);
        LEAVE_LIBRARY("sctp_registerInstance");
        return SCTP_OUT_OF_RESOURCES;
    }

    sctpInstance->supportsPRSCTP = librarySupportsPRSCTP; /* global library variable */
    sctpInstance->supportsADDIP  = librarySupportsADDIP;  /* global library variable */

    sctpInstance->localPort      = localPort;
    sctpInstance->noOfInStreams  = noOfInStreams;
    sctpInstance->noOfOutStreams = noOfOutStreams;
    sctpInstance->has_INADDR_ANY_set    = FALSE;
    sctpInstance->has_IN6ADDR_ANY_set   = FALSE;

    /* check if we got INADDR_ANY from the ULP */
    if (noOfLocalAddresses == 1) {
        adl_str2sockunion((localAddressList[0]), &su);
        switch(sockunion_family(&su)) {
            case AF_INET:
                if (sock2ip(&su) == INADDR_ANY){
                    sctpInstance->has_INADDR_ANY_set = TRUE;
                    with_ipv4 = TRUE;
#ifdef HAVE_IPV6
                    with_ipv6 = FALSE;
#endif
                }
                break;
#ifdef HAVE_IPV6
            case AF_INET6:
  #if defined (LINUX)
                if (IN6_IS_ADDR_UNSPECIFIED(sock2ip6(&su))) {
  #else
                if (IN6_IS_ADDR_UNSPECIFIED(&sock2ip6addr(&su))) {
  #endif
                    with_ipv4 = TRUE;
                    with_ipv6 = TRUE;
                    sctpInstance->has_IN6ADDR_ANY_set = TRUE;
                }
                break;
#endif
            default:
                /* user gave us bad address */
                free(sctpInstance);
                LEAVE_LIBRARY("sctp_registerInstance");
                return SCTP_PARAMETER_PROBLEM;
                break;
        }
    }

    sctpInstance->supportedAddressTypes = 0;
    if (with_ipv4) sctpInstance->supportedAddressTypes |= SUPPORT_ADDRESS_TYPE_IPV4;
#ifdef HAVE_IPV6
    if (with_ipv6) sctpInstance->supportedAddressTypes |= SUPPORT_ADDRESS_TYPE_IPV6;
#endif

    /* put all normal addresses in our array of addresses */
    if (sctpInstance->has_INADDR_ANY_set == FALSE && sctpInstance->has_IN6ADDR_ANY_set == FALSE) {
        for (i=0; i< noOfLocalAddresses; i++) {
            adl_str2sockunion(localAddressList[i], &su);
            if (mdi_checkForValidLocalAddress(&su) == FALSE){
                free(sctpInstance);
                error_log(ERROR_MAJOR, "User gave incorrect address !");
                LEAVE_LIBRARY("sctp_registerInstance");
                return SCTP_WRONG_ADDRESS;
            }
        }
        sctpInstance->numberOfLocalAddresses = noOfLocalAddresses;
        sctpInstance->localAddresses   = g_array_new(FALSE, FALSE, sizeof(union sockunion));
        sctpInstance->localAddressKeys = g_array_new(FALSE, FALSE, sizeof(unsigned int));
        for (i=0; i< noOfLocalAddresses; i++) {
            adl_str2sockunion(localAddressList[i], &su);
            sctpInstance->localAddresses   = g_array_append_val(sctpInstance->localAddresses, su);
            sctpInstance->localAddressKeys = g_array_append_val(sctpInstance->localAddressKeys, initialKey);
            event_logi(VVERBOSE, "sctp_registerInstance : appended address %s to local addresses",
                        localAddressList[i]);

            initialKey++;
        }
    } else {
        sctpInstance->numberOfLocalAddresses = 0;
        sctpInstance->localAddresses = NULL;
        sctpInstance->localAddressKeys = NULL;
    }


    list_result = g_list_find_custom(InstanceList, sctpInstance, &checkForAddressInInstance);

    if (list_result) {
        if (sctpInstance->localAddresses) g_array_free(sctpInstance->localAddresses, TRUE);
        if (sctpInstance->localAddressKeys) g_array_free(sctpInstance->localAddressKeys, TRUE);
        free(sctpInstance);
        error_log(ERROR_MAJOR, "Instance already existed ! Returning error !");
        LEAVE_LIBRARY("sctp_registerInstance");
        return 0;
    }

    /* register callbacks on the raw sockets, so we receive data properly */
#ifdef HAVE_IPV6
    if (with_ipv6 && ipv6_sctp_socket==0) {
         ipv6_sctp_socket = adl_get_sctpv6_socket();
        /*
         * here some operating system specialties may kick in (i.e. opening only ONE
         * socket MIGHT be enough, provided IPv6 socket implicitly reveives IPv4 packets, too
         */
         if (ipv6_sctp_socket > 0) {
             adl_rscb_code = adl_register_socket_cb(ipv6_sctp_socket,&mdi_dummy_callback);
             if (!adl_rscb_code) error_log(ERROR_FATAL, "register ipv6 socket call back function failed");
         }
    }
    if (with_ipv6 == TRUE) {
	    ipv6_users++;
	    sctpInstance->uses_IPv6 = TRUE;
    } else {
	     sctpInstance->uses_IPv6 = FALSE;
    }

#endif
    if (with_ipv4 && sctp_socket==0) {
         sctp_socket = adl_get_sctpv4_socket();
         if (!sctp_socket) error_log(ERROR_FATAL, "IPv4 socket creation failed");
         adl_rscb_code = adl_register_socket_cb(sctp_socket,&mdi_dummy_callback);
         if (!adl_rscb_code) error_log(ERROR_FATAL, "registration of IPv4 socket call back function failed");
    }
    if (with_ipv4 == TRUE) {
	    ipv4_users++;
	    sctpInstance->uses_IPv4 = TRUE;
    } else {
            sctpInstance->uses_IPv4 = FALSE;
    }
		     

    /* fill in all other data */
    sctpInstance->sctpInstanceName = nextSctpInstanceName++;
    if (sctpInstance->sctpInstanceName <= 0) sctpInstance->sctpInstanceName = 1;

    sctpInstance->ULPcallbackFunctions = ULPcallbackFunctions;

    sctpInstance->default_rtoInitial = RTO_INITIAL;
    sctpInstance->default_validCookieLife = VALID_COOKIE_LIFE_TIME;
    sctpInstance->default_assocMaxRetransmits = ASSOCIATION_MAX_RETRANS;
    sctpInstance->default_pathMaxRetransmits = MAX_PATH_RETRANSMITS ;
    sctpInstance->default_maxInitRetransmits = MAX_INIT_RETRANSMITS;
    /* using the static variable defined after initialization of the adaptation layer */
    sctpInstance->default_myRwnd = myRWND/2;
    sctpInstance->default_delay = SACK_DELAY;
    sctpInstance->default_ipTos = IPTOS_DEFAULT;
    sctpInstance->default_rtoMin = RTO_MIN;
    sctpInstance->default_rtoMax = RTO_MAX;
    sctpInstance->default_maxSendQueue = DEFAULT_MAX_SENDQUEUE;
    sctpInstance->default_maxRecvQueue = 0;

    InstanceList = g_list_insert_sorted(InstanceList, sctpInstance, &compareInstanceNames);

    result = sctpInstance->sctpInstanceName;

    LEAVE_LIBRARY("sctp_registerInstance");
    return result;

}                               /* end: sctp_registerInstance */



int sctp_unregisterInstance(int instanceName)
{
    /*
     *  Look through the instance list, and delete instance, when
     *  found, else return error.
     */
    SCTP_instance temporary;
    SCTP_instance* instance;
    /*union sockunion* alist;  */
    /*guint16 num_addresses; */
    guint32 /*n, */ fds;
    GList* result = NULL;
    gboolean with_ipv4=FALSE;
#ifdef HAVE_IPV6
    gboolean with_ipv6=FALSE;
#endif

    ENTER_LIBRARY("sctp_unregisterInstance");
    CHECK_LIBRARY;
    event_logi(EXTERNAL_EVENT, "Removing SCTP Instance %d from list", instanceName);

    temporary.sctpInstanceName = instanceName;
    result = g_list_find_custom(InstanceList, &temporary, &compareInstanceNames);

    if (result != NULL) {
        instance =  result->data;
#ifdef HAVE_IPV6
        with_ipv6 = instance->uses_IPv6;
#endif
        with_ipv4 = instance->uses_IPv4;

        event_logi(INTERNAL_EVENT_0, "sctp_unregisterInstance: SCTP Instance %d found !!!", instanceName);
#ifdef HAVE_IPV6
        event_logi(VERBOSE, "sctp_unregisterInstance : with_ipv6: %s ",(with_ipv6==TRUE)?"TRUE":"FALSE" );
        if (with_ipv6 == TRUE) ipv6_users--;
        event_logi(VERBOSE, "sctp_unregisterInstance : ipv6_users now: %u ",ipv6_users);
#endif
        if (with_ipv4 == TRUE) ipv4_users--;
        event_logi(VERBOSE, "sctp_unregisterInstance : with_ipv4: %s ",(with_ipv4==TRUE)?"TRUE":"FALSE" );
        event_logi(VERBOSE, "sctp_unregisterInstance : ipv4_users now: %u ",ipv4_users);

        /* adjust registered callbacks */
        if (sctp_socket != 0 &&  ipv4_users == 0) {
            fds = adl_remove_poll_fd(sctp_socket);
            event_logi(VVERBOSE, "sctp_unregisterInstance : Removed IPv4 cb, registered FDs: %u ",fds);
            /* if there are no ipv4_users, deregister callback for ipv4-socket, if it was registered ! */
            sctp_socket = 0;
        }
#ifdef HAVE_IPV6
        if (ipv6_sctp_socket != 0 &&  ipv6_users == 0) {
            fds = adl_remove_poll_fd(ipv6_sctp_socket);
            /* if there are no ipv6_users, deregister callback for ipv6-socket, if it was registered ! */
            event_logi(VVERBOSE, "sctp_unregisterInstance : Removed IPv6 cb, registered FDs: %u ",fds);
            ipv6_sctp_socket = 0;
        }
#endif
        event_log(INTERNAL_EVENT_0, "FIXME: are s there till associations with my port ? I remove instance now !");

        /* loop through all assocs, and see whether they stem from this instance ! Abort them ! */

        if (instance->localAddresses)   g_array_free(instance->localAddresses, TRUE);
        if (instance->localAddressKeys) g_array_free(instance->localAddressKeys, TRUE);
        free(instance);
        InstanceList = g_list_remove(InstanceList, result->data);

        LEAVE_LIBRARY("sctp_unregisterInstance");
        return SCTP_SUCCESS;
    } else {
        event_logi(INTERNAL_EVENT_0, "SCTP Instance %d not in list", instanceName);
    }
    LEAVE_LIBRARY("sctp_unregisterInstance");
    return SCTP_INSTANCE_NOT_FOUND;

}


/**
 * This function should be called AFTER an association has indicated a
 * COMMUNICATION_LOST or a SHUTDOWN_COMPLETE, and the upper layer has
 * retrieved all data it is interested in (possibly using the currently
 * not implemented functions  sctp_receive_unsent() or sctp_receive_unacked())
 * it really removes all data belonging to the association, and removes the
 * association instance from the list, on explicit upper layer instruction !
 * @param  associationID the association ID of the assoc that shall be removed
 * @return error_code  0 for success, 1 if assoc is already gone, -1 if assocs
 *         deleted flag is not set (then assoc should be in a state different from CLOSED)
 */
int sctp_deleteAssociation(unsigned int associationID)
{
    Association *assoc;
    Association *assocFindP;
    GList* result = NULL;

    ENTER_LIBRARY("sctp_deleteAssociation");

    CHECK_LIBRARY;

    event_logi(INTERNAL_EVENT_0, "sctp_deleteAssociation: getting assoc %08x from list", associationID);

    tmpAssoc.assocId = associationID;
    tmpAssoc.deleted = FALSE;
    assocFindP = &tmpAssoc;
    assoc = NULL;

    result = g_list_find_custom(AssociationList, assocFindP, &compareAssociationIDs);
    if (result != NULL) {
        assoc = (Association *)result->data;
        if (!assoc->deleted) {
            error_log(ERROR_MAJOR, "Deleted-Flag not set, returning from sctp_deleteAssociation !");
            LEAVE_LIBRARY("sctp_deleteAssociation");
            return SCTP_SPECIFIC_FUNCTION_ERROR;
        }
        /* remove the association from the list */
        AssociationList = g_list_remove(AssociationList, assoc);
        event_log(INTERNAL_EVENT_0, "sctp_deleteAssociation: Deleted Association from list");
        /* free all association data */
        mdi_removeAssociationData(assoc);
        LEAVE_LIBRARY("sctp_deleteAssociation");
        return SCTP_SUCCESS;
    } else {
        event_logi(INTERNAL_EVENT_0, "association %08x not in list", associationID);
        LEAVE_LIBRARY("sctp_deleteAssociation");
        return SCTP_ASSOC_NOT_FOUND;
    }
    /* should not be reached */
    LEAVE_LIBRARY("sctp_deleteAssociation");
    return SCTP_SUCCESS;
}




/**
 * This function is called to setup an association.
 *  The ULP must specify the SCTP-instance to which this association belongs to.
 *  @param SCTP_InstanceName     the SCTP instance this association belongs to.
 *                               if the local port of this SCTP instance is zero, we will get a port num,
                                 else we will use the one from the SCTP instance !
 *  @param noOfOutStreams        number of output streams the ULP would like to have
 *  @param destinationAddress    destination address
 *  @param destinationPort       destination port
 *  @param ulp_data             pointer to an ULP data structure, will be passed with callbacks !
 *  @return association ID of this association, 0 in case of failures
 */
unsigned int sctp_associatex(int instanceName,
                             unsigned short noOfOutStreams,
                             unsigned char  destinationAddresses[][SCTP_MAX_IP_LEN],
                             unsigned int   noOfDestinationAddresses,
                             unsigned int   maxSimultaneousInits,
                             unsigned short destinationPort,
                             void* ulp_data)
{

    unsigned int assocID, count;
    unsigned short zlocalPort;
    union sockunion dest_su[SCTP_MAX_NUM_ADDRESSES];
    SCTP_instance temporary;
    SCTP_instance * sctpInstance = NULL;
    Association* currentAssociation = NULL;
    GList* result = NULL;
    gboolean portSeized = FALSE;

    ENTER_LIBRARY("sctp_associatex");
    ZERO_CHECK_LIBRARY;

    if (destinationPort == 0 || noOfDestinationAddresses > SCTP_MAX_NUM_ADDRESSES) {
        error_log(ERROR_MAJOR, "sctp_associate: destination port was 0, or too many addresses");
        LEAVE_LIBRARY("sctp_associate");
        return 0;
    }

    for (count = 0; count <  noOfDestinationAddresses; count++) {
        if (adl_str2sockunion(destinationAddresses[count], &dest_su[count]) < 0) {
            error_log(ERROR_MAJOR, "sctp_associate: could not convert destination adress");
            LEAVE_LIBRARY("sctp_associate");
            return 0;
        }
    }

    event_log(EXTERNAL_EVENT, "sctp_associatex called");

    event_logi(VERBOSE, "Looking for SCTP Instance %d in the list", instanceName);
    temporary.sctpInstanceName =  instanceName;
    result = g_list_find_custom(InstanceList, &temporary, &compareInstanceNames);
    if (result == NULL) {
        error_log(ERROR_MAJOR, "sctp_associatex: SCTP instance not in the list !!!");
        LEAVE_LIBRARY("sctp_associatex");
        return 0;
    }
    sctpInstance = result->data;

    /* autoselect ports for a server, else use specified local port */
    if (((SCTP_instance*)result->data)->localPort == 0) {
       zlocalPort = seizePort();
       portSeized = TRUE;
    } else {
       zlocalPort = ((SCTP_instance*)result->data)->localPort;
    }
    event_logi(VERBOSE, "Chose local port %u for associate !", zlocalPort);

    /* Create new association, and put it into the list of assocs */
    currentAssociation =  mdi_newAssociation(sctpInstance, zlocalPort,
                                             destinationPort,
                                             mdi_generateTag(),
                                             1, noOfDestinationAddresses, dest_su);

    if (currentAssociation == NULL) {
        error_log(ERROR_MAJOR, "Creation of association failed");
        LEAVE_LIBRARY("sctp_associate");
        return 0;
    }

    currentAssociation->ulp_dataptr = ulp_data;
    currentAssociation->supportsPRSCTP = librarySupportsPRSCTP;
    currentAssociation->selectedPort = (portSeized == TRUE)?1:0;

    /* call associate at SCTP-control */
    scu_associate(sctpInstance, currentAssociation,
                  noOfOutStreams,
                  ((SCTP_instance*)result->data)->noOfInStreams,
                  dest_su,
                  noOfDestinationAddresses,
                  currentAssociation->supportsPRSCTP);

    assocID = currentAssociation->assocId;

    LEAVE_LIBRARY("sctp_associatex");
    return assocID;

}   /* end: sctp_associate */

unsigned int sctp_associate(int instanceName,
                            unsigned short noOfOutStreams,
                            unsigned char  destinationAddress[SCTP_MAX_IP_LEN],
                            unsigned short destinationPort,
                            void* ulp_data)
{
    unsigned char dAddress[1][SCTP_MAX_IP_LEN];
    
    event_log(EXTERNAL_EVENT, "sctp_associate called");
    memcpy(dAddress, destinationAddress, SCTP_MAX_IP_LEN);
    
    return   sctp_associatex(instanceName,
                             noOfOutStreams,
                             dAddress,
                             1,
                             1,
                             destinationPort,
                             ulp_data);
}


/**
 * sctp_shutdown initiates the shutdown of the specified association.
 *  @param    associationID  the ID of the addressed association.
 *  @return   0 for success, 1 for error (assoc. does not exist)
 */
int sctp_shutdown(unsigned int associationID)
{
    Association* currentAssociation = NULL;
    ENTER_LIBRARY("sctp_shutdown");
    CHECK_LIBRARY;

    /* Retrieve association from list  */
    currentAssociation = mdi_getAssociationByID(associationID, FALSE);

    if (currentAssociation != NULL) {
        /* Forward shutdown to the addressed association */
        scu_shutdown(currentAssociation);
    } else {
        error_log(ERROR_MAJOR, "sctp_shutdown: addressed association does not exist");
        LEAVE_LIBRARY("sctp_shutdown");
        return SCTP_ASSOC_NOT_FOUND;
    }
    LEAVE_LIBRARY("sctp_shutdown");
    return SCTP_SUCCESS;
}         /* end: sctp_shutdown */

/**
 * sctp_abort initiates the abort of the specified association.
 * @param    associationID  the ID of the addressed association.
 * @return   0 for success, 1 for error (assoc. does not exist)
 */
int sctp_abort(unsigned int associationID, int userDataLen, void* abortUserData)
{
    Association* currentAssociation = NULL;
    /* Retrieve association from list  */
    ENTER_LIBRARY("sctp_abort");
    CHECK_LIBRARY;

    if ((userDataLen != 0 && abortUserData == NULL) || userDataLen > 200 || userDataLen < 0) {
        LEAVE_LIBRARY("sctp_abort");
        return SCTP_PARAMETER_PROBLEM;
    }
    currentAssociation = mdi_getAssociationByID(associationID, FALSE);
    if (currentAssociation != NULL) {
        /* Forward shutdown to the addressed association */
        scu_abort(currentAssociation, ECC_USER_INITIATED_ABORT, userDataLen, abortUserData);
    } else {
        error_log(ERROR_MAJOR, "sctp_abort: addressed association does not exist");
        LEAVE_LIBRARY("sctp_abort");
        return SCTP_ASSOC_NOT_FOUND;
    }
    LEAVE_LIBRARY("sctp_abort");
    return SCTP_SUCCESS;
}         /* end: sctp_abort */

/**
 * sctp_send is used by the ULP to send data chunks.
 *
 *  @param    associationID  the ID of the addressed association.
 *  @param    streamID       identifies the stream on which the chunk is sent.
 *  @param    buffer         chunk data.
 *  @param    length         length of chunk data.
 *  @param    protocolId     the payload protocol identifier
 *  @param    path_id        index of destination address, if different from primary pat, negative for primary
 *  @param    context        ULP context, i.e. a pointer that will may be retunred with certain callbacks.
                             (in case of send errors).
 *  @param    lifetime       maximum time of chunk in send queue in msecs, 0 for infinite
 *  @param    unorderedDelivery chunk is delivered to peer without resequencing, if true (==1), else ordered (==0).
 *  @param    dontBundle     chunk must not be bundled with other data chunks.
 *                           boolean, 0==normal bundling, 1==do not bundle message
 *  @return   error code     -1 for send error, 1 for association error, 0 if successful
 */
int sctp_send(unsigned int associationID,
              unsigned short streamID,
              unsigned char *buffer,
              unsigned int length,
              unsigned int protocolId,
              unsigned int pathId,      /* 0 for primary path, else key of the address to be taken */
              unsigned int lifetime,    /* 0xFFFFFFFF-> infinite, 0->no retransmit, else msecs */
              void * context,           /* use e.g. SCTP_NO_CONTEXT */
              int unorderedDelivery,    /* use constants SCTP_ORDERED_DELIVERY, SCTP_UNORDERED_DELIVERY */
              int dontBundle)          /* use constants SCTP_BUNDLING_ENABLED, SCTP_BUNDLING_DISABLED */
{
    Association* currentAssociation = NULL;
    int result = SCTP_SUCCESS;

    ENTER_LIBRARY("sctp_send");
    CHECK_LIBRARY;

    /* Retrieve association from list  */
    currentAssociation = mdi_getAssociationByID(associationID, FALSE);

    if (currentAssociation != NULL) {
        if (pathId != 0) {
            result =   pm_checkValidPathKey(currentAssociation, pathId);
        }

        if (result == SCTP_SUCCESS) {
            event_logiiii(INTERNAL_EVENT_1, "sctp_send: got chunk from ULP (SID: %u, len: %u, path: %u, lifetime: %u",
                streamID, length, pathId, lifetime);
            /* Forward chunk to the addressed association */
            result = se_ulpsend(currentAssociation, streamID, buffer, length, protocolId, pathId,
                                lifetime, context, unorderedDelivery, dontBundle);
        } else {
            error_logi(ERROR_MAJOR, "sctp_send: invalid destination address %u", pathId);
            LEAVE_LIBRARY("sctp_send");
            return SCTP_PARAMETER_PROBLEM;
        }
    } else {
        error_log(ERROR_MAJOR, "sctp_send: addressed association does not exist");
        result = SCTP_ASSOC_NOT_FOUND ;
    }
    LEAVE_LIBRARY("sctp_send");
    return result;
}                               /* end: sctp_send */

/**
 * sctp_setPrimary changes the primary path of an association.
 * @param  associationID     ID of assocation.
 * @param  destAddressIndex  index to the new primary path
 * @return error code
 */
int sctp_setPrimary(unsigned int associationID, unsigned int pathId)
{
    Association* currentAssociation = NULL;
    int retval;

    ENTER_LIBRARY("sctp_setPrimary");
    CHECK_LIBRARY;
    /* Retrieve association from list  */
    currentAssociation = mdi_getAssociationByID(associationID, FALSE);

    if (currentAssociation != NULL) {
        if (sci_getState(currentAssociation) != SCTP_ESTABLISHED) {
            LEAVE_LIBRARY("sctp_setPrimary");
            return SCTP_SPECIFIC_FUNCTION_ERROR;
        }
        retval = pm_checkValidPathKey(currentAssociation, pathId);

        if (retval == SCTP_SUCCESS) {
            /* Forward shutdown to the addressed association */
            retval = pm_setPrimaryPathKey(currentAssociation, pathId);
        }
    } else {
        error_log(ERROR_MAJOR, "sctp_setPrimary: addressed association does not exist");
        retval =  SCTP_ASSOC_NOT_FOUND;
    }
    LEAVE_LIBRARY("sctp_setPrimary");
    return retval;
}                       /* end: sctp_setPrimary */

/**
 * sctp_receive is called in response to the dataArriveNotification to
 * get the received data. The application must provide a pointer to a
 * buffer with size *length. After the call, the SCTPLIB sets length to
 * the actual number of bytes copied to  the User Process
 *  @param   associationID  ID of association.
 *  @param   streamID       the stream on which the data chunk is received.
 *  @param   buffer         pointer to where payload data of arrived chunk will be copied
 *  @param   length         length of chunk data.
 *  @return  SCTP_SUCCESS if okay, SCTP_NO_CHUNKS_IN_QUEUE if there was no data
 */
int
sctp_receive(unsigned int    associationID,
             unsigned short  streamID,
             unsigned char   *buffer,
             unsigned int    *length,
             unsigned short  *streamSN,
             unsigned int    *tsn,
             unsigned int    flags)
{
    Association* currentAssociation = NULL;
    int result;

    ENTER_LIBRARY("sctp_receive");
    CHECK_LIBRARY;

    if (buffer == NULL || length == NULL) {
        LEAVE_LIBRARY("sctp_receive");
        return SCTP_PARAMETER_PROBLEM;
    }
    /* Retrieve association from list, as long as the data is not actually gone ! */
    currentAssociation = mdi_getAssociationByID(associationID, TRUE);

    if (currentAssociation != NULL) {
        /* retrieve data from streamengine instance */
        result = se_ulpreceive(currentAssociation, buffer, length, streamID, streamSN, tsn, flags);
    } else {
        error_log(ERROR_MAJOR, "sctp_receive: addressed association does not exist");
        LEAVE_LIBRARY("sctp_receive");
        return SCTP_ASSOC_NOT_FOUND;
    }

    if (result == 0) {
        LEAVE_LIBRARY("sctp_receive");
        return SCTP_SUCCESS;
    } else if (result == 1) {
        LEAVE_LIBRARY("sctp_receive");
        return SCTP_PARAMETER_PROBLEM;
    } else if (result == SCTP_MODULE_NOT_FOUND) {
        LEAVE_LIBRARY("sctp_receive");
        return SCTP_MODULE_NOT_FOUND;
    }
    /* else result == SCTP_NO_CHUNKS_IN_QUEUE, i.e. no data available */
    LEAVE_LIBRARY("sctp_receive");
    return result;
}                       /* end: sctp_receive */

/**
 * sctp_changeHeartBeat turns the hearbeat on a path of an association on or
 * off, or modifies the interval
 * @param  associationID   ID of assocation.
 * @param  path_id         index of the path where to do heartbeat
 * @param  heartbeatON     turn heartbeat on (==1) or off (==0)
 * @param  timeIntervall   heartbeat time intervall in milliseconds
 * @return error code,     SCTP_LIBRARY_NOT_INITIALIZED, SCTP_SUCCESS, SCTP_PARAMETER_PROBLEM,
 *                         SCTP_MODULE_NOT_FOUND, SCTP_ASSOC_NOT_FOUND
 */
int sctp_changeHeartBeat(unsigned int   associationID,
                         unsigned int   pathId,
                         int            heartbeatON,
                         unsigned int   timeIntervall)
{
    Association* currentAssociation = NULL;
    int result;

    ENTER_LIBRARY("sctp_changeHeartbeat");
    CHECK_LIBRARY;

    /* Retrieve association from list  */
    currentAssociation = mdi_getAssociationByID(associationID, FALSE);

    if (currentAssociation != NULL) {
        result = pm_checkValidPathKey(currentAssociation, pathId);

        if (result == SCTP_SUCCESS) {
            /* Forward change HB to the addressed association */
            if (heartbeatON) {
                result = pm_enableHB(currentAssociation, pathId, timeIntervall);
                event_logiii(VERBOSE, "Setting HB interval for address %u to %u msecs, result: %d !",
                             pathId, timeIntervall, result);
            } else {
                result = pm_disableHB(currentAssociation, pathId);
                event_logii(VERBOSE, "Disabling HB for address %u, result: %d !",
                            pathId, result);
            }
        }
    } else {
        error_log(ERROR_MAJOR, "sctp_changeHeartBeat: addressed association does not exist");
        result = SCTP_ASSOC_NOT_FOUND;
    }
    LEAVE_LIBRARY("sctp_changeHeartbeat");
    return result;
}                               /* end: sctp_changeHeartBeat */



/**
 * sctp_requestHeartbeat sends a heartbeat to the given address of an association.
 * @param  associationID  ID of assocation.
 * @param  path_id        destination address to which the heartbeat is sent.
 * @return error code (SCTP_SUCCESS, SCTP_ASSOC_NOT_FOUND, SCTP_MODULE_NOT_FOUND, SCTP_LIBRARY_NOT_INITIALIZED,
                        SCTP_UNSPECIFIED_ERROR)
 */
int sctp_requestHeartbeat(unsigned int associationID, unsigned int pathId)
{
    Association* currentAssociation = NULL;
    int result;

    ENTER_LIBRARY("sctp_requestHeartbeat");
    CHECK_LIBRARY;

    /* Retrieve association from list  */
    currentAssociation = mdi_getAssociationByID(associationID, FALSE);

    if (currentAssociation != NULL) {
        result = pm_checkValidPathKey(currentAssociation, pathId);
        if (result == SCTP_SUCCESS) {
            result = pm_doHB(currentAssociation,pathId);
            event_logi(VERBOSE, "Sending HB on user request to path ID: %u !",pathId);
        }
    } else {
        error_log(ERROR_MAJOR, "sctp_requestHeartbeat: addressed association does not exist");
        result = SCTP_ASSOC_NOT_FOUND;
    }
    LEAVE_LIBRARY("sctp_requestHeartbeat");
    return result;
}                               /* sctp_requestHeartbeat */

/**
 * sctp_getSrttReport returns a smoothed RTT value for a path to a given address
 * @param  associationID    ID of assocation.
 * @param  destAddressIndex destination address where to get SRTT from
 * @return SRTT of the address in msecs, negative on error
 */
int sctp_getSrttReport(unsigned int associationID, unsigned int pathId)
{
    Association* currentAssociation = NULL;
    int srtt, result;

    ENTER_LIBRARY("sctp_getSrttReport");
    CHECK_LIBRARY;

    /* Retrieve association from list  */
    currentAssociation = mdi_getAssociationByID(associationID, FALSE);

    if (currentAssociation != NULL) {
        result = pm_checkValidPathKey(currentAssociation, pathId);

        if (result == SCTP_SUCCESS) {
            srtt = pm_readSRTT(currentAssociation, pathId);
            event_logiii(VERBOSE, "sctp_getSrttReport(asoc=%u, address=%u) result: %d !",
                                  associationID, pathId, srtt);
            if (srtt == -1) {
                LEAVE_LIBRARY("sctp_getSrttReport");
                return SCTP_PARAMETER_PROBLEM;
            } else {
                LEAVE_LIBRARY("sctp_getSrttReport");
                return srtt;
            }
        } else {
            return result;
        }
    } else {
        error_log(ERROR_MAJOR, "sctp_getSrttReport: addressed association does not exist");
    }
    LEAVE_LIBRARY("sctp_getSrttReport");
    return  SCTP_ASSOC_NOT_FOUND;
}


/**
 *  sctp_setFailureThreshold is used to set the threshold for retransmissions on the given
 *  address of an association. If the threshold is exeeded, the the destination address is
 *  considered as  unreachable.
 *  @param  associationID :            ID of assocation.
 *  @param  destAddressIndex :         destination address that gets a new failure threshold.
 *  @param  pathMaxRetransmissions :   threshold for retransmissions.
 *  @return
 */
int
sctp_setFailureThreshold(unsigned int associationID, unsigned int pathMaxRetransmissions)
{
    Association* currentAssociation = NULL;
    int result;

    ENTER_LIBRARY("sctp_setFailureThreshold");
    CHECK_LIBRARY;

    event_logii(VERBOSE, "sctp_setFailureThreshold: Association %u, pathMaxRetr. %u", associationID, pathMaxRetransmissions);

    currentAssociation = mdi_getAssociationByID(associationID, FALSE);

    if (currentAssociation != NULL) {
        pm_setMaxPathRetransmisions(currentAssociation, pathMaxRetransmissions);
        result = SCTP_SUCCESS;
    } else {
        error_logi(ERROR_MAJOR, "sctp_setFailureThreshold : association %u does not exist", associationID);
        result = SCTP_ASSOC_NOT_FOUND;
    }
    LEAVE_LIBRARY("sctp_setFailureThreshold");
    return result;

}                               /* end:  sctp_setFailureThreshold */



/**
 * sctp_getPathStatus : IP_TOS support is still missing !
 * Can be used to get path specific parameters in an existing association.
 *  @param  associationID   ID of assocation.
 *  @param  path_id         path for which to get parameters
 *  @param  status      pointer to new parameters
 *  @return 0 for success, not zero for error
 */
int sctp_getPathStatus(unsigned int associationID, unsigned int pathId, SCTP_PathStatus* status)
{
    Association* currentAssociation = NULL;
    unsigned int inFlight;
    int result;
    int index;

    ENTER_LIBRARY("sctp_getPathStatus");
    CHECK_LIBRARY;
    event_logii(VERBOSE, "sctp_getPathStatus: Association %u, Path %u", associationID, pathId);

    if (status == NULL)
    {
        LEAVE_LIBRARY("sctp_getPathStatus");
        return SCTP_PARAMETER_PROBLEM;
    }
    currentAssociation = mdi_getAssociationByID(associationID, FALSE);

    if (currentAssociation != NULL)
    {
        result = pm_checkValidPathKey(currentAssociation, pathId);
        if (result != SCTP_SUCCESS) return SCTP_PARAMETER_PROBLEM;
        index = mdi_getIndexForKey(currentAssociation, pathId);
        
        adl_sockunion2str(&g_array_index(currentAssociation->destAddresses, union sockunion, index),
                          &(status->destinationAddress[0]), SCTP_MAX_IP_LEN);
        status->pathIndex = index;
        status->pathKey = pathId;
        status->state = pm_readState(currentAssociation, pathId);
        status->srtt = pm_readSRTT(currentAssociation, pathId);
        status->rto = pm_readRTO(currentAssociation, pathId);
        status->rttvar = pm_readRttVar(currentAssociation, pathId);
        status->cwnd = fc_readCWND(currentAssociation, pathId);
        status->cwnd2 = fc_readCWND2(currentAssociation, pathId);
        status->partialBytesAcked = fc_readPBA(currentAssociation, pathId);
        status->ssthresh = fc_readSsthresh(currentAssociation, pathId);
        status->outstandingBytesPerAddress = rtx_get_obpa(currentAssociation, pathId, &inFlight);
        status->mtu = pm_readPathSCTPMTU(currentAssociation, pathId);
        status->ipTos = currentAssociation->ipTos;
        result = SCTP_SUCCESS;
    }
    else
    {
        error_logi(ERROR_MAJOR, "sctp_getPathStatus : association %u does not exist", associationID);
        result = SCTP_ASSOC_NOT_FOUND;
    }
    LEAVE_LIBRARY("sctp_getPathStatus");
    return result;
}

/**
 * sctp_setPathStatus is currently NOT implemented !
 * Can be used to set path specific parameters in an existing association.
 *
 *  @param  associationID   ID of assocation.
 *  @param  path_id         path for which to set parameters
 *  @param  new_status      pointer to new parameters
 *  @return -1
 */
int sctp_setPathStatus(unsigned int associationID, unsigned int pathId, SCTP_PathStatus* new_status)
{
    /* Association* currentAssociation = NULL; */
    CHECK_LIBRARY;
    ENTER_LIBRARY("sctp_setPathStatus");

    error_log(ERROR_MAJOR, "sctp_setPathStatus : unimplemented function");
    LEAVE_LIBRARY("sctp_setPathStatus");
    return SCTP_UNSPECIFIED_ERROR;
}


/**
 * sctp_setAssocStatus allows for setting a number of association parameters.
 * _Not_ all values that the corresponding sctp_getAssocStatus-function returns
 * may be SET here !
 * Will set protocol parameters per SCTP association
 *
 *  @param  associationID   ID of assocation.
 *  @param  new_status      pointer to new parameters
 *  @return -1
*/
int sctp_setAssocStatus(unsigned int associationID, SCTP_AssociationStatus* new_status)
{
    Association* currentAssociation = NULL;
    guint16 result;

    ENTER_LIBRARY("sctp_setAssocStatus");
    CHECK_LIBRARY;

    if (new_status == NULL) {
        LEAVE_LIBRARY("sctp_setAssocStatus");
        return SCTP_PARAMETER_PROBLEM;
    }
    currentAssociation = mdi_getAssociationByID(associationID, FALSE);

    if (currentAssociation != NULL) {
        result = pm_checkValidPathKey(currentAssociation, new_status->primaryAddressKey);
        if (result != SCTP_SUCCESS) return SCTP_PARAMETER_PROBLEM;
        event_logi(VERBOSE, "sctp_setAssocStatus: Association %u", associationID);
        if (pm_setPrimaryPathKey(currentAssociation, new_status->primaryAddressKey)) {
            error_logi(ERROR_MINOR, "pm_setPrimary(%u) returned error", new_status->primaryAddressKey);
            LEAVE_LIBRARY("sctp_setAssocStatus");
            return SCTP_PARAMETER_PROBLEM;
        }
        if (pm_setRtoInitial(currentAssociation, new_status->rtoInitial)) {
            error_logi(ERROR_MINOR, "pm_setRtoInitial(%u) returned error", new_status->rtoInitial);
            LEAVE_LIBRARY("sctp_setAssocStatus");
            return SCTP_PARAMETER_PROBLEM;
        }
        if (pm_setRtoMin(currentAssociation, new_status->rtoMin)) {
            error_logi(ERROR_MINOR, "pm_setRtoMin(%u) returned error", new_status->rtoMin);
            LEAVE_LIBRARY("sctp_setAssocStatus");
            return SCTP_PARAMETER_PROBLEM;
        }
        if (pm_setRtoMax(currentAssociation, new_status->rtoMax)) {
            error_logi(ERROR_MINOR, "pm_setRtoMax(%u) returned error", new_status->rtoMax);
            LEAVE_LIBRARY("sctp_setAssocStatus");
            return SCTP_PARAMETER_PROBLEM;
        }
        if(pm_setMaxPathRetransmisions(currentAssociation, new_status->pathMaxRetransmits)) {
            error_logi(ERROR_MINOR, "pm_getMaxPathRetransmisions(%u) returned error", new_status->pathMaxRetransmits);
            LEAVE_LIBRARY("sctp_setAssocStatus");
            return SCTP_PARAMETER_PROBLEM;
        }
        sci_setCookieLifeTime(currentAssociation, new_status->validCookieLife);
        sci_setMaxAssocRetransmissions(currentAssociation, new_status->assocMaxRetransmits);
        sci_setMaxInitRetransmissions(currentAssociation, new_status->maxInitRetransmits);

        rxc_set_local_receiver_window(currentAssociation, new_status->myRwnd);
        rxc_set_sack_delay(currentAssociation, new_status->delay);
        currentAssociation->ipTos = new_status->ipTos;
        result = fc_set_maxSendQueue(currentAssociation, new_status->maxSendQueue);

        result = SCTP_SUCCESS;

    } else {
        error_logi(ERROR_MAJOR, "sctp_getAssocStatus : association %u does not exist", associationID);
        result = SCTP_ASSOC_NOT_FOUND;
    }
    LEAVE_LIBRARY("sctp_setAssocStatus");
    return result;
}                               /* end: sctp_setAssocStatus */

/**
 * Will get the protocol parameters per SCTP association
 *
 *  @param  associationID   ID of assocation.
 *  @return  pointer to a structure containing association parameters
*/
int sctp_getAssocStatus(unsigned int associationID, SCTP_AssociationStatus* status)
{
    Association* currentAssociation = NULL;
    int result, index;

    ENTER_LIBRARY("sctp_getAssocStatus");
    CHECK_LIBRARY;

    if (status == NULL) {
        LEAVE_LIBRARY("sctp_getAssocStatus");
        return SCTP_PARAMETER_PROBLEM;
    }
    currentAssociation = mdi_getAssociationByID(associationID, FALSE);

    if (currentAssociation != NULL) {
        event_logi(VERBOSE, "sctp_getAssocStatus: Association %u", associationID);
        status->state = sci_getState(currentAssociation);
        status->numberOfDestinationPaths = currentAssociation->noOfDestAddresses;
        status->sourcePort = currentAssociation->localPort;
        status->destPort = currentAssociation->remotePort;
        status->primaryAddressKey = pm_getPrimaryPathKey(currentAssociation);
        for (index = 0; index < currentAssociation->noOfDestAddresses; index++) {
            status->destinationPathIDs[index] = g_array_index(currentAssociation->destAddressKeys, unsigned int, index);
        }
        index = mdi_getIndexForKey(currentAssociation, status->primaryAddressKey);

        adl_sockunion2str(&g_array_index(currentAssociation->destAddresses, union sockunion, index),
                          &(status->primaryDestinationAddress[0]), SCTP_MAX_IP_LEN);

        se_readNumberOfStreams(currentAssociation, &(status->inStreams), &(status->outStreams));
        status->currentReceiverWindowSize =  rtx_read_remote_receiver_window(currentAssociation);
        status->outstandingBytes = fc_readOutstandingBytes(currentAssociation);
        status->noOfChunksInSendQueue = fc_readNumberOfQueuedChunks(currentAssociation) +
                                        se_getQueuedChunksSendQueue(currentAssociation);
        status->noOfChunksInRetransmissionQueue = rtx_readNumberOfUnackedChunks(currentAssociation);
        status->noOfChunksInReceptionQueue = se_numOfQueuedChunks(currentAssociation);
        status->rtoInitial = pm_getRtoInitial(currentAssociation);
        status->rtoMin = pm_getRtoMin(currentAssociation);
        status->rtoMax = pm_getRtoMax(currentAssociation);
        status->validCookieLife = sci_getCookieLifeTime(currentAssociation);
        status->assocMaxRetransmits = sci_getMaxAssocRetransmissions(currentAssociation);
        status->pathMaxRetransmits = pm_getMaxPathRetransmisions(currentAssociation);
        status->maxInitRetransmits = sci_getMaxInitRetransmissions(currentAssociation);
        status->myRwnd = rxc_get_local_receiver_window(currentAssociation);
        status->delay = rxc_get_sack_delay(currentAssociation);
        result = fc_get_maxSendQueue(currentAssociation, &(status->maxSendQueue));
        status->maxRecvQueue = 0;
        status->ipTos = 0;
        result = SCTP_SUCCESS;

    } else {
        error_logi(ERROR_MAJOR, "sctp_getAssocStatus : association %u does not exist", associationID);
        result = SCTP_ASSOC_NOT_FOUND;
    }
    LEAVE_LIBRARY("sctp_getAssocStatus");
    return result;
}                      /* end: sctp_getAssocStatus */

/**
 * sctp_setAssocDefaults allows for setting a few association default parameters !
 * Will set protocol default parameters per given SCTP instance
 *
 *  @param  SCTP_InstanceName   instance for which to set the parameters
 *  @param  params       pointer to parameter data structure
 *  @return -1
*/
int sctp_setAssocDefaults(int SCTP_InstanceName, SCTP_InstanceParameters* params)
{
    SCTP_instance temporary;
    SCTP_instance* instance;
    GList* result = NULL;

    ENTER_LIBRARY("sctp_setAssocDefaults");
    CHECK_LIBRARY;
    event_logi(VERBOSE, "sctp_setInstanceParams: Instance %u", SCTP_InstanceName);

    temporary.sctpInstanceName = SCTP_InstanceName;
    result = g_list_find_custom(InstanceList, &temporary, &compareInstanceNames);
    if (result != NULL) {
        instance =  result->data;
    } else {
        error_logi(ERROR_MINOR, "sctp_setAssocDefaults : Did not find Instance Number %u", SCTP_InstanceName);
        LEAVE_LIBRARY("sctp_setAssocDefaults");
        return SCTP_INSTANCE_NOT_FOUND;
    }
    if (params == NULL) {
        error_log(ERROR_MINOR, "sctp_setAssocDefaults : Passed NULL Pointer !");
        LEAVE_LIBRARY("sctp_setAssocDefaults");
        return SCTP_PARAMETER_PROBLEM;
    }
    instance->default_rtoInitial =  params->rtoInitial;
    instance->default_rtoMin = params->rtoMin;
    instance->default_rtoMax = params->rtoMax;
    instance->default_validCookieLife = params->validCookieLife;
    instance->default_assocMaxRetransmits =  params->assocMaxRetransmits;
    instance->default_pathMaxRetransmits = params->pathMaxRetransmits;
    instance->default_maxInitRetransmits =  params->maxInitRetransmits;
    instance->default_myRwnd =  params->myRwnd;
    instance->default_delay = params->delay;
    instance->default_ipTos = params->ipTos;
    instance->default_maxSendQueue = params->maxSendQueue;
    instance->default_maxRecvQueue = params->maxRecvQueue;
    instance->noOfInStreams = params->inStreams;
    instance->noOfOutStreams = params->outStreams;
    LEAVE_LIBRARY("sctp_setAssocDefaults");
    return SCTP_SUCCESS;
}                               /* end: sctp_setInstanceParams */

/**
 * sctp_getAssocDefaults returns a struct with default parameter values !
 * Will get protocol parameters per given SCTP instance
 *
 *  @param  SCTP_InstanceName   instance for which to set the parameters
 *  @param  params       pointer to parameter data
 *  @return -1
*/
int sctp_getAssocDefaults(int SCTP_InstanceName, SCTP_InstanceParameters* params)
{
    SCTP_instance temporary;
    SCTP_instance* instance;
    GList* result = NULL;
    int count;
    unsigned int numOfAddresses = 0;

    ENTER_LIBRARY("sctp_getAssocDefaults");
    CHECK_LIBRARY;
    event_logi(VERBOSE, "sctp_getInstanceParams: Instance %u", SCTP_InstanceName);

    temporary.sctpInstanceName = SCTP_InstanceName;
    result = g_list_find_custom(InstanceList, &temporary, &compareInstanceNames);
    if (result != NULL) {
        instance =  result->data;
    } else {
        error_logi(ERROR_MINOR, "sctp_getAssocDefaults : Did not find Instance Number %u", SCTP_InstanceName);
        LEAVE_LIBRARY("sctp_getAssocDefaults");
        return SCTP_INSTANCE_NOT_FOUND;
    }
    if (params == NULL) {
        error_log(ERROR_MINOR, "sctp_getAssocDefaults : Passed NULL Pointer !");
        LEAVE_LIBRARY("sctp_getAssocDefaults");
        return SCTP_PARAMETER_PROBLEM;
    }

    if (instance->numberOfLocalAddresses > SCTP_MAX_NUM_ADDRESSES) numOfAddresses = SCTP_MAX_NUM_ADDRESSES;
    else numOfAddresses = instance->numberOfLocalAddresses;

    if (numOfAddresses == 0) { /* get all local addresses from IN(6)ADDR_ANY */
        params->noOfLocalAddresses = myNumberOfAddresses;
        for (count = 0; count < myNumberOfAddresses; count++) {
            adl_sockunion2str(&myAddressList[count], params->localAddressList[count], SCTP_MAX_IP_LEN);
        }
    } else { /* get all specified addresses */
        params->noOfLocalAddresses = numOfAddresses;
        for (count = 0; count < numOfAddresses; count++) {
            adl_sockunion2str(&g_array_index(instance->localAddresses, union sockunion, count),
                                             params->localAddressList[count], SCTP_MAX_IP_LEN);

        }
    }
    params->rtoInitial = instance->default_rtoInitial;
    params->rtoMin  = instance->default_rtoMin;
    params->rtoMax  = instance->default_rtoMax;
    params->validCookieLife = instance->default_validCookieLife;
    params->assocMaxRetransmits = instance->default_assocMaxRetransmits;
    params->pathMaxRetransmits = instance->default_pathMaxRetransmits;
    params->maxInitRetransmits = instance->default_maxInitRetransmits;
    params->myRwnd = instance->default_myRwnd;
    params->delay = instance->default_delay ;
    params->ipTos = instance->default_ipTos ;
    params->maxSendQueue = instance->default_maxSendQueue;
    params->maxRecvQueue = instance->default_maxRecvQueue;
    params->inStreams = instance->noOfInStreams;
    params->outStreams = instance->noOfOutStreams;

    LEAVE_LIBRARY("sctp_getAssocDefaults");
    return SCTP_SUCCESS;
}                               /* end: sctp_getAssocDefaults */

/**
 * sctp_bindAddressToInstance adds a new address to an instance, if this instance
 * does not have IN_ADDR_ANY set.
 *
 *  @param  SCTP_InstanceName   instance for which to set the parameters
 *  @param  address             the address to be added
 *  @return -1 for error, SCTP_SUCCESS if everything went alright
*/
int sctp_bindAddressToInstance(int SCTP_InstanceName, unsigned char newAddress[SCTP_MAX_IP_LEN])
{
    SCTP_instance temporary;
    SCTP_instance* instance;
    GList* result = NULL;
    union sockunion su;
    int returnCode;
    
    gboolean with_ipv4 = FALSE;
#ifdef HAVE_IPV6
    gboolean with_ipv6 = FALSE;
#endif


    ENTER_LIBRARY("sctp_bindAddressToInstance");
    CHECK_LIBRARY;
    event_logi(VERBOSE, "sctp_bindAddressToInstance: Instance %u", SCTP_InstanceName);

    temporary.sctpInstanceName = SCTP_InstanceName;
    result = g_list_find_custom(InstanceList, &temporary, &compareInstanceNames);
    if (result != NULL) {
        instance =  result->data;
    } else {
        error_logi(ERROR_MINOR, "sctp_bindAddressToInstance : Did not find Instance Number %u", SCTP_InstanceName);
        LEAVE_LIBRARY("sctp_bindAddressToInstance");
        return SCTP_INSTANCE_NOT_FOUND;
    }

    if (adl_str2sockunion(newAddress, &su) < 0) {
        error_logi(ERROR_MAJOR, "Address Error in sctp_bindAddressToInstance(%s)", newAddress);
        LEAVE_LIBRARY("sctp_bindAddressToInstance");
        return SCTP_PARAMETER_PROBLEM;
    } else {
        if (su.sa.sa_family == AF_INET) with_ipv4 = TRUE;
#ifdef HAVE_IPV6
        if (su.sa.sa_family == AF_INET6) with_ipv6 = TRUE;
#endif
    }

    /* the instance must not have INADDR_ANY set to add an address */
    if (instance->numberOfLocalAddresses == 0) return SCTP_NOT_SUPPORTED;

#ifdef HAVE_IPV6
    if (with_ipv6 && ipv6_sctp_socket==0) {
         ipv6_sctp_socket = adl_get_sctpv6_socket();
         if (ipv6_sctp_socket > 0) {
             returnCode = adl_register_socket_cb(ipv6_sctp_socket,&mdi_dummy_callback);
             if (!returnCode) error_log(ERROR_FATAL, "register ipv6 socket call back function failed");
         }
    }
    if (with_ipv6 == TRUE) {
	    ipv6_users++;
	    instance->uses_IPv6 = TRUE;
    } 
#endif
    if (with_ipv4 && sctp_socket==0) {
         sctp_socket = adl_get_sctpv4_socket();
         if (!sctp_socket) error_log(ERROR_FATAL, "IPv4 socket creation failed");
         returnCode = adl_register_socket_cb(sctp_socket,&mdi_dummy_callback);
         if (!returnCode) error_log(ERROR_FATAL, "registration of IPv4 socket call back function failed");
    }
    if (with_ipv4 == TRUE) {
	    ipv4_users++;
	    instance->uses_IPv4 = TRUE;
    } 
    returnCode = mdi_addAddressToInstanceForced(instance, &su); 
        
    return SCTP_SUCCESS;
}


int sctp_setLibraryParameters(SCTP_LibraryParameters *params)
{
    ENTER_LIBRARY("sctp_setLibraryParameters");

    CHECK_LIBRARY;
    if (params == NULL) {
        LEAVE_LIBRARY("sctp_setLibraryParameters");
        return SCTP_PARAMETER_PROBLEM;
    }

    event_logi(VERBOSE, "sctp_setLibraryParameters: Parameter sendAbortForOOTB is %s",
                        (sendAbortForOOTB==TRUE)?"TRUE":"FALSE");
    if (params->sendOotbAborts == 0) {
        sendAbortForOOTB = FALSE;
    } else if (params->sendOotbAborts == 1) {
        sendAbortForOOTB = TRUE;
    } else {
        LEAVE_LIBRARY("sctp_setLibraryParameters");
        return SCTP_PARAMETER_PROBLEM;
    }
    if (params->checksumAlgorithm == SCTP_CHECKSUM_ALGORITHM_CRC32C ||
        params->checksumAlgorithm == SCTP_CHECKSUM_ALGORITHM_ADLER32) {
        if (checksumAlgorithm != params->checksumAlgorithm) {
            checksumAlgorithm = params->checksumAlgorithm;
            set_checksum_algorithm(checksumAlgorithm);
        } /* else nothing changes */
    } else {
        LEAVE_LIBRARY("sctp_setLibraryParameters");
        return SCTP_PARAMETER_PROBLEM;
    }
    if (params->supportPRSCTP == 0) {
        librarySupportsPRSCTP = FALSE;
    } else {
        librarySupportsPRSCTP = TRUE;
    } 
    if (params->supportADDIP == 0) {
        librarySupportsADDIP = FALSE;
    } else {
        librarySupportsADDIP = TRUE;
    } 
    event_logi(INTERNAL_EVENT_0, "sctp_setLibraryParameters: sendAbortForOOTB is now %s",
                                  (params->sendOotbAborts==TRUE)?"TRUE":"FALSE");
    event_logi(INTERNAL_EVENT_0, "sctp_setLibraryParameters: Checksum Algorithm is now %s",
                                  (params->checksumAlgorithm==SCTP_CHECKSUM_ALGORITHM_CRC32C)?"CRC32C":"ADLER32");
    event_logi(INTERNAL_EVENT_0, "sctp_setLibraryParameters: Support of PRSCTP is now %s",
                                  (params->supportPRSCTP==TRUE)?"ENABLED":"DISABLED");
    event_logi(INTERNAL_EVENT_0, "sctp_setLibraryParameters: Support of ADDIP is now %s",
                                  (params->supportADDIP==TRUE)?"ENABLED":"DISABLED");

    LEAVE_LIBRARY("sctp_setLibraryParameters");
    return SCTP_SUCCESS;

}

int sctp_getLibraryParameters(SCTP_LibraryParameters *params)
{
    ENTER_LIBRARY("sctp_getLibraryParameters");

    CHECK_LIBRARY;
    if (params == NULL) {
        LEAVE_LIBRARY("sctp_getLibraryParameters");
        return SCTP_PARAMETER_PROBLEM;
    }


    params->sendOotbAborts = sendAbortForOOTB;
    params->checksumAlgorithm = checksumAlgorithm;
    params->supportPRSCTP = librarySupportsPRSCTP;
    params->supportADDIP =  librarySupportsADDIP;

    event_logi(INTERNAL_EVENT_0, "sctp_getLibraryParameters: Parameter sendAbortForOOTB is currently %s",
                        (sendAbortForOOTB==TRUE)?"TRUE":"FALSE");
    event_logi(INTERNAL_EVENT_0, "sctp_getLibraryParameters: Checksum Algorithm is currently %s",
                                  (checksumAlgorithm==SCTP_CHECKSUM_ALGORITHM_CRC32C)?"CRC32C":"ADLER32");
    event_logi(INTERNAL_EVENT_0, "sctp_getLibraryParameters: Support of PRSCTP is now %s",
                                  (params->supportPRSCTP==TRUE)?"ENABLED":"DISABLED");
    event_logi(INTERNAL_EVENT_0, "sctp_getLibraryParameters: Support of ADDIP is now %s",
                                  (params->supportADDIP==TRUE)?"ENABLED":"DISABLED");

    LEAVE_LIBRARY("sctp_getLibraryParameters");
    return SCTP_SUCCESS;

}

/**
 * sctp_receive_unsent returns messages that have not been sent before the termination of an association
 *
 *  @param  associationID       ID of assocation.
 *  @param  buffer              pointer to a buffer that the application needs to pass. Data is copied there.
 *  @param  length              pointer to size of the buffer passed by application, contains actual length
 *                              of the copied chunk after the function call.
 *  @param  streamID            pointer to the stream id, where data should have been sent
 *  @param  streamSN            pointer to stream sequence number of the data chunk that was not sent
 *  @param  protocolID          pointer to the protocol ID of the unsent chunk
 *  @return number of unsent chunks still in the queue, else error code as  SCTP_NO_CHUNKS_IN_QUEUE, or
 *  SCTP_PARAMETER_PROBLEM, SCTP_WRONG_STATE, SCTP_ASSOC_NOT_FOUND, SCTP_LIBRARY_NOT_INITIALIZED
 */
int sctp_receiveUnsent(unsigned int associationID, unsigned char *buffer, unsigned int *length,
                       unsigned int *tsn, unsigned short *streamID, unsigned short *streamSN,
                       unsigned int* protocolId, unsigned char* flags, void** context)
{
    Association* currentAssociation = NULL;
    int result;

    ENTER_LIBRARY("sctp_receiveUnsent");
    CHECK_LIBRARY;

    if (buffer == NULL || length == NULL || tsn==NULL || streamID == NULL || streamSN == NULL || protocolId == NULL) {
        LEAVE_LIBRARY("sctp_receiveUnsent");
        return SCTP_PARAMETER_PROBLEM;
    }
    currentAssociation = mdi_getAssociationByID(associationID, TRUE);

    if (currentAssociation != NULL) {
        if (currentAssociation->deleted == FALSE) {
            result =  SCTP_WRONG_STATE;
        } else if (fc_readNumberOfUnsentChunks(currentAssociation) == 0) {
            result = SCTP_NO_CHUNKS_IN_QUEUE;
        } else {
            result = fc_dequeueOldestUnsentChunk(currentAssociation,
                                                 buffer,
                                                 length,
                                                 tsn,
                                                 streamID,
                                                 streamSN,
                                                 protocolId,
                                                 flags,context);
        }
    } else {
        error_logi(ERROR_MAJOR, "sctp_receiveUnsent : association %u does not exist", associationID);
        result = SCTP_ASSOC_NOT_FOUND;
    }
    LEAVE_LIBRARY("sctp_receiveUnsent");
    return result;
}

/**
 * sctp_receive_unacked returns messages that were already put on the wire, but have not been
 * acknowledged by the peer before termination of the association
 *
 *  @param  associationID       ID of assocation.
 *  @param  buffer              pointer to a buffer that the application needs to pass. Data is copied there.
 *  @param  length              pointer to size of the buffer passed by application, contains actual length
 *                              of the copied chunk after the function call.
 *  @param  streamID            pointer to the stream id, where data should have been sent
 *  @param  streamSN            pointer to stream sequence number of the data chunk that was not acked
 *  @param  protocolID          pointer to the protocol ID of the unacked chunk
 *  @return number of unacked chunks still in the queue, else SCTP_NO_CHUNKS_IN_QUEUE if no chunks there, else
 *  appropriate error code:  SCTP_PARAMETER_PROBLEM, SCTP_WRONG_STATE, SCTP_ASSOC_NOT_FOUND, SCTP_LIBRARY_NOT_INITIALIZED
*/
int sctp_receiveUnacked(unsigned int associationID, unsigned char *buffer, unsigned int *length,
                        unsigned int *tsn, unsigned short *streamID, unsigned short *streamSN,
                        unsigned int* protocolId,unsigned char* flags, void** context)
{
    Association* currentAssociation = NULL;
    int result;

    ENTER_LIBRARY("sctp_receiveUnacked");
    CHECK_LIBRARY;

    if (buffer == NULL || length == NULL || tsn==NULL || streamID == NULL || streamSN == NULL || protocolId == NULL) {
        LEAVE_LIBRARY("sctp_receiveUnacked");
        return SCTP_PARAMETER_PROBLEM;
    }
    currentAssociation = mdi_getAssociationByID(associationID, TRUE);

    if (currentAssociation != NULL) {
        if (currentAssociation->deleted == FALSE) {
            result =  SCTP_WRONG_STATE;
        } else if (rtx_readNumberOfUnackedChunks(currentAssociation) == 0) {
            result = SCTP_NO_CHUNKS_IN_QUEUE;
        } else {
            result = rtx_dequeueOldestUnackedChunk(currentAssociation,
                                                   buffer,
                                                   length,
                                                   tsn,
                                                   streamID,
                                                   streamSN,
                                                   protocolId,
                                                   flags,context);
        }
    } else {
        error_logi(ERROR_MAJOR, "sctp_receiveUnacked : association %u does not exist", associationID);
        result = SCTP_ASSOC_NOT_FOUND;
    }
    LEAVE_LIBRARY("sctp_receiveUnacked");
    return result;
}


/**
 * sctp_getPrimary returns the index of the current primary path
 * @param  associationID       ID of assocation.
 * @return  the index/key of the current primary path, or 0 on error
 */
unsigned int sctp_getPrimary(unsigned int associationID)
{
    Association* currentAssociation = NULL;
    unsigned int primary;

    ENTER_LIBRARY("sctp_getPrimary");
    CHECK_LIBRARY;

    currentAssociation = mdi_getAssociationByID(associationID, FALSE);

    if (currentAssociation != NULL) {
        primary = pm_getPrimaryPathKey(currentAssociation);
        event_logii(VERBOSE, "sctp_getPrimary: Association %u, primary: %u", associationID, primary);
    }else{
        error_logi(ERROR_MAJOR, "sctp_getPrimary : association %u does not exist", associationID);
        primary = 0;
    }
    LEAVE_LIBRARY("sctp_getPrimary");
    return primary;
}

int sctp_getInstanceID(unsigned int associationID, int* instanceID)
{
    Association* currentAssociation = NULL;
    SCTP_instance* sctpInstance = NULL;
    int result = SCTP_SUCCESS;

    ENTER_LIBRARY("sctp_getInstanceID");
    CHECK_LIBRARY;

    if (instanceID == NULL) {
        LEAVE_LIBRARY("sctp_getInstanceID");
        return SCTP_PARAMETER_PROBLEM;
    }
    currentAssociation = mdi_getAssociationByID(associationID, TRUE);

    if (currentAssociation != NULL) {
        sctpInstance = currentAssociation->sctpInstance;
        event_logii(VERBOSE, "sctp_getInstanceID: Association %u, Instance %d",
            associationID, sctpInstance->sctpInstanceName);
        (*instanceID) =  sctpInstance->sctpInstanceName;
    }else{
        error_logi(ERROR_MINOR, "sctp_getInstanceID: association %u does not exist", associationID);
        result = SCTP_ASSOC_NOT_FOUND;
    }
    LEAVE_LIBRARY("sctp_getInstanceID");
    return result;
}

/* ----------------------------------------------------------------------------------------*/
/* ------------------------------------ HELPER FUNCTIONS from adaptation ------------------*/
/* ----------------------------------------------------------------------------------------*/
int sctp_registerUdpCallback(unsigned char me[],
                             unsigned short my_port,
                             sctp_socketCallback scf)
{
    int result;
    ENTER_LIBRARY("sctp_registerUdpCallback");
    CHECK_LIBRARY;

    result = adl_registerUdpCallback(me,my_port,scf);

    LEAVE_LIBRARY("sctp_registerUdpCallback");
    return result;
}

int sctp_unregisterUdpCallback(int udp_sfd)
{
    int result;
    ENTER_LIBRARY("sctp_unregisterUdpCallback");
    CHECK_LIBRARY;

    result = adl_unregisterUdpCallback(udp_sfd);

    LEAVE_LIBRARY("sctp_unregisterUdpCallback");
    return result;
}

int sctp_sendUdpData(int sfd, unsigned char* buf, int length,
                     unsigned char destination[], unsigned short dest_port)
{
    int result;
    ENTER_LIBRARY("sctp_sendUdpData");
    CHECK_LIBRARY;

    result = adl_sendUdpData(sfd, buf, length, destination, dest_port);

    LEAVE_LIBRARY("sctp_sendUdpData");
    return result;
}

int sctp_registerUserCallback(int fd, sctp_userCallback sdf, void* userData, short int eventMask)
{
    int result;
    ENTER_LIBRARY("sctp_registerUserCallback");
    CHECK_LIBRARY;

    result = adl_registerUserCallback(fd, sdf, userData, eventMask);

    LEAVE_LIBRARY("sctp_registerUserCallback");
    return result;
}

int sctp_unregisterUserCallback(int fd)
{
    int result;
    ENTER_LIBRARY("sctp_unregisterUserCallback");
    CHECK_LIBRARY;

    result = adl_unregisterUserCallback(fd);

    LEAVE_LIBRARY("sctp_registerUserCallback");
    return result;
}

unsigned int sctp_startTimer(unsigned int seconds , unsigned int microseconds,
                        sctp_timerCallback timer_cb, void *param1, void *param2)
{
    unsigned int result;
    ENTER_LIBRARY("sctp_startTimer");
    CHECK_LIBRARY;

    result = adl_startMicroTimer(seconds, microseconds, timer_cb,TIMER_TYPE_USER, param1, param2);

    LEAVE_LIBRARY("sctp_startTimer");
    return result;
}

int sctp_stopTimer(unsigned int tid)
{
    int result;
    ENTER_LIBRARY("sctp_stopTimer");
    CHECK_LIBRARY;

    result = adl_stopTimer(tid);

    LEAVE_LIBRARY("sctp_stopTimer");
    return result;

}

unsigned int sctp_restartTimer(unsigned int timer_id, unsigned int seconds, unsigned int microseconds)
{
    int result;
    ENTER_LIBRARY("sctp_restartTimer");
    CHECK_LIBRARY;

    result = adl_restartMicroTimer(timer_id, seconds, microseconds);

    LEAVE_LIBRARY("sctp_restartTimer");
    return result;
}

int sctp_getEvents(void)
{
    int result;
    ENTER_LIBRARY("sctp_getEvents");
    CHECK_LIBRARY;

    result = adl_getEvents();

    LEAVE_LIBRARY("sctp_getEvents");
    return result;
}

int sctp_eventLoop(void)
{
    int result;
    ENTER_LIBRARY("sctp_eventLoop");
    CHECK_LIBRARY;

    result = adl_eventLoop();

    LEAVE_LIBRARY("sctp_eventLoop");
    return result;
}

int sctp_extendedEventLoop(void (*lock)(void* data), void (*unlock)(void* data), void* data)
{
    int result;
    ENTER_LIBRARY("sctp_extendedEventLoop");
    CHECK_LIBRARY;

    result = adl_extendedEventLoop(lock, unlock, data);

    LEAVE_LIBRARY("sctp_extendedEventLoop");
    return result;
}


/* ---------------------------  END of HELPER FUNCTIONS from adaptation -------------------*/

/* ----------------------------------------------------------------------------------------*/
/* ------------------------------------ ASCONF functions  -------------- ------------------*/
/* ----------------------------------------------------------------------------------------*/

int sctp_setRemotePrimary(unsigned int associationID, unsigned char address[SCTP_MAX_IP_LEN])
{
    int result;
    gboolean validAddress = FALSE;
    union sockunion su;
    Association* currentAssociation = NULL;
    /* SCTP_instance* sctpInstance = NULL; */
    ENTER_LIBRARY("sctp_setRemotePrimary");
    CHECK_LIBRARY;

    currentAssociation = mdi_getAssociationByID(associationID, FALSE);
    if (currentAssociation != NULL) {
        result = adl_str2sockunion(address, &su);
        /* this will return 0 if address is a valid IP address */
        validAddress = mdi_checkForValidLocalAddressInAssociation(currentAssociation, &su);        
        /* this will return 0 if address is not valid, i.e. has a valid path key */

        if (result == 0 && validAddress == TRUE) {
            event_logi(VERBOSE, "sctp_setRemotePrimary: Association %u", associationID);
            result = asc_setRemotePrimary(currentAssociation, &su);
        } else {
            result = SCTP_PARAMETER_PROBLEM;
        }
    }else{
        error_logi(ERROR_MAJOR, "sctp_asc_setRemotePrimary : association %u does not exist", associationID);
        result =  SCTP_ASSOC_NOT_FOUND;
    }
    LEAVE_LIBRARY("sctp_setRemotePrimary");
    return result;
}


/* ------------------------------END of ASCONF functions  --------------------------------*/


#ifdef BAKEOFF
int sctp_sendRawData(unsigned int associationID, unsigned int pathId,
                     unsigned char *buffer, unsigned int length)
{
    int result = 0;
    Association* currentAssociation = NULL;

    if (sctpLibraryInitialized == FALSE) return SCTP_LIBRARY_NOT_INITIALIZED;

    /* Retrieve association from list  */
    currentAssociation = mdi_getAssociationByID(associationID, FALSE);

    if (currentAssociation != NULL) {
        
        result = pm_checkValidPathKey(currentAssociation, pathId);

        if (result == SCTP_SUCCESS) {
            event_logiii(INTERNAL_EVENT_1, "sctp_sendRawData(assoc:%u, path: %d): send %u bytes",associationID, pathId,length);
            /* Forward chunk to the addressed association */
            result = mdi_send_message(currentAssociation, (SCTP_message *) buffer, length, pathId);
        } else {
            error_log(ERROR_MAJOR, "sctp_sendRawData: invalid destination address");
            return SCTP_PARAMETER_PROBLEM;
        }
    } else {
        error_log(ERROR_MAJOR, "sctp_send: addressed association does not exist");
        result = 1;
    }
    return result;
}                               /* end: sctp_send */
#endif

/*------------------- Functions called by the SCTP bundling --------------------------------------*/
                                                  
/**
 * Used by bundling to send a SCTP-datagramm.
 *
 * Bundling passes a static pointer and leaves space for common header, so
 * we can fill that header in up front !
 * Before calling send_message at the adaption-layer, this function does:
 * \begin{itemize}
 * \item add the SCTP common header to the message
 * \item convert the SCTP message to a byte string
 * \item retrieve the socket-file descriptor of the SCTP-instance
 * \item retrieve the destination address
 * \item retrieve destination port ???
 * \end{itemize}
 *
 *  @param SCTP_message     SCTP message as a struct (i.e. common header and chunks)
 *  @param length           length of complete SCTP message.
 *  @param destAddresIndex  Index of address in the destination address list.
 *  @return                 Errorcode (0 for good case: length bytes sent; 1 or -1 for error)
*/
int mdi_send_message(Association* asoc,
                     SCTP_message * message,
                     unsigned int length,
                     unsigned int destAddressKey)
{
    union sockunion dest_su, *dest_ptr;
    SCTP_simple_chunk *chunk;
    unsigned char tos = 0;
    unsigned int pathIndex=0;
    int txmit_len = 0;
    /* guchar hoststring[SCTP_MAX_IP_LEN]; */

    if (message == NULL) {
        error_log(ERROR_MINOR, "mdi_send_message: no message to send !!!");
        return 1;
    }
    chunk = (SCTP_simple_chunk *) & message->sctp_pdu[0];

    if (asoc == NULL) {
        /* possible cases : initAck, no association exists yet, and OOTB packets
           use last from address as destination address */

        /* force a termination here */
        assert(lastFromAddress);

        if (lastFromAddress == NULL) {
            error_log(ERROR_FATAL, "mdi_send_message: lastFromAddress does not exist for initAck");
            return 1;
        } else {
            /* only if the sctp-message received before contained an init-chunk */
            memcpy(&dest_su, lastFromAddress, sizeof(union sockunion));
            dest_ptr = &dest_su;
            message->common_header.verification_tag = htonl(lastInitiateTag);
            /* write invalid tag value to lastInitiateTag (reset it) */
            lastInitiateTag = 0;
            /* swap ports */
            message->common_header.src_port  = htons(mdi_readLastDestPort());
            message->common_header.dest_port = htons(mdi_readLastFromPort());
            event_logiii(VERBOSE, "mdi_send_message (I) : tag = %x, src_port = %u , dest_port = %u",
                         lastInitiateTag, mdi_readLastDestPort(), mdi_readLastFromPort());
            tos = IPTOS_DEFAULT;
        }
    } else {
        if (destAddressKey == 0) destAddressKey = pm_getPrimaryPathKey(asoc);
        assert(destAddressKey != 0);
        pathIndex = mdi_getIndexForKey(asoc, destAddressKey);

        /* force a termination here, too */
        assert(pathIndex >= 0);

        if (pathIndex < 0) {
            error_log(ERROR_MAJOR, "mdi_send_message: invalid destination address");
            return 1;
        }
        /* Use given destination address from current association */
        dest_ptr = &g_array_index(asoc->destAddresses, union sockunion, pathIndex);

        if (isInitAckChunk(chunk)) {
            if (lastInitiateTag == 0) {
                error_log(ERROR_FATAL, "mdi_send_message: No verification tag");
                return 1;
            }
            message->common_header.verification_tag = htonl(lastInitiateTag);
        } else {
            message->common_header.verification_tag = htonl(asoc->tagRemote);
        }
        message->common_header.src_port  = htons(asoc->localPort);
        message->common_header.dest_port = htons(asoc->remotePort);

        event_logiii(VERBOSE, "mdi_send_message (II): tag = %x, src_port = %u , dest_port = %u",
                     ntohl(message->common_header.verification_tag), asoc->localPort, asoc->remotePort);

        tos = asoc->ipTos;
    }

    /* calculate and insert checksum */
    aux_insert_checksum((unsigned char *) message, length);

    switch (sockunion_family(dest_ptr)) {
        case AF_INET:
#ifdef DO_PMTUD
            if(pm_pathMTUDMustBeDone(asoc, destAddressKey)
               && (length > pm_readPathSCTPMTU(asoc, destAddressKey) - PATH_MTU_COMMON_DIFFERENCE))
            {
              event_logi(INTERNAL_EVENT_0, "PMTUD done with packet of length = %u", length);
              txmit_len = adl_send_message(sctp_socket, message, length, dest_ptr, tos, TRUE);
              pm_pathMTUDWasDone(asoc, destAddressKey) ;
            }
            else
            {
              txmit_len = adl_send_message(sctp_socket, message, length, dest_ptr, tos, FALSE);
            }
#else
            txmit_len = adl_send_message(sctp_socket, message, length, dest_ptr, tos, FALSE);
#endif
            break;
#ifdef HAVE_IPV6
        case AF_INET6:
#ifdef DO_PMTUD
            if((length > 1280) && (length > (pm_readPathSCTPMTU(asoc, destAddressKey) + sizeof(SCTP_common_header))))
            {
              txmit_len = adl_send_message(ipv6_sctp_socket, message, length, dest_ptr, tos, FALSE);
              pm_pathMTUDWasDone(asoc, destAddressKey) ;
            }
            else
            {
              txmit_len = adl_send_message(ipv6_sctp_socket, message, length, dest_ptr, tos, TRUE);
            }
#else
            txmit_len = adl_send_message(ipv6_sctp_socket, message, length, dest_ptr, tos, FALSE);
#endif
            break;
#endif
        default:
            error_log(ERROR_MAJOR, "mdi_send_message: Unsupported AF_TYPE");
            break;
    }

    /* adl_sockunion2str(dest_ptr, hoststring, SCTP_MAX_IP_LEN);                            */
    /* event_logiii(INTERNAL_EVENT_0, "sent SCTP message of %d bytes to %s, result was %d", */
    /*                length, hoststring, txmit_len);                                       */

    return (txmit_len == length) ? 0 : -1;

}                               /* end: mdi_send_message */



/*------------------- Functions called by the SCTP to forward primitives to ULP ------------------*/


/**
 *  indicates new data has arrived from peer (chapter 10.2.) destined for the ULP
 *
 *  @param streamID  received data belongs to this stream
 *  @param  length   so many bytes have arrived (may be used to reserve space)
 *  @param  protoID  the protocol ID of the arrived payload
 *  @param  unordered  unordered flag (TRUE==1==unordered, FALSE==0==normal,numbered chunk)
 */
void mdi_dataArriveNotif(Association* asoc,
                         unsigned int streamID,
                         unsigned int length,
                         unsigned short streamSN,
                         unsigned int tsn,
                         unsigned int protoID,
                         unsigned int unordered)
{
    SCTP_instance * sI;

    if (asoc != NULL) {
        event_logiii(INTERNAL_EVENT_0, "mdi_dataArriveNotif(assoc %u, streamID %u, length %u)",
               asoc->assocId, streamID,  length);
        sI = asoc->sctpInstance;
        /* Forward dataArriveNotif to the ULP */
        if (sI->ULPcallbackFunctions.dataArriveNotif) {
            ENTER_CALLBACK("dataArriveNotif");
            sI->ULPcallbackFunctions.dataArriveNotif(asoc->assocId, streamID,
                                                     length, streamSN, tsn,
                                                     protoID, unordered,
                                                     asoc->ulp_dataptr);
            LEAVE_CALLBACK("dataArriveNotif");
        }
    } else {
        error_log(ERROR_MAJOR, "mdi_dataArriveNotif: association not set");
    }
}                               /* end: mdi_dataArriveNotif */



/**
 * indicates a change of network status (chapter 10.2.C). Calls the respective ULP callback function.
 * @param  destinationAddress   index to address that has changed
 * @param  newState             state to which indicated address has changed (PM_ACTIVE/PM_INACTIVE)
 */
void mdi_networkStatusChangeNotif(Association* asoc, unsigned int destinationAddress, int newState)
{
    SCTP_instance * sI;

    if (asoc != NULL) {
        sI = asoc->sctpInstance;
        event_logiii(INTERNAL_EVENT_0, "mdi_networkStatusChangeNotif(assoc %u, path-id %u, state %d)",
               asoc->assocId, destinationAddress,newState);
        if (sI->ULPcallbackFunctions.networkStatusChangeNotif) {
            ENTER_CALLBACK("networkStatusChangeNotif");
            sI->ULPcallbackFunctions.networkStatusChangeNotif(asoc->assocId,
                                                              destinationAddress, newState,
                                                              asoc->ulp_dataptr);
            LEAVE_CALLBACK("networkStatusChangeNotif");
        }
    } else {
        error_log(ERROR_MAJOR, "mdi_networkStatusChangeNotif: association not set");
    }
}                               /* end: mdi_networkStatusChangeNotif */

/**
 * indicates a send failure (chapter 10.2.B). Calls the respective ULP callback function.
 * @param data          pointer to the data that has not been sent
 * @param dataLength    length of the data that has not been sent
 * @param context       from sendChunk (CHECKME : may be obsolete ?)
 */
void mdi_sendFailureNotif(Association* asoc, unsigned char *data, unsigned int dataLength, unsigned int *context)
{
    SCTP_instance * sI;

    if (asoc != NULL) {
        sI = asoc->sctpInstance;
        if(sI->ULPcallbackFunctions.sendFailureNotif) {
            ENTER_CALLBACK("sendFailureNotif");
            sI->ULPcallbackFunctions.sendFailureNotif(asoc->assocId,
                                                      data, dataLength, context,
                                                      asoc->ulp_dataptr);
            LEAVE_CALLBACK("sendFailureNotif");
        }
    } else {
        error_log(ERROR_MAJOR, "mdi_sendFailureNotif: association not set");
    }
}                               /* end: mdi_sendFailureNotif */

/**
 * indicates that association has been gracefully shut down (chapter 10.2.H).
 * Calls the respective ULP callback function.
 */
void mdi_shutdownCompleteNotif(Association* asoc)
{
    SCTP_instance * sI;

    if (asoc != NULL) {
        sI = asoc->sctpInstance;

        event_logi(INTERNAL_EVENT_0, "mdi_shutdownCompleteNotif(assoc %u)", asoc->assocId);
        if(sI->ULPcallbackFunctions.shutdownCompleteNotif) {
            ENTER_CALLBACK("shutdownCompleteNotif");
            sI->ULPcallbackFunctions.shutdownCompleteNotif(asoc->assocId,
                                                           asoc->ulp_dataptr);
            LEAVE_CALLBACK("shutdownCompleteNotif");
        }
    } else {
        error_log(ERROR_MAJOR, "mdi_shutdownCompleteNotif: association not set");
    }
}

/**
 * indicates that a restart has occured(chapter 10.2.G).
 * Calls the respective ULP callback function.
 */
void mdi_restartNotif(Association* asoc)
{
    SCTP_instance * sI;

    if (asoc != NULL) {

        sI = asoc->sctpInstance;
        event_logi(INTERNAL_EVENT_0, "mdi_restartNotif(assoc %u)", asoc->assocId);

        if(sI->ULPcallbackFunctions.restartNotif) {
            ENTER_CALLBACK("restartNotif");
            sI->ULPcallbackFunctions.restartNotif(asoc->assocId,
                                                  asoc->ulp_dataptr);
            LEAVE_CALLBACK("restartNotif");
        }
    } else {
        error_log(ERROR_MAJOR, "mdi_restartNotif: association not set");
    }
}

/**
 * indicates that communication was lost to peer (chapter 10.2.E). Calls the respective ULP callback function.
 *
 * @param  status  type of event, that has caused the association to be terminated
 */
void mdi_communicationLostNotif(Association* asoc, int status)
{
    SCTP_instance * sI;

    if (asoc != NULL) {

        sI = asoc->sctpInstance;
        event_logii(INTERNAL_EVENT_0, "mdi_communicationLostNotif(assoc %u, status %d)",
                                        asoc->assocId, status);
        if(sI->ULPcallbackFunctions.communicationLostNotif) {
            ENTER_CALLBACK("communicationLostNotif");
            sI->ULPcallbackFunctions.communicationLostNotif(asoc->assocId,
                                                            status,
                                                            asoc->ulp_dataptr);
            LEAVE_CALLBACK("communicationLostNotif");
        }
    } else {
        error_log(ERROR_MAJOR, "mdi_communicationLostNotif: association not set");
    }
}                               /* end: mdi_communicationLostNotif */

/**
 * indicates that an association is established (chapter 10.2.D).
 *
 * @param status     type of event that caused association to come up;
 *                   either SCTP_COMM_UP_RECEIVED_VALID_COOKIE, SCTP_COMM_UP_RECEIVED_COOKIE_ACK
 *                   or SCTP_COMM_UP_RECEIVED_COOKIE_RESTART
 */
void mdi_communicationUpNotif(Association* asoc, unsigned short status)
{
    SCTP_instance * sI;
    union sockunion lastAddress;
    int result, pathNum;
    unsigned int primaryPath, pathKey;
    unsigned short noOfInStreams;
    unsigned short noOfOutStreams;

    if (asoc != NULL) {
        sI = asoc->sctpInstance;
        /* Find primary path */
        result = mdi_readLastFromAddress(&lastAddress);
        if (result != 1) {
            primaryPath = mdi_getKeyForAddress(asoc, &lastAddress);
        } else {
            /* just use the first one */
            /* CHECKME: maybe set already in path management ?! */
            primaryPath = mdi_getKeyForIndex(asoc, 0);
        }
        /* set number of paths and primary path at pathmanegement and start heartbeat */
        pm_setPaths(asoc, asoc->noOfDestAddresses, primaryPath);
        se_readNumberOfStreams(asoc, &noOfInStreams, &noOfOutStreams);

        event_logiii(VERBOSE, "Distribution: COMM-UP, assocId: %u, status: %d, noOfDestAddresses: %u",
                     asoc->assocId, status, asoc->noOfDestAddresses);
        event_logii(VERBOSE, "noOfInStreams: %u,noOfOutStreams  %u", noOfInStreams, noOfOutStreams);

        /* Forward mdi_communicationup Notification to the ULP */
        if(sI->ULPcallbackFunctions.communicationUpNotif) {
            ENTER_CALLBACK("communicationUpNotif");

            asoc->ulp_dataptr = sI->ULPcallbackFunctions.communicationUpNotif(asoc->assocId,
                                                                              status,
                                                                              asoc->noOfDestAddresses,
                                                                              noOfInStreams, noOfOutStreams,
                                                                              asoc->peerSupportsPRSCTP && asoc->supportsPRSCTP,
                                                                              asoc->adaptationLayerIndicationLen,
                                                                              asoc->adaptationLayerIndication,
                                                                              asoc->ulp_dataptr);
            LEAVE_CALLBACK("communicationUpNotif");
            for (pathNum = 0; pathNum < asoc->noOfDestAddresses; pathNum++)
            {
                pathKey = mdi_getKeyForIndex(asoc, pathNum);
                if (pm_pathConfirmed(asoc, pathKey) == TRUE)
                {
                    mdi_networkStatusChangeNotif(asoc, pathKey, PM_PATH_CONFIRMED);
                }
            }
        } else {
           asoc->ulp_dataptr = NULL;
        }
    } else {
        error_log(ERROR_MAJOR, "mdi_communicationUpNotif: association not set");
    }
}                               /* end: mdi_communicationUpNotif */

void mdi_peerShutdownReceivedNotif(Association* asoc)
{
    SCTP_instance * sI;

    if (asoc != NULL) {
        sI = asoc->sctpInstance;

        event_logi(INTERNAL_EVENT_0, "mdi_peerShutdownReceivedNotif(assoc %u)",asoc->assocId);

        if (sI->ULPcallbackFunctions.peerShutdownReceivedNotif) {
            ENTER_CALLBACK("peerShutdownReceivedNotif");
            sI->ULPcallbackFunctions.peerShutdownReceivedNotif(asoc->assocId,
                                                               asoc->ulp_dataptr);
            LEAVE_CALLBACK("peerShutdownReceivedNotif");
        }
    } else {
        error_log(ERROR_MAJOR, "mdi_peerShutdownReceivedNotif: association not set");
    }
}

/**
 * Function that notifies the ULP of a change in the queue status.
 * I.e. a limit may be exceeded, and therefore subsequent send-primitives will
 * fail, OR the queue length has dropped below a previously set queue length
 *
 * @param  queueType i.e. an outbound queue, stream-engine queue, per stream queue (?)
 * @param  queueId   i.e. i.e. stream id for a per stream queue
 * @param  queueLen  in bytes or in messages, depending on the queue type
 */
void mdi_queueStatusChangeNotif(Association* asoc, int queueType, int queueId, int queueLen)
{
    SCTP_instance * sI;

    if (asoc != NULL) {
        sI = asoc->sctpInstance;

        event_logiiii(INTERNAL_EVENT_0, "mdi_queueStatusChangeNotif(assoc %u, queueType %d, queueId %d, len: %d)",
                                         asoc->assocId, queueType, queueId, queueLen);

        if (sI->ULPcallbackFunctions.queueStatusChangeNotif) {
            ENTER_CALLBACK("queueStatusChangeNotif");
            sI->ULPcallbackFunctions.queueStatusChangeNotif(asoc->assocId,
                                                            queueType, queueId, queueLen,
                                                            asoc->ulp_dataptr);
            LEAVE_CALLBACK("queueStatusChangeNotif");
        }
    } else {
        error_log(ERROR_MAJOR, "mdi_queueuStatusChangeNotif: association not set");
    }
}                               /* end: mdi_queueStatusChangeNotif */



/*------------------- Functions called by the SCTP to get current association data----------------*/

/* When processing external events from outside the SCTP (socket events, timer events and
   function calls from the ULP), first the data of the addressed association are read
   from the list of associations and stored in a private but static datastructure.
   Elements of this association data can be read by the following functions.
*/
unsigned int mdi_getUnusedAssocId(void)
{
    Association * tmp = NULL;
    unsigned int newId;

    do {
        newId =  nextAssocId;
        tmp =    mdi_getAssociationByID(newId, TRUE);
        nextAssocId++;
    } while (tmp != NULL);
    return newId;
}

/**
 * generates a random tag value for a new association, but not 0
 * @return   generates a random tag value for a new association, but not 0
 */
unsigned int mdi_generateTag(void)
{
    unsigned int tag;
    while ((tag = adl_random()) == 0);
    return tag;
}

/**
 * generates a random tsn value for a new association (may also be 0)
 * @return   generates a random tsn value for a new association (may also be 0)
 */
unsigned int mdi_generateStartTSN(void)
{
#ifdef BAKEOFF
    long int tmp = adl_random();
    if ((tmp%3) == 0) return 0;
    if ((tmp%3) == 1) return 0xFFFFFFFF-100;
    return tmp;
#else
    return adl_random();
#endif
}

/*------------- functions for the cookie mechanism --------------------------------------------*/

/**
 * sets the address from which the last datagramm was received (host byte order).
 * @return  0 if successful, 1 if address could not be set !
 */
int mdi_readLastFromAddress(union sockunion* fromAddress)
{
    if (lastFromAddress == NULL) {
        error_log(ERROR_FATAL, "mdi_readLastFromAddress: no last from address");
    } else {
        memcpy(fromAddress, lastFromAddress, sizeof(union sockunion));
        return 0;
    }
    return 1;
}


/**
 * read the index of the path from which the last DG was received (-1 if no DG was received)
 * @return index of the path from which the last DG was received (-1 if no DG was received)
 */
unsigned int mdi_readLastFromPathKey(void)
{
    return lastFromPathKey;
}

int mdi_readLastFromPathIndex(void)
{
    return lastFromPathIndex;
}

/**
 * read the port of the sender of the last received DG (host byte order)
 * @return the port of the sender of the last received DG (host byte order)
 */
unsigned short mdi_readLastFromPort(void)
{
    if (lastFromAddress == NULL) {
        error_log(ERROR_FATAL, "readLastFromPort: no last from address");
        return 0;
    } else {
        return lastFromPort;
    }
}


/**
 * read the port of the destination of the last received DG (host byte order)
 * @return the port of the destination of the last received DG (host byte order)
 */
unsigned short mdi_readLastDestPort(void)
{
    if (lastFromAddress == NULL) {
        error_log(ERROR_FATAL, "readLastDestPort: no last from address");
        return 0;
    } else {
        return lastDestPort;
    }
}


/* return the last initiate tag  */
unsigned int mdi_readLastInitiateTag(void)
{
    return lastInitiateTag;
}

/* write the initiate tag of a-side to be used as verification tag for the initAck */
void mdi_writeLastInitiateTag(unsigned int initiateTag)
{
    lastInitiateTag = initiateTag;
}

/* rewrite the initiate tag of peer in case of a peer reset. */
void mdi_rewriteTagRemote(Association* asoc, unsigned int newInitiateTag)
{
    if (asoc == NULL) {
        error_log(ERROR_FATAL, "mdi_rewriteRemoteTag: association not set");
    } else {
        asoc->tagRemote = newInitiateTag;
    }
}

/* rewrite the initiate tag of peer in case of a peer reset. */
void mdi_rewriteLocalTag(Association* asoc, unsigned int newTag)
{
    if (asoc == NULL) {
        error_log(ERROR_FATAL, "mdi_rewriteLocalTag: association not set");
    } else {
        asoc->tagLocal = newTag;
    }
}

/*------------- functions to write and read addresses --------------------------------------------*/

int mdi_getIndexForAddress(Association* asok, union sockunion* address)
{
    int index = 0;

    if (asok == NULL) {
        error_log(ERROR_MAJOR, "mdi_getIndexForAddress: association not set");
        return -1;
    } else {
        if (asok->destAddresses == NULL) {
            error_logi(ERROR_MINOR, "mdi_getIndexForAddress: addresses for asoc %u not set", asok->assocId);
            return -1;
        }
        /* send cookie back to the address where we got it from     */
        for (index = 0; index < asok->noOfDestAddresses; index++) {
            if (adl_equal_address(
                    &g_array_index(asok->destAddresses, union sockunion, index),
                    address) == TRUE){
                    break;
            }
        }

        if (index == asok->noOfDestAddresses) /* not found */ {
            return -1;
        }
    }
    event_logi(VERBOSE, "mdi_getIndexForAddress returns %d",index);
    return index;
}

/**
 * get the key corresponding to a destination path address
 * address is associated with the key (the return value) for all modules
 */
unsigned int mdi_getKeyForAddress(Association* asok,union sockunion* address)
{
    int index;
    assert(asok);

    for (index = 0; index < asok->noOfDestAddresses; index++) {
        if (adl_equal_address(address,
                            &g_array_index(asok->destAddresses, union sockunion, index))){
            /* got it */
            event_logi(VERBOSE, "mdi_getKeyForAddress returns %u",
                        g_array_index(asok->destAddressKeys, unsigned int, index));

            return (g_array_index(asok->destAddressKeys, unsigned int, index));
        }
    }
    /* haven't got it */
    return 0;
}

/**
 * given a key of a path identifier, returns the corresponding, currently valid
 * index into the array - valid for all modules
 * @param asok the asok for which we search the index
 * @param pathKey the key for which we search the index
 * @return SCTP_PARAMETER_PROBLEM if pathKey not in list, else the index to the passed pathKey
 */
int mdi_getIndexForKey(Association* asok, unsigned int pathKey)
{
    int i;
    assert(asok);
    for (i = 0; i < asok->noOfDestAddresses; i++) {
        if (g_array_index(asok->destAddressKeys, unsigned int, i) == pathKey) {
            event_logii(VERBOSE, "Function mdi_getIndexForKey(%u) returns %d", pathKey, i);
            return i;
        }
    }
    event_logi(VERBOSE, "Function mdi_getIndexForKey(%u) did not find index", pathKey);
    return SCTP_PARAMETER_PROBLEM;
}

/**
 * given a key of a path identifier, returns the corresponding, currently valid
 * index into the array - valid for all modules
 */
unsigned int mdi_getKeyForIndex(Association* asok, int pathIndex)
{
    assert(asok);

    if (pathIndex >= asok->noOfDestAddresses) return 0;
    if (pathIndex < 0) return 0;

    event_logii(VERBOSE, "Function mdi_getKeyForIndex(%d) returns: %u",
                pathIndex, g_array_index(asok->destAddressKeys, unsigned int, pathIndex));

    return (g_array_index(asok->destAddressKeys, unsigned int, pathIndex));
}


/*------------- Functions for the ADDIP-DELIP thing --------------------------------------------*/

/**
 * adds local address to the passed association
 * @param newAddress    the address that has been added 
 * @aram asoc          the association where the address is added
 */
int mdi_addLocalAddressToAssoc(Association* asoc, union sockunion* newAddress)
{
    guchar newAddrString[SCTP_MAX_IP_LEN];

    adl_sockunion2str(newAddress, newAddrString, SCTP_MAX_IP_LEN);
    event_logiii(VERBOSE, "mdi_addLocalAddressToAssoc: asoc=%u, new address: %s, key=%u",
                            asoc->assocId, newAddrString, asoc->lastLocalKey);

    asoc->localAddresses = g_array_append_vals(asoc->localAddresses, newAddress, 1);
    asoc->localAddressKeys = g_array_append_val(asoc->localAddressKeys, asoc->lastLocalKey);
    asoc->lastLocalKey++;
    asoc->noOfLocalAddresses++;
    
    return SCTP_SUCCESS;
}

/**
 * adds destination address to the passed association
 * Also adds appropriate structures in pathmanagement, reliableTransfer, flowControl.
 *  
 * @param newAddress    the address that has been added by an AS-CONF
 * @aram asoc          the association where the address is added
        error_log(ERROR_MAJOR, "mdi_addDestinationAddressToAssoc: try to add known address !"); */
unsigned int mdi_addDestinationAddressToAssoc(Association* asoc,
                                     union sockunion* newAddress)
{
    int newIndex = -1, result = 0;
    unsigned int newKey = 0;
    guchar newAddrString[SCTP_MAX_IP_LEN];

    if (asoc->destAddresses == NULL || asoc->destAddressKeys == NULL) {
            error_log(ERROR_FATAL, "mdi_addDestinationAddressToAssoc: arrays not initialized");
            return 0;
    }
    if (mdi_getKeyForAddress(asoc, newAddress) != 0) {
            error_log(ERROR_MAJOR, "mdi_addDestinationAddressToAssoc: try to add known address !");
            return 0;
    }
    asoc->destAddresses   = g_array_append_vals(asoc->destAddresses, newAddress, 1);
    asoc->destAddressKeys = g_array_append_val(asoc->destAddressKeys, asoc->lastDestKey);
    newKey =    asoc->lastDestKey;
    newIndex =  asoc->noOfDestAddresses;
    asoc->lastDestKey++;
    asoc->noOfDestAddresses++;
    
    adl_sockunion2str(newAddress, newAddrString, SCTP_MAX_IP_LEN);
    event_logiiii(VERBOSE, "mdi_addDestinationAddress: asoc=%u, new address: %s, newKey=%u, newIndex=%d",
                            asoc->assocId, newAddrString, newKey, newIndex);

    pm_addDestinationAddress(asoc, newKey, newIndex);
    
    fc_addDestinationAddress(asoc, newKey, newIndex);
    
    rtx_addDestinationAddress(asoc,newKey, newIndex);

    /* this is a hack for mobility testing :-) */
    result = pm_setPrimaryPathKey(asoc, newKey);
    if (result != SCTP_SUCCESS) error_log(ERROR_MAJOR, "mdi_addDestinationAddressToAssoc: Setting new primary failed !");
    /* this is a hack for mobility testing :-) */    
    return newKey;
}

/**
 * deletes destination address from the passed association
 * Also does this for structures in pathmanagement, reliableTransfer, flowControl.
 *
 * @param newAddress    the address that has been added by an AS-CONF
 * @aram asoc          the association where the address is added
 */
int mdi_delLocalAddressFromAssoc(Association* asoc,
                                       union sockunion* delAddress)
{
    int index;

    for (index = 0; index < asoc->noOfLocalAddresses; index++) {
        if (adl_equal_address(
                &g_array_index(asoc->localAddresses, union sockunion, index),
                delAddress) == TRUE){
                break;
        }
    }

    if (index == asoc->noOfLocalAddresses) /* not found */ {
        return SCTP_WRONG_ADDRESS;
    }
    
    asoc->noOfLocalAddresses--;
    asoc->localAddresses   = g_array_remove_index(asoc->localAddresses, index);
    asoc->localAddressKeys = g_array_remove_index(asoc->localAddressKeys, index); 
    return SCTP_SUCCESS;
}

/**
 * deletes destination address from the passed association
 * Also does this for structures in pathmanagement, reliableTransfer, flowControl.
 *
 * @param newAddress    the address that has been added by an AS-CONF
 * @aram asoc          the association where the address is added
 */
int mdi_delDestinationAddressFromAssoc(Association* asoc,
                                       union sockunion* delAddress)
{
    int delIndex, alterIndex;
    unsigned int delKey, alterKey;

    if (asoc->destAddresses == NULL || asoc->destAddressKeys == NULL) {
            error_log(ERROR_FATAL, "mdi_addDestinationAddressToAssoc: arrays not initialized");
            return SCTP_MODULE_NOT_FOUND;
    }
    /* check if the address is there, at all */
    if ((delKey = mdi_getKeyForAddress(asoc, delAddress)) == 0) {
            error_log(ERROR_MAJOR, "mdi_delDestinationAddressFromAssoc: try to delete unknown address !");
            return SCTP_PARAMETER_PROBLEM;
    }
    if ((delIndex = mdi_getIndexForAddress(asoc, delAddress)) == -1) {
            error_log(ERROR_FATAL, "mdi_delDestinationAddressFromAssoc: Problem with Program Logic !");
            return SCTP_UNSPECIFIED_ERROR;
    }

    alterIndex = (delIndex + 1) % asoc->noOfDestAddresses;
    assert (alterIndex != delIndex);
    alterKey = mdi_getKeyForIndex(asoc, alterIndex);
    assert(alterKey != 0);
    
    /* if it is, remove from the array, and shift the array */
    asoc->noOfDestAddresses--;
    asoc->destAddresses   = g_array_remove_index(asoc->destAddresses, delIndex);
    asoc->destAddressKeys = g_array_remove_index(asoc->destAddressKeys, delIndex); 

    pm_delDestinationAddress(asoc, delKey, delIndex);
    fc_delDestinationAddress(asoc, delKey, delIndex, alterKey, alterIndex);
    rtx_delDestinationAddress(asoc, delKey, delIndex, alterKey, alterIndex);

    return SCTP_SUCCESS;
}

/**
 * adds address to  an sctp instance structure, if IN(6)ADDR_ANY had been specified
 * @param newAddress    the address that has been added 
 * @aram ins           the instance where the address is added
 */
int mdi_addressUpdate(void)
{
    int sfd=0, oldNum = 0, newMaxMTU = 0;

    sfd = adl_get_sctpv4_socket();
    if (myAddressList != NULL)  free(myAddressList);
    oldNum = myNumberOfAddresses;

    min_MTU = DEFAULT_MTU_CEILING;
    event_logi(VVERBOSE, "mdi_addressUpdate: number of addresses before: %d", oldNum);
    if (adl_gatherLocalAddresses(&myAddressList, (int*)&myNumberOfAddresses, sfd, TRUE, &newMaxMTU, &min_MTU, flag_Default) == FALSE) {
        LEAVE_LIBRARY("sctp_initLibrary");
        return SCTP_SPECIFIC_FUNCTION_ERROR;
    }

    event_logi(VVERBOSE, "mdi_addressUpdate: number of addresses after: %d", myNumberOfAddresses);

    return SCTP_SUCCESS;
}

/**
 * adds address to  an sctp instance structure
 * @param newAddress    the address that has been added
 * @aram ins           the instance where the address is added
 */
int mdi_addAddressToInstanceForced(SCTP_instance* ins, union sockunion* addAddress)
{
    unsigned int highestKey = 0, currentKey = 0;
    int cnt;

    if (ins->numberOfLocalAddresses == 0) return SCTP_NOT_SUPPORTED;
    
    for (cnt = 0; cnt < ins->numberOfLocalAddresses; cnt++) {
        currentKey = g_array_index(ins->localAddressKeys, unsigned int, cnt);
            if (currentKey > highestKey) highestKey = currentKey;
    }
    highestKey++;
    ins->localAddresses = g_array_append_vals(ins->localAddresses, addAddress, 1);
    ins->localAddressKeys = g_array_append_val(ins->localAddressKeys, highestKey);
    ins->numberOfLocalAddresses++;

    return SCTP_SUCCESS;
}

/**
 * takes away an address from an sctp instance structure
 * @param delAddress    the address that is being deleted
 * @aram asoc          the instance where the address to be removed
 */
int mdi_delAddressFromInstance(SCTP_instance* ins, union sockunion* delAddress)
{
    int idx;

    if (ins->numberOfLocalAddresses == 1) {
        error_log(ERROR_FATAL, "mdi_delAddressFromInstance: try to delete last address !");
        exit(-3);
        return SCTP_SPECIFIC_FUNCTION_ERROR;
    }
    
    for (idx = 0; idx < ins->numberOfLocalAddresses; idx++) {
        if (adl_equal_address(&g_array_index(ins->localAddresses, union sockunion, idx),
                              delAddress) == TRUE) {
            break;
        }
    }

    if (idx == ins->numberOfLocalAddresses) return SCTP_WRONG_ADDRESS;

    ins->localAddresses   = g_array_remove_index(ins->localAddresses, idx);
    ins->localAddressKeys = g_array_remove_index(ins->localAddressKeys, idx);
    ins->numberOfLocalAddresses--;
    return SCTP_SUCCESS;
}

/**
 * copies destination addresses from the array passed as parameter to  the current association
 * @param addresses array that will hold the destination addresses after returning
 * @aram noOfAddresses number of addresses that the peer has (and sends along in init/initAck)
 */
int mdi_writeDestinationAddresses(Association* asoc,
                                  union sockunion addresses[MAX_NUM_ADDRESSES],
                                  int noOfAddresses)
{
    int index;
    unsigned initialKey = 1;

    if (asoc == NULL) {
        error_log(ERROR_MAJOR, "mdi_writeDestinationAddresses: association not set");
        return -1;
    } else {
        if (asoc->destAddresses) {
            error_log(ERROR_MINOR, "mdi_writeDestinationAddresses: overwriting old address...");
            g_array_free(asoc->destAddresses, TRUE);
        }
        if (asoc->destAddressKeys) {
            error_log(ERROR_MINOR, "mdi_writeDestinationAddresses: overwrite old keys");
            g_array_free(asoc->destAddressKeys, TRUE);
        }
        asoc->noOfDestAddresses = noOfAddresses;
        asoc->destAddresses     = g_array_new(FALSE, FALSE, sizeof(union sockunion));
        asoc->destAddressKeys   = g_array_new(FALSE, FALSE, sizeof(unsigned int));

        if (asoc->destAddresses == NULL || asoc->destAddressKeys == NULL) {
            error_log(ERROR_FATAL, "mdi_writeDestinationAddresses: out of memory");
            return -1;
        }
        for (index = 0; index < noOfAddresses; index++) {
            asoc->destAddresses   = g_array_append_val(asoc->destAddresses, addresses[index]);
            asoc->destAddressKeys = g_array_append_val(asoc->destAddressKeys, initialKey);
            initialKey++;
        }
        asoc->lastDestKey = initialKey;
    }
    return 0;
}


/**
 * Copies local addresses of this instance into the array passed as parameter
 * this is for adding all our relevant addresses to INIT/INIT ACK chunks
 * CHECKME : does this function work in all circumstances ?
 * --> Under what conditions can we NOT find the SCTP instance ?
 *
 * @param addresses array that will hold the local host's addresses after returning
 * @aram noOfAddresses number of addresses that local host/current association has
 * @aram paddress      pointer to the destination address of INIT/INIT ACK chunk
 */
void mdi_readLocalAddresses(Association* asok,
                            SCTP_instance* sctpInstance,
                            union sockunion laddresses[MAX_NUM_ADDRESSES],
                            unsigned int * noOfAddresses,
                            union sockunion *paddress,
                            unsigned int numPeerAddresses,
                            unsigned int addressTypes,
                            gboolean receivedFromPeer)
{
    unsigned int count=0, tmp;
    AddressScopingFlags filterFlags;
    gboolean localHostFound=FALSE, linkLocalFound = FALSE, siteLocalFound = FALSE;

    if (asok == NULL && sctpInstance == NULL) {
        error_log(ERROR_FATAL, "mdi_readLocalAddresses: both assoc and instance not set - error !");
        *noOfAddresses = 0;
        return;
    }
    if (sctpInstance == NULL) {
        error_log(ERROR_FATAL, "mdi_readLocalAddresses: instance not set - error !");
        sctpInstance = asok->sctpInstance;
    }

    for (count = 0; count <  numPeerAddresses; count++)  {
        localHostFound = localHostFound || mdi_addressContainsLocalhost(sctpInstance, &paddress[count]);
        linkLocalFound = linkLocalFound || !( adl_filterInetAddress(&paddress[count], flag_HideLinkLocal));
        siteLocalFound = siteLocalFound || !( adl_filterInetAddress(&paddress[count], flag_HideSiteLocal));
    }

    event_logiii(VERBOSE, "mdi_readLocalAddresses: localHostFound=%s, linkLocalFound=%s, siteLocalFound=%s",
                (localHostFound == TRUE)?"TRUE":"FALSE", (linkLocalFound == TRUE)?"TRUE":"FALSE", (siteLocalFound == TRUE)?"TRUE":"FALSE");

    /* if (receivedFromPeer == FALSE) I send an INIT with my addresses to the peer */
    if ((receivedFromPeer == FALSE) && (localHostFound == TRUE)) {
        /* if paddress == loopback then add my loopback */
        filterFlags = flag_Default;
    } else if ((receivedFromPeer == FALSE) && (localHostFound == FALSE)) {
        /* only add loopback, if sending to a loopback */
        filterFlags = flag_Default|flag_HideLocal;

    /* if (receivedFromPeer == TRUE) I got an INIT with addresses from the peer */
    } else if ((receivedFromPeer == TRUE) && (localHostFound == FALSE)) {
        /* this is from a normal address, get all except loopback */
        if (linkLocalFound) {
            filterFlags =  flag_HideAllExceptLinkLocal;
        } else if (siteLocalFound) {
            filterFlags =  flag_Default|flag_HideLinkLocal|flag_HideLoopback;
        } else {
            filterFlags = flag_Default|flag_HideLocal;
        }
    } else  /* if ((receivedFromPeer == TRUE) && (localHostFound == TRUE)) */ {
        /* this is from a loopback, get all my addresses */
        filterFlags = flag_Default; 
    }

    count = 0;
    
    if (sctpInstance->has_INADDR_ANY_set == TRUE) {
        for (tmp = 0; tmp < myNumberOfAddresses; tmp++) {
            switch(sockunion_family( &(myAddressList[tmp]))) {
                case AF_INET :
                    if ((addressTypes & SUPPORT_ADDRESS_TYPE_IPV4) != 0) {
                        if ( adl_filterInetAddress(&(myAddressList[tmp]), filterFlags) == TRUE) {
                            memcpy(&(laddresses[count]), &(myAddressList[tmp]),sizeof(union sockunion));
                            count++;
                        }
                    }
                    break;
                default: break;
            }
        }
        event_logii(VERBOSE, "mdi_readLocalAddresses: found %u local addresses from INADDR_ANY (from %u)", count,myNumberOfAddresses );
    } else if (sctpInstance->has_IN6ADDR_ANY_set == TRUE) {
        for (tmp = 0; tmp < myNumberOfAddresses; tmp++) {
            switch(sockunion_family( &(myAddressList[tmp]))) {
                case AF_INET :
                    if ((addressTypes & SUPPORT_ADDRESS_TYPE_IPV4) != 0) {
                        if ( adl_filterInetAddress(&(myAddressList[tmp]), filterFlags) == TRUE) {
                            memcpy(&(laddresses[count]), &(myAddressList[tmp]),sizeof(union sockunion));
                            count++;
                        }
                    }
                    break;
#ifdef HAVE_IPV6
                case AF_INET6 :
                    if ((addressTypes & SUPPORT_ADDRESS_TYPE_IPV6) != 0) {
                        if ( adl_filterInetAddress(&(myAddressList[tmp]), filterFlags) == TRUE) {
                            memcpy(&(laddresses[count]), &(myAddressList[tmp]),sizeof(union sockunion));
                            count++;
                        }
                    }
                    break;
#endif
                default: break;
            }
        }
        event_logii(VERBOSE, "mdi_readLocalAddresses: found %u local addresses from IN6ADDR_ANY (from %u)", count, myNumberOfAddresses);

    } else {
        for (tmp = 0; tmp < sctpInstance->numberOfLocalAddresses; tmp++) {

            switch(sockunion_family(&g_array_index(sctpInstance->localAddresses ,union sockunion,tmp))) {
                case AF_INET :
                    if ((addressTypes & SUPPORT_ADDRESS_TYPE_IPV4) != 0) {
                        if (adl_filterInetAddress(&g_array_index(sctpInstance->localAddresses ,union sockunion,tmp), filterFlags) == TRUE) {
                            memcpy(&(laddresses[count]),
                                   &g_array_index(sctpInstance->localAddresses ,union sockunion,tmp),
                                   sizeof(union sockunion));
                            count++;
                        }
                    }
                    break;
#ifdef HAVE_IPV6
                case AF_INET6 :
                    if ((addressTypes & SUPPORT_ADDRESS_TYPE_IPV6) != 0) {
                        if (adl_filterInetAddress(&g_array_index(sctpInstance->localAddresses ,union sockunion,tmp), filterFlags) == TRUE) {
                            memcpy(&(laddresses[count]),
                                   &g_array_index(sctpInstance->localAddresses ,union sockunion,tmp),
                                   sizeof(union sockunion));
                            count++;
                        }
                    }
                    break;
#endif
                default: break;
            }
        }
        event_logii(VERBOSE, "mdi_readLocalAddresses: found %u local addresses from instance (from %u)", count,
            sctpInstance->numberOfLocalAddresses);
    }
    event_logi(INTERNAL_EVENT_0, "mdi_readLocalAddresses() : returning %u addresses !",count);
    error_logii(ERROR_MAJOR, "mdi_readLocalAddresses() : found %u addresses (from %u local) !",count, myNumberOfAddresses);
    if (count == 0) exit(-100);

    *noOfAddresses = count;
}


/*------------------- Functions to create and delete associations --------------------------------*/

/**
 *  This function allocates memory for a new association.
 *  For the active side of an association, this function is called when ULP calls Associate
 *  For the passive side this function is called when a valid cookie message is received.
 *  It also creates all the modules path management, bundling and SCTP-control.
 *  The rest of the modules are created with mdi_initAssociation.
 *  The created association is put into the list of associations.
 *
 *  @param SCTP_InstanceName    identifier for an SCTP instance (if there are more)
 *  @param  local_port          src port (which this association listens to)
 *  @param  remote_port         destination port (peers source port)
 *  @param   tagLocal           randomly generated tag belonging to this association
 *  @param  primaryDestinitionAddress   index of the primary address
 *  @param  noOfDestinationAddresses    number of addresses the peer has
 *  @param  destinationAddressList      pointer to the array of peer's addresses
 *  @return 0 for success, else 1 for failure
 */
Association*   mdi_newAssociation(SCTP_instance* sInstance,
                                  unsigned short local_port,
                                  unsigned short remote_port,
                                  unsigned int tagLocal,
                                  unsigned int primaryDestinationAddress,
                                  unsigned int noOfDestinationAddresses,
                                  union sockunion *destinationAddressList)
{
    int i;
    Association* newAssoc = NULL;
    unsigned int initialKey = 1;

    if (!sInstance) {
        error_log(ERROR_MAJOR, "instance is NULL ! ");
        return NULL;
    }

    event_logiiiii(VERBOSE," mdi_newAssociation: Instance: %d, local port %u, rem.port: %u, local tag: %u, primary: %u",
           sInstance->sctpInstanceName, local_port, remote_port, tagLocal, primaryDestinationAddress);

    /* Do plausi checks on the addresses. */
    if (noOfDestinationAddresses > SCTP_MAX_NUM_ADDRESSES || destinationAddressList == NULL) {
        error_log(ERROR_MAJOR, "Problem with addressesfor new association");
        return NULL;
    }
    if (primaryDestinationAddress > noOfDestinationAddresses) {
        error_log(ERROR_MAJOR, "Invalid primary destination address for new association");
        return NULL;
    }
    newAssoc = (Association *) malloc(sizeof(Association));
    if (!newAssoc) {
        error_log_sys(ERROR_FATAL, errno);
        return NULL;
    }

    newAssoc->sctpInstance  = sInstance;
    newAssoc->localPort     = local_port;
    newAssoc->remotePort    = remote_port;
    newAssoc->tagLocal      = tagLocal;
    newAssoc->assocId       = mdi_getUnusedAssocId();
    newAssoc->tagRemote     = 0;
    newAssoc->deleted       = FALSE;

    newAssoc->ulp_dataptr   = NULL;
    newAssoc->ipTos         = sInstance->default_ipTos;
    newAssoc->maxSendQueue  = sInstance->default_maxSendQueue;
    newAssoc->localAddresses     = g_array_new(FALSE, FALSE, sizeof(union sockunion));
    newAssoc->localAddressKeys   = g_array_new(FALSE, FALSE, sizeof(unsigned int));

    /* assign local addresses first */
    if (sInstance->has_IN6ADDR_ANY_set) {
        /* get ALL addresses */
        newAssoc->noOfLocalAddresses = myNumberOfAddresses;

        for (i=0; i< myNumberOfAddresses; i++) {
            newAssoc->localAddresses   = g_array_append_val(newAssoc->localAddresses, (myAddressList[i]));
            newAssoc->localAddressKeys = g_array_append_val(newAssoc->localAddressKeys, initialKey);
            initialKey++;
        }
        newAssoc->lastLocalKey = initialKey;
        event_logi(VERBOSE," mdi_newAssociation: Assoc has has_IN6ADDR_ANY_set, and %d addresses",myNumberOfAddresses);

    } else if (sInstance->has_INADDR_ANY_set) {
        /* get all IPv4 addresses */
        newAssoc->noOfLocalAddresses = 0;
        for (i = 0; i <  myNumberOfAddresses; i++) {
            if (sockunion_family(&(myAddressList[i])) == AF_INET) {
                newAssoc->noOfLocalAddresses++;
                newAssoc->localAddresses   = g_array_append_val(newAssoc->localAddresses, (myAddressList[i]));
                newAssoc->localAddressKeys = g_array_append_val(newAssoc->localAddressKeys, initialKey);
                initialKey++;
            }
        }
        newAssoc->lastLocalKey = initialKey;
        event_logi(VERBOSE," mdi_newAssociation: Assoc has has_INADDR_ANY_set, and %d addresses",newAssoc->noOfLocalAddresses);
    } else {
        /* get all specified addresses */
        newAssoc->noOfLocalAddresses = sInstance->numberOfLocalAddresses;
        for (i = 0; i < newAssoc->noOfLocalAddresses ; i++) {
            newAssoc->localAddresses   = g_array_append_val(newAssoc->localAddresses,
                                                            g_array_index(sInstance->localAddresses,union sockunion, i));
            newAssoc->localAddressKeys = g_array_append_val(newAssoc->localAddressKeys, initialKey);
            initialKey++;
        }
        newAssoc->lastLocalKey = initialKey;
    }
    newAssoc->had_IN6ADDR_ANY_set   = sInstance->has_IN6ADDR_ANY_set;
    newAssoc->had_INADDR_ANY_set    = sInstance->has_INADDR_ANY_set;

    newAssoc->noOfDestAddresses     = noOfDestinationAddresses;
    newAssoc->destAddresses     = g_array_new(FALSE, FALSE, sizeof(union sockunion));
    newAssoc->destAddressKeys   = g_array_new(FALSE, FALSE, sizeof(unsigned int));
    initialKey = 1;

    newAssoc->destAddresses   = g_array_append_vals(newAssoc->destAddresses,destinationAddressList, noOfDestinationAddresses);
    for (i = 0; i <  noOfDestinationAddresses; i++) {
        newAssoc->destAddressKeys = g_array_append_val(newAssoc->destAddressKeys, initialKey);
        initialKey++;
    }

    /* check if newly created association already exists. Addresses must be assigned before that */
    if (checkForExistingAssociations(newAssoc) == 1) {
        error_log(ERROR_MAJOR, "tried to establish an existing association");
        g_array_free(newAssoc->localAddresses, TRUE);
        g_array_free(newAssoc->localAddressKeys, TRUE);
        g_array_free(newAssoc->destAddresses, TRUE);
        g_array_free(newAssoc->destAddressKeys, TRUE);
        free(newAssoc);
        return NULL;
    }
    /* initialize pointer to other modules of SCTP */
    newAssoc->flowControl       = NULL;
    newAssoc->reliableTransfer  = NULL;
    newAssoc->rx_control        = NULL;
    newAssoc->streamengine      = NULL;
    newAssoc->sctp_asconf       = NULL;

    /* only pathman, bundling and sctp-control are created at this point, the rest is created
       with mdi_initAssociation */
    newAssoc->bundling      = bu_new();
    newAssoc->pathMan       = pm_newPathman(newAssoc, noOfDestinationAddresses, primaryDestinationAddress);
    newAssoc->sctp_control  = sci_newSCTP_control(newAssoc, sInstance);

    event_logii(INTERNAL_EVENT_1, "new Association created ID=%08x, local tag=%08x",
                                   newAssoc->assocId, newAssoc->tagLocal);

    newAssoc->supportsPRSCTP        = librarySupportsPRSCTP;
    newAssoc->supportsADDIP         = librarySupportsADDIP;
    newAssoc->peerSupportsPRSCTP    = FALSE;
    newAssoc->peerSupportsADDIP     = FALSE;
    newAssoc->adaptationLayerIndicationLen = 0;
    newAssoc->adaptationLayerIndication = NULL;

    /* Enter association into list */
    event_logi(INTERNAL_EVENT_0, "putting association %08x into list", newAssoc->assocId);
    AssociationList = g_list_insert_sorted(AssociationList, newAssoc, &compareAssociationIDs);

    return newAssoc;
}                               /* end: mdi_newAssociation */


/**
 * This is the second function needed to fully create and initialize an association (after
 * mdi_newAssociation()) The association is created in two steps because data become available
 * at the  client side in two steps
 * \begin{enumerate}
 * \item associate
 * \item init acknowledgement
 * \end{enumerate}
 * At the server side, with the cookie message all data is available at once. So mdi_newAssociation
 * and mdi_initAssociation must be called when the initAck with valid Cookie is received.
 *
 * @param  remoteSideReceiverWindow  rwnd size that the peer allowed in this association
 * @param  noOfInStreams  number of incoming (receive) streams after negotiation
 * @param  noOfOutStreams number of outgoing (send) streams after negotiation
 * @param  remoteInitialTSN     initial  TSN of the peer
 * @param  tagRemote            tag of the peer
 * @param  localInitialTSN      my initial TSN, needed for initializing my flow control
 * @return 0 for success, else 1 for error
*/
int mdi_initAssociation(Association* asok,
                        unsigned int remoteSideReceiverWindow,
                        unsigned short noOfInStreams,
                        unsigned short noOfOutStreams,
                        unsigned int remoteInitialTSN,
                        unsigned int tagRemote, unsigned int localInitialTSN,
                        gboolean peerSupportsPRSCTP, gboolean peerSupportsADDIP,
                        int adaptationIndicationLen, void* adaptationIndication)
{
    if (!asok) {
        error_log(ERROR_MAJOR, "mdi_initAssociation: association pointer is NULL");
        return 1;
    }

    /*
     * if  mdi_initAssociation has already be called, delete modules and make new ones
     * with possibly new data. Multiple calls of of mdi_initAssociation can occur on the
     * client side in the case of stale cookie errors.
     */
    if (asok->tagRemote != 0) {
        event_log(INTERNAL_EVENT_1, "Deleting and re-creating modules in mdi_initAssociation() !!!!");
        /* association init was already completed */
        fc_delete_flowcontrol(asok);
        rtx_delete_reltransfer(asok);
        rxc_delete_recvctrl(asok);
        se_delete_stream_engine(asok);
        asc_delete(asok);
    }

    asok->tagRemote = tagRemote;
    asok->peerSupportsPRSCTP = peerSupportsPRSCTP;
    asok->peerSupportsADDIP  = peerSupportsADDIP;

    asok->reliableTransfer = rtx_new_reltransfer(asok, asok->noOfDestAddresses, localInitialTSN);

    asok->flowControl      = fc_new_flowcontrol(asok, remoteSideReceiverWindow, localInitialTSN,
                                                asok->noOfDestAddresses, asok->maxSendQueue);

    asok->rx_control       = rxc_new_recvctrl(asok, remoteInitialTSN);

    asok->streamengine     = se_new_stream_engine(asok, noOfInStreams, noOfOutStreams,
                                                   asok->supportsPRSCTP && asok->peerSupportsPRSCTP);

    asok->sctp_asconf      = asc_new(asok, localInitialTSN, remoteInitialTSN);

    asok->adaptationLayerIndicationLen = adaptationIndicationLen;
    if (adaptationIndicationLen >0) {
        asok->adaptationLayerIndication = malloc(adaptationIndicationLen);
        memcpy(asok->adaptationLayerIndication, adaptationIndication, adaptationIndicationLen);
    } else {
        asok->adaptationLayerIndicationLen = 0;
        asok->adaptationLayerIndication = NULL;
    }

    event_logii(INTERNAL_EVENT_1, "association initialisation (2) performed ID=%08x, local tag=%08x",
               asok->assocId, asok->tagLocal);

    return 0;

}                               /* end: mdi_initAssociation */


int mdi_restartAssociation(Association* asok,
                           unsigned short noOfInStreams,
                           unsigned short noOfOutStreams,
                           unsigned int new_rwnd,
                           unsigned int remoteInitialTSN,
                           unsigned int localInitialTSN,
                           unsigned int noOfDestinationAddresses,
                           unsigned int primaryAddress,
                           union sockunion *destinationAddressList,
                           gboolean peerSupportsPRSCTP, gboolean peerSupportsADDIP,
                           int adaptationIndicationLen, void* adaptationIndication)
{
    int result;

    if (!asok) {
        error_log(ERROR_MAJOR, "mdi_restartAssociation: association is NULL !");
        return 1;
    }
    if (!asok->sctpInstance) {
        error_log(ERROR_MAJOR, "mdi_restartAssociation: sctpInstance is NULL !");
        return 1;
    }

    if (noOfDestinationAddresses > asok->noOfDestAddresses) {
        error_log(ERROR_MAJOR, "mdi_restartAssociation tries to increase number of paths !");
        /**
         * TODO - TODO - FIXME:
         * check if new addresses have been added ! If so, decline the RESTART !
         * discard silently
         */
        /* we just need to compare destinationAddressList and asok->destAddresses */
        return -1;
    }

    event_logiiii(EXTERNAL_EVENT, "ASSOCIATION RESTART: in streams: %u, out streams: %u, rwnd: %u, paths: %u",
                                   noOfInStreams,noOfOutStreams,new_rwnd,noOfDestinationAddresses);
    event_logii(EXTERNAL_EVENT,   "ASSOCIATION RESTART: remote initial TSN:  %u, local initial TSN",
                                   remoteInitialTSN, localInitialTSN);

    asok->reliableTransfer = rtx_restart_reliable_transfer(asok,
                                                           noOfDestinationAddresses,
                                                           localInitialTSN);

    fc_restart(asok, new_rwnd, localInitialTSN, asok->maxSendQueue);

    rxc_restart_receivecontrol(asok, mdi_getDefaultMyRwnd(asok->sctpInstance), remoteInitialTSN);

    se_delete_stream_engine(asok);
    asok->streamengine     = (void *) se_new_stream_engine(asok, noOfInStreams, noOfOutStreams,
                                                           asok->supportsPRSCTP && asok->peerSupportsPRSCTP);

    asc_delete(asok);
    asok->sctp_asconf = (void *) asc_new(asok, localInitialTSN, remoteInitialTSN);

    pm_deletePathman(asok);
    asok->pathMan = NULL;
    /* frees old address-list before assigning new one */
    mdi_writeDestinationAddresses(asok, destinationAddressList, noOfDestinationAddresses);

    asok->pathMan = pm_newPathman(asok, noOfDestinationAddresses, primaryAddress);

    if (!asok->pathMan) {
        error_log(ERROR_FATAL, "Error 1 in RESTART --> Fix implementation");
        return -1;
    }

    event_logii(VERBOSE, "ASSOCIATION RESTART: calling pm_setPaths(%u, %u)", noOfDestinationAddresses,primaryAddress);
    result = pm_setPaths(asok, noOfDestinationAddresses, primaryAddress);

    if (result != 0) {
        error_log(ERROR_FATAL, "Error 2 in RESTART --> Fix implementation");
        return -1;
    }
    if (asok->adaptationLayerIndication != NULL) free(asok->adaptationLayerIndication);
    asok->adaptationLayerIndicationLen = adaptationIndicationLen;
    if (adaptationIndicationLen > 0) {
        asok->adaptationLayerIndication = malloc(adaptationIndicationLen);
        memcpy(asok->adaptationLayerIndication, adaptationIndication, adaptationIndicationLen);
    } else {
        asok->adaptationLayerIndicationLen = 0;
        asok->adaptationLayerIndication = NULL;
    }


    return 0;
}


/**
 *  mdi_deleteAssociation deletes the current association.
 *
 *  The association will not be deleted at once, but is only marked for deletion. This is done in
 *  this way to allow other modules to finish their current activities. To prevent them to start
 *  new activities, the currentAssociation pointer is set to NULL.
 */
void mdi_deleteAssociation(Association* currentAssociation)
{
    /* unsigned int pathID; */

    if (currentAssociation != NULL) {
        if (currentAssociation->tagRemote != 0) {
            /* stop timers */
            pm_disableAllHB(currentAssociation);
            fc_stop_timers(currentAssociation);
            rxc_stop_sack_timer(currentAssociation);
            asc_stop_timer(currentAssociation);
        }
        /* mark association as deleted, it will be deleted when retrieveAssociation(..) encounters
           a "deleted" association. */
        currentAssociation->deleted = TRUE;
        event_logi(INTERNAL_EVENT_1, "association ID=%08x marked for deletion", currentAssociation->assocId);
    } else {
        error_log(ERROR_MAJOR,
                  "mdi_deleteAssociation: current association does not exist, can not delete");
    }
}


unsigned int mdi_getCurrentMinAdapterMTU()
{
    return min_MTU;
}
