/***************************************************************************
 *   copyright           : (C) 2004 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "pbookphone.h"
#include "common.h"
#include "helper.h"
#include "atcommand.h"

#include <ctype.h>
#include <string.h>
#include <strings.h>

void struct_pbook_phone_entry_init (struct pbook_phone_entry* p) {
  p->slot = 0;
  p->number = NULL;
  p->numtype = numtype(NULL);
  p->text = NULL;
}
void struct_pbook_phone_entry_delete (struct pbook_phone_entry* p) {
  mem_realloc(p->number,0);
  mem_realloc(p->text,0);
}

int pbook_phone_write_entry (struct pbook_phone_entry* entry) {
  char* temp;
  int retval = 1;

  at_gen_phonebook_write(entry->slot,entry->number,
			 entry->numtype,entry->text);
  temp = at_read_line();
  if (at_line_type(temp,NULL,NULL) != AT_RET_OK) retval = 0;
  mem_realloc(temp,0);
  return retval;
}

int pbook_phone_write (char* mem, struct pbook_phone_entry** entries) {
  unsigned int min,max;
  unsigned int nlen,tlen;
  unsigned int i = 0;
  unsigned int count = 0;

  if (!pbook_get_ranges(mem,1,&min,&max,&nlen,&tlen)) {
    errexit("Error on getting limits of phonebook %s.\n",mem);
  }

  for (i = 0; entries[i] != NULL; ++i) {
    if (min > entries[i]->slot || entries[i]->slot > max) {
      errexit ("Error: slot %d is out of range %d-%d.\n",entries[i]->slot,min,max);
    }
  }
  count = i;
  if (count > max-min+1) {
    errexit("Error: too many entries (max. %d).\n",max-min+1);
  }

  if (count > 1) {
    myprintf(0,"Updating entries %s(%d-%d)...\n",mem,min,max);
  } else {
    myprintf(0,"Updating one entry in slot %d of memory %s...",entries[0]->slot,mem);    
  }
  for (i = 0; entries[i] != NULL; ++i) {
    if (pbook_phone_write_entry(entries[i]) == 0) return 0;
    if (count > 1) myprintf(0," %d",entries[i]->slot);
  }
  if (count > 1) myprintf(0,"%s","\n");
  myprintf(0,"%s","done\n");
  return 1;
}

/* fill the pbook_phone_entry from a text line
 */
int pbook_phone_str2entry (char* line,
			   struct pbook_phone_entry* entry)
{
  //pointers in line
  char* slot;
  char* number;
  char* numtype;
  char* text;

  if (line == NULL || entry == NULL) return -1;

  /* Policy is to not trust in what the phone delivers
   * Thus, full checking must be done.
   * Format: slot,"number",numtype,"text"
   */
  slot = line;
  while(isdigit((int)*line) && *line != 0) ++line;
  if (*line != ',' || *line == 0) return 0;
  ++line;
  if (*line != '"' || *line == 0) return 0;
  number = ++line;
  while (*line != '"' && *line != 0) ++line;
  if (*line != '"' || *line == 0) return 0;
  ++line;
  if (*line != ',' || *line == 0) return 0;
  numtype = ++line;
  while(isdigit((int)*line) && *line != 0) ++line;
  if (*line != ',' || *line == 0) return 0;
  ++line;
  if (*line != '"' || *line == 0) return 0;
  text = ++line;
  if ((line=rindex(text-1,'"')) <= text-1) return 0;
  ++line;
  if (*line != 0) return 0;

  entry->slot = atoi(slot);
  entry->number = strn_dup(number,index(number,'"')-number);
  entry->numtype = atoi(numtype);
  entry->text = strn_dup(text,rindex(text,'"')-text);
  return 1;
}

/* slot_max-slot_min can be used to loop over a limited range
 * The returned pointer is malloc'd, NULL-terminated list,
 * each list enty is malloc'd, too.
 */
#include <unistd.h>
struct pbook_phone_entry**
pbook_phone_get_range (char* mem,
		       unsigned int slot_min,
		       unsigned int slot_max,
		       int print_counter)
{
  unsigned int i = 0;
  unsigned range = slot_max-slot_min+1;
  struct pbook_phone_entry** retval;
  struct pbook_phone_entry* buffer;
  struct pbook_phone_entry** vcfbuf;
  char* command;
  char* ausgabe;
  char* temp;
  enum return_code linestatus;
  int status;
  int enable_slot_range_test = 1;
  unsigned int j,k = 0;

  if (strcasecmp(mem,VCARD_PHONEBOOK_NAME) == 0) {
    if (slot_min != slot_max) {
      retval = mem_alloc(((range*4)+1)*sizeof(*retval),0);
      for (i = slot_min; i <= slot_max; ++i) {
	//call this function recursively (one level)
	vcfbuf = pbook_phone_get_range(mem,i,i,print_counter);
	for (j = 0; vcfbuf[j] != NULL; ++j) {
	  retval[k++] = vcfbuf[j];
	}
	vcfbuf = mem_realloc(vcfbuf,0);
      }
      retval[k] = NULL;
      return retval;
    } else {
      at_sie_phonebook_vcf_read(slot_min, -1);
      command = AT_SIE_PB_VCF_READ;
      enable_slot_range_test = 0;
      range = 4;
    }
  } else if (strcasecmp(mem,"RD") == 0 ||
	     strcasecmp(mem,"CS") == 0) {
    /* Get sorted pb */
    at_sie_phonebook_read_sorted(slot_min,slot_max);
    command = AT_SIE_PB_READ_SORTED;
  } else {
    /* Get pb */
    at_gen_phonebook_read(slot_min,slot_max);
    command = AT_GEN_PB_READ;
  }

  ausgabe = at_read_line();
  linestatus = at_line_type(ausgabe,command,&temp);
  if (linestatus&AT_RET_ERROR) errexit("Error on reading from phonebook %s: %s\n",mem,temp);

  retval = mem_alloc((range+1)*sizeof(*retval),0);
  while ((linestatus == AT_RET_ANSWER ||
	  linestatus == AT_RET_OTHER) &&
	 i <= i+range-1) {
    buffer = mem_alloc(sizeof(**retval),0);
    struct_pbook_phone_entry_init(buffer);
    status = pbook_phone_str2entry(temp,buffer);
    if (status <= 0) errexit("Error in parsing entry: %s\n", temp);

    if (enable_slot_range_test) {
      if (buffer->slot > slot_max) errexit("Error: phone returned unexpected entry %u\n",buffer->slot);
      for (; i+slot_min < buffer->slot; ++i) {
	retval[i] = mem_alloc(sizeof(**retval),0);
	struct_pbook_phone_entry_init(retval[i]);
	retval[i]->slot = i+slot_min;
	if (VERBOSE_LEVEL < 2 && print_counter) myprintf(0," %d",retval[i]->slot);
      }
    }
    retval[i] = buffer;
    if (strcasecmp(mem,VCARD_PHONEBOOK_NAME) == 0) retval[i]->slot = slot_min;
    if (VERBOSE_LEVEL < 2 && print_counter) myprintf(0," %d",retval[i]->slot);

    mem_realloc(ausgabe,0);
    ausgabe = at_read_line();
    linestatus = at_line_type(ausgabe,command,&temp);
    if (linestatus&AT_RET_ERROR) errexit("Error on reading from phonebook %s: %s\n",mem,temp);
    ++i;
  }
  if (enable_slot_range_test) {
    for (; i+slot_min <= slot_max; ++i) {
      retval[i] = mem_alloc(sizeof(**retval),0);
      struct_pbook_phone_entry_init(retval[i]);
      retval[i]->slot = i+slot_min;
      if (VERBOSE_LEVEL < 2 && print_counter) myprintf(0," %d",retval[i]->slot);
    }
  }
  retval[i] = NULL; //termination of list
  return retval;
}

// number of requesting numbers
// higher means a little bit more speed
// lower helps slow phones/SIM-cards to not time out
#define delta 5

struct pbook_phone_entry**
pbook_phone_get (char* mem) {
  unsigned cur_min, cur_max;
  int min = -1, max = -1;
  struct pbook_phone_entry** retval;
  struct pbook_phone_entry** temp;
  unsigned int i; //index in temp
  unsigned int k = 0; //index in retval

  //reading available indexes
  if (!pbook_get_ranges(mem,0,&min,&max,NULL,NULL)) {
    errexit("Error on getting limits of phonebook %s.\n",mem);
  }
  if (min == 0 && max == 0) {
    //emty phonebook
    return NULL;
  }
  myprintf(0,"Receiving phonebook entries %s(%d-%d)...\n",
	   mem,min,max);
  //4 times as big due to phonebook vcf
  retval = mem_alloc(((4*(max-min+1))+1)*sizeof(*retval),0);
  retval[0] = NULL;
  for (cur_min = min; cur_min <= (unsigned int)max; cur_min += delta) {
    cur_max = cur_min+delta-1;
    if (cur_max > (unsigned int)max) cur_max = max;
    temp = pbook_phone_get_range(mem,cur_min,cur_max,1);
    i = 0;
    while (temp[i] != NULL) retval[k++] = temp[i++];
    retval[k] = NULL;    
  }
  myprintf(0,"%s","\ndone\n");
  return retval;
}
