/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
	 
#include "common.h"
#include "helper.h"
#include "memtypes.h"
#include "atcommand.h"

#include <stdlib.h>
#include <string.h>
#include <strings.h>

inline char* get_vendor (void) {
  at_command_send(AT_GEN_VENDOR,NULL);
  return at_get_value(AT_GEN_VENDOR);
}

inline char* get_model (void) {
  at_command_send(AT_GEN_MODEL,NULL);
  return at_get_value(AT_GEN_MODEL);
}

inline char* get_revision (void) {
  at_command_send(AT_GEN_REVISION,NULL);
  return at_get_value(AT_GEN_REVISION);
}

inline char* get_phoneserial (void) {
  at_command_send(AT_GEN_IMSI,NULL);
  return at_get_value(AT_GEN_IMSI);
}

inline char* get_simserial (void) {
  at_command_send(AT_GEN_IMEI,NULL);
  return at_get_value(AT_GEN_IMEI);
}

int pbook_select_mem (char* mem) {
  char* command;
  char* temp;
  enum return_code status;

  if (strcasecmp(mem,VCARD_PHONEBOOK_NAME) == 0) return 1;
  temp = get_vendor();
  if (temp != NULL && strcmp(temp,"SIEMENS") == 0) {
    at_sie_phonebook_select(mem);
    command = AT_SIE_PB_SELECT;
  } else {
    at_gen_phonebook_select(mem);
    command = AT_GEN_PB_SELECT;
  }
  mem_realloc(temp,0);
  temp = at_read_line();
  status = at_line_type(temp,NULL,NULL);
  mem_realloc(temp,0);
  if (status == AT_RET_OK) return 1;
  else return 0;
}

int sms_select_mem (char* mem, struct slot_range* r,
		    int* current_fill)
{
  char* temp;
  char* ausgabe;
  int retval = 0;

  if (str_len(mem) == 0) {
    return 0;
  }

  at_gen_sms_mem(mem,mem,NULL);
  ausgabe = at_get_value(AT_GEN_SMS_MEM);
  //example: +CPMS: 10,112,10,112,8,12
  if (str_len(ausgabe) != 0) {
    if (r != NULL) {
      strncpy(r->name,mem,sizeof(r->name));
      temp = index(ausgabe,',');
      if (str_len(temp)) {
	r->min = 1;
	r->max = atoi(temp+1);
      }
    }
    if (current_fill != NULL && str_len(ausgabe)) {
      *current_fill = atoi(ausgabe);
    }
    retval = 1;
  }
  mem_realloc(ausgabe,0);
  return retval;
}

#include "ttyaccess.h"
int phone_init (void) {
  int retval = 1;
  char* ack;

  tty_write("\r",1);
  at_command_send(AT_GEN_INIT,NULL);
  ack = at_read_line();
  if (at_line_type(ack,NULL,NULL) != AT_RET_OK) {
    mem_realloc(ack,0);
    ack = at_read_line();
    if (at_line_type(ack,NULL,NULL) != AT_RET_OK) {
      retval = 0;
    }
    mem_realloc(ack,0);
  }
  return retval;;
}

int command_echo (int enable) {
  char* command;
  char* ack;
  enum return_code status;
  if (enable) {
    command = AT_GEN_ECHO_ENABLE;
  } else {
    command = AT_GEN_ECHO_DISABLE;
  }
  at_command_send(command,NULL);
  ack = at_read_line();
  status = at_line_type(ack,NULL,NULL);
  mem_realloc(ack,0);
  return (status == AT_RET_OK);
}

int verbose_errors (int enable) {
  char* parmlist;
  char* ack;
  enum return_code status;
  if (enable) {
    parmlist = "2";
  } else {
    parmlist = "0";
  }
  at_command_send(AT_GEN_ERROR,parmlist);
  ack = at_read_line();
  status = at_line_type(ack,NULL,NULL);
  mem_realloc(ack,0);
  return (status == AT_RET_OK);
}

char* get_charset (void) {
  char* ausgabe;
  char* temp;
    
  at_syntax_request(AT_GEN_CHARSET);
  ausgabe = at_get_value(AT_GEN_CHARSET);
  if (ausgabe != NULL) {
    if (*ausgabe == '"' && 
	(temp = index(ausgabe,'"')) != NULL) {
      temp = strn_dup(ausgabe+1,temp-ausgabe-1);
      mem_realloc(ausgabe,0);
      return temp;
    }
  }
  return NULL;
}

char** get_charsets (void) {
  char* ausgabe;
  char** retval;

  at_command_send(AT_GEN_CHARSET,"?");
  ausgabe = at_get_value(AT_GEN_CHARSET);
  //example: +CSCS: ("GSM","UCS2")
  if (str_len(ausgabe) != 0) {
    retval = at_parse_stringlist(ausgabe);
  } else {
    retval = NULL;
  }
  mem_realloc(ausgabe,0);
  return retval;
}

int set_charset(char *charset) {
  char *temp;
  int  retval = 0;

  if (str_len(charset) == 0) {
    errexit ("%s","Error: no charset specified.\n");
  }
  at_gen_charset(charset);
  temp = at_read_line();
  if (at_line_type(temp,NULL,NULL) == AT_RET_OK) {
    retval = 1;
  }
  mem_realloc(temp,0);
  return retval;
}

void set_smsc (char* number) {
  char* temp;
  char* error;
  enum return_code status;

  if (str_len(number) == 0) {
    errexit("%s","Error: no number specified.\n");
  } else if (is_telephone_number(number) == 0) {
    errexit("%s","Error: no valid number specified.\n");
  }
  at_gen_smsc(number);
  temp = at_read_line();
  status = at_line_type(temp,AT_GEN_SMSC,&error);
  if (status == AT_RET_OK) {
    myprintf(0,"SMS server number was set to %s\n",number);
  } else if (status == AT_RET_ERROR) {
    errexit("%s","Error: unknown cause\n");
  } else {
    errexit("Error: %s\n",error);
  }
  mem_realloc(temp,0);
}

char* get_smsc (void) {
  char* ausgabe;
  char* temp;
    
  at_syntax_request(AT_GEN_SMSC);
  ausgabe = at_get_value(AT_GEN_SMSC);
  if (ausgabe != NULL && *ausgabe == '"') {
    if ((temp=strstr(ausgabe,"\",")) != NULL) {
      temp = strn_dup(ausgabe+1,temp-ausgabe-1);
      mem_realloc(ausgabe,0);
      return temp;
    }
  }
  return NULL;
}

#include "timeincl.h"

void set_time (void) {
  char* ack;
  time_t seconds;
  char timestr[20]; //6*2 + 7 + 1
	
  memset(timestr,0,sizeof(timestr));
	
  time(&seconds); //seconds since Unix
  //string formatting with conversion to local time
  strftime(timestr,sizeof(timestr),"\"%y/%m/%d,%H:%M:%S\"",localtime(&seconds));
  at_command_send(AT_GEN_TIME,timestr);
  ack = at_read_line();
  if (at_line_type(ack,NULL,NULL) == AT_RET_OK) {
    myprintf(0,"%s","Time was synchronized\n");
  } else {
    errexit("%s","Time could not be synchronized\n");
  }
  mem_realloc(ack,0);
}

struct tm* get_time (void) {
  char* ausgabe;
  struct tm* phonetime;
  char* status;
  
  at_syntax_request(AT_GEN_TIME);
  ausgabe = at_get_value(AT_GEN_TIME);
  if (ausgabe != NULL) {
    //example: +CCLK: "04/06/10,02:30:37"
    phonetime = mem_alloc(sizeof(*phonetime),1);
    status = strptime(ausgabe,"\"%y/%m/%d,%H:%M:%S\"",phonetime);
    mem_realloc(ausgabe,0);
    if (status != NULL) {
      return phonetime;
    }
    mem_realloc(phonetime,0);
  }
  return NULL;
}

char* get_simid (void) {
  char* ausgabe;

  ausgabe = get_vendor();
  if(ausgabe != NULL &&
     strcasecmp(ausgabe,"SIEMENS") == 0) {
    at_command_send(AT_SIE_CARD_ID,NULL);
    return at_get_value(AT_SIE_CARD_ID);
  }
  return NULL;
}

char* get_operator (void) {
  char* ausgabe;
  char* temp;

  at_syntax_request(AT_GEN_OPERATOR);
  ausgabe = at_get_value(AT_GEN_OPERATOR);
  if (ausgabe != NULL) {
    //example: +COPS: 0,0,"E-Plus"
    if ((temp=index(ausgabe,'"')) != NULL) {
      temp = strn_dup(temp+1,strlen(temp+1)-1);
      mem_realloc(ausgabe,0);
      return temp;
    }
    mem_realloc(ausgabe,0);
  }
  return NULL;
}

char* get_battery (void) {
  char* ausgabe;
  char* temp;
    
  at_command_send(AT_GEN_BATTERY,NULL);
  ausgabe = at_get_value(AT_GEN_BATTERY);
  if (ausgabe != NULL) {
    //example: +CBC: 0,80
    if ((temp=index(ausgabe+6,',')) != NULL) {
      temp=strn_dup(temp+1,strlen(temp+1));
      mem_realloc(ausgabe,0);
      return temp;
    }
    mem_realloc(ausgabe,0);
  }
  return NULL;
}

int get_signal (int* ber_p) {
  char* ausgabe;
  char* temp;
  int signal = -1;
  int b;
  

  if (ber_p != NULL) {
    *ber_p = -1;
  }    
  at_command_send(AT_GEN_SIGNAL,NULL);
  ausgabe = at_get_value(AT_GEN_SIGNAL);
  if (ausgabe != NULL) {
    //example: +CSQ: 28,99
    signal = atoi(ausgabe+6);
    if (signal == 99) {
      signal = -1;
    } else {
      if ((signal=111-((signal-1)*2)) < 0) {
	signal = -1;
      }
    }
    if (ber_p != NULL) {
      if ((temp=index(ausgabe,',')) != NULL) {
	b = atoi(temp+1);
	if (b == 7) {
	  *ber_p = 255;
	} else if (b >= 0 && b < 7) {
	  *ber_p = 2<<b;
	}
      }
    }
    mem_realloc(ausgabe,0);
  }
  return signal;
}

char* get_netstatus (unsigned int* areacode_p, unsigned int* cellid_p) {
  char* ausgabe;
  char* index1;
  char* index2;
  char buffer;

  if (areacode_p != NULL) *areacode_p=0;
  if (cellid_p != NULL) *cellid_p=0;

  at_command_send(AT_GEN_CSD_STATUS,"?");
  ausgabe = at_get_value(AT_GEN_CSD_STATUS);
  //example: +CREG: (0-2)
  if (ausgabe != NULL && index(ausgabe,'2') == NULL) {
    mem_realloc(ausgabe,0);
    at_syntax_request(AT_GEN_CSD_STATUS);
    ausgabe = at_get_value(AT_GEN_CSD_STATUS);
  } else {
    mem_realloc(ausgabe,0);
    /* Fix: firmware may forget to send LAC/IC (e.g. ME45 v23,S55)
     * Original was provided by Dominik Neubauer
     */
    at_command_send(AT_GEN_CSD_STATUS,"2");
    ausgabe = at_read_line();
    if (at_line_type(ausgabe,NULL,NULL) == AT_RET_OK) {
      mem_realloc(ausgabe,0);
      at_syntax_request(AT_GEN_CSD_STATUS);
      ausgabe = at_get_value(AT_GEN_CSD_STATUS);
      if (ausgabe != NULL) {
	at_command_send(AT_GEN_CSD_STATUS,"0");
	mem_realloc(at_read_line(),0);
      }
    } else {
      ausgabe = mem_realloc(ausgabe,0);
    }
  }
  //example: +CREG: 2,1,"0044","6E30"
  if (ausgabe != NULL) {
    index1 = index(ausgabe,'"');
    if (index1 != NULL) {
      index2 = index(++index1,'"');
      if (index2 != NULL) {
	if (areacode_p != NULL) {
	  *areacode_p = hexstr2int(index1,index2-index1);
	}
	index1 = index(index2+1,'"');
	if (index1 != NULL) {
	  index2 = index(++index1,'"');
	  if (index2 != NULL && cellid_p != NULL) {
	    *cellid_p=hexstr2int(index1,index2-index1);
	  }
	}
      }
    }
    if ((index1=index(ausgabe,',')) != NULL) {
      buffer = index1[1];
      mem_realloc(ausgabe,0);
      switch (buffer) {
      case '0':	return "not checked in, not seeking";
      case '1':	return "checked in";
      case '2':	return "not checked in, but seeking a network";
      case '3':	return "check-in denied by network";
      case '5':	return "registered, roaming";
      default:	return "unknown status";
      }
    }
  }
  return NULL;
}

char* get_gprs_status (unsigned int* areacode_p, unsigned int* cellid_p) {
  char* ausgabe;
  char* index1;
  char* index2;
  char buffer;

  if (areacode_p != NULL) *areacode_p=0;
  if (cellid_p != NULL) *cellid_p=0;

  at_syntax_request(AT_GEN_GPRS_STATUS);
  ausgabe = at_get_value(AT_GEN_GPRS_STATUS);
  if (ausgabe != NULL) {
    index1 = index(ausgabe,'"');
    if (index1 != NULL) {
      index2 = index(++index1,'"');
      if (index2!=NULL) {
	if (areacode_p!=NULL) {
	  *areacode_p = hexstr2int(index1,index2-index1);
	}
	index1 = index(index2+1,'"');
	if (index1 != NULL) {
	  index2 = index(++index1,'"');
	  if (index2 != NULL && cellid_p != NULL) {
	    *cellid_p = hexstr2int(index1,index2-index1);
	  }
	}
      }
    }
    if ((index1=index(ausgabe,',')) != NULL) {
      buffer = index1[1];
      mem_realloc(ausgabe,0);
      switch (buffer) {
      case '0':	return "not registered, not searching";
      case '1':	return "registered, home network";
      case '2':	return "not registered, searching";
      case '3':	return "registration denied by network";
      case '5':	return "registered, roaming";
      default:	return "unknown status";
      }
    }
  }
  return NULL;
}

int get_gprs_attach (void) {
  char* ausgabe;
  char buffer;

  at_syntax_request(AT_GEN_GPRS_ATTACH);
  ausgabe = at_get_value(AT_GEN_GPRS_ATTACH);
  if (ausgabe != NULL) {
    buffer = ausgabe[0];
    mem_realloc(ausgabe,0);
    switch (buffer) {
    case '0': return 0;
    case '1': return 1;
    }
  }
  return -1;
}

char* get_gprs_class (void) {
  char* ausgabe;
  char* temp;

  at_syntax_request(AT_GEN_GPRS_CLASS);
  ausgabe = at_get_value(AT_GEN_GPRS_CLASS);
  if (ausgabe != NULL) {
    //example: +CGCLASS: "B"
    if (ausgabe[0] == '"') {
      memmove(ausgabe,ausgabe+1,strlen(ausgabe)+1);
    }
    if ((temp=index(ausgabe,'"')) != NULL) {
      *temp = 0;
    }
    return ausgabe;
  }
  return NULL;
}

void info_misc (FILE* fd) {
  char *temp;
  int i,j;
  struct tm* ltime;
  char** memlist;

  fprintf(fd,"Phone related information:\n");
  fprintf(fd,"Vendor:\t\t%s\n",temp=get_vendor());
  mem_realloc(temp,0);
  fprintf(fd,"Model:\t\t%s\n",temp=get_model());
  mem_realloc(temp,0);
  fprintf(fd,"Revision:\t%s\n",temp=get_revision());
  mem_realloc(temp,0);
  fprintf(fd,"IMEI:\t\t%s\n",temp=get_phoneserial());
  mem_realloc(temp,0);
  fprintf(fd,"Battery:\t%s%%\n",temp=get_battery());
  mem_realloc(temp,0);
  memlist=get_charsets();
  fprintf(fd,"Charsets:\t");
  if (memlist == NULL) {
    fprintf(fd,"none");
  } else {
    for (i=0;memlist[i]!=NULL;++i) {
      if (i != 0) {
	fprintf(fd,", ");
      }
      fprintf(fd,"%s",memlist[i]);
      mem_realloc(memlist[i],0);
    }
  }
  fprintf(fd,"\n");
  mem_realloc(memlist,0);
  temp=mem_alloc(81,1);
  ltime=get_time();
  if (ltime!=NULL && strftime(temp,80,"%c",ltime)) {
    fprintf(fd,"Time:\t\t%s\n",temp);
    mem_realloc(ltime,0);
  }
  mem_realloc(temp,0);

  fprintf(fd,"\nSIM card related information:\n");
  fprintf(fd,"IMSI:\t\t%s\n",temp=get_simserial());
  mem_realloc(temp,0);
  if ((temp=get_simid()) != NULL) {
    fprintf(fd,"card ID:\t%s\n",temp);
  }
  mem_realloc(temp,0);

  fprintf(fd,"\nNetwork related information:\n");
  temp=get_netstatus(&i,&j);
  if (temp!=NULL) {
    fprintf(fd,"Status:\t\t%s\n",temp);
    if (i) {
      fprintf(fd,"Area code:\t%04X\n",i);
      if (j) {
	fprintf(fd,"Cell ID:\t%04X\n",j);
      }
    }
  }
  temp=get_operator();
  if (temp!=NULL) {
    fprintf(fd,"Operator:\t%s\n",temp);
  }
  mem_realloc(temp,0);
  temp=get_smsc();
  if (temp!=NULL) {
    fprintf(fd,"SMS Server:\t%s\n",temp);
  }
  mem_realloc(temp,0);
  if ((i=get_signal(&j)) >= 0) {
    fprintf(fd,"Signal:\t\t%d dBm\n",-i);
    if (j >= 0) {
      if (j <= 128) {
	fprintf(fd,"BER:\t\t<=%d,%d%%\n",j/10,j%10);
      } else {
	fprintf(fd,"BER:\t\t>12,8%%\n");
      }
    }
  }
  temp=get_gprs_class();
  if (temp!=NULL) {
    fprintf(fd,"GPRS class:\t");
    temp=(char *)strtok(temp,"()\",");
    while (temp!=0) {
      fprintf(fd,"%s",temp);
      temp=(char *)strtok(NULL,"()\",");
      if (temp!=0) {
	fprintf(fd,", ");
      }
    }
    fprintf(fd,"\n");
  }
  mem_realloc(temp,0);
  temp=get_gprs_status(NULL,NULL);
  if (temp!=NULL) {
    fprintf(fd,"GRPS status:\t%s",temp);
    i=get_gprs_attach();
    if (i==0) {
      fprintf(fd,", detached\n");
    } else if (i==1) {
      fprintf(fd,", attached\n");
    }
  }

  fprintf(fd,"\nAvailable memories:\n");
}

void info_mem (FILE* fd, int full) {
  fprintf(fd,"Binary files:\t");
  file_print_memlist(fd,full);
  fprintf(fd,"Phonebooks:\t");
  pbook_print_memlist(fd,full);
  fprintf(fd,"SMS storages:\t");
  sms_print_memlist(fd,full);
}

void info (char* file, int misc, int mem){
  if (misc || mem) {
    FILE* fd;
    if (!str_len(file) || !strcmp(file,"-")) {
      fd=stdout;
    } else {
      fd=fdopen(open_myFile_rw(file),"w+");
    }
    if (misc) {
      info_misc(fd);
    }
    info_mem(fd,mem);
    if (str_len(file) && strcmp(file,"-")) {
      fclose(fd);
    }
  }
}

