/***************************************************************************
 *   copyright           : (C) 2002 by Hendrik Sattler                     *
 *   mail                : post@hendrik-sattler.de                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "memtypes.h"
#include "common.h"
#include "helpers.h"
#include "options.h"
#include <errno.h>
#include <stdlib.h>

#include <string.h>
//needed by at least Solaris8
#include <strings.h>

void sms_manage (int action, char** files,
		 char* outfile, char* pipe,
		 struct parameters* myparams,
		 struct smsopts* mysmsopts)
{
  char* storetype;
  char** memlist;
  int i = 0;
  struct slot_range range;
  int current_fill;
  struct sms_data** slist;
  char* pdu;
  int mark_read = 1; //getting SMS with changing status from "unread" to "read"
    
  memlist = NULL;
  slist = NULL;

  if (str_len(myparams->mem) != 0 &&
      strcmp(myparams->mem,"?") == 0) {
    sms_print_memlist(NULL,1);
    exit(0);
  }

  if (action == 0) {
    errexit ("You must specify a valid action.\n");
  }

  if (myparams->slot == SCMXX_SLOT_UNDEFINED) {
    switch (action) {
    case SCMXX_ACTION_GET:
      myparams->slot = SCMXX_SLOT_SMS_UNREAD;
      break;
    case SCMXX_ACTION_SEND:
      break;
    default:
      errexit("If you want to remove short messages or mix actions, you must define a slot or slot type.\n");
      break;
    }
  }

  if (str_len(myparams->mem)) {
    storetype = myparams->mem;
  } else {
    memlist = sms_get_types();
    if (memlist[i] == NULL) {
      mem_realloc(memlist,0);
      return;
    } else {
      do {
	if (!strcasecmp(memlist[i],"MT")) {
	  break;
	}
      } while (memlist[++i] != NULL);
      if (memlist[i] == NULL) { // MT not present
	i=0;
	do {
	  if (!strcasecmp(memlist[i],"SM")) {
	    break;
	  }
	} while (memlist[++i] != NULL);
	if (memlist[i] == NULL) { // SM not present (should never happen)
	  i=0; //choosing first one in list
	}
      }
    }
    storetype=memlist[i];
  }
  if (!sms_select_mem(storetype,&range,&current_fill)) {
    errexit("SMS storage type \"%s\" could not be selected.\n",storetype);
  }
  if (memlist != NULL) {
    i=0;
    while (memlist[i] != NULL) {
      mem_realloc(memlist[i++],0);
    }
    mem_realloc(memlist,0);
  }

  if (str_len(outfile) == 0 && str_len(pipe) == 0) {
    mark_read = 0;
  }
    
  if (action&SCMXX_ACTION_GET) {
    if (mysmsopts->direct) {
      //getting messages as unexpected notice from the phone
      sms_get_direct(outfile,pipe);
    } else {
      do {
	if (i > current_fill) {
	  current_fill = i;
	}
	slist = mem_alloc((current_fill+1)*sizeof(*slist),0);
	i = sms_get(slist,current_fill+1,myparams->slot,mark_read);
      } while (i > current_fill);
      sms_print_all(outfile,pipe,slist);
    }
  }
  if (action&SCMXX_ACTION_SEND) {
    if (myparams->slot != SCMXX_SLOT_UNDEFINED &&
	(myparams->slot < SCMXX_SLOT_SMS_MIN ||
	 (myparams->slot >= SCMXX_SLOT_SMS_MIN && 
	  str_len(myparams->number) == 0))) {
      if (slist == NULL) {
	do {
	  if (i > current_fill) {
	    current_fill = i;
	  }
	  slist = mem_alloc((current_fill+1)*sizeof(*slist),0);
	  i = sms_get(slist,current_fill+1,myparams->slot,mark_read);
	} while (i > current_fill);
      }
      if (slist != NULL) {
	/* These mails come from SMS storage and thus are sent, not stored.
	 * This was also the behaviour in previous versions.
	 */
	sms_send_pdu_all(slist,myparams->number,1);
      }
    } else {
      if (myparams->slot >= SCMXX_SLOT_SMS_MIN) {
	sms_send_slot(myparams->slot,myparams->number);
      } else {
	if (myparams->text == NULL &&
	    (files == NULL || str_len(files[0]) == 0)) {
	  myprintf(0,"Hmm, nothing to send?\n");
	}
	if (myparams->text != NULL) {
	  pdu = sms_pdu_create(NULL,myparams->text,myparams->number,mysmsopts);
	  sms_send_pdu(pdu,mysmsopts->direct);
	  //so we do not submit the same sms multiple times
	  myparams->text = NULL;
	}
	if (files != NULL) {
	  i=0;
	  while (files[i] != NULL) {
	    pdu = sms_pdu_create(files[i++],NULL,myparams->number,mysmsopts);
	    sms_send_pdu(pdu,mysmsopts->direct);	
	  }
	}
      }
    }
  }
  if (action&SCMXX_ACTION_REMOVE) {
    if (slist == NULL &&
	myparams->slot < SCMXX_SLOT_SMS_MIN) {
      do {
	if (i > current_fill) {
	  current_fill = i;
	}
	slist = mem_alloc((current_fill+1)*sizeof(*slist),0);
	i = sms_get(slist,current_fill+1,myparams->slot,mark_read);
      } while (i > current_fill);
    }
    if (slist != NULL) {
      sms_delete_all(slist);
    } else {
      sms_delete_slot(myparams->slot);
    }
  }
}

void sms_delete_slot (int smsslot) {
  char at_command[128];
  char* ack;

  memset(at_command,0,sizeof(at_command));

  if (smsslot >= SCMXX_SLOT_SMS_MIN) {
    new_at_command(at_command,"+CMGD");
    add_at_command(at_command,"=%d",smsslot);
    ack=tty_write_read(at_command);
    if (!strcmp(ack,"OK")) {
      myprintf(0,"SMS slot %d was deleted.\n",smsslot);
    } else if(!strcmp(ack,"+CMS ERROR: INVALID MEMORY INDEX")) {
      errexit("This index is not available.\n");
    } else if(strstr(ack,"ERROR")!=NULL) {
      errexit("%s, aborting.\n",ack);
    }
    mem_realloc(ack,0);
  } else {
    errexit("You must specify a valid slot number.\n");
  }
}

void sms_delete_all (struct sms_data** slist) {
  size_t i = 0;

  if (slist == NULL) {
    return;
  }
  while (slist[i] != NULL) {
    if (slist[i]->slot >= SCMXX_SLOT_SMS_MIN) {
      sms_delete_slot(slist[i]->slot);
    }
    ++i;
  }
}

void sms_send_slot (int smsslot, char* smsnumber) {
  char at_command[128];
  char* ausgabe;

  memset(at_command,0,sizeof(at_command));

  new_at_command(at_command,"+CMSS");
  add_at_command(at_command,"=%d",smsslot);
  if (str_len(smsnumber)>0) {
    add_at_command(at_command,",%s,%d",smsnumber,numtype(smsnumber));
  }
  ausgabe=tty_write_read(at_command);
  if (!strcmp(ausgabe,"+CMS ERROR: INVALID MEMORY INDEX")) {
    errexit("This index is not available.\n");
  } else if(strstr(ausgabe,"ERROR")!=NULL) {
    errexit("%s, aborting.\n",ausgabe);
  }
  myprintf(0,"The message was sent. Message referrer: %s\n",ausgabe+7);
  mem_realloc(ausgabe,0);
  ausgabe=tty_read(at_command);
  myprintf(0,"The phone returned: %s\n",ausgabe);
  mem_realloc(ausgabe,0);
}

void sms_send_pdu (char* pdu, int direct) {
  char at_command[128];
  char* ausgabe;
  char* ack;

  memset(at_command,0,sizeof(at_command));

  if (direct) {
    new_at_command(at_command,"+CMGS");
  } else {
    new_at_command(at_command,"+CMGW");
  }
  add_at_command(at_command,"=%d",sms_tpdu_len(pdu));
  tty_write_command(at_command);
  myprintf(0,"Waiting for data request...\n");
  ack=tty_read(at_command);
  if (!strncmp(ack,"> ",2)) {
    mem_realloc(ack,0);
    myprintf(0,"Sending data...\n");
    if (tty_write_data(pdu,strlen(pdu)) == -1) {
      errexit("ERROR on sending data: %s\n", strerror(errno));
    }
    ausgabe=tty_read(at_command);
    if (strstr(ausgabe,"ERROR")!=NULL) {
      errexit("An error occured on sending the SMS: %s\n",ausgabe);
    }
    if (direct) {
      myprintf(0,"The message was sent. Message referrer: %s\n",ausgabe+7);
    } else {
      myprintf(0,"The message was saved to SMS memory slot %s\n",ausgabe+7);
    }
    mem_realloc(ausgabe,0);
    ack=tty_read(at_command);
    myprintf(0,"The phone returned: %s\n",ack);
    mem_realloc(ack,0);
  } else {
    if (!strncmp(ack,"ERROR",5)) {
      errexit("An unknown error occured.");
    } else if (!strcasecmp(ack,"+CMS ERROR: MEMORY FULL")) {
      errexit("There is no slot free to store the sms.\n");
    } else {
      errexit("ERROR: %s\n",ack+12);
    }
    mem_realloc(ack,0);
  }
}

void sms_send_pdu_all (struct sms_data** slist, char* number, int direct) {
  int i = 0;
  
  if (slist == NULL || *slist == NULL) {
    return;
  }
  while (slist[i] != NULL) {
    if (slist[i]->slot >= SCMXX_SLOT_SMS_MIN) {
      myprintf(0,"Trying to send SMS from slot %d...\n",slist[i]->slot);
    }
    if (slist[i]->type != SMS_OUTGOING) {
      myprintf(0,"Sending is only supported with SMS of outgoing type.\n");
    } else {
      if (str_len(number) && slist[i]->slot >= SCMXX_SLOT_SMS_MIN) {
	sms_send_slot(slist[i]->slot, number);
      } else {
	sms_send_pdu(slist[i]->tpdu,direct);
      }
    }
    ++i;
  }
}

char* sms_pdu_create (char* file, char* text, char* number, 
		      struct smsopts* mysmsopts)
{
  char* pdu;
  char smsinput[(255*155)+1];
  int myfd,i,eof_present=0,read_ret;

  memset(smsinput,0,sizeof(smsinput));

  if (str_len(text)==0) {// --sms-text was not used, look for other sources
    myfd=open_myFile_ro(file);
    for (i=0;i<sizeof(smsinput)-1;i++) {
      read_ret=read(myfd,&smsinput[i],1);
      if (read_ret==0) {
	eof_present=1;
	break;
      } else if (read_ret==-1) {
	errexit("Error while reading from file.\n");
      }
    }
    if (eof_present==0) {
      errexit("SMS text exceeds buffer size.\n",sizeof(smsinput)-1);
    }
    if (str_len(smsinput)==0) {
      errexit("No SMS text found (or zero length).\n");
    } else {
      myprintf(0,"%s has %d bytes.\n",file,str_len(smsinput));
    }
  } else {
    if (str_len(text) < sizeof(smsinput)) {
      strncpy(smsinput,text,sizeof(smsinput)-1);
    }
  }
  myprintf(0,"Creating PDU...\n");
  pdu = sms_pdu_create_submit(smsinput,number,mysmsopts);
  myprintf(1,"PDU: %s\n",pdu);
  return pdu;
}

/* slist must be of size (ssize*sizeof(struct sms_data*))
 * slist ends with NULL
 * If all entries (and NULL) fit into slist, 0 is returned
 * else the suggested number of entries is returned.
 *
 * The pointers in slist are malloc'ed and need to be free'd.
 */
int sms_get (struct sms_data** slist, size_t ssize, int slot, int mark_read) {
  char* command;
  char* ausgabe;
  char* error;
  size_t esize;
  char* temp;
  size_t retval = 0;
  int type = 0;

  if (slot >= SCMXX_SLOT_SMS_MIN) {
    command = mem_alloc(sizeof(AT_PREFIX)-1 + 5
			+ 1 + numlen(slot) +1, 1);
    if (mark_read) {
      new_at_command(command,"+CMGR");
    } else {
      new_at_command(command,"^SMGR");
    }
    add_at_command(command,"=%d",slot);    
  } else { // -1=unread, -2=read, -3=unsent, -4=sent, -5=all
    command = mem_alloc(sizeof(AT_PREFIX)-1 + 8, 1);
    if (mark_read) {
      new_at_command(command,"+CMGL");
    } else {
      new_at_command(command,"^SMGL");
    }
    switch (slot) {
    case SCMXX_SLOT_ALL:
      add_at_command(command,"=%d",4);
      break;
    case SCMXX_SLOT_SMS_SENT:
      add_at_command(command,"=%d",3);
      break;
    case SCMXX_SLOT_SMS_UNSENT:
      add_at_command(command,"=%d",2);
      break;
    case SCMXX_SLOT_SMS_READ:
      add_at_command(command,"=%d",1);
      break;
    case SCMXX_SLOT_SMS_UNREAD:
      add_at_command(command,"=%d",0);
      break;
    default:
      return 0;
      break;
    }
  }
  ausgabe = tty_write_read(command);
  /* example:
   * +CMGR: 1,,159
   * +CMGL: 2,2,,14
   */
  esize = strlen(ausgabe)+1;
  error = mem_alloc(esize,1);
  switch (check_return_code(ausgabe,error,esize)) {
  case -1:
    break;
  case 0:
    if (strcasecmp(error,"INVALID MEMORY INDEX") == 0) {
      errexit("This index is not available.\n");
    } else {
      errexit("Error: %s.\n",error);
    }
    break;
  case 1:
    errexit("There are no SMS of this type on the phone.\n");
    break;
  }
  mem_realloc(error,0);
  if (slot < SCMXX_SLOT_SMS_MIN) {
    myprintf(0,"Looking for SMS of specified type...\n");
  }
  do {
    slist[retval]=mem_alloc(sizeof(**slist),0);
    if (slot >= SCMXX_SLOT_SMS_MIN) {
      slist[retval]->slot = slot;
      type = atoi(ausgabe+7);
    } else {
      slist[retval]->slot = atoi(ausgabe+7);
      temp = index(ausgabe+7,',');
      if (temp != NULL) {
	type = atoi(temp+1);;
      } else {
	errexit("Output parsing error.\n");
      }
    }
    if (type < 2) {
      slist[retval]->type = SMS_INCOMING;
    } else {
      slist[retval]->type = SMS_OUTGOING;
    }
    mem_realloc(ausgabe,0);
    ausgabe = tty_read(command);
    if (!strcmp(ausgabe,"OK")) {
      errexit("Slot %d is empty.\n", slist[retval]->slot);
    } else {
      myprintf(0,"Found ");
      switch (type) {
      case 0:
	myprintf(0,"incoming, unread SMS ");
	break;
      case 1:
	myprintf(0,"incoming, read SMS ");
	break;
      case 2:
	myprintf(0,"outgoing, unsent SMS ");
	break;
      case 3:
	myprintf(0,"outgoing, sent SMS ");
	break;
      default:
	myprintf(0,"SMS of unknown type ");
	break;
      }	    
      myprintf(0,"in slot %d.\n",slist[retval]->slot);
      if (strlen(ausgabe) >= sizeof(slist[retval]->tpdu)) {
	errexit("Returned PDU exceeds buffer size.\n");
      } else {
	memset(slist[retval]->tpdu,0,sizeof(slist[retval]->tpdu));
	strncpy(slist[retval]->tpdu,ausgabe,sizeof(slist[retval]->tpdu));
      }
      mem_realloc(ausgabe,0);
      ausgabe = tty_read(command);
    }
  } while (retval++ <= ssize && check_return_code(ausgabe,NULL,0) < 0);
  mem_realloc(ausgabe,0);
  slist[retval] = NULL;
  return retval;
}

void sms_print (FILE* filefd, char* pipe, struct sms_data* sms) {
  FILE* pipefd = NULL;

  if (filefd != NULL) { //send output to a file
    sms_pdu_decode(filefd,sms->slot,sms->type,sms->tpdu);
  }

  if (str_len(pipe)) { //also send output (splitted) to a given pipe
    pipefd = popen(pipe,"w");
    if (pipefd != NULL) {
      sms_pdu_decode(pipefd,sms->slot,sms->type,sms->tpdu);
      if (pclose(pipefd) == -1) {
	errexit("\nERROR on closing pipe.\n");
      }
    }
  }
}

void sms_print_all (char* file, char* pipe, struct sms_data** slist) {
  FILE* filefd = NULL;
  size_t i = 0;

  if (slist == NULL) {
    return;
  }

  if (str_len(file) != 0) {
    if (!strcmp(file,"-")) {
      filefd=stdout;
    } else {
      filefd=fdopen(open_myFile_rw(file),"w+");
    }
  } else {
    if (str_len(pipe) == 0) {
      return;
    }
  }
  while (slist[i] != NULL) {
    sms_print(filefd,pipe,slist[i++]);
  }
  if (str_len(file) && strcmp(file,"-")) {
    fclose(filefd);
  }
}

void sms_get_direct (char* file, char* pipe) {
  char at_command[128];
  char* ausgabe;
  char* temp;
  FILE* filefd = NULL;
  struct sms_data sms;
	
  memset(at_command,0,sizeof(at_command));
  sms.slot = SCMXX_SLOT_SMS_MIN-1;
  sms.type = SMS_INCOMING;

  if (str_len(file) == 0 && str_len(pipe) == 0) {
    errexit("No output method was specified.\n");
  }
  new_at_command(at_command,"+CSMS=1");
  ausgabe=tty_write_read(at_command);
  if (strstr(ausgabe,"ERROR")==NULL) {
    mem_realloc(tty_read(at_command),0);
  } else {
    errexit("Could not set Phase 2+ compatible mode.\n");
  }
  mem_realloc(ausgabe,0);
  new_at_command(at_command,"+CNMI=1,2,0,0,1");
  ausgabe=tty_write_read(at_command);
  if (strcmp(ausgabe,"OK")) {
    errexit("Could not set direct mode.\n");
  }
  if (str_len(file) != 0) {
    if (!strcmp(file,"-")) {
      filefd=stdout;
    } else {
      filefd=fdopen(open_myFile_rw(file),"w+");
    }
  }
  while (1) {
    do {
      mem_realloc(ausgabe,0);
      ausgabe=tty_readline();
      if (ausgabe==NULL) {
	errexit("Error on reading from device: %s\n", strerror(errno));
      }
      if ((temp=index(ausgabe,'\r')) || (temp=index(ausgabe,'\n'))) {	
	memset(temp,0,1);
      }
    } while (strlen(ausgabe)==0);
    if (strncmp(ausgabe,"+CMT: ,",7)) {
      ausgabe=tty_read("");
      strncpy(sms.tpdu,ausgabe,sizeof(sms.tpdu));
      ausgabe=mem_realloc(ausgabe,0);
      sms_print(filefd,pipe,&sms);
      //if this is not sent, SMSC will resent the SMS so we would receive it multiple times
      mem_realloc(ausgabe,0);
      ausgabe=tty_write_read("AT+CNMA=0");
      if (strcmp(ausgabe,"OK")) {
	errexit("SMS ACK should have been ok but was not.\n");
      }
    }
  }
}
