/**
 * @file scim_socket.h
 * @brief Socket interfaces.
 */

/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2004 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2003 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_socket.h,v 1.15 2004/02/06 07:53:15 suzhe Exp $
 */

#ifndef __SCIM_SOCKET_H
#define __SCIM_SOCKET_H

namespace scim {

/**
 * @addtogroup Helper
 * @{
 */

class Socket;
class SocketAddress;
class SocketServer;
class SocketClient;

typedef Slot2<void, SocketServer *, const Socket &>
        SocketServerSlotSocket;

typedef Signal2<void, SocketServer *, const Socket &>
        SocketServerSignalSocket;

class SocketError: public Exception
{
public:
    SocketError (const String& what_arg)
        : Exception (String("scim::Socket: ") + what_arg) { }
};

/**
 * The vaild Socket address/protocol family,
 * corresponding to libc PF_LOCAL/AF_LOCAL and PF_INET/AF_INET
 */
enum SocketFamily
{
    SCIM_SOCKET_UNKNOWN,
    SCIM_SOCKET_LOCAL,
    SCIM_SOCKET_INET
};

/**
 * class SocketAddress encapsulates the details of
 * socket address, like socketaddr_un and socketaddr_in.
 */
class SocketAddress
{
    class SocketAddressImpl;
    SocketAddressImpl *m_impl;

public:
    /**
     * constructor.
     *
     * @param addr the string format of the address.
     */
    SocketAddress (const String &addr = String ());

    /**
     * copy constructor.
     */
    SocketAddress (const SocketAddress &addr);

    /**
     * destructor.
     */
    ~SocketAddress ();

    /**
     * copy operator.
     */
    const SocketAddress& operator = (const SocketAddress &addr);

    /**
     * check if this address is valid.
     */
    bool valid () const;

    /**
     * get the family of this socket address.
     */
    SocketFamily get_family () const;

    /**
     * set a new address.
     *
     * @param addr the string format of the address.
     */
    bool set_address (const String &addr);

    /**
     * get the string format of the address.
     */
    String get_address () const;

    /**
     * get the data of socket address,
     * used by class Socket
     *
     * @return the pointer to the data, usually a sockaddr struct.
     */
    const void * get_data () const;

    /**
     * get the length of the data.
     */
    int get_data_length () const;
};

/**
 * class Socket provides basic operation of socket,
 * such as bind connect, read, write etc.
 *
 * This class cannot be created by user, it can only
 * be created by class SocketClient and SocketServer.
 */
class Socket
{
    class SocketImpl;

    SocketImpl *m_impl;

    /**
     * null declaration of copy constructor and operator,
     * to prevent from being constructed by user.
     */
    Socket (const Socket&);
    const Socket& operator = (const Socket&);

public:
    /**
     * create a Socket object from an already created socket_id.
     */
    Socket (int id = -1);

    /**
     * destructor, call close.
     */
    ~Socket ();

    /**
     * check if the socket is valid.
     */
    bool valid () const;

    /**
     * read data from socket.
     *
     * @param buf the buffer to store the data.
     * @param size size of the buffer.
     * 
     * @return the amount of data actually read, -1 means error occurred.
     */
    int read (void *buf, size_t size) const;

    /**
     * read data from socket with a timeout.
     *
     * @param buf the buffer to store the data.
     * @param size size of the buffer, and the amount of data to be read.
     * @param timeout time out in millisecond (1/1000 second), -1 means infinity.
     * 
     * @return the amount of data actually read,
     *         0 means the connection is closed,
     *         -1 means error occurred.
     */
    int read_with_timeout (void *buf, size_t size, int timeout) const;

    /**
     * write data to socket.
     *
     * @param buf the buffer stores the data.
     * @param size size of the data to be sent.
     *
     * @return the amount of data acutally sent, or -1 if an error occurred.
     */
    int write (const void *buf, size_t size) const;

    /**
     * wait for data is ready to read.
     *
     * @param timeout time out in millisecond (1/1000 second), -1 means infinity.
     *
     * @return > 0 if data is OK, == 0 if time is out, < 0 if an error occurred.
     */
    int wait_for_data (int timeout = -1) const;

    /**
     * get the number of the last occurred error.
     */
    int get_error_number () const;

    /**
     * get the message of the last occurred error.
     */
    String get_error_message () const;

    /**
     * get the socket id.
     */
    int get_id () const;

protected:

    /**
     * initiate a connection on a socket.
     *
     * @param addr the address to be connected.
     */
    bool connect (const SocketAddress &addr) const;

    /**
     * bind a socket to an address, used by SocketServer.
     */
    bool bind (const SocketAddress &addr) const;

    /**
     * listen for connections on a socket.
     *
     * @param queue_length the length of the waiting queue.
     */
    bool listen (int queue_length = 5) const;

    /**
     * accept a connection on the socket, used by SocketServer.
     *
     * @return the id of the accepted socket, or -1 if an error is occurred.
     */
    int accept () const;

    /**
     * create a socket for specific family.
     */
    bool create (SocketFamily family);

    /**
     * close the socket.
     */
    void close ();
};

/**
 * class SocketServer provides basic operations to create a Socket Server,
 * such as create, run etc.
 */
class SocketServer : private Socket
{
    class SocketServerImpl;

    SocketServerImpl *m_impl;

public:
    /**
     * default constructor, do nothing.
     */
    SocketServer (int max_clients = -1);

    /**
     * constructor.
     *
     * @param address create a server on this address.
     */
    SocketServer (const SocketAddress &address, int max_clients = -1);

    /**
     * destructor.
     */
    ~SocketServer ();

    /**
     * test if the server is valid.
     */
    bool valid ();

    /**
     * create a socket on the address.
     *
     * @param address the address to be listen.
     *
     * @return true if OK.
     */
    bool create (const SocketAddress &address);

    /**
     * run the server.
     */
    bool run ();

    /**
     * check if the server is running.
     */
    bool is_running () const;

    /**
     * shutdown the server.
     */
    void shutdown ();

    /**
     * close a connection.
     */
    void close_connection (const Socket &socket);

    /**
     * get the number of the last occurred error.
     */
    int get_error_number () const;

    /**
     * get the message of the last occurred error.
     */
    String get_error_message () const;

    /**
     * get the max number of clients.
     */
    int get_max_clients () const;

    /**
     * set the max number of clients.
     */
    void set_max_clients (int max_clients);

public:
    /**
     * connect a slot to accept signal, if a client connection is accepted,
     * this signal will be emitted.
     */
    Connection signal_connect_accept (SocketServerSlotSocket *slot);

    /**
     * connect a slot to receive signal, if a client send data to server,
     * this signal will be emitted.
     */
    Connection signal_connect_receive (SocketServerSlotSocket *slot);

    /**
     * connect a slot to exception signal, if a exception was occurred
     * to a connection, this signal will be emitted.
     */
    Connection signal_connect_exception (SocketServerSlotSocket *slot);
};

/**
 * class SocketClient provides basic operations to create a Socket Client,
 * such as connect, read, write, etc.
 */
class SocketClient : public Socket
{
    bool m_connected;

public:
    /**
     * constructor.
     */
    SocketClient ();

    /**
     * constructor.
     *
     * @param address the server address to be connected.
     */
    SocketClient (const SocketAddress &address);

    /**
     * destructor.
     */
    ~SocketClient ();

    /**
     * check if the socket is connected.
     */
    bool is_connected () const;

    /**
     * connect to a server.
     *
     * @param address the server address.
     */
    bool connect (const SocketAddress &address);

    /**
     * close the client.
     */
    void close ();
};

/** @} */

} // namespace scim

#endif //__SCIM_SOCKET_H

/*
vi:ts=4:nowrap:ai:expandtab
*/

