/** @file scim_frontend.h
 *  @brief Defines scim::FrontEndBase interface.
 *  
 */

/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2004 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2003 James Su <suzhe@turbolinux.com.cn>
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_frontend.h,v 1.23 2004/02/12 09:40:12 suzhe Exp $
 */

#ifndef __SCIM_FRONTEND_H
#define __SCIM_FRONTEND_H

namespace scim {

/**
 * @addtogroup FrontEnd
 * The base classes for FrontEnd modules.
 * @{
 */

/**
 * @brief An exception class to hold FrontEnd related errors.
 *
 * scim::FrontEndBase and its derived classes must throw
 * scim::FrontEndError object when error.
 */
class FrontEndError: public Exception
{
public:
    FrontEndError (const String& what_arg)
        : Exception (String("scim::FrontEnd: ") + what_arg) { }
};

class FrontEndBase;

/**
 * @typedef typedef Pointer <FrontEndBase> FrontEndPointer;
 *
 * A smart pointer for scim::FrontEndBase and its derived classes.
 */
typedef Pointer <FrontEndBase> FrontEndPointer;

/**
 * @brief The base class to implement the FrontEnd objects.
 *
 * FrontEnd is an interface between ServerFactory/ServerInstance objects
 * and the user applications. It forward the user requests to
 * ServerFactory/ServerInstance objects, and handle the requests sent back.
 */
class FrontEndBase : public ReferencedObject
{
    typedef std::vector <ServerInstancePointer> ServerInstanceRepository;
    typedef std::vector <ServerFactoryPointer>  ServerFactoryRepository;

    BackEndPointer m_backend;

    ServerInstanceRepository m_server_instance_repository;
    ServerFactoryRepository  m_server_factory_repository;

    int m_server_instance_id_count;

public:
    /**
     * @brief Constructor.
     * @param backend A BackEnd object which holds all ServerFactory objects.
     */
    FrontEndBase (const BackEndPointer &backend);

    /**
     * @brief Virtual destructor.
     */
    virtual ~FrontEndBase ();

private:
    /**
     * @name Slot functions.
     * These functions will be connected to the corresponding Signals
     * of the ServerInstance objects.
     *
     * @{
     */
    void slot_show_preedit_string (ServerInstanceBase * si);
    void slot_show_status_string  (ServerInstanceBase * si);
    void slot_show_aux_string     (ServerInstanceBase * si);
    void slot_show_lookup_table   (ServerInstanceBase * si);

    void slot_hide_preedit_string (ServerInstanceBase * si);
    void slot_hide_status_string  (ServerInstanceBase * si);
    void slot_hide_aux_string     (ServerInstanceBase * si);
    void slot_hide_lookup_table   (ServerInstanceBase * si);

    void slot_update_preedit_caret  (ServerInstanceBase * si, int caret);
    void slot_update_preedit_string (ServerInstanceBase * si, const WideString & str, const AttributeList & attrs);
    void slot_update_status_string  (ServerInstanceBase * si, const WideString & str, const AttributeList & attrs);
    void slot_update_aux_string     (ServerInstanceBase * si, const WideString & str, const AttributeList & attrs);
    void slot_commit_string         (ServerInstanceBase * si, const WideString & str);
    void slot_forward_keyevent      (ServerInstanceBase * si, const KeyEvent & key);
    void slot_update_lookup_table   (ServerInstanceBase * si, const LookupTable & table);

    void slot_update_full_width_punctuation (ServerInstanceBase * si, bool full);
    void slot_update_full_width_letter      (ServerInstanceBase * si, bool full);
    /**
     * @}
     */

private:
    void query_server_factories ();

    ServerFactoryPointer  find_server_factory (const String &uuid) const;

    ServerInstancePointer find_server_instance (int id) const;

    void add_server_instance (const ServerInstancePointer &si);

    void attach_server_instance (const ServerInstancePointer &si);

protected:
    /**
     * @name functions can be used by derived classes.
     * 
     * @{
     */

    /**
     * @brief Get the server factories list for specific encoding
     *
     * @param uuids    the vector to store the factories' uuids which
     *                 support the encoding.
     * @param encoding the encoding to be queried. If empty,
     *                 all server factories will be returned.
     *
     * @return the number of server factories found.
     */
    uint32 get_server_factory_list (std::vector<String> &uuids, const String &encoding) const;

    /**
     * @brief get the name of a server factory.
     *
     * @param uuid the uuid of the server factory
     * @return the name of the server factory.
     */
    WideString get_server_factory_name (const String &uuid) const;

    /**
     * @brief get the authors info of a server factory.
     * @param uuid the uuid of the server factory
     * @return the authors info of the server factory.
     */
    WideString get_server_factory_authors (const String &uuid) const;

    /**
     * @brief get the credits info of a server factory.
     * @param uuid the uuid of the server factory
     * @return the credits info of the server factory.
     */
    WideString get_server_factory_credits (const String &uuid) const;

    /**
     * @brief get the help info of a server factory.
     * @param uuid the uuid of the server factory
     * @return the help info of the server factory.
     */
    WideString get_server_factory_help (const String &uuid) const;

    /**
     * @brief get the icon file of a server factory.
     * @param uuid the uuid of the server factory
     * @return the icon file name of the server factory.
     */
    String get_server_factory_icon_file (const String &uuid) const;

    /**
     * @brief get the supported locales of a server factory.
     * @param uuid the uuid of the server factory
     * @return a comma separated list of the supported locales.
     */
    String get_server_factory_locales (const String &uuid) const;

    /**
     * @brief get all locales supported by BackEnd.
     * @return a comman separated list of all supported locales.
     */
    String get_all_locales () const;

    // server instance related functions.

    /**
     * @brief create a new server instance for specific encoding.
     *
     * @param sf_uuid the ServerFactory UUID.
     * @param encoding the encoding to be used.
     *
     * @return the newly created server instance id, -1 means error occurred.
     */
    int  new_server_instance (const String &sf_uuid, const String &encoding);

    /**
     * @brief replace a server instance by a new instance created by another factory.
     *
     * This function is used to change the input method for an input context on the fly.
     *
     * @param si_id the server instance to be replaced.
     * @param sf_uuid the new server factory to be used.
     */
    bool replace_server_instance (int si_id, const String &sf_uuid);

    /**
     * @brief delete a server instance according to its id.
     * @param id the id of the server instance to be deleted.
     * @return true if success, false if there is no such instance.
     */
    bool delete_server_instance (int id);

    /**
     * @brief delete all server instances.
     *
     * This function should be called just before quitting the FrontEnd.
     */
    void delete_all_server_instances ();

    /**
     * @brief get the working encoding of a server instance.
     * @param id the server instance id.
     * @return the working encoding of this server instance.
     */
    String get_server_instance_encoding (int id) const;

    /**
     * @brief get the name of a server instance.
     * @param id the server instance id.
     * @return the name of this server instance,
     *         aka. the name of its factory.
     */
    WideString get_server_instance_name (int id) const;

    /**
     * @brief get the authors info of a server instance.
     * @param id the server instance id.
     * @return the authors info of this server instance,
     *         aka. the authors of its factory.
     */
    WideString get_server_instance_authors (int id) const;

    /**
     * @brief get the credits info of a server instance.
     * @param id the server instance id.
     * @return the credits info of this server instance,
     *         aka. the credits of its factory.
     */
    WideString get_server_instance_credits (int id) const;

    /**
     * @brief get the help of a server instance.
     * @param id the server instance id.
     * @return the help of this server instance,
     *         aka. the help of its factory.
     */
    WideString get_server_instance_help (int id) const;

    /**
     * @brief get the icon file of a server instance.
     * @param id the server instance id.
     * @return the icon file name of this server instance.
     */
    String get_server_instance_icon_file (int id) const;

    /**
     * @brief process a key event using specific server instance.
     * @param id the server instance id.
     * @param key the key event to be processed.
     * @return true if the event was processed successfully,
     *         false if the event was not processed and should
     *         be forward to the client application.
     */
    bool process_key_event (int id, const KeyEvent& key) const;

    /**
     * @brief let a specific server instance move its preedit caret.
     * @param id the server instance id.
     * @param pos the new preedit caret position.
     */
    void move_preedit_caret (int id, unsigned int pos) const;

    /**
     * @brief let a specific server instance select an item in its current lookup table.
     * @param id the server instance id.
     * @param item the lookup table item to be selected.
     */
    void select_lookup_table (int id, unsigned int item) const;

    /**
     * @brief update the page size of a specific server instance's lookup table.
     * @param id the server instance id.
     * @param page_size the new page size to be used.
     */
    void update_lookup_table_page_size (int id, unsigned int page_size) const;

    /**
     * @brief reset a specific server instance.
     * @param id the id of the server instance to be reset.
     */
    void reset_server_instance (int id) const;

    /**
     * @brief focus in a specific server instance.
     * @param id the id of the server instance to be focused in.
     */
    void focus_in_server_instance (int id) const;

    /**
     * @brief focus out a specific server instance.
     * @param id the id of the server instance to be focused out.
     */
    void focus_out_server_instance (int id) const;

    /**
     * @brief let a specific server instance toggle its full/half width punctuation state.
     * @param id the server instance id.
     */
    void toggle_full_width_punctuation (int id) const;

    /**
     * @brief let a specific server instance toggle its full/half width letter state.
     * @param id the server instance id.
     */
    void toggle_full_width_letter (int id) const;

    /**
     * @brief let a specific server instance toggle its input status.
     * @param id the server instance id.
     */
    void toggle_input_status (int id) const;

    /**
     * @}
     */

protected:
    /**
     * @name Pure virtual protected methods.
     *
     * The following methods should be implemented by derivation classes.
     * these functions handle the real things.
     * 
     * @{
     */
 
    /**
     * @brief show preedit string area for a server instance.
     * @param id the id of the server instance, it must be focused in.
     */
    virtual void show_preedit_string (int id) = 0;

    /**
     * @brief show status string area for a server instance.
     * @param id the id of the server instance, it must be focused in.
     */
    virtual void show_status_string  (int id) = 0;

    /**
     * @brief show aux string area for a server instance.
     * @param id the id of the server instance, it must be focused in.
     */
    virtual void show_aux_string     (int id) = 0;

    /**
     * @brief show lookup table area for a server instance.
     * @param id the id of the server instance, it must be focused in.
     */
    virtual void show_lookup_table   (int id) = 0;

    /**
     * @brief hide preedit string area for a server instance.
     * @param id the id of the server instance, it must be focused in.
     */
    virtual void hide_preedit_string (int id) = 0;

    /**
     * @brief hide status string area for a server instance.
     * @param id the id of the server instance, it must be focused in.
     */
    virtual void hide_status_string  (int id) = 0;

    /**
     * @brief hide aux string area for a server instance.
     * @param id the id of the server instance, it must be focused in.
     */
    virtual void hide_aux_string     (int id) = 0;

    /**
     * @brief hide lookup table area for a server instance.
     * @param id the id of the server instance, it must be focused in.
     */
    virtual void hide_lookup_table   (int id) = 0;

    /**
     * @brief update the position of preedit caret for a server instance.
     * @param id the id of the server instance, it must be focused in.
     * @param caret the new caret position.
     */
    virtual void update_preedit_caret  (int id, int caret) = 0;

    /**
     * @brief update the content of preedit string for a server instance.
     * @param id the id of the server instance, it must be focused in.
     * @param str the new content of preedit string.
     * @param attrs the string attributes.
     */
    virtual void update_preedit_string (int id, const WideString & str, const AttributeList & attrs) = 0;

    /**
     * @brief update the content of status string for a server instance.
     * @param id the id of the server instance, it must be focused in.
     * @param str the new content of status string.
     * @param attrs the string attributes.
     */
    virtual void update_status_string  (int id, const WideString & str, const AttributeList & attrs) = 0;

    /**
     * @brief update the content of aux string for a server instance.
     * @param id the id of the server instance, it must be focused in.
     * @param str the new content of aux string.
     * @param attrs the string attributes.
     */
    virtual void update_aux_string     (int id, const WideString & str, const AttributeList & attrs) = 0;

    /**
     * @brief update the content of lookup table for a server instance.
     * @param id the id of the server instance, it must be focused in.
     * @param table the new lookup table.
     */
    virtual void update_lookup_table   (int id, const LookupTable & table) = 0;

    /**
     * @brief commit a string to client for a server instance.
     * @param id the id of the server instance to commit the string.
     * @param str the string to be committed.
     */
    virtual void commit_string         (int id, const WideString & str) = 0;

    /**
     * @brief forward a keyevent to the client of a server instance.
     * @param id the id of the server instance, it must be focused in.
     * @param key the key event to be forwarded.
     */
    virtual void forward_keyevent      (int id, const KeyEvent & key) = 0;

    /**
     * @brief update the full width punctuation status for a server instance.
     * @param id the id of the server instance, it must be focused in.
     * @param full indicates the punctuation input status,
     *             true = full width, false = half width.
     */
    virtual void update_full_width_punctuation (int id, bool full) = 0;

    /**
     * @brief update the full width letter status for a server instance.
     * @param id the id of the server instance, it must be focused in.
     * @param full indicates the letter input status,
     *             true = full width, false = half width.
     */
    virtual void update_full_width_letter      (int id, bool full) = 0;

    /**
     * @}
     */

public:
    /**
     * @brief init the frontend.
     *
     * This method must be implemented by derivation classes.
     */
    virtual void init (int argc, char **argv) = 0;

    /**
     * @brief run the frontend.
     *
     * This method must be implemented by derivation classes.
     */
    virtual void run () = 0;
};

/** @} */

} // namespace scim

#endif //__SCIM_FRONTEND_H

/*
vi:ts=4:nowrap:ai:expandtab
*/
