/** @file scim_gconf_config.cpp
 * implementation of GConfConfig class.
 */

/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_gconf_config.cpp,v 1.12 2004/03/02 09:07:00 suzhe Exp $
 */

#define Uses_SCIM_CONFIG_BASE
#include <gconf/gconf.h>
#include <gconf/gconf-client.h>
#include <glib-object.h>
#include "scim_private.h"
#include "scim.h"
#include "scim_gconf_config.h"

#define scim_module_init gconf_LTX_scim_module_init
#define scim_module_exit gconf_LTX_scim_module_exit
#define scim_config_module_init gconf_LTX_scim_config_module_init
#define scim_config_module_create_config gconf_LTX_scim_config_module_create_config

using namespace scim;

extern "C" {
    void scim_module_init (void)
    {
        SCIM_DEBUG_CONFIG(1) << "Initializing GConf Config module...\n";
    }

    void scim_module_exit (void)
    {
        SCIM_DEBUG_CONFIG(1) << "Exiting GConf Config module...\n";
    }

    void scim_config_module_init ()
    {
        SCIM_DEBUG_CONFIG(1) << "Initializing GConf Config module (more)...\n";
        g_type_init ();
    }

    ConfigPointer scim_config_module_create_config (const String &app_name)
    {
        SCIM_DEBUG_CONFIG(1) << "Creating a GConf Config instance...\n";
        return new GConfConfig (app_name);
    }
}

GConfConfig::GConfConfig (const String& app_name)
    : ConfigBase (app_name), m_client (NULL)
{
    SCIM_DEBUG_CONFIG(1) << "Constructing GConfConfig object...\n";

    m_client = gconf_client_get_default ();

    if (!m_client)
        throw ConfigError ("Failed to create GConfClient!");

    GError *err = NULL;

    gconf_client_add_dir (m_client, "/apps/scim", GCONF_CLIENT_PRELOAD_RECURSIVE, &err);

    if (err) {
        SCIM_DEBUG_CONFIG(2) << " Error in GConfConfig : " << err->message << "\n";
        g_error_free (err);
    }
}

GConfConfig::~GConfConfig ()
{
    SCIM_DEBUG_CONFIG(1) << "Destroying GConf Config instance...\n";
    SCIM_DEBUG_CONFIG(2) << " Flushing GConf Server...\n";
    flush ();
    if (m_client) {
        SCIM_DEBUG_CONFIG(2) << " Unreferencing GConf client...\n";
        g_object_unref (m_client);
    }
}

bool
GConfConfig::valid () const
{
    return ConfigBase::valid() && m_client;
}

// String
bool
GConfConfig::read (const String& key, String *pStr) const
{
    if (!valid () || !pStr || key.empty()) return false;

    GConfValue *result = get_gconf_value (key);

    if (result) {
        if (result->type == GCONF_VALUE_STRING) {
            *pStr = String (gconf_value_get_string (result));
            gconf_value_free (result);
            return true;
        } else {
            *pStr = String ("");
            gconf_value_free (result);
            return false;
        }
    }

    *pStr = String ("");
    return false;
}

// int
bool
GConfConfig::read (const String& key, int *pl) const
{
    if (!valid () || !pl || key.empty()) return false;
    
    GConfValue *result = get_gconf_value (key);

    if (result) {
        if (result->type == GCONF_VALUE_INT) {
            *pl = gconf_value_get_int (result);
            gconf_value_free (result);
            return true;
        } else {
            *pl = 0;
            gconf_value_free (result);
            return false;
        }
    }

    *pl = 0;
    return false;
}

// double
bool
GConfConfig::read (const String& key, double* val) const
{
    if (!valid () || !val || key.empty()) return false;
    
    GConfValue *result = get_gconf_value (key);

    if (result) {
        if (result->type == GCONF_VALUE_FLOAT) {
            *val = gconf_value_get_float (result);
            gconf_value_free (result);
            return true;
        } else {
            *val = 0;
            gconf_value_free (result);
            return false;
        }
    }

    *val = 0;
    return false;
}

// bool
bool
GConfConfig::read (const String& key, bool* val) const
{
    if (!valid () || !val || key.empty()) return false;
    
    GConfValue *result = get_gconf_value (key);

    if (result) {
        if (result->type == GCONF_VALUE_BOOL) {
            *val = gconf_value_get_bool (result);
            gconf_value_free (result);
            return true;
        } else {
            *val = false;
            gconf_value_free (result);
            return false;
        }
    }

    *val = false;
    return false;
}

//String list
bool
GConfConfig::read (const String& key, std::vector <String>* val) const
{
    if (!valid () || !val || key.empty()) return false;
    
    GConfValue *result = get_gconf_value (key);

    if (result) {
        if (result->type == GCONF_VALUE_LIST &&
            gconf_value_get_list_type (result) == GCONF_VALUE_STRING) {
            GSList *iter = gconf_value_get_list (result);
            while (iter != NULL) {
                if (iter->data) {
                    val->push_back (String (static_cast<char*>(iter->data)));
                } else {
                    val->push_back (String (""));
                }

                iter = g_slist_next (iter);
            }
            gconf_value_free (result);
            return true;
        } else {
            val->clear ();
            gconf_value_free (result);
            return false;
        }
    }

    val->clear();
    return false;
}

//int list
bool
GConfConfig::read (const String& key, std::vector <int>* val) const
{
    if (!valid () || !val || key.empty()) return false;
    
    GConfValue *result = get_gconf_value (key);

    if (result) {
        if (result->type == GCONF_VALUE_LIST &&
            gconf_value_get_list_type (result) == GCONF_VALUE_INT) {
            GSList *iter = gconf_value_get_list (result);
            while (iter != NULL) {
                val->push_back ((int)(iter->data));
                iter = g_slist_next (iter);
            }
            gconf_value_free (result);
            return true;
        } else {
            val->clear ();
            gconf_value_free (result);
            return false;
        }
    }

    val->clear();
    return false;
}

// write the value (return true on success)
bool
GConfConfig::write (const String& key, const String& value)
{
    if (!valid () || key.empty()) return false;
    
    GError *err = NULL;
    String real_key = compose_key (key);

    if (!gconf_client_set_string (m_client, real_key.c_str (), value.c_str (), &err)) {
        SCIM_DEBUG_CONFIG(2) << " Error in GConfConfig : "
             << err->message << " | " << real_key << "\n";
        g_error_free (err);
        return false;
    }

    return true;
}

bool
GConfConfig::write (const String& key, int value)
{
    if (!valid () || key.empty()) return false;
    
    GError *err = NULL;
    String real_key = compose_key (key);

    if (!gconf_client_set_int (m_client, real_key.c_str (), value, &err)) {
        SCIM_DEBUG_CONFIG(2) << " Error in GConfConfig : "
             << err->message << " | " << real_key << "\n";
        g_error_free (err);
        return false;
    }

    return true;
}

bool
GConfConfig::write (const String& key, double value)
{
    if (!valid () || key.empty()) return false;
    
    GError *err = NULL;
    String real_key = compose_key (key);

    if (!gconf_client_set_float (m_client, real_key.c_str (), value, &err)) {
        SCIM_DEBUG_CONFIG(2) << " Error in GConfConfig : "
             << err->message << " | " << real_key << "\n";
        g_error_free (err);
        return false;
    }

    return true;
}

bool
GConfConfig::write (const String& key, bool value)
{
    if (!valid () || key.empty()) return false;
    
    GError *err = NULL;
    String real_key = compose_key (key);

    if (!gconf_client_set_bool (m_client, real_key.c_str (), value, &err)) {
        SCIM_DEBUG_CONFIG(2) << " Error in GConfConfig : "
             << err->message << " | " << real_key << "\n";
        g_error_free (err);
        return false;
    }

    return true;
}

bool
GConfConfig::write (const String& key, const std::vector <String>& value)
{
    if (!valid () || key.empty()) return false;
    
    GError *err = NULL;
    String real_key = compose_key (key);

    GSList *result = g_slist_alloc ();

    if (!result) throw ConfigError ("Memory allocation error (g_slist_alloc)!");

    for (std::vector <String>::const_iterator i=value.begin(); i!=value.end(); ++i)
        result = g_slist_append (result, static_cast<void*>(const_cast<char*>(i->c_str ())));

    if (!gconf_client_set_list (m_client, real_key.c_str (), GCONF_VALUE_STRING, result, &err)) {
        SCIM_DEBUG_CONFIG(2) << " Error in GConfConfig : "
             << err->message << " | " << real_key << "\n";
        g_error_free (err);
        g_slist_free (result);
        return false;
    }

    g_slist_free (result);
    return true;
}

bool
GConfConfig::write (const String& key, const std::vector <int>& value)
{
    if (!valid () || key.empty()) return false;
    
    GError *err = NULL;
    String real_key = compose_key (key);

    GSList *result = g_slist_alloc ();

    if (!result) throw ConfigError ("Memory allocation error (g_slist_alloc)!");

    for (std::vector <int>::const_iterator i=value.begin(); i!=value.end(); ++i)
        result = g_slist_append (result, GINT_TO_POINTER (*i));

    if (!gconf_client_set_list (m_client, real_key.c_str (), GCONF_VALUE_INT, result, &err)) {
        SCIM_DEBUG_CONFIG(2) << " Error in GConfConfig : "
             << err->message << " | " << real_key << "\n";
        g_error_free (err);
        g_slist_free (result);
        return false;
    }

    g_slist_free (result);
    return true;
}


// permanently writes all changes
bool
GConfConfig::flush()
{
    if (!valid ()) return false;

    GError *err = NULL;
    gconf_client_suggest_sync (m_client, &err);

    if (err) {
        SCIM_DEBUG_CONFIG(2) << " Error in GConfConfig : " << err->message << "\n";
        g_error_free (err);
        return false;
    }
    return true;
}

// delete entries
bool
GConfConfig::erase (const String& key)
{
    if (!valid ()) return false;

    GError *err = NULL;
    String real_key = compose_key (key);

    gconf_client_unset (m_client, real_key.c_str (), &err);

    if (err) {
        SCIM_DEBUG_CONFIG(2) << " Error in GConfConfig : "
             << err->message << " | " << real_key << "\n";
        g_error_free (err);
        return false;
    }
    return true;
}

String
GConfConfig::compose_key (const String &key) const
{
    if (key[0] == '/')
        return String ("/apps/") + get_app_name () + key;
    else
        return String ("/apps/") + get_app_name () + String ("/") + key;
}

GConfValue *
GConfConfig::get_gconf_value (const String& key) const
{
    GError *err = NULL;
    String real_key = compose_key (key);

    GConfValue * result = gconf_client_get (m_client, real_key.c_str (), &err);

    if (err) {
        SCIM_DEBUG_CONFIG(2) << " Error in GConfConfig : "
             << err->message << " | " << real_key << "\n";
        g_error_free (err);

        if (result) gconf_value_free (result);
        
        return NULL;
    }

    return result;
}

/*
vi:ts=4:nowrap:ai:expandtab
*/
