/** @file scim_panel_gtk.cpp
 */

/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_panel_gtk.cpp,v 1.47.2.1 2004/05/01 13:23:33 suzhe Exp $
 */

#include <sys/wait.h>
#include <sys/types.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <glib.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>

#define Uses_C_STDIO
#define Uses_SCIM_LOOKUP_TABLE
#define Uses_SCIM_SOCKET
#define Uses_SCIM_SOCKET_TRANSACTION
#define Uses_SCIM_CONFIG
#define Uses_SCIM_CONFIG_MODULE
#define Uses_SCIM_DEBUG
#define Uses_SCIM_PANEL

#include "scim_private.h"
#include "scim.h"

#include "scimstringview.h"

#if ENABLE_TRAY_ICON
  #include "scimtrayicon.h"
#endif

using namespace scim;

#include "icons/up.xpm"
#include "icons/down.xpm"
#include "icons/left.xpm"
#include "icons/right.xpm"
#include "icons/setup.xpm"
#include "icons/help.xpm"
#include "icons/trademark.xpm"
#include "icons/full-punct.xpm"
#include "icons/half-punct.xpm"
#include "icons/full-letter.xpm"
#include "icons/half-letter.xpm"
#include "icons/pin-up.xpm"
#include "icons/pin-down.xpm"

#define SCIM_CONFIG_PANEL_GTK_FONT                     "/Panel/Gtk/Font"
#define SCIM_CONFIG_PANEL_GTK_COLOR_NORMAL_BG          "/Panel/Gtk/Color/NormalBackground"
#define SCIM_CONFIG_PANEL_GTK_COLOR_ACTIVE_BG          "/Panel/Gtk/Color/ActiveBackground"
#define SCIM_CONFIG_PANEL_GTK_COLOR_NORMAL_TEXT        "/Panel/Gtk/Color/NormalText"
#define SCIM_CONFIG_PANEL_GTK_COLOR_ACTIVE_TEXT        "/Panel/Gtk/Color/ActiveText"
#define SCIM_CONFIG_PANEL_GTK_TOOLBAR_ALWAYS_SHOW      "/Panel/Gtk/ToolBar/AlwaysShow"
#define SCIM_CONFIG_PANEL_GTK_TOOLBAR_AUTO_SNAP        "/Panel/Gtk/ToolBar/AutoSnap"
#define SCIM_CONFIG_PANEL_GTK_TOOLBAR_HIDE_TIMEOUT     "/Panel/Gtk/ToolBar/HideTimeout"
#define SCIM_CONFIG_PANEL_GTK_TOOLBAR_POS_X            "/Panel/Gtk/ToolBar/POS_X"
#define SCIM_CONFIG_PANEL_GTK_TOOLBAR_POS_Y            "/Panel/Gtk/ToolBar/POS_Y"
#define SCIM_CONFIG_PANEL_GTK_TOOLBAR_SHOW_SERVER_ICON "/Panel/Gtk/ToolBar/ShowServerIcon"
#define SCIM_CONFIG_PANEL_GTK_TOOLBAR_SHOW_SERVER_NAME "/Panel/Gtk/ToolBar/ShowServerName"
#define SCIM_CONFIG_PANEL_GTK_TOOLBAR_SHOW_STICK_ICON  "/Panel/Gtk/ToolBar/ShowStickIcon"
#define SCIM_CONFIG_PANEL_GTK_TOOLBAR_SHOW_SETUP_ICON  "/Panel/Gtk/ToolBar/ShowSetupIcon"
#define SCIM_CONFIG_PANEL_GTK_TOOLBAR_SHOW_HELP_ICON   "/Panel/Gtk/ToolBar/ShowHelpIcon"
#define SCIM_CONFIG_PANEL_GTK_LOOKUP_TABLE_EMBEDDED    "/Panel/Gtk/LookupTableEmbedded"
#define SCIM_CONFIG_PANEL_GTK_LOOKUP_TABLE_VERTICAL    "/Panel/Gtk/LookupTableVertical"
#define SCIM_CONFIG_PANEL_GTK_SHOW_STATUS_BOX          "/Panel/Gtk/ShowStatusBox"
#define SCIM_CONFIG_PANEL_GTK_DEFAULT_STICKED          "/Panel/Gtk/DefaultSticked"
#define SCIM_CONFIG_PANEL_GTK_SHOW_TRAY_ICON           "/Panel/Gtk/ShowTrayIcon"

#ifndef SCIM_SETUP_PROGRAM
  #define SCIM_SETUP_PROGRAM "/scim-setup"
#endif

#define SCIM_SETUP_PROGRAM_PATH (SCIM_BINDIR "/" SCIM_SETUP_PROGRAM)

#define SCIM_KEYBOARD_ICON_FILE     (SCIM_ICONDIR "/keyboard.png")
#define SCIM_TRADEMARK_ICON_FILE    (SCIM_ICONDIR "/trademark.png")
#define SCIM_SETUP_ICON_FILE        (SCIM_ICONDIR "/setup.png")
#define SCIM_HELP_ICON_FILE         (SCIM_ICONDIR "/help.png")
#define SCIM_UP_ICON_FILE           (SCIM_ICONDIR "/up.png")
#define SCIM_DOWN_ICON_FILE         (SCIM_ICONDIR "/down.png")
#define SCIM_LEFT_ICON_FILE         (SCIM_ICONDIR "/left.png")
#define SCIM_RIGHT_ICON_FILE        (SCIM_ICONDIR "/right.png")
#define SCIM_FULL_LETTER_ICON_FILE  (SCIM_ICONDIR "/full-letter.png")
#define SCIM_HALF_LETTER_ICON_FILE  (SCIM_ICONDIR "/half-letter.png")
#define SCIM_FULL_PUNCT_ICON_FILE   (SCIM_ICONDIR "/full-punct.png")
#define SCIM_HALF_PUNCT_ICON_FILE   (SCIM_ICONDIR "/half-punct.png")
#define SCIM_PIN_UP_ICON_FILE       (SCIM_ICONDIR "/pin-up.png")
#define SCIM_PIN_DOWN_ICON_FILE     (SCIM_ICONDIR "/pin-down.png")


#define TOOLBAR_ICON_SIZE                     16 
#define TRAY_ICON_SIZE                        16

// Declaration of internal functions.
static void     ui_initialize                          (int argc, char **argv);

static void     ui_settle_input_window                 (bool relative = false);
static void     ui_settle_lookup_table_window          (bool relative = false);
static void     ui_settle_toolbar_window               (void);

static int      ui_screen_width                        (void);
static int      ui_screen_height                       (void);

#if ENABLE_TRAY_ICON
static gboolean ui_create_tray_icon_when_idle          (gpointer        data);
#endif

#if GDK_MULTIHEAD_SAFE
static void     ui_switch_screen                       (GdkScreen      *screen);
#endif

static GdkPixbuf * ui_scale_pixbuf                     (GdkPixbuf      *pixbuf,
                                                        int             width,
                                                        int             height);

static GtkWidget * ui_create_label                     (const String   &name,
                                                        const String   &iconfile,
                                                        const char    **xpm,
                                                        bool            show_icon_only = false,
                                                        bool            force_icon = false);

static GtkWidget * ui_create_icon_from_file            (const String   &iconfile,
                                                        int             width = -1,
                                                        int             height = -1,
                                                        bool            force_create = false);

static GtkWidget * ui_create_icon_from_xpm             (const char    **xpm,
                                                        int             width = -1,
                                                        int             height = -1,
                                                        bool            force_create = false);

static GtkWidget * ui_create_icon                      (const String   &iconfile,
                                                        const char    **xpm = NULL,
                                                        int             width = -1,
                                                        int             height = -1,
                                                        bool            force_create = false);

static GtkWidget * ui_create_trademark_icon            (void);
static GtkWidget * ui_create_stick_icon                (bool            sticked);
static GtkWidget * ui_create_punct_icon                (bool            full);
static GtkWidget * ui_create_letter_icon               (bool            full);
static GtkWidget * ui_create_help_icon                 (void);
static GtkWidget * ui_create_up_icon                   (void);
static GtkWidget * ui_create_down_icon                 (void);
static GtkWidget * ui_create_left_icon                 (void);
static GtkWidget * ui_create_right_icon                (void);

static GtkWidget * ui_create_stick_command_label       (bool            sticked);
static GtkWidget * ui_create_help_command_label        (void);

#if HAVE_SCIM_SETUP
static GtkWidget * ui_create_setup_icon                (void);
static GtkWidget * ui_create_setup_command_label       (void);
#endif

// callback functions
static void     ui_preedit_area_move_cursor_cb         (ScimStringView *view,
                                                        guint           position);
static void     ui_full_width_punct_button_click_cb    (GtkButton      *button,
                                                        gpointer        user_data);
static void     ui_full_width_letter_button_click_cb   (GtkButton      *button,
                                                        gpointer        user_data);

#if HAVE_SCIM_SETUP
static void     ui_setup_button_click_cb               (GtkButton      *button,
                                                        gpointer        user_data);
#endif

static void     ui_help_button_click_cb                (GtkButton      *button,
                                                        gpointer        user_data);
static gboolean ui_server_button_click_cb              (GtkWidget      *button,
                                                        GdkEvent       *event,
                                                        gpointer        user_data);
static void     ui_server_menu_activate_cb             (GtkMenuItem    *item,
                                                        gpointer        user_data);
static void     ui_server_menu_deactivate_cb           (GtkMenuItem    *item,
                                                        gpointer        user_data);

static gboolean ui_lookup_table_vertical_click_cb      (GtkWidget      *item,
                                                        GdkEventButton *event,
                                                        gpointer        user_data);

static void     ui_lookup_table_horizontal_click_cb    (GtkWidget      *item,
                                                        guint           position);

static void     ui_lookup_table_up_button_click_cb     (GtkButton      *button,
                                                        gpointer        user_data);
static void     ui_lookup_table_down_button_click_cb   (GtkButton      *button,
                                                        gpointer        user_data);

static void     ui_window_stick_button_click_cb        (GtkButton      *button,
                                                        gpointer        user_data);
static void     ui_status_button_click_cb              (GtkButton      *button,
                                                        gpointer        user_data);

static gboolean ui_input_window_motion_cb              (GtkWidget      *window,
                                                        GdkEventMotion *event,
                                                        gpointer        user_data);
static gboolean ui_input_window_click_cb               (GtkWidget      *window,
                                                        GdkEventButton *event,
                                                        gpointer        user_data);
static gboolean ui_toolbar_window_crossing_cb          (GtkWidget      *window,
                                                        GdkEventCrossing *event,
                                                        gpointer        user_data);
static gboolean ui_toolbar_window_motion_cb            (GtkWidget      *window,
                                                        GdkEventMotion *event,
                                                        gpointer        user_data);
static gboolean ui_toolbar_window_click_cb             (GtkWidget      *window,
                                                        GdkEventButton *event,
                                                        gpointer        user_data);
static gboolean ui_lookup_table_window_motion_cb       (GtkWidget      *window,
                                                        GdkEventMotion *event,
                                                        gpointer        user_data);
static gboolean ui_lookup_table_window_click_cb        (GtkWidget      *window,
                                                        GdkEventButton *event,
                                                        gpointer        user_data);

static gboolean ui_hide_window_timeout_cb              (gpointer data);

static void     ui_command_menu_stick_activate_cb      (GtkMenuItem    *item,
                                                        gpointer        user_data);
static void     ui_command_menu_help_activate_cb       (GtkMenuItem    *item,
                                                        gpointer        user_data);
#if HAVE_SCIM_SETUP
static void     ui_command_menu_setup_activate_cb      (GtkMenuItem    *item,
                                                        gpointer        user_data);
#endif

static void     ui_command_menu_deactivate_cb          (GtkMenuItem    *item,
                                                        gpointer        user_data);


#if ENABLE_TRAY_ICON
static void     ui_tray_icon_destroy_cb                (GtkObject      *object,
                                                        gpointer        user_data);
#endif

static bool     ui_can_hide_input_window               (void);

static PangoAttrList * create_pango_attrlist           (const String    &str,
                                                        const AttributeList &attrs);

// Action function
static void     action_move_preedit_caret              (uint32 position);
static void     action_toggle_full_width_punct         (void);
static void     action_toggle_full_width_letter        (void);
static void     action_request_help                    (void);
static void     action_request_server_menu             (void);
static void     action_change_server_factory           (const String &uuid);
static void     action_select_lookup_table             (uint32 item);
static void     action_lookup_table_page_up            (void);
static void     action_lookup_table_page_down          (void);
static void     action_toggle_window_stick             (void);
static void     action_toggle_input_status             (void);
static void     action_show_command_menu               (void);
#if HAVE_SCIM_SETUP
static void     action_launch_setup_tool               (void);
#endif

// Socket related functions
static bool     initialize_socket                      (void);
static bool     run_socket                             (void);

static gpointer socket_thread_func                     (gpointer data);

static bool     check_client_connection                (const Socket   &client);

static void     socket_accept_callback                 (SocketServer   *server,
                                                        const Socket   &client);
static void     socket_receive_callback                (SocketServer   *server,
                                                        const Socket   &client);
static void     socket_exception_callback              (SocketServer   *server,
                                                        const Socket   &client);

static void     socket_turn_on                         (void);
static void     socket_turn_off                        (void);
static void     socket_update_display                  (void);
static void     socket_update_screen                   (void);
static void     socket_update_spot_location            (void);
static void     socket_update_server_info              (void);
static void     socket_show_help                       (void);
static void     socket_show_server_menu                (void);

static void     socket_show_preedit_string             (void);
static void     socket_show_status_string              (void);
static void     socket_show_aux_string                 (void);
static void     socket_show_lookup_table               (void);
static void     socket_hide_preedit_string             (void);
static void     socket_hide_status_string              (void);
static void     socket_hide_aux_string                 (void);
static void     socket_hide_lookup_table               (void);
static void     socket_update_preedit_string           (void);
static void     socket_update_preedit_caret            (void);
static void     socket_update_status_string            (void);
static void     socket_update_aux_string               (void);
static void     socket_update_lookup_table             (void);
static void     socket_update_full_width_punctuation   (void);
static void     socket_update_full_width_letter        (void);

static gboolean check_exit_timeout_cb                  (gpointer data);

#if HAVE_SCIM_SETUP
static bool     check_scim_setup                       (void);
static void     run_scim_setup                         (void);
static void     sigchld_handler                        (int signum);
#endif

// Declaration of internal variables.
#if GDK_MULTIHEAD_SAFE
static GdkDisplay        *_current_display             = 0;
static GdkScreen         *_current_screen              = 0;
#endif

static GtkWidget         *_input_window                = 0;
static GtkWidget         *_status_area                 = 0;
static GtkWidget         *_preedit_area                = 0;
static GtkWidget         *_aux_area                    = 0;

static GtkWidget         *_lookup_table_window         = 0;
static GtkWidget         *_lookup_table_up_button      = 0;
static GtkWidget         *_lookup_table_down_button    = 0;
static GtkWidget         *_lookup_table_items [SCIM_LOOKUP_TABLE_MAX_PAGESIZE];

static GtkWidget         *_toolbar_window              = 0;
static GtkWidget         *_window_stick_button         = 0;
static GtkWidget         *_server_button               = 0;
static GtkWidget         *_server_menu                 = 0;
static GtkWidget         *_status_button               = 0;
static GtkWidget         *_full_width_punct_button     = 0;
static GtkWidget         *_full_width_letter_button    = 0;
static GtkWidget         *_help_button                 = 0;

#if HAVE_SCIM_SETUP
static GtkWidget         *_setup_button                = 0;
#endif

static GtkWidget         *_help_dialog                 = 0;
static GtkWidget         *_help_scroll                 = 0;
static GtkWidget         *_help_area                   = 0;
static GtkWidget         *_command_menu                = 0;

static GtkTooltips       *_tooltips                    = 0;

#if ENABLE_TRAY_ICON
static ScimTrayIcon      *_tray_icon                   = 0;
static GtkWidget         *_tray_icon_server_button     = 0;
#endif

static gboolean           _input_window_draging        = FALSE;
static gint               _input_window_drag_x         = 0;
static gint               _input_window_drag_y         = 0;

static gboolean           _toolbar_window_draging      = FALSE;
static gint               _toolbar_window_drag_x       = 0;
static gint               _toolbar_window_drag_y       = 0;

static gboolean           _lookup_table_window_draging = FALSE;
static gint               _lookup_table_window_drag_x  = 0;
static gint               _lookup_table_window_drag_y  = 0;

static int                _spot_location_x             = 0;
static int                _spot_location_y             = 0;

static bool               _lookup_table_embedded       = true;
static bool               _lookup_table_vertical       = false;
static bool               _window_sticked              = false;

static bool               _show_status_box             = true;
static bool               _toolbar_always_show         = true;
static bool               _toolbar_auto_snap           = true;
static bool               _toolbar_show_server_icon    = true;
static bool               _toolbar_show_server_name    = false;
static bool               _toolbar_show_stick_icon     = false;
static bool               _toolbar_show_setup_icon     = false;
static bool               _toolbar_show_help_icon      = false;
static bool               _toolbar_should_hide         = false;
static bool               _toolbar_hidden              = false;
static bool               _server_menu_activated       = false;
static bool               _command_menu_activated      = false;

static int                _toolbar_hide_timeout_max    = 0;
static int                _toolbar_hide_timeout_count  = 0;
static guint              _toolbar_hide_timeout        = 0;

static bool               _ui_initialized              = false;

#if HAVE_SCIM_SETUP
static bool               _scim_setup_running          = false;
#endif

static int                _lookup_table_index [SCIM_LOOKUP_TABLE_MAX_PAGESIZE+1];
static KeyEvent           _lookup_table_page_up_key;
static KeyEvent           _lookup_table_page_down_key;

static GdkColor           _normal_bg;
static GdkColor           _normal_text;
static GdkColor           _active_bg;
static GdkColor           _active_text;

static ConfigModule      *_config_module               = 0;
static ConfigPointer      _config;

static SocketServer       _socket_server;
static SocketTransaction  _send_transaction;
static SocketTransaction  _receive_transaction;
static String             _socket_address;
static int                _socket_timeout              = 500;

static int                _socket_client_count         = 0;

static int                _current_socket_client       = -1;
static uint32             _current_client_context      = 0;

static int                _last_socket_client          = -1;
static uint32             _last_client_context         = 0;

static GThread           *_socket_server_thread        = 0;

static guint              _check_exit_timeout          = 0;

static volatile bool      _should_exit                 = false;

static bool               _should_resident             = true;

static bool               _panel_is_on                 = false;

static std::vector<String> _server_menu_uuids;

G_LOCK_DEFINE_STATIC     (_global_resource_lock);

// Implementation of internal functions.
static void
ui_initialize (int argc, char **argv)
{
    SCIM_DEBUG_MAIN (1) << "Initialize UI...\n";

    PangoFontDescription *font_desc = 0;
    GtkWidget *input_window_vbox;
    String str;

#if ENABLE_TRAY_ICON
    bool show_tray_icon = true;
#endif

    int toolbar_window_x = -1;
    int toolbar_window_y = -1;

    gtk_init (&argc, &argv);

    // Read configurations.
    gdk_color_parse ("gray92",     &_normal_bg);
    gdk_color_parse ("black",      &_normal_text);
    gdk_color_parse ("light blue", &_active_bg);
    gdk_color_parse ("black",      &_active_text);

    if (!_config.null ()) {
        str = _config->read (String (SCIM_CONFIG_PANEL_GTK_FONT),
                              String ("default"));

        if (str != String ("default"))
            font_desc = pango_font_description_from_string (str.c_str ());

        str = _config->read (String (SCIM_CONFIG_PANEL_GTK_COLOR_NORMAL_BG),
                             String ("gray92"));
        gdk_color_parse (str.c_str (), &_normal_bg);

        str = _config->read (String (SCIM_CONFIG_PANEL_GTK_COLOR_NORMAL_TEXT),
                             String ("black"));
        gdk_color_parse (str.c_str (), &_normal_text);

        str = _config->read (String (SCIM_CONFIG_PANEL_GTK_COLOR_ACTIVE_BG),
                             String ("light blue"));
        gdk_color_parse (str.c_str (), &_active_bg);

        str = _config->read (String (SCIM_CONFIG_PANEL_GTK_COLOR_ACTIVE_TEXT),
                             String ("black"));
        gdk_color_parse (str.c_str (), &_active_text);

        toolbar_window_x = _config->read (String (SCIM_CONFIG_PANEL_GTK_TOOLBAR_POS_X),
                                          toolbar_window_x);

        toolbar_window_y = _config->read (String (SCIM_CONFIG_PANEL_GTK_TOOLBAR_POS_Y),
                                          toolbar_window_y);

        _show_status_box =
            _config->read (String (SCIM_CONFIG_PANEL_GTK_SHOW_STATUS_BOX),
                           _show_status_box);

        _window_sticked  =
            _config->read (String (SCIM_CONFIG_PANEL_GTK_DEFAULT_STICKED),
                           _window_sticked);

        _lookup_table_vertical =
            _config->read (String (SCIM_CONFIG_PANEL_GTK_LOOKUP_TABLE_VERTICAL),
                           _lookup_table_vertical);

        _lookup_table_embedded =
            _config->read (String (SCIM_CONFIG_PANEL_GTK_LOOKUP_TABLE_EMBEDDED),
                           _lookup_table_embedded);

        _toolbar_always_show =
            _config->read (String (SCIM_CONFIG_PANEL_GTK_TOOLBAR_ALWAYS_SHOW),
                           _toolbar_always_show);

        _toolbar_auto_snap =
            _config->read (String (SCIM_CONFIG_PANEL_GTK_TOOLBAR_AUTO_SNAP),
                           _toolbar_auto_snap);

        _toolbar_show_server_icon =
            _config->read (String (SCIM_CONFIG_PANEL_GTK_TOOLBAR_SHOW_SERVER_ICON),
                           _toolbar_show_server_icon);

        _toolbar_show_server_name =
            _config->read (String (SCIM_CONFIG_PANEL_GTK_TOOLBAR_SHOW_SERVER_NAME),
                           _toolbar_show_server_name);

        _toolbar_show_setup_icon =
            _config->read (String (SCIM_CONFIG_PANEL_GTK_TOOLBAR_SHOW_SETUP_ICON),
                           _toolbar_show_setup_icon);

        _toolbar_show_stick_icon =
            _config->read (String (SCIM_CONFIG_PANEL_GTK_TOOLBAR_SHOW_STICK_ICON),
                           _toolbar_show_stick_icon);

        _toolbar_show_help_icon =
            _config->read (String (SCIM_CONFIG_PANEL_GTK_TOOLBAR_SHOW_HELP_ICON),
                           _toolbar_show_help_icon);

        _toolbar_hide_timeout_max =
            _config->read (String (SCIM_CONFIG_PANEL_GTK_TOOLBAR_HIDE_TIMEOUT),
                           _toolbar_hide_timeout_max);

#if ENABLE_TRAY_ICON
        show_tray_icon =
            _config->read (String (SCIM_CONFIG_PANEL_GTK_SHOW_TRAY_ICON),
                           show_tray_icon);
#endif
    }

#if GDK_MULTIHEAD_SAFE
    // Initialize the Display and Screen.
    {
        _current_display = gdk_display_get_default ();
        _current_screen  = gdk_display_get_default_screen (_current_display);
    }
#endif

    // Create input window
    {
        GtkWidget *vbox;
        GtkWidget *hbox;
        GtkWidget *frame;

        _input_window = gtk_window_new (GTK_WINDOW_POPUP);
        gtk_widget_modify_bg (_input_window, GTK_STATE_NORMAL, &_normal_bg);
        gtk_window_set_policy (GTK_WINDOW (_input_window), TRUE, TRUE, FALSE);
        gtk_window_set_resizable (GTK_WINDOW (_input_window), FALSE);
        gtk_widget_add_events (_input_window,GDK_BUTTON_PRESS_MASK);
        gtk_widget_add_events (_input_window,GDK_BUTTON_RELEASE_MASK);
        gtk_widget_add_events (_input_window,GDK_POINTER_MOTION_MASK);
        g_signal_connect (G_OBJECT (_input_window), "button-press-event",
                          G_CALLBACK (ui_input_window_click_cb),
                          GINT_TO_POINTER (0));
        g_signal_connect (G_OBJECT (_input_window), "button-release-event",
                          G_CALLBACK (ui_input_window_click_cb),
                          GINT_TO_POINTER (1));

        frame = gtk_frame_new (0);
        gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_OUT);
        gtk_container_add (GTK_CONTAINER (_input_window), frame);

        hbox = gtk_hbox_new (FALSE, 0);
        gtk_container_add (GTK_CONTAINER (frame), hbox);

        if (_show_status_box) {
            vbox = gtk_vbox_new (FALSE, 0);
            gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);

            //Create status area
            _status_area = gtk_label_new ("");
            if (font_desc)
                gtk_widget_modify_font (_status_area, font_desc);

            gtk_box_pack_start (GTK_BOX (vbox), _status_area, FALSE, FALSE, 0);
        } else {
            _status_area = 0;
        }

        vbox = gtk_vbox_new (FALSE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), vbox, TRUE, TRUE, 0);
        input_window_vbox = vbox;

        //Create preedit area
        _preedit_area = scim_string_view_new ();
        if (font_desc)
            gtk_widget_modify_font (_preedit_area, font_desc);
        gtk_widget_modify_base (_preedit_area, GTK_STATE_NORMAL, &_normal_bg);
        gtk_widget_modify_base (_preedit_area, GTK_STATE_ACTIVE, &_active_bg);
        gtk_widget_modify_text (_preedit_area, GTK_STATE_NORMAL, &_normal_text);
        gtk_widget_modify_text (_preedit_area, GTK_STATE_ACTIVE, &_active_text);
        scim_string_view_set_width_chars (SCIM_STRING_VIEW (_preedit_area), 24);
        scim_string_view_set_forward_event (SCIM_STRING_VIEW (_preedit_area), TRUE);
        scim_string_view_set_auto_resize (SCIM_STRING_VIEW (_preedit_area), TRUE);
        scim_string_view_set_has_frame (SCIM_STRING_VIEW (_preedit_area), FALSE);
        scim_string_view_set_max_width (SCIM_STRING_VIEW (_preedit_area), ui_screen_width () / 2);
        g_signal_connect (G_OBJECT (_preedit_area), "move_cursor",
                          G_CALLBACK (ui_preedit_area_move_cursor_cb),
                          0);
        gtk_box_pack_start (GTK_BOX (vbox), _preedit_area, TRUE, TRUE, 0);

        //Create aux area
        _aux_area = scim_string_view_new ();
        if (font_desc)
            gtk_widget_modify_font (_aux_area, font_desc);
        gtk_widget_modify_base (_aux_area, GTK_STATE_NORMAL, &_normal_bg);
        gtk_widget_modify_base (_aux_area, GTK_STATE_ACTIVE, &_active_bg);
        gtk_widget_modify_text (_aux_area, GTK_STATE_NORMAL, &_normal_text);
        gtk_widget_modify_text (_aux_area, GTK_STATE_ACTIVE, &_active_text);
        scim_string_view_set_width_chars (SCIM_STRING_VIEW (_aux_area), 24);
        scim_string_view_set_draw_cursor (SCIM_STRING_VIEW (_aux_area), FALSE);
        scim_string_view_set_forward_event (SCIM_STRING_VIEW (_aux_area), TRUE);
        scim_string_view_set_auto_resize (SCIM_STRING_VIEW (_aux_area), TRUE);
        scim_string_view_set_has_frame (SCIM_STRING_VIEW (_aux_area), FALSE);
        scim_string_view_set_max_width (SCIM_STRING_VIEW (_aux_area), ui_screen_width () / 2);
        gtk_box_pack_start (GTK_BOX (vbox), _aux_area, TRUE, TRUE, 0);

        gtk_window_move (GTK_WINDOW (_input_window), ui_screen_width (), ui_screen_height ());

        gtk_widget_show_all (_input_window);
        gtk_widget_hide (_input_window);
    }

    //Create lookup table window
    {
        GtkWidget *vbox;
        GtkWidget *hbox;
        GtkWidget *frame;
        GtkWidget *lookup_table_parent;
        GtkWidget *image;
        GtkWidget *separator;
        GtkRequisition size;

        if (_lookup_table_embedded) {
            _lookup_table_window = gtk_vbox_new (FALSE, 0);
            gtk_box_pack_start (GTK_BOX (input_window_vbox), _lookup_table_window, TRUE, TRUE, 0);
            lookup_table_parent = _lookup_table_window;
            separator = gtk_hseparator_new ();
            gtk_box_pack_start (GTK_BOX (lookup_table_parent), separator, FALSE, FALSE, 0);
        } else {
            _lookup_table_window = gtk_window_new (GTK_WINDOW_POPUP);
            gtk_widget_modify_bg (_lookup_table_window, GTK_STATE_NORMAL, &_normal_bg);
            gtk_window_set_policy (GTK_WINDOW (_lookup_table_window), TRUE, TRUE, FALSE);
            gtk_window_set_resizable (GTK_WINDOW (_lookup_table_window), FALSE);
            gtk_widget_add_events (_lookup_table_window,GDK_BUTTON_PRESS_MASK);
            gtk_widget_add_events (_lookup_table_window,GDK_BUTTON_RELEASE_MASK);
            gtk_widget_add_events (_lookup_table_window,GDK_POINTER_MOTION_MASK);
            g_signal_connect (G_OBJECT (_lookup_table_window), "button-press-event",
                              G_CALLBACK (ui_lookup_table_window_click_cb),
                              GINT_TO_POINTER (0));
            g_signal_connect (G_OBJECT (_lookup_table_window), "button-release-event",
                              G_CALLBACK (ui_lookup_table_window_click_cb),
                              GINT_TO_POINTER (1));
            gtk_container_set_border_width (GTK_CONTAINER (_lookup_table_window), 0);

            frame = gtk_frame_new (0);
            gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_OUT);
            gtk_container_add (GTK_CONTAINER (_lookup_table_window), frame);
            lookup_table_parent = frame;
        }

        //Vertical lookup table
        if (_lookup_table_vertical) {
            vbox = gtk_vbox_new (FALSE, 0);
            gtk_container_add (GTK_CONTAINER (lookup_table_parent), vbox);

            //New table items
            for (int i=0; i<SCIM_LOOKUP_TABLE_MAX_PAGESIZE; ++i) {
                _lookup_table_items [i] = scim_string_view_new ();
                if (font_desc)
                    gtk_widget_modify_font (_lookup_table_items [i], font_desc);
                gtk_widget_modify_base (_lookup_table_items [i], GTK_STATE_NORMAL, &_normal_bg);
                gtk_widget_modify_base (_lookup_table_items [i], GTK_STATE_ACTIVE, &_active_bg);
                gtk_widget_modify_text (_lookup_table_items [i], GTK_STATE_NORMAL, &_normal_text);
                gtk_widget_modify_text (_lookup_table_items [i], GTK_STATE_ACTIVE, &_active_text);
                scim_string_view_set_width_chars (SCIM_STRING_VIEW (_lookup_table_items [i]), 80);
                scim_string_view_set_has_frame (SCIM_STRING_VIEW (_lookup_table_items [i]), FALSE);
                scim_string_view_set_forward_event (SCIM_STRING_VIEW (_lookup_table_items [i]), TRUE);
                scim_string_view_set_auto_resize (SCIM_STRING_VIEW (_lookup_table_items [i]), TRUE);
                scim_string_view_set_draw_cursor (SCIM_STRING_VIEW (_lookup_table_items [i]), FALSE);
                scim_string_view_set_auto_move_cursor (SCIM_STRING_VIEW (_lookup_table_items [i]), FALSE);
                g_signal_connect (G_OBJECT (_lookup_table_items [i]), "button-press-event",
                                  G_CALLBACK (ui_lookup_table_vertical_click_cb),
                                  GINT_TO_POINTER (i));
                gtk_box_pack_start (GTK_BOX (vbox), _lookup_table_items [i], TRUE, TRUE, 0);
            }

            separator = gtk_hseparator_new ();
            gtk_box_pack_start (GTK_BOX (vbox), separator, FALSE, FALSE, 0);

            hbox = gtk_hbox_new (FALSE, 0);
            gtk_box_pack_start (GTK_BOX (vbox), hbox, TRUE, TRUE, 0);

            //New down button
            image = ui_create_down_icon ();
            gtk_widget_size_request (image, &size);
            _lookup_table_down_button = gtk_button_new ();
            gtk_widget_set_size_request (_lookup_table_down_button, size.width + 4, size.height + 4);
            gtk_container_add (GTK_CONTAINER (_lookup_table_down_button), image);
            gtk_box_pack_end (GTK_BOX (hbox), _lookup_table_down_button, FALSE, FALSE, 0);
            g_signal_connect (G_OBJECT (_lookup_table_down_button), "clicked",
                                G_CALLBACK (ui_lookup_table_down_button_click_cb),
                                image);

            //New up button
            image = ui_create_up_icon ();
            gtk_widget_size_request (image, &size);
            _lookup_table_up_button = gtk_button_new ();
            gtk_widget_set_size_request (_lookup_table_up_button, size.width + 4, size.height + 4);
            gtk_container_add (GTK_CONTAINER (_lookup_table_up_button), image);
            gtk_box_pack_end (GTK_BOX (hbox), _lookup_table_up_button, FALSE, FALSE, 0);
            g_signal_connect (G_OBJECT (_lookup_table_up_button), "clicked",
                                G_CALLBACK (ui_lookup_table_up_button_click_cb),
                                image);
        } else {
            hbox = gtk_hbox_new (FALSE, 0);
            gtk_container_add (GTK_CONTAINER (lookup_table_parent), hbox);

            _lookup_table_items [0] = scim_string_view_new ();
            if (font_desc)
                gtk_widget_modify_font (_lookup_table_items [0], font_desc);
            gtk_widget_modify_base (_lookup_table_items [0], GTK_STATE_NORMAL, &_normal_bg);
            gtk_widget_modify_base (_lookup_table_items [0], GTK_STATE_ACTIVE, &_active_bg);
            gtk_widget_modify_text (_lookup_table_items [0], GTK_STATE_NORMAL, &_normal_text);
            gtk_widget_modify_text (_lookup_table_items [0], GTK_STATE_ACTIVE, &_active_text);
            scim_string_view_set_forward_event (SCIM_STRING_VIEW (_lookup_table_items [0]), TRUE);
            scim_string_view_set_auto_resize (SCIM_STRING_VIEW (_lookup_table_items [0]), TRUE);
            scim_string_view_set_has_frame (SCIM_STRING_VIEW (_lookup_table_items [0]), FALSE);
            scim_string_view_set_draw_cursor (SCIM_STRING_VIEW (_lookup_table_items [0]), FALSE);
            scim_string_view_set_auto_move_cursor (SCIM_STRING_VIEW (_lookup_table_items [0]), FALSE);
            g_signal_connect (G_OBJECT (_lookup_table_items [0]), "move_cursor",
                            G_CALLBACK (ui_lookup_table_horizontal_click_cb),
                            0);
            gtk_box_pack_start (GTK_BOX (hbox), _lookup_table_items [0], TRUE, TRUE, 0);

            separator = gtk_vseparator_new ();
            gtk_box_pack_start (GTK_BOX (hbox), separator, FALSE, FALSE, 0);

            //New left button
            image = ui_create_left_icon ();
            gtk_widget_size_request (image, &size);
            _lookup_table_up_button = gtk_button_new ();
            gtk_widget_set_size_request (_lookup_table_up_button, size.width + 4, size.height + 4);
            gtk_container_add (GTK_CONTAINER (_lookup_table_up_button), image);

            gtk_box_pack_start (GTK_BOX (hbox), _lookup_table_up_button, FALSE, FALSE, 0);
            g_signal_connect (G_OBJECT (_lookup_table_up_button), "clicked",
                                G_CALLBACK (ui_lookup_table_up_button_click_cb),
                                image);

            //New right button
            image = ui_create_right_icon ();
            gtk_widget_size_request (image, &size);
            _lookup_table_down_button = gtk_button_new ();
            gtk_widget_set_size_request (_lookup_table_down_button, size.width + 4, size.height + 4);
            gtk_container_add (GTK_CONTAINER (_lookup_table_down_button), image);

            gtk_box_pack_start (GTK_BOX (hbox), _lookup_table_down_button, FALSE, FALSE, 0);

            g_signal_connect (G_OBJECT (_lookup_table_down_button), "clicked",
                                G_CALLBACK (ui_lookup_table_down_button_click_cb),
                                image);
        }

        gtk_button_set_relief (GTK_BUTTON (_lookup_table_up_button), GTK_RELIEF_NONE);
        gtk_widget_modify_bg (_lookup_table_up_button, GTK_STATE_ACTIVE, &_normal_bg);
        gtk_widget_modify_bg (_lookup_table_up_button, GTK_STATE_INSENSITIVE, &_normal_bg);
        gtk_widget_modify_bg (_lookup_table_up_button, GTK_STATE_PRELIGHT, &_normal_bg);

        gtk_button_set_relief (GTK_BUTTON (_lookup_table_down_button), GTK_RELIEF_NONE);
        gtk_widget_modify_bg (_lookup_table_down_button, GTK_STATE_ACTIVE, &_normal_bg);
        gtk_widget_modify_bg (_lookup_table_down_button, GTK_STATE_INSENSITIVE, &_normal_bg);
        gtk_widget_modify_bg (_lookup_table_down_button, GTK_STATE_PRELIGHT, &_normal_bg);

        if (!_lookup_table_embedded)
            gtk_window_move (GTK_WINDOW (_lookup_table_window), ui_screen_width (), ui_screen_height ());

        gtk_widget_show_all (_lookup_table_window);
        gtk_widget_hide (_lookup_table_window);
    }

    //Create toolbar window
    {
        GtkWidget *hbox;
        GtkWidget *frame;
        GtkWidget *image;
        GtkRequisition size;

        _toolbar_window = gtk_window_new (GTK_WINDOW_POPUP);
        gtk_window_set_policy (GTK_WINDOW (_toolbar_window), TRUE, TRUE, FALSE);
        gtk_window_set_resizable (GTK_WINDOW (_toolbar_window), FALSE);
        gtk_widget_add_events (_toolbar_window,GDK_BUTTON_PRESS_MASK);
        gtk_widget_add_events (_toolbar_window,GDK_BUTTON_RELEASE_MASK);
        gtk_widget_add_events (_toolbar_window,GDK_POINTER_MOTION_MASK);
        g_signal_connect (G_OBJECT (_toolbar_window), "button-press-event",
                          G_CALLBACK (ui_toolbar_window_click_cb),
                          GINT_TO_POINTER (0));
        g_signal_connect (G_OBJECT (_toolbar_window), "button-release-event",
                          G_CALLBACK (ui_toolbar_window_click_cb),
                          GINT_TO_POINTER (1));

        frame = gtk_frame_new (0);
        gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_OUT);
        gtk_container_add (GTK_CONTAINER (_toolbar_window), frame);

        hbox = gtk_hbox_new (FALSE, 0);
        gtk_container_add (GTK_CONTAINER (frame), hbox);

        //New trademark pixmap
        image = ui_create_trademark_icon ();
        gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 0);

        //New stick button
        if (_toolbar_show_stick_icon) {
            image = ui_create_stick_icon (_window_sticked);
            gtk_widget_size_request (image, &size);
            _window_stick_button = gtk_button_new ();
            gtk_button_set_relief (GTK_BUTTON (_window_stick_button), GTK_RELIEF_NONE);
            gtk_container_add (GTK_CONTAINER (_window_stick_button), image);
            gtk_widget_set_size_request (_window_stick_button, size.width + 6, size.height + 6);
            gtk_box_pack_start (GTK_BOX (hbox), _window_stick_button, TRUE, TRUE, 0);
            g_signal_connect (G_OBJECT (_window_stick_button), "clicked",
                              G_CALLBACK (ui_window_stick_button_click_cb),
                              0);
        }

        //New server button
        if (_toolbar_show_server_icon || _toolbar_show_server_name) {
            _server_button = gtk_button_new ();
            gtk_widget_set_size_request (_server_button, 0, 0);
            gtk_button_set_relief (GTK_BUTTON (_server_button), GTK_RELIEF_NONE);
            gtk_box_pack_start (GTK_BOX (hbox), _server_button, TRUE, TRUE, 0);
            g_signal_connect (G_OBJECT (_server_button), "button-release-event",
                              G_CALLBACK (ui_server_button_click_cb),
                              0);
        }

        //New status button.
        _status_button = gtk_button_new ();
        gtk_widget_set_size_request (_status_button, 0, 0);
        gtk_button_set_relief (GTK_BUTTON (_status_button), GTK_RELIEF_NONE);
        gtk_box_pack_start (GTK_BOX (hbox), _status_button, TRUE, TRUE, 0);
        g_signal_connect (G_OBJECT (_status_button), "clicked",
                          G_CALLBACK (ui_status_button_click_cb),
                          0);

        //New full_width_letter button
        image = ui_create_letter_icon (false);
        gtk_widget_size_request (image, &size);
        _full_width_letter_button = gtk_button_new ();
        gtk_button_set_relief (GTK_BUTTON (_full_width_letter_button), GTK_RELIEF_NONE);
        gtk_container_add (GTK_CONTAINER (_full_width_letter_button), image);
        gtk_widget_set_size_request (_full_width_letter_button, size.width + 6, size.height + 6);
        gtk_box_pack_start (GTK_BOX (hbox), _full_width_letter_button, TRUE, TRUE, 0);
        g_signal_connect (G_OBJECT (_full_width_letter_button), "clicked",
                          G_CALLBACK (ui_full_width_letter_button_click_cb),
                          0);

        //New full_width_punctuation button
        image = ui_create_punct_icon (false);
        gtk_widget_size_request (image, &size);
        _full_width_punct_button = gtk_button_new ();
        gtk_button_set_relief (GTK_BUTTON (_full_width_punct_button), GTK_RELIEF_NONE);
        gtk_container_add (GTK_CONTAINER (_full_width_punct_button), image);
        gtk_widget_set_size_request (_full_width_punct_button, size.width + 6, size.height + 6);
        gtk_box_pack_start (GTK_BOX (hbox), _full_width_punct_button, TRUE, TRUE, 0);
        g_signal_connect (G_OBJECT (_full_width_punct_button), "clicked",
                          G_CALLBACK (ui_full_width_punct_button_click_cb),
                          0);

#if HAVE_SCIM_SETUP
        //New setup button
        if (check_scim_setup () && _toolbar_show_setup_icon) {
            image = ui_create_setup_icon ();
            gtk_widget_size_request (image, &size);
            _setup_button = gtk_button_new ();
            gtk_button_set_relief (GTK_BUTTON (_setup_button), GTK_RELIEF_NONE);
            gtk_container_add (GTK_CONTAINER (_setup_button), image);
            gtk_widget_set_size_request (_setup_button, size.width + 6, size.height + 6);
            gtk_box_pack_start (GTK_BOX (hbox), _setup_button, TRUE, TRUE, 0);
            g_signal_connect (G_OBJECT (_setup_button), "clicked",
                              G_CALLBACK (ui_setup_button_click_cb),
                              0);
        }
#endif

        //New help button
        if (_toolbar_show_help_icon) {
            image = ui_create_help_icon ();
            gtk_widget_size_request (image, &size);
            _help_button = gtk_button_new ();
            gtk_button_set_relief (GTK_BUTTON (_help_button), GTK_RELIEF_NONE);
            gtk_container_add (GTK_CONTAINER (_help_button), image);
            gtk_widget_set_size_request (_help_button, size.width + 6, size.height + 6);
            gtk_box_pack_start (GTK_BOX (hbox), _help_button, TRUE, TRUE, 0);
            g_signal_connect (G_OBJECT (_help_button), "clicked",
                              G_CALLBACK (ui_help_button_click_cb),
                              image);
        }

        gtk_window_move (GTK_WINDOW (_toolbar_window), ui_screen_width (), ui_screen_height ());

        gtk_widget_show_all (_toolbar_window);
        gtk_widget_hide (_toolbar_window);

        GtkRequisition ws;
        gtk_widget_size_request (_toolbar_window, &ws);

        if (toolbar_window_x < 0)
            toolbar_window_x = ui_screen_width () - ws.width - 16;
        if (toolbar_window_y < 0)
            toolbar_window_y = ui_screen_height () - ws.height - 16;

        gtk_window_move (GTK_WINDOW (_toolbar_window), toolbar_window_x, toolbar_window_y);
    }

    // Create help window
    {
        GtkWidget *frame;
        GtkWidget *vbox;
        GtkWidget *scroll;

        _help_dialog = gtk_dialog_new_with_buttons (_("SCIM Help"),
                                NULL,
                                GtkDialogFlags (0),
                                GTK_STOCK_OK,
                                GTK_RESPONSE_OK,
                                NULL);

        g_signal_connect_swapped (GTK_OBJECT (_help_dialog), 
                                  "response", 
                                  G_CALLBACK (gtk_widget_hide),
                                  GTK_OBJECT (_help_dialog));

        g_signal_connect_swapped (GTK_OBJECT (_help_dialog), 
                                  "delete_event", 
                                  G_CALLBACK (gtk_widget_hide_on_delete),
                                  GTK_OBJECT (_help_dialog));

        frame = gtk_frame_new (_("Smart Common Input Method"));

        gtk_box_pack_start (GTK_BOX (GTK_DIALOG (_help_dialog)->vbox), frame, TRUE, TRUE, 0);
        gtk_widget_show (frame);

        vbox = gtk_vbox_new (FALSE, 8);
        gtk_container_add (GTK_CONTAINER (frame), vbox);
        gtk_widget_show (vbox);

        _help_scroll = gtk_scrolled_window_new (NULL, NULL);
        gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (_help_scroll), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
        gtk_box_pack_start (GTK_BOX (vbox), _help_scroll, TRUE, TRUE, 0);
        gtk_widget_show (_help_scroll);

        _help_area = gtk_label_new ("");
        gtk_label_set_justify (GTK_LABEL (_help_area), GTK_JUSTIFY_LEFT);
        gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (_help_scroll), _help_area);
        gtk_widget_show (_help_area);
    }

#if ENABLE_TRAY_ICON
    // Create Tray Icon
    {
        if (show_tray_icon)
            ui_create_tray_icon_when_idle (0);
    }
#endif

    //Settle input/lookup windows to default position
    {
        uint32 spot_x, spot_y;

        spot_x = ui_screen_width () / 2 - 64;
        spot_y = ui_screen_height () * 3 / 4;
        gtk_window_move (GTK_WINDOW (_input_window), spot_x, spot_y);

        if (!_lookup_table_embedded)
            gtk_window_move (GTK_WINDOW (_lookup_table_window), spot_x, spot_y + 32);
    }

    //Init timeout callback
    if (_toolbar_always_show && _toolbar_hide_timeout_max > 0) {
        _toolbar_hide_timeout = gtk_timeout_add (1000, ui_hide_window_timeout_cb, NULL);
        g_signal_connect (G_OBJECT (_toolbar_window), "enter-notify-event",
                          G_CALLBACK (ui_toolbar_window_crossing_cb),
                          GINT_TO_POINTER (0));
        g_signal_connect (G_OBJECT (_toolbar_window), "leave-notify-event",
                          G_CALLBACK (ui_toolbar_window_crossing_cb),
                          GINT_TO_POINTER (1));
    }

    // Init the tooltips
    {
        _tooltips = gtk_tooltips_new ();

        gtk_tooltips_set_delay (_tooltips, 1000);

        if (_window_stick_button)
            gtk_tooltips_set_tip (_tooltips, _window_stick_button,
                                  _("Stick/unstick the input window and the toolbar."),
                                  NULL);

        gtk_tooltips_set_tip (_tooltips, _status_button,
                              _("The status of the current input method. Click to change it."),
                              NULL);

        gtk_tooltips_set_tip (_tooltips, _full_width_punct_button,
                              _("The input mode of the puncutations. Click to toggle between half and full."),
                              NULL);

        gtk_tooltips_set_tip (_tooltips, _full_width_letter_button,
                              _("The input mode of the letters. Click to toggle between half and full."),
                              NULL);

#if HAVE_SCIM_SETUP
        if (_setup_button)
            gtk_tooltips_set_tip (_tooltips, _setup_button,
                                  _("Start the SCIM Setup Utility."),
                                  NULL);
#endif

        if (_help_button)
            gtk_tooltips_set_tip (_tooltips, _help_button,
                                  _("Show a brief help about SCIM and the current input method."),
                                  NULL);
    }

    if (font_desc)
        pango_font_description_free (font_desc);

    _ui_initialized = true;
}

static void
ui_settle_input_window (bool relative)
{
    SCIM_DEBUG_MAIN (2) << " Settle input window...\n";

    if (_window_sticked)
        return;

    GtkRequisition ws;
    gint old_x, old_y;
    gint spot_x, spot_y;

    gtk_widget_size_request (_input_window, &ws);
    gtk_window_get_position (GTK_WINDOW (_input_window), &old_x, &old_y);

    if (!relative) {
        spot_x = _spot_location_x; 
        spot_y = _spot_location_y; 
    } else {
        spot_x = old_x; 
        spot_y = old_y;
    }

    if (spot_x < 0) spot_x = 0;
    if (spot_y < 0) spot_y = 0;

    if (spot_x + ws.width > ui_screen_width () - 4)
        spot_x = ui_screen_width () - ws.width - 4;
    if (spot_y + ws.height + 8 > ui_screen_height () - 4)
        spot_y = ui_screen_height () - ws.height - 4;

    if (old_x != spot_x || old_y != spot_y)
        gtk_window_move (GTK_WINDOW (_input_window), spot_x, spot_y);
}

static void
ui_settle_lookup_table_window(bool relative)
{
    SCIM_DEBUG_MAIN (2) << " Settle lookup table window...\n";

    if (_lookup_table_embedded)
        return ui_settle_input_window (relative);

    if (_window_sticked)
        return;

    gint input_x, input_y;
    gint pos_x, pos_y;
    gint old_x, old_y;

    GtkRequisition iws;
    GtkRequisition ws;

    gtk_widget_size_request (_input_window, &iws);
    gtk_widget_size_request (_lookup_table_window, &ws);
    gtk_window_get_position (GTK_WINDOW (_input_window), &input_x, &input_y);
    gtk_window_get_position (GTK_WINDOW (_lookup_table_window), &old_x, &old_y);

    if (!relative) {
        pos_x = input_x + 8;
        pos_y = input_y + iws.height + 8;
    } else {
        pos_x = old_x;
        pos_y = old_y;
    }

    if (pos_x + ws.width > ui_screen_width () - 4) {
        pos_x = ui_screen_width () - ws.width - 4;
    }

    if (pos_y + ws.height > ui_screen_height () - 4) {
        pos_y = ui_screen_height () - ws.height - 4;
    }

    if (old_x != pos_x || old_y != pos_y)
        gtk_window_move (GTK_WINDOW (_lookup_table_window), pos_x, pos_y);
}

static void
ui_settle_toolbar_window (void)
{
    SCIM_DEBUG_MAIN (2) << " Settle toolbar window...\n";

    if (_window_sticked) 
        return;

    gint old_x, old_y;
    gint pos_x, pos_y;

    GtkRequisition ws;

    gtk_widget_size_request (_toolbar_window, &ws);
    gtk_window_get_position (GTK_WINDOW (_toolbar_window), &old_x, &old_y);

    pos_x = old_x;
    pos_y = old_y;

    if (_toolbar_auto_snap) {
        if ((ui_screen_width () - (pos_x + ws.width)) < pos_x)
            pos_x = ui_screen_width () - ws.width;
        else
            pos_x = 0;
    } else if (pos_x + ws.width > ui_screen_width ()) {
        pos_x = ui_screen_width () - ws.width;
    } else if (pos_x < 0) {
        pos_x = 0;
    }

    if (pos_y + ws.height > ui_screen_height ())
        pos_y = ui_screen_height () - ws.height;
    else if (pos_y < 0)
        pos_y = 0;

    if (old_x != pos_x || old_y != pos_y)
        gtk_window_move (GTK_WINDOW (_toolbar_window), pos_x, pos_y);
}

static int
ui_screen_width (void)
{
#if GDK_MULTIHEAD_SAFE
    if (_current_screen)
        return gdk_screen_get_width (_current_screen);
#endif
    return gdk_screen_width ();
}

static int
ui_screen_height (void)
{
#if GDK_MULTIHEAD_SAFE
    if (_current_screen)
        return gdk_screen_get_height (_current_screen);
#endif
    return gdk_screen_height ();
}

#if GDK_MULTIHEAD_SAFE
static void
ui_switch_screen (GdkScreen *screen)
{
    if (screen) {
        if (_input_window) {
            gtk_window_set_screen (GTK_WINDOW (_input_window), screen);
            gtk_window_move (GTK_WINDOW (_input_window),
                             ui_screen_width (),
                             ui_screen_height ());
        }

        if (_toolbar_window) {
            GtkRequisition ws;
            gtk_widget_size_request (_toolbar_window, &ws);

            gtk_window_set_screen (GTK_WINDOW (_toolbar_window), screen);
            gtk_window_move (GTK_WINDOW (_toolbar_window),
                             ui_screen_width () - ws.width - 16,
                             ui_screen_height () - ws.height - 16);
        }

        if (!_lookup_table_embedded && _lookup_table_window) {
            gtk_window_set_screen (GTK_WINDOW (_lookup_table_window), screen);
            gtk_window_move (GTK_WINDOW (_lookup_table_window),
                             ui_screen_width (),
                             ui_screen_height ());
        }

#if ENABLE_TRAY_ICON
        if (_tray_icon) {
            gtk_window_set_screen (GTK_WINDOW (_tray_icon), screen);
        }
#endif

        ui_settle_input_window ();
        ui_settle_lookup_table_window ();
        ui_settle_toolbar_window ();
    }
}
#endif

#if ENABLE_TRAY_ICON
static gboolean
ui_create_tray_icon_when_idle (gpointer data)
{
    GtkWidget *image;

    _tray_icon = scim_tray_icon_new ("SCIM Tray Icon");

    g_signal_connect (G_OBJECT (_tray_icon), "destroy",
                      G_CALLBACK (ui_tray_icon_destroy_cb),
                      0);

    image = ui_create_icon_from_file (SCIM_KEYBOARD_ICON_FILE,
                                      TRAY_ICON_SIZE,
                                      TRAY_ICON_SIZE,
                                      true);

    _tray_icon_server_button = gtk_button_new ();
    gtk_container_add (GTK_CONTAINER (_tray_icon_server_button), image);
    gtk_button_set_relief (GTK_BUTTON (_tray_icon_server_button), GTK_RELIEF_NONE);
    gtk_container_add (GTK_CONTAINER (_tray_icon), _tray_icon_server_button);
    g_signal_connect (G_OBJECT (_tray_icon_server_button), "button-release-event",
                      G_CALLBACK (ui_server_button_click_cb),
                      0);

    gtk_widget_show_all (GTK_WIDGET (_tray_icon));

    g_object_ref (G_OBJECT (_tray_icon));

    return FALSE;
}
#endif

static GdkPixbuf *
ui_scale_pixbuf (GdkPixbuf *pixbuf,
                 int        width,
                 int        height)
{
    if (pixbuf) {
        if (gdk_pixbuf_get_width (pixbuf) != width ||
            gdk_pixbuf_get_height (pixbuf) != height) {
            GdkPixbuf *dest = gdk_pixbuf_scale_simple (pixbuf, width, height, GDK_INTERP_BILINEAR);
            gdk_pixbuf_unref (pixbuf);
            pixbuf = dest;
        }
    }
    return pixbuf;
}

static GtkWidget *
ui_create_label (const String   &name,
                 const String   &iconfile,
                 const char    **xpm,
                 bool            show_icon_only,
                 bool            force_icon)
{
    GtkWidget * hbox = gtk_hbox_new (FALSE, 0);
    GtkWidget * label = gtk_label_new (name.c_str ());

    GtkRequisition iconsize;

    gtk_widget_size_request (label, &iconsize);

    iconsize.height += 4;
            
    if (iconsize.height < 16)
        iconsize.height = 16;

    iconsize.width = iconsize.height;

    GtkWidget *icon = ui_create_icon (iconfile,
                                      xpm,
                                      iconsize.width,
                                      iconsize.height,
                                      force_icon);

    if (icon) {
        gtk_box_pack_start (GTK_BOX (hbox), icon, FALSE, FALSE, 0);
        if (!show_icon_only)
            gtk_box_set_spacing (GTK_BOX (hbox), 4);
    }

    if (!show_icon_only || !icon)
        gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    else
        gtk_widget_unref (label);

    gtk_widget_show_all (hbox);

    return hbox;
}

static GtkWidget *
ui_create_icon_from_file (const String   &iconfile,
                          int             width,
                          int             height,
                          bool            force_create)
{
    String path = iconfile;

    // Not a absolute path, prepend SCIM_ICONDIR
    if (path [0] != SCIM_PATH_DELIM)
        path = String (SCIM_ICONDIR) + String (SCIM_PATH_DELIM_STRING) + path;

    GdkPixbuf *pixbuf = gdk_pixbuf_new_from_file (path.c_str (), 0);

    if (!pixbuf && !force_create)
         return 0;

    if (!pixbuf) {
        if (width <= 0 || height <= 0)
            return 0;

        pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, true, 8, width, height);

        if (!pixbuf)
            return 0;

        gdk_pixbuf_fill (pixbuf, 0);
    }

    if (width <= 0) width = gdk_pixbuf_get_width (pixbuf);
    if (height <= 0) height = gdk_pixbuf_get_height (pixbuf);

    pixbuf = ui_scale_pixbuf (pixbuf, width, height);

    GtkWidget *icon = gtk_image_new_from_pixbuf (pixbuf);
    gtk_widget_show (icon);

    gdk_pixbuf_unref (pixbuf);

    return icon;
}

static GtkWidget *
ui_create_icon_from_xpm (const char **xpm,
                         int          width,
                         int          height,
                         bool         force_create)
{
    GdkPixbuf *pixbuf = gdk_pixbuf_new_from_xpm_data (xpm);

    if (!pixbuf && !force_create)
         return 0;

    if (!pixbuf) {
        if (width <= 0 || height <= 0)
            return 0;

        pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, true, 8, width, height);

        if (!pixbuf)
            return 0;

        gdk_pixbuf_fill (pixbuf, 0);
    }

    if (width <= 0) width = gdk_pixbuf_get_width (pixbuf);
    if (height <= 0) height = gdk_pixbuf_get_height (pixbuf);

    pixbuf = ui_scale_pixbuf (pixbuf, width, height);

    GtkWidget *icon = gtk_image_new_from_pixbuf (pixbuf);
    gtk_widget_show (icon);

    gdk_pixbuf_unref (pixbuf);

    return icon;
}

static GtkWidget *
ui_create_icon (const String  &iconfile,
                const char   **xpm,
                int            width,
                int            height,
                bool           force_create)
{
    GtkWidget *icon = 0;

    if (iconfile.length ())
        icon = ui_create_icon_from_file (iconfile, width, height, !xpm);

    if (!icon && xpm)
        icon = ui_create_icon_from_xpm (xpm, width, height, force_create);

    return icon;
}

static GtkWidget *
ui_create_trademark_icon (void)
{
    return ui_create_icon (SCIM_TRADEMARK_ICON_FILE,
                           (const char **) trademark_xpm,
                           TOOLBAR_ICON_SIZE + 4,
                           TOOLBAR_ICON_SIZE + 4);
}

static GtkWidget *
ui_create_stick_icon (bool sticked)
{
    return ui_create_icon ((sticked ? SCIM_PIN_DOWN_ICON_FILE : SCIM_PIN_UP_ICON_FILE),
                           (const char **) (sticked ? pin_down_xpm : pin_up_xpm),
                           TOOLBAR_ICON_SIZE,
                           TOOLBAR_ICON_SIZE);
}

static GtkWidget *
ui_create_punct_icon (bool full)
{
    return ui_create_icon ((full ? SCIM_FULL_PUNCT_ICON_FILE : SCIM_HALF_PUNCT_ICON_FILE),
                           (const char **) (full ? full_punct_xpm : half_punct_xpm),
                           TOOLBAR_ICON_SIZE,
                           TOOLBAR_ICON_SIZE);
}

static GtkWidget *
ui_create_letter_icon (bool full)
{
    return ui_create_icon ((full ? SCIM_FULL_LETTER_ICON_FILE : SCIM_HALF_LETTER_ICON_FILE),
                           (const char **) (full ? full_letter_xpm : half_letter_xpm),
                           TOOLBAR_ICON_SIZE,
                           TOOLBAR_ICON_SIZE);
}

#if HAVE_SCIM_SETUP
static GtkWidget *
ui_create_setup_icon (void)
{
    return ui_create_icon (SCIM_SETUP_ICON_FILE,
                           (const char **) setup_xpm,
                           TOOLBAR_ICON_SIZE,
                           TOOLBAR_ICON_SIZE);
}
#endif

static GtkWidget *
ui_create_help_icon (void)
{
    return ui_create_icon (SCIM_HELP_ICON_FILE,
                           (const char **) help_xpm,
                           TOOLBAR_ICON_SIZE,
                           TOOLBAR_ICON_SIZE);
}

static GtkWidget *
ui_create_up_icon (void)
{
    return ui_create_icon (SCIM_UP_ICON_FILE,
                           (const char **) up_xpm,
                           TOOLBAR_ICON_SIZE,
                           TOOLBAR_ICON_SIZE);
}

static GtkWidget *
ui_create_left_icon (void)
{
    return ui_create_icon (SCIM_LEFT_ICON_FILE,
                           (const char **) left_xpm,
                           TOOLBAR_ICON_SIZE,
                           TOOLBAR_ICON_SIZE);
}

static GtkWidget *
ui_create_right_icon (void)
{
    return ui_create_icon (SCIM_RIGHT_ICON_FILE,
                           (const char **) right_xpm,
                           TOOLBAR_ICON_SIZE,
                           TOOLBAR_ICON_SIZE);
}

static GtkWidget *
ui_create_down_icon (void)
{
    return ui_create_icon (SCIM_DOWN_ICON_FILE,
                           (const char **) down_xpm,
                           TOOLBAR_ICON_SIZE,
                           TOOLBAR_ICON_SIZE);
}

static GtkWidget *
ui_create_stick_command_label (bool sticked)
{
    return ui_create_label (
                    (sticked ? String (_("Unstick Windows")) : String (_("Stick Windows"))),
                    (sticked ? SCIM_PIN_DOWN_ICON_FILE : SCIM_PIN_UP_ICON_FILE),
                    (const char **)(sticked ? pin_down_xpm : pin_up_xpm),
                    false,
                    true);
}

#if HAVE_SCIM_SETUP
static GtkWidget *
ui_create_setup_command_label (void)
{
    return ui_create_label (
                    String (_("Setup ...")),
                    SCIM_SETUP_ICON_FILE,
                    (const char **) setup_xpm,
                    false,
                    true);
}
#endif

static GtkWidget *
ui_create_help_command_label (void)
{
    return ui_create_label (
                    String (_("Help ...")),
                    SCIM_HELP_ICON_FILE,
                    (const char **) help_xpm,
                    false,
                    true);
}

/* Implementation of callback functions */
static void
ui_preedit_area_move_cursor_cb (ScimStringView *view,
                                guint           position)
{
    SCIM_DEBUG_MAIN (3) << "  ui_preedit_area_move_cursor_cb...\n";

    action_move_preedit_caret (position);
}

static void
ui_full_width_punct_button_click_cb (GtkButton *button,
                                     gpointer user_data)
{
    SCIM_DEBUG_MAIN (3) << "  ui_full_width_punct_button_click_cb...\n";

    action_toggle_full_width_punct ();
}

static void
ui_full_width_letter_button_click_cb (GtkButton *button,
                                      gpointer user_data)
{
    SCIM_DEBUG_MAIN (3) << "  ui_full_width_letter_button_click_cb...\n";

    action_toggle_full_width_letter ();
}

static void
ui_status_button_click_cb (GtkButton *button,
                           gpointer user_data)
{
    SCIM_DEBUG_MAIN (3) << "  ui_status_button_click_cb...\n";

    action_toggle_input_status ();
}

#if HAVE_SCIM_SETUP
static void
ui_setup_button_click_cb (GtkButton *button,
                          gpointer   user_data)
{
    action_launch_setup_tool ();
}
#endif

static void
ui_help_button_click_cb (GtkButton *button,
                         gpointer   user_data)
{
    SCIM_DEBUG_MAIN (3) << "  ui_help_button_click_cb...\n";

    if (GTK_WIDGET_VISIBLE (_help_dialog)) {
        gtk_widget_hide (_help_dialog);
    } else {
        action_request_help ();
    }
}

static gboolean
ui_server_button_click_cb (GtkWidget *button,
                           GdkEvent  *event,
                           gpointer   user_data)
{
    SCIM_DEBUG_MAIN (3) << "  ui_server_button_click_cb...\n";

    GdkEventButton *bevent = (GdkEventButton *) event;

    if (bevent->button <= 1)
        action_request_server_menu ();
    else
        action_show_command_menu ();

    return FALSE;
}

static void
ui_server_menu_activate_cb (GtkMenuItem *item,
                            gpointer     user_data)
{
    int id = GPOINTER_TO_INT (user_data);

    if (id >= 0 && id < _server_menu_uuids.size ())
        action_change_server_factory (_server_menu_uuids [id]);
    else
        action_change_server_factory (String (""));
}

static void
ui_server_menu_deactivate_cb (GtkMenuItem *item,
                              gpointer     user_data)
{
    _server_menu_activated = false;
}

static gboolean
ui_lookup_table_vertical_click_cb (GtkWidget      *item,
                                   GdkEventButton *event,
                                   gpointer        user_data)
{
    SCIM_DEBUG_MAIN (3) << "  ui_lookup_table_vertical_click_cb...\n";

    action_select_lookup_table ((uint32)GPOINTER_TO_INT (user_data));

    return TRUE;
}

static void
ui_lookup_table_horizontal_click_cb (GtkWidget *item,
                                     guint      position)
{
    SCIM_DEBUG_MAIN (3) << "  ui_lookup_table_horizontal_click_cb...\n";

    int *index = _lookup_table_index;
    int pos = (int) position;

    for (int i=0; i<SCIM_LOOKUP_TABLE_MAX_PAGESIZE && index [i] >= 0; ++i) {
        if (pos >= index [i] && pos < index [i+1]) {
            action_select_lookup_table ((uint32) i);
            return;
        }
    }
}

static void
ui_lookup_table_up_button_click_cb (GtkButton *button,
                                    gpointer user_data)
{
    SCIM_DEBUG_MAIN (3) << "  ui_lookup_table_up_button_click_cb...\n";

    action_lookup_table_page_up ();
}

static void
ui_lookup_table_down_button_click_cb (GtkButton *button,
                                      gpointer user_data)
{
    SCIM_DEBUG_MAIN (3) << "  ui_lookup_table_down_button_click_cb...\n";

    action_lookup_table_page_down ();
}

static void
ui_window_stick_button_click_cb (GtkButton *button,
                                 gpointer user_data)
{
    action_toggle_window_stick ();
}

static gboolean
ui_input_window_motion_cb (GtkWidget *window,
                           GdkEventMotion *event,
                           gpointer user_data)
{
    gint pos_x, pos_y;

    if ((event->state & (GDK_BUTTON1_MASK | GDK_BUTTON2_MASK | GDK_BUTTON3_MASK)) != 0 &&
        _input_window_draging) {
        gtk_window_get_position (GTK_WINDOW (window), &pos_x, &pos_y);
        gtk_window_move (GTK_WINDOW (window), 
            pos_x + ((gint) event->x_root - _input_window_drag_x),
            pos_y + ((gint) event->y_root - _input_window_drag_y));

        _input_window_drag_x = (gint) event->x_root;
        _input_window_drag_y = (gint) event->y_root;

        return TRUE;
    }
    return FALSE;
}

static gboolean
ui_input_window_click_cb (GtkWidget *window,
                          GdkEventButton *event,
                          gpointer user_data)
{
    int click_type = GPOINTER_TO_INT (user_data);
    static gulong motion_handler;
    GdkCursor *cursor;

    if (click_type == 0) {
        if (_input_window_draging)
            return FALSE;

        // Connection pointer motion handler to this window.
        motion_handler = g_signal_connect (G_OBJECT (window), "motion-notify-event",
                                           G_CALLBACK (ui_input_window_motion_cb),
                                           NULL);

        _input_window_draging = TRUE;
        _input_window_drag_x = (gint) event->x_root;
        _input_window_drag_y = (gint) event->y_root;

        cursor = gdk_cursor_new (GDK_TOP_LEFT_ARROW);
    
        // Grab the cursor to prevent losing events.
        gdk_pointer_grab (window->window, TRUE,
                          (GdkEventMask) (GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK),
                          NULL, cursor, event->time);
        gdk_cursor_unref (cursor);
        return TRUE;
    } else if (click_type == 1) {
        if (!_input_window_draging)
            return FALSE;

        g_signal_handler_disconnect (G_OBJECT (window), motion_handler);
        gdk_pointer_ungrab (event->time);
        _input_window_draging = FALSE;
        return TRUE;
    }

    return FALSE;
}

static gboolean
ui_toolbar_window_crossing_cb (GtkWidget        *window,
                               GdkEventCrossing *event,
                               gpointer          user_data)
{
    if (!_toolbar_always_show || _panel_is_on || _toolbar_window_draging)
        return FALSE;

    int crossing_type = GPOINTER_TO_INT (user_data);

    // 0 == enter, otherwise leave 
    if (crossing_type == 0) {
        if (_toolbar_hidden) {
            if (_window_stick_button)
                gtk_widget_show (_window_stick_button);

            if (_server_button)
                gtk_widget_show (_server_button);

#if HAVE_SCIM_SETUP
            if (_setup_button)
                gtk_widget_show (_setup_button);
#endif

            if (_help_button)
                gtk_widget_show (_help_button);

            _toolbar_hidden = false;
            ui_settle_toolbar_window ();
        }
        _toolbar_should_hide = false;
    } else {
        _toolbar_should_hide = true;
    }

    return FALSE;
}

static gboolean
ui_toolbar_window_motion_cb (GtkWidget *window,
                             GdkEventMotion *event,
                             gpointer user_data)
{
    gint pos_x, pos_y;
    if ((event->state & (GDK_BUTTON1_MASK | GDK_BUTTON2_MASK | GDK_BUTTON3_MASK)) != 0 &&
        _toolbar_window_draging) {
        gtk_window_get_position (GTK_WINDOW (window), &pos_x, &pos_y);
        gtk_window_move (GTK_WINDOW (window), 
            pos_x + ((gint) event->x_root - _toolbar_window_drag_x),
            pos_y + ((gint) event->y_root - _toolbar_window_drag_y));

        _toolbar_window_drag_x = (gint) event->x_root;
        _toolbar_window_drag_y = (gint) event->y_root;

        return TRUE;
    }
    return FALSE;
}

static gboolean
ui_toolbar_window_click_cb (GtkWidget *window,
                            GdkEventButton *event,
                            gpointer user_data)
{
    int click_type = GPOINTER_TO_INT (user_data);
    static gulong motion_handler;
    GdkCursor *cursor;

    if (click_type == 0 && event->button <= 1) {
        if (_toolbar_window_draging)
            return FALSE;

        // Connection pointer motion handler to this window.
        motion_handler = g_signal_connect (G_OBJECT (window), "motion-notify-event",
                                           G_CALLBACK (ui_toolbar_window_motion_cb),
                                           NULL);

        _toolbar_window_draging = TRUE;
        _toolbar_window_drag_x = (gint) event->x_root;
        _toolbar_window_drag_y = (gint) event->y_root;

        cursor = gdk_cursor_new (GDK_TOP_LEFT_ARROW);

        // Grab the cursor to prevent losing events.
        gdk_pointer_grab (window->window, TRUE,
                          (GdkEventMask) (GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK),
                          NULL, cursor, event->time);
        gdk_cursor_unref (cursor);
        return TRUE;
    } else if (click_type == 1 && event->button <= 1) {
        if (!_toolbar_window_draging)
            return FALSE;

        g_signal_handler_disconnect (G_OBJECT (window), motion_handler);
        gdk_pointer_ungrab (event->time);
        _toolbar_window_draging = FALSE;

        gint pos_x, pos_y;

        gtk_window_get_position (GTK_WINDOW (window), &pos_x, &pos_y);

        if (!_config.null ()) {
            _config->write (
                SCIM_CONFIG_PANEL_GTK_TOOLBAR_POS_X, pos_x);

            _config->write (
                SCIM_CONFIG_PANEL_GTK_TOOLBAR_POS_Y, pos_y);
        }
        return TRUE;
    } else if (click_type == 1 && event->button > 1) {
        action_show_command_menu ();
        return TRUE;
    }
    return FALSE;
}

static gboolean
ui_lookup_table_window_motion_cb (GtkWidget      *window,
                                  GdkEventMotion *event,
                                  gpointer       user_data)
{
    gint pos_x, pos_y;
    if ((event->state & (GDK_BUTTON1_MASK | GDK_BUTTON2_MASK | GDK_BUTTON3_MASK)) != 0 &&
        _lookup_table_window_draging) {
        gtk_window_get_position (GTK_WINDOW (window), &pos_x, &pos_y);
        gtk_window_move (GTK_WINDOW (window), 
            pos_x + ((gint) event->x_root - _lookup_table_window_drag_x),
            pos_y + ((gint) event->y_root - _lookup_table_window_drag_y));

        _lookup_table_window_drag_x = (gint) event->x_root;
        _lookup_table_window_drag_y = (gint) event->y_root;

        return TRUE;
    }
    return FALSE;
}

static gboolean
ui_lookup_table_window_click_cb (GtkWidget *window,
                                 GdkEventButton *event,
                                 gpointer user_data)
{
    int click_type = GPOINTER_TO_INT (user_data);
    static gulong motion_handler;
    GdkCursor *cursor;

    if (click_type == 0) {
        if (_lookup_table_window_draging)
            return FALSE;

        // Connection pointer motion handler to this window.
        motion_handler = g_signal_connect (G_OBJECT (window), "motion-notify-event",
                                           G_CALLBACK (ui_lookup_table_window_motion_cb),
                                           NULL);

        _lookup_table_window_draging = TRUE;
        _lookup_table_window_drag_x = (gint) event->x_root;
        _lookup_table_window_drag_y = (gint) event->y_root;

        cursor = gdk_cursor_new (GDK_TOP_LEFT_ARROW);
    
        // Grab the cursor to prevent losing events.
        gdk_pointer_grab (window->window, TRUE,
                          (GdkEventMask) (GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK),
                          NULL, cursor, event->time);
        gdk_cursor_unref (cursor);
        return TRUE;
    } else if (click_type == 1) {
        if (!_lookup_table_window_draging)
            return FALSE;

        g_signal_handler_disconnect (G_OBJECT (window), motion_handler);
        gdk_pointer_ungrab (event->time);
        _lookup_table_window_draging = FALSE;
        return TRUE;
    }

    return FALSE;
}

static gboolean
ui_hide_window_timeout_cb (gpointer data)
{
    gdk_threads_enter ();

    if (!_toolbar_always_show) {
        gdk_threads_leave ();
        return TRUE;
    }

    if (!_toolbar_should_hide || _panel_is_on || _server_menu_activated ||
        _toolbar_window_draging || _toolbar_hidden) {
        _toolbar_hide_timeout_count = 0;
        gdk_threads_leave ();
        return TRUE;
    }

    _toolbar_hide_timeout_count ++;

    if (_toolbar_hide_timeout_count > _toolbar_hide_timeout_max) {
        _toolbar_hide_timeout_count = 0;

        if (_help_button)
            gtk_widget_hide (_help_button);

#if HAVE_SCIM_SETUP
        if (_setup_button)
            gtk_widget_hide (_setup_button);
#endif

        gtk_widget_hide (_status_button);

        if (_server_button)
            gtk_widget_hide (_server_button);

        if (_window_stick_button)
            gtk_widget_hide (_window_stick_button);

        _toolbar_hidden = true;
        ui_settle_toolbar_window ();
    }

    gdk_threads_leave ();
    return TRUE;
}

static bool
ui_can_hide_input_window (void)
{
    if (!_panel_is_on) return true;

    if (GTK_WIDGET_VISIBLE (_preedit_area) ||
        GTK_WIDGET_VISIBLE (_aux_area) ||
        (_status_area && GTK_WIDGET_VISIBLE (_status_area)) ||
        (_lookup_table_embedded && GTK_WIDGET_VISIBLE (_lookup_table_window)))
        return false;
    return true;
}

static PangoAttrList *
create_pango_attrlist (const String        &mbs,
                       const AttributeList &attrs)
{
    PangoAttrList  *attrlist = pango_attr_list_new ();
    PangoAttribute *attr;

    guint start_index, end_index;
    guint wlen = g_utf8_strlen (mbs.c_str (), mbs.length ());

    for (int i=0; i<attrs.size (); ++i) {
        start_index = attrs[i].get_start ();
        end_index = attrs[i].get_end ();

        if (end_index <= wlen && start_index < end_index) {
            start_index = g_utf8_offset_to_pointer (mbs.c_str (), attrs[i].get_start ()) - mbs.c_str ();
            end_index = g_utf8_offset_to_pointer (mbs.c_str (), attrs[i].get_end ()) - mbs.c_str ();

            switch (attrs[i].get_type ()) {
                case SCIM_ATTR_UNDERLINE:
                    attr = pango_attr_underline_new (PANGO_UNDERLINE_DOUBLE);
                    attr->start_index = start_index;
                    attr->end_index = end_index;
                    pango_attr_list_insert (attrlist, attr);
                    break;
                case SCIM_ATTR_REVERSE:
                    attr = pango_attr_foreground_new (_normal_bg.red, _normal_bg.green, _normal_bg.blue);
                    attr->start_index = start_index;
                    attr->end_index = end_index;
                    pango_attr_list_insert (attrlist, attr);

                    attr = pango_attr_background_new (_normal_text.red, _normal_text.green, _normal_text.blue);
                    attr->start_index = start_index;
                    attr->end_index = end_index;
                    pango_attr_list_insert (attrlist, attr);
                    break;
                case SCIM_ATTR_HIGHLIGHT:
                    attr = pango_attr_foreground_new (_active_text.red, _active_text.green, _active_text.blue);
                    attr->start_index = start_index;
                    attr->end_index = end_index;
                    pango_attr_list_insert (attrlist, attr);

                    attr = pango_attr_background_new (_active_bg.red, _active_bg.green, _active_bg.blue);
                    attr->start_index = start_index;
                    attr->end_index = end_index;
                    pango_attr_list_insert (attrlist, attr);
                    break;
            }
        }
    }
    return attrlist;
}

static void
ui_command_menu_stick_activate_cb (GtkMenuItem *item,
                                   gpointer     user_data)
{
    action_toggle_window_stick ();
}

#if HAVE_SCIM_SETUP
static void
ui_command_menu_setup_activate_cb (GtkMenuItem *item,
                                   gpointer     user_data)
{
    action_launch_setup_tool ();
}
#endif

static void
ui_command_menu_help_activate_cb (GtkMenuItem *item,
                                  gpointer     user_data)
{
    if (GTK_WIDGET_VISIBLE (_help_dialog)) {
        gtk_widget_hide (_help_dialog);
    } else {
        action_request_help ();
    }
}

static void
ui_command_menu_deactivate_cb (GtkMenuItem *item,
                               gpointer     user_data)
{
    _command_menu_activated = false;
}

#if ENABLE_TRAY_ICON
static void
ui_tray_icon_destroy_cb (GtkObject      *object,
                         gpointer        user_data)
{
    SCIM_DEBUG_MAIN (1) << "Tray Icon destroyed!\n";

    g_object_unref (_tray_icon);

    _tray_icon = 0;
    _tray_icon_server_button = 0;

    g_idle_add (ui_create_tray_icon_when_idle, NULL);
}
#endif

//Implementation of the action functions
static void
action_move_preedit_caret (uint32 position)
{
    int client;
    uint32 context;

    if (_current_socket_client >= 0) {
        client = _current_socket_client;
        context = _current_client_context;
    } else {
        client = _last_socket_client;
        context = _last_client_context;
    }

    if (client >= 0) {
        _send_transaction.clear ();
        _send_transaction.put_command (SCIM_TRANS_CMD_REPLY);
        _send_transaction.put_data ((uint32) context);
        _send_transaction.put_command (SCIM_TRANS_CMD_MOVE_PREEDIT_CARET);
        _send_transaction.put_data ((uint32) position);
        _send_transaction.write_to_socket (Socket (client));
    }
}

static void
action_toggle_full_width_punct (void)
{
    int client;
    uint32 context;

    if (_current_socket_client >= 0) {
        client = _current_socket_client;
        context = _current_client_context;
    } else {
        client = _last_socket_client;
        context = _last_client_context;
    }

    if (client >= 0) {
        _send_transaction.clear ();
        _send_transaction.put_command (SCIM_TRANS_CMD_REPLY);
        _send_transaction.put_data ((uint32) context);
        _send_transaction.put_command (SCIM_TRANS_CMD_TOGGLE_FULL_WIDTH_PUNCTUATION);
        _send_transaction.write_to_socket (Socket (client));
    }
}

static void
action_toggle_full_width_letter (void)
{
    int client;
    uint32 context;

    if (_current_socket_client >= 0) {
        client = _current_socket_client;
        context = _current_client_context;
    } else {
        client = _last_socket_client;
        context = _last_client_context;
    }

    if (client >= 0) {
        _send_transaction.clear ();
        _send_transaction.put_command (SCIM_TRANS_CMD_REPLY);
        _send_transaction.put_data ((uint32) context);
        _send_transaction.put_command (SCIM_TRANS_CMD_TOGGLE_FULL_WIDTH_LETTER);
        _send_transaction.write_to_socket (Socket (client));
    }
}

#if HAVE_SCIM_SETUP
static void
action_launch_setup_tool (void)
{
    G_LOCK (_global_resource_lock);
    run_scim_setup ();
    G_UNLOCK (_global_resource_lock);
}
#endif

static void
action_request_help (void)
{
    int client;
    uint32 context;

    if (_current_socket_client >= 0) {
        client = _current_socket_client;
        context = _current_client_context;
    } else {
        client = _last_socket_client;
        context = _last_client_context;
    }

    if (client >= 0) {
        _send_transaction.clear ();
        _send_transaction.put_command (SCIM_TRANS_CMD_REPLY);
        _send_transaction.put_data ((uint32) context);
        _send_transaction.put_command (SCIM_TRANS_CMD_PANEL_REQUEST_HELP);
        _send_transaction.write_to_socket (Socket (client));
    }
}

static void
action_request_server_menu (void)
{
    int client;
    uint32 context;

    if (_current_socket_client >= 0) {
        client = _current_socket_client;
        context = _current_client_context;
    } else {
        client = _last_socket_client;
        context = _last_client_context;
    }

    if (client >= 0) {
        _send_transaction.clear ();
        _send_transaction.put_command (SCIM_TRANS_CMD_REPLY);
        _send_transaction.put_data ((uint32) context);
        _send_transaction.put_command (SCIM_TRANS_CMD_PANEL_REQUEST_SERVER_MENU);
        _send_transaction.write_to_socket (Socket (client));
    }
}

static void
action_change_server_factory (const String &uuid)
{
    int client;
    uint32 context;

    if (_current_socket_client >= 0) {
        client = _current_socket_client;
        context = _current_client_context;
    } else {
        client = _last_socket_client;
        context = _last_client_context;
    }

    if (client >= 0) {
        _send_transaction.clear ();
        _send_transaction.put_command (SCIM_TRANS_CMD_REPLY);
        _send_transaction.put_data ((uint32) context);
        _send_transaction.put_command (SCIM_TRANS_CMD_PANEL_CHANGE_SERVER_FACTORY);
        _send_transaction.put_data (uuid);
        _send_transaction.write_to_socket (Socket (client));
    }
}

static void
action_select_lookup_table (uint32 item)
{
    int client;
    uint32 context;

    if (_current_socket_client >= 0) {
        client = _current_socket_client;
        context = _current_client_context;
    } else {
        client = _last_socket_client;
        context = _last_client_context;
    }

    if (client >= 0) {
        _send_transaction.clear ();
        _send_transaction.put_command (SCIM_TRANS_CMD_REPLY);
        _send_transaction.put_data ((uint32) context);
        _send_transaction.put_command (SCIM_TRANS_CMD_SELECT_LOOKUP_TABLE);
        _send_transaction.put_data ((uint32)item);
        _send_transaction.write_to_socket (Socket (client));
    }
}

static void
action_lookup_table_page_up (void)
{
    int client;
    uint32 context;

    if (_current_socket_client >= 0) {
        client = _current_socket_client;
        context = _current_client_context;
    } else {
        client = _last_socket_client;
        context = _last_client_context;
    }

    if (client >= 0) {
        _send_transaction.clear ();
        _send_transaction.put_command (SCIM_TRANS_CMD_REPLY);
        _send_transaction.put_data ((uint32) context);
        _send_transaction.put_command (SCIM_TRANS_CMD_PROCESS_KEYEVENT);
        _send_transaction.put_data (_lookup_table_page_up_key);
        _send_transaction.write_to_socket (Socket (client));
    }
}

static void
action_lookup_table_page_down (void)
{
    int client;
    uint32 context;

    if (_current_socket_client >= 0) {
        client = _current_socket_client;
        context = _current_client_context;
    } else {
        client = _last_socket_client;
        context = _last_client_context;
    }

    if (client >= 0) {
        _send_transaction.clear ();
        _send_transaction.put_command (SCIM_TRANS_CMD_REPLY);
        _send_transaction.put_data ((uint32) context);
        _send_transaction.put_command (SCIM_TRANS_CMD_PROCESS_KEYEVENT);
        _send_transaction.put_data (_lookup_table_page_down_key);
        _send_transaction.write_to_socket (Socket (client));
    }
}

static void
action_toggle_window_stick (void)
{
    GtkWidget *image;

    _window_sticked = ! _window_sticked;

    if (_window_stick_button) {
        image = gtk_bin_get_child (GTK_BIN (_window_stick_button));
        gtk_container_remove (GTK_CONTAINER (_window_stick_button), image);

        image = ui_create_stick_icon (_window_sticked);
        gtk_container_add (GTK_CONTAINER (_window_stick_button), image);
    }
}

static void
action_toggle_input_status (void)
{
    int client;
    uint32 context;

    if (_current_socket_client >= 0) {
        client = _current_socket_client;
        context = _current_client_context;
    } else {
        client = _last_socket_client;
        context = _last_client_context;
    }

    if (client >= 0) {
        _send_transaction.clear ();
        _send_transaction.put_command (SCIM_TRANS_CMD_REPLY);
        _send_transaction.put_data ((uint32) context);
        _send_transaction.put_command (SCIM_TRANS_CMD_TOGGLE_INPUT_STATUS);
        _send_transaction.write_to_socket (Socket (client));
    }
}

static void
action_show_command_menu (void)
{
    if (_command_menu_activated)
        return;

    _command_menu_activated = true;

    guint32 activate_time = gtk_get_current_event_time ();

    if (_command_menu) {
        gtk_widget_destroy (_command_menu);
        _command_menu = 0;
    }

    _command_menu = gtk_menu_new ();

    GtkWidget *menu_item;
    GtkWidget *item_widget;

    //Stick
    menu_item = gtk_menu_item_new ();
    item_widget = ui_create_stick_command_label (_window_sticked);
    gtk_container_add (GTK_CONTAINER (menu_item), item_widget);
    gtk_menu_shell_append (GTK_MENU_SHELL (_command_menu), menu_item);
    g_signal_connect (G_OBJECT (menu_item), "activate",
                      G_CALLBACK (ui_command_menu_stick_activate_cb),
                      0);
    gtk_widget_show_all (menu_item);

#if HAVE_SCIM_SETUP
    //Setup
    menu_item = gtk_menu_item_new ();
    item_widget = ui_create_setup_command_label ();
    gtk_container_add (GTK_CONTAINER (menu_item), item_widget);
    gtk_menu_shell_append (GTK_MENU_SHELL (_command_menu), menu_item);
    g_signal_connect (G_OBJECT (menu_item), "activate",
                      G_CALLBACK (ui_command_menu_setup_activate_cb),
                      0);
    gtk_widget_show_all (menu_item);
#endif

    //Help
    menu_item = gtk_menu_item_new ();
    item_widget = ui_create_help_command_label ();
    gtk_container_add (GTK_CONTAINER (menu_item), item_widget);
    gtk_menu_shell_append (GTK_MENU_SHELL (_command_menu), menu_item);
    g_signal_connect (G_OBJECT (menu_item), "activate",
                      G_CALLBACK (ui_command_menu_help_activate_cb),
                      0);
    gtk_widget_show_all (menu_item);

    g_signal_connect (G_OBJECT (_command_menu), "deactivate",
                      G_CALLBACK (ui_command_menu_deactivate_cb),
                      NULL);

    gtk_menu_popup (GTK_MENU (_command_menu), 0, 0, 0, 0, 2, activate_time);
}

// Implementation of socket related functions.
static bool 
initialize_socket (void)
{
    SCIM_DEBUG_MAIN (1) << "Initialize Socket Server...\n";

    // Read configurations
    _socket_address = SCIM_PANEL_DEFAULT_SOCKET_ADDRESS;
    _socket_timeout = 500;
    _current_socket_client = -1;
    _current_client_context = 0;
    _last_socket_client = -1;
    _last_client_context = 0;
    _socket_client_count = 0;

    if (!_config.null ()) {
        _socket_address = _config->read (String (SCIM_CONFIG_PANEL_SOCKET_ADDRESS),
                                        _socket_address);
        _socket_timeout = _config->read (String (SCIM_CONFIG_PANEL_SOCKET_TIMEOUT),
                                        _socket_timeout);
    }

    // Override the config vaule if env is set.
    const char *env = getenv ("SCIM_PANEL_SOCKET_ADDRESS");
    if (env) {
        _socket_address = String (env);
    }

    env = getenv ("SCIM_SOCKET_TIMEOUT");
    if (env) {
        _socket_timeout = atoi (env);
    } else {
        env = getenv ("SCIM_PANEL_SOCKET_TIMEOUT");
        if (env)
        _socket_timeout = atoi (env);
    }

    if (_socket_address == "default")
        _socket_address = SCIM_PANEL_DEFAULT_SOCKET_ADDRESS;

    // No timeout is ridiculous, set it to unlimited.
    if (_socket_timeout == 0)
        _socket_timeout = -1;

    if (!_socket_server.create (SocketAddress (_socket_address))) {
        std::cerr << "Cannot create SocketServer.\n";
        return false;
    }

    _socket_server.set_max_clients (1024);

    _socket_server.signal_connect_accept (slot (socket_accept_callback));
    _socket_server.signal_connect_receive (slot (socket_receive_callback));
    _socket_server.signal_connect_exception(slot (socket_exception_callback));

    return true;
}

static bool
run_socket (void)
{
    SCIM_DEBUG_MAIN (1) << "Run Socket Server...\n";

    _socket_server_thread = NULL;

    if (_socket_server.valid ())
        _socket_server_thread = g_thread_create (socket_thread_func, NULL, TRUE, NULL);

    return (_socket_server_thread != NULL);
}

static gpointer
socket_thread_func (gpointer data)
{
    SCIM_DEBUG_MAIN (2) << " In Socket Server Thread...\n";
    _socket_server.run ();
    g_thread_exit (NULL);
    return ((gpointer) NULL);
}

static bool
check_client_connection (const Socket &client)
{
    SCIM_DEBUG_MAIN (3) << "  Check Client Connection...\n";

    unsigned char buf [sizeof(uint32)];

    int nbytes = client.read_with_timeout (buf, sizeof(uint32), _socket_timeout);

    if (nbytes == sizeof (uint32))
        return true;

    if (nbytes < 0) {
        SCIM_DEBUG_MAIN (4) << "   Error occurred when reading socket (" << client.get_id ()
            << "):" << client.get_error_message () << "\n";
    } else {
        SCIM_DEBUG_MAIN (4) << "   Timeout when reading socket (" << client.get_id ()
            << ").\n";
    }

    return false;
}

static void
socket_accept_callback (SocketServer *server, const Socket &client)
{
    SCIM_DEBUG_MAIN (2) << " Accept Client Connection...\n";

    _socket_client_count ++;
}

static void
socket_receive_callback (SocketServer *server, const Socket &client)
{
    int id = client.get_id ();
    int cmd = 0;
    uint32 context = 0;

    SCIM_DEBUG_MAIN (2) << " Receive data from client " << id << " ...\n";

    if (!check_client_connection (client)) {
        _socket_client_count --;
        server->close_connection (client);

        // The focused client is closed.
        if (_current_socket_client == id) {

            _current_socket_client = -1;
            _current_client_context = 0;
            _last_socket_client = -1;
            _last_client_context = 0;

            if (_ui_initialized) {
                gdk_threads_enter ();
                socket_turn_off ();
                gdk_threads_leave ();
            }
        }

        SCIM_DEBUG_MAIN (3) << "  Client " << id << " closed...\n";

        // Exit panel if there is no connected client anymore.
        if (_socket_client_count <= 0 && !_should_resident) {
            G_LOCK (_global_resource_lock);
            _should_exit = true;
            G_UNLOCK (_global_resource_lock);
        }

        return;
    }

    if (!_receive_transaction.read_from_socket (client, _socket_timeout))
        return;

    _send_transaction.clear ();
    _send_transaction.put_command (SCIM_TRANS_CMD_REPLY);

    if (_ui_initialized &&
        _receive_transaction.get_data_type () == SCIM_TRANS_DATA_COMMAND &&
        _receive_transaction.get_command (cmd) && cmd == SCIM_TRANS_CMD_REQUEST &&
        _receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        _receive_transaction.get_data (context)) {

        _send_transaction.put_data (context);

        gdk_threads_enter ();
        while (_receive_transaction.get_data_type () == SCIM_TRANS_DATA_COMMAND &&
               _receive_transaction.get_command (cmd)) {

            if (cmd == SCIM_TRANS_CMD_PANEL_EXIT) {
                G_LOCK (_global_resource_lock);
                if (_should_exit) {
                    SCIM_DEBUG_MAIN (3) << "  Exit Socket Server Thread.\n";
                    server->shutdown ();
                }
                G_UNLOCK (_global_resource_lock);
                gdk_threads_leave ();
                return;
            } else if (cmd == SCIM_TRANS_CMD_FOCUS_IN) {
                _current_socket_client = id;
                _current_client_context = context;
            } else if ((_current_socket_client == id &&
                        _current_client_context == context) ||
                       (_current_socket_client < 0 &&
                        (_last_socket_client == id &&
                         _last_client_context == context))) {
                // Client must focus in before do any other things.
                if (cmd == SCIM_TRANS_CMD_PANEL_TURN_ON)
                    socket_turn_on ();
                else if (cmd == SCIM_TRANS_CMD_PANEL_TURN_OFF)
                    socket_turn_off ();
                else if (cmd == SCIM_TRANS_CMD_PANEL_UPDATE_DISPLAY)
                    socket_update_display ();
                else if (cmd == SCIM_TRANS_CMD_PANEL_UPDATE_SCREEN)
                    socket_update_screen ();
                else if (cmd == SCIM_TRANS_CMD_PANEL_UPDATE_SPOT_LOCATION)
                    socket_update_spot_location ();
                else if (cmd == SCIM_TRANS_CMD_PANEL_UPDATE_SERVER_INFO)
                    socket_update_server_info ();
                else if (cmd == SCIM_TRANS_CMD_SHOW_PREEDIT_STRING)
                    socket_show_preedit_string ();
                else if (cmd == SCIM_TRANS_CMD_SHOW_STATUS_STRING)
                    socket_show_status_string ();
                else if (cmd == SCIM_TRANS_CMD_SHOW_AUX_STRING)
                    socket_show_aux_string ();
                else if (cmd == SCIM_TRANS_CMD_SHOW_LOOKUP_TABLE)
                    socket_show_lookup_table ();
                else if (cmd == SCIM_TRANS_CMD_HIDE_PREEDIT_STRING)
                    socket_hide_preedit_string ();
                else if (cmd == SCIM_TRANS_CMD_HIDE_STATUS_STRING)
                    socket_hide_status_string ();
                else if (cmd == SCIM_TRANS_CMD_HIDE_AUX_STRING)
                    socket_hide_aux_string ();
                else if (cmd == SCIM_TRANS_CMD_HIDE_LOOKUP_TABLE)
                    socket_hide_lookup_table ();
                else if (cmd == SCIM_TRANS_CMD_UPDATE_PREEDIT_STRING)
                    socket_update_preedit_string ();
                else if (cmd == SCIM_TRANS_CMD_UPDATE_PREEDIT_CARET)
                    socket_update_preedit_caret ();
                else if (cmd == SCIM_TRANS_CMD_UPDATE_STATUS_STRING)
                    socket_update_status_string ();
                else if (cmd == SCIM_TRANS_CMD_UPDATE_AUX_STRING)
                    socket_update_aux_string ();
                else if (cmd == SCIM_TRANS_CMD_UPDATE_LOOKUP_TABLE)
                    socket_update_lookup_table ();
                else if (cmd == SCIM_TRANS_CMD_UPDATE_FULL_WIDTH_PUNCTUATION)
                    socket_update_full_width_punctuation ();
                else if (cmd == SCIM_TRANS_CMD_UPDATE_FULL_WIDTH_LETTER)
                    socket_update_full_width_letter ();
                else if (cmd == SCIM_TRANS_CMD_FOCUS_OUT) {
                    _last_socket_client = id;
                    _last_client_context = context;
                    _current_socket_client = -1;
                    _current_client_context = 0;
                    socket_turn_off ();
                    break;
                } else if (cmd == SCIM_TRANS_CMD_PANEL_SHOW_HELP) {
                    socket_show_help ();
                } else if (cmd == SCIM_TRANS_CMD_PANEL_SHOW_SERVER_MENU) {
                    socket_show_server_menu ();
                }
            }
        }
        gdk_threads_leave ();
    }

    // cmd == SCIM_TRANS_CMD_REPLY
    _send_transaction.get_command (cmd);

    // the context id
    _send_transaction.get_data (context);

    // Send the reply only if necessary.
    if (_send_transaction.get_data_type () != SCIM_TRANS_DATA_UNKNOWN)
        _send_transaction.write_to_socket (client);
}

static void
socket_exception_callback (SocketServer *server, const Socket &client)
{
    SCIM_DEBUG_MAIN (2) << "  Exception occurred on client " << client.get_id () << " ...\n";

    // The focused client is closed.
    if (_current_socket_client == client.get_id ()) {
        _current_socket_client = -1;
        _current_client_context = 0;
        _last_socket_client = -1;
        _last_client_context = 0;

        if (_ui_initialized) {
            gdk_threads_enter ();
            socket_turn_off ();
            gdk_threads_leave ();
        }
    }

    _socket_client_count --;

    SCIM_DEBUG_MAIN (3) << "  Client " << client.get_id () << " closed...\n";

    server->close_connection (client);

    // Exit panel if there is no connected client anymore.
    if (_socket_client_count <= 0 && !_should_resident) {
        G_LOCK (_global_resource_lock);
        _should_exit = true;
        G_UNLOCK (_global_resource_lock);
    }

    return;
}

static void
socket_turn_on (void)
{
    SCIM_DEBUG_MAIN (3) << "  Turn on panel...\n";

    _toolbar_should_hide = false;
    _toolbar_hidden = false;
    _panel_is_on = true;

    gtk_widget_hide (_lookup_table_window);
    gtk_widget_hide (_input_window);
    gtk_widget_hide (_preedit_area);
    gtk_widget_hide (_aux_area);
    gtk_widget_hide (_status_button);
    gtk_widget_hide (_full_width_punct_button);
    gtk_widget_hide (_full_width_letter_button);

    if (_status_area)
        gtk_widget_hide (_status_area);

    if (_window_stick_button)
        gtk_widget_show (_window_stick_button);

    if (_server_button)
        gtk_widget_show (_server_button);

#if HAVE_SCIM_SETUP
    if (_setup_button)
        gtk_widget_show (_setup_button);
#endif

    if (_help_button)
        gtk_widget_show (_help_button);

    gtk_widget_show (_toolbar_window);

    ui_settle_toolbar_window ();
}

static void
socket_turn_off (void)
{
    SCIM_DEBUG_MAIN (3) << "  Turn off panel...\n";

    _panel_is_on = false;

    gtk_widget_hide (_input_window);
    gtk_widget_hide (_lookup_table_window);

    gtk_widget_hide (_preedit_area);
    gtk_widget_hide (_aux_area);
    gtk_widget_hide (_status_button);
    gtk_widget_hide (_full_width_letter_button);
    gtk_widget_hide (_full_width_punct_button);

    if (_status_area)
        gtk_widget_hide (_status_area);

    if (_toolbar_always_show) {
        if (!_toolbar_hidden) {
            if (_window_stick_button)
                gtk_widget_show (_window_stick_button);

            if (_server_button)
                gtk_widget_show (_server_button);

#if HAVE_SCIM_SETUP
            if (_setup_button)
                gtk_widget_show (_setup_button);
#endif

            if (_help_button)
                gtk_widget_show (_help_button);

        }
        ui_settle_toolbar_window ();
        gtk_widget_show (_toolbar_window);
        _toolbar_should_hide = true;
    } else {
        gtk_widget_hide (_toolbar_window);
        _toolbar_hidden = true;
    }
}

static void
socket_update_display (void)
{
    SCIM_DEBUG_MAIN (3) << "  socket_update_display...\n";

    String name;

    if (_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        _receive_transaction.get_data (name)) {

#if GDK_MULTIHEAD_SAFE
        if (name == String (gdk_display_get_name (_current_display)))
            return;

        GdkDisplay *display = gdk_display_open (name.c_str ());

        if (!display) return;

        GdkScreen *screen = gdk_display_get_default_screen (display);

        if (!screen) return;

        _current_display = display;
        _current_screen  = screen;

        gdk_display_manager_set_default_display (gdk_display_manager_get (), display);

        ui_switch_screen (screen);
#endif
    }
}

static void
socket_update_screen (void)
{
    SCIM_DEBUG_MAIN (3) << "  socket_update_screen...\n";

    uint32 number;

    if (_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        _receive_transaction.get_data (number)) {

#if GDK_MULTIHEAD_SAFE
        if (gdk_screen_get_number (_current_screen) == number ||
            gdk_display_get_n_screens (_current_display) <= number)
            return;

        GdkScreen *screen = gdk_display_get_screen (_current_display, number);

        if (!screen) return;

        _current_screen  = screen;

        ui_switch_screen (screen);
#endif
    }
}

static void
socket_update_server_info (void)
{
    SCIM_DEBUG_MAIN (3) << "  socket_update_server_info...\n";

    String name;
    String iconfile;

    if (_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        _receive_transaction.get_data (name) &&
        _receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        _receive_transaction.get_data (iconfile)) {

        SCIM_DEBUG_MAIN (4) << "  Icon file: " << iconfile <<"\n";

        if (_server_button) {
            GtkWidget * newlabel = 0;

            if (_toolbar_show_server_icon) {
                newlabel = ui_create_label (name,
                                            iconfile,
                                            0,
                                            !_toolbar_show_server_name,
                                            false);
            } else {
                newlabel = gtk_label_new (name.c_str ());
                gtk_widget_show (newlabel);
            }

            if (newlabel) {
                GtkWidget * old = gtk_bin_get_child (GTK_BIN (_server_button));
                if (old)
                    gtk_container_remove (GTK_CONTAINER (_server_button), old);
                gtk_container_add (GTK_CONTAINER (_server_button), newlabel);

                GtkRequisition size;

                gtk_widget_size_request (newlabel, &size);
                gtk_widget_set_size_request (_server_button, -1, size.height + 6);
            }

            if (!GTK_WIDGET_VISIBLE (_server_button) && !_toolbar_hidden)
                gtk_widget_show (_server_button);

            if (_tooltips)
                gtk_tooltips_set_tip (_tooltips, _server_button, name.c_str (), NULL);

            ui_settle_toolbar_window ();
        }

#if ENABLE_TRAY_ICON
        if (_tray_icon_server_button) {
            GtkWidget *icon = gtk_bin_get_child (GTK_BIN (_tray_icon_server_button));

            if (icon)
                gtk_container_remove (GTK_CONTAINER (_tray_icon_server_button), icon);

            icon = ui_create_icon_from_file (iconfile, TRAY_ICON_SIZE, TRAY_ICON_SIZE, true);

            gtk_container_add (GTK_CONTAINER (_tray_icon_server_button), icon);

            if (_tooltips)
                gtk_tooltips_set_tip (_tooltips, _tray_icon_server_button, name.c_str (), NULL);
        }
#endif
    }
}

static void
socket_show_help (void)
{
    SCIM_DEBUG_MAIN (3) << "  socket_show_help...\n";
    
    String help;
    if (_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        _receive_transaction.get_data (help)) {
        GtkRequisition size;

        gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (_help_scroll), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);

        gtk_label_set_text (GTK_LABEL (_help_area), help.c_str ());

        gtk_widget_size_request (_help_area, &size);

        if (size.width > ui_screen_width ()/2) {
            size.width = ui_screen_width ()/2;
            gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (_help_scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
        }

        if (size.height > ui_screen_height ()/2)
            size.height = ui_screen_height ()/2;

        if (size.height < size.width/2)
            size.height = size.width/2;

        gtk_widget_set_size_request (_help_scroll, size.width, size.height);

        gtk_window_set_position (GTK_WINDOW (_help_dialog), GTK_WIN_POS_CENTER_ALWAYS);
        gtk_widget_show (_help_dialog);
    }
}

static void
socket_show_server_menu (void)
{
    SCIM_DEBUG_MAIN (3) << "  socket_show_server_menu...\n";

    std::vector <String> names;
    std::vector <String> uuids;
    std::vector <String> icons;

    String name;
    String uuid;
    String icon;

    guint32 activate_time = gtk_get_current_event_time ();

    while (_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
           _receive_transaction.get_data (uuid) &&
           _receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
           _receive_transaction.get_data (name) &&
           _receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
           _receive_transaction.get_data (icon)) {
        uuids.push_back (uuid);
        names.push_back (name);
        icons.push_back (icon);
    }

    if (_server_menu_activated || !uuids.size ()) return;

    _server_menu_activated = true;

    _server_menu_uuids = uuids;

    if (_server_menu) {
        gtk_widget_destroy (_server_menu);
        _server_menu = 0;
    }

    _server_menu = gtk_menu_new ();

    GtkWidget *menu_item;
    GtkWidget *item_widget;

    for (size_t i=0; i<uuids.size (); ++i) {
        menu_item = gtk_menu_item_new ();

        item_widget = ui_create_label (names [i], icons [i], 0, false, true);

        gtk_container_add (GTK_CONTAINER (menu_item), item_widget);

        gtk_menu_shell_append (GTK_MENU_SHELL (_server_menu), menu_item);

        g_signal_connect (G_OBJECT (menu_item), "activate",
                          G_CALLBACK (ui_server_menu_activate_cb),
                          GINT_TO_POINTER ((int)i));
        gtk_widget_show (menu_item);
    }

    //Append an entry for forward mode.
    menu_item = gtk_menu_item_new ();

    item_widget = ui_create_label (String (_("English")),
                                   String (SCIM_KEYBOARD_ICON_FILE),
                                   0,
                                   false,
                                   true);

    gtk_container_add (GTK_CONTAINER (menu_item), item_widget);

    gtk_menu_shell_append (GTK_MENU_SHELL (_server_menu), menu_item);
    g_signal_connect (G_OBJECT (menu_item), "activate",
                      G_CALLBACK (ui_server_menu_activate_cb),
                      GINT_TO_POINTER (-1));
    gtk_widget_show (menu_item);

    g_signal_connect (G_OBJECT (_server_menu), "deactivate",
                      G_CALLBACK (ui_server_menu_deactivate_cb),
                      NULL);
    gtk_menu_popup (GTK_MENU (_server_menu), 0, 0, 0, 0, 1, activate_time);
}

static void
socket_update_spot_location (void)
{
    SCIM_DEBUG_MAIN (3) << "  Set spot location...\n";

    uint32 x, y;

    if (_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        _receive_transaction.get_data (x) &&
        _receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        _receive_transaction.get_data (y)) {

        _spot_location_x = (int) x;
        _spot_location_y = (int) y;

        ui_settle_input_window ();
        if (!_lookup_table_embedded)
            ui_settle_lookup_table_window ();
    }
}

static void
socket_show_preedit_string (void)
{
    SCIM_DEBUG_MAIN (3) << "  Show preedit string...\n";

    gtk_widget_show (_preedit_area);

    if (_panel_is_on && !GTK_WIDGET_VISIBLE (_input_window))
        gtk_widget_show (_input_window);

    ui_settle_input_window (true);
}

static void
socket_show_status_string (void)
{
    SCIM_DEBUG_MAIN (3) << "  Show status string...\n";

    if (_status_area) {
        gtk_widget_show (_status_area);
        if (_panel_is_on && !GTK_WIDGET_VISIBLE (_input_window))
            gtk_widget_show (_input_window);
    }

    if (!_toolbar_hidden && !GTK_WIDGET_VISIBLE (_status_button)) {
        gtk_widget_show (_status_button);
        ui_settle_toolbar_window ();
    }

    ui_settle_input_window (true);
}

static void
socket_show_aux_string (void)
{
    SCIM_DEBUG_MAIN (3) << "  Show aux string...\n";

    gtk_widget_show (_aux_area);

    if (_panel_is_on && !GTK_WIDGET_VISIBLE (_input_window))
        gtk_widget_show (_input_window);

    ui_settle_input_window (true);
}

static void
socket_show_lookup_table (void)
{
    SCIM_DEBUG_MAIN (3) << "  Show lookup table...\n";

    gtk_widget_show (_lookup_table_window);

    if (_panel_is_on && _lookup_table_embedded && !GTK_WIDGET_VISIBLE (_input_window))
        gtk_widget_show (_input_window);

    ui_settle_lookup_table_window (true);
}

static void
socket_hide_preedit_string (void)
{
    SCIM_DEBUG_MAIN (3) << "  Hide preedit string...\n";

    gtk_widget_hide (_preedit_area);
    scim_string_view_set_text (SCIM_STRING_VIEW (_preedit_area), "");

    if (ui_can_hide_input_window ())
        gtk_widget_hide (_input_window);
}

static void
socket_hide_status_string (void)
{
    SCIM_DEBUG_MAIN (3) << "  Hide status string...\n";

    if (_status_area) {
        gtk_widget_hide (_status_area);
        gtk_label_set_text (GTK_LABEL (_status_area), "");
    }

    gtk_widget_hide (_status_button);

    if (ui_can_hide_input_window ())
        gtk_widget_hide (_input_window);

}

static void
socket_hide_aux_string (void)
{
    SCIM_DEBUG_MAIN (3) << "  Hide aux string...\n";

    gtk_widget_hide (_aux_area);
    scim_string_view_set_text (SCIM_STRING_VIEW (_aux_area), "");

    if (ui_can_hide_input_window ())
        gtk_widget_hide (_input_window);
}

static void
socket_hide_lookup_table (void)
{
    SCIM_DEBUG_MAIN (3) << "  Hide lookup table...\n";

    gtk_widget_hide (_lookup_table_window);

    if (_lookup_table_embedded && ui_can_hide_input_window ())
        gtk_widget_hide (_input_window);
}

static void
socket_update_preedit_string (void)
{
    SCIM_DEBUG_MAIN (3) << "  Update preedit string...\n";

    String str;
    AttributeList attrs;

    if (_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        _receive_transaction.get_data (str) &&
        _receive_transaction.get_data_type () == SCIM_TRANS_DATA_ATTRLIST &&
        _receive_transaction.get_data (attrs)) {

        PangoAttrList  *attrlist = create_pango_attrlist (str, attrs);

        scim_string_view_set_attributes (SCIM_STRING_VIEW (_preedit_area), attrlist);
        scim_string_view_set_text (SCIM_STRING_VIEW (_preedit_area), str.c_str ());

        pango_attr_list_unref (attrlist);

        ui_settle_input_window (true);
    }
}

static void
socket_update_preedit_caret (void)
{
    SCIM_DEBUG_MAIN (3) << "  Update preedit caret...\n";

    uint32 caret;

    if (_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        _receive_transaction.get_data (caret)) {
        scim_string_view_set_position (SCIM_STRING_VIEW (_preedit_area), caret);
    }
}

static void
socket_update_status_string (void)
{
    SCIM_DEBUG_MAIN (3) << "  Update status string...\n";

    String str;
    AttributeList attrs;

    if (_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        _receive_transaction.get_data (str) &&
        _receive_transaction.get_data_type () == SCIM_TRANS_DATA_ATTRLIST &&
        _receive_transaction.get_data (attrs)) {

        PangoAttrList  *attrlist = create_pango_attrlist (str, attrs);

        if (_status_area) {
            gtk_label_set_text (GTK_LABEL (_status_area), str.c_str ());
            gtk_label_set_attributes (GTK_LABEL (_status_area), attrlist);
        }

        if (_status_button) {
            GtkWidget * old = gtk_bin_get_child (GTK_BIN (_status_button));
            if (old)
                gtk_container_remove (GTK_CONTAINER (_status_button), old);

            GtkWidget * label = gtk_label_new (str.c_str ());
            gtk_widget_show (label);
            gtk_label_set_attributes (GTK_LABEL (label), attrlist);
            gtk_container_add (GTK_CONTAINER (_status_button), label);

            GtkRequisition size;

            gtk_widget_size_request (label, &size);
            gtk_widget_set_size_request (_status_button, -1, size.height + 6);
        }

        pango_attr_list_unref (attrlist);

        ui_settle_input_window (true);
        ui_settle_toolbar_window ();
    }
}

static void
socket_update_aux_string (void)
{
    SCIM_DEBUG_MAIN (3) << "  Update aux string...\n";

    String str;
    AttributeList attrs;

    if (_receive_transaction.get_data_type () == SCIM_TRANS_DATA_STRING &&
        _receive_transaction.get_data (str) &&
        _receive_transaction.get_data_type () == SCIM_TRANS_DATA_ATTRLIST &&
        _receive_transaction.get_data (attrs)) {

        PangoAttrList  *attrlist = create_pango_attrlist (str, attrs);

        scim_string_view_set_attributes (SCIM_STRING_VIEW (_aux_area), attrlist);
        scim_string_view_set_text (SCIM_STRING_VIEW (_aux_area), str.c_str ());

        pango_attr_list_unref (attrlist);

        ui_settle_input_window (true);
    }
}

static void
socket_update_lookup_table (void)
{
    SCIM_DEBUG_MAIN (3) << "  Update lookup table...\n";

    CommonLookupTable table;
    if (_receive_transaction.get_data_type () == SCIM_TRANS_DATA_LOOKUP_TABLE &&
        _receive_transaction.get_data (table)) {

        size_t i;
        size_t item_num = table.get_current_page_size ();

        KeyEvent       key;
        String         mbs;
        WideString     wcs;
        GtkRequisition size;

        if (_lookup_table_vertical) {
            for (i = 0; i < SCIM_LOOKUP_TABLE_MAX_PAGESIZE; ++ i) {
                if (i < item_num) {
                    mbs = String ();

                    wcs = table.get_content_in_page (i);

                    key = table.get_page_index_key (i);

                    if (key.get_ascii_code ())
                        mbs.push_back (key.get_ascii_code ());
                    else
                        mbs.push_back (' ');

                    mbs += String (". ");
                    mbs += utf8_wcstombs (wcs);

                    if (i == table.get_cursor_pos_in_page () && table.is_cursor_visible ())
                        scim_string_view_set_highlight (SCIM_STRING_VIEW (_lookup_table_items [i]),
                                                        0, wcs.length () + 3);
                    else
                        scim_string_view_set_highlight (SCIM_STRING_VIEW (_lookup_table_items [i]),
                                                        -1, -1);

                    scim_string_view_set_text (SCIM_STRING_VIEW (_lookup_table_items [i]),
                                               mbs.c_str ());
                    gtk_widget_show (_lookup_table_items [i]);
                } else {
                    gtk_widget_hide (_lookup_table_items [i]);
                }
            }
        } else {
            _lookup_table_index [0] = 0;
            for (i=0; i<SCIM_LOOKUP_TABLE_MAX_PAGESIZE; ++i) {
                mbs = String ();

                if (i<item_num) {
                    key = table.get_page_index_key (i);

                    if (key.get_ascii_code () != 0) {
                        mbs.push_back (key.get_ascii_code ());
                    } else {
                        mbs.push_back (' ');
                    }

                    mbs.push_back ('.');

                    wcs += utf8_mbstowcs (mbs);
                    wcs += table.get_content_in_page (i);
                    wcs .push_back (0x20);

                    _lookup_table_index [i+1] = wcs.length ();

                    scim_string_view_set_text (SCIM_STRING_VIEW (_lookup_table_items [0]),
                                               utf8_wcstombs (wcs).c_str ());

                    gtk_widget_size_request (_lookup_table_window, &size);

                    if (size.width >= ui_screen_width () / 3) {
                        item_num = i+1;
                    }
                } else {
                    _lookup_table_index [i+1] = -1;
                }
            }

            if (table.is_cursor_visible ()) {
                int start = _lookup_table_index [table.get_cursor_pos_in_page ()];
                int end = _lookup_table_index [table.get_cursor_pos_in_page ()+1] - 1;
                scim_string_view_set_highlight (SCIM_STRING_VIEW (_lookup_table_items [0]), start, end);
            } else {
                scim_string_view_set_highlight (SCIM_STRING_VIEW (_lookup_table_items [0]), -1, -1);
            }
        }

        _lookup_table_page_up_key = table.get_page_up_key ();
        _lookup_table_page_down_key = table.get_page_down_key ();

        if (table.get_current_page_start ())
            gtk_widget_set_sensitive (_lookup_table_up_button, TRUE);
        else
            gtk_widget_set_sensitive (_lookup_table_up_button, FALSE);

        if (table.get_current_page_start () + item_num < table.number_of_entries ())
            gtk_widget_set_sensitive (_lookup_table_down_button, TRUE);
        else
            gtk_widget_set_sensitive (_lookup_table_down_button, FALSE);

        if (item_num < table.get_current_page_size ()) {
            _send_transaction.put_command (SCIM_TRANS_CMD_UPDATE_LOOKUP_TABLE_PAGE_SIZE);
            _send_transaction.put_data ((uint32) item_num);
        }

        ui_settle_lookup_table_window (true);
    }
}

static void
socket_update_full_width_punctuation (void)
{
    SCIM_DEBUG_MAIN (3) << "  Update full width punctuation...\n";

    uint32 full;

    if (_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        _receive_transaction.get_data (full)) {

        GtkWidget *image;

        image = gtk_bin_get_child (GTK_BIN (_full_width_punct_button));
        gtk_container_remove (GTK_CONTAINER (_full_width_punct_button), image);

        image = ui_create_punct_icon (full);
        gtk_container_add (GTK_CONTAINER (_full_width_punct_button), image);

        if (!_toolbar_hidden && !GTK_WIDGET_VISIBLE (_full_width_punct_button))
            gtk_widget_show (_full_width_punct_button);

        ui_settle_toolbar_window ();
    }
}

static void
socket_update_full_width_letter (void)
{
    SCIM_DEBUG_MAIN (3) << "  Update full width letter...\n";

    uint32 full;

    if (_receive_transaction.get_data_type () == SCIM_TRANS_DATA_UINT32 &&
        _receive_transaction.get_data (full)) {

        GtkWidget *image;

        image = gtk_bin_get_child (GTK_BIN (_full_width_letter_button));
        gtk_container_remove (GTK_CONTAINER (_full_width_letter_button), image);

        image = ui_create_letter_icon (full);
        gtk_container_add (GTK_CONTAINER (_full_width_letter_button), image);

        if (!_toolbar_hidden && !GTK_WIDGET_VISIBLE (_full_width_letter_button))
            gtk_widget_show (_full_width_letter_button);

        ui_settle_toolbar_window ();
    }
}

static gboolean
check_exit_timeout_cb (gpointer data)
{
    G_LOCK (_global_resource_lock);
    if (_should_exit) {
        gdk_threads_enter ();
        gtk_main_quit ();
        gdk_threads_leave ();
    }
    G_UNLOCK (_global_resource_lock);

    return TRUE;
}

static void
signalhandler(int sig)
{
    SCIM_DEBUG_MAIN (1) << "In signal handler...\n";

    G_LOCK (_global_resource_lock);
    _should_exit = true;
    G_UNLOCK (_global_resource_lock);
}

void make_daemon ()
{
#if HAVE_DAEMON
    if (daemon (0, 0) == -1)
        std::cerr << "Error to make myself into a daemon!\n";

    return;
#else        
    pid_t id;
 
    id = fork ();
    if (id == -1) {
        std::cerr << "Error to make myself into a daemon!\n";
        return;
    } else if (id > 0) {
        exit (0);
    }

    id = fork ();
    if (id == -1) {
        std::cerr << "Error to make myself into a daemon!\n";
        return;
    } else if (id > 0) {
        exit (0);
    }

    return;
#endif
}

#if HAVE_SCIM_SETUP
static bool
check_scim_setup ()
{
    return access (SCIM_SETUP_PROGRAM_PATH, X_OK) == 0;
}

static void
sigchld_handler (int signum)
{
    int pid, status, serrno;

    serrno = errno;

    signal (SIGCHLD, SIG_DFL);

    while (1) {
        pid = waitpid (WAIT_ANY, &status, WNOHANG);
        if (pid <= 0) break;
    }

    if (pid < 0)
        _scim_setup_running = false;

    if (pid == 0)
        signal (SIGCHLD, sigchld_handler);

    errno = serrno;
}

static void
run_scim_setup ()
{
    static char * argv [] = { SCIM_SETUP_PROGRAM, NULL };
    int pid;

    if (!check_scim_setup () || _scim_setup_running)
        return;

    pid = fork ();

    if (pid == 0) {
        execv (SCIM_SETUP_PROGRAM_PATH, argv);
        exit (-1);
    } else if (pid > 0) {
        _scim_setup_running = true;
        signal (SIGCHLD, sigchld_handler);
    }
}
#endif

int main (int argc, char *argv [])
{
    std::vector<String>  config_list;

    String def_config;
    String debug_output;

    uint32 verbose_level = 0;

    int i;

    bool daemon = false;

    char *argv0 = argv [0];

    //Display version info
    std::cout << "GTK Panel of SCIM " << SCIM_VERSION << "\n\n";

    //get modules list
    scim_get_config_module_list (config_list);

    //Add a dummy config module, it's not really a module!
    config_list.push_back ("dummy");

    //Use socket Config module as default if available.
    if (config_list.size ()) {
        def_config = String ("simple");
        if (std::find (config_list.begin (),
                       config_list.end (),
                       def_config) == config_list.end ())
            def_config = config_list [0];
    }

    //parse command options
    i = 0;
    while (i<argc) {
        if (++i >= argc) break;

        if (String ("-l") == argv [i] ||
            String ("--list") == argv [i]) {
            std::vector<String>::iterator it;

            std::cout << "\n";
            std::cout << "Available Config module:\n";
            for (it = config_list.begin (); it != config_list.end (); it++)
                std::cout << "    " << *it << "\n";

            return 0;
        }

        if (String ("-c") == argv [i] ||
            String ("--config") == argv [i]) {
            if (++i >= argc) {
                std::cerr << "No argument for option " << argv [i-1] << "\n";
                return -1;
            }
            def_config = argv [i];
            continue;
        }

        if (String ("-h") == argv [i] ||
            String ("--help") == argv [i]) {
            std::cout << "Usage: " << argv [0] << " [option]...\n\n"
                 << "The options are: \n"
                 << "  -l, --list\t\tList all of available config modules.\n"
                 << "  -c, --config name\tUses specified Config module.\n"
                 << "  -d, --daemon\t\tRun " << argv [0] << " as a daemon.\n"
                 << "  -ns, --no-stay\t\tQuit if no connected client.\n"
#if ENABLE_DEBUG
                 << "  -v, --verbose\t\tEnable debug info, multiple -v for more.\n"
                 << "  -o, --output file\tOutput debug information into file.\n"
#endif
                 << "  -h, --help\t\tShow this help message.\n";
            return 0;
        }

        if (String ("-d") == argv [i] ||
            String ("--daemon") == argv [i]) {
            daemon = true;
            continue;
        }

        if (String ("-ns") == argv [i] ||
            String ("--no-stay") == argv [i]) {
            _should_resident = false;
            continue;
        }

        if (String ("-v") == argv [i] ||
            String ("--verbose") == argv [i]) {
            verbose_level += 1;
            continue;
        }

        if (String ("-o") == argv [i] ||
            String ("--output") == argv [i]) {
            if (++i >= argc) {
                std::cerr << "No argument for option " << argv [i-1] << "\n";
                return -1;
            }
            debug_output = argv [i];
            continue;
        }

        if (String ("--") == argv [i])
            break;
    } //End of command line parsing.

    DebugOutput::set_output (debug_output);
    DebugOutput::set_verbose_level (verbose_level);
    DebugOutput::disable_debug (SCIM_DEBUG_AllMask);
    DebugOutput::enable_debug (SCIM_DEBUG_MainMask | SCIM_DEBUG_SocketMask);

    if (!def_config.length ()) {
        std::cerr << "No Config module is available!\n";
        return -1;
    }

    if (def_config != "dummy") {
        //load config module
        _config_module = new ConfigModule (def_config);

        if (!_config_module || !_config_module->valid ()) {
            std::cerr << "Can not load " << def_config << " Config module.\n";
            return -1;
        }

        //create config instance
        _config = _config_module->create_config ("scim");
    } else {
        _config = new DummyConfig ("scim");
    }

    if (_config.null ()) {
        std::cerr << "Failed to create Config instance from "
             << def_config << " Config module.\n";
        return -1;
    }

    /* init threads */
    g_thread_init (NULL);
    gdk_threads_init ();

    signal(SIGQUIT, signalhandler);
    signal(SIGTERM, signalhandler);
    signal(SIGINT,  signalhandler);
    signal(SIGHUP,  signalhandler);

    if (daemon) {
        std::cout << "Starting as daemon ...\n";
            make_daemon ();
    }

    ui_initialize (argc, argv);

    if (!initialize_socket ()) {
        std::cerr << "Failed to initialize Socket Server!\n";
        return -1;
    }
 
    if (!run_socket ()) {
        std::cerr << "Failed to run Socket Server!\n";
        return -1;
    }

    _check_exit_timeout = gtk_timeout_add (500, check_exit_timeout_cb, NULL);

    gdk_threads_enter ();
    gtk_main ();
    gdk_threads_leave ();

    // Exiting...
    {
        SocketTransaction trans;
        SocketClient  client;

        if (client.connect (SocketAddress (_socket_address))) {
            trans.put_command (SCIM_TRANS_CMD_REQUEST);
            trans.put_data ((uint32) 0);
            trans.put_command (SCIM_TRANS_CMD_PANEL_EXIT);
            trans.write_to_socket (client);
            client.close ();
            g_thread_join (_socket_server_thread);
        }

        _config.reset ();

        std::cout << "Successfully exited.\n";
    }

    return 0;
}

/*
vi:ts=4:nowrap:expandtab
*/
