/*
 * SCIM Bridge
 *
 * Copyright (c) 2006 Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.*
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.*
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

#include <assert.h>
#include <alloca.h>
#include <errno.h>
#include <fcntl.h>
#include <pthread.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/socket.h>
#include <sys/types.h>

#include "scim-bridge-exception.h"
#include "scim-bridge-messenger.h"
#include "scim-bridge-output.h"

/* Type definition */
struct _ScimBridgeMessenger
{
    int socket_fd;
};

/* Functions */
retval_t scim_bridge_messenger_send (ScimBridgeMessenger *messenger, const ScimBridgeMessage *message)
{
    scim_bridge_exception_clear ();
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 4, "scim_bridge_messenger_send");
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 4, "message: %s", scim_bridge_message_get_header (message));

    size_t buffer_capacity = 10;
    char *buffer = alloca (sizeof (char) * (buffer_capacity + 1));
    size_t buffer_index = 0;

    const ssize_t arg_count = (ssize_t) scim_bridge_message_get_argument_count (message);

    ssize_t i;
    for (i = -1; i < arg_count; ++i) {
        const char *str;
        if (i == -1) {
            str = scim_bridge_message_get_header (message);
        } else {
            str = scim_bridge_message_get_argument (message, i);
        }

        int j;
        for (j = 0; j < strlen (str); ++j) {
            if (buffer_index + 1 >= buffer_capacity) {
                const size_t new_buffer_capacity = buffer_index + 10;
                char *new_buffer = alloca (sizeof (char) * (new_buffer_capacity + 1));
                strncpy (new_buffer, buffer, buffer_capacity);
                buffer = new_buffer;
                buffer_capacity = new_buffer_capacity;
            }

            switch (str[j]) {
                case '\n':
                case '\\':
                case ' ':
                    buffer[buffer_index] = '\\';
                    buffer[buffer_index + 1] = str[j];
                    buffer_index += 2;
                    break;
                default:
                    buffer[buffer_index] = str[j];
                    buffer_index += 1;
            }
        }

        if (buffer_index >= buffer_capacity) {
            const size_t new_buffer_capacity = buffer_index + 10;
            char *new_buffer = alloca (sizeof (char) * (new_buffer_capacity + 1));
            strncpy (new_buffer, buffer, buffer_capacity);
            buffer = new_buffer;
            buffer_capacity = new_buffer_capacity;
        }
        if (i + 1 == arg_count) {
            buffer[buffer_index] = '\0';
        } else {
            buffer[buffer_index] = ' ';
        }
        ++buffer_index;
    }
    const size_t buffer_length = buffer_index - 1;

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "full message: %s", buffer);
    buffer[buffer_length] = '\n';

    void *seek = buffer;
    size_t remain_size = sizeof (char) * (buffer_length + 1);
    while (remain_size > 0) {
        if (messenger->socket_fd < 0) {
            scim_bridge_exception_occured (IO_EXCEPTION, "Invalid socket fd at scim_bridge_messenger_send");
            scim_bridge_exception_push_stack ("scim_bridge_messenger_send ()");
            return RETVAL_FAILED;
        }

        ssize_t written_size = send (messenger->socket_fd, seek, remain_size, MSG_NOSIGNAL);

        if (written_size <= 0) {
            scim_bridge_exception_occured (IO_EXCEPTION, "IOException at scim_bridge_messenger_send: %s", errno != 0 ? strerror (errno):"unknown reason");
            scim_bridge_exception_push_stack ("scim_bridge_messenger_send ()");
            return RETVAL_FAILED;
        } else {
            seek += written_size;
            remain_size -= written_size;
        }
    }

    return RETVAL_SUCCEEDED;
}


retval_t scim_bridge_messenger_receive (ScimBridgeMessenger *messenger, ScimBridgeMessage **dst)
{
    scim_bridge_exception_clear ();
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 4, "scim_bridge_messenger_receive");

    *dst = NULL;

    boolean escaped = FALSE;

    size_t buffer_index = 0;
    size_t buffer_capacity = 4;
    char **buffers = alloca (sizeof (char*) * buffer_capacity);

    size_t buffer_line_index = 0;
    size_t buffer_line_capacity = 50;
    char *buffer_line = alloca (sizeof (char) * (buffer_line_capacity + 1));

    while (TRUE) {
        if (messenger->socket_fd < 0) {
            scim_bridge_exception_occured (IO_EXCEPTION, "Invalid socket fd at scim_bridge_messenger_receive");
            scim_bridge_exception_push_stack ("scim_bridge_messenger_receive ()");
            return RETVAL_FAILED;
        }

        char c;
        const int read_size = recv (messenger->socket_fd, &c, sizeof (char), 0);

        if (read_size <= 0) {
            scim_bridge_exception_occured (IO_EXCEPTION, "IOException at scim_bridge_messenger_receive: %s", errno != 0 ? strerror (errno):"unknown reason");
            scim_bridge_exception_push_stack ("scim_bridge_messenger_receive ()");
            *dst = NULL;
            return RETVAL_FAILED;
        }

        if (buffer_line_index >= buffer_line_capacity) {
            const size_t new_buffer_line_capacity = buffer_line_index + 50;
            char *new_buffer_line = alloca (sizeof (char) * (new_buffer_line_capacity + 1));
            strncpy (new_buffer_line, buffer_line, buffer_line_capacity);

            buffer_line = new_buffer_line;
            buffer_line_capacity = new_buffer_line_capacity;
        }

        if (c == '\\') {
            if (escaped) {
                buffer_line[buffer_line_index] = '\\';
                ++buffer_line_index;
                escaped = FALSE;
            } else {
                escaped = TRUE;
            }
        } else if (c == ' ') {
            if (escaped) {
                buffer_line[buffer_line_index] = ' ';
                ++buffer_line_index;
                escaped = FALSE;
            } else {
                if (buffer_index >= buffer_capacity) {
                    const size_t new_buffer_capacity = buffer_index + 4;
                    char **new_buffers = alloca (sizeof (char*) * new_buffer_capacity);
                    memcpy (new_buffers, buffers, sizeof (char*) * buffer_capacity);

                    buffers = new_buffers;
                    buffer_capacity = new_buffer_capacity;
                }

                buffer_line[buffer_line_index] = '\0';
                buffers[buffer_index] = buffer_line;
                buffer_line_index = 0;
                buffer_line_capacity = 50;
                buffer_line = alloca (sizeof (char) * (buffer_line_capacity + 1));
                ++buffer_index;
            }
        } else if (c == '\n') {
            if (escaped) {
                buffer_line[buffer_line_index] = '\n';
                ++buffer_line_index;
                escaped = FALSE;
            } else {
                if (buffer_index > buffer_capacity) {
                    const size_t new_buffer_capacity = buffer_index + 4;
                    char **new_buffers = alloca (sizeof (char*) * new_buffer_capacity);
                    memcpy (new_buffers, buffers, sizeof (char*) * buffer_capacity);

                    buffers = new_buffers;
                    buffer_capacity = new_buffer_capacity;
                }

                buffer_line[buffer_line_index] = '\0';
                buffers[buffer_index] = buffer_line;
                buffer_line_index = 0;
                buffer_line_capacity = 50;
                buffer_line = alloca (sizeof (char) * (buffer_line_capacity + 1));
                ++buffer_index;
                break;
            }
        } else {
            buffer_line[buffer_line_index] = c;
            ++buffer_line_index;
            escaped = FALSE;
        }
    }

    const size_t argument_count = buffer_index - 1;
    ScimBridgeMessage *message = scim_bridge_alloc_message (buffers[0], argument_count);

    scim_bridge_pdebug (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "full message: %s ", buffers[0]);
    int i;
    for (i = 0; i < argument_count; ++i) {
        scim_bridge_message_set_argument (message, i, buffers[i + 1]);

        scim_bridge_pdebug (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "%s ", buffers[i + 1]);
    }
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "");

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 4, "message: %s", scim_bridge_message_get_header (message));

    *dst = message;
    return RETVAL_SUCCEEDED;
}


ScimBridgeMessenger *scim_bridge_alloc_messenger (int socket_fd)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 5, "scim_bridge_alloc_messenger");

    ScimBridgeMessenger *messenger = malloc (sizeof (ScimBridgeMessenger));
    messenger->socket_fd = socket_fd;
    return messenger;
}


void scim_bridge_free_messenger (ScimBridgeMessenger *messenger)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 5, "scim_bridge_free_messenger");
    if (messenger == NULL) return;

    scim_bridge_messenger_close (messenger);
    free (messenger);
}


void scim_bridge_messenger_close (ScimBridgeMessenger *messenger)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 5, "scim_bridge_messenger_close");

    if (messenger->socket_fd < 0) return;

    shutdown (messenger->socket_fd, SHUT_RDWR);
    close (messenger->socket_fd);
    messenger->socket_fd = -1;
}


boolean scim_bridge_messenger_is_closed (const ScimBridgeMessenger *messenger)
{
    return messenger->socket_fd < 0;
}
