/*
 * SCIM Bridge
 *
 * Copyright (c) 2006 Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.*
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.*
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

#include <sys/time.h>

#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>

#include "scim-bridge-client-key-event-utility-gtk.h"
#include "scim-bridge-key-event.h"

static GdkKeymap *get_gdk_keymap (GdkWindow *window)
{
#if GDK_MULTIHEAD_SAFE
    if (window) return gdk_keymap_get_for_display (gdk_drawable_get_display (window));
#endif

    return gdk_keymap_get_default ();
}


void scim_bridge_key_event_bridge_to_gdk (GdkEventKey *gdk_key_event, GdkWindow *client_window, const ScimBridgeKeyEvent *key_event)
{
    gdk_key_event->state = 0;
    if (scim_bridge_key_event_is_shift_down (key_event)) gdk_key_event->state |= GDK_SHIFT_MASK;
    if (scim_bridge_key_event_is_caps_lock_down (key_event)) gdk_key_event->state |= GDK_LOCK_MASK;
    if (scim_bridge_key_event_is_control_down (key_event)) gdk_key_event->state |= GDK_CONTROL_MASK;
    if (scim_bridge_key_event_is_alt_down (key_event)) gdk_key_event->state |= GDK_MOD1_MASK;
    /*if (scim_bridge_key_event_is_num_lock_down (key_event)) gdk_key_event->state |= GDK_MOD2_MASK;*/

    if (scim_bridge_key_event_is_pressed (key_event)) {
        gdk_key_event->type = GDK_KEY_PRESS;
    } else {
        gdk_key_event->type = GDK_KEY_RELEASE;
        gdk_key_event->state |= GDK_RELEASE_MASK;
    }

    gdk_key_event->window = client_window;

    struct timeval current_time;
    gettimeofday (&current_time, NULL);

    gdk_key_event->send_event = TRUE;
    gdk_key_event->time = current_time.tv_sec * 1000 + current_time.tv_usec / 1000;
    gdk_key_event->keyval = scim_bridge_key_event_get_code (key_event);
    gdk_key_event->length = 0;
    gdk_key_event->string = 0;

    GdkKeymap *key_map = get_gdk_keymap (gdk_key_event->window);

    GdkKeymapKey *keys;
    gint n_keys;

    if (gdk_keymap_get_entries_for_keyval (key_map, gdk_key_event->keyval, &keys, &n_keys)) {
        gdk_key_event->hardware_keycode = keys[0].keycode;
        gdk_key_event->group = keys [0].group;
    } else {
        gdk_key_event->hardware_keycode = 0;
        gdk_key_event->group = 0;
    }
}


void scim_bridge_key_event_gdk_to_bridge (ScimBridgeKeyEvent *bridge_key_event, GdkWindow *window, const GdkEventKey *key_event)
{
    // Use Key Symbole provided by gtk.
    scim_bridge_key_event_set_code (bridge_key_event, (scim_bridge_key_code_t) key_event->keyval);

    scim_bridge_key_event_clear_modifiers (bridge_key_event);
    if (key_event->state & GDK_SHIFT_MASK || key_event->keyval == GDK_Shift_L || key_event->keyval == GDK_Shift_R) scim_bridge_key_event_set_shift_down (bridge_key_event, TRUE);
    if (key_event->state & GDK_LOCK_MASK || key_event->keyval == GDK_Caps_Lock) scim_bridge_key_event_set_caps_lock_down (bridge_key_event, TRUE);
    if (key_event->state & GDK_CONTROL_MASK || key_event->keyval == GDK_Control_L || key_event->keyval == GDK_Control_R) scim_bridge_key_event_set_control_down (bridge_key_event, TRUE);
    if (key_event->state & GDK_MOD1_MASK || key_event->keyval == GDK_Alt_L || key_event->keyval == GDK_Alt_R) scim_bridge_key_event_set_alt_down (bridge_key_event, TRUE);
    /*if (key_event->state & GDK_MOD2_MASK) scim_bridge_key_event_set_num_lock_down (bridge_key_event, TRUE);*/

    if (key_event->type != GDK_KEY_RELEASE) {
        scim_bridge_key_event_set_pressed (bridge_key_event, TRUE);
    } else {
        scim_bridge_key_event_set_pressed (bridge_key_event, FALSE);
    }
}
