#include <assert.h>
#include <alloca.h>
#include <errno.h>
#include <fcntl.h>
#include <pthread.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/poll.h>
#include <sys/socket.h>
#include <sys/types.h>

#include "scim-bridge-messenger.h"
#include "scim-bridge-output.h"

typedef enum _MessageHeaderCode
{
    CODE_OPEN,
    CODE_CLOSE,
    CODE_WRITE,
} MessageHeaderCode;

typedef struct _MessageHeader
{
    MessageHeaderCode code;

    int value;
} MessageHeader;

typedef enum _Status
{
    STATUS_INITIALIZED,
    STATUS_CONNECTION_ESTABLISHED,
    STATUS_FINALIZED,
    STATUS_ERROR
} Status;

typedef struct _Messenger
{
    Status input_status;
    Status output_status;

    int input_fd;
    int output_fd;

    pthread_mutex_t output_mutex;
    pthread_mutex_t input_mutex;
} Messenger;

typedef struct _MessengerContainer
{
    ScimBridgeMessengerID id;

    ScimBridgeMessengerID prev;
    ScimBridgeMessengerID next;

    Messenger *messenger;
} MessengerContainer;

/* Private variables */

static MessengerContainer *messenger_containers = NULL;
static size_t messenger_capacity;

static ScimBridgeMessengerID free_messenger_container_root;
static ScimBridgeMessengerID used_messenger_container_root;

static pthread_mutex_t messenger_mutex;

/* Helper functions */
static int do_write (ScimBridgeException *except, Messenger *messenger, int input, const void *buf, size_t size)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 1, "scim_bridge_messenger_write: size = %d", size);

    if (size <= 0) return 0;

    Status *status;
    int *fd;
    pthread_mutex_t *mutex;

    if (input) {
        status = &messenger->input_status;
        fd = &messenger->input_fd;
        mutex = &messenger->input_mutex;
    } else {
        status = &messenger->output_status;
        fd = &messenger->output_fd;
        mutex = &messenger->output_mutex;
    }

    pthread_mutex_lock (mutex);
    if (*status != STATUS_CONNECTION_ESTABLISHED) {
        /* No connection/Not permitted */
        scim_bridge_exception_set_errno (except, EPERM);
        scim_bridge_exception_set_message (except, "No connection");

        pthread_mutex_unlock (mutex);
        return -1;
    }

    MessageHeader header;
    header.code = CODE_WRITE;
    header.value = size;

    const void *seek = &header;
    size_t remain_size = sizeof (header);
    while (remain_size > 0) {

        struct pollfd poll_fd;
        poll_fd.revents = 0;
        poll_fd.events = POLLOUT | POLLERR | POLLHUP | POLLNVAL;
        while (! (poll_fd.revents & POLLOUT)) {
            poll_fd.fd = *fd;
            if (poll_fd.fd < 0) {
                scim_bridge_exception_set_errno (except, EIO);
                scim_bridge_exception_set_message (except, "Socket is closed while polling the socket at scim_bridge_messenger_write");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            }
            if (poll (&poll_fd, 1, -1) < 0 && (poll_fd.revents & (POLLERR | POLLHUP | POLLNVAL))) {
                scim_bridge_exception_set_errno (except, errno);
                scim_bridge_exception_set_message (except, "IO exception while polling the socket at scim_bridge_messenger_write");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            }
        }

        const ssize_t write_size = send (*fd, seek, remain_size, MSG_NOSIGNAL);
        if (write_size < 0) {
            if (errno != EAGAIN) {
                scim_bridge_exception_set_errno (except, errno);
                scim_bridge_exception_set_message (except, "IO exception while sending 'WRITE' command at scim_bridge_messenger_write");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            }
        } else {
            remain_size -= write_size;
            seek += write_size;
        }
    }

    seek = buf;
    remain_size = size;
    while (remain_size > 0) {

        struct pollfd poll_fd;
        poll_fd.revents = 0;
        poll_fd.events = POLLOUT | POLLERR | POLLHUP | POLLNVAL;
        while (! (poll_fd.revents & POLLOUT)) {
            poll_fd.fd = *fd;
            if (poll_fd.fd < 0) {
                scim_bridge_exception_set_errno (except, EIO);
                scim_bridge_exception_set_message (except, "Socket is closed while polling the socket at scim_bridge_messenger_write");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            } else if (poll (&poll_fd, 1, -1) < 0 && (poll_fd.revents & (POLLERR | POLLHUP | POLLNVAL))) {
                scim_bridge_exception_set_errno (except, errno);
                scim_bridge_exception_set_message (except, "IO exception while polling the socket at scim_bridge_messenger_write");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            }
        }

        const ssize_t write_size = send (*fd, seek, remain_size, MSG_NOSIGNAL);

        if (write_size < 0) {
            if (errno != EAGAIN) {
                scim_bridge_exception_set_errno (except, errno);
                scim_bridge_exception_set_message (except, "IO exception while sending the data scim_bridge_messenger_write");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            }
        } else {
            remain_size -= write_size;
            seek += write_size;
        }
    }

    pthread_mutex_unlock (mutex);
    return 0;
}


static int do_read (ScimBridgeException *except, Messenger *messenger, int input, void *buf, size_t size)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 1, "scim_bridge_messenger_read: size = %d", size);

    if (size <= 0) return 0;

    Status *status;
    int *fd;
    pthread_mutex_t *mutex;

    if (input) {
        status = &messenger->input_status;
        fd = &messenger->input_fd;
        mutex = &messenger->input_mutex;
    } else {
        status = &messenger->output_status;
        fd = &messenger->output_fd;
        mutex = &messenger->output_mutex;
    }

    pthread_mutex_lock (mutex);
    if (*status != STATUS_CONNECTION_ESTABLISHED) {
        /* No connection/Not permitted */
        scim_bridge_exception_set_errno (except, EPERM);
        scim_bridge_exception_set_message (except, "No connection");

        pthread_mutex_unlock (mutex);
        return -1;
    }

    MessageHeader header;

    void *seek = &header;
    size_t remain_size = sizeof (header);
    while (remain_size > 0) {

        struct pollfd poll_fd;
        poll_fd.revents = 0;
        poll_fd.events = POLLIN | POLLERR | POLLHUP | POLLNVAL;
        while (! (poll_fd.revents & POLLIN)) {
            poll_fd.fd = *fd;
            if (poll_fd.fd < 0) {
                scim_bridge_exception_set_errno (except, EIO);
                scim_bridge_exception_set_message (except, "Socket is closed while polling the socket at scim_bridge_messenger_read");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            } else if (poll (&poll_fd, 1, -1) < 0 && (poll_fd.revents & (POLLERR | POLLHUP | POLLNVAL))) {
                scim_bridge_exception_set_errno (except, errno);
                scim_bridge_exception_set_message (except, "IO exception while polling the socket at scim_bridge_messenger_read");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            }
        }

        const ssize_t read_size = recv (*fd, seek, remain_size, 0);

        if (read_size < 0) {
            if (errno != EAGAIN) {
                scim_bridge_exception_set_errno (except, errno);
                scim_bridge_exception_set_message (except, "IO exception while reading a command at scim_bridge_messenger_read");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            }
        } else {
            remain_size -= read_size;
            seek += read_size;
        }
    }

    if (header.code != CODE_WRITE) {
        scim_bridge_exception_set_errno (except, EPERM);
        scim_bridge_exception_set_message (except, "The opponent won't send me enough data at scim_bridge_messenger_read");

        *status = STATUS_ERROR;
        pthread_mutex_unlock (mutex);
        return -1;
    }

    remain_size = size;
    seek = buf;
    while (remain_size > 0) {

        struct pollfd poll_fd;
        poll_fd.revents = 0;
        poll_fd.events = POLLIN | POLLERR | POLLHUP | POLLNVAL;
        while (! (poll_fd.revents & POLLIN)) {
            poll_fd.fd = *fd;
            if (poll_fd.fd < 0) {
                scim_bridge_exception_set_errno (except, EIO);
                scim_bridge_exception_set_message (except, "Socket is closed while polling the socket at scim_bridge_messenger_read");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            } else if (poll (&poll_fd, 1, -1) < 0 && (poll_fd.revents & (POLLERR | POLLHUP | POLLNVAL))) {
                scim_bridge_exception_set_errno (except, errno);
                scim_bridge_exception_set_message (except, "IO exception while polling the socket at scim_bridge_messenger_read");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            }
        }

        const ssize_t read_size = recv (*fd, seek, remain_size, 0);

        if (read_size < 0) {
            if (errno != EAGAIN) {
                scim_bridge_exception_set_errno (except, EPERM);
                scim_bridge_exception_set_message (except, "The opponent won't send me correct data at scim_bridge_messenger_read");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            }
        } else {
            remain_size -= read_size;
            seek += read_size;
        }
    }

    pthread_mutex_unlock (mutex);
    return 0;
}


static int do_open (ScimBridgeException *except, Messenger *messenger, int input)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 1, "scim_bridge_messenger_open");

    Status *status;
    int *fd;
    pthread_mutex_t *mutex;

    if (input) {
        status = &messenger->input_status;
        fd = &messenger->input_fd;
        mutex = &messenger->input_mutex;
    } else {
        status = &messenger->output_status;
        fd = &messenger->output_fd;
        mutex = &messenger->output_mutex;
    }

    pthread_mutex_lock (mutex);
    if (*status == STATUS_ERROR) {
        /* Error/Not permitted */
        scim_bridge_exception_set_errno (except, EPERM);
        scim_bridge_exception_set_message (except, "The messenger is confused with previous error");

        pthread_mutex_unlock (mutex);
        return -1;
    } else if (*status == STATUS_CONNECTION_ESTABLISHED) {
        /* Already opened/Not permitted */
        scim_bridge_exception_set_errno (except, EPERM);
        scim_bridge_exception_set_message (except, "Already opend");

        pthread_mutex_unlock (mutex);
        return -1;
    } else if (*status == STATUS_FINALIZED) {
        /* Finalized/Not permitted */
        scim_bridge_exception_set_errno (except, EPERM);
        scim_bridge_exception_set_message (except, "The messenger is finalized");

        pthread_mutex_unlock (mutex);
        return -1;
    }

    MessageHeader header;
    header.code = CODE_OPEN;

    void *seek = &header;
    size_t remain_size = sizeof (header);
    while (remain_size > 0) {

        struct pollfd poll_fd;
        poll_fd.revents = 0;
        poll_fd.events = POLLOUT | POLLERR | POLLHUP | POLLNVAL;
        while (! (poll_fd.revents & POLLOUT)) {
            poll_fd.fd = *fd;
            if (poll_fd.fd < 0) {
                scim_bridge_exception_set_errno (except, EIO);
                scim_bridge_exception_set_message (except, "Socket is closed while polling the socket at scim_bridge_messenger_open");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            } else if (poll (&poll_fd, 1, -1) < 0 && (poll_fd.revents & (POLLERR | POLLHUP | POLLNVAL))) {
                scim_bridge_exception_set_errno (except, errno);
                scim_bridge_exception_set_message (except, "IO exception while polling the socket at scim_bridge_messenger_open");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            }
        }

        const ssize_t write_size = send (*fd, seek, remain_size, MSG_NOSIGNAL);

        if (write_size < 0) {
            if (errno != EAGAIN) {
                scim_bridge_exception_set_errno (except, errno);
                scim_bridge_exception_set_message (except, "IO exception while sending 'OPEN' command at scim_bridge_messenger_open");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            }
        } else {
            remain_size -= write_size;
            seek += write_size;
        }
    }

    seek = &header;
    remain_size = sizeof (header);
    while (remain_size > 0) {

        struct pollfd poll_fd;
        poll_fd.revents = 0;
        poll_fd.events = POLLIN | POLLERR | POLLHUP | POLLNVAL;
        while (! (poll_fd.revents & POLLIN)) {
            poll_fd.fd = *fd;
            if (poll_fd.fd < 0) {
                scim_bridge_exception_set_errno (except, EIO);
                scim_bridge_exception_set_message (except, "Socket is closed while polling the socket at scim_bridge_messenger_open");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            } else if (poll (&poll_fd, 1, -1) < 0 && (poll_fd.revents & (POLLERR | POLLHUP | POLLNVAL))) {
                scim_bridge_exception_set_errno (except, errno);
                scim_bridge_exception_set_message (except, "IO exception while polling the socket at scim_bridge_messenger_open");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            }
        }

        const ssize_t read_size = recv (*fd, seek, remain_size, 0);

        if (read_size < 0) {
            if (errno != EAGAIN) {
                scim_bridge_exception_set_errno (except, errno);
                scim_bridge_exception_set_message (except, "IO exception while reading the retval at scim_bridge_messenger_open");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            }
        } else {
            seek += read_size;
            remain_size -= read_size;
        }
    }

    if (header.code == CODE_OPEN) {
        *status = STATUS_CONNECTION_ESTABLISHED;

        pthread_mutex_unlock (mutex);
        return 0;
    } else {
        scim_bridge_exception_set_errno (except, EPERM);
        scim_bridge_exception_set_message (except, "Rejected at scim_bridge_messenger_open");

        pthread_mutex_unlock (mutex);
        return -1;
    }
}


static int do_close (ScimBridgeException *except, Messenger *messenger, int input)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 1, "scim_bridge_messenger_close");

    Status *status;
    int *fd;
    pthread_mutex_t *mutex;

    if (input) {
        status = &messenger->input_status;
        fd = &messenger->input_fd;
        mutex = &messenger->input_mutex;
    } else {
        status = &messenger->output_status;
        fd = &messenger->output_fd;
        mutex = &messenger->output_mutex;
    }

    pthread_mutex_lock (mutex);
    if (*status != STATUS_CONNECTION_ESTABLISHED) {
        /* No connection/Not permitted */
        scim_bridge_exception_set_errno (except, EPERM);
        scim_bridge_exception_set_message (except, "No connection");

        pthread_mutex_unlock (mutex);
        return -1;
    }

    MessageHeader header;
    header.code = CODE_CLOSE;

    void *seek = &header;
    size_t remain_size = sizeof (header);
    while (remain_size > 0) {

        struct pollfd poll_fd;
        poll_fd.revents = 0;
        poll_fd.events = POLLOUT | POLLERR | POLLHUP | POLLNVAL;
        while (! (poll_fd.revents & POLLOUT)) {
            poll_fd.fd = *fd;
            if (poll_fd.fd < 0) {
                scim_bridge_exception_set_errno (except, EIO);
                scim_bridge_exception_set_message (except, "Socket is closed while polling the socket at scim_bridge_messenger_close");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            } else if (poll (&poll_fd, 1, -1) < 0 && (poll_fd.revents & (POLLERR | POLLHUP | POLLNVAL))) {
                scim_bridge_exception_set_errno (except, errno);
                scim_bridge_exception_set_message (except, "IO exception while polling the socket at scim_bridge_messenger_close");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            }
        }

        const ssize_t write_size = send (*fd, seek, remain_size, MSG_NOSIGNAL);

        if (write_size < 0) {
            if (errno != EAGAIN) {
                scim_bridge_exception_set_errno (except, errno);
                scim_bridge_exception_set_message (except, "IO exception is occured while writing for closing at scim_bridge_messenger_close");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            }
        } else {
            remain_size -= write_size;
            seek += write_size;
        }
    }

    seek = &header;
    remain_size = sizeof (header);

    while (remain_size > 0) {

        struct pollfd poll_fd;
        poll_fd.revents = 0;
        poll_fd.events = POLLIN | POLLERR | POLLHUP | POLLNVAL;
        while (! (poll_fd.revents & POLLIN)) {
            poll_fd.fd = *fd;
            if (poll_fd.fd < 0) {
                scim_bridge_exception_set_errno (except, EIO);
                scim_bridge_exception_set_message (except, "Socket is closed while polling the socket at scim_bridge_messenger_close");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            } else if (poll (&poll_fd, 1, -1) < 0 && (poll_fd.revents & (POLLERR | POLLHUP | POLLNVAL))) {
                scim_bridge_exception_set_errno (except, errno);
                scim_bridge_exception_set_message (except, "IO exception while polling the socket at scim_bridge_messenger_close");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            }
        }

        const ssize_t read_size = recv (*fd, seek, remain_size, 0);

        if (read_size < 0) {
            if (errno != EAGAIN) {
                scim_bridge_exception_set_errno (except, errno);
                scim_bridge_exception_set_message (except, "IO exception is occured while waiting for the retval at scim_bridge_messenger_close");

                *status = STATUS_ERROR;
                pthread_mutex_unlock (mutex);
                return -1;
            }
        } else {
            remain_size -= read_size;
            seek += read_size;
        }
    }

    if (header.code != CODE_CLOSE) {
        scim_bridge_exception_set_errno (except, errno);
        scim_bridge_exception_set_message (except, "The opponent won't close at scim_bridge_messenger_close");

        *status = STATUS_ERROR;
        pthread_mutex_unlock (mutex);
        return -1;
    }

    *status = STATUS_INITIALIZED;
    pthread_mutex_unlock (mutex);
    return 0;
}


/* Implementations */
int scim_bridge_initialize_messenger (ScimBridgeException *except, ScimBridgeMessengerID *messenger_id, int new_input_fd, int new_output_fd)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "scim_bridge_initialize_messenger");

    if (messenger_containers == NULL) {
        pthread_mutex_init (&messenger_mutex, NULL);
        messenger_capacity = 0;
        used_messenger_container_root = -1;
        free_messenger_container_root = -1;
    }

    pthread_mutex_lock (&messenger_mutex);
    if (free_messenger_container_root == -1) {
        const size_t new_messenger_capacity = messenger_capacity + 5;
        messenger_containers = realloc (messenger_containers, sizeof (MessengerContainer) * new_messenger_capacity);

        int i;
        for (i = messenger_capacity; i < new_messenger_capacity; ++i) {
            MessengerContainer *container = &messenger_containers[i];

            container->id = i;

            if (i > messenger_capacity) container->prev = i - 1;
            else container->prev = -1;
            if (i < new_messenger_capacity - 1) container->next = i + 1;
            else container->next = -1;

            Messenger *messenger = malloc (sizeof (Messenger));

            pthread_mutex_init (&messenger->input_mutex, NULL);
            pthread_mutex_init (&messenger->output_mutex, NULL);

            messenger->input_status = STATUS_FINALIZED;
            messenger->output_status = STATUS_FINALIZED;

            container->messenger = messenger;
        }

        free_messenger_container_root = messenger_capacity;
        messenger_capacity = new_messenger_capacity;
    }

    MessengerContainer *container = &messenger_containers[free_messenger_container_root];
    Messenger *messenger = container->messenger;

    if (container->prev != -1) messenger_containers[container->prev].next = container->next;
    if (container->next != -1) messenger_containers[container->next].prev = container->prev;

    free_messenger_container_root = container->next;

    container->prev = -1;
    container->next = used_messenger_container_root;
    if (used_messenger_container_root != -1) messenger_containers[used_messenger_container_root].prev = container->id;
    used_messenger_container_root = container->id;

    pthread_mutex_lock (&messenger->input_mutex);
    pthread_mutex_lock (&messenger->output_mutex);
    if (messenger->input_status != STATUS_FINALIZED || messenger->output_status != STATUS_FINALIZED) {
        /* Reinitialization/Not permitted */
        scim_bridge_exception_set_errno (except, EPERM);
        scim_bridge_exception_set_message (except, "Already initialized");

        pthread_mutex_unlock (&messenger->input_mutex);
        pthread_mutex_unlock (&messenger->output_mutex);

        return -1;
    }

    int input_fd_options = fcntl (new_input_fd, F_GETFL);
    int output_fd_options = fcntl (new_output_fd, F_GETFL);
    int input_fcntl_retval = fcntl (new_input_fd, F_SETFL, input_fd_options | O_NONBLOCK);
    int output_fcntl_retval = fcntl (new_output_fd, F_SETFL, output_fd_options | O_NONBLOCK);
    if (input_fd_options < 0 || input_fcntl_retval < 0 || output_fd_options < 0 || output_fcntl_retval < 0) {
        scim_bridge_exception_set_errno (except, EPERM);
        scim_bridge_exception_set_message (except, "Failed to change the FD flags of the sockets");

        pthread_mutex_unlock (&messenger->input_mutex);
        pthread_mutex_unlock (&messenger->output_mutex);

        return -1;
    }

    messenger->input_fd = new_input_fd;
    messenger->output_fd = new_output_fd;

    messenger->input_status = STATUS_INITIALIZED;
    messenger->output_status = STATUS_INITIALIZED;

    pthread_mutex_unlock (&messenger->input_mutex);
    pthread_mutex_unlock (&messenger->output_mutex);

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 1, "Initialize done..");

    pthread_mutex_unlock (&messenger_mutex);

    *messenger_id = container->id;
    return 0;
}


int scim_bridge_finalize_messenger (ScimBridgeException *except, ScimBridgeMessengerID messenger_id)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "scim_bridge_finalize_messenger");

    if (messenger_containers == NULL) {
        /* No such messenger/Not permitted */
        scim_bridge_exception_set_errno (except,  EPERM);
        scim_bridge_exception_set_message (except, "No such messenger");
        return -1;
    }

    pthread_mutex_lock (&messenger_mutex);

    if (messenger_id < 0 || messenger_id >= messenger_capacity) {
        /* No such messenger/Not permitted */
        scim_bridge_exception_set_errno (except,  EPERM);
        scim_bridge_exception_set_message (except, "No such messenger");

        pthread_mutex_unlock (&messenger_mutex);
        return -1;
    }

    MessengerContainer *container = &messenger_containers[messenger_id];
    Messenger *messenger = container->messenger;

    if (messenger->input_status == STATUS_FINALIZED && messenger->output_status == STATUS_FINALIZED) {
        /* Not initialized/Not permitted */
        scim_bridge_exception_set_errno (except,  EPERM);
        scim_bridge_exception_set_message (except, "Not initialized");

        pthread_mutex_unlock (&messenger_mutex);
        return -1;
    }

    int fd;

    fd = messenger->input_fd;
    messenger->input_fd = -1;
    shutdown (fd, SHUT_RDWR);
    close (fd);

    fd = messenger->output_fd;
    messenger->output_fd = -1;
    shutdown (fd, SHUT_RDWR);
    close (fd);

    while (pthread_mutex_destroy (&messenger->input_mutex)) {
        usleep (100);
    }
    while (pthread_mutex_destroy (&messenger->output_mutex)) {
        usleep (100);
    }

    messenger->input_status = STATUS_FINALIZED;
    messenger->output_status = STATUS_FINALIZED;

    pthread_mutex_unlock (&messenger_mutex);
    return 0;
}


int scim_bridge_messenger_open_input (ScimBridgeException *except, ScimBridgeMessengerID messenger_id)
{
    pthread_mutex_lock (&messenger_mutex);
    if (messenger_id < 0 || messenger_id >= messenger_capacity) {
        /* No such messenger/Not permitted */
        scim_bridge_exception_set_errno (except,  EPERM);
        scim_bridge_exception_set_message (except, "No such messenger");

        pthread_mutex_unlock (&messenger_mutex);
        return -1;
    }

    Messenger *messenger = messenger_containers[messenger_id].messenger;
    pthread_mutex_unlock (&messenger_mutex);
    return do_open (except, messenger, 1);
}


int scim_bridge_messenger_close_input (ScimBridgeException *except, ScimBridgeMessengerID messenger_id)
{
    pthread_mutex_lock (&messenger_mutex);
    if (messenger_id < 0 || messenger_id >= messenger_capacity) {
        /* No such messenger/Not permitted */
        scim_bridge_exception_set_errno (except,  EPERM);
        scim_bridge_exception_set_message (except, "No such messenger");

        pthread_mutex_unlock (&messenger_mutex);
        return -1;
    }

    Messenger *messenger = messenger_containers[messenger_id].messenger;
    pthread_mutex_unlock (&messenger_mutex);
    return do_close (except, messenger, 1);
}


int scim_bridge_messenger_write_input (ScimBridgeException *except, ScimBridgeMessengerID messenger_id, const void *buf, size_t size)
{
    pthread_mutex_lock (&messenger_mutex);
    if (messenger_id < 0 || messenger_id >= messenger_capacity) {
        /* No such messenger/Not permitted */
        scim_bridge_exception_set_errno (except,  EPERM);
        scim_bridge_exception_set_message (except, "No such messenger");

        pthread_mutex_unlock (&messenger_mutex);
        return -1;
    }

    Messenger *messenger = messenger_containers[messenger_id].messenger;
    pthread_mutex_unlock (&messenger_mutex);
    return do_write (except, messenger, 1, buf, size);
}


int scim_bridge_messenger_read_input (ScimBridgeException *except, ScimBridgeMessengerID messenger_id, void *buf, size_t size)
{
    pthread_mutex_lock (&messenger_mutex);
    if (messenger_id < 0 || messenger_id >= messenger_capacity) {
        /* No such messenger/Not permitted */
        scim_bridge_exception_set_errno (except,  EPERM);
        scim_bridge_exception_set_message (except, "No such messenger");

        pthread_mutex_unlock (&messenger_mutex);
        return -1;
    }

    Messenger *messenger = messenger_containers[messenger_id].messenger;
    pthread_mutex_unlock (&messenger_mutex);
    return do_read (except, messenger, 1, buf, size);
}


int scim_bridge_messenger_open_output (ScimBridgeException *except, ScimBridgeMessengerID messenger_id)
{
    pthread_mutex_lock (&messenger_mutex);
    if (messenger_id < 0 || messenger_id >= messenger_capacity) {
        /* No such messenger/Not permitted */
        scim_bridge_exception_set_errno (except,  EPERM);
        scim_bridge_exception_set_message (except, "No such messenger");

        pthread_mutex_unlock (&messenger_mutex);
        return -1;
    }

    Messenger *messenger = messenger_containers[messenger_id].messenger;
    pthread_mutex_unlock (&messenger_mutex);
    return do_open (except, messenger, 0);
}


int scim_bridge_messenger_close_output (ScimBridgeException *except, ScimBridgeMessengerID messenger_id)
{
    pthread_mutex_lock (&messenger_mutex);
    if (messenger_id < 0 || messenger_id >= messenger_capacity) {
        /* No such messenger/Not permitted */
        scim_bridge_exception_set_errno (except,  EPERM);
        scim_bridge_exception_set_message (except, "No such messenger");

        pthread_mutex_unlock (&messenger_mutex);
        return -1;
    }

    Messenger *messenger = messenger_containers[messenger_id].messenger;
    pthread_mutex_unlock (&messenger_mutex);
    return do_close (except, messenger, 0);
}


int scim_bridge_messenger_write_output (ScimBridgeException *except, ScimBridgeMessengerID messenger_id, const void *buf, size_t size)
{
    pthread_mutex_lock (&messenger_mutex);
    if (messenger_id < 0 || messenger_id >= messenger_capacity) {
        /* No such messenger/Not permitted */
        scim_bridge_exception_set_errno (except,  EPERM);
        scim_bridge_exception_set_message (except, "No such messenger");

        pthread_mutex_unlock (&messenger_mutex);
        return -1;
    }

    Messenger *messenger = messenger_containers[messenger_id].messenger;
    pthread_mutex_unlock (&messenger_mutex);
    return do_write (except, messenger, 0, buf, size);
}


int scim_bridge_messenger_read_output (ScimBridgeException *except, ScimBridgeMessengerID messenger_id, void *buf, size_t size)
{
    pthread_mutex_lock (&messenger_mutex);
    if (messenger_id < 0 || messenger_id >= messenger_capacity) {
        /* No such messenger/Not permitted */
        scim_bridge_exception_set_errno (except,  EPERM);
        scim_bridge_exception_set_message (except, "No such messenger");

        pthread_mutex_unlock (&messenger_mutex);
        return -1;
    }

    Messenger *messenger = messenger_containers[messenger_id].messenger;
    pthread_mutex_unlock (&messenger_mutex);
    return do_read (except, messenger, 0, buf, size);\
}
