#include <assert.h>
#include <errno.h>

#include "scim-bridge-client-kernel.h"
#include "scim-bridge-client-kernel-protected.h"
#include "scim-bridge-client-messenger.h"
#include "scim-bridge-imcontext-manager.h"

/* Private variables */
static int initialized = 0;

/* Implementations */
int scim_bridge_client_initialize_kernel (ScimBridgeException *except)
{
    scim_bridge_initialize_imcontext_manager ();

    if (scim_bridge_client_initialize_messenger (except))
        return -1;

    initialized = 1;
    return 0;
}


int scim_bridge_client_finalize_kernel (ScimBridgeException *except)
{
    if (scim_bridge_client_finalize_messenger (except))
        return -1;

    if (scim_bridge_client_kernel_cleanup (except))
        return -1;

    return 0;
}


int scim_bridge_client_kernel_cleanup (ScimBridgeException *except)
{
    scim_bridge_client_kernel_impl_cleanup ();

    scim_bridge_finalize_imcontext_manager ();

    scim_bridge_client_kernel_impl_finalized ();

    initialized = 0;
    return 0;
}


int scim_bridge_client_trigger_kernel_finalizer (ScimBridgeException *except)
{
    return scim_bridge_client_finalize_messenger (except);
}


int scim_bridge_client_kernel_alloc_imcontext (ScimBridgeException *except, ScimBridgeIMContext *ic)
{
    if (!initialized) {
        scim_bridge_exception_set_errno (except, EPERM);
        scim_bridge_exception_set_message (except, "The kernel is not initialized");

        return -1;
    }

    scim_bridge_add_imcontext (ic);
    ScimBridgeIMContextID opponent_id;
    if (scim_bridge_client_call_alloc_imcontext (except, ic->parent.id, &opponent_id)) {
        scim_bridge_remove_imcontext (ic);
        return -1;
    } else if (opponent_id < 0) {
        scim_bridge_exception_set_message (except, "Alloc message is refused");
        scim_bridge_exception_set_errno (except, EPERM);
        scim_bridge_remove_imcontext (ic);
        return -1;
    }

    ic->parent.opponent_id = opponent_id;

    return 0;
}


int scim_bridge_client_kernel_free_imcontext (ScimBridgeException *except, ScimBridgeIMContext *ic)
{
    if (!initialized) {
        scim_bridge_exception_set_errno (except, EPERM);
        scim_bridge_exception_set_message (except, "The kernel is not initialized");

        return -1;
    }

    if (scim_bridge_client_call_free_imcontext (except, ic->parent.opponent_id))
        return -1;

    ic->parent.opponent_id = -1;
    scim_bridge_remove_imcontext (ic);

    return 0;
}


ScimBridgeIMContext *scim_bridge_client_kernel_get_first_imcontext ()
{
    return scim_bridge_get_first_imcontext ();
}


ScimBridgeIMContext *scim_bridge_client_kernel_get_next_imcontext (ScimBridgeIMContext *imcontext)
{
    return scim_bridge_get_next_imcontext (imcontext);
}


int scim_bridge_client_kernel_focus_changed (ScimBridgeException *except, ScimBridgeIMContext *ic, int focus_in)
{
    if (!initialized) {
        scim_bridge_exception_set_errno (except, EPERM);
        scim_bridge_exception_set_message (except, "The kernel is not initialized");

        return -1;
    }

    if (!focus_in) {
        scim_bridge_client_kernel_set_preedit_shown (ic->parent.id, 0);
        scim_bridge_client_kernel_update_preedit (ic->parent.id);
    }
    return scim_bridge_client_call_focus_changed (except, ic->parent.opponent_id, focus_in);
}


int scim_bridge_client_kernel_reset_imcontext (ScimBridgeException *except, ScimBridgeIMContext *ic)
{
    if (!initialized) {
        scim_bridge_exception_set_errno (except, EPERM);
        scim_bridge_exception_set_message (except, "The kernel is not initialized");

        return -1;
    }

    return scim_bridge_client_call_reset_imcontext (except, ic->parent.opponent_id);
}


int scim_bridge_client_kernel_cursor_location_changed (ScimBridgeException *except, ScimBridgeIMContext *ic, int cursor_x, int cursor_y)
{
    if (!initialized) {
        scim_bridge_exception_set_errno (except, EPERM);
        scim_bridge_exception_set_message (except, "The kernel is not initialized");

        return -1;
    }

    return scim_bridge_client_call_cursor_location_changed (except, ic->parent.opponent_id, cursor_x, cursor_y);
}


int scim_bridge_client_kernel_keyevent_occured (ScimBridgeException *except, ScimBridgeIMContext *ic, const ScimBridgeKeyEvent *keyevent, int *consumed)
{
    if (!initialized) {
        scim_bridge_exception_set_errno (except, EPERM);
        scim_bridge_exception_set_message (except, "The kernel is not initialized");

        return -1;
    }

    return scim_bridge_client_call_keyevent_occured (except, ic->parent.opponent_id, keyevent, consumed);
}


void scim_bridge_client_kernel_commit (ScimBridgeIMContextID id)
{
    if (!initialized) return;

    ScimBridgeIMContext *ic = scim_bridge_find_imcontext (id);

    if (ic != NULL) {
        assert (ic->parent.opponent_id != -1);
        scim_bridge_client_kernel_impl_commit (ic);
    }
}


void scim_bridge_client_kernel_set_preedit_string (ScimBridgeIMContextID id, ucs4_t *wstr, size_t wstr_len)
{
    if (!initialized) return;

    ScimBridgeIMContext *ic = scim_bridge_find_imcontext (id);

    if (ic != NULL) {
        assert (ic->parent.opponent_id != -1);
        scim_bridge_client_kernel_impl_set_preedit_string (ic, wstr, wstr_len);
    }
}


void scim_bridge_client_kernel_set_preedit_attributes (ScimBridgeIMContextID id, ScimBridgeAttribute *attrs, size_t attr_length)
{
    if (!initialized) return;

    ScimBridgeIMContext *ic = scim_bridge_find_imcontext (id);

    if (ic != NULL) {
        assert (ic->parent.opponent_id != -1);
        scim_bridge_client_kernel_impl_set_preedit_attributes (ic, attrs, attr_length);
    }
}


void scim_bridge_client_kernel_set_preedit_cursor_position (ScimBridgeIMContextID id, int cursor_position)
{
    if (!initialized) return;

    ScimBridgeIMContext *ic = scim_bridge_find_imcontext (id);

    if (ic != NULL) {
        assert (ic->parent.opponent_id != -1);
        scim_bridge_client_kernel_impl_set_preedit_cursor_position (ic, cursor_position);
    }
}


void scim_bridge_client_kernel_set_preedit_shown (ScimBridgeIMContextID id, int shown)
{
    if (!initialized) return;

    ScimBridgeIMContext *ic = scim_bridge_find_imcontext (id);

    if (ic != NULL) {
        assert (ic->parent.opponent_id != -1);
        scim_bridge_client_kernel_impl_set_preedit_shown (ic, shown);
    }
}


void scim_bridge_client_kernel_update_preedit (ScimBridgeIMContextID id)
{
    if (!initialized) return;

    ScimBridgeIMContext *ic = scim_bridge_find_imcontext (id);

    if (ic != NULL) {
        assert (ic->parent.opponent_id != -1);
        scim_bridge_client_kernel_impl_update_preedit (ic);
    }
}


void scim_bridge_client_kernel_forward_keyevent (ScimBridgeIMContextID id, const ScimBridgeKeyEvent *keyevent)
{
    if (!initialized) return;

    ScimBridgeIMContext *ic = scim_bridge_find_imcontext (id);

    if (ic != NULL) {
        assert (ic->parent.opponent_id != -1);
        scim_bridge_client_kernel_impl_forward_keyevent (ic, keyevent);
    }
}


void scim_bridge_client_kernel_beep (ScimBridgeIMContextID id)
{
    if (!initialized) return;

    ScimBridgeIMContext *ic = scim_bridge_find_imcontext (id);

    if (ic != NULL) {
        assert (ic->parent.opponent_id != -1);
        scim_bridge_client_kernel_impl_beep (ic);
    }
}


void scim_bridge_client_kernel_get_surrounding_string (ScimBridgeIMContextID id, ucs4_t *wstr, size_t max_length, size_t *fetch_length, int *cursor_position)
{
    if (!initialized) {
        *fetch_length = 0;
        return;
    }

    ScimBridgeIMContext *ic = scim_bridge_find_imcontext (id);

    if (ic != NULL) {
        assert (ic->parent.opponent_id != -1);
        scim_bridge_client_kernel_impl_get_surrounding_string (ic, wstr, max_length, fetch_length, cursor_position);
    }
}


void scim_bridge_client_kernel_delete_surrounding_string (ScimBridgeIMContextID id, size_t offset, size_t length, int *retval)
{
    if (!initialized) return;

    ScimBridgeIMContext *ic = scim_bridge_find_imcontext (id);

    if (ic != NULL) {
        assert (ic->parent.opponent_id != -1);
        scim_bridge_client_kernel_impl_delete_surrounding_string (ic, offset, length, retval);
    }
}


void scim_bridge_client_kernel_exception_occured (ScimBridgeException *except)
{
    scim_bridge_client_kernel_impl_exception_occured (except);
}
