#include <alloca.h>
#include <assert.h>
#include <malloc.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>

#include "scim-bridge-agent-output.h"
#include "scim-bridge-environment.h"

static bool syslog_enabled = false;

static const char SCIM_BRIDGE_AGENT_PROCESS_NAME[] = "scim-bridge-agent";

/* Helper functions */
static void vxfprintf (FILE *stream, const char *format, va_list ap)
{
    const size_t format_len = strlen (format);
    char *new_format = static_cast <char*> (alloca (sizeof (char) * (format_len + 1)));
    strcpy (new_format, format);
    new_format[format_len] = '\0';
    vfprintf (stream, new_format, ap);
}


static void vxfprintfln (FILE *stream, const char *format, va_list ap)
{
    const size_t format_len = strlen (format);
    char *new_format = static_cast <char*> (alloca (sizeof (char) * (format_len + 2)));
    strcpy (new_format, format);
    new_format[format_len] = '\n';
    new_format[format_len + 1] = '\0';
    vfprintf (stream, new_format, ap);
}


static void vxsyslog (int priority, const char *format, va_list ap)
{
    char *expanded_format;
    vasprintf (&expanded_format, format, ap);
    syslog (priority, "%s", expanded_format);
    free (expanded_format);
}


/* Implementations */
bool scim_bridge_agent_is_syslog_enabled ()
{
    return syslog_enabled;
}


void scim_bridge_agent_enable_syslog ()
{
    if (!syslog_enabled) {
        setlogmask (LOG_UPTO (LOG_INFO));
        closelog ();
        openlog (SCIM_BRIDGE_AGENT_PROCESS_NAME, LOG_NDELAY, LOG_DAEMON);
        syslog_enabled = true;
    }
}


void scim_bridge_agent_disable_syslog ()
{
    if (syslog_enabled) {
        closelog ();
        syslog_enabled = false;
    }
}


void scim_bridge_pdebugln (ScimBridgeDebugFlag flag, ScimBridgeDebugLevel level, const char* format,...)
{
    if ((flag & scim_bridge_environment_get_debug_flags ()) && ((10 - level) <= scim_bridge_environment_get_debug_level ())) {
        va_list ap;
        va_start (ap, format);
        vxfprintfln (stdout, format, ap);
        va_end (ap);
    }
}


void scim_bridge_pdebug (ScimBridgeDebugFlag flag, ScimBridgeDebugLevel level, const char* format,...)
{
    if ((flag & scim_bridge_environment_get_debug_flags ()) && ((10 - level) <= scim_bridge_environment_get_debug_level ())) {
        va_list ap;
        va_start (ap, format);
        vxfprintf (stdout, format, ap);
        va_end (ap);
    }
}


void scim_bridge_perrorln (const char* format,...)
{
    va_list ap;
    va_start (ap, format);
    if (syslog_enabled) vxsyslog (LOG_ERR, format, ap);
    va_end (ap);
    va_start (ap, format);
    vxfprintfln (stdout, format, ap);
    va_end (ap);
}


void scim_bridge_perror (const char* format,...)
{
    va_list ap;
    va_start (ap, format);
    if (syslog_enabled) vxsyslog (LOG_ERR, format, ap);
    va_end (ap);
    va_start (ap, format);
    vxfprintf (stderr, format, ap);
    va_end (ap);
}


void scim_bridge_println (const char* format,...)
{
    va_list ap;
    va_start (ap, format);
    if (syslog_enabled) vxsyslog (LOG_NOTICE, format, ap);
    va_end (ap);
    va_start (ap, format);
    vxfprintfln (stdout, format, ap);
    va_end (ap);
}


void scim_bridge_print (const char* format,...)
{
    va_list ap;
    va_start (ap, format);
    if (syslog_enabled) vxsyslog (LOG_NOTICE, format, ap);
    va_end (ap);
    va_start (ap, format);
    vxfprintf (stdout, format, ap);
    va_end (ap);
}
