//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function CL_plot_earthMap(win_id,color_id,tick_steps,data_bounds,thickness,res,coord)
// Plots an Earth map
//
// Calling Sequence
// CL_plot_earthMap([win_id,color_id,tick_steps,data_bounds,thickness,res,coord])
//
// Description
// <itemizedlist><listitem>
// <p>Plots an Earth map in the selected figure (win_id). </p>
// <p>The x-axis and y-axis are respectively the longitude and latitude 
// in degrees. </p>
// <p> Depending on type of coordinates ("coord" option), the longitude is considered 
// to be the spherical or elliptical (geodetic) longitude. </p>
// <p>There are 2 available resolutions: </p>
// <p>- "low": outlines of Earth continents and major islands (low resolution) </p>
// <p>- "high": outlines of Earth continents, islands and lakes</p>
// </listitem></itemizedlist>
//
// Parameters
// win_id: (optional) Figure Id - Id of the current window if omitted.
// color_id: (optional) Color index - Default is 2.
// tick_steps: (optional) Steps of the grid in longitude and latitude in degrees. Default is [60 ,30].
// data_bounds: (optional) Definition of the view area: [longmin, latmin; longmax, latmax] in degrees. Default is [-180, -90; 180, 90].
// thickness: (optional) Line thickness. Default is 1.
// res: (optional) (string) Resolution: "low" or "high". Default is "low" (1x1)
// coord: (optional) (string) Type of coordinates: "sph"=spherical, "ell"=elliptical. Default is "sph" (1x1)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_plot_ephem
//
// Examples
// // basic plot
// CL_plot_earthMap();
//
// // plot in figure 10, with color 2
// CL_plot_earthMap(win_id=10, color_id=2);
//
// // plot in figure 10, with color 1, and focus on Europe
// CL_plot_earthMap(win_id=10, color_id=1, ..
// data_bounds=[-10,30;40,60], tick_steps=[10,10]);
//
// // plot in high res, elliptical coordinates and focus on Europe
// CL_plot_earthMap(color_id=5,data_bounds=[-10,30;40,60], ..
// tick_steps=[10,10], res = "high", coord = "ell");


function [M] = read_earthmap(id)
// Loads the map from a file (if not been loaded yet).
// Storage in global variable %CL__earthMap 
// (as a matrix of short integers to save memory)
// accuracy is about 6.e-3 degrees
// id: (string) map identifier
// M: (2xN) matrix of doubles (lon/lat) in degrees 
// (%nan between blocks)

  // map stored (structure)
  global %CL__earthMap;

  // F = conversion factor: [-1,1] to short integers
  // NAN = integer value representing %nan
  F = 2^15 - 1; 
  NAN = int16(-(2^15)); 

  // initialize structure (in case)
  if (%CL__earthMap == []); %CL__earthMap=struct(); end

  // load map if it hasn't been loaded yet
  if ~isfield(%CL__earthMap,id) 
    // file path
    fname = fullfile(%CL_home, 'data', 'utils', 'earthMap' + '_' + id + '.pla');
    if (isfile(fname))
      earthMap = (fscanfMat(fname))'; 
    else
      CL__error("File: " + fname + ": not found"); 
    end

    // conversion to integers and storage
    %CL__earthMap(id) = int16(round(min(max(earthMap/180,-1),1)*F));
    I = find(earthMap >= 900.);
    %CL__earthMap(id)(I) = NAN; 
  end
  
  // convert map to double (degrees)
  M = double(%CL__earthMap(id)) * 180 / F;
  I = find(%CL__earthMap(id) == NAN);
  M(I) = %nan;
      
endfunction
  

// Declarations:
if(~exists('%CL_home')) then global %CL_home; end;

// Global variable for map storage (used in sub-function) 
global %CL__earthMap;

// Code:

// CODE FOR COMPATIBILITY
if exists('winId','local'); win_id=winId; end
if exists('colorId','local'); color_id=colorId; end
if exists('tickInterval','local'); tick_steps=tickInterval; end
if exists('dataBounds','local'); data_bounds=dataBounds; end


if exists('win_id','local') then
  f = scf(win_id)
else
  f = gcf();
end

if ~exists('color_id','local') then
  color_id = 2;
end

if ~exists('tick_steps','local') then
  tick_steps = [60, 30]; // degrees
end

if ~exists('data_bounds','local') then
  data_bounds = [-180, -90; 180, 90]; // degrees
end

if ~exists('thickness','local'); thickness = 1; end

if ~exists('res','local'); res = "low"; end 

if ~exists('coord','local'); coord = "sph"; end 

// error checking

if (size(data_bounds,1) <> 2 | size(data_bounds,2) <> 2)
  CL__error('Invalid size for data_bounds');
end
if (data_bounds(2,1)<=data_bounds(1,1) | data_bounds(2,1)>data_bounds(1,1)+360)
  CL__error('Invalid data bounds in longitude');
end 
if (data_bounds(2,2)<=data_bounds(1,2) | data_bounds(1,2) < -90 | data_bounds(2,2) > 90)
  CL__error('Invalid data bounds in latitude');
end 

if (length(tick_steps) <> 2)
  CL__error('Invalid size for tick_steps');
end 
if (tick_steps(1) <= 0 | tick_steps(2) <= 0)
  CL__error('Invalid values for tick_steps');
end 

// plot
immediate_drawing_save = f.immediate_drawing; // store field
f.immediate_drawing = "off"; 

// loading of the map (separations of blocks = %nan)
M = read_earthmap(coord+'_'+res);

// interval in longitude containing the view
bmin = (data_bounds(1,1)+data_bounds(2,1))/2 - 180; 
bmax = (data_bounds(1,1)+data_bounds(2,1))/2 + 180; 

// beginnings / ends of blocks (%nan separations)
I = [0, find(isnan(M(1,:))), size(M,2)+1]; // beginning/end added

for k = 1 : length(I)-1
  ind = I(k)+1 : I(k+1)-1; 
   
  if (length(ind) >= 2)
    liste_seg = CL__plot_contSegsList(M(1,ind), M(2,ind), bmin, bmax);   
    for (seg = liste_seg) 
      plot2d(seg(1,:), seg(2,:), style=color_id);
      e = gce();
      e.children(1).thickness = thickness; 
    end
  end
end

// adjustments (grid, databounds)
a = gca();
a.data_bounds = data_bounds;
a.tight_limits = "on";

bticks = [ tick_steps(1) * floor(data_bounds(1,:)/tick_steps(1)); 
           tick_steps(2) * ceil((data_bounds(2,:))/tick_steps(2)) ];

x_ticks = bticks(1,1) : tick_steps(1) : bticks(2,1);
y_ticks = bticks(1,2) : tick_steps(2) : bticks(2,2);

// strsplit+msprintf : used to avoid "string"
x_ticks_labels = strsplit( stripblanks( msprintf("%.8g ",x_ticks')) , " ")' ; 
y_ticks_labels = strsplit( stripblanks( msprintf("%.8g ",y_ticks')) , " ")' ;

a.x_ticks = tlist(["ticks", "locations", "labels"], x_ticks, x_ticks_labels ); 
a.y_ticks = tlist(["ticks", "locations", "labels"], y_ticks, y_ticks_labels ); 

CL_g_stdaxes(a);

f.immediate_drawing = immediate_drawing_save; // restore field

endfunction


