//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [om,omp] = CL_rot_angularVelocity(naxes,angles,angles_der,angles_der2)
// Rotation angles to angular velocity and acceleration vectors
//
// Calling Sequence
// [om,omp] = CL_rot_angularVelocity(naxes,angles,angles_der,angles_der2)
//
// Description
// <itemizedlist><listitem>
// <p>Computes the angular rotation and acceleration vectors resulting from a combination of 3 
// rotations.</p> 
// <p>The rotation axes are defined by numbers: 1=x-axis, 2=y-axis, 3=z-axis.</p>  
// </listitem>
// </itemizedlist>
//
// Parameters
// naxes : Axes numbers: 1=x-axis, 2=y-axis or 3=z-axis (1x3 or 3x1)
// angles : Rotation angles around respective axes [rad] (3xN) 
// angles_der : First derivatives of angles with respect to time [rad/s] (3xN) 
// angles_der2 : Second derivatives of angles with respect to time [rad/s^2] (3xN) 
// om : Angular rotation vector (3xN)
// omp : Angular acceleration vector (3xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_rot_defQuat
// CL_rot_matrix2quat
//
// Examples
// naxes = [1,2,3]; // XYZ
// angles = [%pi/4 ; %pi/8 ; %pi/6]
// angles_der = [%pi/4 ; %pi/8 ; %pi/6]
// angles_der2 = [%pi/4 ; %pi/8 ; %pi/6]
// [om,omp] = CL_rot_angularVelocity(naxes,angles,angles_der,angles_der2)

// Declarations:


// Code:

[lhs,rhs]=argn();
if rhs~=4 CL__error("check number of input arguments"); end,

phi=angles(1,:);
theta=angles(2,:);
psi=angles(3,:);

phip=angles_der(1,:);
thetap=angles_der(2,:);
psip=angles_der(3,:);

phipp=angles_der2(1,:);
thetapp=angles_der2(2,:);
psipp=angles_der2(3,:);

// correspondance
// axis sequence - index value - kinematic equations of motions
set_sequences = [[1,2,3];[2,3,1]; [3,1,2]; [1,3,2]; [3,2,1]; [2,1,3]; [1,2,1];[2,3,2]; [3,1,3]; [1,3,1]; [3,2,3]; [2,1,2]];

nn = find(naxes(1) == set_sequences(:,1) & naxes(2) == set_sequences(:,2) & naxes(3) == set_sequences(:,3));

if (nn>=1 & nn<=3)
  transformation="A";
elseif (nn>=4 & nn<=6)
  transformation="B"
elseif (nn>=7 & nn<=9)
  transformation="C";
elseif (nn>=10 & nn<=12)
  transformation="D";
else
  CL__error("Unknown rotation order");
end


om = zeros(3,size(theta,2));
omp = zeros(3,size(theta,2));

I1 = naxes(1);
I2 = naxes(2);
I3 = naxes(3);

select transformation

  case "A"

    om(I1,:)  = phip.*cos(theta).*cos(psi) + thetap.*sin(psi);
    om(I2,:)  = -phip.*cos(theta).*sin(psi) + thetap.*cos(psi);
    om(I3,:)  = psip + phip.*sin(theta);

    omp(I1,:) = phipp.*cos(theta).*cos(psi) - phip.*thetap.*sin(theta).*cos(psi) - phip.*cos(theta).*psip.*sin(psi) + ...
        thetapp.*sin(psi) + thetap.*psip.*cos(psi);
    omp(I2,:) = -phipp.*cos(theta).*sin(psi) + phip.*thetap.*sin(theta).*sin(psi) - phip.*cos(theta).*psip.*cos(psi) + ...
        thetapp.*cos(psi) - thetap.*psip.*sin(psi);
    omp(I3,:) = psipp + phipp.*sin(theta) + phip.*thetap.*cos(theta);

  case "B"

    om(I1,:)  = phip.*cos(theta).*cos(psi) - thetap.*sin(psi);
    om(I2,:)  = phip.*cos(theta).*sin(psi) + thetap.*cos(psi);
    om(I3,:)  = psip - phip.*sin(theta);

    omp(I1,:) = phipp.*cos(theta).*cos(psi) - phip.*thetap.*sin(theta).*cos(psi) - phip.*cos(theta).*psip.*sin(psi) - ...
        thetapp.*sin(psi) - thetap.*psip.*cos(psi);
    omp(I2,:) = phipp.*cos(theta).*sin(psi) - phip.*thetap.*sin(theta).*sin(psi) + phip.*cos(theta).*psip.*cos(psi) + ...
        thetapp.*cos(psi) - thetap.*psip.*sin(psi);
    omp(I3,:) = psipp - phipp.*sin(theta) - phip.*thetap.*cos(theta);

  case "C"

    om(I3,:)  = phip.*sin(theta).*cos(psi) - thetap.*sin(psi);
    om(I2,:)  = phip.*sin(theta).*sin(psi) + thetap.*cos(psi);
    om(I1,:)  = psip - phip.*cos(theta);

    omp(I3,:) = phipp.*sin(theta).*cos(psi) + phip.*thetap.*cos(theta).*cos(psi) - phip.*sin(theta).*psip.*sin(psi) - ...
        thetapp.*sin(psi) - thetap.*psip.*cos(psi);
    omp(I2,:) = phipp.*sin(theta).*sin(psi) + phip.*thetap.*cos(theta).*sin(psi) + phip.*sin(theta).*psip.*cos(psi) + ...
        thetapp.*cos(psi) - thetap.*psip.*sin(psi);
    omp(I1,:) = psipp - phipp.*cos(theta) + phip.*thetap.*sin(theta);

  case "D"

    om(I3,:) = -phip.*sin(theta).*cos(psi) + thetap.*sin(psi);
    om(I2,:) = phip.*sin(theta).*sin(psi) + thetap.*cos(psi);
    om(I1,:) = psip + phip.*cos(theta);

    omp(I3,:) = -phipp.*sin(theta).*cos(psi) - phip.*thetap.*cos(theta).*cos(psi) + phip.*sin(theta).*psip.*sin(psi) + ...
        thetapp.*sin(psi) + thetap.*psip.*cos(psi);
    omp(I2,:) = phipp.*sin(theta).*sin(psi) + phip.*thetap.*cos(theta).*sin(psi) + phip.*sin(theta).*psip.*cos(psi) + ...
        thetapp.*cos(psi) - thetap.*psip.*sin(psi);
    omp(I1,:) = psipp + phipp.*cos(theta) - phip.*thetap.*sin(theta);

end

endfunction
