//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function CL_plot_earthMap(win_id,color_id,tick_steps,data_bounds,thickness)
// Plots an Earth map
//
// Calling Sequence
// CL_plot_earthMap([win_id,color_id,tick_steps,data_bounds,thickness])
//
// Parameters
// win_id: (optional) Figure Id - Id of the current window if omitted.
// color_id: (optional) Color index - Default is 2.
// tick_steps: (optional) Steps of the grid in longitude and latitude in degrees. Default is [60 ,30].
// data_bounds: (optional) Definition of the view area: [longmin, latmin; longmax, latmax] in degrees. Default is [-180, -90; 180, 90].
// thickness: (optional) Line thickness. Default is 1.
//
// Description
// <itemizedlist><listitem>
// This function plots an Earth map in the selected figure (win_id). The plot x-axis and y-axis are respectively the (spherical) longitude and latitude in degrees. 
// </listitem></itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_plot_ephem
//
// Examples
// // basic plot
// CL_plot_earthMap();
//
// // plot in figure 10, with color 2
// CL_plot_earthMap(win_id=10, color_id=2);
//
// // plot in figure 10, with color 2, and focus on Europe
// CL_plot_earthMap(win_id=10, color_id=2, ..
// data_bounds=[-10,30;40,60], tick_steps=[10,10]);

// Declarations:
  global %CL_home; 

// Code:

 // CODE FOR COMPATIBILITY
 if exists('winId','local'); win_id=winId; end
 if exists('colorId','local'); color_id=colorId; end
 if exists('tickInterval','local'); tick_steps=tickInterval; end
 if exists('dataBounds','local'); data_bounds=dataBounds; end


 if exists('win_id','local') then
    f = scf(win_id)
 else
    f = gcf();
 end

 if ~exists('color_id','local') then
    color_id = 2;
 end

 if ~exists('tick_steps','local') then
    tick_steps = [60, 30]; // degrees
 end

 if ~exists('data_bounds','local') then
    data_bounds = [-180, -90; 180, 90]; // degrees
 end

 if ~exists('thickness','local'); thickness = 1; end

 // error checking
 if (size(data_bounds,1) <> 2 | size(data_bounds,2) <> 2)
    CL__error('Invalid size for data_bounds');
 end
 if (data_bounds(2,1)<=data_bounds(1,1) | data_bounds(2,1)>data_bounds(1,1)+360)
    CL__error('Invalid data bounds in longitude');
 end 
 if (data_bounds(2,2)<=data_bounds(1,2) | data_bounds(1,2) < -90 | data_bounds(2,2) > 90)
    CL__error('Invalid data bounds in latitude');
 end 

 if (length(tick_steps) <> 2)
    CL__error('Invalid size for tick_steps');
 end 
 if (tick_steps(1) <= 0 | tick_steps(2) <= 0)
    CL__error('Invalid values for tick_steps');
 end 

 // plot
 f.immediate_drawing = "off"; 

 // file path
 sep = filesep();
 nomfic = %CL_home+sep+'data'+sep+'utils'+sep+'terre_sph.pla';
 
 // interval in longitude containing the view
 bmin = (data_bounds(1,1)+data_bounds(2,1))/2 - 180; 
 bmax = (data_bounds(1,1)+data_bounds(2,1))/2 + 180; 

 // loading of the map 
 M = fscanfMat(nomfic);
 M = M'; 

 // beginnings / ends of blocks
 I = [0, find(M(1,:) == 999.), size(M,2)+1]; // beginning/end added

 for k = 1 : length(I)-1
   ind = I(k)+1 : I(k+1)-1; 
   
   if (length(ind) >= 2)
     liste_seg = CL__plot_contSegsList(M(1,ind), M(2,ind), bmin, bmax);   
     for (seg = liste_seg) 
       plot2d(seg(1,:), seg(2,:), style=color_id);
       e = gce();
       e.children(1).thickness = thickness; 
     end
   end
 end

 // adjustments (grid, databounds)
 a = gca();
 a.data_bounds = data_bounds;
 a.tight_limits = "on";

 bticks = [ tick_steps(1) * floor(data_bounds(1,:)/tick_steps(1)); 
            tick_steps(2) * ceil((data_bounds(2,:))/tick_steps(2)) ];

 x_ticks = bticks(1,1) : tick_steps(1) : bticks(2,1);
 y_ticks = bticks(1,2) : tick_steps(2) : bticks(2,2);

 a.x_ticks = tlist(["ticks", "locations", "labels"], x_ticks, string(x_ticks)); 
 a.y_ticks = tlist(["ticks", "locations", "labels"], y_ticks, string(y_ticks)); 

 CL_g_stdaxes(a);

 f.immediate_drawing = "on"; 

endfunction


