//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [dv,anv] = CL_man_sma(ai,ei,af,posman,mu)
// Delta V required to change the semi-major axis
//
// Calling Sequence
// [dv,anv] = CL_man_sma(ai,ei,af [,posman,mu])
//
// Description
// <itemizedlist><listitem>
// This function computes the delta-v required to change the semi major axis.
// <para> The maneuver consists in modifying the periapsis or apoapsis radius only, and 
// only one maneuver is performed: at periapsis or at apoapsis. </para>
// <para>The output argument <emphasis role="bold">dv</emphasis> is the delta-V required expressed in spherical coordinates in the QSW frame: [lambda; phi; |dv|], where lambda is the in-plane angle (+%pi: towards planet and +%pi/2: ~along velocity), phi is the out-of-plane angle, positive towards the angular momentum vector (the angular momentum vector is perpendicular to the orbit plane and oriented according to right hand convention), dv is the norm of the velocity increment. </para>
// <para><emphasis role="bold">anv</emphasis> is the true anomaly at the position of maneuver.</para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// ai : Initial semi major axis [m] (1xN)
// ei: Initial eccentricity (1xN)
// af : Final semi major axis [m] (1xN)
// posman: (optional) flag specifying the position of maneuver (0->periapsis, 1->apoapsis; default is 0) (1xN)
// mu : (optionnal) Gravitational constant. [m^3/s^2] (default value is %CL_mu)
// dv : Delta_v in spherical coordinates in the QSW frame [lambda;phi;|dv|] [rad,rad,m/s] (3xN)
// anv: True anomaly at the position of maneuver [rad] (1xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_man_biElliptic
// CL_man_hohmann
// CL_man_hohmannG
//
// Examples
// ai = 7200.e3;
// af = 7000.e3;
// ei = 0.1;
// [dv,anv] = CL_man_sma(ai,ei,af)
// // Check results :
// kep = [ai ; ei ; %pi/2 ; 0 ; 0 ; anv];
// kep1 = CL_man_applyDv(kep,dv)

// Declarations:
global %CL_mu;

// Code:

if ~exists('posman','local') then posman=0; end

Nai = size(ai,2);
Nei = size(ei,2);
Naf = size(af,2);
Nposman = size(posman,2);
N = max(Nai,Naf,Nei,Nposman);
coherence = (Nai==N | Nai==1) &..
            (Nei==N | Nei==1) &..
            (Naf==N | Naf==1) &..
            (Nposman==N | Nposman==1);
if ~coherence then CL__error('bad input argument size'); end
if N~=1
  if Nai==1 then ai=ai*ones(1,N); end
  if Naf==1 then af=af*ones(1,N); end
  if Nei==1 then ei=ei*ones(1,N); end
  if Nposman==1 then posman=posman*ones(1,N); end
end

if ~exists('mu','local') then mu=%CL_mu; end

rman = zeros(1,N);

//calculation of radius at maneuver point
i11 = find(ei~=0 & posman==1);  //maneuver at apoapsis and not circular orbit
i12 = find(ei~=0 & posman==0);  //maneuver at periapsis and not circular orbit
i2 = find(ei==0); //circular orbit
rman(i11) = ai(i11).*(1+ei(i11));
rman(i12) = ai(i12).*(1-ei(i12));
rman(i2) = ai(i2);

[dv,anv] = CL__man_raps(rman,ai,af,mu);

endfunction
