//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [mean_cir] = CL_ex_meanEckHech (osc_cir,er,mu, zonals)
// Eckstein Hechler orbit propagation analytical model (mean elements)
//
// Calling Sequence
// [mean_cir] = CL_ex_meanEckHech(osc_cir [,er,mu,zonals])
//
// Description
// <itemizedlist><listitem>
// Computes the mean orbital elements <emphasis role="bold">mean_cir</emphasis> from the osculating elements <emphasis role="bold">osc_cir</emphasis>
// </listitem>
// <listitem> The orbital elements are the following:  
// <para><inlinemediaobject><imageobject><imagedata fileref="cir_par.gif"/></imageobject></inlinemediaobject></para>
// </listitem>
// <listitem>Warnings :
// <para> - This function does not work for inclinations close to the critical inclinations (63.43494882 deg and 116.5650512 deg)</para>
// <para> - This function nominally works for eccentricities smaller than 5.e-3</para>
// <para> - This function works with a lesser accuracy for eccentricities between 5.e-3 and 0.1</para>
// <para> - This function does not work for eccentricities greater than 0.1</para></listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// osc_cir: Osculating circular-adapted elements [sma;ex;ey;inc;raan;pom+anm] (6xN)
// er: (optional) Equatorial radius [m] (default is %CL_eqRad)
// mu: (optional) Gravitational constant [m^3/s^2] (default value is %CL_mu)
// zonals: (optional) Vector of zonals coefficients J1 to Jn (troncated to J6) to be used (default is %CL_j1jn(1:6)) (1xN)
// mean_cir: Mean circular-adapted elements [sma;ex;ey;inc;raan;pom+anm] (6xN)
//
// Authors
// CNES - DCT/SB
//
// Bibliography
// 1 CNES - MSLIB FORTRAN 90, Volume E (me_eck_hech)
//
// See also
// CL_ex_lyddane
// CL_ex_eckHech
//
// Examples
// // Example 1 : one orbit extrapolated at several dates :
// osc_cir = [42166712;2.9e-3;1.2e-3;CL_deg2rad([97.2;125;5])]
// [mean_cir] = CL_ex_meanEckHech(osc_cir)
//

//************************************************************************
//
// But:  Modele analytique d'extrapolation d'orbite de ECKSTEIN-HECHLER.
// ===
//
// Note d'utilisation:  L'excentricite  doit etre superieure >= 0 et < 0.1.
// ==================   Le domaine d'utilisation du modele est .. [ 0E+0 , 5E-3 [.
//                      Le domaine de precision degradee est .... [ 5E-3 , 1E-1 [.
//                      Le domaine d'erreur est ................. [ 1E-1 , +Oo  [.
//                      L'inclinaison  doit etre > 0. et < pi et non proches des inclinaisons critiques
//                      ( pm_i_critique_non_retro, pm_i_critique_retro ) a pm_eps_i_critique pres.
//
// Historique:
// ==========
//   + Version 2.0 (SP 341 ed01 rev00): creation a partir de la routine MEEHJ6 de la MSLIB f77
//   + Version 3.1 (DE globale 439 ed01 rev00) : ajout des champs %biblio et %message pour le code retour
//   + Version 4.1 (DE globale 482 ed01 rev00): Corrections qualite
//   + Version 4.1 (DE globale 482 ed01 rev00): Corrections qualite
//   + Version 6.0 (DE globale 618 ed01 rev00): ajout de tests sur mu et les coefficients CN0
//************************************************************************



// Declarations:
global %CL_criticalInc1 %CL_criticalInc2 %CL_eqRad %CL_j1jn %CL_mu;

// Code:

if ~exists('mu','local') then mu=%CL_mu; end
if ~exists('er','local') then er=%CL_eqRad; end
if ~exists('zonals','local') then zonals=%CL_j1jn(1:6); end

if(length(zonals) < 6)
  warning('zonals must be a vector of size 6 (or more), completing with zeros...');
  zonals = [zonals 0 0 0 0 0 0];
end

// initialisation de la valeur epsilon * 100
eps100 = 100 * %eps
eccel = 0.05

//controle des donnees
//demi grand axe
if or( osc_cir(1,:) <= eps100 )
  CL__error("Error : semi major axis < or = 0");
end
//eccentricite
ecc = sqrt(osc_cir(2,:)**2+(osc_cir(3,:)**2));
if or( ecc < 0 )
  CL__error("Error : negative eccentricity");
end
if or( ecc > 0.1 )
  CL__error("Error : eccentricity higher than 0.1");
end
if or( ecc > 5.e-3)
  warning("Warning : lesser precision for eccentricities between 5e-3 and 0.1");
end

// inclinaison
if or( osc_cir(4,:) < 0 )
  CL__error("negative inclination");
end
if or( osc_cir(4,:) > %pi )
  CL__error("inclination > pi");
end

// inclinaison proche de l'inclinaison critique.
eps_i_critique = 4e-5; // obtenu experimentalement : au dela ca plante
if (abs(osc_cir(4,:) - %CL_criticalInc1) < eps_i_critique | ...
     abs(osc_cir(4,:) - %CL_criticalInc2) < eps_i_critique) then
   CL__error("inclination close (less than "+string(eps_i_critique)+" rad) to the critical value");
end

nmax        = 51          // nombre maximum d'iterations admises
n = ones(1,size(osc_cir,2))           // initialisation du numero d'iterations
convergence = zeros(1,size(osc_cir,2))// initialisation de l'indicateur de convergence
eps1000 = 1000 * %eps // epsilon pour test de convergence
dat = 1.0  //la date est inutile dans l'appel a me_lyddane: on l'initialise a 1 j 0 sec,

mean_local=zeros(osc_cir)
ecart = zeros(osc_cir)
ecart_admi=zeros(osc_cir)
mean_sortie=zeros(osc_cir)
oscn=zeros(osc_cir)

// initialisation des mean aux osculateurs en entree et passage des angles dans l'intervalle [0,2*pi]
mean_local(1:3,:) = osc_cir(1:3,:);
mean_local(4:6,:) = pmodulo(osc_cir(4:6,:),2*%pi);

// ecarts admis
ecart_admi(1,:) = eps1000*(1 + abs(mean_local(1,:)))
ecart_admi(2,:) = eps1000*(1 + abs(mean_local(2,:)))
ecart_admi(3,:) = eps1000*(1 + abs(mean_local(3,:)))
ecart_admi(4,:) = eps1000*(1 + abs(mean_local(4,:)))
ecart_admi(5,:) = eps1000*(1 + abs(mean_local(5,:)))
ecart_admi(6,:) = eps1000*(1 + abs(mean_local(6,:)))


// debut d'iteration
ii = find((n < nmax) & (~convergence))
while ii~=[]  // iterations tant que le nombre maximum d'iterations n'est pas atteint et convergence non atteinte

  //  calcul des osculateurs correspondant aux mean
   [mean_sortie,oscn] = CL_ex_eckHech( dat, mean_local, dat, er, mu, zonals );

  //calcul des ecarts entre parametres osculateurs calcules et ceux de depart. .
   ecart  = osc_cir- oscn;

   // Ces ecarts sont ramenes sur [-pi,pi] : inc
  jj1 = find( (ecart(4,:) < (-%pi)) & ((n < nmax) & (~convergence)) );
  jj2 = find( (ecart(4,:) > %pi) & ((n < nmax) & (~convergence)) );
  ecart(4,jj1) = ecart(4,jj1) + 2*%pi;
  ecart(4,jj2) = ecart(4,jj2) - 2*%pi

   // Ces ecarts sont ramenes sur [-pi,pi] : gom
  jj1 = find( (ecart(5,:) < (-%pi)) & ((n < nmax) & (~convergence)) );
  jj2 = find( (ecart(5,:) > %pi) & ((n < nmax) & (~convergence)) );
  ecart(5,jj1) = ecart(5,jj1) + 2*%pi;
  ecart(5,jj2) = ecart(5,jj2) - 2*%pi

   // Ces ecarts sont ramenes sur [-pi,pi] : anm
  jj1 = find( (ecart(6,:) < (-%pi)) & ((n < nmax) & (~convergence)) );
  jj2 = find( (ecart(6,:) > %pi) & ((n < nmax) & (~convergence)) );
  ecart(6,jj1) = ecart(6,jj1) + 2*%pi;
  ecart(6,jj2) = ecart(6,jj2) - 2*%pi

  //  test de convergence
  nn = find( ...
            ( abs(ecart(1,:)) <= ecart_admi(1,:) ) &...
            ( abs(ecart(2,:)) <= ecart_admi(2,:) ) &...
            ( abs(ecart(3,:)) <= ecart_admi(3,:) ) &...
            ( abs(ecart(4,:)) <= ecart_admi(4,:) ) &...
            ( abs(ecart(5,:)) <= ecart_admi(5,:) ) &...
            ( abs(ecart(6,:)) <= ecart_admi(6,:) ) &...
            ( (n < nmax) & (~convergence) )...
            );
  jj = find( ~(...
                ( abs(ecart(1,:)) <= ecart_admi(1,:) ) &...
                ( abs(ecart(2,:)) <= ecart_admi(2,:) ) &...
                ( abs(ecart(3,:)) <= ecart_admi(3,:) ) &...
                ( abs(ecart(4,:)) <= ecart_admi(4,:) ) &...
                ( abs(ecart(5,:)) <= ecart_admi(5,:) ) &...
                ( abs(ecart(6,:)) <= ecart_admi(6,:) ) &...
                ( (n < nmax) & (~convergence) ) ...
              ) ...
           );

   convergence(nn) = 1

   //  reinitialisation des elements means
  mean_local(1,jj) = mean_local(1,jj)+ecart(1,jj);
  mean_local(2,jj) = mean_local(2,jj)+ecart(2,jj);
  mean_local(3,jj) = mean_local(3,jj)+ecart(3,jj);
  mean_local(4,jj) = mean_local(4,jj)+ecart(4,jj);
  mean_local(5,jj) = mean_local(5,jj)+ecart(5,jj);
  mean_local(6,jj) = CL_rMod(mean_local(6,jj)+ecart(6,jj),2*%pi);

  n(ii) = n(ii) + 1
  ii = find((n < nmax) & (~convergence))

end

if or(~convergence) then
  CL__error("No convergence mean Eckstein Hechler in "+string(ii))
end // pas de convergence

// affectation des parametres de sortie
mean_cir = mean_local;

endfunction
