//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [sma,ecc] = CL_op_rarp2ae(ra,rp,mu)
// Apoapsis radius and periapsis radius to semi major axis and eccentricity (elliptic orbits)
//
// Calling Sequence
// [sma,ecc] = CL_op_rarp2ae(ra,rp[,mu])
//
// Description
// <itemizedlist><listitem>
// Computes keplerian parameters semi major axis and eccentricity from
// apoapsis and periapsis radius (only ellipse).
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// ra: apoapsis radius [m] (1xN)
// rp: periapsis radius [m] (1xN)
// mu : (optionnal) geocentric gravitational constant [m^3/s^2] (default value is %CL_mu)
// sma: semi major axis [m] (1xN)
// ecc: eccentricity (1xN)
//
// Authors
// CNES - DCT/SB
//
// Examples
// rp = 7078.e3;
// ra = 7178.e3;
// [sma,ecc] = CL_op_rarp2ae(ra,rp)

// Declarations:
global %CL_mu;

// Code:

if ~exists('mu','local') then mu=%CL_mu; end

sma = (ra+rp)/2
ecc = (ra-rp)./(ra+rp)

endfunction
