//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [result] = CL_gm_eclipseCir(sma,inc,raan,alpha_sun,delta_sun,er,mu,j2)
// Analytical eclipse calculation for circular orbits
//
// Calling Sequence
// [result] = CL_gm_eclipseCir(sma,inc,raan,alpha_sun,delta_sun[,er,mu,j2])
//
// Description
// <itemizedlist><listitem>
// This function computes various results that characterize the portion of the orbit path where the satellite is 
// in the shadow of the planet. 
// <para>The eclipsed area is a half cylinder of diameter the planet's diameter, and axis the Sun direction. 
// The calculation is purely geometrical: the Sun direction is supposed constant with respect to the orbit plane. </para>
// <para><emphasis role="bold">result</emphasis> is a structure with the following fields :</para>
// <para><emphasis role="bold">result.start</emphasis> and <emphasis role="bold">result.end</emphasis> contain quantities that define 
// the eclipse start and end positions: 'pso' : argument of latitude(w+M); 'ra' : right ascension; 'decl' :declination</para>
// <para><emphasis role="bold">result.sun_orb</emphasis> contains quantities that define the Sun's position 
// ('alpha' and 'delta': spheric coordinates) in a frame tied to the orbit</para>
// <para>The frame tied to the orbit is defined as follows:</para>
// <para>X-axis: Towards the ascending node</para>
// <para>Z-axis: Parallel to (and same direction as) the angular momentum vector</para>
// <para>Y-axis: Such that the frame is direct. </para>
// <para><emphasis role="bold">result.angle</emphasis> is the total eclipse's length (result.end.pso - result.start.pso)</para>
// <para><emphasis role="bold">result.duration</emphasis> is the duration corresponding to result.angle taking into account keplerian mean motion</para>
// </listitem>
// <listitem>
// <para>Notes: </para>
// <para> - The planet is assumed spherical</para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// sma: Semi major axis [m] (1xN)
// inc: Inclination [rad] (1xN)
// raan: Right ascension of ascending node [rad] (1xN)
// alpha_sun: Sun right ascension [rad] (1xN)
// delta_sun: Sun declination [rad] (1xN)
// er: (optional) Equatorial radius [m] (default is %CL_eqRad)
// mu: (optional) Geocentric gravitational constant [m^3/s^2] (default value is %CL_mu)
// result: (structure) Various quantities that define the eclipse [rad,sec] (each field is 1xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_gm_betaEclipse
// CL_mod_moonSunG50
//
// Examples
// //Example 1
// cjd = CL_dat_cal2cjd(2009,03,21,6,0,0);
// [r_sun,rs] = CL_mod_moonSunG50(cjd,'s');
// sun_sph = CL_co_car2sph(r_sun);
// alpha_sun = sun_sph(1,:);
// delta_sun = sun_sph(2,:);
// raan=0;
// sma = 1000000+%CL_eqRad;
// inc = CL_deg2rad(98);
// [result] = CL_gm_eclipseCir(sma,inc,raan,alpha_sun,delta_sun);
// result.start
// result.end
// result.sun_orb
// result.angle
// result.duration
//
// // Example 2
// inc = CL_deg2rad(90);
// sma = 6978.e3;
// raan = 0;
// // Eclipse position is around 180deg of pso :
// [result] = CL_gm_eclipseCir(sma,inc,raan,alpha_sun,delta_sun) 

// Declarations:
global %CL_eqRad %CL_mu;

// Code:

Nsma = size(sma,2)
Ninc = size(inc,2)
Ncjd = size(cjd,2)
Nraan = size(raan,2)
N = max(Nsma,Ninc,Ncjd,Nraan)
coherence = (Nsma==1|Nsma==N)&(Ninc==1|Ninc==N)&(Ncjd==1|Ncjd==N)&(Nraan==1|Nraan==N)
if ~coherence then CL__error('bad size of input arguments'); end
if N~=1
  if Nsma==1 then sma=sma*ones(1,N); end
  if Ninc==1 then inc=inc*ones(1,N); end
  if Ncjd==1 then cjd=cjd*ones(1,N); end
  if Nraan==1 then raan=raan*ones(1,N); end
end

if ~exists('er','local') then er=%CL_eqRad; end
if ~exists('mu','local') then mu=%CL_mu; end

// Keplerian Mean motion
xnmoy = CL_kp_params('mm',sma,mu)

// Sun in orbit's frame
[alpha_sun_orb,delta_sun_orb] = CL_gm_inertial2orbitSph(inc,raan,alpha_sun,delta_sun)
// Earth semi angle of vision seen from satellite
betaa = asin(er./(sma))
// Anomaly semi angle on eclipse
dalpha = zeros(betaa)
cosdalpha = zeros(betaa)
aux = cos(betaa)./cos(delta_sun_orb)
i1 = find( (cos(delta_sun_orb)<=0) | (aux>1))
dalpha(i1) = 0
ii = find(~( (cos(delta_sun_orb)<=0) | (aux>1) ))
cosdalpha(ii) = cos(betaa(ii))./cos(delta_sun_orb(ii))
dalpha(ii) = acos(cosdalpha(ii))

// PSO start and end of eclipse
dir_deb_orb = zeros(2,N)
dir_fin_orb = zeros(2,N)
dir_deb_orb(1,:) = alpha_sun_orb + %pi - dalpha
dir_fin_orb(1,:) = alpha_sun_orb + %pi + dalpha
dir_deb_orb(2,:) = 0
dir_fin_orb(2,:) = 0

// Right ascension / declination: eclipse start
dir_deg_g50 = zeros(2,N)
[aa,bb] = CL_gm_orbit2inertialSph(inc,raan,dir_deb_orb(1,:),dir_deb_orb(2,:))
dir_deb_g50(1,:) = aa
dir_deb_g50(2,:) = bb

// Right ascension / declination : eclipse end
dir_fin_g50 = zeros(2,N)
[cc,dd] = CL_gm_orbit2inertialSph(inc,raan,dir_fin_orb(1,:),dir_fin_orb(2,:))
dir_fin_g50(1,:) = cc
dir_fin_g50(2,:) = dd

// Local hour start/end eclipse
hl_deb_ecl = modulo(alpha_sun-dir_deb_g50(1,:),2*%pi)*12/%pi+12
hl_fin_ecl = modulo(alpha_sun-dir_fin_g50(1,:),2*%pi)*12/%pi+12
jj = find(hl_fin_ecl<hl_deb_ecl)
hl_fin_ecl(jj) = hl_fin_ecl(jj) + 24

// Angle on eclipse
delta_anom = 2*dalpha

// Duration
duree_eclipse = delta_anom ./ xnmoy

// Angle Sun / orbit plan
dir_deb_orb(1,:) = modulo(dir_deb_orb(1,:),2*%pi)
dir_fin_orb(1,:) = dir_deb_orb(1,:) + modulo(dir_fin_orb(1,:)-dir_deb_orb(1,:),2*%pi)

dir_deb_g50(1,:) = modulo(dir_deb_g50(1,:),2*%pi)
dir_fin_g50(1,:) = dir_deb_g50(1,:) + modulo(dir_fin_g50(1,:)-dir_deb_g50(1,:),2*%pi)

alpha_sun_orb = pmodulo(alpha_sun_orb,2*%pi)

pso_start = dir_deb_orb(1,:)
ra_start = dir_deb_g50(1,:)
dec_start = dir_deb_g50(2,:)
pso_end = dir_fin_orb(1,:)
ra_end = dir_fin_g50(1,:)
dec_end = dir_fin_g50(2,:)
ra_sun_orb = alpha_sun_orb
dec_sun_orb = delta_sun_orb


// -------------------------------------------------------------
result = struct('start', 0, 'end', 0, 'sun_orb', 0, 'angle', 0, 'duration', 0);
start = struct('pso', 0, 'ra', 0, 'decl', 0);
end = struct('pso', 0, 'ra', 0, 'decl', 0);
sun_orb = struct('alpha', 0, 'delta', 0);

result.start.pso = pso_start;
result.start.ra = ra_start;
result.start.decl = dec_start;
result.end.pso = pso_end;
result.end.ra = ra_end;
result.end.decl = dec_end;
result.sun_orb.alpha = ra_sun_orb;
result.sun_orb.delta = dec_sun_orb;
result.angle = delta_anom;
result.duration = duree_eclipse;



endfunction
