//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [u,alpha] = CL_rot_quat2axAng(q)
// Quaternion to rotation axis and angle
//
// Calling Sequence
// [u,alpha] = CL_rot_quat2axAng(q)
//
// Description
// <itemizedlist><listitem>
// Converts a quaternion into rotation axis and angle. 
// </listitem>
// <listitem>
// Notes: 
// <para> - The rotation axis returned may be undefined (%nan). </para>
// <para> - See <link linkend="Conventions">Conventions</link> or <link linkend="CL_rot_defQuat">CL_rot_defQuat</link> for more details on quaternions. </para>
// </listitem>
// </itemizedlist>
// <para><emphasis role="bold">( Last updated: 2010-02-17 )</emphasis></para>
//
// Parameters
// u : Rotation axis [ux;uy;uz] (3xN)
// q : Quaternion (dim N)
// alpha : Rotation angle [rad] (1xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_rot_defQuat
// CL_rot_quat2axAng
//
// Examples
// phi = CL_deg2rad(33);
// theta = CL_deg2rad(12);
// psi = CL_deg2rad(133);
// q = CL_rot_eul2quat(phi,theta,psi);
//
// [u, alpha] = CL_rot_quat2axAng(q)

// Declarations:


// Code:

[u, u_norm] = CL_unitVector(q.i);

alpha = 2 * real(acos(q.r));

endfunction
