/*
 * taskset.c - taskset
 * Command-line utility for setting and retrieving a task's CPU affinity
 *
 * Robert Love <rml@tech9.net>		25 April 2002
 *
 * Linux kernels as of 2.5.8 provide the needed syscalls for
 * working with a task's cpu affinity.  Currently 2.4 does not
 * support these syscalls, but patches are available at:
 *
 * 	http://www.kernel.org/pub/linux/kernel/people/rml/cpu-affinity/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License, v2, as
 * published by the Free Software Foundation
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Copyright (C) 2004 Robert Love
 */

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>
#include <sched.h>
#include <errno.h>

static void show_usage(const char *cmd)
{
	fprintf(stderr, "taskset version " VERSION "\n");
	fprintf(stderr, "usage: %s [options] [mask] [pid | cmd [args...]]\n",
			cmd);
	fprintf(stderr, "set or get the affinity of a process\n\n");
	fprintf(stderr, "  -p, --pid                  "
			"operate on existing given pid\n");
	fprintf(stderr, "  -h, --help                 "
			"display this help\n");
	fprintf(stderr, "  -v, --version              "\
			"output version information\n\n");
	fprintf(stderr, "The default behavior is to run a new command:\n");
	fprintf(stderr, "  %s 03 sshd -b 1024\n", cmd);
	fprintf(stderr, "You can retrieve the mask of an existing task:\n");
	fprintf(stderr, "  %s -p 700\n", cmd);
	fprintf(stderr, "Or set it:\n");
	fprintf(stderr, "  %s -p 03 700\n\n", cmd);
}

/*
 * cpu_set_to_ulong - convert a "cpu_set_t" to an "unsigned long".
 *
 * Yes, this limits the size of the CPU mask to "sizeof (unsigned long)".  The
 * interface is stupid.  I can say that, since I wrote it.  We use a pointer
 * instead of returning the value to be consistent with "ulong_to_cpu_set()".
 */
static void cpu_set_to_ulong(cpu_set_t cpu_set_mask, unsigned long *ulong_mask)
{
	size_t i;

	*ulong_mask = 0;

	for (i = 0; i < sizeof (unsigned long) * 8; i++)
		if (CPU_ISSET(i, &cpu_set_mask))
			*ulong_mask |= (1UL << i);
}

/*
 * ulong_to_cpu_set - convert an "unsigned long" to a "cpu_set_t".
 *
 * We receive the "cpu_set_t" as a pointer to avoid returning an aggregate
 * type.
 */
static void ulong_to_cpu_set(unsigned long ulong_mask, cpu_set_t *cpu_set_mask)
{
	size_t i;

	CPU_ZERO(cpu_set_mask);
	for (i = 0; i < sizeof (unsigned long) * 8; i++)
		if (ulong_mask & (1UL << i))
			CPU_SET(i, cpu_set_mask);
}

int main(int argc, char *argv[])
{
	cpu_set_t new_mask, cur_mask;
	unsigned long the_mask;
	pid_t pid = 0;
	int opt;

	struct option longopts[] = {
		{ "pid",	0, NULL, 'p' },
		{ "help",	0, NULL, 'h' },
		{ "version",	0, NULL, 'V' },
		{ NULL,		0, NULL, 0 }
	};

	while ((opt = getopt_long(argc, argv, "+phV", longopts, NULL)) != -1) {
		int ret = 1;

		switch (opt) {
		case 'p':
			pid = atoi(argv[argc - 1]);
			break;
		case 'V':
			printf("taskset version " VERSION "\n");
			return 0;
		case 'h':
			ret = 0;
		default:
			show_usage(argv[0]);
			return ret;
		}
	}

	if ((!pid && argc - optind < 2)
			|| (pid && (argc - optind < 1 || argc - optind > 2))) {
		show_usage(argv[0]);
		return 1;
	}

	if (pid) {
		unsigned long mask = 0;

		if (sched_getaffinity(pid, &cur_mask) < 0) {
			perror("sched_getaffinity");
			fprintf(stderr, "failed to get pid %d's affinity\n",
				pid);
			return 1;
		}
		cpu_set_to_ulong(cur_mask, &mask);
		printf("pid %d's current affinity mask: %08lx\n", pid, mask);
		if (argc - optind == 1)
			return 0;
	}

	errno = 0;
	the_mask = strtoul(argv[optind], NULL, 16);
	if (errno) {
		perror("strtoul");
		fprintf(stderr, "failed to parse CPU mask %s\n", argv[optind]);
		return 1;
	}

	ulong_to_cpu_set(the_mask, &new_mask);

	if (sched_setaffinity(pid, &new_mask)) {
		perror("sched_setaffinity");
		fprintf(stderr, "failed to set pid %d's affinity.\n", pid);
		return 1;
	}

	if (sched_getaffinity(pid, &cur_mask) < 0) {
		perror("sched_setaffinity");
		fprintf(stderr, "failed to get pid %d's affinity.\n", pid);
		return 1;
	}

	if (pid) {
		unsigned long mask = 0;

		cpu_set_to_ulong(cur_mask, &mask);
		printf("pid %d's new affinity mask: %08lx\n", pid, mask);
	} else {
		argv += optind + 1;
		execvp(argv[0], argv);
		perror("execvp");
		fprintf(stderr, "failed to execute %s\n", argv[0]);
		return 1;
	}

	return 0;
}
