/*
 * warts-dump
 *
 * $Id: warts-dump.c,v 1.121 2010/05/07 04:38:40 mjl Exp $
 *
 *        Matthew Luckie
 *        mjl@wand.net.nz
 *
 * Copyright (C) 2004-2010 The University of Waikato
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef lint
static const char rcsid[] =
  "$Id: warts-dump.c,v 1.121 2010/05/07 04:38:40 mjl Exp $";
#endif

#ifdef _WIN32
#include <winsock2.h>
#define STDIN_FILENO 0
#endif

#if defined(_MSC_VER)
typedef unsigned __int8 uint8_t;
typedef unsigned __int16 uint16_t;
typedef unsigned __int32 uint32_t;
#endif

#if defined(__APPLE__)
#include <stdint.h>
#endif

#include <sys/types.h>

#ifndef _WIN32
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#endif

#include <stdio.h>
#include <string.h>
#include <time.h>

#ifdef _WIN32
#define snprintf _snprintf
#endif

#if defined(DMALLOC)
#include <dmalloc.h>
#endif

#include "scamper_addr.h"
#include "scamper_list.h"
#include "scamper_tlv.h"
#include "scamper_icmpext.h"
#include "scamper_trace.h"
#include "scamper_ping.h"
#include "scamper_tracelb.h"
#include "scamper_dealias.h"
#include "scamper_neighbourdisc.h"
#include "scamper_tbit.h"
#include "scamper_file.h"
#include "utils.h"

#define TCP_MAX_SEQNUM 4294967295U

static void usage()
{
  fprintf(stderr, "usage: warts-dump <file>\n");
  return;
}

static void dump_list_summary(scamper_list_t *list)
{
  if(list != NULL)
    {
      printf(" list id: %d", list->id);
      if(list->name != NULL)
	printf(", name: %s", list->name);
      if(list->monitor != NULL)
	printf(", monitor: %s", list->monitor);
      printf("\n");
    }
  return;
}

static void dump_cycle_summary(scamper_cycle_t *cycle)
{
  if(cycle != NULL)
    printf(" cycle id: %d\n", cycle->id);
  return;
}

static void dump_tcp_flags(uint8_t flags)
{
  if(flags != 0)
    {
      printf(" (%s%s%s%s%s%s%s%s )",
	     (flags & 0x01) ? " fin" : "",
	     (flags & 0x02) ? " syn" : "",
	     (flags & 0x04) ? " rst" : "",
	     (flags & 0x08) ? " psh" : "",
	     (flags & 0x10) ? " ack" : "",
	     (flags & 0x20) ? " urg" : "",
	     (flags & 0x40) ? " ece" : "",
	     (flags & 0x80) ? " cwr" : "");
    }
  return;
}

static void dump_start(struct timeval *start)
{
  time_t tt = start->tv_sec;
  char buf[32];
  memcpy(buf, ctime(&tt), 24); buf[24] = '\0';
  printf(" start: %s %06d\n", buf, (int)start->tv_usec);
  return;
}

static void dump_trace_hop(scamper_trace_hop_t *hop)
{
  scamper_icmpext_t *ie;
  scamper_tlv_t *tlv;
  uint32_t u32;
  char addr[256];
  int i;

  printf("hop %2d  %s\n",
	 hop->hop_probe_ttl,
	 scamper_addr_tostr(hop->hop_addr, addr, sizeof(addr)));

  printf(" attempt: %d, rtt: %d.%06ds\n",
	 hop->hop_probe_id,
	 (int)hop->hop_rtt.tv_sec, (int)hop->hop_rtt.tv_usec);

  printf(" probe_size: %d", hop->hop_probe_size);
  if(hop->hop_flags & SCAMPER_TRACE_HOP_FLAG_REPLY_TTL)
    printf(", reply_ttl: %d", hop->hop_reply_ttl);
  printf("\n");

  if((hop->hop_flags & SCAMPER_TRACE_HOP_FLAG_TCP) == 0)
    {
      printf(" icmp type: %d, code: %d\n",
	     hop->hop_icmp_type, hop->hop_icmp_code);
    }
  else
    {
      printf(" tcp flags: 0x%02x", hop->hop_tcp_flags);
      dump_tcp_flags(hop->hop_tcp_flags);
      printf("\n");
    }

  printf(" flags: 0x%02x", hop->hop_flags);
  if(hop->hop_flags != 0)
    {
      printf(" (");
      if(hop->hop_flags & SCAMPER_TRACE_HOP_FLAG_TS_SOCK_RX)
	printf(" sockrxts");
      if(hop->hop_flags & SCAMPER_TRACE_HOP_FLAG_TS_DL_TX)
	printf(" dltxts");
      if(hop->hop_flags & SCAMPER_TRACE_HOP_FLAG_TS_DL_RX)
	printf(" dlrxts");
      if(hop->hop_flags & SCAMPER_TRACE_HOP_FLAG_TS_TSC)
	printf(" tscrtt");
      if(hop->hop_flags & SCAMPER_TRACE_HOP_FLAG_REPLY_TTL)
	printf(" replyttl");
      printf(" )");
    }
  printf("\n");

  if(hop->hop_tlvs != NULL)
    {
      for(tlv = hop->hop_tlvs; tlv != NULL; tlv = tlv->tlv_next)
	{
	  switch(tlv->tlv_type)
	    {
	    case SCAMPER_TRACE_HOP_TLV_REPLY_IPID:
	      printf(" ipid outer 0x%04x", tlv->tlv_val_16);
	      break;

	    case SCAMPER_TRACE_HOP_TLV_REPLY_IPTOS:
	      printf(" iptos outer 0x%02x", tlv->tlv_val_8);
	      break;

	    case SCAMPER_TRACE_HOP_TLV_NHMTU:
	      printf(" nhmtu %d", tlv->tlv_val_16);
	      break;

	    case SCAMPER_TRACE_HOP_TLV_INNER_IPTTL:
	      printf(" turn ttl %d", tlv->tlv_val_8);
	      break;

	    case SCAMPER_TRACE_HOP_TLV_INNER_IPLEN:
	      printf(" iplen inner %d", tlv->tlv_val_16);
	      break;

	    case SCAMPER_TRACE_HOP_TLV_INNER_IPTOS:
	      printf(" iptos inner 0x%02x", tlv->tlv_val_8);
	      break;
	    }
	}
      printf("\n");
    }

  for(ie = hop->hop_icmpext; ie != NULL; ie = ie->ie_next)
    {
      if(SCAMPER_ICMPEXT_IS_MPLS(ie))
	{
	  for(i=0; i<SCAMPER_ICMPEXT_MPLS_COUNT(ie); i++)
	    {
	      u32 = SCAMPER_ICMPEXT_MPLS_LABEL(ie, i);
	      printf("%9s ttl: %d, s: %d, exp: %d, label: %d\n",
		     (i == 0) ? "mpls ext" : "",
		     SCAMPER_ICMPEXT_MPLS_TTL(ie, i),
		     SCAMPER_ICMPEXT_MPLS_S(ie, i),
		     SCAMPER_ICMPEXT_MPLS_EXP(ie, i), u32);
	    }
	}
    }

  return;
}

static void dump_trace(scamper_trace_t *trace)
{
  scamper_trace_hop_t *hop;
  uint16_t i;
  char buf[256];

  if(trace->src != NULL)
    {
      scamper_addr_tostr(trace->src, buf, sizeof(buf));
      printf("traceroute from %s to ", buf);
      scamper_addr_tostr(trace->dst, buf, sizeof(buf));
      printf("%s\n", buf);
    }
  else
    {
      printf("traceroute to %s\n",
	     scamper_addr_tostr(trace->dst, buf, sizeof(buf)));
    }

  dump_list_summary(trace->list);
  dump_cycle_summary(trace->cycle);
  printf(" user-id: %d\n", trace->userid);
  dump_start(&trace->start);

  printf(" type: ");
  switch(trace->type)
    {
    case SCAMPER_TRACE_TYPE_ICMP_ECHO:
      printf("icmp, echo id: %d", trace->sport);
      break;

    case SCAMPER_TRACE_TYPE_ICMP_ECHO_PARIS:
      /*
       * if the byte ordering of the trace->sport used in the icmp csum
       * is unknown -- that is, not known to be correct, print that detail
       */
      printf("icmp paris, echo id: %d", trace->sport);
      if(SCAMPER_TRACE_IS_ICMPCSUMDP(trace))
	printf(", csum: 0x%04x", trace->dport);
      break;

    case SCAMPER_TRACE_TYPE_UDP:
      printf("udp, sport: %d, base dport: %d",
	     trace->sport, trace->dport);
      break;

    case SCAMPER_TRACE_TYPE_UDP_PARIS:
      printf("udp paris, sport: %d, dport: %d",
	     trace->sport, trace->dport);
      break;

    case SCAMPER_TRACE_TYPE_TCP:
      printf("tcp, sport: %d, dport: %d", trace->sport, trace->dport);
      break;

    case SCAMPER_TRACE_TYPE_TCP_ACK:
      printf("tcp-ack, sport: %d, dport: %d",
	     trace->sport, trace->dport);
      break;

    default:
      printf("%d", trace->type);
      break;
    }
  printf("\n");

  if(trace->dtree != NULL)
    {
      printf(" doubletree firsthop: %d", trace->dtree->firsthop);
      if(trace->dtree->lss != NULL)
	printf(", lss-name: %s", trace->dtree->lss);
      if(trace->dtree->lss_stop != NULL)
	printf(", lss-stop: %s",
	       scamper_addr_tostr(trace->dtree->lss_stop, buf, sizeof(buf)));
      if(trace->dtree->gss_stop != NULL)
	printf(", gss-stop: %s",
	       scamper_addr_tostr(trace->dtree->gss_stop, buf, sizeof(buf)));
      printf("\n");
    }

  printf(" attempts: %d, hoplimit: %d, loops: %d, probec: %d\n",
	 trace->attempts, trace->hoplimit, trace->loops, trace->probec);
  printf(" gaplimit: %d, gapaction: ", trace->gaplimit);
  if(trace->gapaction == SCAMPER_TRACE_GAPACTION_STOP)
    printf("stop");
  else if(trace->gapaction == SCAMPER_TRACE_GAPACTION_LASTDITCH)
    printf("lastditch");
  else
    printf("0x%02x", trace->gapaction);
  printf("\n");

  printf(" wait-timeout: %ds", trace->wait);
  if(trace->wait_probe != 0)
    printf(", wait-probe: %dms", trace->wait_probe * 10);
  if(trace->confidence != 0)
    printf(", confidence: %d%%", trace->confidence);
  printf("\n");

  printf(" flags: 0x%02x", trace->flags);
  if(trace->flags != 0)
    {
      printf(" (");
      if(trace->flags & SCAMPER_TRACE_FLAG_ALLATTEMPTS)
	printf(" all-attempts");
      if(trace->flags & SCAMPER_TRACE_FLAG_PMTUD)
	printf(" pmtud");
      if(trace->flags & SCAMPER_TRACE_FLAG_DL)
	printf(" dltxts");
      if(trace->flags & SCAMPER_TRACE_FLAG_IGNORETTLDST)
	printf(" ignorettldst");
      if(trace->flags & SCAMPER_TRACE_FLAG_DOUBLETREE)
	printf(" doubletree");
      if(trace->flags & SCAMPER_TRACE_FLAG_ICMPCSUMDP)
	printf(" icmp-csum-dport");
      printf(" )");
    }
  printf("\n");

  printf(" stop reason: ");
  switch(trace->stop_reason)
    {
    case SCAMPER_TRACE_STOP_NONE:
      printf("none");
      break;

    case SCAMPER_TRACE_STOP_COMPLETED:
      printf("done");
      break;

    case SCAMPER_TRACE_STOP_UNREACH:
      printf("icmp unreach %d", trace->stop_data);
      break;

    case SCAMPER_TRACE_STOP_ICMP:
      printf("icmp type %d", trace->stop_data);
      break;

    case SCAMPER_TRACE_STOP_LOOP:
      printf("loop");
      break;

    case SCAMPER_TRACE_STOP_GAPLIMIT:
      printf("gaplimit");
      break;

    case SCAMPER_TRACE_STOP_ERROR:
      printf("errno %d", trace->stop_data);
      break;

    case SCAMPER_TRACE_STOP_HOPLIMIT:
      printf("hoplimit");
      break;

    case SCAMPER_TRACE_STOP_GSS:
      printf("dtree-gss");
      break;

    default:
      printf("reason 0x%02x data 0x%02x",trace->stop_reason,trace->stop_data);
      break;
    }
  printf("\n");

  for(i=0; i<trace->hop_count; i++)
    {
      for(hop = trace->hops[i]; hop != NULL; hop = hop->hop_next)
	{
	  dump_trace_hop(hop);
	}
    }

  /* dump any last-ditch probing hops */
  for(hop = trace->lastditch; hop != NULL; hop = hop->hop_next)
    {
      dump_trace_hop(hop);
    }

  if(trace->pmtud != NULL)
    {
      printf("pmtud: ifmtu %d, pmtu %d\n",
	     trace->pmtud->ifmtu, trace->pmtud->pmtu);
      for(hop = trace->pmtud->hops; hop != NULL; hop = hop->hop_next)
	{
	  dump_trace_hop(hop);
	}
    }

  printf("\n");

  scamper_trace_free(trace);

  return;
}

static void dump_tracelb_reply(scamper_tracelb_probe_t *probe,
			       scamper_tracelb_reply_t *reply)
{
  scamper_icmpext_t *ie;
  struct timeval rtt;
  char from[32];
  uint32_t u32;
  uint16_t m;

  timeval_diff_tv(&rtt, &probe->tx, &reply->reply_rx);
  scamper_addr_tostr(reply->reply_from, from, sizeof(from));

  printf("   reply from: %s, rtt: %d.%06d, ttl: %d",
	 from, (int)rtt.tv_sec, (int)rtt.tv_usec, reply->reply_ttl);

  if(reply->reply_from->type == SCAMPER_ADDR_TYPE_IPV4)
    printf(", ipid: 0x%04x", reply->reply_ipid);
  printf("\n     ");

  if(reply->reply_flags & SCAMPER_TRACELB_REPLY_FLAG_TCP)
    {
      printf("tcp flags 0x%02x", reply->reply_tcp_flags);
      dump_tcp_flags(reply->reply_tcp_flags);
      printf("\n");
    }
  else
    {
      printf("icmp: %d/%d, q-tos: 0x%02x",
	     reply->reply_icmp_type, reply->reply_icmp_code,
	     reply->reply_icmp_q_tos);
      if(SCAMPER_TRACELB_REPLY_IS_ICMP_UNREACH(reply) ||
	 SCAMPER_TRACELB_REPLY_IS_ICMP_TTL_EXP(reply))
	{
	  printf(", q-ttl: %d", reply->reply_icmp_q_ttl);
	}
      printf("\n");

      for(ie = reply->reply_icmp_ext; ie != NULL; ie = ie->ie_next)
	{
	  if(SCAMPER_ICMPEXT_IS_MPLS(ie))
	    {
	      for(m=0; m<SCAMPER_ICMPEXT_MPLS_COUNT(ie); m++)
		{
		  u32 = SCAMPER_ICMPEXT_MPLS_LABEL(ie, m);
		  printf("   %9s: label %d exp %d s %d ttl %d\n",
			 (m == 0) ? "  icmp-ext mpls" : "", u32,
			 SCAMPER_ICMPEXT_MPLS_EXP(ie, m),
			 SCAMPER_ICMPEXT_MPLS_S(ie, m),
			 SCAMPER_ICMPEXT_MPLS_TTL(ie, m));
		}
	    }
	}
    }

  return;
}

static void dump_tracelb_probe(scamper_tracelb_t *trace,
			       scamper_tracelb_probe_t *probe)
{
  uint32_t i;

  printf("  probe flowid: %d, ttl: %d, attempt: %d, tx: %d.%06d\n",
	 probe->flowid, probe->ttl, probe->attempt,
	 (int)probe->tx.tv_sec, (int)probe->tx.tv_usec);

  for(i=0; i<probe->rxc; i++)
    {
      dump_tracelb_reply(probe, probe->rxs[i]);
    }

  return;
}

static void dump_tracelb(scamper_tracelb_t *trace)
{
  scamper_tracelb_link_t *link;
  scamper_tracelb_node_t *node;
  scamper_tracelb_probeset_t *set;
  char src[256], dst[256];
  uint16_t i, j, k, l;

  if(trace->src != NULL)
    {
      printf("tracelb from %s to %s\n",
	     scamper_addr_tostr(trace->src, src, sizeof(src)),
	     scamper_addr_tostr(trace->dst, dst, sizeof(dst)));
    }
  else
    {
      printf("tracelb to %s\n",
	     scamper_addr_tostr(trace->dst, dst, sizeof(dst)));
    }

  dump_list_summary(trace->list);
  dump_cycle_summary(trace->cycle);
  printf(" user-id: %d\n", trace->userid);
  dump_start(&trace->start);

  printf(" type: ");
  switch(trace->type)
    {
    case SCAMPER_TRACELB_TYPE_ICMP_ECHO:
      printf("icmp-echo id: %d", trace->sport);
      break;

    case SCAMPER_TRACELB_TYPE_UDP_DPORT:
      printf("udp-dport %d:%d", trace->sport, trace->dport);
      break;

    case SCAMPER_TRACELB_TYPE_UDP_SPORT:
      printf("udp-sport %d:%d", trace->sport, trace->dport);
      break;

    case SCAMPER_TRACELB_TYPE_TCP_SPORT:
      printf("tcp-sport %d:%d", trace->sport, trace->dport);
      break;

    case SCAMPER_TRACELB_TYPE_TCP_ACK_SPORT:
      printf("tcp-ack-sport %d:%d", trace->sport, trace->dport);
      break;

    default:
      printf("%d", trace->type);
      break;
    }
  printf(", tos: 0x%02x\n", trace->tos);

  printf(" firsthop: %d, attempts: %d, confidence: %d\n",
	 trace->firsthop, trace->attempts, trace->confidence);
  printf(" probe-size: %d, wait-probe: %dms, wait-timeout %ds\n",
	 trace->probe_size, trace->wait_probe * 10, trace->wait_timeout);
  printf(" nodec: %d, linkc: %d, probec: %d, probec_max: %d\n",
	 trace->nodec, trace->linkc, trace->probec, trace->probec_max);

  for(i=0; i<trace->nodec; i++)
    {
      node = trace->nodes[i];

      if(node->addr != NULL)
	scamper_addr_tostr(node->addr, src, sizeof(src));
      else
	snprintf(src, sizeof(src), "*");

      printf("node %d %s", i, src);
      if(SCAMPER_TRACELB_NODE_QTTL(node) != 0)
	printf(", qttl %d", node->q_ttl);
      printf("\n");

      for(j=0; j<node->linkc; j++)
	{
	  link = node->links[j];
	  if(link->from->addr != NULL)
	    scamper_addr_tostr(link->from->addr, src, sizeof(src));
	  else
	    snprintf(src, sizeof(src), "*");
	  if(link->to != NULL)
	    scamper_addr_tostr(link->to->addr, dst, sizeof(dst));
	  else
	    snprintf(dst, sizeof(dst), "*");
	  printf(" link %s -> %s hopc %d\n", src, dst, link->hopc);

	  for(k=0; k<link->hopc; k++)
	    {
	      set = link->sets[k];
	      for(l=0; l<set->probec; l++)
		dump_tracelb_probe(trace, set->probes[l]);
	    }
	}
    }

  printf("\n");

  scamper_tracelb_free(trace);
  return;
}

static void dump_ping_reply(scamper_ping_reply_t *reply)
{
  scamper_ping_reply_v4rr_t *v4rr;
  scamper_ping_reply_v4ts_t *v4ts;
  uint8_t i;
  char buf[256];

  printf("reply from %s, attempt: %d, rtt: %d.%06ds\n",
	 scamper_addr_tostr(reply->addr, buf, sizeof(buf)),
	 reply->probe_id+1, (int)reply->rtt.tv_sec, (int)reply->rtt.tv_usec);

  printf(" size: %d", reply->reply_size);
  if(reply->flags & SCAMPER_PING_REPLY_FLAG_REPLY_TTL)
    printf(", ttl: %d", reply->reply_ttl);
  if(reply->flags & SCAMPER_PING_REPLY_FLAG_PROBE_IPID)
    printf(", probe-ipid: 0x%04x", reply->probe_ipid);
  if(reply->flags & SCAMPER_PING_REPLY_FLAG_REPLY_IPID)
    printf(", reply-ipid: 0x%04x", reply->reply_ipid);
  printf("\n");

  if(SCAMPER_PING_REPLY_IS_ICMP(reply))
    {
      printf(" icmp type: %d, code: %d\n", reply->icmp_type, reply->icmp_code);
    }
  else if(SCAMPER_PING_REPLY_IS_TCP(reply))
    {
      printf(" tcp flags: %02x", reply->tcp_flags);
      dump_tcp_flags(reply->tcp_flags);
      printf("\n");
    }

  if((v4rr = reply->v4rr) != NULL)
    {
      printf(" record route:");
      for(i=0; i<v4rr->rrc; i++)
	{
	  if((i % 3) == 0 && i != 0)
	    printf("\n              ");

	  printf(" %-15s",
		 scamper_addr_tostr(v4rr->rr[i],buf,sizeof(buf)));
	}
      printf("\n");
    }

  if((v4ts = reply->v4ts) != NULL)
    {
      printf(" timestamp: ");
      if(v4ts->ips != NULL)
	{
	  for(i=0; i<v4ts->tsc; i++)
	    {
	      if((i % 2) == 0 && i != 0)
		printf("\n            ");
	      else if(i != 0)
		printf("    ");

	      printf("%-15s %d",
		     scamper_addr_tostr(v4ts->ips[i], buf, sizeof(buf)),
		     v4ts->tss[i]);
	    }
	}
      else
	{
	  for(i=0; i<v4ts->tsc; i++)
	    {
	      if((i % 3) == 0 && i != 0)
		printf("\n            ");
	      printf("%16d", v4ts->tss[i]);
	    }
	}
      printf("\n");
    }

  return;
}

static void dump_ping(scamper_ping_t *ping)
{
  scamper_ping_reply_t *reply;
  char buf[256];
  int i;

  scamper_addr_tostr(ping->src, buf, sizeof(buf));
  printf("ping from %s", buf);
  if(ping->flags & SCAMPER_PING_FLAG_SPOOF)
    printf(" (spoofed)");
  scamper_addr_tostr(ping->dst, buf, sizeof(buf));
  printf(" to %s\n", buf);

  dump_list_summary(ping->list);
  dump_cycle_summary(ping->cycle);
  printf(" user-id: %d\n", ping->userid);
  dump_start(&ping->start);

  printf(" probe count: %d, size: %d, wait: %d, ttl: %d",
	 ping->probe_count,ping->probe_size,ping->probe_wait,ping->probe_ttl);

  if(ping->reply_count > 0)
    printf(", replies-req: %d", ping->reply_count);
  printf("\n");

  switch(ping->probe_method)
    {
    case SCAMPER_PING_METHOD_ICMP_ECHO:
      printf(" method: icmp-echo");
      if((ping->flags & SCAMPER_PING_FLAG_ICMPSUM) != 0)
	printf(", icmp-csum: %04x", ping->probe_icmpsum);
      printf("\n");
      break;

    case SCAMPER_PING_METHOD_TCP_ACK:
      printf(" method: tcp-ack, sport: %d, dport: %d\n",
	     ping->probe_sport, ping->probe_dport);
      break;

    case SCAMPER_PING_METHOD_TCP_ACK_SPORT:
      printf(" method: tcp-ack-sport, base-sport: %d, dport: %d\n",
	     ping->probe_sport, ping->probe_dport);
      break;

    case SCAMPER_PING_METHOD_UDP:
      printf(" method: udp, sport: %d, dport %d\n",
	     ping->probe_sport, ping->probe_dport);
      break;

    case SCAMPER_PING_METHOD_UDP_DPORT:
      printf(" method: udp-dport, sport: %d, base-dport %d\n",
	     ping->probe_sport, ping->probe_dport);
      break;

    default:
      printf(" method: %d\n", ping->probe_method);
      break;
    }

  if(ping->probe_tsps != NULL)
    {
      printf(" timestamp-prespec:");
      for(i=0; i<ping->probe_tsps->ipc; i++)
	printf(" %s",
	       scamper_addr_tostr(ping->probe_tsps->ips[i],buf,sizeof(buf)));
      printf("\n");
    }

  /* dump pad bytes, if used */
  if(ping->probe_datalen > 0 && ping->probe_data != NULL)
    {
      if((ping->flags & SCAMPER_PING_FLAG_PAYLOAD) != 0)
	printf(" payload");
      else
	printf(" pattern");
      printf(" bytes (%d): ", ping->probe_datalen);
      for(i=0; i<ping->probe_datalen; i++)
	printf("%02x", ping->probe_data[i]);
      printf("\n");
    }

  printf(" probes-sent: %d, stop-reason: ", ping->ping_sent);
  switch(ping->stop_reason)
    {
    case SCAMPER_PING_STOP_NONE:
      printf("none"); break;

    case SCAMPER_PING_STOP_COMPLETED:
      printf("done"); break;

    case SCAMPER_PING_STOP_ERROR:
      printf("sendto errno %d", ping->stop_data); break;

    default:
      printf("reason 0x%02x data 0x%02x",
	      ping->stop_reason, ping->stop_data);
      break;
    }
  printf("\n");

  for(i=0; i<ping->ping_sent; i++)
    {
      for(reply = ping->ping_replies[i]; reply != NULL; reply = reply->next)
	{
	  dump_ping_reply(reply);
	}
    }

  printf("\n");

  scamper_ping_free(ping);

  return;
}

static void dump_dealias_probedef(scamper_dealias_probedef_t *def)
{
  scamper_dealias_probedef_icmp_t *icmp;
  char dst[128], src[128];

  printf(" probedef %d: dst: %s, src: %s, ttl: %d, tos: 0x%02x\n",
	  def->id,
	  scamper_addr_tostr(def->dst, dst, sizeof(dst)),
	  scamper_addr_tostr(def->src, src, sizeof(src)),
	  def->ttl, def->tos);
  if(SCAMPER_DEALIAS_PROBEDEF_PROTO_IS_ICMP(def))
    {
      icmp = &def->un.icmp;
      printf("  icmp type: %d, code: %d, csum: %04x, id: %04x\n",
	     icmp->type, icmp->code, icmp->csum, icmp->id);
    }
  else if(SCAMPER_DEALIAS_PROBEDEF_PROTO_IS_UDP(def))
    {
      if(def->method == SCAMPER_DEALIAS_PROBEDEF_METHOD_UDP)
	printf("  udp");
      else if(def->method == SCAMPER_DEALIAS_PROBEDEF_METHOD_UDP_DPORT)
	printf("  udp-dport");
      else
	printf("  udp-%d", def->method);
      printf(" %d:%d\n", def->un.udp.sport, def->un.udp.dport);
    }
  else if(SCAMPER_DEALIAS_PROBEDEF_PROTO_IS_TCP(def))
    {
      if(def->method == SCAMPER_DEALIAS_PROBEDEF_METHOD_TCP_ACK)
	printf("  tcp-ack");
      else if(def->method == SCAMPER_DEALIAS_PROBEDEF_METHOD_TCP_ACK_SPORT)
	printf("  tcp-ack-sport");
      else if(def->method == SCAMPER_DEALIAS_PROBEDEF_METHOD_TCP_SYN_SPORT)
	printf("  tcp-syn-sport");
      else
	printf("  tcp-%d", def->method);
      printf(" %d:%d ", def->un.tcp.sport, def->un.tcp.dport);
      dump_tcp_flags(def->un.tcp.flags);
      printf("\n");
    }
  else
    {
      printf("%d\n", def->method);
    }
  return;
}

static void dump_dealias(scamper_dealias_t *dealias)
{
  scamper_dealias_prefixscan_t *ps = dealias->data;
  scamper_dealias_mercator_t *mercator = dealias->data;
  scamper_dealias_radargun_t *radargun = dealias->data;
  scamper_dealias_ally_t *ally = dealias->data;
  scamper_dealias_bump_t *bump = dealias->data;
  scamper_dealias_probe_t *probe;
  scamper_dealias_reply_t *reply;
  struct timeval rtt;
  uint32_t i;
  uint16_t u16;
  uint8_t u8;
  char buf[256];
  int j;

  /* first line: dealias */
  printf("dealias");
  if(dealias->method == SCAMPER_DEALIAS_METHOD_MERCATOR)
    {
      scamper_addr_tostr(mercator->probedef.src, buf, sizeof(buf));
      printf(" from %s", buf);
      scamper_addr_tostr(mercator->probedef.dst, buf, sizeof(buf));
      printf(" to %s", buf);
    }
  printf("\n");

  /* dump list, cycle, start time */
  dump_list_summary(dealias->list);
  dump_cycle_summary(dealias->cycle);
  printf(" user-id: %d\n", dealias->userid);
  dump_start(&dealias->start);

  printf(" probes: %d, result: ", dealias->probec);
  switch(dealias->result)
    {
    case SCAMPER_DEALIAS_RESULT_NONE:
      printf("none");
      break;

    case SCAMPER_DEALIAS_RESULT_ALIASES:
      printf("aliases");
      break;

    case SCAMPER_DEALIAS_RESULT_NOTALIASES:
      printf("not aliases");
      break;
    }
  printf("\n");

  /* method headers */
  printf(" method: ");
  if(dealias->method == SCAMPER_DEALIAS_METHOD_MERCATOR)
    {
      printf("mercator, attempts: %d, timeout: %ds\n",
	     mercator->attempts, mercator->wait_timeout);
      dump_dealias_probedef(&mercator->probedef);
    }
  else if(dealias->method == SCAMPER_DEALIAS_METHOD_ALLY)
    {
      printf("ally, attempts: %d, fudge: %d, "
	     "wait-probe: %dms, wait-timeout: %ds",
	     ally->attempts,ally->fudge,ally->wait_probe,ally->wait_timeout);
      if(SCAMPER_DEALIAS_ALLY_IS_NOBS(dealias))
	printf(", nobs");
      printf("\n");

      dump_dealias_probedef(&ally->probedefs[0]);
      dump_dealias_probedef(&ally->probedefs[1]);
    }
  else if(dealias->method == SCAMPER_DEALIAS_METHOD_BUMP)
    {
      printf("bump, attempts: %d, wait-probe: %dms, bump-limit: %d\n",
	     bump->attempts, bump->wait_probe, bump->bump_limit);
      dump_dealias_probedef(&bump->probedefs[0]);
      dump_dealias_probedef(&bump->probedefs[1]);
    }
  else if(dealias->method == SCAMPER_DEALIAS_METHOD_RADARGUN)
    {
      printf("radargun, wait-probe: %dms, wait-round: %dms\n"
	     "  wait-timeout: %ds, attempts: %d, probedefc: %d\n",
	     radargun->wait_probe, radargun->wait_round,
	     radargun->wait_timeout, radargun->attempts, radargun->probedefc);
      if((u8 = radargun->flags) != 0)
	{
	  printf("  flags: ");
	  for(i=0; i<8; i++)
	    {
	      if((u8 & (1 << i)) == 0)
		continue;
	      switch(1 << i)
		{
		case SCAMPER_DEALIAS_RADARGUN_FLAG_SHUFFLE:
		  printf("shuffle");
		  break;

		default:
		  printf("0x%02x", 1<<i);
		  break;
		}

	      u8 &= ~(1 << i);
	      if(u8 != 0)
		printf(", ");
	      else
		break;
	    }
	  printf("\n");
	}
      for(i=0; i<radargun->probedefc; i++)
	dump_dealias_probedef(&radargun->probedefs[i]);
    }
  else if(dealias->method == SCAMPER_DEALIAS_METHOD_PREFIXSCAN)
    {
      printf("prefixscan, %s:",
	     scamper_addr_tostr(ps->a, buf, sizeof(buf)));
      printf("%s/%d",
	     scamper_addr_tostr(ps->b, buf, sizeof(buf)), ps->prefix);
      if(ps->ab != NULL)
	printf(", alias: %s/%d",
	       scamper_addr_tostr(ps->ab, buf, sizeof(buf)),
	       scamper_addr_prefixhosts(ps->b, ps->ab));
      printf("\n");

      printf("  attempts: %d, replyc: %d, fudge: %d, wait-probe: %dms, "
	     "wait-timeout: %ds", ps->attempts, ps->replyc, ps->fudge,
	     ps->wait_probe, ps->wait_timeout);
      if(SCAMPER_DEALIAS_PREFIXSCAN_IS_NOBS(dealias))
	printf(", nobs");
      printf("\n");
      if(ps->xc > 0)
	{
	  printf("  exclude:");
	  for(u16=0; u16<ps->xc; u16++)
	    printf(" %s", scamper_addr_tostr(ps->xs[u16], buf, sizeof(buf)));
	  printf("\n");
	}
      for(i=0; i<ps->probedefc; i++)
	dump_dealias_probedef(&ps->probedefs[i]);
    }
  else
    {
      printf("%d\n", dealias->method);
    }

  for(i=0; i<dealias->probec; i++)
    {
      probe = dealias->probes[i];
      printf(" probe: %d, def: %d, seq: %d, ipid: %04x, tx: %d.%06d\n",
	     i, probe->probedef->id, probe->seq, probe->ipid,
	     (int)probe->tx.tv_sec, (int)probe->tx.tv_usec);
      for(j=0; j<probe->replyc; j++)
	{
	  reply = probe->replies[j];
	  timeval_diff_tv(&rtt, &probe->tx, &reply->rx);
	  printf("  reply: %d, src: %s",
		 j, scamper_addr_tostr(reply->src, buf, sizeof(buf)));
	  printf(" ipid: %04x, ttl: %d, rtt: %d.%06d\n",
		 reply->ipid, reply->ttl,
		 (int)rtt.tv_sec, (int)rtt.tv_usec);

	  if(SCAMPER_DEALIAS_REPLY_IS_ICMP(reply))
	    {
	      printf("  icmp-type: %d, icmp-code: %d",
		     reply->icmp_type, reply->icmp_code);

	      if(SCAMPER_DEALIAS_REPLY_IS_ICMP_UNREACH(reply) ||
		 SCAMPER_DEALIAS_REPLY_IS_ICMP_TTL_EXP(reply))
		{
		  printf(", icmp-q-ttl: %d", reply->icmp_q_ip_ttl);
		}
	      printf("\n");
	    }
	  else if(SCAMPER_DEALIAS_REPLY_IS_TCP(reply))
	    {
	      printf("   tcp flags:");
	      dump_tcp_flags(reply->tcp_flags);
	      printf("\n");
	    }
	  else
	    {
	      printf("  reply proto %d\n", reply->proto);
	    }
	}
    }

  printf("\n");

  scamper_dealias_free(dealias);
  return;
}

static void dump_neighbourdisc(scamper_neighbourdisc_t *nd)
{
  scamper_neighbourdisc_probe_t *probe;
  scamper_neighbourdisc_reply_t *reply;
  struct timeval rtt;
  uint16_t i, j;
  char a[64], b[64];

  printf("neighbourdisc\n");
  dump_list_summary(nd->list);
  dump_cycle_summary(nd->cycle);
  printf(" user-id: %d\n", nd->userid);
  dump_start(&nd->start);

  if(nd->method == SCAMPER_NEIGHBOURDISC_METHOD_ARP ||
     nd->method == SCAMPER_NEIGHBOURDISC_METHOD_ND_NSOL)
    {
      if(nd->method == SCAMPER_NEIGHBOURDISC_METHOD_ARP)
	printf(" method: arp");
      else
	printf(" method: ipv6 nsol");

      printf(", attempts: %d, wait: %ds, replyc: %d, iface: %s\n",
	     nd->attempts, nd->wait, nd->replyc, nd->ifname);
      printf(" our-mac: %s\n",
	     scamper_addr_tostr(nd->src_mac, a, sizeof(a)));
      printf(" flags: 0x%02x", nd->flags);
      if(nd->flags != 0)
	{
	  printf(" (");
	  if(nd->flags & SCAMPER_NEIGHBOURDISC_FLAG_ALLATTEMPTS)
	    printf(" all-attempts");
	  if(nd->flags & SCAMPER_NEIGHBOURDISC_FLAG_FIRSTRESPONSE)
	    printf(" first-response");
	  printf(" )");
	}
      printf("\n");
      printf(" query:  who-has %s tell %s\n",
	     scamper_addr_tostr(nd->dst_ip,  a, sizeof(a)),
	     scamper_addr_tostr(nd->src_ip,  b, sizeof(b)));
      printf(" result: %s is-at %s\n", a,
	     scamper_addr_tostr(nd->dst_mac, b, sizeof(b)));
    }

  for(i=0; i<nd->probec; i++)
    {
      probe = nd->probes[i];
      printf(" probe: %d, tx: %d.%06d\n",
	     i, (int)probe->tx.tv_sec, (int)probe->tx.tv_usec);

      for(j=0; j<probe->rxc; j++)
	{
	  reply = probe->rxs[j];
	  timeval_diff_tv(&rtt, &probe->tx, &reply->rx);
	  printf("  reply: %d, rtt: %d.%06d, mac: %s\n",
		 i, (int)rtt.tv_sec, (int)rtt.tv_usec,
		 scamper_addr_tostr(reply->mac, a, sizeof(a)));
	}
    }

  printf("\n");

  scamper_neighbourdisc_free(nd);
  return;
}

static void dump_tbit(scamper_tbit_t *tbit)
{
  scamper_tbit_pmtud_t *pmtud;
  scamper_tbit_app_http_t *http;
  scamper_tbit_pkt_t *pkt;
  struct timeval diff;
  uint16_t len, i, u16, datalen;
  uint8_t proto, flags, iphlen, tcphlen, mf;
  uint32_t seq, ack, server_isn, client_isn, off, id, u32;
  char src[64], dst[64], buf[32], ipid[12], fstr[32];
  char *str;
  int frag;

  ipid[0] = '\0';

  /* Start dumping the tbit test information */
  printf("tbit from %s to %s\n",
	 scamper_addr_tostr(tbit->src, src, sizeof(src)),
	 scamper_addr_tostr(tbit->dst, dst, sizeof(dst)));

  dump_list_summary(tbit->list);
  dump_cycle_summary(tbit->cycle);
  printf(" user-id: %d\n", tbit->userid);
  dump_start(&tbit->start);

  printf(" sport: %d, dport: %d\n", tbit->sport, tbit->dport);
  printf(" client-mss: %d, server-mss: %d\n",
	 tbit->client_mss, tbit->server_mss);

  switch(tbit->result)
    {
    case SCAMPER_TBIT_RESULT_NONE: str = "none"; break;
    case SCAMPER_TBIT_RESULT_TCP_NOCONN: str = "tcp-noconn"; break;
    case SCAMPER_TBIT_RESULT_TCP_NOCONN_RST: str = "tcp-noconn-rst"; break;
    case SCAMPER_TBIT_RESULT_TCP_RST: str = "tcp-rst"; break;
    case SCAMPER_TBIT_RESULT_TCP_ERROR: str = "tcp-error"; break;
    case SCAMPER_TBIT_RESULT_ERROR: str = "error"; break;
    case SCAMPER_TBIT_RESULT_ABORTED: str = "aborted"; break;
    case SCAMPER_TBIT_RESULT_PMTUD_NOACK: str = "pmtud-noack"; break;
    case SCAMPER_TBIT_RESULT_PMTUD_NODATA: str = "pmtud-nodata"; break;
    case SCAMPER_TBIT_RESULT_PMTUD_TOOSMALL: str = "pmtud-toosmall"; break;
    case SCAMPER_TBIT_RESULT_PMTUD_NODF: str = "pmtud-nodf"; break;
    case SCAMPER_TBIT_RESULT_PMTUD_FAIL: str = "pmtud-fail"; break;
    case SCAMPER_TBIT_RESULT_PMTUD_SUCCESS: str = "pmtud-success"; break;
    case SCAMPER_TBIT_RESULT_PMTUD_CLEARDF: str = "pmtud-cleardf"; break;
    default: snprintf(buf, sizeof(buf), "%d", tbit->result); str = buf; break;
    }
  printf(" result: %s\n", str);

  if(tbit->type == SCAMPER_TBIT_TYPE_PMTUD && tbit->data != NULL)
    {
      pmtud = tbit->data;
      printf(" mtu: %d, ptb_retx: %d\n", pmtud->mtu, pmtud->ptb_retx);
    }

  if(tbit->app_proto == SCAMPER_TBIT_APP_HTTP && tbit->app_data != NULL)
    {
      http = tbit->app_data;
      printf(" app: http");
      if(http->host != NULL && http->file != NULL)
	printf(", url: http://%s%s", http->host, http->file);
      else if(http->host != NULL)
	printf(", url: http://%s", http->host);
      else
	printf(", file: %s", http->file);
      printf("\n");
    }

  client_isn = 0;
  server_isn = 0;

  for(i=0; i<tbit->pktc; i++)
    {
      pkt = tbit->pkts[i];
      frag = 0; mf = 0; id = 0; off = 0;

      if((pkt->data[0] >> 4) == 4)
        {
	  iphlen = (pkt->data[0] & 0xf) * 4;
	  len = bytes_ntohs(pkt->data+2);
	  proto = pkt->data[9];
	  if(pkt->data[6] & 0x20)
	    mf = 1;
	  id  = bytes_ntohs(pkt->data+4);
	  off = (bytes_ntohs(pkt->data+6) & 0x1fff) * 8;
	  if(mf != 0 || off != 0)
	    frag = 1;
	  snprintf(ipid, sizeof(ipid), " %04x", bytes_ntohs(pkt->data+4));
        }
      else if((pkt->data[0] >> 4) == 6)
        {
	  iphlen = 40;
	  len = bytes_ntohs(pkt->data+4) + iphlen;
	  proto = pkt->data[6];

	  for(;;)
            {
	      switch(proto)
                {
		case IPPROTO_HOPOPTS:
		case IPPROTO_DSTOPTS:
		case IPPROTO_ROUTING:
		  proto = pkt->data[iphlen+0];
		  iphlen += (pkt->data[iphlen+1] * 8) + 8;
		  continue;

		case IPPROTO_FRAGMENT:
		  if(pkt->data[iphlen+3] & 0x1)
		    mf = 1;
		  off = (bytes_ntohs(pkt->data+iphlen+2) & 0xfff8);
		  id  = bytes_ntohl(pkt->data+iphlen+4);
		  proto = pkt->data[iphlen+0];
		  iphlen += 8;
		  frag = 1;
		  continue;
                }
	      break;
            }
        }
      else
	{
	  continue;
	}	

      timeval_diff_tv(&diff, &tbit->start, &pkt->tv);
      printf(" [%3d.%03d] %s ", (int)diff.tv_sec, (int)(diff.tv_usec / 1000),
	     pkt->dir == SCAMPER_TBIT_PKT_DIR_TX ? "TX" : "RX");

      if(frag != 0)
	snprintf(fstr,sizeof(fstr),"%u:%u%s", id, off, mf != 0 ? " MF" : "");
      else
	fstr[0] = '\0';

      if(off != 0)
	{
	  printf("        %4dF%21s %s %s", len, "", ipid, fstr);
	}
      else if(proto == IPPROTO_TCP)
        {
	  seq     = bytes_ntohl(pkt->data+iphlen+4);
	  ack     = bytes_ntohl(pkt->data+iphlen+8);
	  flags   = pkt->data[iphlen+13];
	  tcphlen = ((pkt->data[iphlen+12] & 0xf0) >> 4) * 4;

	  if(flags & 0x2)
            {
	      if(flags & 0x10)
                {
		  server_isn = seq;
		  str = "SYN/ACK";
                }
	      else
                {
		  client_isn = seq;
		  str = "SYN";
                }
            }
	  else if(flags & 0x1)
	    str = "FIN";
	  else if(flags & 0x4)
	    str = "RST";
	  else
	    str = "";

	  if(pkt->dir == SCAMPER_TBIT_PKT_DIR_TX)
            {
	      seq -= client_isn + ((seq >= client_isn) ? 0 : TCP_MAX_SEQNUM+1);
	      ack -= server_isn + ((ack >= server_isn) ? 0 : TCP_MAX_SEQNUM+1);
            }
	  else    
            {
	      seq -= server_isn + ((seq >= server_isn) ? 0 : TCP_MAX_SEQNUM+1);
	      ack -= client_isn + ((ack >= client_isn) ? 0 : TCP_MAX_SEQNUM+1);
            }

	  datalen = len - iphlen - tcphlen;

	  printf("%-7s %4d%s", str, len, frag != 0 ? "F" : " ");
	  if(datalen != 0)
	    snprintf(buf, sizeof(buf), " seq = %u:%u(%d)", seq, ack, datalen);
	  else
	    snprintf(buf, sizeof(buf), " seq = %u:%u", seq, ack);
	  printf("%-22s%s", buf, ipid);
	  if(frag != 0) printf(" %s", fstr);
	  if(datalen > 0 && (pkt->data[0] >> 4) == 4 && pkt->data[6] & 0x40)
	    printf(" DF");
        }
      else if(proto == IPPROTO_ICMP)
        {
	  if(pkt->data[iphlen+0] == 3 && pkt->data[iphlen+1] == 4)
	    {
	      u16 = bytes_ntohs(pkt->data+iphlen+6);
	      printf("PTB     %4d  mtu = %d", len, u16);
	    }
        }
      else if(proto == IPPROTO_ICMPV6)
        {
	  if(pkt->data[iphlen+0] == 2)
	    {
	      u32 = bytes_ntohl(pkt->data+iphlen+4);
	      printf("PTB     %4d  mtu = %d", len, u32);
	    }
	}

      printf("\n");
    }

  fprintf(stdout,"\n");

  scamper_tbit_free(tbit);
  return;
}

static void dump_cycle(scamper_cycle_t *cycle, const char *type)
{
  time_t tt;
  char buf[32];

  if(strcmp(type, "start") == 0 || strcmp(type, "def") == 0)
    tt = cycle->start_time;
  else
    tt = cycle->stop_time;

  memcpy(buf, ctime(&tt), 24); buf[24] = '\0';

  printf("cycle %s, list %s %d, cycle %d, time %s\n",
	 type, cycle->list->name, cycle->list->id, cycle->id, buf);
  scamper_cycle_free(cycle);
  return;
}

static void dump_list(scamper_list_t *list)
{
  printf("list id %d, name %s", list->id, list->name);
  if(list->descr != NULL) printf(", descr \"%s\"", list->descr);
  printf("\n");
  scamper_list_free(list);
  return;
}

static void dump_addr(scamper_addr_t *addr)
{
  char buf[128];
  printf("addr %s\n", scamper_addr_tostr(addr, buf, sizeof(buf)));
  scamper_addr_free(addr);
  return;
}

int main(int argc, char *argv[])
{
  scamper_file_t        *file;
  scamper_file_filter_t *filter;
  uint16_t filter_types[] = {
    SCAMPER_FILE_OBJ_LIST,
    SCAMPER_FILE_OBJ_CYCLE_START,
    SCAMPER_FILE_OBJ_CYCLE_DEF,
    SCAMPER_FILE_OBJ_CYCLE_STOP,
    SCAMPER_FILE_OBJ_TRACE,
    SCAMPER_FILE_OBJ_PING,
    SCAMPER_FILE_OBJ_TRACELB,
    SCAMPER_FILE_OBJ_DEALIAS,
    SCAMPER_FILE_OBJ_NEIGHBOURDISC,
    SCAMPER_FILE_OBJ_TBIT,
  };
  uint16_t filter_cnt = sizeof(filter_types)/sizeof(uint16_t);
  void     *data;
  uint16_t  type;
  int       f;

#ifdef _WIN32
  WSADATA wsaData;
  WSAStartup(MAKEWORD(2,2), &wsaData);
#endif

#if defined(DMALLOC)
  free(malloc(1));
#endif

  if((filter = scamper_file_filter_alloc(filter_types, filter_cnt)) == NULL)
    {
      usage();
      fprintf(stderr, "could not alloc fitler\n");
      return -1;
    }

  for(f=0; f<argc; f++)
    {
      if(f == 0)
	{
	  if(argc > 1)
	    continue;

	  if((file=scamper_file_openfd(STDIN_FILENO,"-",'r',"warts")) == NULL)
	    {
	      usage();
	      fprintf(stderr, "could not use stdin\n");
	      return -1;
	    }
	}
      else
	{
	  if((file = scamper_file_open(argv[f], 'r', NULL)) == NULL)
	    {
	      usage();
	      fprintf(stderr, "could not open %s\n", argv[f]);
	      return -1;
	    }
	}

      while(scamper_file_read(file, filter, &type, &data) == 0)
	{
	  /* hit eof */
	  if(data == NULL)
	    goto done;

	  switch(type)
	    {
	    case SCAMPER_FILE_OBJ_ADDR:
	      dump_addr(data);
	      break;

	    case SCAMPER_FILE_OBJ_TRACE:
	      dump_trace(data);
	      break;

	    case SCAMPER_FILE_OBJ_PING:
	      dump_ping(data);
	      break;

	    case SCAMPER_FILE_OBJ_TRACELB:
	      dump_tracelb(data);
	      break;

	    case SCAMPER_FILE_OBJ_DEALIAS:
	      dump_dealias(data);
	      break;

	    case SCAMPER_FILE_OBJ_NEIGHBOURDISC:
	      dump_neighbourdisc(data);
	      break;

	    case SCAMPER_FILE_OBJ_TBIT:
	      dump_tbit(data);
	      break;

	    case SCAMPER_FILE_OBJ_LIST:
	      dump_list(data);
	      break;

	    case SCAMPER_FILE_OBJ_CYCLE_START:
	      dump_cycle(data, "start");
	      break;

	    case SCAMPER_FILE_OBJ_CYCLE_STOP:
	      dump_cycle(data, "stop");
	      break;

	    case SCAMPER_FILE_OBJ_CYCLE_DEF:
	      dump_cycle(data, "def");
	      break;
	    }
	}

    done:
      scamper_file_close(file);

      if(argc == 1)
	break;
    }

  scamper_file_filter_free(filter);
  return 0;
}
