#ifndef INCLUDE_MANAGER_HH
#define INCLUDE_MANAGER_HH

// Copyright (c) 1995-2000 The University of Cincinnati.
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Author: Dale E. Martin          dmartin@cliftonlabs.com

//---------------------------------------------------------------------------

#include <iostream>
#include <string>
using std::string;
using std::ostream;
using std::cerr;
using std::endl;

template <class Type> class hash_table;

/**
   This class is responsible for managing the #includes generated by the
   code generator.  The basic idea is that it will keep track of what has
   already been included on a per file basis and only generate a particular
   include once per file.
*/

class include_manager {
public:
  include_manager();
  ~include_manager();
  
  void add_include( const string file_to_include, bool system_include );

  void publish_includes( ostream &os );

private:
  /**
     This inner class defines the hash container for us.  It simply
     associates an include file name with a flag saying whether it's
     already been included or not.
  */
  class included_file_record {
  private:
    /**
       The included file name.
    */
    string included_file_name;

    /**
       Tells us if this is a system include <> or a user include "".
    */
    bool system_include;
    
    /**
       The flag saying whether it's been included or not.
    */
    bool already_included;


  public:
    /**
       Constructor.
    */
    included_file_record() : included_file_name( "" ), 
			     system_include( false ),
			     already_included( false ){}
    ~included_file_record(){}
    /**
       Tell this record that it's been included.
    */
    void set_already_included(){ already_included = true; }

    /**
       See if this file has already been included.
    */
    bool get_already_included(){ return already_included; }

    /**
       Set that this is a system include.
    */
    void set_system_include(){ system_include = true; }

    /**
       See if this is a system include
    */
    bool get_system_include(){ return system_include; }

    /**
       "set_key" required by the hash table.
    */
    void set_key( const string key,  const int len );

    /**
       "test_key" required by the hash table.
    */
    int test_key( const string key, 
		  const int len ){ return strncmp( key.c_str(), included_file_name.c_str(), len ); }

    /**
       Read the file name from this record.
    */
    const string get_included_file_name(){ return included_file_name; }
  };

  hash_table<included_file_record> &my_included_files;

  /**
     Get the hash table for this generated file.
  */
  included_file_record *get_record_for_file( const string included_file_name );

  /**
     Ask if we need to publish this include for this file.  Simply calling
     this method will mark that the include has already been published, so
     make sure if you get "true" back that you really publish the include!
     
     @return Does this include really need published?     
  */
  bool need_to_publish( const string file_to_include,
			const string file_name );


};

#endif
