
// Copyright (c) 1996-2003 The University of Cincinnati.
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Timothy J. McBrayer
//          Malolan Chetlur    
//          Krishnan Subramani 
//          Umesh Kumar V. Rajasekaran
//          Narayanan Thondugulam

#include "IIR_SignalAssignmentStatement.hh"
#include "IIR_AliasDeclaration.hh"
#include "IIR_Label.hh"
#include "IIR_WaveformElement.hh"
#include "IIR_AboveAttribute.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "set.hh"
#include "symbol_table.hh"
#include "savant.hh"
#include "published_file.hh"

extern symbol_table *cgen_sym_tab_ptr;

IIRScram_SignalAssignmentStatement::~IIRScram_SignalAssignmentStatement() {}

void
IIRScram_SignalAssignmentStatement::_publish_vhdl(ostream &_vhdl_out) {

  _publish_vhdl_stmt_label(_vhdl_out);

  get_target()->_publish_vhdl(_vhdl_out);
  _vhdl_out << " <= ";

  _publish_vhdl_delay_mechanism(_vhdl_out, get_delay_mechanism(), get_reject_time_expression());
  waveform._publish_vhdl(_vhdl_out);

}

void
IIRScram_SignalAssignmentStatement::_publish_cc( published_file &_cc_out ) {
  ASSERT(get_target()->_is_resolved());
  IIR_WaveformElement* waveform_element = waveform.first();
  // Indicates the first itereation.  We need this to publish the
  // waveforms other than the first with transport delay mechanism.
  IIR_Boolean first = TRUE;

  while(waveform_element != NULL) {

    if (get_target()->get_kind() == IIR_AGGREGATE) {
      // Create a temporary aggregate and use that in the signal assignment
      _cc_out.open_scope( "{" );
      ASSERT ( get_target()->get_subtype() != NULL );
      _cc_out << get_target()->get_subtype()->_get_cc_type_name();
      _cc_out.open_scope( " tempAggregate(" );
      get_target()->_publish_cc_constructor_args( _cc_out );
      get_target()->_publish_cc_initialization_value( _cc_out );
      _cc_out.close_scope( ");" );
    }
    
    if(_is_currently_publishing_subprogram() == TRUE) {
      _cc_out << "processPtr->";
    }

    if(waveform_element->_is_null_transaction() == TRUE) {
      _cc_out << OS("assignNullTransaction(");
    }
    else {
      _cc_out << OS( "assignSignal( " );
    }

    _publish_cc_target( _cc_out );

    _cc_out << "," << NL();

    if(_is_currently_publishing_subprogram() == TRUE) {
      _cc_out << "processPtr,";
    }
    else {
      _cc_out << "this,";
    }      
    _cc_out << NL();

    waveform_element->_publish_cc_lvalue( _cc_out ); // data source and delay value

    // Now the reject time.
    _publish_cc_time_expression( _cc_out, first );
    _cc_out << "," << NL();
    get_target()->_publish_cc_bounds( _cc_out );
    _cc_out << "," << NL();
    waveform_element->_publish_cc_bounds( _cc_out );
    _cc_out << CS( ");" );
    waveform_element = waveform.successor(waveform_element);
    first = FALSE;

    if (get_target()->get_kind() == IIR_AGGREGATE) {
      _cc_out.close_scope("}");
    }
  }
}

void 
IIRScram_SignalAssignmentStatement::_type_check(){
  _type_check_target_and_waveform( &waveform );
  _type_check_mechanism_and_time( &waveform );
}

void
IIRScram_SignalAssignmentStatement::_get_list_of_input_signals( set<IIR> *list ){
  waveform._get_list_of_input_signals(list);
}

void
IIRScram_SignalAssignmentStatement::_get_signal_source_info( set<IIR> *siginfo ){
  ASSERT(get_target()->_is_signal() == TRUE);
  IIR* my_target = get_target();
  ASSERT(my_target != NULL);
  if(my_target->get_kind() == IIR_ALIAS_DECLARATION) {
    siginfo->add( ((IIR_AliasDeclaration*)my_target)->get_name() );
  } 
  else if( my_target->_is_name() == TRUE  || 
	   my_target->_is_aggregate_expression() == TRUE ){
    my_target->_get_signal_source_info(siginfo);
  }
  else {
    ASSERT( my_target->_is_iir_declaration() == TRUE );
    siginfo->add( my_target );
  }
}

IIR *
IIRScram_SignalAssignmentStatement::_clone() {
  IIR_SignalAssignmentStatement *stmt;
  IIR *target;
  IIR_WaveformElement *newwave, *oldwave;

  stmt = new IIR_SignalAssignmentStatement();
  IIRScram_SequentialStatement::_clone(stmt);

  target = get_target();
  target = target->_clone();
  stmt->set_target(target);
  stmt->set_delay_mechanism(get_delay_mechanism());
  stmt->set_reject_time_expression(get_reject_time_expression());

  oldwave = waveform.first();
  while(oldwave != NULL) {
    newwave = (IIR_WaveformElement*)oldwave->_clone();
    stmt->waveform.append(newwave);
    oldwave = waveform.successor(oldwave);
  }
  return stmt;
}

IIR *
IIRScram_SignalAssignmentStatement::_get_target(){
  return get_target();
}

void 
IIRScram_SignalAssignmentStatement::_set_target( IIR *new_target ){
  set_target( new_target );
}

IIR_DelayMechanism 
IIRScram_SignalAssignmentStatement::_get_delay_mechanism(){
  return get_delay_mechanism();
}

void 
IIRScram_SignalAssignmentStatement::_set_delay_mechanism( IIR_DelayMechanism new_mech ){
  set_delay_mechanism( new_mech );
}

IIR *
IIRScram_SignalAssignmentStatement::_get_reject_time_expression(){
  return get_reject_time_expression();
}

IIR_Boolean 
IIRScram_SignalAssignmentStatement::_is_resolved(){
  IIR_Boolean retval = TRUE;

  if( get_target() && get_target()->_is_resolved() == FALSE ){
    retval = FALSE;
  }
  else if( get_reject_time_expression() && get_reject_time_expression()->_is_resolved() == FALSE ){
    retval = FALSE;
  }
  else if ( waveform._is_resolved() == FALSE ){
    retval = FALSE;
  }

  return retval;
}

void 
IIRScram_SignalAssignmentStatement::_set_reject_time_expression( IIR *new_reject_time ){
  set_reject_time_expression( new_reject_time );
}

void
IIRScram_SignalAssignmentStatement::_publish_cc_target( published_file &_cc_out ){
  IIR_Declaration *destsig  = get_target()->_get_prefix_declaration();
  if( destsig != NULL && cgen_sym_tab_ptr->in_scope(destsig) == FALSE ){
    cgen_sym_tab_ptr->add_declaration(destsig);
  }
  switch(get_target()->get_kind()) {
  case IIR_SLICE_NAME:
    destsig->_publish_cc_lvalue( _cc_out );
    break;
    
  case IIR_AGGREGATE:
    _cc_out << "tempAggregate";
    break;
    
  default:
    get_target()->_publish_cc_lvalue( _cc_out );
  }
}

void
IIRScram_SignalAssignmentStatement::_publish_cc_time_expression( published_file &_cc_out, 
								 IIR_Boolean first ){
  // REJECT TIME
  if( first == FALSE || get_delay_mechanism() == IIR_TRANSPORT_DELAY ){
    _cc_out << "SAVANT_ZERO_TIME";
  } 
  else {
    if( get_reject_time_expression() != NULL ){
      get_reject_time_expression()->_publish_cc_lvalue( _cc_out );
    }
    else{
      if( waveform.first()->get_time() != NULL ){
	waveform.first()->get_time()->_publish_cc_lvalue( _cc_out );
      } else {
	_cc_out << "SAVANT_ZERO_TIME";
      }
    } 
  }
}

IIR_Boolean
IIRScram_SignalAssignmentStatement::_is_above_attribute_found() {
  return waveform._is_above_attribute_found();
}

void
IIRScram_SignalAssignmentStatement::
_build_above_attribute_set(set<IIR_AboveAttribute> *to_build) {
  waveform._build_above_attribute_set(to_build);
}

visitor_return_type *
IIRScram_SignalAssignmentStatement::_accept_visitor( node_visitor *visitor, 
						     visitor_argument_type *arg ){
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_SignalAssignmentStatement(this, arg);
}

