// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <iomanip.h>

#include "grid.h"
#include "util.h"
#include "framebase.h"
#include "fitsimage.h"
#include "rotstr.h"

#include <Xutil.h>

extern "C" {
  #include "wcs.h"
  #include "ast.h"
  #include "grf.h"
}

static char* black = "black";
static char* white = "white";
static char* red = "red";
static char* green = "green";
static char* blue = "blue";
static char* cyan = "cyan";
static char* magenta = "magenta";
static char* yellow = "yellow";

Attribute::Attribute(FrameBase* p) : parent(p)
{
  style_ = SOLID;
  width_ = 1;

  font_ = HELVETICA;
  size_ = 10;
  tkfont_ = Tk_GetFont(parent->interp, parent->tkwin, "helvetica 10 normal");

  colour_ = WHITE;
  colorName_ = white;
  setColour(colour_);
}

Attribute::~Attribute()
{
  if (tkfont_)
    Tk_FreeFont(tkfont_);
}

void Attribute::setStyle(double v)
{
  switch ((int)v) {
  case SOLID:
  case DASH:
    break;
  default:
    return;
  }

  style_ = (Style)((int)v);
}

void Attribute::setWidth(double v)
{
  width_ = v;
}

void Attribute::setSize(double v)
{
  if (v >= 1) {
    size_ = (int)v;
    initFont();
  }
}

void Attribute::setFont(double v)
{
  switch ((Font)((int)v)) {
  case DEFAULT:
  case TIMES:
  case HELVETICA:
  case SYMBOL:
  case COURIER:

  case DEFAULTBOLD:
  case TIMESBOLD:
  case HELVETICABOLD:
  case SYMBOLBOLD:
  case COURIERBOLD:

  case DEFAULTITALIC:
  case TIMESITALIC:
  case HELVETICAITALIC:
  case SYMBOLITALIC:
  case COURIERITALIC:
    font_ = (Font)((int)(v));
    initFont();
    return;

  default:
    return;
  }
}

void Attribute::initFont()
{
  if (tkfont_)
    Tk_FreeFont(tkfont_);
  tkfont_ = NULL;

  ostringstream str;
  switch (font_) {
  case DEFAULT:
    str << "system " << size_ << " normal" << ends;
    break;
  case TIMES:
    str << "times " << size_ << " normal" << ends;
    break;
  case HELVETICA:
    str << "helvetica " << size_ << " normal" << ends;
    break;
  case SYMBOL:
    str << "symbol " << size_ << " normal" << ends;
    break;
  case COURIER:
    str << "courier " << size_ << " normal" << ends;
    break;

  case DEFAULTBOLD:
    str << "system " << size_ << " bold" << ends;
    break;
  case TIMESBOLD:
    str << "times " << size_ << " bold" << ends;
    break;
  case HELVETICABOLD:
    str << "helvetica " << size_ << " bold" << ends;
    break;
  case SYMBOLBOLD:
    str << "symbol " << size_ << " bold" << ends;
    break;
  case COURIERBOLD:
    str << "courier " << size_ << " bold" << ends;
    break;

  case DEFAULTITALIC:
    str << "system " << size_ << " italic" << ends;
    break;
  case TIMESITALIC:
    str << "times " << size_ << " italic" << ends;
    break;
  case HELVETICAITALIC:
    str << "helvetica " << size_ << " italic" << ends;
    break;
  case SYMBOLITALIC:
    str << "symbol " << size_ << " italic" << ends;
    break;
  case COURIERITALIC:
    str << "courier " << size_ << " italic" << ends;
    break;
  default:
    return;
  }

  tkfont_ = Tk_GetFont(parent->interp, parent->tkwin, str.str().c_str());
}

void Attribute::setColour(double v)
{
  // check for color change
  if (colour_ == (Colour)((int)(v)))
    return;

  switch ((int)v) {
  case BLACK:
    colorName_ = black;
    break;
  case WHITE:
    colorName_ = white;
    break;
  case RED:
    colorName_ = red;
    break;
  case GREEN:
    colorName_ = green;
    break;
  case BLUE:
    colorName_ = blue;
    break;
  case CYAN:
    colorName_ = cyan;
    break;
  case MAGENTA:
    colorName_ = magenta;
    break;
  case YELLOW:
    colorName_ = yellow;
    break;
  default:
    return;
  }

  color_ = parent->getColor(colorName_);
  colour_ = (Colour)((int)(v));
}

Grid* gr = NULL;

Grid::Grid(FrameBase* p, 
	   CoordSystem sys, SkyFrame sky, SkyFormat format, 
	   GridType t, const char* o) 
  : parent(p), system_(sys), sky_(sky), skyFormat_(format), type_(t)
{
  option_ = dupstr(o);

  line = new Attribute(parent);
  text = new Attribute(parent);
  psMode = FrameBase::RGB;
}

Grid::~Grid()
{
  if (option_)
    delete [] option_;

  if (line)
    delete line;
  if (text)
    delete text;
}

int Grid::render()
{
  switch (type_) {
  case ANALYSIS:
    return doit(X11,WIDGETBB);
  case PUBLICATION:
    if (!parent->isMosaic())
      return doit(X11,IMAGEBB);
    else
      return doit(X11,MOSAICBB);
  }
}

int Grid::ps(int mode)
{
  psMode = mode;

  switch (type_) {
  case ANALYSIS:
    return doit(PS,WIDGETBB);
  case PUBLICATION:
    if (!parent->isMosaic())
      return doit(PS,IMAGEBB);
    else
      return doit(PS,MOSAICBB);
  }
}

int Grid::matrixMap(void* fs, Matrix& mx, const char* str)
{
  AstFrameSet* frameSet = (AstFrameSet*)fs;

  double ss[] = {mx.matrix(0,0),mx.matrix(1,0),mx.matrix(0,1),mx.matrix(1,1)};
  double tt[] = {mx.matrix(2,0),mx.matrix(2,1)};

  AstMatrixMap* mm;
  if (!(mm= astMatrixMap(2, 2, 0, ss, "")))
    return 0;

  AstShiftMap* sm;
  if (!(sm = astShiftMap(2, tt, "")))
    return 0;

  AstCmpMap* cmp;
  if (!(cmp = astCmpMap(mm, sm, 1, "")))
    return 0;

  astAddFrame(frameSet, AST__CURRENT, cmp, astFrame(2, str));
}

int Grid::doit(RenderMode rm, RenderBound bound)
{
  FitsImage* fits = *(parent->keyFits);

  if (!fits)
    return 1;

  int width = fits->width();
  int height = fits->height();

  astClearStatus; // just to make sure
  astBegin; // start memory management

  AstFrameSet* frameSet = NULL;
  AstPlot* plot = NULL;

  if (!(frameSet = astFrameSet(astFrame(2,"Domain=WIDGET"),"")))
    goto error;

  // map from Widget to Image
  matrixMap(frameSet,fits->getWidgetToImage(),"Domain=IMAGE");

  switch (system_) {
  case IMAGE:
    break;

  case PHYSICAL:
    matrixMap(frameSet,fits->getImageToPhysical(),"Domain=PHYSICAL");
    break;

  case AMPLIFIER:
    matrixMap(frameSet,fits->getImageToAmplifier(),"Domain=AMPLIFIER");
    break;

  case DETECTOR:
    matrixMap(frameSet,fits->getImageToDetector(),"Domain=DETECTOR");
    break;

  default:
    if (!fits->hasWCS(system_))
      goto error;

    if (fits->hasWCSEqu(system_)) {
      // map from Image to Equatorial WCS

      WorldCoor* wcs = fits->getWCS(system_);

      AstFrameSet* wcsfs = NULL;

      // we can handle most projections, but not these
      if (wcs->prjcode==WCS_PIX ||
	  wcs->prjcode==WCS_LIN ||
	  wcs->prjcode==WCS_PLT)
	goto error;

      // try to build a valid header, so that wcssubs and ast will agree.
      // however, for certain projections, just feed the header directly to
      // ast.
      if (wcs->prjcode==WCS_DSS)
	wcsfs = (AstFrameSet*)fits2ast();
      else
      	wcsfs = (AstFrameSet*)wcs2ast(wcs);

      if (!wcsfs)
	goto error;

      // set desired WCS system

      switch (sky_) {
      case FK4:
	astSet(wcsfs, "System=FK4, Equinox=B1950");
	break;
      case FK5:
	astSet(wcsfs, "System=FK5, Equinox=J2000");
	break;
      case ICRS:
	astSet(wcsfs, "System=FK5, Equinox=J2000");
	break;
      case GALACTIC:
	astSet(wcsfs, "System=Galactic");
	break;
      case ECLIPTIC:
	astSet(wcsfs, "System=Ecliptic");
	// this is a kludge to get AST to agree with wcssubs
	astSetD(wcsfs, "Equinox", astGetD(wcsfs, "Epoch"));
	break;
      }

      // invert wcsfs  base/current 
      astInvert(wcsfs);

      // add wcsfs to frameset
      // this will link frame 2 of frameset to frame 3 wcsfs
      astAddFrame(frameSet,2,astUnitMap(2,""),wcsfs);
      // set the current of frameset to 4 (was 3)
      astSetI(frameSet,"current",4);
    }
    else {
      // map from Image to Linear WCS
      WorldCoor* wcs = fits->getWCS(system_);
      Matrix m(wcs->cd[0],wcs->cd[1],wcs->cd[2],wcs->cd[3],
	       wcs->crval[0]-
	       (wcs->cd[0]*wcs->crpix[0])-(wcs->cd[2]*wcs->crpix[1]),
	       wcs->crval[1]-
	       (wcs->cd[1]*wcs->crpix[0])-(wcs->cd[3]*wcs->crpix[1]));

      matrixMap(frameSet,fits->getImageToDetector(),"Domain=WCS");
    }
    break;
  }

  astSet(frameSet,"Title=%s", " ");

  if (DebugAST) {
    int status = astStatus;
    astClearStatus;
    astShow(frameSet);
    astSetStatus(status);
  }

  // create astPlot
  float gbox[4];
  double pbox[4];

  switch (bound) {
  case WIDGETBB:
    {
      gbox[0] = pbox[0] = 0;
      gbox[1] = pbox[1] = 0;
      gbox[2] = pbox[2] = parent->options->width-1;
      gbox[3] = pbox[3] = parent->options->height-1;
    }
    break;
  case IMAGEBB:
    {
      Matrix m = fits->getImageToWidget();
      BBox b = parent->imageBBox();
      b.expand(.5);

      Vector ll = b.ll   * m;
      Vector lr = b.lr() * m;
      Vector ur = b.ur   * m;
      Vector ul = b.ul() * m;
      
      BBox bb(ll,ll);
      bb.bound(lr);
      bb.bound(ur);
      bb.bound(ul);

      gbox[0] = pbox[0] = (int)bb.ll[0];
      gbox[1] = pbox[1] = (int)bb.ll[1];
      gbox[2] = pbox[2] = (int)bb.ur[0];
      gbox[3] = pbox[3] = (int)bb.ur[1];
    }
    break;
  case MOSAICBB:
    {
      Matrix m = fits->getImageToWidget();
      BBox b = parent->imageBBox() * *(parent->currentWCSmatrix);
      b.expand(.5);

      Vector ll = b.ll   * m;
      Vector lr = b.lr() * m;
      Vector ur = b.ur   * m;
      Vector ul = b.ul() * m;
      
      BBox bb(ll,ll);
      bb.bound(lr);
      bb.bound(ur);
      bb.bound(ul);

      gbox[0] = pbox[0] = (int)bb.ll[0];
      gbox[1] = pbox[1] = (int)bb.ll[1];
      gbox[2] = pbox[2] = (int)bb.ur[0];
      gbox[3] = pbox[3] = (int)bb.ur[1];
    }
    break;
  }

  plot = astPlot(frameSet, gbox, pbox, option_);

  // and now create astGrid

  gr = this;
  renderMode = rm;
  astGrid(plot);

  astEnd; // now, clean up memory
  return 1;

error:
  astEnd;
  return 0;
}

// Generic

int Grid::gAttr(int which, double value, double* old, int prim)
{
  if (value == AST__BAD) {
    if (old)
      *old = 0;
    return 1;
  }

  Attribute* attr;
  switch (prim) {
  case GRF__TEXT:
    attr = text;

    switch (which) {
    case GRF__STYLE:
      break;
    case GRF__WIDTH:
      break;
    case GRF__SIZE:
      if (old)
	*old = attr->size();
      attr->setSize(value);
      break;
    case GRF__FONT:
      if (old)
	*old = attr->font();
      attr->setFont(value);
      break;
    case GRF__COLOUR:
      if (old)
	*old = attr->colour();
      attr->setColour(value);
      break;
    }

    break;
  case GRF__LINE:
    attr = line;

    switch (which) {
    case GRF__STYLE:
      if (old)
	*old = attr->style();
      attr->setStyle(value);
      break;
    case GRF__WIDTH:
      if (old)
	*old = attr->width();
      attr->setWidth(value);
      break;
    case GRF__SIZE:
      break;
    case GRF__FONT:
      break;
    case GRF__COLOUR:
      if (old)
	*old = attr->colour();
      attr->setColour(value);
      break;
    }

    break;
  }

  return 1;
}

int Grid::gQch(float* chv, float* chh)
{
  if (text->tkfont()) {
    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(text->tkfont(), &metrics);

    *chv = (float)metrics.linespace;
    *chh = (float)metrics.linespace;
    return 1;
  }
  else {
    *chv = *chh = 0;
    return 0;
  }
}

int Grid::gTxExt(const char* txt, float x, float y, const char* just,
		 float upx, float upy, float* xb, float* yb)
{
  if (txt && text->tkfont()) {
    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(text->tkfont(), &metrics);
    int width = Tk_TextWidth(text->tkfont(), txt, strlen(txt));

    // Translate from CC to CC justification
    Matrix m1,m2;
    if (just) {
      switch (just[0]) {
      case 'T':
	m1 = Translate(0,metrics.linespace/2);
	break;
      case 'C':
	break;
      case 'B':
	m1 = Translate(0,-(metrics.ascent-metrics.linespace/2));
	break;
      case 'M':
	m1 = Translate(0,-metrics.linespace/2);
      }

      switch (just[1]) {
      case 'L':
	m2 = Translate(width/2,0);
	break;
      case 'C':
	break;
      case 'R':
	m2 = Translate(-width/2,0);
	break;
      }
    }

    Vector c = Vector(x,y) *  m1 * m2;

    BBox bb = BBox(c*Translate(-width/2,-metrics.linespace/2), 
    		   c*Translate(width/2,metrics.linespace/2));

    xb[0] = bb.ll[0];
    yb[0] = bb.ll[1];

    xb[1] = bb.ur[0];
    yb[1] = bb.ll[1];

    xb[2] = bb.ur[0];
    yb[2] = bb.ur[1];

    xb[3] = bb.ll[0];
    yb[3] = bb.ur[1];
  }
  else {
    xb[0] = xb[1] = xb[2] = xb[3] = 0;
    yb[0] = yb[1] = yb[2] = yb[3] = 0;
  }

  return 1;
}

int Grid::gScales(float* alpha, float* beta)
{
  /*
  *alpha = float(XDisplayWidthMM(parent->display,parent->screenNumber)) / 
    float(XDisplayWidth(parent->display,parent->screenNumber));

  *beta = float(XDisplayHeightMM(parent->display,parent->screenNumber)) / 
    float(XDisplayHeight(parent->display,parent->screenNumber));
  */
  return 1;
}

int Grid::gCap(int cap, int value)
{
  switch (cap) {
  case GRF__SCALES:
    return 0;
  case GRF__MJUST:
    return 1;
  case GRF__ESC:
    return 0;
  }
}

// X11 Render functions

int Grid::x11Line(int n, const float* x, const float* y)
{
  if (n<2 || !x || !y)
    return 1;

  XSetForeground(parent->display, parent->gridGC, line->color());
  int w = line->width();
  if (w<1)
    w = 1;
  switch (line->style()) {
  case Attribute::SOLID:
    XSetLineAttributes(parent->display, parent->gridGC, w, 
		       LineSolid, CapButt, JoinMiter);
    break;
  case Attribute::DASH:
    XSetLineAttributes(parent->display, parent->gridGC, w, 
		       LineOnOffDash, CapButt, JoinMiter);
    char dlist[] = {8,3};
    XSetDashes(parent->display, parent->gridGC, 0, dlist, 2);
    break;
  }

  Vector s = Vector(x[0],y[0]);
  Vector e;
  for (int i=0; i<n; i++) {
    e = Vector(x[i],y[i]);
    XDRAWLINE(parent->display, parent->basePixmap, parent->gridGC, 
	      (int)s[0],(int)s[1],(int)e[0],(int)e[1]);
    s = e;
  }

  return 1;
}

int Grid::x11Text(const char* txt, float x, float y, 
		   const char* just, float upx, float upy)
{
  //  cerr << txt << ' ' << x << ' ' << y << ' ' 
  //       << just << ' ' << upx << ' ' << upy << endl;

  if (txt && txt[0] && text->tkfont() && just) {
    XSetFont(parent->display, parent->gridGC, Tk_FontId(text->tkfont()));
    XSetForeground(parent->display, parent->gridGC, text->color());

    Vector v = Vector(x,y);
    double angle = calcTextAngle(Vector(upx,upy).angle());

    if ((angle>-.01 && angle<+.01))
      drawString(txt, v, just);
    else
      drawRotString(txt, v, just, angle);
    return 1;
  }

  return 0;
}

void Grid::drawString(const char* txt, Vector v, const char* just)
{
  Tk_FontMetrics metrics;
  Tk_GetFontMetrics(text->tkfont(), &metrics);
  int width = Tk_TextWidth(text->tkfont(), txt, strlen(txt));

  // Translate from CC to left,baseline justification
  Matrix m1,m2;
  switch (just[0]) {
  case 'T':
    m1 = Translate(0,metrics.ascent);
    break;
  case 'C':
    m1 = Translate(0,metrics.ascent-metrics.linespace/2);
    break;
  case 'B':
    break;
  case 'M':
    m1 = Translate(0,-metrics.descent);
  }

  switch (just[1]) {
  case 'L':
    break;
  case 'C':
    m2 = Translate(-width/2,0);
    break;
  case 'R':
    m2 = Translate(-width,0);
    break;
  }

  Vector c = v *  m1 * m2;

  XDrawString(parent->display, parent->basePixmap, parent->gridGC,
	      (int)c[0], (int)c[1], txt, strlen(txt));
}

void Grid::drawRotString(const char* txt, Vector v, 
			 const char* just, double angle)
{
  Tk_FontMetrics metrics;
  Tk_GetFontMetrics(text->tkfont(), &metrics);
  int width = Tk_TextWidth(text->tkfont(), txt, strlen(txt));
  int height = metrics.linespace;

  // Translate from CC to CC justification
  Matrix m1,m2;
  if (just) {
    switch (just[0]) {
    case 'T':
      m1 = Translate(0,metrics.linespace/2);
      break;
    case 'C':
      break;
    case 'B':
      m1 = Translate(0,-(metrics.ascent-metrics.linespace/2));
      break;
    case 'M':
      m1 = Translate(0,-metrics.linespace/2);
    }

    switch (just[1]) {
    case 'L':
      m2 = Translate(width/2,0);
      break;
    case 'C':
      break;
    case 'R':
      m2 = Translate(-width/2,0);
      break;
    }
  }

  Matrix m = Translate(-v)*Rotate(-angle)*m1*m2*Rotate(angle)*Translate(v);

  XDrawRotString(parent->display, parent->basePixmap, parent->gridGC,
		 parent->baseXImage->byte_order, 
		 parent->getBlackColor(), parent->getWhiteColor(),
		 txt, text->tkfont(), angle, v*m);
}

// PS Render functions

int Grid::psLine(int n, const float* x, const float* y)
{
  if (n<2 || !x || !y)
    return 1;

  Matrix m = parent->getWidgetToCanvas();

  psColor(line);
  psWidth(line);
  psStyle(line);

  {
    ostringstream str;
    Vector v = Vector(x[0],y[0]) * m;
    str << "newpath " << endl
	<< v.TkCanvasPs(parent->canvas) << " moveto" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }

  for (int i=1; i<n; i++) {
    ostringstream str;
    Vector v = Vector(x[i],y[i]) * m;
    str << v.TkCanvasPs(parent->canvas) << " lineto" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }

  {
    ostringstream str;
    str << "stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }

  return 1;
}

int Grid::psText(const char* txt, float x, float y, 
		const char* just, float upx, float upy)
{
  if (txt && txt[0] && text->tkfont() && just) {
    double angle = calcTextAngle(Vector(upx,upy).angle());

    Tcl_DString psFont;
    Tcl_DStringInit(&psFont);
    int psSize = Tk_PostscriptFontName(text->tkfont(), &psFont);

    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(text->tkfont(), &metrics);
    int width = Tk_TextWidth(text->tkfont(), txt, strlen(txt));

  // Translate from CC to left,baseline justification
    Matrix m1,m2;
    switch (just[0]) {
    case 'T':
      m1 = Translate(0,metrics.ascent);
      break;
    case 'C':
      m1 = Translate(0,metrics.ascent-metrics.linespace/2);
      break;
    case 'B':
      break;
    case 'M':
      m1 = Translate(0,-metrics.descent);
    }

    switch (just[1]) {
    case 'L':
      break;
    case 'C':
      m2 = Translate(-width/2.,0);
      break;
    case 'R':
      m2 = Translate(-width,0);
      break;
    }

    Vector v = Vector(x,y) * parent->getWidgetToCanvas();
    Matrix m = Translate(-v)*Rotate(-angle)*m1*m2*Rotate(angle)*Translate(v);
    
    ostringstream str;
    str << '/' << Tcl_DStringValue(&psFont) 
	<< " findfont " << psSize << " scalefont setfont" << endl;

    Tcl_DStringFree(&psFont);

    psColor(text);

    str << "gsave " 
	<< (v*m).TkCanvasPs(parent->canvas) << " moveto" << endl
      	<< radToDeg(angle) << " rotate " 
	<< '(' << psQuote(txt) << ')' << " show"
	<< " grestore" << endl << ends;

    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
    return 1;
  }

  return 0;
}

// PS Support

void Grid::psColor(Attribute* attr)
{
  ostringstream str;
  switch ((FrameBase::PSColorSpace)psMode) {
  case FrameBase::BW:
  case FrameBase::GRAY:
    psColorGray(attr->colorName(), str);
    str << " setgray";
    break;
  case FrameBase::RGB:
    psColorRGB(attr->colorName(), str);
    str << " setrgbcolor";
    break;
  case FrameBase::CMYK:
    psColorCMYK(attr->colorName(), str);
    str << " setcmykcolor";
    break;
  }
  str << endl << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
}

void Grid::psWidth(Attribute* attr)
{
  ostringstream str;
  str << attr->width() << " setlinewidth" << endl << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
}

void Grid::psStyle(Attribute* attr)
{
  ostringstream str;
  switch (attr->style()) {
  case Attribute::SOLID:
    str << "[] 0 setdash" << endl << ends;
    break;
  case Attribute::DASH:
    str << "[8 3] 0 setdash" << endl << ends;
    break;
  }
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
}

double Grid::calcTextAngle(double a)
{
  // our angle is 90 off from ast's, and the other direction
  double r = -(a - M_PI/2);

  // and add any image rotation

  // normalize
  if (r>0)
    while (r>2*M_PI)
      r -= 2*M_PI;
  else 
    while (r<0)
      r += 2*M_PI;

  // and flip if upside down
  // we only want to do this if the number is really, really upside down
  // you might have the case of a curved vertical line
  if (r > M_PI/2 && r <= 3*M_PI/2)
    r -= M_PI;

  return r;
}

// AST support

void* Grid::fits2ast() 
{
  // read FITS header into astChannel

  // we may have an error, just reset
  astClearStatus;

  // new fitschan
  AstFitsChan* chan = astFitsChan(NULL, NULL, "");
  if (!astOK || chan == AST__NULL)
    return NULL;

  // no warning messages
  astClear(chan,"Warnings");

  // fill up chan
  char* header = (*(parent->channelFits))->getCards();
  int ncards = (*(parent->channelFits))->getNCards();
  for (int i=0; i<ncards; i++) {
    char buf[81];
    strncpy(buf,header+(i*80),80);
    buf[80] = '\0';

    astPutFits(chan, buf, 0);
    // sometimes, we get a bad parse, just ignore
    if (!astOK)
      astClearStatus;
  }

  // parse FITS header for WCS

  // we may have an error, just reset
  astClearStatus;
  astClear(chan, "Card");

  // parse header
  AstFrameSet* frameSet = (AstFrameSet*)astRead(chan);

  // do we have anything?
  if (!astOK || frameSet == AST__NULL || 
      strcmp(astGetC(frameSet,"Class"), "FrameSet"))
    return NULL;

  return frameSet;
}

void* Grid::wcs2ast(void* vwcs) 
{
  // read wcs struct into astChannel

  // we may have an error, just reset
  astClearStatus;

  // new fitschan
  AstFitsChan* chan;
  chan = astFitsChan(NULL, NULL, "");
  if (!astOK || chan == AST__NULL)
    return NULL;

  // no warning messages
  astClear(chan,"Warnings");

  // fill up chan

  FitsFile* fits = (*(parent->keyFits))->imageFile();
  WorldCoor* wcs = (WorldCoor*)vwcs;

  if (!(strncmp("RA---TNX", wcs->ctype[0], 8))) {
    putFitsCard(chan, "CTYPE1", "RA---TAN");
    putFitsCard(chan, "CTYPE2", "DEC--TAN");
  }
  else if (!(strncmp("XLON", wcs->ctype[0], 4))) {
    // this is really just ra/dec with north pole of value 0
    putFitsCard(chan, "CTYPE1", "RA---TAN");
    putFitsCard(chan, "CTYPE2", "DEC--TAN");
    putFitsCard(chan, "LATPOLE", -90);
  }
  else {
    putFitsCard(chan, "CTYPE1", wcs->ctype[0]);
    putFitsCard(chan, "CTYPE2", wcs->ctype[1]);
  }
 
  putFitsCard(chan, "CRPIX1", wcs->crpix[0]);
  putFitsCard(chan, "CRPIX2", wcs->crpix[1]);

  putFitsCard(chan, "CRVAL1", wcs->crval[0]);
  putFitsCard(chan, "CRVAL2", wcs->crval[1]);

  // This is very complicated/. AST is very, very, very picky as to which
  // keywords it use...
  //
  //   first see if any CD's present
  //     if so, if there is a rotation, use CD's, else use just CDELTS
  //   else use CDELT's and
  //     see if any PC's present, use them
  //       else see if any CROTA's present, use them

  if (fits->find("CD1_1")) {
    if (!wcs->cd[1] && !wcs->cd[2]) {
      putFitsCard(chan, "CDELT1", wcs->cdelt[0]);
      putFitsCard(chan, "CDELT2", wcs->cdelt[1]);
    }
    else {
      putFitsCard(chan, "CD1_1", wcs->cd[0]);
      putFitsCard(chan, "CD1_2", wcs->cd[1]);
      putFitsCard(chan, "CD2_1", wcs->cd[2]);
      putFitsCard(chan, "CD2_2", wcs->cd[3]);
    }
  }
  else if (fits->find("CDELT1")) {
    putFitsCard(chan, "CDELT1", wcs->cdelt[0]);
    putFitsCard(chan, "CDELT2", wcs->cdelt[1]);

    if (fits->find("PC1_1")) {
      putFitsCard(chan, "PC1_1", wcs->pc[0]);
      putFitsCard(chan, "PC1_2", wcs->pc[1]);
      putFitsCard(chan, "PC2_1", wcs->pc[2]);
      putFitsCard(chan, "PC2_2", wcs->pc[3]);
    }
    else {
      if (fits->find("CROTA1"))
	putFitsCard(chan, "CROTA1", wcs->rot);
      if (fits->find("CROTA2"))
	putFitsCard(chan, "CROTA2", wcs->rot);
    }
  }

  // equiniox
  putFitsCard(chan, "EQUINOX", wcs->equinox);

  // from wcssub/wcsinit.c line 800
  // wcs->epoch = 1900.0 + (mjd - 15019.81352) / 365.242198781;
  putFitsCard(chan, "MJD-OBS", (wcs->epoch-1900)*365.242198781+15019.81352);

  if (!strncmp("RA",wcs->ctype[0],2) || !strncmp("RA",wcs->ctype[1],2)) {
    if (!strncmp("FK4",wcs->radecsys,3) ||
	!strncmp("FK5",wcs->radecsys,3) ||
	!strncmp("ICRS",wcs->radecsys,3))
      putFitsCard(chan, "RADESYS", wcs->radecsys);
  }

  if (wcs->latpole != 999)
    putFitsCard(chan, "LATPOLE", wcs->latpole);
  if (wcs->longpole != 999)
    putFitsCard(chan, "LONPOLE", wcs->longpole);

  // Projection parameters
  // PVi_m
  for (int i=1; i<=2; i++)
    for (int m=0; m<=9; m++) {
      ostringstream str;
      str << "PV" << i << '_' << m << ends;
      if (fits->find(str.str().c_str())) {
	float val = fits->getReal(str.str().c_str(),0);
	putFitsCard(chan, str.str().c_str(), val);
      }
    }

  // PSi_m
  for (int i=1; i<=2; i++)
    for (int m=0; m<=9; m++) {
      ostringstream str;
      str << "PS" << i << '_' << m << ends;
      if (fits->find(str.str().c_str())) {
	float val = fits->getReal(str.str().c_str(),0);
	putFitsCard(chan, str.str().c_str(), val);
      }
    }
   
  // rewind chan
  astClear(chan, "Card");

  // parse header
  AstFrameSet* frameSet = (AstFrameSet*)astRead(chan);

  // do we have anything?
  if (!astOK || frameSet == AST__NULL || 
      strcmp(astGetC(frameSet,"Class"), "FrameSet"))
    return NULL;

  if (wcs->coorflip) {
    int orr[] = {2,1};
    astPermAxes(frameSet,orr);
  }

  return frameSet;
}

void Grid::putFitsCard(void* chan, const char* key, const char* value)
{
  char buf[80];
  memset(buf,'\0', 80);

  ostringstream str;
  str.setf(ios::left,ios::adjustfield);
  str.width(8);
  str << key << "= '" << value << "'";
  memcpy(buf,str.str().c_str(),str.str().length());

  astPutFits(chan, buf, 0);
  astClearStatus;

  if (DebugAST)
    cerr << buf << endl;
}

void Grid::putFitsCard(void* chan, const char* key, double value)
{
  char buf[80];
  memset(buf,'\0', 80);

  ostringstream str;
  str.setf(ios::left,ios::adjustfield);
  str.setf(ios::scientific,ios::floatfield);
  str.width(8);
  str.precision(12);
  str << key << "= " << value;
  memcpy(buf,str.str().c_str(),str.str().length());

  astPutFits(chan, buf, 0);
  astClearStatus;

  if (DebugAST)
    cerr << buf << endl;
}

