// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "tcl.h"
#include <Xlib.h>

#include "panner.h"
#include "util.h"

// Parser Stuff

#undef yyFlexLexer
#define yyFlexLexer pnFlexLexer
#include <FlexLexer.h>

static pnFlexLexer* lex = NULL;    // used by pnlex
static Panner* pn = NULL;          // used by pnerror
extern int pnparse(void*);

int pnlex()
{
  return (lex ? lex->yylex() : 0);
}

void pnerror(const char* m)
{
  if (pn) {
    pn->error(m);
    const char* cmd = lex ? lex->YYText() : (const char*)NULL;
    if (cmd && cmd[0] != '\n') {
      pn->error(": ");
      pn->error(cmd);
    }
  }
}

// Public Member Functions

Panner::Panner(Tcl_Interp* i, Tk_Canvas c, Tk_Item* item) : Widget(i, c, item)
{
  thumbnail = 0;
  highLite = 0;
  panning = 0;
  useBBox = 1;
  useImageCompass = 1;
  useWCSCompass = 1;
  validWCSCompass = 0;
  useFont = 1;
  needsUpdate = 0;

  font = NULL;

  bboxGC = NULL;
  compassGC = NULL;
}

Panner::~Panner()
{ 
  if (bboxGC)
    XFreeGC(display, bboxGC);

  if (compassGC)
    XFreeGC(display, compassGC);

  if (font)
    Tk_FreeFont(font);
}

#if __GNUC__ >= 3
int Panner::parse(istringstream& istr)
#else
int Panner::parse(istrstream& istr)
#endif
{
  result = TCL_OK;
  lex = new pnFlexLexer(&istr);
  pn = this;
  pnparse(this);

  delete lex;
  lex = NULL;
  return result;
}

void Panner::update()
{
  needsUpdate = 1; 
  redraw();
}

// Required Virtual Functions

// UpdatePixmap. This function is responsable for creating a valid 
// pixmap the size of the current Panner

int Panner::updatePixmap(const BBox& bb)
{
  // bb is in canvas coords
  // create a valid pixmap if needed

  if (!pixmap) {
    if (!(pixmap = Tk_GetPixmap(display, Tk_WindowId(tkwin), 
				options->width, options->height, depth))) {
      cerr << "Panner Internal Error: Unable to Create Pixmap" << endl;
      exit(1);
    }
    updateGCs();
  }

  if (needsUpdate) {
    if (thumbnail) {
      XSetClipOrigin(display, gc, 0, 0);
      XCopyArea(display, thumbnail, pixmap, gc, 0, 0,
		options->width, options->height, 0, 0);

      if (useBBox)
	renderBBox();

      if (useImageCompass)
	renderImageCompass();

      if (useWCSCompass && validWCSCompass)
	renderWCSCompass();
    }
    else
      clearPixmap();

    needsUpdate = 0;
  }

  return TCL_OK;
}

void Panner::invalidPixmap()
{
  Widget::invalidPixmap();
  update();
}

// Command Functions

#if __GNUC__ >= 3
void Panner::getBBoxCmd()
{
  ostringstream str;
  str << bbox << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);  
}
#else
void Panner::getBBoxCmd()
{
  char buf[64];
  ostrstream str(buf, 64);
  str << bbox << ends;
  Tcl_AppendResult(interp, buf, NULL);  
}
#endif

#if __GNUC__ >= 3
void Panner::getSizeCmd()
{
  ostringstream str;
  str << options->width << " " << options->height << ends;
  Tcl_AppendResult(interp, str.str().c_str(), NULL);  
}
#else
void Panner::getSizeCmd()
{
  char buf[64];
  ostrstream str(buf, 64);
  str << options->width << " " << options->height << ends;
  Tcl_AppendResult(interp, buf, NULL);  
}
#endif

void Panner::highLiteCmd(int which)
{
  if (highLite != which) {
    highLite = !highLite;
    update();
  }
}

void Panner::highLiteCmd(Vector v)
{
  if (highLite != bbox.isIn(v)) {
    highLite = !highLite;
    update();
  }
}

void Panner::panToCmd(Vector v)
{
  bbox = v;
  update();
}

void Panner::panBeginCmd(Vector v)
{
  if (useBBox && bbox.isIn(v)) {
    panStart = v;
    panning = 1;
  }
}

void Panner::panMotionCmd(Vector v)
{
  if (panning && useBBox) {
    bbox += v - panStart;
    panStart = v;
    update();
  }
}

void Panner::panEndCmd(Vector v)
{
  if (panning && useBBox) {
    bbox += v - panStart;
    panning = 0;
    update();
  }
}

void Panner::setImageCompassCmd(int w)
{
  useImageCompass = w ? 1 : 0;
  update();
}

void Panner::setWCSCompassCmd(int w)
{
  useWCSCompass = w ? 1 : 0;
  update();
}

void Panner::setBBoxCmd(int w)
{
  useBBox = w ? 1 : 0;
  update();
}

void Panner::updateCmd(Pixmap p)
{
  thumbnail = p;
  update();
}

void Panner::updateBBoxCmd(BBox bb)
{
  bbox = bb;
  update();
}

void Panner::updateImageCompassCmd(Rotate o)
{
  imageCompass = (Matrix)o;
  update();
}

void Panner::updateWCSCompassCmd()
{
  validWCSCompass = 0;
  update();
}

void Panner::updateWCSCompassCmd(Vector n, Vector e)
{
  validWCSCompass = 1;
  wcsNorth = n;
  wcsEast = e;
  update();
}

void Panner::warpCmd(Vector v)
{
  XWARPPOINTER(display, None, None, 0, 0, 0, 0, (int)v[0], (int)v[1]);
}

// Private Functions

void Panner::updateGCs()
{
  if (!bboxGC) {
    bboxGC = XCreateGC(display, pixmap, 0, NULL);
    XSetForeground(display, bboxGC, cyanColor->pixel);
  }

  if (!font) {
    font = Tk_GetFont(interp, tkwin, "helvetica 9 normal");
    if (font)
      Tk_GetFontMetrics(font, &metric);
  }  

  if (!compassGC) {
    compassGC = XCreateGC(display, pixmap, 0, NULL);
    XSetLineAttributes(display, compassGC, 1, LineSolid, CapButt, JoinMiter);
    if (font)
      XSetFont(display, compassGC, Tk_FontId(font));
  }
}

void Panner::renderBBox()
{
  XSetLineAttributes(display, bboxGC, (highLite ? 2 : 1), LineSolid, 
		     CapButt,JoinMiter);
  Vector size = bbox.size();
  XDRAWRECTANGLE(display, pixmap, bboxGC, 
		 (int)bbox.ll[0], (int)bbox.ll[1], (int)size[0], (int)size[1]);
}

void Panner::renderImageCompass()
{
  const int length = (int)((options->width/2 + options->height/2)/2 * .4);
  Vector center(options->width/2., options->height/2.);

  // X arm

  renderArm(length, center, imageCompass, "X", greenColor->pixel);

  // Y arm

  Rotate rotY = Rotate(M_PI/2) * imageCompass;
  renderArm(length, center, rotY, "Y", greenColor->pixel); // Y
}

void Panner::renderWCSCompass()
{
  const int length = (int)((options->width/2 + options->height/2)/2 * .25);
  Vector center(options->width/2., options->height/2.);

  // East arm

  Rotate rotEast(-wcsEast.angle());
  renderArm(length, center, rotEast, "E", yellowColor->pixel);

  // North arm

  Rotate rotNorth(-wcsNorth.angle());
  renderArm(length, center, rotNorth, "N", yellowColor->pixel);
}

void Panner::renderArm(int length, Vector& center, Rotate& rot, char* str, 
		       int color)
{
  // set GC

  XSetForeground(display, compassGC, color);
  const int textOffset = 15; // Text offset
  const int tip = 6;  // length from end of line to tip of arrow
  const int tail = 2; // length from end of line to tails of arrow
  const int wc = 2;   // width of arrow at end of line
  const int wt = 3;   // width of arrow at tails

  // Arrow-- oriented on Y axis

  Vector arrow[6];
  arrow[0] = Vector(0, tip);
  arrow[1] = Vector(-wc, 0);
  arrow[2] = Vector(-wt, -tail);
  arrow[3] = Vector(0, 0);
  arrow[4] = Vector(wt, -tail);
  arrow[5] = Vector(wc, 0);

  // Staff-- oriented on X axis

  XPoint arrowArray[6];
  Matrix arrowMatrix = Rotate(M_PI/2) * 
    Translate(length,0) * 
    rot * 
    Translate(center);
  for (int i=0; i<6; i++) {
    Vector r = (arrow[i] * arrowMatrix).round();
    arrowArray[i].x = (int)r[0];
    arrowArray[i].y = (int)r[1];
  }

  Vector c = center.round();
  Vector end = (Vector(length, 0) * rot * Translate(center)).round();
  XDRAWLINE(display, pixmap, compassGC, (int)c[0], (int)c[1], 
	    (int)end[0], (int)end[1]);
  XFILLPOLYGON(display, pixmap, compassGC, arrowArray, 6, 
	       Nonconvex, CoordModeOrigin);

  if (useFont && font) {
    Vector et = Vector((length + textOffset), 0) * rot * Translate(center) *
                Translate(-Tk_TextWidth(font, str, 1)/2., metric.ascent/2.);
    XDrawString(display, pixmap, compassGC, (int)et[0], (int)et[1], str, 1);
  }
}

