//  Copyright (C) 2007-2008  CEA/DEN, EDF R&D, OPEN CASCADE
//
//  Copyright (C) 2003-2007  OPEN CASCADE, EADS/CCR, LIP6, CEA/DEN,
//  CEDRAT, EDF R&D, LEG, PRINCIPIA R&D, BUREAU VERITAS
//
//  This library is free software; you can redistribute it and/or
//  modify it under the terms of the GNU Lesser General Public
//  License as published by the Free Software Foundation; either
//  version 2.1 of the License.
//
//  This library is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//  Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with this library; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
//
//  See http://www.salome-platform.org/ or email : webmaster.salome@opencascade.com
//
//  File   : VVTK_PickingDlg.cxx
//  Author : Oleg Uvarov
//  Module : VISU
//
#include "VVTK_PickingDlg.h"

#include "VISU_GaussPtsAct.h"
#include "VISU_PickingSettings.h"
#include "VISU_GaussPointsPL.hxx"

#include "SUIT_MessageBox.h"
#include "SUIT_ResourceMgr.h"
#include "SUIT_Session.h"

#include "CAM_Module.h"

#include "SVTK_Selector.h"
#include "SVTK_MainWindow.h"
#include "SVTK_RenderWindowInteractor.h"

#include "VTKViewer_Algorithm.h"
#include "SVTK_Functor.h"

#include <vtkActorCollection.h>
#include <vtkCallbackCommand.h>
#include <vtkObjectFactory.h>
#include <vtkRenderer.h>
#include <vtkGenericRenderWindowInteractor.h>
#include <vtkSmartPointer.h>

#include "utilities.h"

#include <QCheckBox>
#include <QColorDialog>
#include <QComboBox>
#include <QGroupBox>
#include <QLabel>
#include <QLayout>
#include <QPushButton>
#include <QVBoxLayout>
#include <QKeyEvent>
#include <QIntValidator>

#include "QtxAction.h"
#include "QtxDoubleSpinBox.h"
#include "QtxIntSpinBox.h"
#include "QtxColorButton.h"

#include "LightApp_Application.h"

namespace
{
  struct SelectorHelper
  {
  public:
    SelectorHelper( SVTK_RenderWindowInteractor* theInteractor ):
      myInteractor( theInteractor )
    {}

    bool
    get()
    {
      bool aResult = false;
      myMapIndex.Clear();
      mySelector = NULL;
      myPipeLine = NULL;
      myActor = NULL;

      if ( !myInteractor )
	return aResult;
      
      mySelector = myInteractor->GetSelector();
      if ( !mySelector )
	return aResult;
      
      const SALOME_ListIO& aListIO = mySelector->StoredIObjects();
      if ( aListIO.Extent() != 1 ) 
	return aResult;
      
      myIO = aListIO.First();
      if ( mySelector->HasIndex( myIO ) )
	mySelector->GetIndex(myIO, myMapIndex);
	
      VTK::ActorCollectionCopy aCopy(myInteractor->getRenderer()->GetActors());
      myActor = SVTK::Find<VISU_GaussPtsAct>(aCopy.GetActors(),
					     SVTK::TIsSameIObject<VISU_GaussPtsAct>( myIO ));
      if ( !myActor )
	return aResult;
      
      myPipeLine = myActor->GetGaussPointsPL();

      return true;
    }

    SVTK_RenderWindowInteractor* myInteractor;
    TColStd_IndexedMapOfInteger myMapIndex;
    Handle(SALOME_InteractiveObject) myIO;
    SVTK_Selector* mySelector;

    VISU_GaussPointsPL* myPipeLine;
    VISU_GaussPtsAct* myActor;
  };


  
  class GaussPtsIDValidator: public QIntValidator
  {
  public:
    GaussPtsIDValidator( SVTK_RenderWindowInteractor* theInteractor,
			 QObject * theParent ):
      QIntValidator( 0, VTK_LARGE_ID, theParent ),
      myHelper(theInteractor)
    {}

    virtual 
    State
    validate ( QString & theInput, int & thePos ) const
    {
      if ( QIntValidator::validate( theInput, thePos ) == QValidator::Invalid)
	return QValidator::Invalid;
      
      if ( !myHelper.get() )
	return QValidator::Invalid;

      return QValidator::Acceptable;
    }

  protected:
    mutable SelectorHelper myHelper;
  };


  class GaussCellIDValidator: public GaussPtsIDValidator
  {
  public:
    GaussCellIDValidator( QLineEdit* theLocalPointLabel,
			  SVTK_RenderWindowInteractor* theInteractor,
			  QObject * theParent ):
      GaussPtsIDValidator( theInteractor, theParent ),
      myLocalPointLineEdit( theLocalPointLabel )
    {}

    virtual 
    State
    validate ( QString & theInput, int & thePos ) const
    {
      if ( GaussPtsIDValidator::validate( theInput, thePos ) == QValidator::Invalid)
	return QValidator::Invalid;

      VISU::TCellID aCellID = theInput.toInt();
      VISU::TLocalPntID aLocalPntID = myLocalPointLineEdit->text().toInt();
      VISU::PGaussPtsIDMapper anIDMapper = myHelper.myPipeLine->GetGaussPtsIDMapper();
      if ( anIDMapper->GetVTKID( VISU::TGaussPointID( aCellID, aLocalPntID ) ) < 0 )
	return QValidator::Intermediate;

      return QValidator::Acceptable;
    }

  private:
    QLineEdit* myLocalPointLineEdit;
  };


  class GaussLocalPointIDValidator: public GaussPtsIDValidator
  {
  public:
    GaussLocalPointIDValidator( QLineEdit* theParentElementLineEdit,
				SVTK_RenderWindowInteractor* theInteractor,
				QObject * theParent ):
      GaussPtsIDValidator( theInteractor, theParent ),
      myParentElementLineEdit( theParentElementLineEdit )
    {}

    virtual 
    State
    validate ( QString & theInput, int & thePos ) const
    {
      if ( GaussPtsIDValidator::validate( theInput, thePos ) == QValidator::Invalid)
	return QValidator::Invalid;

      VISU::TLocalPntID aLocalPntID = theInput.toInt();
      VISU::TCellID aCellID = myParentElementLineEdit->text().toInt();
      VISU::PGaussPtsIDMapper anIDMapper = myHelper.myPipeLine->GetGaussPtsIDMapper();
      if ( anIDMapper->GetVTKID( VISU::TGaussPointID( aCellID, aLocalPntID ) ) < 0 )
	return QValidator::Intermediate;

      return QValidator::Acceptable;
    }

  private:
    QLineEdit* myParentElementLineEdit;
  };
}


VVTK_ValidatedLineEdit::VVTK_ValidatedLineEdit( QWidget* parent ):
  QLineEdit( parent )
{
  connect( this, SIGNAL( textChanged( const QString& ) ), this, SLOT( MarkValidated( const QString& ) ) );
}

void VVTK_ValidatedLineEdit::MarkValidated( const QString& theText )
{
  if ( !validator() )
    return;
  
  int aPos;
  QString aText( theText );
  QPalette pal = palette();
  switch ( validator()->validate( aText, aPos ) ) {
  case QValidator::Invalid:
  case QValidator::Intermediate:
    pal.setColor( foregroundRole(), QColor( 255, 0, 0 ));
    setPalette( pal );
    break;
  case QValidator::Acceptable:
    pal.setColor( foregroundRole(), QColor( 0, 0, 0 ));
    setPalette( pal );
    break;
  }
}

//---------------------------------------------------------------------------------

VVTK_PickingDlg::VVTK_PickingDlg(QtxAction* theAction,
				 SVTK_MainWindow* theParent,
				 const char* theName):
  SVTK_DialogBase(theAction,
		  theParent, 
		  theName),
  myEventCallbackCommand( vtkCallbackCommand::New() ),
  myPickingSettings( VISU_PickingSettings::New() )
{
  myPriority = 0.0;
  myEventCallbackCommand->Delete();
  myEventCallbackCommand->SetClientData(this); 
  myEventCallbackCommand->SetCallback(VVTK_PickingDlg::ProcessEvents);

  setWindowTitle( tr( "PICKING_DLG_TITLE" ) );
  setSizeGripEnabled(TRUE);

  QVBoxLayout* TopLayout = new QVBoxLayout( this );
  TopLayout->setSpacing(6);
  TopLayout->setMargin(11);

  QWidget* aBox = new QWidget( this );
  QVBoxLayout* BoxLayout = new QVBoxLayout( aBox );
  BoxLayout->setMargin(0);
  BoxLayout->setSpacing(6);

  // Cursor
  QGroupBox* CursorGroup = new QGroupBox( tr( "CURSOR_TITLE" ), aBox );
  //CursorGroup->setColumnLayout(0, Qt::Vertical );
  //CursorGroup->layout()->setSpacing( 0 );
  //CursorGroup->layout()->setMargin( 0 );

  QGridLayout* CursorGroupLayout = new QGridLayout (CursorGroup);
  CursorGroupLayout->setAlignment(Qt::AlignTop | Qt::AlignCenter);
  CursorGroupLayout->setSpacing(6);
  CursorGroupLayout->setMargin(11);

  QLabel* CursorSizeLabel = new QLabel( tr( "CURSOR_SIZE" ), CursorGroup );
  myCursorSizeSpinBox = new QtxDoubleSpinBox( 0, 1, 0.1, CursorGroup );
  myCursorSizeSpinBox->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed ) );

  QLabel* PyramidHeightLabel = new QLabel( tr( "PYRAMID_HEIGHT" ), CursorGroup );
  double aHeightMin=1.e-7;
  double aHeightMax=10.;
  double aHeightStep=0.1;
  myPyramidHeightSpinBox = new QtxDoubleSpinBox(aHeightMin, aHeightMax, aHeightStep, CursorGroup );
  myPyramidHeightSpinBox->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed ) );
  
  QLabel* SelectionColorLabel = new QLabel( tr( "SELECTION_COLOR" ), CursorGroup );
  mySelectionColorButton = new QtxColorButton( CursorGroup );
  mySelectionColorButton->setColor( Qt::blue );
  //QPalette aPalette( mySelectionColorButton->palette() );
  //aPalette.setColor( mySelectionColorButton->backgroundRole(), Qt::blue );
  //mySelectionColorButton->setPalette( aPalette );
  //connect( mySelectionColorButton, SIGNAL( clicked() ), this, SLOT( onColorButtonPressed() ) );

  CursorGroupLayout->addWidget( CursorSizeLabel, 0, 0 );
  CursorGroupLayout->addWidget( myCursorSizeSpinBox, 0, 1 );
  CursorGroupLayout->addWidget( PyramidHeightLabel, 1, 0 );
  CursorGroupLayout->addWidget( myPyramidHeightSpinBox, 1, 1 );
  CursorGroupLayout->addWidget( SelectionColorLabel, 2, 0 );
  CursorGroupLayout->addWidget( mySelectionColorButton, 2, 1 );

  BoxLayout->addWidget( CursorGroup );

  // Tolerance
  QGroupBox* ToleranceGroup = new QGroupBox( tr( "TOLERANCE_TITLE" ), aBox );
  //ToleranceGroup->setColumnLayout(0, Qt::Vertical );
  //ToleranceGroup->layout()->setSpacing( 0 );
  //ToleranceGroup->layout()->setMargin( 0 );

  QGridLayout* ToleranceGroupLayout = new QGridLayout (ToleranceGroup);
  ToleranceGroupLayout->setAlignment(Qt::AlignTop | Qt::AlignCenter);
  ToleranceGroupLayout->setSpacing(6);
  ToleranceGroupLayout->setMargin(11);

  QLabel* PointToleranceLabel = new QLabel( tr( "POINT_TOLERANCE" ), ToleranceGroup );
  myPointToleranceSpinBox = new QtxDoubleSpinBox( 0.001, 10.0, 0.01, ToleranceGroup );
  myPointToleranceSpinBox->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed ) );

  ToleranceGroupLayout->addWidget( PointToleranceLabel, 0, 0 );
  ToleranceGroupLayout->addWidget( myPointToleranceSpinBox, 0, 1 );

  BoxLayout->addWidget( ToleranceGroup );

  // Information window
  QGroupBox* InfoWindowGroup = new QGroupBox( tr( "INFO_WINDOW_TITLE" ), aBox );
  //InfoWindowGroup->setColumnLayout(0, Qt::Vertical );
  //InfoWindowGroup->layout()->setSpacing( 0 );
  //InfoWindowGroup->layout()->setMargin( 0 );

  QGridLayout* InfoWindowGroupLayout = new QGridLayout (InfoWindowGroup);
  InfoWindowGroupLayout->setAlignment(Qt::AlignTop | Qt::AlignCenter);
  InfoWindowGroupLayout->setSpacing(6);
  InfoWindowGroupLayout->setMargin(11);

  QLabel* TransparencyLabel = new QLabel( tr( "TRANSPARENCY" ), InfoWindowGroup );
  myTransparencySpinBox = new QtxIntSpinBox( 0, 100, 10, InfoWindowGroup );
  myTransparencySpinBox->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed ) );

  InfoWindowGroupLayout->addWidget( TransparencyLabel, 0, 0 );
  InfoWindowGroupLayout->addWidget( myTransparencySpinBox, 0, 1 );

  QLabel* PositionLabel = new QLabel( tr( "POSITION" ), InfoWindowGroup );
  myPositionComboBox = new QComboBox( InfoWindowGroup );
  myPositionComboBox->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed ) );

  myPositionComboBox->addItem( tr( "BELOW_POINT" ) );
  myPositionComboBox->addItem( tr( "TOP_LEFT_CORNER" ) );

  InfoWindowGroupLayout->addWidget( TransparencyLabel, 0, 0 );
  InfoWindowGroupLayout->addWidget( myTransparencySpinBox, 0, 1 );
  InfoWindowGroupLayout->addWidget( PositionLabel, 1, 0 );
  InfoWindowGroupLayout->addWidget( myPositionComboBox, 1, 1 );

  BoxLayout->addWidget( InfoWindowGroup );

  // Movement of the camera
  QGroupBox* CameraGroup = new QGroupBox( tr( "CAMERA_TITLE" ), aBox );
  //CameraGroup->setColumnLayout(0, Qt::Vertical );
  //CameraGroup->layout()->setSpacing( 0 );
  //CameraGroup->layout()->setMargin( 0 );

  QGridLayout* CameraGroupLayout = new QGridLayout (CameraGroup);
  CameraGroupLayout->setAlignment(Qt::AlignTop | Qt::AlignCenter);
  CameraGroupLayout->setSpacing(6);
  CameraGroupLayout->setMargin(11);

  QLabel* ZoomFactorLabel = new QLabel( tr( "ZOOM_FACTOR" ), CameraGroup );
  myZoomFactorSpinBox = new QtxDoubleSpinBox( 0.1, 10.0, 0.1, CameraGroup );
  myZoomFactorSpinBox->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed ) );

  QLabel* StepNumberLabel = new QLabel( tr( "STEP_NUMBER" ), CameraGroup );
  myStepNumberSpinBox = new QtxIntSpinBox( 1, 100, 1, CameraGroup );
  myStepNumberSpinBox->setSizePolicy( QSizePolicy( QSizePolicy::Expanding, QSizePolicy::Fixed ) );

  CameraGroupLayout->addWidget( ZoomFactorLabel, 0, 0 );
  CameraGroupLayout->addWidget( myZoomFactorSpinBox, 0, 1 );
  CameraGroupLayout->addWidget( StepNumberLabel, 1, 0 );
  CameraGroupLayout->addWidget( myStepNumberSpinBox, 1, 1 );

  BoxLayout->addWidget( CameraGroup );

  // Display parent mesh element
  QGroupBox* PositionGroup = new QGroupBox( tr( "DATA_POSITION" ), aBox );
  //PositionGroup->setColumnLayout(0, Qt::Vertical );
  //PositionGroup->layout()->setSpacing( 0 );
  //PositionGroup->layout()->setMargin( 0 );

  QGridLayout* PositionGroupLayout = new QGridLayout (PositionGroup);
  PositionGroupLayout->setAlignment(Qt::AlignTop | Qt::AlignLeft);
  PositionGroupLayout->setSpacing(6);
  PositionGroupLayout->setMargin(11);

  QLabel* ParentElementLabel = new QLabel( tr( "PARENT_ELEMENT" ), PositionGroup );

  myParentElementLineEdit = new VVTK_ValidatedLineEdit( PositionGroup );

  QLabel* LocalPointLabel = new QLabel( tr( "LOCAL_POINT" ), PositionGroup );

  myLocalPointLineEdit = new VVTK_ValidatedLineEdit( PositionGroup );

  myDisplayParentMeshCheckBox = new QCheckBox( tr( "DISPLAY_PARENT_MESH" ), PositionGroup );

  PositionGroupLayout->addWidget( ParentElementLabel,          0, 0, 1, 3 );
  PositionGroupLayout->addWidget( myParentElementLineEdit,     0, 3 );
  PositionGroupLayout->addWidget( LocalPointLabel,             1, 0, 1, 3 );
  PositionGroupLayout->addWidget( myLocalPointLineEdit,        1, 3 );
  PositionGroupLayout->addWidget( myDisplayParentMeshCheckBox, 2, 0, 1, 4 );

  BoxLayout->addWidget( PositionGroup );

  // Common buttons ===========================================================
  QGroupBox* GroupButtons = new QGroupBox( this );
  //GroupButtons->setColumnLayout(0, Qt::Vertical );
  //GroupButtons->layout()->setSpacing( 0 );
  //GroupButtons->layout()->setMargin( 0 );
  QGridLayout* GroupButtonsLayout = new QGridLayout( GroupButtons );
  GroupButtonsLayout->setAlignment( Qt::AlignTop );
  GroupButtonsLayout->setSpacing( 6 );
  GroupButtonsLayout->setMargin( 11 );

  QPushButton* buttonApply = new QPushButton( tr( "&Apply" ), GroupButtons );
  buttonApply->setAutoDefault( TRUE );
  buttonApply->setDefault( TRUE );
  GroupButtonsLayout->addWidget( buttonApply, 0, 0 );
  GroupButtonsLayout->addItem( new QSpacerItem( 5, 5, QSizePolicy::Expanding, QSizePolicy::Minimum ), 0, 1 );

  QPushButton* buttonClose = new QPushButton( tr( "&Close" ) , GroupButtons );
  buttonClose->setAutoDefault( TRUE );
  GroupButtonsLayout->addWidget( buttonClose, 0, 2 );

  QPushButton* buttonHelp = new QPushButton( tr( "&Help" ) , GroupButtons );
  buttonHelp->setAutoDefault( TRUE );
  GroupButtonsLayout->addWidget( buttonHelp, 0, 3 );

  TopLayout->addWidget( aBox );
  TopLayout->addWidget( GroupButtons );

  connect( buttonApply, SIGNAL( clicked() ), this, SLOT( onClickApply() ) );
  connect( buttonClose, SIGNAL( clicked() ), this, SLOT( onClickClose() ) );
  connect( buttonHelp,  SIGNAL( clicked() ), this, SLOT( onClickHelp() ) );

  connect( myParentElementLineEdit, SIGNAL( textChanged( const QString& ) ), this, SLOT( onSelectionValidate() ) );
  connect( myLocalPointLineEdit,    SIGNAL( textChanged( const QString& ) ), this, SLOT( onSelectionValidate() ) );

}

VVTK_PickingDlg::~VVTK_PickingDlg()
{
}

void VVTK_PickingDlg::AddActor( VISU_GaussPtsAct* theActor )
{
  //theActor->SetPickingSettings( myPickingSettings.GetPointer() );
}

void VVTK_PickingDlg::RemoveActor( VISU_GaussPtsAct* theActor )
{
  //theActor->SetPickingSettings( NULL );
}

void VVTK_PickingDlg::Update()
{
  float aCursorSize = 0.5;
  float aPyramidHeight = 10.0;
  float aPointTolerance = 0.1;
  QColor aColor = Qt::yellow;
  int anInfoWindowTransparency = 50;
  int anInfoWindowPosition = VISU_PickingSettings::BelowPoint;
  float aZoomFactor = 1.5;
  int aStepNumber = 10;
  bool aDisplayParentMesh = false;

  //if( !myPickingSettings->GetInitial() )
  {
    myCursorSizeSpinBox->setValue( myPickingSettings->GetCursorSize() );
    myPyramidHeightSpinBox->setValue( myPickingSettings->GetPyramidHeight() );
    myPointToleranceSpinBox->setValue( myPickingSettings->GetPointTolerance() );
    myTransparencySpinBox->setValue( int(myPickingSettings->GetInfoWindowTransparency() * 100.0) );
    myPositionComboBox->setCurrentIndex( myPickingSettings->GetInfoWindowPosition() );
    myZoomFactorSpinBox->setValue( myPickingSettings->GetZoomFactor() );
    myStepNumberSpinBox->setValue( myPickingSettings->GetStepNumber() );
    myDisplayParentMeshCheckBox->setChecked( myPickingSettings->GetDisplayParentMesh() );

    vtkFloatingPointType* aColor = myPickingSettings->GetColor();
   
    //QPalette aPalette( mySelectionColorButton->palette() );
    /*aPalette.setColor( mySelectionColorButton->backgroundRole(),
		       QColor( ( int )( aColor[0] * 255.0 ),
			       ( int )( aColor[1] * 255.0 ),
			       ( int )( aColor[2] * 255.0 ) ) );
                               mySelectionColorButton->setPalette( aPalette );*/
    mySelectionColorButton->setColor(
                 QColor( ( int )( aColor[0] * 255.0 ),
		 ( int )( aColor[1] * 255.0 ),
                 ( int )( aColor[2] * 255.0 ) ) );
    
    return;
  }

  SUIT_ResourceMgr* aResourceMgr = SUIT_Session::session()->resourceMgr();

  aCursorSize = aResourceMgr->doubleValue( "VISU", "picking_cursor_size", aCursorSize );
  myCursorSizeSpinBox->setValue( aCursorSize );

  aPyramidHeight = aResourceMgr->doubleValue( "VISU", "picking_pyramid_height", aPyramidHeight );
  myPyramidHeightSpinBox->setValue( aPyramidHeight );

  aPointTolerance = aResourceMgr->doubleValue( "VISU", "picking_point_tolerance", aPointTolerance );
  myPointToleranceSpinBox->setValue( aPointTolerance );

  aColor = aResourceMgr->colorValue( "VISU", "picking_selection_color", aColor );
  //QPalette aPalette( mySelectionColorButton->palette() );
  //aPalette.setColor( mySelectionColorButton->backgroundRole(), aColor );
  mySelectionColorButton->setColor( aColor );

  anInfoWindowTransparency = aResourceMgr->integerValue( "VISU", "picking_transparency", anInfoWindowTransparency );
  myTransparencySpinBox->setValue( anInfoWindowTransparency );

  anInfoWindowPosition = aResourceMgr->integerValue( "VISU", "picking_position", anInfoWindowPosition );
  myPositionComboBox->setCurrentIndex( anInfoWindowPosition );

  aZoomFactor = aResourceMgr->doubleValue( "VISU", "picking_zoom_factor", aZoomFactor );
  myZoomFactorSpinBox->setValue( aZoomFactor );

  aStepNumber = aResourceMgr->integerValue( "VISU", "picking_step_number", aStepNumber );
  myStepNumberSpinBox->setValue( aStepNumber );

  aDisplayParentMesh = aResourceMgr->booleanValue( "VISU", "picking_display_parent_mesh", aDisplayParentMesh );
  myDisplayParentMeshCheckBox->setChecked( aDisplayParentMesh );

  onClickApply();
}

void VVTK_PickingDlg::onSelectionValidate() 
{
  myParentElementLineEdit->MarkValidated( myParentElementLineEdit->text() );
  myLocalPointLineEdit->MarkValidated( myLocalPointLineEdit->text() );
}

void VVTK_PickingDlg::onSelectionEvent() 
{
  SelectorHelper aHelper( myInteractor );
  if ( !aHelper.get() )
    return;

  const TColStd_IndexedMapOfInteger& aMapIndex = aHelper.myMapIndex;
  if ( aMapIndex.Extent() != 1 )
    return;

  int anObjId = aHelper.myMapIndex(1);

  VISU::TGaussPointID aGaussPointID = aHelper.myPipeLine->GetObjID( anObjId );

  VISU::TCellID aCellID = aGaussPointID.first;
  myParentElementLineEdit->setText( QString::number( aCellID ) );

  VISU::TLocalPntID aLocalPntID = aGaussPointID.second;
  myLocalPointLineEdit->setText( QString::number( aLocalPntID ) );
}

VISU_PickingSettings* 
VVTK_PickingDlg
::GetPickingSettings()
{
  return myPickingSettings.GetPointer();
}

void VVTK_PickingDlg::SetInteractor( SVTK_RenderWindowInteractor* theInteractor )
{
  myInteractor = theInteractor;

  theInteractor->GetDevice()->AddObserver(vtkCommand::KeyPressEvent, 
					  myEventCallbackCommand.GetPointer(), 
					  myPriority);
  theInteractor->GetDevice()->AddObserver(vtkCommand::EndPickEvent, 
					  myEventCallbackCommand.GetPointer(), 
					  myPriority);
  {
    QValidator* aValidator = new GaussCellIDValidator( myLocalPointLineEdit, theInteractor, myParentElementLineEdit );
    myParentElementLineEdit->setValidator( aValidator );
  }
  {
    QValidator* aValidator = new GaussLocalPointIDValidator( myParentElementLineEdit, theInteractor, myLocalPointLineEdit );
    myLocalPointLineEdit->setValidator( aValidator );
  }
}

void VVTK_PickingDlg::ProcessEvents(vtkObject* vtkNotUsed(theObject), 
				    unsigned long theEvent,
				    void* theClientData, 
				    void* vtkNotUsed(theCallData))
{
  VVTK_PickingDlg* self = reinterpret_cast<VVTK_PickingDlg*>(theClientData);

  switch(theEvent){
  case vtkCommand::KeyPressEvent:
    self->KeyPressed();
    break;
  case vtkCommand::EndPickEvent:
    self->onSelectionEvent();
    break;
  }
}

void VVTK_PickingDlg::KeyPressed()
{
  if( myInteractor->GetDevice()->GetKeyCode() == 'P' )
  {
    bool aDisplayParentMesh = !myPickingSettings->GetDisplayParentMesh();
    myPickingSettings->SetDisplayParentMesh( aDisplayParentMesh );

    myDisplayParentMeshCheckBox->setChecked( aDisplayParentMesh );

    myPickingSettings->InvokeEvent(VISU::UpdatePickingSettingsEvent,NULL);
  }
}

void VVTK_PickingDlg::onClickApply()
{
  //myPickingSettings->SetInitial( false );

  myPickingSettings->SetCursorSize( myCursorSizeSpinBox->value() );
  myPickingSettings->SetPyramidHeight( myPyramidHeightSpinBox->value() );
  myPickingSettings->SetPointTolerance( myPointToleranceSpinBox->value() );
  myPickingSettings->SetInfoWindowTransparency( myTransparencySpinBox->value() / 100.0 );
  myPickingSettings->SetInfoWindowPosition( myPositionComboBox->currentIndex() );
  myPickingSettings->SetZoomFactor( myZoomFactorSpinBox->value() );
  myPickingSettings->SetStepNumber( myStepNumberSpinBox->value() );
  myPickingSettings->SetDisplayParentMesh( myDisplayParentMeshCheckBox->isChecked() );

  //QPalette aPalette( mySelectionColorButton->palette() );
  QColor aButtonColor = mySelectionColorButton->color();//aPalette.color(mySelectionColorButton->backgroundRole() );
  vtkFloatingPointType aColor[3];
  aColor[0] = aButtonColor.red() / 255.0;
  aColor[1] = aButtonColor.green() / 255.0;
  aColor[2] = aButtonColor.blue() / 255.0;
  myPickingSettings->SetColor( aColor );

  myPickingSettings->InvokeEvent( VISU::UpdatePickingSettingsEvent,NULL );

  SelectorHelper aHelper( myInteractor );
  if ( !aHelper.get() )
    return;

  VISU::TCellID aCellID = myParentElementLineEdit->text().toInt();
  VISU::TLocalPntID aLocalPntID = myLocalPointLineEdit->text().toInt();
  VISU::PGaussPtsIDMapper anIDMapper = aHelper.myPipeLine->GetGaussPtsIDMapper();
  vtkIdType anObjId = anIDMapper->GetVTKID( VISU::TGaussPointID( aCellID, aLocalPntID ) );
  if ( anObjId < 0 )
    return;

  aHelper.myMapIndex.Clear();
  aHelper.myMapIndex.Add( anObjId );
  aHelper.mySelector->AddOrRemoveIndex( aHelper.myIO, aHelper.myMapIndex, false );
  aHelper.myActor->Highlight( aHelper.myIO );
  myInteractor->GetDevice()->CreateTimer( VTKI_TIMER_FIRST );
}

void VVTK_PickingDlg::onClickClose()
{
  reject();
}

void VVTK_PickingDlg::onClickHelp()
{
  QString aHelpFileName = "picking.htm";
  LightApp_Application* app = (LightApp_Application*)(SUIT_Session::session()->activeApplication());
  if (app)
    app->onHelpContextModule(app->activeModule() ? app->moduleName(app->activeModule()->moduleName()) : QString(""), aHelpFileName);
  else {
		QString platform;
#ifdef WIN32
		platform = "winapplication";
#else
		platform = "application";
#endif
	SUIT_MessageBox::warning( 0,
				  QObject::tr("WRN_WARNING"),
				  QObject::tr("EXTERNAL_BROWSER_CANNOT_SHOW_PAGE").
				    arg(app->resourceMgr()->stringValue("ExternalBrowser", platform)).arg(aHelpFileName),
				  QObject::tr("BUT_OK"));
  }
}

/*void VVTK_PickingDlg::onColorButtonPressed()
{
  QPalette aPalette( mySelectionColorButton->palette() );
  QColor aColor = QColorDialog::
    getColor( aPalette.color(mySelectionColorButton->backgroundRole() ), this );

  if( aColor.isValid() )
  {
      aPalette.setColor( mySelectionColorButton->backgroundRole(),
		     aColor );
      mySelectionColorButton->setPalette( aPalette );
  }
}*/

void VVTK_PickingDlg::keyPressEvent( QKeyEvent* e )
{
  QDialog::keyPressEvent( e );
  if ( e->isAccepted() )
    return;

  if ( e->key() == Qt::Key_F1 )
    {
      e->accept();
      onClickHelp();
    }
}
