/*
 * Copyright (C) 1993-2005 Robert & Jeremy Lain
 * See AUTHORS file for a full list of contributors.
 * 
 * $Id: sailwriter-xml.h,v 1.5 2005/09/03 14:28:58 jeremy_laine Exp $
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef SAILWRITER_XML_H
#define SAILWRITER_XML_H

#include "filewriter.h"

/** This class allows you to write sailcut objects to
 *  an XML file.
 *
 *  @see CSailXmlWriter
 *  @see CSailDefXmlWriter
 */
template <class objtype>
class XmlWriterTempl : public CFileWriter
{
protected :
    /** the object to write */
    objtype _obj;

public:
    /** The constructor.
     *
     * @param obj the object to write
     * @param name the object name
     */
    XmlWriterTempl(const objtype &obj, const QString &name,
                   const QString ext = ".xml", const QString desc = "XML files")
            : CFileWriter(name, ext, desc), _obj(obj)
    {}
    ;

    /** The destructor. */
    virtual ~XmlWriterTempl()
    {}
    ;

    /** Write object to XML format.
     *
     * @param filename the file to write to
     */
    virtual void write(const QString &filename)
    {
        CSailDoc doc;
        doc.put(doc.top, _obj, _name);
        doc.toFile(filename);
    };
};


/** A class used to output a CSail to an XML file. */
class CSailXmlWriter : public XmlWriterTempl<CSail>
{
public:
    CSailXmlWriter(const CSail &obj, const QString &name)
            : XmlWriterTempl<CSail>(obj, name, ".sail3d", "3D sails")
    {}
    ;
};

/** A class used to output a CSailDef to an XML file. */
class CSailDefXmlWriter : public XmlWriterTempl<CSailDef>
{
public:
    CSailDefXmlWriter(const CSailDef &obj, const QString &name)
            : XmlWriterTempl<CSailDef>(obj, name, ".saildef", "Sail definitions")
    {}
    ;
};

/** A class used to output a CRigDef to an XML file. */
class CRigDefXmlWriter : public XmlWriterTempl<CRigDef>
{
public:
    CRigDefXmlWriter(const CRigDef &obj, const QString &name)
            : XmlWriterTempl<CRigDef>(obj, name, ".sailrig", "Rig definitions")
    {}
    ;
};

/** A class used to output a CPrefs to an XML file. */
typedef XmlWriterTempl<CPrefs> CPrefsXmlWriter;


#endif
