/*
 * Copyright (C) 1993-2004 Robert & Jeremy Lain
 * See AUTHORS file for a full list of contributors.
 *
 * $Id: widgetprofilevert.cpp,v 1.3 2005/05/09 22:13:25 jeremy_laine Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "widgetprofilevert.h"
#include "sailmould.h"

#include <qgroupbox.h>
#include <qlayout.h>
#include <qpainter.h>
#include <qslider.h>

/**************************************************************************

                         CLabelProfileVert class

**************************************************************************/

/** The constructor.
 *
 * @param parent the parent dialog
 * @param ptr pointer to the CSailMould
 */
CLabelProfileVert::CLabelProfileVert( QWidget *parent, CSailMould *ptr)
        : QLabel(parent), CDispArea(0.9), mould(ptr)
{
    setMinimumSize( QSize( 110, 400 ) );
    
    QPalette pal;
    QColorGroup cg;
    cg.setColor( QColorGroup::Background, white );
    pal.setActive( cg );
    pal.setInactive( cg );
    setPalette( pal );
}


/** Clear the drawing area.
 */
void CLabelProfileVert::clear()
{
    QPainter painter( this );
    QRect rect = painter.viewport();
    painter.eraseRect(rect);
}


/** Display the vertical repartition in the drawing area.
 */
void CLabelProfileVert::paintEvent( QPaintEvent *)
{
    QPainter painter( this );
    QPoint prev, next;
    unsigned int nbpoints = 32;
    real z, y; // z is horizontal and y is vertical upward
   
    QRect view = painter.viewport();
    real scale =  view.height() / lRect.height();
    
    // do a translation to have from z=0 to z=scale centered
    painter.translate( (lRect.width() / 2 - center.x()) * scale,  view.height() + (center.y() - lRect.height()/2) * scale );

    // flip coordinate system to have the y axis pointing up
    painter.scale(1,-1);

    prev = QPoint( int(scale * mould->interpol(0).getDepth()), 0 );
    for( unsigned i = 1; i < nbpoints; i++)
    {
        y = real(i) / (nbpoints-1);
        z = mould->interpol(y).getDepth();
        
        next = QPoint( int(scale * z) , int(scale * y));
        painter.drawLine(prev, next);
        prev = next;
    }

}


/** The draw area has been resized.
 */
void CLabelProfileVert::resizeEvent( QResizeEvent * )
{
    QPainter painter(this);
    QRect vRect = painter.viewport();
    CRect3d viewRect;
    viewRect.max = CPoint3d(vRect.width(), vRect.height());
    
    CRect3d objRect; 
    objRect.max = CPoint3d(0.2, 1);
    center = objRect.center();
    
    lRect = calcLRect(viewRect, objRect);
}


/**************************************************************************

                         CWidgetProfileVert class

**************************************************************************/

/** The constructor.
 *
 * @param parent the parent dialog
 * @param sailmould pointer to the CSailMould
 * @param caption caption for the profile view (e.g. top,middle..)
 */
CWidgetProfileVert::CWidgetProfileVert(QWidget *parent, CSailMould *mouldptr, QString caption)
    : QWidget(parent), sailmould(mouldptr)
{
    QGridLayout *layout = new QGridLayout( this, 1, 1, 0, 0 );
    
    grpVertical = new QGroupBox( this );
    grpVertical->setTitle(caption);

    //grpVertical->setAlignment( int( QGroupBox::AlignHCenter ) );
    grpVertical->setColumnLayout(0, Qt::Vertical );

////////////    
    QHBoxLayout *drawVerticalHBox = new QHBoxLayout( 0, 0, 6 );

    // add depth vertical repartition draw area
    lblDrawVert = new CLabelProfileVert( grpVertical, sailmould );
    drawVerticalHBox->addWidget( lblDrawVert );
    
    // add vertical slider for mid depth position
    sliderVertical = new QSlider( grpVertical );
    sliderVertical->setMinValue( 30 );
    sliderVertical->setMaxValue( 70 );
    sliderVertical->setOrientation( QSlider::Vertical );
    sliderVertical->setValue( 100 - sailmould->vertpos );
    
    drawVerticalHBox->addWidget( sliderVertical );
    
    grpVertical->layout()->addItem( drawVerticalHBox );
/////////////
    QHBoxLayout *lblVerticalHBox = new QHBoxLayout( 0, 0, 6 );
    
    //QSpacerItem* spacer_1 = new QSpacerItem( 20, 20, QSizePolicy::MinimumExpanding, QSizePolicy::Minimum );
    //lblVerticalHBox->addItem( spacer_1 );

    lblVertical = new QLabel( grpVertical );
    lblVertical->setAlignment( int( QLabel::AlignCenter ) );
    lblVerticalHBox->addWidget( lblVertical );

    //QSpacerItem* spacer_2 = new QSpacerItem( 20, 20, QSizePolicy::MinimumExpanding, QSizePolicy::Minimum );
    //lblVerticalHBox->addItem( spacer_2 );
        
    grpVertical->layout()->addItem( lblVerticalHBox );
    
/////////////

    layout->addWidget( grpVertical, 0, 0 );

    slotSliderVert();

    connect( sliderVertical, SIGNAL( valueChanged(int) ), this, SLOT( slotSliderVert() ) );

}

void CWidgetProfileVert::refresh() 
{
    lblDrawVert->clear();
    lblDrawVert->paintEvent( NULL );
}

/** Trigered when the user moves the vertical repartition slider. */
void CWidgetProfileVert::slotSliderVert()
{
    // store new value of vertical position of mid profile and update label
    sailmould->vertpos = 100 - sliderVertical->value();
    lblVertical->setText( QString::number(sailmould->vertpos) );

    // update vertical profile view
    lblDrawVert->clear();
    lblDrawVert->paintEvent( NULL );
}
