/*
 * Copyright (C) 1993-2004 Robert & Jeremy Lain
 * See AUTHORS file for a full list of contributors.
 * 
 * $Id: sailcutqt.cpp,v 1.10 2005/05/15 23:02:56 jeremy_laine Exp $
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "sailcutqt.h"
#include <qfile.h>
#include <qdir.h>
#include <iostream>
#include "sailreader-xml.h"
#include "sailwriter-xml.h"

using namespace std;

/**
 * Creates an new Sailcut application.
 *
 * @param argc
 * @param argv
 */
CSailApp::CSailApp(int argc, char* argv[]) :
        QApplication(argc, argv)
{
    // create and install translators
    transApp = new QTranslator( this );
    installTranslator( transApp );
    transQt = new QTranslator( this );
    installTranslator( transQt );

    // the file containing the user's preferences
    prefsfile = QDir::home().filePath(".sailcutrc");
}


/**
 * The destructor.
 */
CSailApp::~CSailApp()
{
    // remove and destroy translators
    removeTranslator(transApp);
    removeTranslator(transQt);
    delete transApp;
    delete transQt;
}


/**
 * Tries to locate the Sailcut Handbook.
 */
QString CSailApp::findHandbook(const QString locale)
{
    QString handbook;
    
    // look for handbook in requested language
    QFile::exists(handbook = "html\\" + locale + "\\index.html")
#ifdef DOCDIR
    || QFile::exists(handbook = DOCDIR "/html/" + locale + "/index.html")
#endif
    || (handbook = "");

    if (!handbook.isEmpty())
    {
        cout << "Found Handbook for locale '" << locale << "' : " << handbook << endl;
        return handbook;
    }
          
    // try falling back to English      
    if ((handbook.isEmpty()) && (locale != "en"))
    {    
      QFile::exists(handbook = "html\\en\\index.html")
#ifdef DOCDIR
      || QFile::exists(handbook = DOCDIR "/html/en/index.html")
#endif
      || (handbook = "");
      
      if (!handbook.isEmpty())
      {
          cout << "Falling back to Handbook for locale 'en': " << handbook << endl;
      }      
    }
        
    // no handbook found
    if (handbook.isEmpty())
    {
        cout << "No handbook found for locale '" << locale << "'!" << endl;
    }
    return handbook;
}


/**
 * Loads translation strings for a given language.
 */
void CSailApp::loadTranslation(const QString locale)
{
    // translation file for Qt
    transQt->load( QString( "qt_" ) + locale, "." );

    // translation file for application strings
    if (locale.startsWith("en"))
    {
        transApp->clear();
    }
    else
    {
        QString qm = QString("sailcut.") + locale;
        transApp->load(qm, ".")
        || transApp->load(qm, "ts")
#ifdef DATADIR
        || transApp->load(qm, DATADIR)
#endif
        ;
    }
}


/**
 * Try to read the user's preferences
 */
void CSailApp::readPrefs()
{
    try
    {
        if (QFile(prefsfile).exists())
            prefs = CPrefsXmlReader("prefs").read(prefsfile);
        loadTranslation(prefs.language.data());
    }
    catch (CException e)
    {
        cout << "Could not read preferences" << endl;
    }
}


/**
 * Try to write the user's preferences.
 */
void CSailApp::writePrefs()
{
    try
    {
        CPrefsXmlWriter(prefs,"prefs").write(prefsfile);
    }
    catch (CException e)
    {
        cout << "Could not write preferences" << endl;
    }
}
