/* ====================================================================
 * ===  Copyright (C) 1998-2007 Yutaka Sagiya. All rights reserved. ===
 * ====================================================================
 * 
 *    Project              : SagCAD
 *    Description          : CAD/CAM
 *    Source               : Select.c
 * 
 *    ----------------------------------
 * 
 *    License              : GNU General Public License (GPL)
 *    Copyright            : (C) 1998-2007 by Yutaka Sagiya
 *    email                : kappa@a6s.highway.ne.jp
 *                         : yutaka@sagiya.com
 *    Begin                : 2002/09/08
 *    Last                 : 2007/11/08
 * ====================================================================
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <gtk/gtk.h>

#include <stdio.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include "intl.h"

#include "types.h"
#include "MemoryLeak.h"
#include "List_cad.h"
#include "List_Dimension.h"
#include "List_PolyLine.h"
#include "List_insert.h"
#include "List_Ellipse.h"
#include "List_Vertex.h"
#include "List_Block.h"
#include "List_Undo.h"
#include "List_Select.h"
#include "global.h"
#include "etc.h"
#include "sagcad.h"
#include "culcfunc.h"
#include "Ellipse.h"
#include "Draw.h"
#include "spline.h"
#include "B_spline.h"
#include "MsgBox.h"
#include "Dimension.h"
#include "Mouse.h"
#include "InputBox.h"
#include "Line_dlg.h"
#include "OutsideProcess.h"

#define _SELECT_
#include "Select.h"



/* -------------------------------------------------------
 * セレクトモード
 *	
 * X, Y 		 サーチ座標 (X , Y)
 *	
 * SelectFrag	 
 *	  図形要素	 1 : 点
 *				 2 : 線
 *				 4 : 円弧
 *				 8 : 円
 *				15 : 全図形
 * 
 *	  寸法図形	16 : 始点・終点
 *				32 : 四半点
 *				64 : 中心点
 *			   128 : 任意点
 *			   256 : 寸法図形
 *			   512 : ポリライン
 *			  1024 : ブロック
 *			  2048 : 楕円
 * 
 * Type 		 0 : 重複選択不可
 *				 1 : 重複選択可
 * 
 */
int MouseSelect(GtkWidget *widget, double X, double Y, int SelectFrag, int Type)
{
	SELECT DumySelect, RetSelect;
	int dumy, i, No;
	CAD *ReP;			//1個前の
	char str1[256], str2[256], str[256];


	Already = 0;

	/* マウスクリック点(任意点) */
	if ( (SelectFrag & 128) == 128 ) {
		/* グリッドモードなら */
		if ((sagcad_system.GridMode & 2) == 2) {
			X = sagcad_system.Grid * (int)(X / sagcad_system.Grid);
			Y = sagcad_system.Grid * (int)(Y / sagcad_system.Grid);
		}
		
		/* ------------------------------------------
		 * MouseUp() にあんたは処理しなくていいよと
		 * Select[0].Data に０を入れて知らせる。
		 */
		mouse_select_frag.index = 1;
		mouse_select_frag.diagram_point.Nuru_point = NULL;
		mouse_select_frag.x = sg(X, 6);
		mouse_select_frag.y = sg(Y, 6);
		/* Select Data をリストの最後に追加 */
		select_list_add_last(&mouse_select_frag);
		(*ReturnFunc)(widget, 2, 0, 0);
		return 1;
	}



	/* -----------------------------------------------------
	 * ポリラインデータ 
	 * SelectFrag = 512   
	 * index : 8
	 * 
	 */
	if ( (SelectFrag & 512) == 512 ) {
		mouse_select_frag.diagram_point.polyline_point = PolyLine_search(X, Y);
		/* 見つからない */
		if (mouse_select_frag.diagram_point.polyline_point == NULL) {
			/* ---------------------------------------
			 * MouseUp() にあんたは処理しなくていいよと
			 * Select[0].Data に０を入れて知らせる。
			 */
			mouse_select_frag.index = 0;
			mouse_select_frag.diagram_point.Nuru_point = NULL;
			mouse_select_frag.x = X;
			mouse_select_frag.y = Y;
			//return 0;
		}

		/* 見つかった */
		else {
			mouse_select_frag.index = 8;
			mouse_select_frag.x = X;
			mouse_select_frag.y = Y;


			/* ---------------------------------------
			 * WAIT 秒 図形を消し,BrinkEnd 関数で再表示する。
			 * 一定條ﾔごとに起動したい処理があるときにg用する関数
			 * 引数1 : 何秒毎に関数を呼び出すか(1秒=1000)
			 * 引数2 : 呼び出す関数
			 * 引数3 : 関数に渡す引数
			 */
			PolyLineDraw(drawing_area, mouse_select_frag.diagram_point.polyline_point, SCD_HIDE);
			UpDate();
			gtk_timeout_add(sagcad_system.Wait, (GtkFunction)BrinkEnd, (gpointer)&mouse_select_frag);
			return 1;
		}
	}



	/* -----------------------------------------------------
	 * 楕円データ 
	 * SelectFrag = 2048
	 * index = 16
	 */
	if ( (SelectFrag & 2048) == 2048 ) {
		mouse_select_frag.diagram_point.ellipse_point = ellipse_search(X, Y);
		/* 見つからない */
		if (mouse_select_frag.diagram_point.ellipse_point == NULL) {
			/* ---------------------------------------
			 * MouseUp() にあんたは処理しなくていいよと
			 * Select[0].Data に０を入れて知らせる。
			 */
			mouse_select_frag.index = 0;
			mouse_select_frag.diagram_point.Nuru_point = NULL;
			mouse_select_frag.x = X;
			mouse_select_frag.y = Y;
			//return 0;
		}

		/* 見つかった */
		else {
			mouse_select_frag.index = 16;
			mouse_select_frag.x = X;
			mouse_select_frag.y = Y;


			/* ---------------------------------------
			 * WAIT 秒 図形を消し,BrinkEnd 関数で再表示する。
			 * 一定條ﾔごとに起動したい処理があるときにg用する関数
			 * 引数1 : 何秒毎に関数を呼び出すか(1秒=1000)
			 * 引数2 : 呼び出す関数
			 * 引数3 : 関数に渡す引数
			 */
			EllipseDraw(drawing_area, mouse_select_frag.diagram_point.ellipse_point, SCD_HIDE);
			UpDate();
			gtk_timeout_add(sagcad_system.Wait, BrinkEnd, (gpointer)&mouse_select_frag);
			return 1;
		}
	}



	/* 寸法データ */
	if( (SelectFrag & 256) == 256 ) {
		mouse_select_frag.diagram_point.dimension_point = SnpLook(X, Y);
		/* 見つからない */
		if (mouse_select_frag.diagram_point.dimension_point == NULL) {
			/* ---------------------------------------
			 * MouseUp() にあんたは処理しなくていいよと
			 * Select[0].Data に０を入れて知らせる。
			 */
			mouse_select_frag.index = 0;
			mouse_select_frag.diagram_point.Nuru_point = NULL;
			mouse_select_frag.x = X;
			mouse_select_frag.y = Y;
			//return 0;
		}

		/* 見つかった */
		else {
			mouse_select_frag.index = 3;
			mouse_select_frag.x = X;
			mouse_select_frag.y = Y;


			/* ---------------------------------------
			 * WAIT 秒 図形を消し,BrinkEnd 関数で再表示する。
			 * 一定條ﾔごとに起動したい処理があるときにg用する関数
			 * 引数1 : 何秒毎に関数を呼び出すか(1秒=1000)
			 * 引数2 : 呼び出す関数
			 * 引数3 : 関数に渡す引数
			 */
			DimensionDraw(drawing_area, mouse_select_frag.diagram_point.dimension_point, SCD_HIDE);
			UpDate();
			gtk_timeout_add(sagcad_system.Wait, BrinkEnd, (gpointer)&mouse_select_frag);
			return 1;
		}
	}



	/* 寸法モードのとき */
	if(mode > 500 && mode < 600) {
		if((SelectFrag & 16) == 16 || (SelectFrag & 32) == 32
			 || (SelectFrag & 64) == 64 || (SelectFrag & 1) == 1)
		{
			/* 図形の端点サーチ */
			DumySelect.x = X;
			DumySelect.y = Y;
			dumy = TanTenLook(&DumySelect);
			if(dumy != 0) {
				/*
				 * MouseUp() にあんたは処理しなくていいよと
				 * mouse_select_frag.index に０を入れて知らせる。
				 */
				mouse_select_frag.index = 0;
				mouse_select_frag.diagram_point.Nuru_point = NULL;
				mouse_select_frag.x = 0;
				mouse_select_frag.y = 0;

				/* Select Data をリストの最後に追加 */
				select_list_add_last(&DumySelect);
				ReturnFunc(widget, 1, 0, 0);
				return 1;
			}
			return 0;
		}
	}




	/* -----------------------------------------------------
	 * ブリンク セレクト 処理 
	 * 
	 * -----------------------------------------------------
	 */
	ReP = NULL;
	/* -------------------------------------------
	 * ブリンクより早くマウスのボタンを放したとき。
	 * セレクトが完了していたら Already = 1
	 *
	 * ブリンクより早くマウスのボタンを放したとき。
	 * セレクトが見つからなかった。または完了していないとき Already = 0
	 */
	Already = 0;

	No = 1;
	i = 0;
	while (No != 0) {
		/* とりあえずデータを探す */
		dumy = MouseSelectL(X, Y, SelectFrag, &RetSelect);
		/* ０が来たらそこには無い。*/
		if ( dumy == 0) {
			/* -----------------------------------
			 * ボタンを離したとき(MouseUp)にデータがない
			 * またはすべてセレクトされていることを
			 * Select[0].Data に０を入れて知らせる。
			 */
			mouse_select_frag.index = 0;
			mouse_select_frag.diagram_point.Nuru_point = NULL;
			mouse_select_frag.x = 0;
			mouse_select_frag.y = 0;
			/* ステータスナンバー表示 */
			sprintf(str, _(" There is not a diagram there. or already chosen.") );
			StatusMsg(str);
			//beep();
			return 0;
		}


		/* ---------------------------------------
		 * 選択されているものは選ばない。(選択セット)
		 * (Type == 0:重複選択不可) 
		 */
		if (Type == 0) {
			/* 図形がセレクトされているかを確認 
			 * 0:まだ	1:選択済み
			 */
			No = SelectCheck(RetSelect.diagram_point.cad_point);
			if (i == 0) {
				ReP = RetSelect.diagram_point.cad_point;
				i++;
			}
			else {
				if (ReP == RetSelect.diagram_point.cad_point) {
					/* ---------------------------
					 * ボタンを離したとき(MouseUp)にデータがない
					 * またはすべてセレクトされていることを 		
					 * mouse_select_frag.index に０を入れて知らせる。
					 */
					mouse_select_frag.index = 0;
					mouse_select_frag.diagram_point.Nuru_point = NULL;
					mouse_select_frag.x = 0;
					mouse_select_frag.y = 0;
					/* データナンバー表示 */
					sprintf(str, _("It is already chosen.") );
					StatusMsg(str);
					//Beep;
					return 0;
				}
			}
		}
		/* ---------------------------------------
		 * 選択されているものを選ぶ。(選択済セット)
		 */
		else if (Type == 2) {
			/* 図形がセレクトされているかを確認 
			 * 0:まだ	1:選択済み
			 */
			if (SelectCheck(RetSelect.diagram_point.cad_point) == 1) No = 0;
			if (i == 0) {
				ReP = RetSelect.diagram_point.cad_point;
				i++;
			}
			else {
				if (ReP == RetSelect.diagram_point.cad_point) {
					/* ---------------------------
					 * ボタンを離したとき(MouseUp)にデータがない
					 * またはすべてセレクトされていることを 		
					 * mouse_select_frag.index に０を入れて知らせる。
					 */
					mouse_select_frag.index = 0;
					mouse_select_frag.diagram_point.Nuru_point = NULL;
					mouse_select_frag.x = 0;
					mouse_select_frag.y = 0;
					/* データナンバー表示 */
					sprintf(str, _(" It isn't yet chosen.") );
					StatusMsg(str);
					//Beep;
					return 0;
				}
			}
		}
		else {
			/* データがあれば No = 0 */
			No = 0;
		}
	}


	/* データが見つかったときの処理 */
	if (RetSelect.diagram_point.cad_point != NULL && No == 0) {
		/* ---------------------------------------
		 * マウスを押した今。
		 * MdX & MdY にその座標を入力。
		 *	
		 * マウスを放したときの座標と
		 * MdX & MdY とのずれで、セレクトするかを決める。
		 */
		mouse_select_frag.index = 2;
		mouse_select_frag.diagram_point.cad_point = RetSelect.diagram_point.cad_point;
		mouse_select_frag.x = sg(X, 6);
		mouse_select_frag.y = sg(Y, 6);


		/* ---------------------------------------
		 * データナンバー表示
		 * 左ボタンを押している間だけ表示
		 */
		DataInfo(&mouse_select_frag, str2);				/* 図形データ情報の謔闕桙ﾝ */
		sprintf(str1,"Search %s", str2);				/* サーチ用 */
		StatusMsg(str1); 								/* ステータスバーに表示 */


		/* ---------------------------------------
		 * WAIT 秒 図形を消し,BrinkEnd 関数で再表示する。
		 * 一定時間ごとに起動したい処理があるときに使用する関数
		 * 引数1 : 何秒毎に関数を呼び出すか(1秒=1000)
		 * 引数2 : 呼び出す関数
		 * 引数3 : 関数に渡す引数
		 */
		SelectCadDraw (widget, mouse_select_frag.diagram_point.cad_point, SCD_HIDE);
		gtk_timeout_add(sagcad_system.Wait, BrinkEnd, (gpointer)&mouse_select_frag);
		return 1;
	}


	/* データが見つからなかったきの処理 */
	else {
		//Beep;
		return 0;
	}
	return 0;
}





/* -------------------------------------------------------
 * MouseSelect 関数のブリンクによって消された図形を再表示
 */
gint BrinkEnd(gpointer data)
{
//char str[256];
//g_print("Mouse.c : BrinkEnd() : Already = %d  index = %d\n", Already, ((SELECT *)data)->index);

	/* -----------------------------------------------------
	 * ブリンクが終わったとき。マウスのボタンが放されていない。
	 * またはセレクト図形じゃない。
	 * 消したあとの再表示は、オリジナルカラー
	 */
	if (Already == 0 && data != NULL) {
		SelectDraw (drawing_area, (SELECT *)data, SCD_ORG, UPDATE_ON);
	}

	/* -----------------------------------------------------
	 * ブリンクが終わったとき。マウスのボタンが放されている。
	 * セレクトが完了している。
	 * 消したあとの再表示は、セレクトカラー
	 * とりむの後は再表示して欲しくないので Select[0].Data に 0 を入れて置くので
	 * 再表示しない
	 */
	else if (Already == 1 && data != NULL && ((SELECT *)data)->index != 0) {
		SelectDraw (drawing_area, (SELECT *)data, SCD_SELECT, UPDATE_ON);
	}
	return FALSE;
}





/* -------------------------------------------------------
 * 図形データ情報
 */
int DataInfo(SELECT *select, char *str)
{
	char str1[256], str2[256];
	long no;


	if (select->diagram_point.Nuru_point == NULL) return 0;



	/* Mouse Point Data */
	if (select->index == 1) {

	}

	/* CAD Data */
	else if (select->index == 2) {
		/* データの検索(CAD Data から Data No. の long 値を求める) */
		no = cad_search_cad_to_num(select->diagram_point.cad_point, &cad_list_info);

		/* 点データのとき */
		if(select->diagram_point.cad_point->code == 0) {
			FloatOut(str2, select->diagram_point.cad_point->sx, 0);
			strcpy(str1, "(");
			strcat(str1, str2);
			strcat(str1, ",");
			FloatOut(str2, select->diagram_point.cad_point->sy, 0);
			strcat(str1, str2);
			strcat(str1, ")");
			sprintf(str, " CAD No.%d [POINT］ %s", (int)no, str1);
		}

		/* 線データのとき */
		else if(select->diagram_point.cad_point->code == 1) {
			strcpy(str1, "(");
			FloatOut(str2, select->diagram_point.cad_point->sx, 0);
			strcat(str1, str2);
			strcat(str1, ",");
			FloatOut(str2, select->diagram_point.cad_point->sy, 0);
			strcat(str1, str2);
			strcat(str1, ")-(");
			FloatOut(str2, select->diagram_point.cad_point->ex, 0);
			strcat(str1, str2);
			strcat(str1, ",");
			FloatOut(str2, select->diagram_point.cad_point->ey, 0);
			strcat(str1, str2);
			strcat(str1, ")");
			sprintf(str, " CAD No.%d [LINE］ %s", (int) no, str1);
		}

		/* 円弧データのとき */
		else if(select->diagram_point.cad_point->code == 2) {
			strcpy(str1, "(");
			FloatOut(str2, select->diagram_point.cad_point->cx, 0);
			strcat(str1, str2);
			strcat(str1, ",");
			FloatOut(str2, select->diagram_point.cad_point->cy, 0);
			strcat(str1, str2);
			strcat(str1, "), R");
			FloatOut(str2, select->diagram_point.cad_point->r, 0);
			strcat(str1, str2);
			strcat(str1, ", (");
			FloatOut(str2, select->diagram_point.cad_point->sx, 0);
			strcat(str1, str2);
			strcat(str1, ",");
			FloatOut(str2, select->diagram_point.cad_point->sy, 0);
			strcat(str1, str2);
			strcat(str1, ")-(");
			FloatOut(str2, select->diagram_point.cad_point->ex, 0);
			strcat(str1, str2);
			strcat(str1, ",");
			FloatOut(str2, select->diagram_point.cad_point->ey, 0);
			strcat(str1, str2);
			strcat(str1, ")");
			sprintf(str, " CAD No.%d [ARC］ %s", (int) no, str1);
		}

		/* 円データのとき */
		else if(select->diagram_point.cad_point->code == 4)
		{
			strcpy(str1, "(");
			FloatOut(str2, select->diagram_point.cad_point->cx, 0);
			strcat(str1, str2);
			strcat(str1, ",");
			FloatOut(str2, select->diagram_point.cad_point->cy, 0);
			strcat(str1, str2);
			strcat(str1, "), R");
			FloatOut(str2, select->diagram_point.cad_point->r, 0);
			strcat(str1, str2);
			sprintf(str, " CAD No.%d [CIRCLE] %s", (int) no, str1);
		}
		else return 0;
	}


	/* Dimension Data */
	else if (select->index == 3) {
		/* データの検索(DIMENSION Data から Data No. の long 値を求める) */
		no = dimension_search_dimension_to_num(select->diagram_point.dimension_point, &dimension_list_info);
		sprintf(str, " DIMENSION No.%d ", (int)no);
	}


	/* polyline Data */
	else if (select->index == 8) {
		/* データの検索(POLYLINE Data から Data No. の long 値を求める) */
		no = polyline_search_polyline_to_num(select->diagram_point.polyline_point, &polyline_list_info);
		if (select->diagram_point.polyline_point->code == 8) {
			if (  (select->diagram_point.polyline_point->frag & 1) == 1  ) {
				sprintf(str, _(" POLYLINE No.%d [broken line (close)") , (int)no);
			}
			else {
				sprintf(str, _(" POLYLINE No.%d [broken line (open)") , (int)no);
			}
		}
		else if (select->diagram_point.polyline_point->code == 16) {
			if (  (select->diagram_point.polyline_point->frag & 1) == 1  ) {
				sprintf(str, _(" POLYLINE No.%d [spline(close)]") , (int)no);
			}
			else {
				sprintf(str, _(" POLYLINE No.%d [spline(open)]") , (int)no);
			}
		}
		else if (select->diagram_point.polyline_point->code == 32) {
			sprintf(str, " POLYLINE No.%d [Ｂスプライン曲線]", (int)no);
		}
		else {
			sprintf(str, " POLYLINE No.%d ", (int)no);
		}
	}


	/* ellipse Data */
	else if (select->index == 16) {
		/* データの検索(ELLIPSE Data から Data No. の long 値を求める) */
		no = ellipse_search_ellipse_to_num(select->diagram_point.ellipse_point, &ellipse_list_info);

		strcpy(str1, "(");
		FloatOut(str2, select->diagram_point.ellipse_point->cx, 0);
		strcat(str1, str2);
		strcat(str1, ",");
		FloatOut(str2, select->diagram_point.ellipse_point->cy, 0);
		strcat(str1, str2);
		strcat(str1, "), a = ");
//g_print ("DataInfo() : call\n");
		FloatOut(str2, get_ellipse_a(*select->diagram_point.ellipse_point), 0);
		strcat(str1, str2);
		strcat(str1, ", b = ");
		FloatOut(str2, get_ellipse_b(*select->diagram_point.ellipse_point), 0);
		strcat(str1, str2);
		strcat(str1, ", angle = ");
		FloatOut(str2, get_ellipse_angle(*select->diagram_point.ellipse_point), 0);
		strcat(str1, str2);

		strcat(str1, ", sa = ");
		FloatOut(str2, select->diagram_point.ellipse_point->sa, 0);
		strcat(str1, str2);
		strcat(str1, ", ea = ");
		FloatOut(str2, select->diagram_point.ellipse_point->ea, 0);
		strcat(str1, str2);


		sprintf(str, " ELLIPSE No.%d %s", (int)no, str1);
	}


	/* Error Data */
	else {
		// 
	}


	return 1;
}




/* -------------------------------------------------------
 * セレクトモードの Sub
 */
int MouseSelectL(double x, double y, int SelectFrag, SELECT *RetSelect)
{
	CAD *Ret;

	if (mode == 200) {
		/* 点を探す */
		if ((SelectFrag & 1) == 1) {
			Ret = TenLook(x, y);
			if (Ret != NULL) {
				RetSelect->index = 2;
				RetSelect->diagram_point.cad_point = Ret;
				RetSelect->x = 0;
				RetSelect->y = 0;
				return 1;
			}
		}
	}

	Ret = Look(x, y, SelectFrag);
	if (Ret != NULL) {
		RetSelect->index = 2;
		RetSelect->diagram_point.cad_point = Ret;
		RetSelect->x = 0;
		RetSelect->y = 0;
		return 1;
	}
	return 0;
}





/* -------------------------------------------------------------------
 * 探す
 * マウスから読み謔ﾁた座標 (x , y) から、その点に近い [点・線・円弧・円] を探す
 * ( DC の値よりも近い値 )
 *	
 * 戻値  CAD *	 CAD Data へのポインタを返す。
 * 外部  CAD_LIST StartData
 */
CAD *Look(double x, double y, int SelectFrag)
{
//#define LOOK
	CAD_LIST *p;
	int debug = 0;

#ifdef LOOK
	debug = 1;
#endif

	if (StartData == NULL && cad_list_info.head != NULL) {
		if (debug > 0) g_print("Look Function in StartData = NULL\n");
		StartData = cad_list_info.tail;
	}
	else if (cad_list_info.head == NULL) {
		if (debug > 0) g_print("Look CAD Data がない！");
		return NULL;
	}

	/* p に StartData を代入 */
	p = StartData;

	/* -----------------------------------------------------
	 * p が最後のデータ tail なら p を最初のデータ head にする。
	 * それ以外なら、p の氓ﾌデータ(StartData の樊から探す
	 */
	if (debug > 0) g_print("Look Function Start StartData = %d\n", (int)cad_search_cad_to_num(p->cad, &cad_list_info));
	if (p == cad_list_info.tail) {
		p = cad_list_info.head;
	}
	else if (p != cad_list_info.tail) { 
		p = p->next;
	}
	if (debug > 0) g_print("Look : First Search Start Data No = %d\n", (int)cad_search_cad_to_num(p->cad, &cad_list_info));


	/* p から データの最後までを探す */
	while (p != NULL) {
		if (TenCheck(x, y, p) != 0 && (SelectFrag & 1) == 1) {
			/* ----------------------------------------------
			 * 見つかった場合、そのデータを氓ﾌ図形サーチの
			 * 最初のデータとするために StartData に代入する。
			 */
			StartData = p;
			if (debug > 0) g_print("    Look (point) : First Search Data No = %d  <YES>\n", (int)cad_search_cad_to_num(p->cad, &cad_list_info));
			return p->cad;
		}
		if (LineCheck(x, y, p) != 0 && (SelectFrag & 2) == 2) {
			StartData = p;
			if (debug > 0) g_print("    Look (line) : First Search Data No = %d  <YES>\n", (int)cad_search_cad_to_num(p->cad, &cad_list_info));
			return p->cad;
		}
		if (ArcCheck(x, y, p) != 0 && (SelectFrag & 4) == 4) {
			StartData = p;
			if (debug > 0) g_print("    Look (arc) : First Search Data No = %d  <YES>\n", (int)cad_search_cad_to_num(p->cad, &cad_list_info));
			return p->cad;
		}
		if (CirCheck(x, y, p) != 0 && (SelectFrag & 8) == 8) {
			StartData = p;
			if (debug > 0) g_print("    Look (circle) : First Search Data No = %d  <YES>\n", (int)cad_search_cad_to_num(p->cad, &cad_list_info));
			return p->cad;
		}
		if (debug > 0) g_print("    Look : First Search Data No = %d  <NO>\n", (int)cad_search_cad_to_num(p->cad, &cad_list_info));
		p = p->next;
	}


	/* -----------------------------------------------------
	 * StartData が tail だったら
	 * 最初から最後まで探しているからこの後の処理はいらない。
	 */
	if (StartData == cad_list_info.tail) {
		if (debug > 0) g_print("Look First Search Only Return\n");
		return NULL;
	}


	/* ------------------------------------------------------------
	 * データの最初から StartData までを探す。
	 */
	p =  cad_list_info.head;
	if (debug > 0) g_print("Look : Second Search Start Data No = %d\n", (int)cad_search_cad_to_num(p->cad, &cad_list_info));
	while (p != NULL && p != StartData->next) {
		if (TenCheck(x, y, p) != 0 && (SelectFrag & 1) == 1) {
			StartData = p;
			if (debug > 0) g_print("    Look : Second Search Data No = %d  <YES>\n", (int)cad_search_cad_to_num(p->cad, &cad_list_info));
			return p->cad;
		}
		if (LineCheck(x, y, p) != 0 && (SelectFrag & 2) == 2) {
			StartData = p;
			if (debug > 0) g_print("    Look : Second Search Data No = %d  <YES>\n", (int)cad_search_cad_to_num(p->cad, &cad_list_info));
			return p->cad;
		}
		if (ArcCheck(x, y, p) != 0 && (SelectFrag & 4) == 4) {
			StartData = p;
			if (debug > 0) g_print("    Look : Second Search Data No = %d  <YES>\n", (int)cad_search_cad_to_num(p->cad, &cad_list_info));
			return p->cad;
		}
		if (CirCheck(x, y, p) != 0 && (SelectFrag & 8) == 8) {
			StartData = p;
			if (debug > 0) g_print("    Look : Second Search Data No = %d  <YES>\n", (int)cad_search_cad_to_num(p->cad, &cad_list_info));
			return p->cad;
		}
		if (debug > 0) g_print("    Look : Second Search Data No = %d  <NO>\n", (int)cad_search_cad_to_num(p->cad, &cad_list_info));
		p = p->next;
	}
	StartData = p;
	if (debug > 0) g_print("Look : Second Search Return StartData = %d\n",  (int)cad_search_cad_to_num(StartData->cad, &cad_list_info));
	return NULL;
}





/* -------------------------------------------------------------------
 * セレクト図形のレイヤー変更
 *	
 * Change_Frag 1 : 移動
 * Change_Frag 2 : 複写
 */
int Change_Layer(int Change_Frag)
{
	CAD NewCad;
	DIMENSION NewDimension;
	POLYLINE Newpolyline;
	ELLIPSE NewEllipse;

	CAD_LIST *cp;
	DIMENSION_LIST *dp;
	POLYLINE_LIST *pp;
	ELLIPSE_LIST *ep;

	SELECT_LIST *sp;
	int LayerNo, Ret = 0;
	char org[256];

	VERTEX_LIST *v;
	VERTEX vertex;


	/* 数値入力ダイアログボックス表示 */
	if (Change_Frag == 1) {
		Ret = InputBox( _("Layer Move") , _("The layer in the movement place.") , _("") , org);
	}
	else if (Change_Frag == 2) {
		Ret = InputBox( _("Layer Copy") , _("The layer in the copy place."), _("") , org);
	}
//Ret = 0; /* ＜---あとでとる */

	if (Ret == -1) return 0;
	LayerNo = atoi(org);
	if (LayerNo < 0 || LayerNo > 255) return 0;


	if (select_list_info.head != NULL) undo_index_plus();
	/* 移動 */
	if (Change_Frag == 1) {
		sp = select_list_info.head;
		while (sp != NULL) {

			/* 元の図を消す */
			SelectDraw(drawing_area, sp->select, SCD_HIDE, UPDATE_OFF);

			/* 図形データ */
			if (sp->select->index == 2 || sp->select->index == 4) {
				NewCad = *sp->select->diagram_point.cad_point;
				NewCad.layer = LayerNo;
				/* CAD Data を変更する、 Undo Baffer に書込む */
				cp = cad_list_edit_with_undo(
							cad_search_cad(sp->select->diagram_point.cad_point, &cad_list_info), 
							&NewCad, 
							&cad_list_info);
			}

			/* 寸法データ */
			else if (sp->select->index == 3) {
				NewDimension = *sp->select->diagram_point.dimension_point;
				NewDimension.Layer = LayerNo;
				/* DIMENSION Data を変更する、 Undo Baffer に書込む */
				dp = dimension_list_edit_with_undo(
							dimension_search_dimension(sp->select->diagram_point.dimension_point, &dimension_list_info), 
							&NewDimension, &dimension_list_info);
			}

			/* 曲線データ */
			else if (sp->select->index == 8) {
				Newpolyline = *sp->select->diagram_point.polyline_point;
				Newpolyline.layer = LayerNo;
				sp->select->diagram_point.polyline_point->layer = LayerNo;
				/* POLYLINE Data を変更する、 Undo Baffer に書込む */
				pp = polyline_list_edit_with_undo(
							polyline_search_polyline(sp->select->diagram_point.polyline_point, &polyline_list_info), 
							&Newpolyline, 
							&polyline_list_info);
			}

			/* 楕円データ */
			else if (sp->select->index == 16) {
				NewEllipse = *sp->select->diagram_point.ellipse_point;
				NewEllipse.layer = LayerNo;
				/* ELLIPSE Data を変更する、 Undo Baffer に書込む */
				ep = ellipse_list_edit_with_undo(
							ellipse_search_ellipse(sp->select->diagram_point.ellipse_point, &ellipse_list_info), 
							&NewEllipse, 
							&ellipse_list_info);
			}

			/* 移動先が表示中なら表示 */
			if (Layer[LayerNo].draw == 1) {
				SelectDraw(drawing_area, sp->select, SCD_SELECT, UPDATE_OFF);
			}
			sp = sp->next;
		}
	}



	/* 複写 */
	else if (Change_Frag == 2) {
		sp = select_list_info.head;
		while (sp != NULL) {

			/* 元の図を消して表示 */
			SelectDraw(drawing_area, sp->select, SCD_HIDE, UPDATE_OFF);
			SelectDraw(drawing_area, sp->select, SCD_ORG, UPDATE_OFF);

			/* 図形データ */
			if (sp->select->index == 2 || sp->select->index == 4) {
				NewCad = *sp->select->diagram_point.cad_point;
				NewCad.layer = LayerNo;
				/* CAD Data をリストの最初に追加 */
				cp = cad_list_add_first_with_undo(&NewCad, &cad_list_info);
				sp->select->diagram_point.cad_point = cp->cad;
			}

			/* 寸法データ */
			else if (sp->select->index == 3) {
				NewDimension = *sp->select->diagram_point.dimension_point;
				NewDimension.Layer = LayerNo;
				/* Dimension Data をリストの最初に追加 */
				dp = dimension_list_add_first_with_undo(&NewDimension, &dimension_list_info);
				sp->select->diagram_point.dimension_point = dp->dimension;
			}

			/* 曲線データ */
			if (sp->select->index == 8) {
				Newpolyline = *sp->select->diagram_point.polyline_point;
				Newpolyline.layer = LayerNo;

				v = sp->select->diagram_point.polyline_point->vertex_list_info.head;
				while (v != NULL) {
					vertex = *v->vertex;
					vertex_list_add_last(&vertex, &Newpolyline.vertex_list_info);
					v = v->next;
				}

				/* POLYLINE Data をリストの最初に追加 */
				pp = polyline_list_add_first(&Newpolyline, &polyline_list_info);
				sp->select->diagram_point.polyline_point = pp->polyline;
			}

			/* 楕円データ */
			else if (sp->select->index == 16) {
				NewEllipse = *sp->select->diagram_point.ellipse_point;
				NewEllipse.layer = LayerNo;
				/* CAD Data をリストの最初に追加 */
				ep = ellipse_list_add_first_with_undo(&NewEllipse, &ellipse_list_info);
				sp->select->diagram_point.ellipse_point = ep->ellipse;
			}

			/* 移動先が表示中なら表示 */
			if (Layer[LayerNo].draw == 1) {
				SelectDraw(drawing_area, sp->select, SCD_SELECT, UPDATE_OFF);
			}
			sp = sp->next;
		}
	}
	UpDate();
	return 1;
}





/* -------------------------------------------------------------------
 * データ№から、ＳＡ　ＥＡを求める
 * 戻値 : int
 *		: RtnDumy.sx1 = sa	RtnDumy.sy1 = ea
 */
int iSaEa(CAD_LIST *p, struct RtnDat *RtnDumy)
{
	struct RtnDat rtn;

	rtn.sx[1] = p->cad->cx;
	rtn.sy[1] = p->cad->cy;
	rtn.ex[1] = p->cad->sx;
	rtn.ey[1] = p->cad->sy;
	la(&rtn);
	RtnDumy->sx[1] = rtn.angle;

	rtn.sx[1] = p->cad->cx;
	rtn.sy[1] = p->cad->cy;
	rtn.ex[1] = p->cad->ex;
	rtn.ey[1] = p->cad->ey;
	la(&rtn);
	RtnDumy->sy[1] = rtn.angle;
	return 1;
}





/* -------------------------------------------------------
 * 図形データ *p の円弧に対して angle は円弧の SA EA 中か外か
 * 戻値 : int	1: 中	0：外
 */
int ArcIn(CAD_LIST *p, double Angle)
{
	double SA, EA;
	struct RtnDat rtn;
	
	/* ＳＡ ＥＡを求める。 */
	iSaEa(p, &rtn);
	SA = rtn.sx[1];
	EA = rtn.sy[1];
	
	/* --- <	> --- */
	if(SA < EA) {
		if(sg(Angle, 6) >= sg(SA, 6) && sg(Angle, 6) <= sg(EA, 6)) {
			return 1;
		}
	}
	else if(SA > EA) {
		if(sg(Angle, 6) >= sg(SA, 6) || sg(Angle, 6) <= sg(EA, 6)) {
			return 1;
		}
	}
	return 0;
}





/* -------------------------------------------------------------------
 * 点を探す
 * マウスから読み謔ﾁた座標 (x , y) から、その点に近い点を探す
 * ( DC の値よりも近い値 )
 *	
 * 戻値  CAD *	 CAD Data へのポインタを返す。
 * 外部  CAD_LIST StartData
 */
CAD *TenLook(double x, double y)
{
	CAD_LIST *p;


	/* StartData */
	if (StartData == NULL && cad_list_info.head != NULL) {
		StartData = cad_list_info.tail;
	}
	/* CAD DATA がない */
	else if (cad_list_info.head == NULL) {
		return NULL;
	}

	/* p に StartData を代入 */
	p = StartData;

	/* -----------------------------------------------------
	 * p が最後のデータ tail なら p を最初のデータ head にする。
	 * それ以外なら、p の氓ﾌデータ(StartData の樊から探す
	 */
	if (p == cad_list_info.tail) {
		p = cad_list_info.head;
	}
	else if (p != cad_list_info.tail) { 
		p = p->next;
	}


	/* p から データの最後までを探す */
	while (p != NULL) {
		if (TenCheck(x, y, p) != 0) {
			/* ----------------------------------------------
			 * 見つかった場合、そのデータを氓ﾌ図形サーチの
			 * 最初のデータとするために StartData に代入する。
			 */
			StartData = p;
			return p->cad;
		}
		p = p->next;
	}

	/* -----------------------------------------------------
	 * StartData が tail だったら
	 * 最初から最後まで探しているからこの後の処理はいらない。
	 */
	if (StartData == cad_list_info.tail) {
		return NULL;
	}

	/* ------------------------------------------------------------
	 * データの最初から StartData までを探す
	 */
	p =  cad_list_info.head;
	while (p != NULL && p != StartData->next) {
		if (TenCheck(x, y, p) != 0) {
			/* ----------------------------------------------
			 * 見つかった場合、そのデータを氓ﾌ図形サーチの
			 * 最初のデータとするために StartData に代入する。
			 */
			StartData = p;
			return p->cad;
		}
		p = p->next;
	}
	StartData = p;
	return NULL;
}





/* -------------------------------------------------------
 * 点を探す (sub)
 * 座標 (x , y) が、データ№ i の点であるかをチェック。
 * (DC の値よりも近い値の点)
 */
int TenCheck(double x, double y, CAD_LIST *p)
{
	struct RtnDat PPH;


//	if (cad_search_cad(p->cad) == NULL) return 0;

	if (p->cad->code == 0) {
		if (Layer[p->cad->layer].draw != 0) {
			PPH.sx[1] = p->cad->sx;
			PPH.sy[1] = p->cad->sy;
			PPH.ex[1] = x;
			PPH.ey[1] = y;
			pp(&PPH);
			if (PPH.l <= DC) {
				return 1;
			}
		}
	}
	return 0;
}





/* -------------------------------------------------------------------
 * 線を探す
 * マウスから読み謔ﾁた座標 (x , y) から、その点に近い線を探す
 * ( DC の値よりも近い値 )
 *	
 * 戻値  CAD *	 CAD Data へのポインタを返す。
 * 外部  CAD_LIST StartData
 */
CAD *LineLook(double x, double y)
{
	CAD_LIST *p;


	if (StartData == NULL && cad_list_info.head != NULL) {
		StartData = cad_list_info.tail;
	}
	else if (cad_list_info.head == NULL) {
		return NULL;
	}

	/* p に StartData を代入 */
	p = StartData;

	/* -----------------------------------------------------
	 * p が最後のデータ tail なら p を最初のデータ head にする。
	 * それ以外なら、p の氓ﾌデータ(StartData の樊から探す
	 */
	if (p == cad_list_info.tail) {
		p = cad_list_info.head;
	}
	else if (p != cad_list_info.tail) { 
		p = p->next;
	}


	/* p から データの最後までを探す */
	while (p != NULL) {
		if (LineCheck(x, y, p) != 0) {
			/* ----------------------------------------------
			 * 見つかった場合、そのデータを氓ﾌ図形サーチの
			 * 最初のデータとするために StartData に代入する。
			 */
			StartData = p;
			return p->cad;
		}
		p = p->next;
	}


	/* -----------------------------------------------------
	 * StartData が tail だったら
	 * 最初から最後まで探しているからこの後の処理はいらない。
	 */
	if (StartData == cad_list_info.tail) {
		return NULL;
	}


	/* ------------------------------------------------------------
	 * データの最初から StartData までを探す。
	 */
	p =  cad_list_info.head;
	while (p != NULL && p != StartData->next) {
		if (LineCheck(x, y, p) != 0) {
			/* ----------------------------------------------
			 * 見つかった場合、そのデータを氓ﾌ図形サーチの
			 * 最初のデータとするために StartData に代入する。
			 */
			StartData = p;
			return p->cad;
		}
		p = p->next;
	}
	StartData = p;
	return NULL;
}





/* -------------------------------------------------------
 * 線を探す (sub)
 */
int LineCheck(double x, double y, CAD_LIST *p)
{
	struct RtnDat PLPH, PPH, LAH;
	double L0, L1, A1, L2, A2;


	/* 線に対して探す */
	if (p->cad->code == 1) {
		/* 表示しているレイヤーから探す */
		if (Layer[p->cad->layer].draw != 0) {
			/* 点と線の最近点を求める */
			PLPH.sx[2] = x;
			PLPH.sy[2] = y;
			PLPH.sx[1] = p->cad->sx;
			PLPH.sy[1] = p->cad->sy;
			PLPH.ex[1] = p->cad->ex;
			PLPH.ey[1] = p->cad->ey;
			plp(&PLPH);
			/* 最近点	 PLPH.ex[2] , PLPH.ey[2] */
			
			/* 点と最近点の距離を求める */
			PPH.sx[1] = x;
			PPH.sy[1] = y;
			PPH.ex[1] = PLPH.ex[2];
			PPH.ey[1] = PLPH.ey[2];
			pp(&PPH);
			/* 距離 PPH.L */
			L0 = PPH.l;
			
			/* 距離が DC 以上なら探している線ではない。*/
			if (L0 > DC) {
				return 0;
			}
			
			
			/* 線のn点から終点の角度と距離を求める */
			PPH.sx[1] = p->cad->sx;
			PPH.sy[1] = p->cad->sy;
			PPH.ex[1] = p->cad->ex;
			PPH.ey[1] = p->cad->ey;
			pp(&PPH);
			/* 距離 PPH.L */
			L1 = PPH.l;
			
			LAH.sx[1] = p->cad->sx;
			LAH.sy[1] = p->cad->sy;
			LAH.ex[1] = p->cad->ex;
			LAH.ey[1] = p->cad->ey;
			la(&LAH);
			/* 角度 LAH.ANGLE */
			A1 = LAH.angle;
			
			/* 線のn点から最近点の角度と距離を求める */
			PPH.sx[1] = p->cad->sx;
			PPH.sy[1] = p->cad->sy;
			PPH.ex[1] = PLPH.ex[2];
			PPH.ey[1] = PLPH.ey[2];
			pp(&PPH);
			/* 距離 PPH.L */
			L2 = PPH.l;

			LAH.sx[1] = p->cad->sx;
			LAH.sy[1] = p->cad->sy;
			LAH.ex[1] = PLPH.ex[2];
			LAH.ey[1] = PLPH.ey[2];
			la(&LAH);
			/* 角度 LAH.ANGLE */
			A2 = LAH.angle;
			
			/* n点からの角度は同じで、n点から最近点までの長さよりも線の長さのが長い(L1>L2)*/
			if (A1 + (1/pow(10, compa_digits)) > A2 
				&& 
				A1 - (1/pow(10, compa_digits)) < A2) 
			{
//			if (sg(A1, 1) == sg(A2, 1)) {
				if (L1 > L2) {
					return 1;
				}
			}
			else {
				return 0;
			}
		}
	}
	return 0;
}





/* -------------------------------------------------------------------
 * 円弧を探す
 * マウスから読み謔ﾁた座標 (x , y) から、その点に近い円弧を探す
 * ( DC の値よりも近い値 )
 *	
 * 戻値  CAD *	 CAD Data へのポインタを返す。
 * 外部  CAD_LIST StartData
 */
CAD *ArcLook(double x, double y)
{
	CAD_LIST *p;
	
	if (StartData == NULL && cad_list_info.head != NULL) {
		StartData = cad_list_info.tail;
	}
	else if (cad_list_info.head == NULL) {
		return NULL;
	}

	/* p に StartData を代入 */
	p = StartData;

	/* -----------------------------------------------------
	 * p が最後のデータ tail なら p を最初のデータ head にする。
	 * それ以外なら、p の氓ﾌデータ(StartData の樊から探す
	 */
	if (p == cad_list_info.tail) {
		p = cad_list_info.head;
	}
	else if (p != cad_list_info.tail) { 
		p = p->next;
	}


	/* p から データの最後までを探す */
	while (p != NULL) {
		if (ArcCheck(x, y, p) != 0) {
			/* ----------------------------------------------
			 * 見つかった場合、そのデータを氓ﾌ図形サーチの
			 * 最初のデータとするために StartData に代入する。
			 */
			StartData = p;
			return p->cad;
		}
		p = p->next;
	}

	/* -----------------------------------------------------
	 * StartData が tail だったら
	 * 最初から最後まで探しているからこの後の処理はいらない。
	 */
	if (StartData == cad_list_info.tail) {
		return NULL;
	}

	/* ------------------------------------------------------------
	 * データの最初から StartData までを探す
	 */
	p =  cad_list_info.head;
	while (p != NULL && p != StartData->next) {
		if (ArcCheck(x, y, p) != 0) {
			/* ----------------------------------------------
			 * 見つかった場合、そのデータを氓ﾌ図形サーチの
			 * 最初のデータとするために StartData に代入する。
			 */
			StartData = p;
			return p->cad;
		}
		p = p->next;
	}
	StartData = p;
	return NULL;
}





/* -------------------------------------------------------
 * 円弧を探す (sub)
 * マウスから読み謔ﾁた座標 (x , y) から、その点に近い円弧を探す。
 * （DCの値よりも近い値の円弧）
 * データ№ i の図形につながている図形をセレクトする
 */
int ArcCheck( double x , double y , CAD_LIST *p)
{
	double MAXR , MINR , CL , SA ,EA ;
	struct RtnDat LAH, ISAEA;


	if (p->cad->code == 2) {
		if (Layer[p->cad->layer].draw != 0) {
			/* まずマウスの座標と円の中心の距離が半径Ｒになるかをチェック。*/
			CL = sqrt((x - p->cad->cx) * (x - p->cad->cx) + (y - p->cad->cy) * (y - p->cad->cy));
			MAXR = p->cad->r + DC;
			MINR = p->cad->r - DC;
			if (MAXR > CL && MINR < CL) {
				/* ＳＡ ＥＡを求める。*/
				iSaEa(p, &ISAEA);
				SA = ISAEA.sx[1];
				EA = ISAEA.sy[1];
				/* 円の中心からクリック点への角度 */
				LAH.sx[1] = p->cad->cx;
				LAH.sy[1] = p->cad->cy;
				LAH.ex[1] = x;
				LAH.ey[1] = y;
				la(&LAH);
				
				if (SA < EA) {
					if (LAH.angle > SA && LAH.angle < EA) {
						return (1);
					}
				}
				else if (SA > EA) {
					if (LAH.angle >= SA || LAH.angle <= EA) {
						return (1);
					}
				}
			}
		}
	}
	return 0;
}





/* -------------------------------------------------------------------
 * 円を探す
 * マウスから読み謔ﾁた座標 (x , y) から、その点に近い円を探す
 * ( DC の値よりも近い値 )
 *	
 * 戻値  CAD *	 CAD Data へのポインタを返す。
 * 外部  CAD_LIST StartData
 */
CAD *CirLook(double x, double y)
{
	CAD_LIST *p;
	
	if (StartData == NULL && cad_list_info.head != NULL) {
		StartData = cad_list_info.tail;
	}
	else if (cad_list_info.head == NULL) {
		return NULL;
	}

	/* p に StartData を代入 */
	p = StartData;

	/* -----------------------------------------------------
	 * p が最後のデータ tail なら p を最初のデータ head にする。
	 * それ以外なら、p の氓ﾌデータ(StartData の樊から探す
	 */
	if (p == cad_list_info.tail) {
		p = cad_list_info.head;
	}
	else if (p != cad_list_info.tail) { 
		p = p->next;
	}


	/* p から データの最後までを探す */
	while (p != NULL) {
		if (CirCheck(x, y, p) != 0) {
			/* ----------------------------------------------
			 * 見つかった場合、そのデータを氓ﾌ図形サーチの
			 * 最初のデータとするために StartData に代入する。
			 */
			StartData = p;
			return p->cad;
		}
		p = p->next;
	}

	/* -----------------------------------------------------
	 * StartData が tail だったら
	 * 最初から最後まで探しているからこの後の処理はいらない。
	 */
	if (StartData == cad_list_info.tail) {
		return NULL;
	}

	/* ------------------------------------------------------------
	 * データの最初から StartData までを探す
	 */
	p =  cad_list_info.head;
	while (p != NULL && p != StartData->next) {
		if (CirCheck(x, y, p) != 0) {
			/* ----------------------------------------------
			 * 見つかった場合、そのデータを氓ﾌ図形サーチの
			 * 最初のデータとするために StartData に代入する。
			 */
			StartData = p;
			return p->cad;
		}
		p = p->next;
	}
	StartData = p;
	return NULL;
}





/* -------------------------------------------------------
 * 円を探す (sub)
 * マウスから読み謔ﾁた座標 (x , y) から、その点に近い円を探す。
 * （DCの値よりも近い値の円）
 */
int CirCheck(double x, double y, CAD_LIST *p)
{
	double MAXR , MINR , CL ;

	if (p->cad->code == 4) {
		if (Layer[p->cad->layer].draw != 0) {
			/* まずマウスの座標と円の中心の距離が半径Ｒになるものを探す。*/
			CL = sqrt((x - p->cad->cx) * (x - p->cad->cx) + (y - p->cad->cy) * (y - p->cad->cy));
			MAXR = p->cad->r + DC;
			MINR = p->cad->r - DC;
			if (MAXR > CL && MINR < CL) {
				return 1;
			}
		}
	}
	return 0;
}


/* -------------------------------------------------------
 * セレクト図形の線種変更
 */
int Change_Line_Style(void)
{
	int style;
	long color;
	SELECT_LIST *ps;

	style = NowStyle;
	color = NowColor;

	g_print("Change_Line_Style() : Line_set_dlg START !\n");
	Line_set_dlg(&style, &color);
	g_print("Change_Line_Style() : Line_set_dlg END !\n");

	/* -----------------------------------------------------------------
	 * 選択図形を処理
	 */
	if (select_list_info.head != NULL) {
		ps = select_list_info.head;
		while (ps != NULL) {

			/* CAD Data */
			if (ps->select->index == 2 || ps->select->index == 4) {
				ps->select->diagram_point.cad_point->color = color;
				ps->select->diagram_point.cad_point->style = style;
			}

			/* DIMENSION Data */
			else if (ps->select->index == 3) {
				ps->select->diagram_point.dimension_point->text_color = color;
				ps->select->diagram_point.dimension_point->Assist_color = color;
			}

			/* POLYLINE Data */
			else if (ps->select->index == 8) {
				ps->select->diagram_point.polyline_point->color = color;
				ps->select->diagram_point.polyline_point->style = style;
			}

			/* ELLIPSE Data */
			else if (ps->select->index == 16) {
				ps->select->diagram_point.ellipse_point->color = color;
				ps->select->diagram_point.ellipse_point->style = style;
			}

			SelectDraw(drawing_area, ps->select, SCD_HIDE, UPDATE_OFF);
			SelectDraw(drawing_area, ps->select, SCD_SELECT, UPDATE_OFF);

			ps = ps->next;
		}
	}
	return 1;
}



/* -------------------------------------------------------
 * データ№ i の図形がセレクトされているかを確認
 *	
 * 引数 ： int	i  データナンバー
 * 戻値 ： int	0：選択されていない 	1：選択済み
 */
int SelectCheck(CAD *data)
{
	SELECT_LIST *p;

	p = select_list_info.head;
	while (p != NULL) {
		if (p->select->diagram_point.cad_point == data) 
			return 1;
		p = p->next;
	}
	return 0;
}




/* -------------------------------------------------------
 * ２点で作られるl角の中にすっぽり入るものを探し、
 * セレクトデータに入れる。
 *	
 * 戻値 ： int	0:セレクト図形無し。	1:セレクト図形有り。
 */
int SqSelect(GtkWidget *widget , double Mode310X1, double Mode310Y1, double Mode310X2, double Mode310Y2)
{
	double Dumy;
	CAD_LIST *p;
	DIMENSION_LIST *dp;
	POLYLINE_LIST *pp;
	ELLIPSE_LIST *pe;
	VERTEX_LIST *pv;
	SELECT slt;
	SAG_POINT s_point, e_point;
	int ATTA;

	SELECT_LIST *ps;


	/*
	 * ２点で作られるl角の中にすっぽり入るものを探し、セレクトデータに入れる。
	 * 選んである要素はまた選ばない。
	 * レイヤーで表示されていないものは選ばない。
	 * サーチ設定で設定されている要素を選ぶ。
	 */
	if (Mode310X1 > Mode310X2) {
		Dumy = Mode310X1;
		Mode310X1 = Mode310X2;
		Mode310X2 = Dumy;
	}

	if ( Mode310Y1 > Mode310Y2) {
		Dumy = Mode310Y1;
		Mode310Y1 = Mode310Y2;
		Mode310Y2 = Dumy;
	}


	/* -----------------------------------------------------
	 * 図形データ 
	 */
	p = cad_list_info.head;
	while (p != NULL) {
		if ( Layer[p->cad->layer].draw != 0) {
			ATTA = 0;
			/* 点 */
			if (( p->cad->code == 0) && ( (DcObj & 1) == 1 )
					&& ( Mode310X1 < p->cad->sx && Mode310X2 > p->cad->sx)
						&& ( Mode310Y1 < p->cad->sy && Mode310Y2 > p->cad->sy)) {
				ATTA = 1;
			}

			/* 線 */
			else if (( p->cad->code == 1) && ( (DcObj & 2) == 2 )
					&& ( Mode310X1 < p->cad->sx && Mode310X2 > p->cad->sx)
						&& ( Mode310Y1 < p->cad->sy && Mode310Y2 > p->cad->sy)
							&& ( Mode310X1 < p->cad->ex && Mode310X2 > p->cad->ex)
								&& ( Mode310Y1 < p->cad->ey && Mode310Y2 > p->cad->ey)) {
				ATTA = 1;
			}

			/* 円弧 */
			else if (( p->cad->code == 2) && ( (DcObj & 4) == 4 )
					&& ( Mode310X1 < p->cad->sx && Mode310X2 > p->cad->sx )
						&& ( Mode310Y1 < p->cad->sy && Mode310Y2 > p->cad->sy )
							&& ( Mode310X1 < p->cad->ex && Mode310X2 > p->cad->ex )
								&& ( Mode310Y1 < p->cad->ey && Mode310Y2 > p->cad->ey)) {
				ATTA = 1;
			}

			/* 円 */
			else if ((p->cad->code == 4 ) && ( (DcObj & 8) == 8 )
					&& ( Mode310X1 < p->cad->cx + p->cad->r && Mode310X2 > p->cad->cx + p->cad->r )
						&& ( Mode310Y1 < p->cad->cy && Mode310Y2 > p->cad->cy )
							&& ( Mode310X1 < p->cad->cx && Mode310X2 > p->cad->cx )
								&& ( Mode310Y1 < p->cad->cy+p->cad->r && Mode310Y2 > p->cad->cy+p->cad->r )
									&& (Mode310X1 < p->cad->cx-p->cad->r && Mode310X2 > p->cad->cx-p->cad->r)
										&& (Mode310Y1<p->cad->cy-p->cad->r && Mode310Y2>p->cad->cy-p->cad->r)) {
				ATTA = 1;
			}


			/* 図形が見つかったら */
			if (ATTA == 1) {
				/* 前に同じデータがなかったら Return 0 */
				if (SelectCheck(p->cad) == 0) {
					slt.index = 2;
					slt.x = 0;
					slt.y = 0;
					slt.diagram_point.cad_point = p->cad;
					/* Select Data をリストの最後に追加 */
					select_list_add_last(&slt);
//					SelectCadDraw(widget, p->cad, SCD_SELECT);
				}
			}
		}
		p = p->next;
	}



	/* -----------------------------------------------------
	 * 寸法データ 
	 */
	dp = dimension_list_info.head;
	while (dp != NULL) {
		if ( Layer[dp->dimension->Layer].draw != 0) {
			if ( (DcObj & 256) == 256 
					&& ( Mode310X1 < dp->dimension->SearchPointX && Mode310X2 > dp->dimension->SearchPointX)
						&& ( Mode310Y1 < dp->dimension->SearchPointY && Mode310Y2 > dp->dimension->SearchPointY)) {
				/* 前に同じデータがなかったら Return 0 */
				if (SelectDimensionCheck(dp->dimension) == 0) {
					slt.index = 3;
					slt.x = 0;
					slt.y = 0;
					slt.diagram_point.dimension_point = dp->dimension;
					/* Select Data をリストの最後に追加 */
					select_list_add_last(&slt);
//					DimensionDraw (drawing_area, dp->dimension, SCD_SELECT);
				}
			}
		}
		dp = dp->next;
	}



	/* -----------------------------------------------------
	 * 曲線データ 
	 */
	pp = polyline_list_info.head;
	while (pp != NULL) {
		if ( Layer[pp->polyline->layer].draw != 0) {
			ATTA = 0;
			if ( (DcObj & 512) == 512 ) {
				pv = pp->polyline->vertex_list_info.head;
				while (pv != NULL) {
					if (
						(
							Mode310X1 < pv->vertex->x 
							&& 
							Mode310X2 > pv->vertex->x 
						)
						&& 
						(
							Mode310Y1 < pv->vertex->y 
							&& 
							Mode310Y2 > pv->vertex->y 
						)
						) 
					{
						ATTA++;
					}
					pv = pv->next;
				}

				if (pp->polyline->index == ATTA) {
					/* 前に同じデータがなかったら Return 0 */
					if (SelectPolyLineCheck(pp->polyline) == 0) {
						slt.index = 8;
						slt.x = 0;
						slt.y = 0;
						slt.diagram_point.polyline_point = pp->polyline;
						/* Select Data をリストの最後に追加 */
						select_list_add_last(&slt);
//						PolyLineDraw(drawing_area, pp->polyline, SCD_SELECT);
					}
				}
			}
		}
		pp = pp->next;
	}



	/* -----------------------------------------------------
	 * 楕円データ 
	 */
	pe = ellipse_list_info.head;
	while (pe != NULL) {
		if ( Layer[pe->ellipse->layer].draw != 0) {

			/* 始点 終点 */
			get_ellipse_se_point(*pe->ellipse, &s_point, &e_point);

			if (
					((DcObj & 2048) == 2048)
					&& 
					( Mode310X1 < s_point.x && Mode310X2 > s_point.x )
					&& 
					( Mode310Y1 < s_point.y && Mode310Y2 > s_point.y )
					&& 
					( Mode310X1 < e_point.x && Mode310X2 > e_point.x )
					&& 
					( Mode310Y1 < e_point.y && Mode310Y2 > e_point.y)
				) 
			{
				/* 図形が見つかったら */
				/* 前に同じデータがなかったら Return 0 */
				if (SelectEllipseCheck(pe->ellipse) == 0) {
					slt.index = 16;
					slt.x = 0;
					slt.y = 0;
					slt.diagram_point.ellipse_point = pe->ellipse;
					/* Select Data をリストの最後に追加 */
					select_list_add_last(&slt);
//					SelectDraw(widget, select_list_info.tail->select, SCD_SELECT, UPDATE_OFF);
				}
			}
		}
		pe = pe->next;
	}



	/* -----------------------------------------------------------------
	 * 選択図形作図
	 */
	if (select_list_info.head != NULL) {
		ps = select_list_info.head;
		while (ps != NULL) {
			SelectDraw(widget, ps->select, SCD_SELECT, UPDATE_OFF);
			ps = ps->next;
		}
	}
	UpDate();


	if (select_list_info.head != NULL) return 1;
	return 0;
}





/* -------------------------------------------------------
 * 選択図形を赤くする。＆キャンセルで元の色に戻す
 *	
 * 選択されている図形を 1 で赤くし、0 で戻す。
 *	
 * 図形描画 	 DrawType
 * 選択敞   ： DrawType = 1	SCD_SELECT
 * 元の色	   ： DrawType = 0	SCD_ORG
 * 消  竿   ： DrawType = 2	SCD_HIDE
 * 	
 */
void SelectCadDraw(GtkWidget *widget, CAD *cad, int DrawType)
{
	long SelectCadDrawColor = cad->color;


	if (cad == NULL) {
		MsgBox("SagCAD error", "CAD Data is NULL !", "SelectCadDraw error", NULL, NULL, "OK");
		return;
	}

	if ( DrawType == SCD_SELECT) SelectCadDrawColor = sagcad_color.Select;
	else if ( DrawType == SCD_ORG) SelectCadDrawColor = cad->color;
	else if ( DrawType == SCD_HIDE) SelectCadDrawColor = sagcad_color.Back;

	/* 点のとき */
	if ( cad->code == 0) {
		if ( Layer[cad->layer].draw != 0) {
			PointDraw(widget, cad->sx, cad->sy, cad->style, sagcad_color.Back);
			PointDraw(widget, cad->sx, cad->sy, cad->style, SelectCadDrawColor);
		}
	}
	/* 直線のとき */
	else if ( cad->code == 1) {
		if ( Layer[cad->layer].draw != 0) {
			LineDraw(widget, cad->sx, cad->sy, cad->ex, cad->ey, cad->style, sagcad_color.Back);
			LineDraw(widget, cad->sx, cad->sy, cad->ex, cad->ey, cad->style, SelectCadDrawColor);
		}
	}
	/* 円弧のとき */
	else if ( cad->code == 2) {
		if ( Layer[cad->layer].draw != 0) {
			/* 円弧を描画 */
			ArcDraw(widget, cad->cx, cad->cy, cad->r, cad->sx, cad->sy, cad->ex, cad->ey, cad->style, sagcad_color.Back);
			ArcDraw(widget, cad->cx, cad->cy, cad->r, cad->sx, cad->sy, cad->ex, cad->ey, cad->style, SelectCadDrawColor);
		}
	}
	/* 円のとき */
	else if ( cad->code == 4) {
		if ( Layer[cad->layer].draw != 0) {
			/* 円を描画 */
			CircleDraw(widget, cad->cx, cad->cy, cad->r, cad->style, sagcad_color.Back);
			CircleDraw(widget, cad->cx, cad->cy, cad->r, cad->style, SelectCadDrawColor);
		}
	}
	UpDate();
}





/* -------------------------------------------------------
 * セレクト図形を削除する
 */
void SelectDelete(GtkWidget *widget)
{
	SELECT_LIST *p;
	char str[256];


	if (select_list_info.head == NULL) return;

	undo_index_plus();

	p = select_list_info.head;
	while (p != NULL) {

		SelectDraw(widget, p->select, SCD_HIDE, UPDATE_OFF);

		/* CAD DATA */
		if (p->select->index == 2 || p->select->index == 4) {
			/* CAD Data の削除機能 (メモリの削除ではなく、Undo Buffer へ移す) */
			cad_list_delete_Undo(cad_search_cad(p->select->diagram_point.cad_point, &cad_list_info), 
								 &cad_list_info);
		}

		/* DIMENSION DATA */
		else if (p->select->index == 3) {
			/* Dimension Data の削除機能 (メモリの削除ではなく、Undo Buffer へ移す) */
			dimension_list_delete_Undo(dimension_search_dimension(p->select->diagram_point.dimension_point, 
									   &dimension_list_info), &dimension_list_info);
		}

		/* POLYLINE DATA */
		else if (p->select->index == 8) {
			/* PolyLine Data の削除機能 (メモリの削除ではなく、Undo Buffer へ移す) */
			polyline_list_delete_Undo(
					polyline_search_polyline(p->select->diagram_point.polyline_point, &polyline_list_info), 
					&polyline_list_info);
		}

		/* ELLIPSE DATA */
		else if (p->select->index == 16) {
			/* ELLIPSE Data の削除機能 (メモリの削除ではなく、Undo Buffer へ移す) */
			ellipse_list_delete_Undo(
					ellipse_search_ellipse(p->select->diagram_point.ellipse_point, &ellipse_list_info), 
					&ellipse_list_info);
		}
		p = p->next;
	}

	/*	全リストを削除 */
	select_list_all_free();
	StartData = NULL;

	/* メッセージ表示 */
	sprintf(str, _("It deleted a diagram.") );
	StatusMsg(str);
	UpDate();
}





/* -------------------------------------------------------
 * セレクト図形を解除する
 */
void SelectCancel(GtkWidget *widget)
{
	SELECT_LIST *p;
	char str[256];


	if (SelectChainFrag == 1 || SelectChainFrag == 2) {
		SelectChainFrag = 0;
		OP_SelectList = NULL;
	}
	if (Chain2_OnlyOne_mode == 211) {
		SelectChain2_OnlyOne (0);
		SelectChainFrag = 0;
		OP_SelectList = NULL;
	}
	if (SelectColorFrag == 1) {
		SelectColorFrag = 0;
		OP_SelectList = NULL;
	}

	if (outside_process_on == 1) {
		outside_process_on = 0;
	}


	/* ------------------------------------------------
	 * 選択されているデータの色を元の色に
	 * 表示する。
	 */
	p = select_list_info.head;
	while (p != NULL) {
		SelectDraw(drawing_area, p->select, SCD_HIDE, UPDATE_OFF);
		SelectDraw(drawing_area, p->select, SCD_ORG, UPDATE_OFF);
		p = p->next;
	}

	/* 寸法線の位置決め中の表示を消す */
	if (Dim_old.Text[0] != '\0') {
		DimensionDraw(drawing_area, &Dim_old, SCD_HIDE);
		Dim_old.Text[0] = '\0';
		
		/*  */
		Dim_old.diagram_of_char_index = 0;
		if (Dim_old.diagram_of_char != NULL) {
			xfree(Dim_old.diagram_of_char);
		}
		Dim_old.diagram_of_char = NULL;

		/*  */
		Dim_old.index = 0;
		if (Dim_old.AssistLine != NULL) {
			xfree(Dim_old.AssistLine);
		}
		Dim_old.AssistLine = NULL;
	}


	/* 全リストを削除 */
	select_list_all_free();
	UpDate();

	mode = 0;
	ReturnFunc = MouseRet_Non;
	/* メッセージ表示 */
	sprintf(str, _(" A command was canceled.") );
	StatusMsg(str);
}





/* -------------------------------------------------------
 * CAD データ p_cad と同じ色の図形をセレクトする
 */
int SelectMaskColor(CAD *p_cad)
{
	CAD_LIST *p;
	SELECT select;
	
	if (cad_list_info.head == NULL) return 0;
	p = cad_list_info.head;
	/* p から データの最後までを探す */
	while (p != NULL) {
		if (Layer[p->cad->layer].draw != 0) {
			if (p->cad->color == p_cad->color) {
				if (SelectCheck(p->cad) == 0) {
					select.index = 2;
					select.x = 0;
					select.y = 0;
					select.diagram_point.cad_point = p->cad;
					/* Select Data をリストの最後に追加 */
					select_list_add_last(&select);
					SelectCadDraw(drawing_area, p->cad, SCD_SELECT);
				}
			}
		}
		p = p->next;
	}
	SelectChainFrag = 0;
	SelectColorFrag = 0;
	(*ReturnFunc)(drawing_area, 1, 0, 0);
	return 1;
}





/* -------------------------------------------------------
 * CAD データ p_cad の図形につながている図形をセレクトする
 */
int SelectChain(CAD *p_cad)
{
	SELECT select;
	CAD_LIST *p;


#ifdef ONE_SHOT_DEBUG
	OneShotLog("Select.c : SelectChain() : in\n");
#endif


	p = cad_list_info.head;
	/* CAD DATA を最初から最後の1個前まで */
	while (p != NULL) {
		if (Layer[p->cad->layer].draw != 0 && p->cad->code != 0) {
			/* n点とn点 */
			if (	(p_cad->sx + (1/pow(10, compa_digits)) > p->cad->sx) 
					&& 
					(p_cad->sx - (1/pow(10, compa_digits)) < p->cad->sx) 
					&& 
					(p_cad->sy + (1/pow(10, compa_digits)) > p->cad->sy) 
					&& 
					(p_cad->sy - (1/pow(10, compa_digits)) < p->cad->sy) ) 
			{
//			if (sg(p->cad->sx, 3) == sg(p_cad->sx, 3) && sg(p->cad->sy, 3) == sg(p_cad->sy, 3)) {
				if (SelectCheck(p->cad) == 0) {
					select.index = 2;
					select.x = 0;
					select.y = 0;
					select.diagram_point.cad_point = p->cad;
					/* Select Data をリストの最後に追加 */
					select_list_add_last(&select);
					SelectCadDraw(drawing_area, p->cad, SCD_SELECT);
					SelectChain(p->cad);
				}
			}
			/* n点と終点 */
			else if (	(p_cad->ex + (1/pow(10, compa_digits)) > p->cad->sx) 
						&& 
						(p_cad->ex - (1/pow(10, compa_digits)) < p->cad->sx) 
						&& 
						(p_cad->ey + (1/pow(10, compa_digits)) > p->cad->sy) 
						&& 
						(p_cad->ey - (1/pow(10, compa_digits)) < p->cad->sy) ) 
			{
//			else if (sg(p->cad->sx, 3) == sg(p_cad->ex, 3) && sg(p->cad->sy, 3) == sg(p_cad->ey, 3)) {
				if (SelectCheck(p->cad) == 0) {
					select.index = 2;
					select.x = 0;
					select.y = 0;
					select.diagram_point.cad_point = p->cad;
					/* Select Data をリストの最後に追加 */
					select_list_add_last(&select);
					SelectCadDraw(drawing_area, p->cad, SCD_SELECT);
					SelectChain(p->cad);
				}
			}
			/* 終点とn点 */
			else if (	(p_cad->sx + (1/pow(10, compa_digits)) > p->cad->ex) 
						&& 
						(p_cad->sx - (1/pow(10, compa_digits)) < p->cad->ex) 
						&& 
						(p_cad->sy + (1/pow(10, compa_digits)) > p->cad->ey) 
						&& 
						(p_cad->sy - (1/pow(10, compa_digits)) < p->cad->ey) ) 
			{
//			else if (sg(p->cad->ex, 3) == sg(p_cad->sx, 3) && sg(p->cad->ey, 3) == sg(p_cad->sy, 3)) {
				if (SelectCheck(p->cad) == 0) {
					select.index = 2;
					select.x = 0;
					select.y = 0;
					select.diagram_point.cad_point = p->cad;
					/* Select Data をリストの最後に追加 */
					select_list_add_last(&select);
					SelectCadDraw(drawing_area, p->cad, SCD_SELECT);
					SelectChain(p->cad);
				}
			}
			/* 終点と終点 */
			else if (	(p_cad->ex + (1/pow(10, compa_digits)) > p->cad->ex) 
						&& 
						(p_cad->ex - (1/pow(10, compa_digits)) < p->cad->ex) 
						&& 
						(p_cad->ey + (1/pow(10, compa_digits)) > p->cad->ey) 
						&& 
						(p_cad->ey - (1/pow(10, compa_digits)) < p->cad->ey) ) 
			{
//			else if (sg(p->cad->ex, 3) == sg(p_cad->ex, 3) && sg(p->cad->ey, 3) == sg(p_cad->ey, 3)) {
				if (SelectCheck(p->cad) == 0) {
					select.index = 2;
					select.x = 0;
					select.y = 0;
					select.diagram_point.cad_point = p->cad;
					/* Select Data をリストの最後に追加 */
					select_list_add_last(&select);
					SelectCadDraw(drawing_area, p->cad, SCD_SELECT);
					SelectChain(p->cad);
				}
			}
		}
		p = p->next;
	}
	SelectChainFrag = 0;
	SelectColorFrag = 0;
	(*ReturnFunc)(drawing_area, 1, 0, 0);
	return 1;
}





/* -------------------------------------------------------
 * < 端点サーチ関数 > 寸法を書くときに寸法の点を取るための関数。
 * 図形要素の端点を取る
 * (点は点、線は始点OR終点、円弧は始点OR終点、
 *円は中心からＸ軸Ｙ軸と平行な線との交点と中心)
 *	
 * 引数 : int i;	図形データ№
 * 戻値 : int		0:セレクトされていない。	1:セレクトされている。
 */
int TanTenLook(SELECT *a)
{
	CAD_LIST *p;
	struct RtnDat RTN;


	p = cad_list_info.head;
	while (p != NULL) {
		/* 点データのとき */
		if(p->cad->code == 0) {
			if(Layer[p->cad->layer].draw != 0) {
				if (( (DcObj & 1) == 1 )) {
					RTN.sx[1] = p->cad->sx;
					RTN.sy[1] = p->cad->sy;
					RTN.ex[1] = a->x;
					RTN.ey[1] = a->y;
					pp(&RTN);
					if(RTN.l <= DC) {
						a->index = 1;
						a->diagram_point.cad_point = p->cad;
						a->x = p->cad->sx;
						a->y = p->cad->sy;
						return 1;
					}
				}
			}
		}
		
		
		/* 線データのとき */
		if(p->cad->code == 1) {
			if (Layer[p->cad->layer].draw != 0) {
				if (( (DcObj & 16) == 16 )) {
					RTN.sx[1] = p->cad->sx;
					RTN.sy[1] = p->cad->sy;
					RTN.ex[1] = a->x;
					RTN.ey[1] = a->y;
					pp(&RTN);
					if(RTN.l <= DC) {
						a->index = 1;
						a->diagram_point.cad_point = p->cad;
						a->x = p->cad->sx;
						a->y = p->cad->sy;
						return 1;
					}
					RTN.sx[1] = p->cad->ex;
					RTN.sy[1] = p->cad->ey;
					RTN.ex[1] = a->x;
					RTN.ey[1] = a->y;
					pp(&RTN);
					if(RTN.l <= DC) {
						a->index = 1;
						a->diagram_point.cad_point = p->cad;
						a->x = p->cad->ex;
						a->y = p->cad->ey;
						return 1;
					}
				}
			}
		}
		
		
		/* 円弧データのとき */
		if(p->cad->code == 2) {
			if (Layer[p->cad->layer].draw != 0) {
				if (( (DcObj & 16) == 16 )) {
					// 円弧の中心
					// RTN.sx[1] = p->cad->cx;
					// RTN.sy[1] = p->cad->cy;
					// RTN.ex[1] = a->x;
					// RTN.ey[1] = a->y;
					// pp(&RTN);
					// if(RTN.l <= DC)
					// {
					//	 a->diagram_point.cad_point = p->cad;
					//	 a->x = p->cad->cx;
					//	 a->y = p->cad->cy;
					//	 return 1;
					// }
					// 円弧のn点
					RTN.sx[1] = p->cad->sx;
					RTN.sy[1] = p->cad->sy;
					RTN.ex[1] = a->x;
					RTN.ey[1] = a->y;
					pp(&RTN);
					if(RTN.l <= DC) {
						a->index = 1;
						a->diagram_point.cad_point = p->cad;
						a->x = p->cad->sx;
						a->y = p->cad->sy;
						return 1;
					}
					// 円弧の終点
					RTN.sx[1] = p->cad->ex;
					RTN.sy[1] = p->cad->ey;
					RTN.ex[1] = a->x;
					RTN.ey[1] = a->y;
					pp(&RTN);
					if(RTN.l <= DC) {
						a->index = 1;
						a->diagram_point.cad_point = p->cad;
						a->x = p->cad->ex;
						a->y = p->cad->ey;
						return 1;
					}
				}
			}
		}
		
		
		/* 円データのとき */
		if (p->cad->code == 4 || p->cad->code == 2) {
			if (Layer[p->cad->layer].draw != 0) {
				if (( (DcObj & 64) == 64 )) {
					/* 円の中心 */
					RTN.sx[1] = p->cad->cx;
					RTN.sy[1] = p->cad->cy;
					RTN.ex[1] = a->x;
					RTN.ey[1] = a->y;
					pp(&RTN);
					if(RTN.l <= DC) {
						a->index = 1;
						a->diagram_point.cad_point = p->cad;
						a->x = p->cad->cx;
						a->y = p->cad->cy;
						return 1;
					}
				}
				
				
				if (( (DcObj & 32) == 32 )) {
					/* 円の 0 度点 */
					RTN.sx[1] = p->cad->cx + p->cad->r;
					RTN.sy[1] = p->cad->cy;
					RTN.ex[1] = a->x;
					RTN.ey[1] = a->y;
					pp(&RTN);
					if(RTN.l <= DC) {
//int ArcIn(CAD_LIST *p, double Angle)
						if((p->cad->code == 2) && (ArcIn(p, 0) == 1)) {
							a->index = 1;
							a->diagram_point.cad_point = p->cad;
							a->x = p->cad->cx + p->cad->r;
							a->y = p->cad->cy;
							return 1;
						}
						else if(p->cad->code == 4) {
							a->index = 1;
							a->diagram_point.cad_point = p->cad;
							a->x = p->cad->cx + p->cad->r;
							a->y = p->cad->cy;
							return 1;
						}
					}
					
					/* 円の 90 度点 */
					RTN.sx[1] = p->cad->cx;
					RTN.sy[1] = p->cad->cy + p->cad->r;
					RTN.ex[1] = a->x;
					RTN.ey[1] = a->y;
					pp(&RTN);
					if(RTN.l <= DC) {
						if((p->cad->code == 2) && (ArcIn(p, 90) == 1)) {
							a->index = 1;
							a->diagram_point.cad_point = p->cad;
							a->x = p->cad->cx;
							a->y = p->cad->cy + p->cad->r;
							return 1;
						}
						else if(p->cad->code == 4) {
							a->index = 1;
							a->diagram_point.cad_point = p->cad;
							a->x = p->cad->cx;
							a->y = p->cad->cy + p->cad->r;
							return 1;
						}
					}
					
					/* 円の 180 度点 */
					RTN.sx[1] = p->cad->cx - p->cad->r;
					RTN.sy[1] = p->cad->cy;
					RTN.ex[1] = a->x;
					RTN.ey[1] = a->y;
					pp(&RTN);
					if(RTN.l <= DC) {
						if((p->cad->code == 2) && (ArcIn(p, 180) == 1)) {
							a->index = 1;
							a->diagram_point.cad_point = p->cad;
							a->x = p->cad->cx - p->cad->r;
							a->y = p->cad->cy;
							return 1;
						}
						else if(p->cad->code == 4) {
							a->index = 1;
							a->diagram_point.cad_point = p->cad;
							a->x = p->cad->cx - p->cad->r;
							a->y = p->cad->cy;
							return 1;
						}
					}
					
					/* 円の 270 度点 */
					RTN.sx[1] = p->cad->cx;
					RTN.sy[1] = p->cad->cy - p->cad->r;
					RTN.ex[1] = a->x;
					RTN.ey[1] = a->y;
					pp(&RTN);
					if(RTN.l <= DC) {
						if((p->cad->code == 2) && (ArcIn(p, 270) == 1)) {
							a->index = 1;
							a->diagram_point.cad_point = p->cad;
							a->x = p->cad->cx;
							a->y = p->cad->cy - p->cad->r;
							return 1;
						}
						else if(p->cad->code == 4) {
							a->index = 1;
							a->diagram_point.cad_point = p->cad;
							a->x = p->cad->cx;
							a->y = p->cad->cy - p->cad->r;
							return 1;
						}
					}
				}
			}
		}
		p = p->next;
	}
	a->index = 0;
	a->diagram_point.cad_point = NULL;
	a->x = 0;
	a->y = 0;
	return 0;
}





/* -------------------------------------------------------
 * 寸法サーチ関数	削除や編集のときに寸法を探すための関数。
 *	
 *	
 * マウスから読み謔ﾁた座標 (x , y) から、その点に近い寸法を探す。
 * (DC の値よりも近い値の寸法)
 *	
 * 戻値 : int 寸法データナンバーを返す。
 *	
 */
DIMENSION *SnpLook(double x, double y)
{
//define SNPLOOK_TEST
#ifdef SNPLOOK_TEST
char str[256];
#endif

	DIMENSION_LIST *p;
	struct RtnDat PPH;


	p = dimension_list_info.head;
	while (p != NULL && p->dimension != NULL) {
		/* 表示レイヤーのみ描く */
		if (Layer[p->dimension->Layer].draw != 0) {
			PPH.sx[1] = p->dimension->SearchPointX;
			PPH.sy[1] = p->dimension->SearchPointY;
			PPH.ex[1] = x;
			PPH.ey[1] = y;
			pp(&PPH);
			if (PPH.l <= (2 * DC) ) {
				if (SelectDimensionCheck(p->dimension) == 0)
					return p->dimension;
			}
		}
		p = p->next;
	}
	return NULL;
}





/* -------------------------------------------------------
 * Dimension Data がセレクトされているかを確認
 *	
 * 引数 ： DIMENSION *data
 * 戻値 ： int	0：選択されていない 	1：選択済み
 */
int SelectDimensionCheck(DIMENSION *data)
{
	SELECT_LIST *p;

	p = select_list_info.head;
	while (p != NULL) {
		if (p->select->diagram_point.dimension_point == data) 
			return 1;
		p = p->next;
	}
	return 0;
}





/* -------------------------------------------------------------------
 * チェイン２セレクト
 * 
 * (Frag == 0) start Data の方向を決める
 * (Frag == 2) start Data の方向はそのままで探す
 * (Frag == 4) start Data の方向を変えて探す
 */
int SelectChain2(CAD *start, int frag)
{
//#define SELECTCHAIN2
	int debug = 0;


#ifdef SELECTCHAIN2
	debug = 1;
	if (debug > 0) g_print("Select.c : SelectChain2() : in\n");
#endif


	/* start Data の方向を決める */
	if (frag == 0) {
		if (debug > 0) g_print("Select.c : SelectChain2() : (Frag == 0) start Data の方向を決める !\n");

		Chain2[1].diagram_point.cad_point = start;
		Chain2[1].index = 2;
		Chain2[2].diagram_point.cad_point = start;
		Chain2[2].index = 4;
		Chain2Data = 2;
		Chain2index = 1;
		Chain2_OnlyOne_mode = 211;
//		KeepReturnFunc = ReturnFunc;
//		ReturnFunc = MouseRet_ChainOne;
//		(*ReturnFunc)(drawing_area, 1, 0, 0);
		SelectChain2_OnlyOne(3);
	}

	/* start Data の方向はそのままで探す */
	else if (frag == 2) {
		if (debug > 0) g_print("Select.c : SelectChain2() : (Frag == 2) start Data の方向はそのままで探す !\n");
		SelectChain2_Search(start->ex, start->ey);
	}

	/* start Data の方向を変えて探す */
	else if (frag == 4) {
		if (debug > 0) g_print("Select.c : SelectChain2() : (Frag == 4) start Data の方向を変えて探す !\n");
		SelectChain2_Search(start->sx, start->sy);
	}

	if (debug > 0) g_print("Select.c : SelectChain2() : out\n");
	return 1;
}





/* -------------------------------------------------------------------
 * 点(x,y) を端点に持つ図形を複数探し、
 * SelectChain2_OnlyOne で一つをセレクトする
 *	
 */
int SelectChain2_Search(double x, double y)
{
//#define DEBUG_SELECTCHAIN2_SEARCH
	CAD_LIST *p;
	int debug = 0;

	Chain2Data = 0;


#ifdef DEBUG_SELECTCHAIN2_SEARCH
	debug = 1;
	if (debug > 0) g_print("Select.c : SelectChain2_Search() : in\n");
#endif


	if (debug > 0) g_print("探点(%f , %f)  ORGN(%f, %f)\n", sg(x, 3), sg(y, 3), x, y);


	/* 各図形要素 */
	p = cad_list_info.head;
	/* -------------------------------------------
	 * 最後まで見つからない場合、 p に NULL が 
	 * 入ってループを抜ける。
	 */
	while(p != NULL) {
		/* 表示されていて、点でない */
		if (Layer[p->cad->layer].draw != 0 && p->cad->code != 0 && p->cad->code != 4) {


			if (debug > 0) g_print("始点(%f , %f)  終点(%f , %f)\n", 
						   sg(p->cad->sx, 3), sg(p->cad->sy, 3), sg(p->cad->ex, 3), sg(p->cad->ey, 3));
			if (debug > 0) g_print("SG 判定(始点) = %d    判定(終点) = %d\n", 
				(sg(x, 2) == sg(p->cad->sx, 2)) && (sg(y, 2) == sg(p->cad->sy, 2)), 
				(sg(x, 2) == sg(p->cad->ex, 2) && sg(y, 2) == sg(p->cad->ey, 2)) );
			if (debug > 0) g_print("dig判定(始点) = %d    判定(終点) = %d\n", 
					( (x + (1/pow(10, compa_digits)) > p->cad->sx) && (x - (1/pow(10, compa_digits)) < p->cad->sx) && (y + (1/pow(10, compa_digits)) > p->cad->sy) &&  (y - (1/pow(10, compa_digits)) < p->cad->sy) ), 
				( (x + (1/pow(10, compa_digits)) > p->cad->ex) && (x - (1/pow(10, compa_digits)) < p->cad->ex) && (y + (1/pow(10, compa_digits)) > p->cad->ey) &&  (y - (1/pow(10, compa_digits)) < p->cad->ey) ) 
				);
			if (debug > 0) g_print("dig = %f\n", (1/pow(10, compa_digits)));


			/* 始点と一致 */
//			if (	(sg(x, compa_digits) == sg(p->cad->sx, compa_digits)) 
//					&& 
//					(sg(y, compa_digits) == sg(p->cad->sy, compa_digits)) ) { 
			if (	(x + (1/pow(10, compa_digits)) > p->cad->sx) 
					&& 
					(x - (1/pow(10, compa_digits)) < p->cad->sx) 
					&& 
					(y + (1/pow(10, compa_digits)) > p->cad->sy) 
					&&	
					(y - (1/pow(10, compa_digits)) < p->cad->sy) ) 
			{
				/* データ№ i の図形がセレクトされていないなら */
				if (SelectCheck(p->cad) == 0) {
					if (debug > 0) g_print("--- << 始点と一致 >> ---\n");
					Chain2Data++;
					Chain2[Chain2Data].index = 2;
					Chain2[Chain2Data].diagram_point.cad_point = p->cad;
					Chain2[Chain2Data].x = 0;
					Chain2[Chain2Data].y = 0;
				}
				else {
					if (debug > 0) g_print("既に図形がセレクトされている(始点)\n");
				}
			}

			/* 終点と一致 */
//			else if (	(sg(x, compa_digits) == sg(p->cad->ex, compa_digits)) 
//						&& 
//						(sg(y, compa_digits) == sg(p->cad->ey, compa_digits)) ) {
			else if (	(x + (1/pow(10, compa_digits)) > p->cad->ex) 
						&& 
						(x - (1/pow(10, compa_digits)) < p->cad->ex) 
						&& 
						(y + (1/pow(10, compa_digits)) > p->cad->ey) 
						&&	
						(y - (1/pow(10, compa_digits)) < p->cad->ey) ) 
			{
				/* データ№ i の図形がセレクトされていないなら */
				if (SelectCheck(p->cad) == 0) {
					if (debug > 0) g_print("--- << 終点と一致 >> ---\n");
					Chain2Data++;
					Chain2[Chain2Data].index = 4;
					Chain2[Chain2Data].diagram_point.cad_point = p->cad;
					Chain2[Chain2Data].x = 0;
					Chain2[Chain2Data].y = 0;
				}
				else {
					if (debug > 0) g_print("既に図形がセレクトされている(終点)\n");
				}
			}
		}
		p = p->next;
	}

if (debug > 0) g_print("Select.c : SelectChain2_Search() : 1\n");

	/* 続く図形が見つからないとき */
	if (Chain2Data == 0) {
if (debug > 0) g_print("Select.c : SelectChain2_Search() : 2\n");
		Chain2_OnlyOne_mode = 0;
		if (debug > 0) g_print("Select.c : SelectChain2_Search() : 続く図形が見つからない : チェイン２終了\n");
		
		MsgBox("SagCAD", _(" It ends chain 2.") , " ", NULL, NULL, "OK");
		ReturnFunc = KeepReturnFunc;
		(*ReturnFunc)(drawing_area, 1, 0, 0);
		return 0;
	}

	/* 続く図形が１個だけのとき */
	else if (Chain2Data == 1) {
if (debug > 0) g_print("Select.c : SelectChain2_Search() : 2-2\n");
		if (debug > 0) g_print("Select.c : SelectChain2_Search() : 続く図形が１個だけ\n");
		select_list_add_last( &Chain2[1] );
		SelectCadDraw(drawing_area, select_list_info.tail->select->diagram_point.cad_point, SCD_SELECT);

		if (Chain2[1].index == 2) 
			SelectChain2(select_list_info.tail->select->diagram_point.cad_point, 2);
		else if (Chain2[1].index == 4) 
			SelectChain2(select_list_info.tail->select->diagram_point.cad_point, 4);
	}

	/* 続く図形が２個以上あるとき */
	else if (Chain2Data > 1) {
if (debug > 0) g_print("Select.c : SelectChain2_Search() : 2-3\n");
		if (debug > 0) g_print("Select.c : SelectChain2_Search() : 続く図形が２個以上ある\n");
		Chain2index = 1;
		Chain2_OnlyOne_mode = 211;
		SelectChain2_OnlyOne(3);
	}

	if (debug > 0) g_print("Select.c : SelectChain2_Search() : out\n");
	return 1;
}





/* -------------------------------------------------------------------
 * チェイン２選択モード
 * (エントリーされている図形の中から目的の図形を選ぶ)
 *	
 * 引数 : int Frag
 *	  0:CANCEL	  1:YES    2:NO
 *	  3:Frag = 2 と同じだけど前のインデックスを消さない。（初期用）
 *	  5:方向選択
 *	
 * 外部 : 
 *	  Chain2		: セレクトデータ形式データ
 *	  Chain2index	: インデックス
 *	  Chain2Data	: 選択データ数
 *	
 *	
 */
void SelectChain2_OnlyOne(int Frag)
{
//#define SELECTCHAIN2_ONLYONE
	int Old_Draw;
	char str[256];
	int debug = 0;


#ifdef SELECTCHAIN2_ONLYONE
	debug = 1;
	if (debug > 0) g_print("Select.c : SelectChain2_OnlyOne() : in\n");
#endif

/*
	if (Frag == 5) {
		Chain2[1].cad = select_list_info.tail->select->diagram_point.cad_point;
		Chain2[1].index = 1;
		Chain2[2].cad = select_list_info.tail->select->diagram_point.cad_point;
		Chain2[2].index = 2;
		Chain2Data = 2;
		Chain2index = 1;
		Chain2_OnlyOne_mode = 211;
		SelectChain2_OnlyOne(3);
		return;
	}
*/

	sprintf(str, _(" Does it go to here? [Yes:(Left/RET)] [No:(Right/SPC)] [Cancel:(Center/ESC)]") );
	StatusMsg(str);


	/* 違う (NO) 次の図形へ */
	if (Frag == 2) {
		if (Chain2index == Chain2Data) Chain2index = 1;
		else	Chain2index++;
	}
	/* 前のデータ */
	if (Chain2index == 1) Old_Draw = Chain2Data;
	else Old_Draw = Chain2index - 1;


	/* 図形を描画 */
	if (Frag == 2) {
		SelectChain2_Draw(*Chain2[Old_Draw].diagram_point.cad_point, Chain2[Old_Draw].index, SCD_HIDE);
		SelectCadDraw(drawing_area, Chain2[Old_Draw].diagram_point.cad_point, SCD_ORG);
	}
	if (Frag == 2 || Frag == 3) 
		SelectChain2_Draw(*Chain2[Chain2index].diagram_point.cad_point, Chain2[Chain2index].index, 3);


	/* 決定 (OK) */
	if (Frag == 1) {
		if (select_list_info.tail->select->diagram_point.cad_point == Chain2[Chain2index].diagram_point.cad_point) {
			//MsgBox("SagCAD", "セレクト済みデータ", "SelectChain2_OnlyOne()", NULL, NULL, "OK");
			select_list_info.tail->select->index = Chain2[Chain2index].index;
		}
		else {
			//MsgBox("SagCAD", "セレクトリストに追加", "SelectChain2_OnlyOne()", NULL, NULL, "OK");
			/* Select Data をリストの最後に追加 */
			select_list_add_last( &Chain2[Chain2index] );
		}

		SelectChain2_Draw(*select_list_info.tail->select->diagram_point.cad_point, select_list_info.tail->select->index, SCD_HIDE);
		SelectCadDraw(drawing_area, select_list_info.tail->select->diagram_point.cad_point, SCD_SELECT);
		Chain2_OnlyOne_mode = 0;
		Chain2Data = 0;
		Chain2index = 0;
		SelectChain2(select_list_info.tail->select->diagram_point.cad_point, select_list_info.tail->select->index);
		if (debug > 0) g_print("Select.c : SelectChain2_OnlyOne() : 決定 (frag = 1)\n");
		if (debug > 0) g_print("Select.c : SelectChain2_OnlyOne() : out 1\n");
		return;
	}

	/* キャンセル (CANCEL) */
	else if (Frag == 0) {
		SelectChain2_Draw(*Chain2[Chain2index].diagram_point.cad_point, Chain2[Chain2index].index, SCD_HIDE);
		SelectCadDraw(drawing_area, Chain2[Chain2index].diagram_point.cad_point, SCD_ORG);
		Chain2_OnlyOne_mode = 0;
		Chain2Data = 0;
		Chain2index = 0;
		Redraw();
		if (debug > 0) g_print("Select.c : SelectChain2_OnlyOne() : キャンセル (frag = 0)\n");
//		MessageBox(NULL, "Cancel しました", "SelectChain2_OnlyOne", MB_OK);
		ReturnFunc = KeepReturnFunc;
		(*ReturnFunc)(drawing_area, 1, 0, 0);
		if (debug > 0) g_print("Select.c : SelectChain2_OnlyOne() : out 2\n");
		return;
	}
}





/* -------------------------------------------------------------------
 * データ№ i の図形を表示
 *	
 * 図形描画 	 DrawType
 * 選択時		: DrawType = 1	SCD_SELECT
 * 元の色		: DrawType = 0	SCD_ORG
 * 消　去		: DrawType = 2	SCD_HIDE
 * オプション	: DrawType = 3	SCD_CAM
 *	
 */
void SelectChain2_Draw(CAD cad, int frag, int DrawType)
{
//#define SELECTCHAIN2_DRAW
	long SelectCadDrawColor = sagcad_color.Cam;
	int debug = 0;


#ifdef SELECTCHAIN2_DRAW
	debug = 1;
	g_print ("Select.c : SelectChain2_Draw() : in\n");
#endif


	if ( DrawType == 1) SelectCadDrawColor = sagcad_color.Select;
	else if ( DrawType == 0)	SelectCadDrawColor = cad.color;
	else if ( DrawType == 2)	SelectCadDrawColor = sagcad_color.Back;
	else if ( DrawType == 3)	SelectCadDrawColor = sagcad_color.Cam;

	if ( Layer[cad.layer].draw != 0) {
		/* 線のとき */
		if (cad.code == 1) {
			if (DrawType ==0 || DrawType == 1) {
				if (frag == 2) {
					SelectChain2_S_Bow(cad.sx, cad.sy, cad.ex, cad.ey, sagcad_color.Back);
					SelectCadDraw(drawing_area, &cad, DrawType);
				}
				else if (frag == 4) {
					SelectChain2_S_Bow(cad.ex, cad.ey, cad.sx, cad.sy, sagcad_color.Back);
					SelectCadDraw(drawing_area, &cad, DrawType);
				}
			}
			else {
				if (frag == 2) {
					SelectChain2_S_Bow(cad.sx, cad.sy, cad.ex, cad.ey, sagcad_color.Back);
					SelectChain2_S_Bow(cad.sx, cad.sy, cad.ex, cad.ey, SelectCadDrawColor);
				}
				else if (frag == 4) {
					SelectChain2_S_Bow(cad.ex, cad.ey, cad.sx, cad.sy, sagcad_color.Back);
					SelectChain2_S_Bow(cad.ex, cad.ey, cad.sx, cad.sy, SelectCadDrawColor);
				}
			}
		}
		/* 円弧のとき */
		else if ( cad.code == 2) {
			if (DrawType ==0 || DrawType == 1) {
				SelectChain2_ArcBow(cad.cx, cad.cy, cad.r, cad.sx, cad.sy, cad.ex, cad.ey, sagcad_color.Back, frag);
				SelectCadDraw(drawing_area, &cad, DrawType);
			}
			else {
				SelectChain2_ArcBow(cad.cx, cad.cy, cad.r, cad.sx, cad.sy, cad.ex, cad.ey, sagcad_color.Back, frag);
				SelectChain2_ArcBow(cad.cx, cad.cy, cad.r, cad.sx, cad.sy, cad.ex, cad.ey, SelectCadDrawColor, frag);
			}
		}
	}
	UpDate();
	if (debug > 0) g_print ("Select.c : SelectChain2_Draw() : out\n");
}





/* -------------------------------------------------------------------
 * 終点に矢印
 *	
 */
int SelectChain2_S_Bow(double SX, double SY, double EX, double EY, long color)
{
	struct RtnDat LAH, PAPH;
	double EX1, EY1, EX2, EY2;


	/* ------------------------------------------------ */
	/* LA	直線の角度 */
	/*		 (A.sx[1],A.SY[1]),(A.ex[1],A.EY[1]) */
	/*				 Ans A.ANGLE */
	/* ------------------------------------------------ */
	LAH.sx[1] = EX;
	LAH.sy[1] = EY;
	LAH.ex[1] = SX;
	LAH.ey[1] = SY;
	la(&LAH);
	/* ------------------------------------------------ */
	/* PAP	始点と角度と距離で直線の終点を求める */
	/*		 (A.sx[1] , A.sy[1]) , A.ANGLE , A.l */
	/*				 Ans  (A.ex[1] , A.EY[1]) */
	/* ------------------------------------------------ */
	PAPH.sx[1] = EX;
	PAPH.sy[1] = EY;
	PAPH.angle = LAH.angle + sagcad_dimension.arrow_angle;
	PAPH.l = 3;//ArrowLength(DRAW_CONST) * 2; /* DRAW_DISP */
	pap(&PAPH);

	EX1 = PAPH.ex[1];
	EY1 = PAPH.ey[1];

	PAPH.sx[1] = EX;
	PAPH.sy[1] = EY;
	PAPH.angle = LAH.angle - sagcad_dimension.arrow_angle;
	PAPH.l = 3;//ArrowLength(DRAW_CONST) * 2;
	pap(&PAPH);
	EX2 = PAPH.ex[1];
	EY2 = PAPH.ey[1];

	LineDraw(drawing_area, SX, SY, EX, EY, 1, color);
	LineDraw(drawing_area, EX, EY, EX1, EY1, 1, color);
	LineDraw(drawing_area, EX, EY, EX2, EY2, 1, color);

	return 0;
}





/* -------------------------------------------------------------------
 * 矢印(円弧)
 *	
 * frag : 4 : 終点に矢印
 *		: 2 : 始点に矢印
 */
int SelectChain2_ArcBow(double CX, double CY, double R, double SX, double SY, double EX, double EY, long color, int frag)
{
	/* Dumy = ArcBow(CX, CY, R, SA,EA, INDEX) */

	struct RtnDat PAPH, LAH;
	double SA, EA;
	double SX1, SY1, SX2, SY2;
	double EX1, EY1, EX2, EY2;


	if (frag == 4) {
		/* 中心点から始点の線の角度 SA を求める。 */
		/* LA	  直線の角度 */
		LAH.sx[1] = CX;
		LAH.sy[1] = CY;
		LAH.ex[1] = SX;
		LAH.ey[1] = SY;
		la(&LAH);
		SA = LAH.angle;
		if(SA > 360) SA = SA - 360;

		PAPH.sx[1] = SX;
		PAPH.sy[1] = SY;
		PAPH.angle = SA + 90 - sagcad_dimension.arrow_angle;
		PAPH.l = 3;//ArrowLength(DRAW_CONST) * 2;
		pap(&PAPH);
		SX1 = PAPH.ex[1];
		SY1 = PAPH.ey[1];

		PAPH.sx[1] = SX;
		PAPH.sy[1] = SY;
		PAPH.angle = SA + 90 + sagcad_dimension.arrow_angle;
		PAPH.l = 3;//ArrowLength(DRAW_CONST) * 2;
		pap(&PAPH);
		SX2 = PAPH.ex[1];
		SY2 = PAPH.ey[1];

		LineDraw(drawing_area, SX, SY, SX1, SY1, 1, color);
		LineDraw(drawing_area, SX, SY, SX2, SY2, 1, color);
	}


	if (frag == 2) {
		/* 中心点から終点の線の角度 EA を求める。 */
		/* LA	  直線の角度 */
		LAH.sx[1] = CX;
		LAH.sy[1] = CY;
		LAH.ex[1] = EX;
		LAH.ey[1] = EY;
		la(&LAH);
		EA = LAH.angle;
		if(EA > 360) EA = EA - 360;

		PAPH.sx[1] = EX;
		PAPH.sy[1] = EY;
		PAPH.angle = EA - 90 - sagcad_dimension.arrow_angle;
		PAPH.l = 3;//ArrowLength(DRAW_CONST) * 2;
		pap(&PAPH);
		EX1 = PAPH.ex[1];
		EY1 = PAPH.ey[1];

		PAPH.sx[1] = EX;
		PAPH.sy[1] = EY;
		PAPH.angle = EA - 90 + sagcad_dimension.arrow_angle;
		PAPH.l = 3;//ArrowLength(DRAW_CONST) * 2;
		pap(&PAPH);
		EX2 = PAPH.ex[1];
		EY2 = PAPH.ey[1];

		LineDraw(drawing_area, EX, EY, EX1, EY1, 1, color);
		LineDraw(drawing_area, EX, EY, EX2, EY2, 1, color);
	}

	ArcDraw(drawing_area, CX, CY, R, SX, SY, EX, EY, 1, color);

	return 0;
}





/* -------------------------------------------------------
 * 線を探す (sub)
 * 0 : なかった
 * 1 : あった
 */
int LineDataCheck(double sx, double sy, double ex, double ey, double x, double y)
{
//#define LINE_DATA_CHECK
	struct RtnDat PLPH, PPH, LAH;
	double L0, L1, A1, L2, A2;
	int debug = 0;


#ifdef LINE_DATA_CHECK
	debug = 1;
#endif


	if (debug > 0) {
		g_print("Select.c : LineDataCheck() : in\n");
		g_print("LINE (%f,%f)-(%f,%f) PicPoint(%f,%f)\n", sx, sy, ex, ey, x, y);
	}


	/* 点と線の最近点を求める */
	PLPH.sx[2] = x;
	PLPH.sy[2] = y;
	PLPH.sx[1] = sx;
	PLPH.sy[1] = sy;
	PLPH.ex[1] = ex;
	PLPH.ey[1] = ey;
	plp(&PLPH);
	/* 最近点	 PLPH.ex[2] , PLPH.ey[2] */
	
	/* 点と最近点の距離を求める */
	PPH.sx[1] = x;
	PPH.sy[1] = y;
	PPH.ex[1] = PLPH.ex[2];
	PPH.ey[1] = PLPH.ey[2];
	pp(&PPH);
	/* 距離 PPH.L */
	L0 = PPH.l;
	
	/* 距離が DC 以上なら探している線ではない。*/
	if (L0 > DC) {
		if (debug > 0) g_print("Select.c : LineDataCheck() : 離れている [L0(%f) > DC(%f)]\n", L0, DC);
		if (debug > 0) g_print("Select.c : LineDataCheck() : out\n");
		return 0;
	}
	
	
	/* 線のn点から終点の角度と距離を求める */
	PPH.sx[1] = sx;
	PPH.sy[1] = sy;
	PPH.ex[1] = ex;
	PPH.ey[1] = ey;
	pp(&PPH);
	/* 距離 PPH.L */
	L1 = PPH.l;
	
	LAH.sx[1] = sx;
	LAH.sy[1] = sy;
	LAH.ex[1] = ex;
	LAH.ey[1] = ey;
	la(&LAH);
	/* 角度 LAH.ANGLE */
	A1 = LAH.angle;
	
	/* 線のn点から最近点の角度と距離を求める */
	PPH.sx[1] = sx;
	PPH.sy[1] = sy;
	PPH.ex[1] = PLPH.ex[2];
	PPH.ey[1] = PLPH.ey[2];
	pp(&PPH);
	/* 距離 PPH.L */
	L2 = PPH.l;

	LAH.sx[1] = sx;
	LAH.sy[1] = sy;
	LAH.ex[1] = PLPH.ex[2];
	LAH.ey[1] = PLPH.ey[2];
	la(&LAH);
	/* 角度 LAH.ANGLE */
	A2 = LAH.angle;
	
	/* n点からの角度は同じで、n点から最近点までの長さよりも線の長さのが長い(L1>L2)*/
	if ( (A1 + 1 > A2) && (A1 - 1 < A2) ) {
		if (L1 > L2) {
			if (debug > 0) g_print("Select.c : LineDataCheck() : あった\n");
			if (debug > 0) g_print("Select.c : LineDataCheck() : out\n");
			return 1;
		}
	}
	else {
		if (debug > 0) g_print("Select.c : LineDataCheck() : 離れていないけど、範囲外\n");
		if (debug > 0) g_print("Select.c : LineDataCheck() : out\n");
		return 0;
	}

	if (debug > 0) g_print("Select.c : LineDataCheck() : out\n");
	return 0;
}





/* -------------------------------------------------------
 * 折れ線を探索
 *	
 *	
 */
int normal_polyline_search(POLYLINE *polyline, double px, double py)
{
//#define NORMAL_POLYLINE_SEARCH
	int i, Ret = 0;
	double old_x = 0, old_y = 0;
	VERTEX_LIST *pv;
	int debug = 0;


#ifdef NORMAL_POLYLINE_SEARCH
	debug = 1;
	g_print("Select.c : normal_polyline_search : in\n");
#endif



	pv = polyline->vertex_list_info.head;
	if (pv == NULL) {
		g_print("Vertex is No Data\n");
		return 0;
	}

	if (pv != NULL) {
		i = 0;
		while (pv != NULL) {
			if (i == 0) {
				// 
			}
			else {
				if (debug > 0) g_print("Select.c : normal_polyline_search : Search %d 番目\n", i);
				Ret = LineDataCheck(old_x, old_y, pv->vertex->x, pv->vertex->y, px, py);
				if (Ret != 0) {
					return Ret;
				}
				if (debug > 0) g_print("Select.c : normal_polyline_search : なかった\n");
			}
			old_x = pv->vertex->x;
			old_y = pv->vertex->y;

			i++;
			pv = pv->next;
		}


		if ((polyline->frag & 1) == 1) {
			if (debug > 0) g_print("Select.c : normal_polyline_search : Search close last\n");
			Ret = LineDataCheck(old_x, old_y, 
								polyline->vertex_list_info.head->vertex->x, polyline->vertex_list_info.head->vertex->y, 
								px, py);
			if (Ret != 0) {
				return Ret;
			}
			if (debug > 0) g_print("Select.c : normal_polyline_search : なかった\n");
		}
	}

	return Ret;
}





/* -------------------------------------------------------
 * 開スプライン曲線を探索
 *	
 *	
 */
int open_spline_search(POLYLINE *polyline, double px, double py)
{
	int i;
	double pitch;
	static double old_x = 0, old_y = 0;

	double u, v;
	double *x, *y, *p, *a, *b;
	double *h, *d;

	int n, split, Ret = 0;
	VERTEX_LIST *pv;


//OneShotLog("Select.c : open_spline_search() : in\n");


	n = (int)vertex_list_num(&polyline->vertex_list_info);
	split = polyline->split * n;

//	vertex = polyline->vertex;

	x = (double *) xmalloc( n * sizeof(double) );
	y = (double *) xmalloc( n * sizeof(double) );
	p = (double *) xmalloc( n * sizeof(double) );
	a = (double *) xmalloc( n * sizeof(double) );
	b = (double *) xmalloc( n * sizeof(double) );

	h = (double *) xmalloc( n * sizeof(double) );
	d = (double *) xmalloc( n * sizeof(double) );


	pv = polyline->vertex_list_info.head;
	i = 0;
	while (pv != NULL) {
		x[i] = pv->vertex->x;
		y[i] = pv->vertex->y;
		p[i] = 0;
		a[i] = 0;
		b[i] = 0;

		h[i] = 0;
		d[i] = 0;

		i++;
		pv = pv->next;
	}


	open_maketable2(n, p, x, y, a, b, h, d);

	pitch = 1.0/(double)split;
	for (i = 0; i <= split ; i++) {
		open_spline2(n, pitch * (double)i, &u, &v, p, x, y, a, b);
		if (i == 0) {
//OneShotLog("Select.c : open_spline_search() : i = 0 でパス\n");
			// pset
		}
		else {
			Ret = LineDataCheck(old_x, old_y, u, v, px, py);
			if (Ret != 0) {
				xfree(x);
				xfree(y);
				xfree(p);
				xfree(a);
				xfree(b);

				xfree(h);
				xfree(d);
//OneShotLog("Select.c : open_spline_search() : あった\n");
//OneShotLog("Select.c : open_spline_search() : out\n");
				return Ret;
			}
			//OneShotLog("Select.c : open_spline_search() : ない\n");
			//LineDraw(drawing_area, old_x, old_y, u, v, 1, 0xffffff);
		}
		old_x = u;
		old_y = v;
	}


	xfree(x);
	xfree(y);
	xfree(p);
	xfree(a);
	xfree(b);

	xfree(h);
	xfree(d);

//OneShotLog("Select.c : open_spline_search() : out\n");
	return Ret;
}





/* -------------------------------------------------------
 * 閉スプライン曲線を探索
 * 
 * 始点と同じ vertex のデータを最後に付け足す
 */
int close_spline_search(POLYLINE *polyline, double px, double py)
{
	int i;
	double pitch;
	static double old_x = 0, old_y = 0;

	double u, v;
	double *x, *y, *p, *a, *b;
	double *h, *d, *w;

	int n, split, Ret = 0;
	VERTEX_LIST *pv;

//OneShotLog("Select.c : close_spline_search() : in\n");

	n = (int)vertex_list_num(&polyline->vertex_list_info);
	split = polyline->split * n;


	x = (double *) xmalloc( (n+1) * sizeof(double) );
	y = (double *) xmalloc( (n+1) * sizeof(double) );
	p = (double *) xmalloc( (n+1) * sizeof(double) );
	a = (double *) xmalloc( (n+1) * sizeof(double) );
	b = (double *) xmalloc( (n+1) * sizeof(double) );

	h = (double *) xmalloc( (n+1) * sizeof(double) );
	d = (double *) xmalloc( (n+1) * sizeof(double) );
	w = (double *) xmalloc( (n+1) * sizeof(double) );

	pv = polyline->vertex_list_info.head;
	i = 0;
	while (pv != NULL) {
		x[i] = pv->vertex->x;
		y[i] = pv->vertex->y;
		p[i] = 0;
		a[i] = 0;
		b[i] = 0;

		h[i] = 0;
		d[i] = 0;
		w[i] = 0;

		i++;
		pv = pv->next;
	}
	x[i] = polyline->vertex_list_info.head->vertex->x;
	y[i] = polyline->vertex_list_info.head->vertex->y;
	p[i] = 0;
	a[i] = 0;
	b[i] = 0;

	h[i] = 0;
	d[i] = 0;
	w[i] = 0;




	close_maketable2(n, p, x, y, a, b, h, d, w);
	pitch = 1.0/(double)split;
	for (i = 0; i <= split ; i++) {
		close_spline2(n, pitch * i, &u, &v, p, x, y, a, b);
		if (i == 0) {
			// pset
		}
		else {
			Ret = LineDataCheck(old_x, old_y, u, v, px, py);
			if (Ret != 0) {
				xfree(x);
				xfree(y);
				xfree(p);
				xfree(a);
				xfree(b);

				xfree(h);
				xfree(d);
				xfree(w);
				return Ret;
			}
			//LineDraw(drawing_area, old_x, old_y, u, v, 1, 0xffffff);
		}
		old_x = u;
		old_y = v;
	}


	xfree(x);
	xfree(y);
	xfree(p);
	xfree(a);
	xfree(b);

	xfree(h);
	xfree(d);
	xfree(w);

	return Ret;
}





/* -------------------------------------------------------
 * Ｂスプライン曲線を探索
 *	
 *	
 */
int b_spline_search(POLYLINE *polyline, double px, double py)
{
	int i, j;
	double u, s1, s2, s3, s4;
	double x, y;
	static double old_x = 0, old_y = 0;
	VERTEX *dummy_vertex = NULL;
	VERTEX *vertex;
	VERTEX_LIST *pv;

	VERTEX sp = {0,0};
	VERTEX ep = {0,0};
	VERTEX ap1 = {0,0};
	VERTEX ap2 = {0,0};

	int n, Ret = 0;
	double pitch;


	pitch = polyline->pitch;

	n = (int)vertex_list_num(&polyline->vertex_list_info);
	vertex = (VERTEX *) xmalloc( (n + 1) * sizeof(VERTEX) );

	pv = polyline->vertex_list_info.head;
	i = 0;
	while (pv != NULL) {
		vertex[i] = *pv->vertex;
		i++;
		pv = pv->next;
	}


	if (n == 3) {
		dummy_vertex = (VERTEX *)xmalloc( 7 * sizeof(VERTEX) );
		j = 0;
		dummy_vertex[j].x = vertex[0].x;
		dummy_vertex[j].y = vertex[0].y;
		for (i = 1 ; i < n ; i++) {
			sp.x = vertex[i-1].x;
			sp.y = vertex[i-1].y;
			ep.x = vertex[i].x;
			ep.y = vertex[i].y;
			split3(sp, ep, &ap1, &ap2);
			j++;
			dummy_vertex[j].x = ap1.x;
			dummy_vertex[j].y = ap1.y;
			j++;
			dummy_vertex[j].x = ap2.x;
			dummy_vertex[j].y = ap2.y;
			j++;
			dummy_vertex[j].x = vertex[i].x;
			dummy_vertex[j].y = vertex[i].y;
		}
		n = 7;
	}
	else if (n > 3 && n < 7) {
		dummy_vertex = (VERTEX *)xmalloc(((n * 2) - 1) * sizeof(VERTEX));
		j = 0;
		dummy_vertex[j].x = vertex[0].x;
		dummy_vertex[j].y = vertex[0].y;
		for (i = 1 ; i < n ; i++) {
			sp.x = vertex[i-1].x;
			sp.y = vertex[i-1].y;
			ep.x = vertex[i].x;
			ep.y = vertex[i].y;
			split2(sp, ep, &ap1);
			j++;
			dummy_vertex[j].x = ap1.x;
			dummy_vertex[j].y = ap1.y;
			j++;
			dummy_vertex[j].x = vertex[i].x;
			dummy_vertex[j].y = vertex[i].y;
		}
		n = (n * 2) - 1;
	}
	else if (n > 6) {
		dummy_vertex = (VERTEX *)xmalloc( n * sizeof(VERTEX) );
		for (i = 0 ; i < n ; i++) {
			dummy_vertex[i].x = vertex[i].x;
			dummy_vertex[i].y = vertex[i].y;
		}
	}



	for (i = 0 ; i < n-3 ; i++) {
		for (u = 0.0 ; u <= 1.0 + pitch/2 ; u = u + pitch) {
			/* 最初の区間 */
			if (i == 0) {
				spl1(u, &s1, &s2, &s3, &s4);
			}
			/* ２番目の区間 */
			else if (i == 1) {
				spl2(u, &s1, &s2, &s3, &s4);
			}
			/* 中間の区間 */
			else if (i > 1 && i < n-5) {
				spl3(u, &s1, &s2, &s3, &s4);
			}
			/* 最後から２番目の区間 */
			else if (i > 1 && i == n-5) {
				spl4(u, &s1, &s2, &s3, &s4);
			}
			/* 最後の区間 */
			else if (i > 2 && i == n-4) {
				spl5(u, &s1, &s2, &s3, &s4);
			}
			x = s1*dummy_vertex[i].x + s2*dummy_vertex[i+1].x + s3*dummy_vertex[i+2].x + s4*dummy_vertex[i+3].x;
			y = s1*dummy_vertex[i].y + s2*dummy_vertex[i+1].y + s3*dummy_vertex[i+2].y + s4*dummy_vertex[i+3].y;
			if (u == 0) {
				//pset;
			}
			else {
				Ret = LineDataCheck(old_x, old_y, x, y, px, py);
				if (Ret != 0) {
					xfree(dummy_vertex);
					return Ret;
				}
				//LineDraw(drawing_area, old_x, old_y, x, y, 1, 0xffffff);
			}
			old_x = x;
			old_y = y;
		}
	}

	xfree(dummy_vertex);
	return Ret;
}





/* -------------------------------------------------------
 * 曲線探索メイン
 * 
 */
POLYLINE *PolyLine_search(double x, double y)
{
//#define POLYLINE_SEARCH

	POLYLINE_LIST *p;
	int debug = 0;


#ifdef POLYLINE_SEARCH
	debug = 1;
#endif


	if (PolyLineStartData == NULL && polyline_list_info.head != NULL) {
		PolyLineStartData = polyline_list_info.tail;
	}
	else if (polyline_list_info.head == NULL) {
		return NULL;
	}

	/* p に PolyLineStartData を代入 */
	p = PolyLineStartData;

	/* -----------------------------------------------------
	 * p が最後のデータ tail なら p を最初のデータ head にする。
	 * それ以外なら、p の氓ﾌデータ(PolyLineStartData の樊から探す
	 */
	if (p == polyline_list_info.tail) {
		p = polyline_list_info.head;
	}
	else if (p != polyline_list_info.tail) { 
		p = p->next;
	}


	/* p から データの最後までを探す */
	while (p != NULL) {
		if (Layer[p->polyline->layer].draw == 1) {
			if (debug > 0) g_print("Select.c : PolyLine_search() : in\n");

			/* ---------------------------------------------
			 * スプライン補間
			 */
			if (p->polyline->code == 16) {
				if (debug > 0) g_print("Select.c : PolyLine_search() : search spline 1\n");

				/* 閉じているとき(閉曲線) */
				if ((p->polyline->frag & 1) == 1) {
					if (debug > 0) g_print("Select.c : PolyLine_search() : search close spline 1\n");
					if (close_spline_search(p->polyline, x, y) != 0) {
						/* 見つかった */
						PolyLineStartData = p;
						if (debug > 0) {
							g_print("Select.c : PolyLine_search() : スプライン補間(閉曲線)がみつかった 1\n");
						}
						if (SelectPolyLineCheck(p->polyline) == 0) {
							return p->polyline;
						}
					}
				}

				/* (開曲線) */
				else {
					if (debug > 0) g_print("Select.c : PolyLine_search() : search open spline 1\n");
					if (open_spline_search(p->polyline, x, y) != 0) {
						/* 見つかった */
						PolyLineStartData = p;
						if (debug > 0) {
							g_print("Select.c : PolyLine_search() : スプライン補間(開曲線)がみつかった 1\n");
						}
						if (SelectPolyLineCheck(p->polyline) == 0)
							return p->polyline;
					}
				}
			}


			/* ---------------------------------------------
			 * Ｂスプライン曲線
			 */
			else if (p->polyline->code == 32) {
				if (debug > 0) g_print("Select.c : PolyLine_search() : search B-spline 1\n");
				if (b_spline_search(p->polyline, x, y) != 0) {
					/* 見つかった */
					PolyLineStartData = p;
					if (debug > 0) {
						g_print("Select.c : PolyLine_search() : Ｂスプライン曲線がみつかった 1\n");
					}
					if (SelectPolyLineCheck(p->polyline) == 0)
						return p->polyline;
				}
			}


			/* ---------------------------------------------
			 * 折れ線
			 */
			else if (p->polyline->code == 8) {
				if (debug > 0) g_print("Select.c : PolyLine_search() : search polyline 1\n");
				if (normal_polyline_search(p->polyline, x, y) != 0) {
					/* 見つかった */
					PolyLineStartData = p;
					if (debug > 0) {
						g_print("Select.c : PolyLine_search() : 折れ線がみつかった 1\n");
					}
					if (SelectPolyLineCheck(p->polyline) == 0)
						return p->polyline;
				}
			}
		}
		else {
			//
		}

		p = p->next;
	}


	/* -----------------------------------------------------
	 * PolyLineStartData が tail だったら
	 * 最初から最後まで探しているからこの後の処理はいらない。
	 */
	if (PolyLineStartData == polyline_list_info.tail) {
		return NULL;
	}


	/* ------------------------------------------------------------
	 * データの最初から PolyLineStartData までを探す。
	 */
	p =  polyline_list_info.head;
	while (p != NULL && p != PolyLineStartData->next) {
		if(Layer[p->polyline->layer].draw == 1) {

			/* ---------------------------------------------
			 * スプライン補間
			 */
			if (p->polyline->code == 16) {

				/* 閉じているとき(閉曲線) */
				if ((p->polyline->frag & 1) == 1) {
					if (close_spline_search(p->polyline, x, y) != 0) {
						/* 見つかった */
						PolyLineStartData = p;
						if (debug > 0) {
							g_print("Select.c : PolyLine_search() : スプライン補間(閉曲線)がみつかった 2\n");
						}
						if (SelectPolyLineCheck(p->polyline) == 0)
							return p->polyline;
					}
				}

				/* スプライン補間(開曲線) */
				else {
					if (open_spline_search(p->polyline, x, y) != 0) {
						/* 見つかった */
						PolyLineStartData = p;
						if (debug > 0) {
							g_print("Select.c : PolyLine_search() : スプライン補間(開曲線)がみつかった 2\n");
						}
						if (SelectPolyLineCheck(p->polyline) == 0)
							return p->polyline;
					}
				}
			}


			/* ---------------------------------------------
			 * Ｂスプライン曲線
			 */
			else if (p->polyline->code == 32) {
				if (b_spline_search(p->polyline, x, y) != 0) {
						/* 見つかった */
						PolyLineStartData = p;
						if (debug > 0) {
							g_print("Select.c : PolyLine_search() : Ｂスプライン曲線がみつかった 2\n");
						}
						if (SelectPolyLineCheck(p->polyline) == 0)
							return p->polyline;
				}
			}

			/* ---------------------------------------------
			 * 折れ線
			 */
			else if (p->polyline->code == 8) {
				if (normal_polyline_search(p->polyline, x, y) != 0) {
						/* 見つかった */
						PolyLineStartData = p;
						if (debug > 0) {
							g_print("Select.c : PolyLine_search() : 折れ線がみつかった 2\n");
						}
						if (SelectPolyLineCheck(p->polyline) == 0)
							return p->polyline;
				}
			}

		}
		else {
			//
		}

		p = p->next;
	}
	PolyLineStartData = p;
	return NULL;
}





/* -------------------------------------------------------
 * polyline Data がセレクトされているかを確認
 *	
 * 引数 ： POLYLINE *data
 * 戻値 ： int	0：選択されていない 	1：選択済み
 */
int SelectPolyLineCheck(POLYLINE *data)
{
	SELECT_LIST *p;

	p = select_list_info.head;
	while (p != NULL) {
		if (p->select->diagram_point.polyline_point == data) 
			return 1;
		p = p->next;
	}
	return 0;
}





/* -------------------------------------------------------
 * 選択図形を赤くする。＆キャンセルで元の色に戻す
 * 
 * 選択されている図形を 1 で赤くし、0 で戻す。
 * 
 * diagram_type
 *     CAD       : 2 or 4
 *     DIMENSION : 3
 *     POLYLINE  : 8
 *     ELLIPSE   : 16
 *     INSERT    : 32
 * 
 * 図形描画 DrawType
 * 選択色   ： DrawType = 1	SCD_SELECT
 * 元の色   ： DrawType = 0	SCD_ORG
 * 消  去   ： DrawType = 2	SCD_HIDE
 * 
 * UpDate
 *     1 : 今描いたものをすぐに表示
 *     0 : 後で表示
 * 
 */
int SelectDraw(GtkWidget *widget, SELECT *select, int DrawType, int update)
{
	/* CAD Data */
	if (select->index == 2 || select->index == 4) {
		if (select->diagram_point.cad_point == NULL) {
			MsgBox("SagCAD error", "CAD Data is NULL !", "SelectDraw()", NULL, NULL, "OK");
			return 0;
		}
		if (Layer[select->diagram_point.cad_point->layer].draw == 1) {
			CadDraw(widget, select->diagram_point.cad_point, DrawType, update);
		}
	}

	/* DIMENSION Data */
	else if (select->index == 3) {
		if (select->diagram_point.dimension_point == NULL) {
			MsgBox("SagCAD error", "Dimension Data is NULL !", "SelectDraw()", NULL, NULL, "OK");
			return 0;
		}
		if (Layer[select->diagram_point.dimension_point->Layer].draw == 1) {
			DimensionDraw (drawing_area, select->diagram_point.dimension_point, DrawType);
		}
	}

	/* POLYLINE Data */
	else if (select->index == 8) {
		if (select->diagram_point.polyline_point == NULL) {
			MsgBox("SagCAD error", "PolyLine Data is NULL !", "SelectDraw()", NULL, NULL, "OK");
			return 0;
		}
		if (Layer[select->diagram_point.polyline_point->layer].draw == 1) {
			PolyLineDraw (widget, select->diagram_point.polyline_point, DrawType);
		}
	}

	/* ELLIPSE Data */
	else if (select->index == 16) {
		if (select->diagram_point.ellipse_point == NULL) {
			MsgBox("SagCAD error", "ELLIPSE Data is NULL !", "SelectDraw()", NULL, NULL, "OK");
			return 0;
		}
		if (Layer[select->diagram_point.ellipse_point->layer].draw == 1) {
			EllipseDraw (widget, select->diagram_point.ellipse_point, DrawType);
		}
	}

	/* INSERT Data */
	else if (select->index == 32) {
		//
	}


	if (update == UPDATE_ON) {
		UpDate();
	}


	return 1;
}





/* ====================================================================
 * ===	Copyright (C) 1998-2007 Yutaka Sagiya. All rights reserved. ===
 * ====================================================================
 *	  Project			   : SagCAD
 *	  Source			   : select.c
 * ====================================================================
 */





/* -------------------------------------------------------
 * 選択図形を赤くする。＆キャンセルで元の色に戻す
 * 
 * 選択されている図形を 1 で赤くし、0 で戻す。
 * 
 * DrawType
 *     SCD_ORG     : 0 : 定義色
 *     SCD_SELECT  : 1 : 選択色
 *     SCD_HIDE    : 2 : 背景色(消去)
 * 
 * UpDate
 *     UPDATE_ON  : 1 : 今描いたものをすぐに表示
 *     UPDATE_OFF : 0 : 後で表示
 * 
 */
int CadDraw(GtkWidget *widget, CAD *cad, int DrawType, int update)
{
	long SelectCadDrawColor = cad->color;


	if (cad == NULL) {
		MsgBox("SagCAD error", "CAD Data is NULL !", "SelectCadDraw error", NULL, NULL, "OK");
		return 0;
	}

	if ( DrawType == SCD_SELECT) SelectCadDrawColor = sagcad_color.Select;
	else if ( DrawType == SCD_ORG) SelectCadDrawColor = cad->color;
	else if ( DrawType == SCD_HIDE) SelectCadDrawColor = sagcad_color.Back;

	/* 点のとき */
	if ( cad->code == 0) {
		if ( Layer[cad->layer].draw != 0) {
			PointDraw(widget, cad->sx, cad->sy, cad->style, sagcad_color.Back);
			PointDraw(widget, cad->sx, cad->sy, cad->style, SelectCadDrawColor);
		}
	}
	/* 直線のとき */
	else if ( cad->code == 1) {
		if ( Layer[cad->layer].draw != 0) {
			LineDraw(widget, cad->sx, cad->sy, cad->ex, cad->ey, cad->style, sagcad_color.Back);
			LineDraw(widget, cad->sx, cad->sy, cad->ex, cad->ey, cad->style, SelectCadDrawColor);
		}
	}
	/* 円弧のとき */
	else if ( cad->code == 2) {
		if ( Layer[cad->layer].draw != 0) {
			/* 円弧を描画 */
			ArcDraw(widget, cad->cx, cad->cy, cad->r, cad->sx, cad->sy, cad->ex, cad->ey, cad->style, sagcad_color.Back);
			ArcDraw(widget, cad->cx, cad->cy, cad->r, cad->sx, cad->sy, cad->ex, cad->ey, cad->style, SelectCadDrawColor);
		}
	}
	/* 円のとき */
	else if ( cad->code == 4) {
		if ( Layer[cad->layer].draw != 0) {
			/* 円を描画 */
			CircleDraw(widget, cad->cx, cad->cy, cad->r, cad->style, sagcad_color.Back);
			CircleDraw(widget, cad->cx, cad->cy, cad->r, cad->style, SelectCadDrawColor);
		}
	}

	if (update == UPDATE_ON) {
		UpDate();
	}

	return 1;
}





/* -------------------------------------------------------
 * 楕円を探索
 * 
 */
int ellipse_search_sub(ELLIPSE *ellipse, double px, double py)
{
//#define ELLIPSE_SEARCH_SUB
	struct RtnDat rtn;
	double a, b, angle;
	double l1, l2, pic_angle;
	double x1, y1, x2, y2;
	int debug = 0;


#ifdef ELLIPSE_SEARCH_SUB
	debug = 1;
	if (debug > 0) g_print("Select.c : ellipse_search_sub() : in\n");
#endif

	/* -----------------------------------------------------
	 * 楕円の要素を求める
	 */
	
	/* 中心から長軸の位置の角度と距離を求める */
	/* -------------------------------------------------------------------
	 * la : [8] 直線の角度
	 */
	rtn.sx[1] = ellipse->cx;
	rtn.sy[1] = ellipse->cy;
	rtn.ex[1] = ellipse->cx + ellipse->dx;
	rtn.ey[1] = ellipse->cy + ellipse->dy;
	la(&rtn);
	angle = rtn.angle;
	if (debug > 0) g_print("Select.c : ellipse_search_sub() : angle = %f\n", angle);

	/* -------------------------------------------------------------------
	 * pp : [9] ２点間の距離
	 */
	pp(&rtn);
	a = rtn.l;
	b = a * ellipse->k;
	if (debug > 0) g_print("Select.c : ellipse_search_sub() : a = %f  b = %f\n", a, b);

	/* 焦点を求める */
	x1 = -1 * (sqrt(a*a - b*b)) + ellipse->cx;
	y1 = 0 + ellipse->cy;
	x2 = sqrt(a*a - b*b) + ellipse->cx;
	y2 = 0 + ellipse->cy;
	rotation(&x1, &y1, ellipse->cx, ellipse->cy, angle);
	rotation(&x2, &y2, ellipse->cx, ellipse->cy, angle);
	if (debug > 0) g_print("Select.c : ellipse_search_sub() : 焦点 (%f,%f) (%f,%f)\n", x1, y1, x2, y2);


	/* (px,py) と 焦点の距離の関係で、DCの2倍で判定 */
	/* -------------------------------------------------------------------
	 * pp : [9] ２点間の距離
	 */
	rtn.sx[1] = x1;
	rtn.sy[1] = y1;
	rtn.ex[1] = px;
	rtn.ey[1] = py;
	pp(&rtn);
	l1 = rtn.l;

	rtn.sx[1] = x2;
	rtn.sy[1] = y2;
	rtn.ex[1] = px;
	rtn.ey[1] = py;
	pp(&rtn);
	l2 = rtn.l;

	if (debug > 0) g_print("Select.c : ellipse_search_sub() : l1 + l2 = %f  OK = %f  DC = %f\n", l1 + l2, 2*a, DC);

	if ( (l1 + l2) + DC > (2*a) && (l1 + l2) - DC < (2*a) ) {
		if (debug > 0) g_print("Select.c : ellipse_search_sub() : Click On Ellipse\n");
		//
	}
	else {
		if (debug > 0) g_print("Select.c : ellipse_search_sub() : Click Not On Ellipse\n");
		return 0;
	}


	/* 開始角・終了角で判定 */
	/* ピック点の楕円に対する角度を求める */
	/* -------------------------------------------------------------------
	 * la : [8] 直線の角度
	 */
	rtn.sx[1] = ellipse->cx;
	rtn.sy[1] = ellipse->cy;
	rtn.ex[1] = px;
	rtn.ey[1] = py;
	la(&rtn);
	pic_angle = rtn.angle - angle;
	
	while (pic_angle < 0 || pic_angle > 360) {
		if (pic_angle > 360) pic_angle = pic_angle - 360;
		if (pic_angle < 0) pic_angle = pic_angle + 360;
	}


	if (debug > 0) {
		g_print("Select.c : ellipse_search_sub() : pic_angle = %f  Ellipse sa = %f  ea = %f\n", 
				 pic_angle, ellipse->sa, ellipse->ea);
	}


	if (ellipse->sa < ellipse->ea) {
		if (debug > 0) g_print("Select.c : ellipse_search_sub() : sa (%f) < ea (%f)\n", ellipse->sa, ellipse->ea);

		if (pic_angle > ellipse->sa && pic_angle < ellipse->ea) {
			if (debug > 0) g_print("Select.c : ellipse_search_sub() : sa < pic (%f) < ea ---> OK !!\n", pic_angle);
			return 1;
		}
	}
	else if (ellipse->sa > ellipse->ea) {
		if (debug > 0) g_print("Select.c : ellipse_search_sub() : sa (%f) > ea (%f)\n", ellipse->sa, ellipse->ea);

		if (pic_angle > ellipse->sa || pic_angle < ellipse->ea) {
			if (debug > 0) g_print("Select.c : ellipse_search_sub() : pic (%f) > sa  pic (%f) < ea ---> OK !!\n", 
					pic_angle, pic_angle);
			return 1;
		}
	}
	else {
		return 0;
	}

	return 0;
}





/* -------------------------------------------------------
 * 楕円探索メイン
 * 
 */
ELLIPSE *ellipse_search(double x, double y)
{
//#define ELLIPSE_SEARCH

	ELLIPSE_LIST *p;
//	static ELLIPSE_LIST *StartData = NULL;
	int debug = 0;


#ifdef ELLIPSE_SEARCH
	debug = 1;
#endif


	if (EllipseStartData == NULL && ellipse_list_info.head != NULL) {
		EllipseStartData = ellipse_list_info.tail;
	}
	else if (ellipse_list_info.head == NULL) {
		return NULL;
	}

	/* p に EllipseStartData を代入 */
	p = EllipseStartData;

	/* -----------------------------------------------------
	 * p が最後のデータ tail なら p を最初のデータ head にする。
	 * それ以外なら、p の氓ﾌデータ(EllipseStartData の樊から探す
	 */
	if (p == ellipse_list_info.tail) {
		p = ellipse_list_info.head;
	}
	else if (p != ellipse_list_info.tail) { 
		p = p->next;
	}


	/* p から データの最後までを探す */
	while (p != NULL) {
		if(Layer[p->ellipse->layer].draw == 1) {
			if (debug > 0) g_print("Select.c : ellipse_search() : in\n");
			/* ---------------------------------------------
			 * 楕円
			 */
			if (debug > 0) g_print("Select.c : ellipse_search() : search ellipse 1\n");
			if (ellipse_search_sub(p->ellipse, x, y) != 0) {
				/* 見つかった */
				EllipseStartData = p;
				if (debug > 0) {
					g_print("Select.c : ellipse_search() : 楕円がみつかった 1\n");
				}
				if (SelectEllipseCheck(p->ellipse) == 0)
					return p->ellipse;
			}
		}
		else {
			//
		}
		p = p->next;
	}


	/* -----------------------------------------------------
	 * EllipseStartData が tail だったら
	 * 最初から最後まで探しているからこの後の処理はいらない。
	 */
	if (EllipseStartData == ellipse_list_info.tail) {
		return NULL;
	}


	/* ------------------------------------------------------------
	 * データの最初から EllipseStartData までを探す。
	 */
	p =  ellipse_list_info.head;
	while (p != NULL && p != EllipseStartData->next) {
		if(Layer[p->ellipse->layer].draw == 1) {
			/* ---------------------------------------------
			 * 楕円
			 */
			if (ellipse_search_sub(p->ellipse, x, y) != 0) {
					/* 見つかった */
					EllipseStartData = p;
					if (debug > 0) {
						g_print("Select.c : ellipse_search() : 楕円がみつかった 2\n");
					}
					if (SelectEllipseCheck(p->ellipse) == 0)
						return p->ellipse;
			}
		}
		else {
			//
		}
		p = p->next;
	}
	EllipseStartData = p;
	return NULL;
}





/* -------------------------------------------------------
 * ELLIPSE Data がセレクトされているかを確認
 *	
 * 引数 ： ELLIPSE *data
 * 戻値 ： int	0：選択されていない 	1：選択済み
 */
int SelectEllipseCheck(ELLIPSE *data)
{
	SELECT_LIST *p;

	p = select_list_info.head;
	while (p != NULL) {
		if (p->select->diagram_point.ellipse_point == data) 
			return 1;
		p = p->next;
	}
	return 0;
}





/* ====================================================================
 * ===  Copyright (C) 1998-2007 Yutaka Sagiya. All rights reserved. ===
 * ====================================================================
 *    Project              : SagCAD
 *    Source               : Select.c
 * ====================================================================
 */
