/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file Su.cxx
    \author Romain BONDUE
    \date 19/12/2005 */
#include <iostream>
#include <cstdlib> // abort()
#include <sstream>
#include <memory>
#include <csignal>

extern"C"{
#include <sys/types.h>
#include <unistd.h> // ::getpid()
}

#include "Su.h"
#include "Msg.h"
#include "StaticSettings.h"
#include "ErrorsCode.h"



namespace
{
    template <typename T>
    std::string ToString (const T& Value) throw()
    {
        std::ostringstream Os;
        Os << Value;
        return Os.str();

    } // ToString()

} // anonymous namespace



nsSystem::CLocalSocket* nsRoot::CSu::pSock (0);
unsigned long nsRoot::CSu::CurrentRemoteHandlerNumber (0);
unsigned long nsRoot::CSu::InstanceCounter (1);



bool nsRoot::CSu::Init (const std::string& RootPassword)
                                    throw (nsErrors::CException, std::bad_alloc)
{
    Close();
    nsSystem::CLocalSocket MasterSock;
    const ::pid_t Pid (::getpid());
    MasterSock.Bind (ServerAddr + ToString (Pid));
    MasterSock.Listen (1);

    if (!nsSystem::Fork())
    {
        std::ostringstream OsArg1;
        try
        {
            MasterSock.Close();
            OsArg1 << Pid;
            const std::string Arg1 (OsArg1.str());
#ifndef EXTERNALLAUNCHER
            const std::string HelperPath (nsCore::HelperPrefix +
                                                            nsCore::HelperName);
#endif // EXTERNALLAUNCHER
            const char* Argv [] = {
#ifdef EXTERNALLAUNCHER
                                   nsCore::HelperLauncherPath.c_str(),
#else
                                   HelperPath.c_str(),
#endif // EXTERNALLAUNCHER
                                   Arg1.c_str(), 0};
            nsSystem::Exec (Argv);
        }
        catch (nsErrors::CException& Exc)
        {
            std::cerr << "Error : " << Exc.GetMsg() << "\nCode : "
                      << Exc.GetCode() << std::endl;
        }
        catch (...) {} // Ignore everything else.
        bool Unblocked (false);
        try
        {
                // The other process may block on accept(), let's prevent that.
            nsSystem::CLocalSocket Socket;
            const std::string Addr (ServerAddr + OsArg1.str());
            ::timespec Timer;
            std::memset (&Timer, 0, sizeof (Timer));
            Timer.tv_nsec = 20000L;
                // Try every 20ms for 3 seconds.
            for (unsigned Cnt (16) ; --Cnt && Unblocked == false ; )
                try
                {
                    ::nanosleep (&Timer, 0);
                    Socket.Connect (Addr);
                    Unblocked = true;
                }
                catch (...) {} // Ignore everything.
        }
        catch (nsErrors::CException& Exc)
        {
            std::cerr << "Error in error handler : " << Exc.GetMsg()
                      << "\nCode : " << Exc.GetCode() << std::endl;
        }
        catch (...) {} // Ignore everything else.
        if (!Unblocked)
            ::kill (::getppid(), SIGKILL);
        abort();
    }
    try
    {       // Could possibly occur after helper's Connect(), but very unlikely.
        pSock = MasterSock.Accept();
        pSock->SendCredential();
        if (!pSock->CheckCredentialUID (0))
            throw nsErrors::CException ("Helper authentication failed.",
                                        nsErrors::HelperAuthenticationFailed);
#ifndef NOROOTPASSCHECK
        const int ACK (SendInternalCommand (CheckRootPassword, RootPassword));
        if (ACK)
        {
            if (ACK == nsErrors::InvalidRootPassword)
                return false;
            throw nsErrors::CException ("Cannot send root password.",
                                        nsErrors::CannotSendRootPasswd);
        }
#endif // NOROOTPASSCHECK
        pSock->Write (reinterpret_cast<const char*> (&InstanceCounter),
                                                    sizeof (InstanceCounter));
    }
    catch (const nsErrors::CException& Exc)
    {
        if (pSock)
            try
            {
                Close();
            }
            catch (...) {} // Ignore everything.
#ifndef NDEBUG
        std::cerr << "Error while checking root password and/or authenticate: "
                  << Exc.GetMsg() << "\nCode : " << Exc.GetCode() << std::endl;
#endif // NDEBUG
        if (Exc.GetCode() == EINTR || Exc.GetCode() ==
                                                nsErrors::CannotSendRootPasswd)
            return false;
        if (Exc.GetCode() == EPIPE || Exc.GetCode() == ECONNRESET)
            throw nsErrors::CException ("Cannot execute helper, check your"
                            " installation.", nsErrors::CannotExecuteHelper);
        throw;
    }
    return true;

} // Init()


nsRoot::CSu::~CSu () throw()
{
    try
    {
        if (pSock)
            SendInternalCommand (DeleteRemoteHandlerCmd,
                                 ToString (m_InstanceNumber));
    }
    catch (...) {} // We just ignore errors.

} // ~CSu()


void nsRoot::CSu::CreateRemoteHandler
        (const std::string& RemoteHandlerTypeName) throw (nsErrors::CSystemExc)
{
    int Value (SendInternalCommand (CreateRemoteHandlerCmd,
                                    RemoteHandlerTypeName));
    if (Value)
    {
        std::auto_ptr<char> Buffer (0);
        try{Buffer.reset (new char [Value + 1]);}
        catch (const std::bad_alloc& Exc)
        {
            throw nsErrors::CSystemExc (Exc.what(), nsErrors::OutOfMemory);
        }
        Buffer.get() [pSock->Read (Buffer.get(), Value)] = '\0';
        pSock->Read (reinterpret_cast<char*> (&Value), sizeof (Value));
        throw nsErrors::CSystemExc
            (std::string ("Can't create remote handler : ") + Buffer.get(),
             Value);
    }

} // CreateRemoteHandler()


void nsRoot::CSu::WriteString (const std::string& Str)
                                                throw (nsErrors::CSystemExc)
{
    const int Size (Str.size());
    pSock->Write (reinterpret_cast<const char*> (&Size), sizeof (Size));
    if (Size) pSock->Write (Str.data(), Size);

} // WriteString()


    /* The main difference with Do() is the order on which fields are written :
           Size, Data, Code for Do(), then the same sequence of data (value may
           differ of course) is expected back.
           Code, Size, Data for this function, an int is expected to be sent
           back.
       That's why commands must be negatives (a size can't be negative) so
       the remote process can identify the data type. */
int nsRoot::CSu::SendInternalCommand (int Code, const std::string& Text)
                                                throw (nsErrors::CSystemExc)
{
    pSock->Write (reinterpret_cast<const char*> (&Code), sizeof (Code));
    WriteString (Text);
    int ACK (nsErrors::InvalidData);
    pSock->Read (reinterpret_cast<char*> (&ACK), sizeof (ACK));
    return ACK;

} // SendInternalCommand()


void nsRoot::CSu::SetRemoteHandler () throw (nsErrors::CException)
{
    if (CurrentRemoteHandlerNumber != m_InstanceNumber)
    {
        int Value (SendInternalCommand (ChangeRemoteHandlerCmd,
                                        ToString (m_InstanceNumber)));
        if (Value)
        {
            std::string ErrorMsg ("Can't set remote handler.");
            if (Value > 0)
                try
                {
                    char Buffer [Value + 1];
                    Buffer [pSock->Read (Buffer, Value)] = '\0';
                    ErrorMsg += '\n';
                    ErrorMsg += Buffer;
                    pSock->Read (reinterpret_cast<char*> (&Value),
                                 sizeof (Value));
                }
                catch (...) {Value = nsErrors::CodeNotTransmitted;}
            else
                Value = nsErrors::CodeNotTransmitted;
            throw nsErrors::CException (ErrorMsg, Value);
        }
        CurrentRemoteHandlerNumber = m_InstanceNumber;
    }

} // SetRemoteHandler()


nsRoot::CMsg nsRoot::CSu::Do (const CMsg& Msg) throw (nsErrors::CException,
                                                      std::bad_alloc)
{
    SetRemoteHandler();
    WriteString (Msg.GetText());
    int Code (Msg.GetCode());
    pSock->Write (reinterpret_cast<const char*> (&Code), sizeof (Code));

    int Size (0);
    pSock->Read (reinterpret_cast<char*> (&Size), sizeof (Size));
    std::auto_ptr<char> Buffer (0);
    if (Size)
    {
        Buffer.reset (new char [Size]);
        Size = pSock->Read (Buffer.get(), Size);
    }
    pSock->Read (reinterpret_cast<char*> (&Code), sizeof (Code));

    return Size ? CMsg (std::string (Buffer.get(), 0, Size), Code)
                : CMsg (std::string(), Code);

} // Do()
