/*
 *   file: form_net.c
 *  
 *  Copyright (C) Nicola Fragale <nicolafragale@libero.it>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>
#include <glade/glade.h>

#include "libral.h"

#include "app.h"
#include "personal_gui.h"
#include "form.h"
#include "form_info.h"
#include "app.h"
#include "groups_cb.h"
#include "models.h"
#include "types.h"
#include "themes.h"


struct _RubricaFormInfoPrivate {
  RubricaApp *app;

  GtkWidget *name;
  GtkWidget *vat;
    
  GtkWidget *logo_bt;
  GtkWidget *logo;

  GtkWidget *contacts_tree;
  GtkWidget *info_add;
  GtkWidget *info_del;
  GtkWidget *info_mod;

  gchar *logo_uri;              /* logo's path */

  GList* cards;                 /* cards referenced */

  gboolean dispose_has_run;
};


static void rubrica_form_info_class_init (RubricaFormInfoClass* klass);
static void rubrica_form_info_init       (RubricaFormInfo* obj);

static void rubrica_form_info_dispose    (RubricaFormInfo* obj);
static void rubrica_form_info_finalize   (RubricaFormInfo* obj);

static gboolean extract_info       (GtkTreeModel *model,GtkTreePath *path,
				    GtkTreeIter *iter, gpointer data); 

static void form_info_extract_data (RubricaFormInfo* form, RCard *card);
static void form_info_display_data (RubricaFormInfo* form, RCard *card);
static void form_info_clean_form   (RubricaFormInfo* form, RCard *card);


void form_info_logo_clicked (GtkWidget *button, gpointer data);


static RubricaFormClass* parent_class;


GType
rubrica_form_info_get_type()
{
  static GType form_info_type = 0;
  
  if (!form_info_type)
    {
      static const GTypeInfo form_info_info =
	{
	  sizeof(RubricaFormInfoClass),
	  NULL,
	  NULL,
	  (GClassInitFunc) rubrica_form_info_class_init,
	  NULL,
	  NULL,
	  sizeof(RubricaFormInfo),
	  0,
	  (GInstanceInitFunc) rubrica_form_info_init
	};

      form_info_type = g_type_register_static (RUBRICA_FORM_TYPE, 
					      "RubricaFormInfo",
					      &form_info_info, 0);
    }
  
  return form_info_type;
}


static void
rubrica_form_info_class_init(RubricaFormInfoClass* klass)
{
  GObjectClass *class;
 
  class  = G_OBJECT_CLASS (klass);
  class->dispose  = (GObjectFinalizeFunc) rubrica_form_info_dispose;
  class->finalize = (GObjectFinalizeFunc) rubrica_form_info_finalize;

  parent_class = RUBRICA_FORM_CLASS(klass);
  parent_class->extract_data = (gpointer) form_info_extract_data;
  parent_class->display_data = (gpointer) form_info_display_data;
  parent_class->clean_form   = (gpointer) form_info_clean_form;
}


static void
rubrica_form_info_init(RubricaFormInfo* self)
{
  self->priv = g_new(RubricaFormInfoPrivate, 1);
  
  self->priv->name          = NULL;
  self->priv->vat           = NULL;
  self->priv->logo_bt       = NULL;
  self->priv->logo          = NULL;
  self->priv->contacts_tree = NULL;
  self->priv->info_add      = NULL;
  self->priv->info_del      = NULL;
  self->priv->info_mod      = NULL;

  self->priv->logo_uri      = NULL;

  self->priv->cards         = NULL;

  self->priv->dispose_has_run = FALSE;
}

 

static void 
rubrica_form_info_dispose (RubricaFormInfo* self)
{
  g_return_if_fail(IS_RUBRICA_FORM(self));
  
  if (self->priv->dispose_has_run)
    return;
  
  g_object_unref(self->priv->name);
  g_object_unref(self->priv->vat);
  g_object_unref(self->priv->logo_bt);
  g_object_unref(self->priv->logo);
  g_object_unref(self->priv->contacts_tree);
  g_object_unref(self->priv->info_add);
  g_object_unref(self->priv->info_del);
  g_object_unref(self->priv->info_mod);

  if (self->priv->logo_uri)
    g_free(self->priv->logo_uri);
  
  if (self->priv->cards)
    {
      // TODO
      // foreach card in list do delete card
    }

  self->priv->dispose_has_run = TRUE;
}


static void 
rubrica_form_info_finalize (RubricaFormInfo* self)
{
  g_return_if_fail(IS_RUBRICA_FORM(self));
  
  g_free(self->priv);
  self->priv = NULL;
}


static gboolean 
extract_info (GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter,
	      gpointer data)
{
  RCard* company = (RCard*) data;
  RRef* ref;
  gchar *card_info;
  glong card_id;
  
  gtk_tree_model_get(model, iter, 
		     INFO_CARD_LABEL_COLUMN, &card_info,
		     INFO_ID_COLUMN,         &card_id,
		     NULL);

  ref = r_ref_new(card_id);
  g_object_set(G_OBJECT(ref), "ref-info", card_info, NULL);
  
  r_card_add_ref(R_CARD(company), ref);
  
  return FALSE;
}


static void 
form_info_extract_data (RubricaFormInfo* form, RCard *card)
{
  RAbook*  abook;
  GtkTreeModel* model = NULL;
  RRef* ref;
  glong company_id;
  const gchar *name;
  const gchar *vat;
  const gchar* logo;

  g_return_if_fail(IS_R_CARD(card));
  
  g_object_get(R_CARD(card), "card-id", &company_id, NULL);
  
  abook = rubrica_app_get_active_addressbook(RUBRICA_APP(form->priv->app));

  name  = gtk_entry_get_text(GTK_ENTRY(form->priv->name));
  vat   = gtk_entry_get_text(GTK_ENTRY(form->priv->vat));
  form->priv->logo_uri ? logo = form->priv->logo_uri : NULL;

  g_object_set(G_OBJECT(card), "company-name", name, "company-logo", logo,
	       "company-vat", vat, NULL);
  
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(form->priv->contacts_tree));
  if (model)
    {
      gtk_tree_model_foreach(GTK_TREE_MODEL(model), extract_info, card);
      g_object_unref(model);
    }  
  
  r_card_reset_ref(card);
  for (; (ref = r_card_get_ref(card)); ref = r_card_get_next_ref(card))
  {
    RCard* card;
    glong id; 
    
    g_object_get(G_OBJECT(ref), "ref-to", &id, NULL);
    if (id)
      {
	card = r_abook_get_card_by_id(abook, id);
	
	r_card_add_ref(card, ref);
      }
  }

  g_print("\n ********* company name: %s"
	  "\n ********* logo: %s"
	  "\n ********* pi: %s", name, logo, vat);	  
}


static void 
form_info_display_data (RubricaFormInfo* form, RCard *company)
{
  RAbook* abook;
  RRef* ref;
  GtkTreeModel* model = NULL;
  GtkTreeIter iter;
  gchar *name, *vat;

  abook = rubrica_app_get_active_addressbook(RUBRICA_APP(form->priv->app));

  g_object_get(G_OBJECT(company), "company-name", &name, 
	       "company-vat", &vat, NULL);

  gtk_entry_set_text(GTK_ENTRY(form->priv->name), name);
  gtk_entry_set_text(GTK_ENTRY(form->priv->vat),  vat);
  
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(form->priv->contacts_tree));
  r_card_reset_ref(R_CARD(company));
  for (; (ref = r_card_get_ref(R_CARD(company))); 
       ref = r_card_get_next_ref(R_CARD(company)));
  {
    RCard* card = NULL;
    RInfos *infos;
    glong id; 
    
    g_object_get(G_OBJECT(ref), "ref-to", &id, NULL);
    if (!id)
      return;

    card = r_abook_get_card_by_id(abook, id);
    if (card)
      {
        infos = r_card_get_infos(R_CARD(card));
	
	gtk_list_store_append(GTK_LIST_STORE(model), &iter);
	gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
			   INFO_CARD_LABEL_COLUMN, r_infos_get_label(infos),
			   INFO_FIRST_NAME_COLUMN, r_infos_get_last(infos),
			   INFO_LAST_NAME_COLUMN,  r_infos_get_first(infos),
			   INFO_ASSIGNMENT_COLUMN, r_infos_get_assignment(infos),
			   INFO_EMAIL_COLUMN,      r_infos_get_email(infos),
			   INFO_TELEPHONE_COLUMN,  r_infos_get_telephone(infos),
			   INFO_ID_COLUMN,         r_infos_get_id(infos),
			   -1);
      }
  }
}


static void 
form_info_clean_form   (RubricaFormInfo* form, RCard *card)
{
  gtk_entry_set_text(GTK_ENTRY(form->priv->name), "");
  gtk_entry_set_text(GTK_ENTRY(form->priv->vat),  ""); 
}



/*   Callbacks
*/
void 
form_info_list_activated(GtkTreeSelection *selection, gpointer data)
{
  // TODO form_info_list_activated
}


void
form_info_add_clicked (GtkButton *button, gpointer data)
{
  RubricaFormInfo *form = (RubricaFormInfo *) data;  
  PersonalDialog* personal;  
  RPersonalCard* card = NULL;
  RInfos *infos;
  GtkTreeModel *model;
  GtkTreeIter iter;
  
  model = gtk_tree_view_get_model(GTK_TREE_VIEW(form->priv->contacts_tree));

  personal = rubrica_personal_dialog_new(RUBRICA_APP(form->priv->app));
  card = rubrica_personal_run_add_dialog(personal);   
    
  if (card)
    {
      infos = r_card_get_infos(R_CARD(card));
      
      gtk_list_store_append(GTK_LIST_STORE(model), &iter);
      gtk_list_store_set(GTK_LIST_STORE(model), &iter, 
			 INFO_CARD_LABEL_COLUMN, r_infos_get_label(infos),
			 INFO_FIRST_NAME_COLUMN, r_infos_get_last(infos),
			 INFO_LAST_NAME_COLUMN,  r_infos_get_first(infos),
			 INFO_ASSIGNMENT_COLUMN, r_infos_get_assignment(infos),
			 INFO_EMAIL_COLUMN,      r_infos_get_email(infos),
			 INFO_TELEPHONE_COLUMN,  r_infos_get_telephone(infos),
			 INFO_ID_COLUMN,         r_infos_get_id(infos),
			 -1);
      
      form->priv->cards = g_list_append(form->priv->cards, card);
    }
}


void
form_info_mod_clicked (GtkButton *button, gpointer data)
{
  // TODO form_info_mod_clicked
}


void
form_info_del_clicked (GtkButton *button, gpointer data)
{
  // TODO form_info_del_clicked

}


void
form_info_logo_clicked (GtkWidget *button, gpointer data)
{
  RubricaFormInfo* form = (RubricaFormInfo*) data;
  GtkWidget* image = NULL;
  GdkPixbuf *pixbuf;

  GtkIconTheme* theme;
  gchar* uri = NULL;

  theme = gtk_icon_theme_get_default();       
  image = (GtkImage*) form->priv->logo;
  uri   = rubrica_themes_load_pixmap(button, image);
 
  if (uri)
    {
      pixbuf = gdk_pixbuf_new_from_file_at_size (uri, 124, 124, NULL);
      form->priv->logo_uri = g_strdup(uri);      
    }
  else
    if (form->priv->logo_uri)
      pixbuf = gdk_pixbuf_new_from_file_at_size (form->priv->logo_uri, 
						 124, 124, NULL);
    else
      {
	pixbuf = gtk_icon_theme_load_icon (theme, "puzzle", 48, 0, NULL);
	
	g_free(form->priv->logo_uri);
	form->priv->logo_uri = NULL;
      }
  
  gtk_image_set_from_pixbuf (GTK_IMAGE (image), pixbuf);  
}


/*   ***************************** Public *****************************
*/


/**
 * rubrica_form_info_new
 *
 * create a #RubricaFormInfo
 *
 * Returns: a new #RubricaFormInfo
 */
RubricaFormInfo*
rubrica_form_info_new(RubricaApp* app, GladeXML *gui)
{
  RubricaFormInfo* form;
  GtkTreeView      *tree;
  GtkTreeModel     *model;
  GtkTreeSelection *selection;

  form = g_object_new(rubrica_form_info_get_type(), NULL);

  form->priv->app           = app;
  form->priv->name          = glade_xml_get_widget(gui, "company_name");
  form->priv->vat           = glade_xml_get_widget(gui, "iva");
  form->priv->logo_bt       = glade_xml_get_widget(gui, "logo_bt");
  form->priv->logo          = glade_xml_get_widget(gui, "image");
  form->priv->contacts_tree = glade_xml_get_widget(gui, "contacts_tree");
  form->priv->info_add      = glade_xml_get_widget(gui, "contact_add");
  form->priv->info_del      = glade_xml_get_widget(gui, "contact_del");
  form->priv->info_mod      = glade_xml_get_widget(gui, "contact_mod");
  
  gtk_widget_set_sensitive(GTK_WIDGET(form->priv->info_mod), FALSE);
  gtk_widget_set_sensitive(GTK_WIDGET(form->priv->info_del), FALSE);

  /*   info urls model/view 
   */
  model = rubrica_model_new(INFO_MODEL);
  tree  = GTK_TREE_VIEW(form->priv->contacts_tree);

  gtk_tree_view_set_model(tree, GTK_TREE_MODEL(model));
  selection = gtk_tree_view_get_selection(tree);
  gtk_tree_selection_set_mode(selection, GTK_SELECTION_SINGLE);

  rubrica_model_add_columns(tree, INFO_COLUMNS);

  g_signal_connect(G_OBJECT(selection), "changed", 
		   G_CALLBACK(form_info_list_activated), form);
  g_object_unref(model);
  
  /*    info buttons 
   */
  g_signal_connect (G_OBJECT (form->priv->info_add), "clicked", 
		    G_CALLBACK (form_info_add_clicked), form); 
  g_signal_connect (G_OBJECT (form->priv->info_mod), "clicked", 
		    G_CALLBACK(form_info_mod_clicked), form); 
  g_signal_connect (G_OBJECT (form->priv->info_del), "clicked", 
		    G_CALLBACK (form_info_del_clicked), form); 

  /*    logo button
   */
  g_signal_connect (G_OBJECT (form->priv->logo_bt), "clicked", 
		    G_CALLBACK (form_info_logo_clicked), form);  

  return form;
}


/**
 * rubrica_form_info_free
 * @form: a #RubricaFormInfo
 * 
 * free the #RubricaFormInfo
 */
void
rubrica_form_info_free(RubricaFormInfo* form)
{
  g_return_if_fail(IS_RUBRICA_FORM(form));

  g_object_unref(form);   
}
