/*
	Copyright (C) 2004 - 2006 rt2x00 SourceForge Project
	<http://rt2x00.serialmonkey.com>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the
	Free Software Foundation, Inc.,
	59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
	Module: rt2500usb
	Abstract: rt2500usb device specific routines.
	Supported chipsets: RT2570.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/usb.h>
#include <linux/delay.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/wireless.h>
#include <linux/ethtool.h>

#include <net/iw_handler.h>
#include <net/d80211.h>

/*
 * Set enviroment defines for rt2x00.h
 */
#define DRV_NAME "rt2500usb"

#ifdef CONFIG_RT2500USB_DEBUG
#define CONFIG_RT2X00_DEBUG
#endif /* CONFIG_RT2500USB_DEBUG */

#include "rt2x00.h"
#include "rt2x00usb.h"
#include "rt2500usb.h"

/*
 * Register access.
 * All access to the CSR registers will go through the methods
 * rt2x00_register_read and rt2x00_register_write.
 * BBP and RF register require indirect register access,
 * and use the CSR registers BBPCSR and RFCSR to achieve this.
 * These indirect registers work with busy bits,
 * and we will try maximal REGISTER_BUSY_COUNT times to access
 * the register while taking a REGISTER_BUSY_DELAY us delay
 * between each attampt. When the busy bit is still set at that time,
 * the access attempt is considered to have failed,
 * and we will print an error.
 */
static int rt2x00_vendor_request(const struct rt2x00_dev *rt2x00dev,
	const u8 request, const u8 type, const u16 offset,
	u32 value, void *buffer, const u16 buffer_length, const u16 timeout)
{
	struct usb_device *usb_dev = interface_to_usbdev(
		rt2x00dev_usb(rt2x00dev));
	int status;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		status = usb_control_msg(
			usb_dev,
			(type == USB_VENDOR_REQUEST_IN) ?
				usb_rcvctrlpipe(usb_dev, 0) :
				usb_sndctrlpipe(usb_dev, 0),
			request, type, value, offset, buffer, buffer_length,
			timeout);
		if (status >= 0)
			return 0;
	}

	ERROR("vendor request error. Request 0x%02x failed "
		"for offset 0x%04x with error %d.\n", request, offset, status);

	return status;
}

static inline void rt2x00_register_read(
	const struct rt2x00_dev *rt2x00dev,
	const u16 offset, u16 *value)
{
	__le16 reg;
	rt2x00_vendor_request(
		rt2x00dev, USB_MULTI_READ, USB_VENDOR_REQUEST_IN,
		offset, 0x00, &reg, 2, REGISTER_TIMEOUT);
	*value = le16_to_cpu(reg);
}

static inline void rt2x00_register_multiread(
	const struct rt2x00_dev *rt2x00dev,
	const u16 offset, u16 *value, const u16 length)
{
	rt2x00_vendor_request(
		rt2x00dev, USB_MULTI_READ, USB_VENDOR_REQUEST_IN,
		offset, 0x00, value, length, REGISTER_TIMEOUT);
}

static inline void rt2x00_register_write(
	const struct rt2x00_dev *rt2x00dev,
	const u16 offset, u16 value)
{
	__le16 reg = cpu_to_le16(value);
	rt2x00_vendor_request(
		rt2x00dev, USB_MULTI_WRITE, USB_VENDOR_REQUEST_OUT,
		offset, 0x00, &reg, 2, REGISTER_TIMEOUT);
}

static inline void rt2x00_register_multiwrite(
	const struct rt2x00_dev *rt2x00dev,
	const u16 offset, u16 *value, const u16 length)
{
	rt2x00_vendor_request(
		rt2x00dev, USB_MULTI_WRITE, USB_VENDOR_REQUEST_OUT,
		offset, 0x00, value, length, REGISTER_TIMEOUT);
}

static u16 rt2x00_bbp_check(const struct rt2x00_dev *rt2x00dev)
{
	u16 reg;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00dev, PHY_CSR8, &reg);
		if (!rt2x00_get_field16(reg, PHY_CSR8_BBP_BUSY))
			return reg;
		udelay(REGISTER_BUSY_DELAY);
	}

	return 0xffff;
}

static void rt2x00_bbp_write(const struct rt2x00_dev *rt2x00dev,
	const u8 reg_id, const u8 value)
{
	u16 reg;

	/*
	 *  Wait untill the BBP becomes ready.
	 */
	if (rt2x00_bbp_check(rt2x00dev) == 0xffff) {
		ERROR("PHY_CSR8 register busy. Write failed.\n");
		return;
	}

	/*
	 * Write the data into the BBP.
	 */
	reg = 0;
	rt2x00_set_field16(&reg, PHY_CSR7_BBP_DATA, value);
	rt2x00_set_field16(&reg, PHY_CSR7_BBP_REG_ID, reg_id);
	rt2x00_set_field16(&reg, PHY_CSR7_BBP_READ_CONTROL, 0);

	rt2x00_register_write(rt2x00dev, PHY_CSR7, reg);
}

static void rt2x00_bbp_read(const struct rt2x00_dev *rt2x00dev,
	const u8 reg_id, u8 *value)
{
	u16 reg;

	/*
	 *  Wait untill the BBP becomes ready.
	 */
	if (rt2x00_bbp_check(rt2x00dev) == 0xffff) {
		ERROR("PHY_CSR8 register busy. Read failed.\n");
		return;
	}

	/*
	 * Write the request into the BBP.
	 */
	reg =0;
	rt2x00_set_field16(&reg, PHY_CSR7_BBP_REG_ID, reg_id);
	rt2x00_set_field16(&reg, PHY_CSR7_BBP_READ_CONTROL, 1);

	rt2x00_register_write(rt2x00dev, PHY_CSR7, reg);

	/*
	 *  Wait untill the BBP becomes ready.
	 */
	if (rt2x00_bbp_check(rt2x00dev) == 0xffff) {
		ERROR("PHY_CSR8 register busy. Read failed.\n");
		*value = 0xff;
		return;
	}

	rt2x00_register_read(rt2x00dev, PHY_CSR7, &reg);
	*value = rt2x00_get_field16(reg, PHY_CSR7_BBP_DATA);
}

static void rt2x00_rf_write(const struct rt2x00_dev *rt2x00dev,
	const u32 value)
{
	u16 reg;
	unsigned int i;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00dev, PHY_CSR10, &reg);
		if (!rt2x00_get_field16(reg, PHY_CSR10_RF_BUSY))
			goto rf_write;
		udelay(REGISTER_BUSY_DELAY);
	}

	ERROR("PHY_CSR10 register busy. Write failed.\n");
	return;

rf_write:
	reg = 0;
	rt2x00_set_field16(&reg, PHY_CSR9_RF_VALUE, value & 0x0000ffff);
	rt2x00_register_write(rt2x00dev, PHY_CSR9, reg);

	reg = 0;
	rt2x00_set_field16(&reg, PHY_CSR10_RF_VALUE,
		(value >> 16) & 0x0000ffff);
	rt2x00_set_field16(&reg, PHY_CSR10_RF_NUMBER_OF_BITS, 20);
	rt2x00_set_field16(&reg, PHY_CSR10_RF_IF_SELECT, 0);
	rt2x00_set_field16(&reg, PHY_CSR10_RF_BUSY, 1);

	rt2x00_register_write(rt2x00dev, PHY_CSR10, reg);
}

/*
 * EEPROM access.
 * The EEPROM is being accessed by word index.
 */
static void* rt2x00_eeprom_addr(const struct rt2x00_dev *rt2x00dev,
	const u8 word)
{
	return (void*)&rt2x00dev->eeprom[word];
}

static void rt2x00_eeprom_read(const struct rt2x00_dev *rt2x00dev,
	const u8 word, u16 *data)
{
	*data = le16_to_cpu(rt2x00dev->eeprom[word]);
}

static void rt2x00_eeprom_write(const struct rt2x00_dev *rt2x00dev,
	const u8 word, u16 data)
{
	rt2x00dev->eeprom[word] = cpu_to_le16(data);
}

/*
 * TX/RX Descriptor access functions.
 */
static inline struct urb* rt2x00_urb(struct data_entry *entry)
{
	return (struct urb*)entry->priv;
}

static inline void* rt2x00_rxdata_addr(struct data_entry *entry)
{
	return entry->data_addr;
}

static inline struct data_desc* rt2x00_rxdesc_addr(struct data_entry *entry)
{
	return entry->data_addr +
		(rt2x00_urb(entry)->actual_length - entry->ring->desc_size);
}

static inline void* rt2x00_txdata_addr(struct data_entry *entry)
{
	return entry->data_addr + entry->ring->desc_size;
}

static inline struct data_desc* rt2x00_txdesc_addr(struct data_entry *entry)
{
	return entry->data_addr;
}

static inline void rt2x00_desc_read(struct data_desc *desc,
	const u8 word, u32 *value)
{
	*value = le32_to_cpu(desc->word[word]);
}

static inline void rt2x00_desc_write(struct data_desc *desc,
	const u8 word, const u32 value)
{
	desc->word[word] = cpu_to_le32(value);
}

/*
 * Ethtool handlers.
 */
static void rt2500usb_get_drvinfo(struct net_device *net_dev,
	struct ethtool_drvinfo *drvinfo)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_get_hw(net_dev)->priv;

	strcpy(drvinfo->driver, DRV_NAME);
	strcpy(drvinfo->version, DRV_VERSION);
	strcpy(drvinfo->fw_version, "N/A");
	strcpy(drvinfo->bus_info, rt2x00dev_usb(rt2x00dev)->dev.bus_id);
	drvinfo->eedump_len = EEPROM_SIZE;
	drvinfo->regdump_len = CSR_REG_SIZE;
}

static int rt2500usb_get_regs_len(struct net_device *net_dev)
{
	return CSR_REG_SIZE;
}

static void rt2500usb_get_regs(struct net_device *net_dev,
	struct ethtool_regs *regs, void *data)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_get_hw(net_dev)->priv;

	rt2x00_register_multiread(rt2x00dev, CSR_REG_BASE, data, CSR_REG_SIZE);
	regs->version = rt2x00_rev(&rt2x00dev->chip);
	regs->len = CSR_REG_SIZE;
}

#ifdef CONFIG_RT2500USB_DEBUG
static u32 rt2500usb_get_msglevel(struct net_device *net_dev)
{
	return rt2x00_debug_level;
}

static void rt2500usb_set_msglevel(struct net_device *net_dev, u32 msg)
{
	rt2x00_debug_level = !!msg;
}
#endif /* CONFIG_RT2500USB_DEBUG */

static int rt2500usb_get_eeprom_len(struct net_device *net_dev)
{
	return EEPROM_SIZE;
}

static int rt2500usb_get_eeprom(struct net_device *net_dev,
	struct ethtool_eeprom *eeprom, u8 *data)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_get_hw(net_dev)->priv;

	eeprom->magic = 0xbad;

	memcpy(data,
		rt2x00_eeprom_addr(rt2x00dev, eeprom->offset / sizeof(u16)),
		eeprom->len);

	return 0;
}

static int rt2500usb_set_eeprom(struct net_device *net_dev,
	struct ethtool_eeprom *eeprom, u8 *data)
{
	struct rt2x00_dev *rt2x00dev = ieee80211_get_hw(net_dev)->priv;
	unsigned int i;
	unsigned int words;

	if (eeprom->magic != 0xbad)
		return -EINVAL;

	words = eeprom->len / sizeof(u16);
	for (i = eeprom->offset / sizeof(u16); i < words; i++)
		rt2x00_eeprom_write(rt2x00dev, i, ((u16*)data)[i]);

	return 0;
}

static struct ethtool_ops rt2500usb_ethtool_ops = {
	.get_drvinfo	= rt2500usb_get_drvinfo,
	.get_regs_len	= rt2500usb_get_regs_len,
	.get_regs	= rt2500usb_get_regs,
#ifdef CONFIG_RT2500USB_DEBUG
	.get_msglevel	= rt2500usb_get_msglevel,
	.set_msglevel	= rt2500usb_set_msglevel,
#endif /* CONFIG_RT2500USB_DEBUG */
	.get_link	= ethtool_op_get_link,
	.get_eeprom_len	= rt2500usb_get_eeprom_len,
	.get_eeprom	= rt2500usb_get_eeprom,
	.set_eeprom	= rt2500usb_set_eeprom,
	.get_perm_addr	= ethtool_op_get_perm_addr,
};

/*
 * Configuration handlers.
 */
static void rt2500usb_config_bssid(struct rt2x00_dev *rt2x00dev, u8 *bssid)
{
	u16 reg[3] = { 0, 0, 0 };

	/*
	 * The BSSID is passed to us as an array of bytes,
	 * that array is little endian, so no need for byte ordering.
	 */
	memcpy(&reg, bssid, ETH_ALEN);
	rt2x00_register_multiwrite(rt2x00dev, MAC_CSR5, &reg[0], sizeof(reg));
}

static void rt2500usb_config_promisc(struct rt2x00_dev *rt2x00dev, int promisc)
{
	u16 reg;

	rt2x00_register_read(rt2x00dev, TXRX_CSR2, &reg);

	if (promisc) {
		rt2x00_set_field16(&reg, TXRX_CSR2_DROP_NOT_TO_ME, 0);
		SET_FLAG(rt2x00dev, INTERFACE_ENABLED_PROMISC);
	} else {
		rt2x00_set_field16(&reg, TXRX_CSR2_DROP_NOT_TO_ME, 1);
		CLEAR_FLAG(rt2x00dev, INTERFACE_ENABLED_PROMISC);
	}

	rt2x00_register_write(rt2x00dev, TXRX_CSR2, reg);
}

static void rt2500usb_config_type(struct rt2x00_dev *rt2x00dev, int type)
{
	u16 reg;

	/*
	 * Only continue when there is something to be done.
	 */
	if (!(GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED) ^
	      GET_FLAG(rt2x00dev, INTERFACE_ENABLED)) &&
	    !(GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) ^
	      GET_FLAG(rt2x00dev, INTERFACE_ENABLED_MONITOR)))
		return;

	rt2x00_register_write(rt2x00dev, TXRX_CSR19, 0);

	/*
	 * Apply hardware packet filter.
	 */
	rt2x00_register_read(rt2x00dev, TXRX_CSR2, &reg);

	if (!GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) &&
	    (type == IEEE80211_IF_TYPE_IBSS || type == IEEE80211_IF_TYPE_STA))
		rt2x00_set_field16(&reg, TXRX_CSR2_DROP_TODS, 1);
	else
		rt2x00_set_field16(&reg, TXRX_CSR2_DROP_TODS, 0);

	rt2x00_set_field16(&reg, TXRX_CSR2_DROP_CRC, 1);
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR)) {
		rt2x00_set_field16(&reg, TXRX_CSR2_DROP_PHYSICAL, 0);
		rt2x00_set_field16(&reg, TXRX_CSR2_DROP_CONTROL, 0);
		rt2x00_set_field16(&reg, TXRX_CSR2_DROP_VERSION_ERROR, 0);
	} else {
		rt2x00_set_field16(&reg, TXRX_CSR2_DROP_PHYSICAL, 1);
		rt2x00_set_field16(&reg, TXRX_CSR2_DROP_CONTROL, 1);
		rt2x00_set_field16(&reg, TXRX_CSR2_DROP_VERSION_ERROR, 1);
	}

	rt2x00_register_write(rt2x00dev, TXRX_CSR2, reg);

	/*
	 * Enable promisc mode when in monitor mode.
	 */
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR))
		rt2500usb_config_promisc(rt2x00dev, 1);

	/*
	 * Enable beacon config
	 */
	rt2x00_register_read(rt2x00dev, TXRX_CSR20, &reg);
	rt2x00_set_field16(&reg, TXRX_CSR20_OFFSET,
		(PREAMBLE + get_duration(IEEE80211_HEADER, 2)) >> 6);
	if (type == IEEE80211_IF_TYPE_STA)
		rt2x00_set_field16(&reg, TXRX_CSR20_BCN_EXPECT_WINDOW, 0);
	else
		rt2x00_set_field16(&reg, TXRX_CSR20_BCN_EXPECT_WINDOW, 2);
	rt2x00_register_write(rt2x00dev, TXRX_CSR20, reg);

	/*
	 * Enable synchronisation.
	 */
	rt2x00_register_read(rt2x00dev, TXRX_CSR18, &reg);
	rt2x00_set_field16(&reg, TXRX_CSR18_OFFSET, 0);
	rt2x00_set_field16(&reg, TXRX_CSR18_INTERVAL, 100 << 2);
	rt2x00_register_write(rt2x00dev, TXRX_CSR18, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR19, &reg);
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED)) {
		rt2x00_set_field16(&reg, TXRX_CSR19_TSF_COUNT, 1);
		rt2x00_set_field16(&reg, TXRX_CSR19_TBCN, 1);
	}

	if (type == IEEE80211_IF_TYPE_IBSS) {
		rt2x00_set_field16(&reg, TXRX_CSR19_TSF_SYNC, 2);
		rt2x00_set_field16(&reg, TXRX_CSR19_BEACON_GEN, 1);
	} else if (type == IEEE80211_IF_TYPE_STA) {
		rt2x00_set_field16(&reg, TXRX_CSR19_TSF_SYNC, 1);
		rt2x00_set_field16(&reg, TXRX_CSR19_BEACON_GEN, 0);
	} else if (type == IEEE80211_IF_TYPE_AP) {
		rt2x00_set_field16(&reg, TXRX_CSR19_TSF_SYNC, 0);
		rt2x00_set_field16(&reg, TXRX_CSR19_BEACON_GEN, 1);
	} else if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) &&
		   !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED)) {
		rt2x00_set_field16(&reg, TXRX_CSR19_TSF_SYNC, 0);
		rt2x00_set_field16(&reg, TXRX_CSR19_BEACON_GEN, 0);
	}

	/*
	 * Beacon generation will fail initially.
	 * To prevent this we need to register the TXRX_CSR19
	 * register several times.
	 */
	rt2x00_register_write(rt2x00dev, TXRX_CSR19, reg);
	rt2x00_register_write(rt2x00dev, TXRX_CSR19, 0);
	rt2x00_register_write(rt2x00dev, TXRX_CSR19, reg);
	rt2x00_register_write(rt2x00dev, TXRX_CSR19, 0);
	rt2x00_register_write(rt2x00dev, TXRX_CSR19, reg);

	/*
	 * Change flags of enabled interfaces.
	 */
	if (type != IEEE80211_IF_TYPE_MNTR) {
		if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
			SET_FLAG(rt2x00dev, INTERFACE_ENABLED);
		else
			CLEAR_FLAG(rt2x00dev, INTERFACE_ENABLED);
	} else {
		if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR))
			SET_FLAG(rt2x00dev, INTERFACE_ENABLED_MONITOR);
		else
			CLEAR_FLAG(rt2x00dev, INTERFACE_ENABLED_MONITOR);
	}
}

static void rt2500usb_config_channel(struct rt2x00_dev *rt2x00dev,
	int rf2, int channel, int freq, int txpower)
{
	u32 rf1 = rt2x00dev->rf1;
	u32 rf3 = rt2x00dev->rf3;
	u32 rf4 = rt2x00dev->rf4;

	/*
	 * Only continue when there is something to be done.
	 */
	if (channel == rt2x00dev->rx_params.channel)
		return;

	if (txpower == 0xff)
		txpower = rt2x00dev->tx_power;
	txpower = TXPOWER_TO_DEV(txpower);

	if ((rt2x00_rf(&rt2x00dev->chip, RF2523) ||
	     rt2x00_rf(&rt2x00dev->chip, RF2524) ||
	     rt2x00_rf(&rt2x00dev->chip, RF2525)) &&
	     channel == 14)
		rf4 &= ~0x00000018;

	if (rt2x00_rf(&rt2x00dev->chip, RF2525E)) {
		if (channel & 0x01)
			rf4 = 0x00000e1b;
		else
			rf4 = 0x00000e07;
		if (channel == 14)
			rf4 = 0x00000e23;
	}

	if (rt2x00_rf(&rt2x00dev->chip, RF5222)) {
		if (channel < 14) {
			rf1 = 0x00022020;
			rf4 = 0x00000a0b;
		} else if (channel == 14) {
			rf1 = 0x00022010;
			rf4 = 0x00000a1b;
		} else if (channel < 64) {
			rf1 = 0x00022010;
			rf4 = 0x00000a1f;
		} else if (channel < 140) {
			rf1 = 0x00022010;
			rf4 = 0x00000a0f;
		} else if (channel < 161) {
			rf1 = 0x00022020;
			rf4 = 0x00000a07;
		}
	}

	/*
	 * Set TXpower.
	 */
	rt2x00_set_field32(&rf3, RF3_TXPOWER, txpower);

	INFO("Switching channel. RF1: 0x%08x, RF2: 0x%08x, RF3: 0x%08x, "
		"RF4: 0x%08x.\n", rf1, rf2, rf3, rf4);

	/*
	 * For RT2525E we should first set the channel to half band higher.
	 */
	if (rt2x00_rf(&rt2x00dev->chip, RF2525E)) {
		static const u32 vals[] = {
			0x000008aa, 0x000008ae, 0x000008ae, 0x000008b2,
			0x000008b2, 0x000008b6, 0x000008b6, 0x000008ba,
			0x000008ba, 0x000008be, 0x000008b7, 0x00000902,
			0x00000902, 0x00000906
		};

		rt2x00_rf_write(rt2x00dev, vals[channel - 1]);
		if (rf4)
			rt2x00_rf_write(rt2x00dev, rf4);
	}

	rt2x00_rf_write(rt2x00dev, rf1);
	rt2x00_rf_write(rt2x00dev, rf2);
	rt2x00_rf_write(rt2x00dev, rf3);
	if (rf4)
		rt2x00_rf_write(rt2x00dev, rf4);

	/*
	 * Update active info for RX.
	 */
	rt2x00dev->rx_params.freq = freq;
	rt2x00dev->rx_params.channel = channel;

	rt2x00dev->tx_power = txpower;

	/*
	 * Update rf fields
	 */
	rt2x00dev->rf1 = rf1;
	rt2x00dev->rf2 = rf2;
	rt2x00dev->rf3 = rf3;
	rt2x00dev->rf4 = rf4;

	rt2x00dev->tx_power = txpower;
}

static void rt2500usb_config_txpower(struct rt2x00_dev *rt2x00dev, int txpower)
{
	txpower = TXPOWER_TO_DEV(txpower);

	/*
	 * Only continue when there is something to be done.
	 */
	if (txpower == rt2x00dev->tx_power)
		return;

	rt2x00_set_field32(&rt2x00dev->rf3, RF3_TXPOWER, txpower);
	rt2x00_rf_write(rt2x00dev, rt2x00dev->rf3);

	rt2x00dev->tx_power = txpower;
}

static void rt2500usb_config_antenna(struct rt2x00_dev *rt2x00dev, int antenna)
{
	u8 reg_rx;
	u8 reg_tx;
	u16 csr5_reg;
	u16 csr6_reg;

	/*
	 * Only continue when there is something to be done.
	 */
	if (rt2x00dev->rx_params.antenna == antenna)
		return;

	rt2x00_bbp_read(rt2x00dev, 2, &reg_tx);
	rt2x00_bbp_read(rt2x00dev, 14, &reg_rx);
	rt2x00_register_read(rt2x00dev, PHY_CSR5, &csr5_reg);
	rt2x00_register_read(rt2x00dev, PHY_CSR6, &csr6_reg);

	/*
	 * Clear current config antenna bits.
	 */
	reg_tx &= ~0x03;
	reg_rx &= ~0x03;

	/*
	 * Ralink devices have have antenna options for both TX as RX.
	 * The ieee80211 stack currently only provide the user to set
	 * 1 antenna, by default this is considered to be the TX antenna.
	 */
	if (antenna == 0) {
		/* Diversity. */
		reg_rx |= 0x01;
		reg_tx |= 0x01;
		rt2x00_set_field16(&csr5_reg, PHY_CSR5_CCK, 1);
		rt2x00_set_field16(&csr6_reg, PHY_CSR6_OFDM, 1);
	} else if (antenna == 1) {
		/* RX: Antenna B */
		reg_rx |= 0x02;
		/* TX: Antenna A */
		reg_tx |= 0x00;
		rt2x00_set_field16(&csr5_reg, PHY_CSR5_CCK, 0);
		rt2x00_set_field16(&csr6_reg, PHY_CSR6_OFDM, 0);
	} else if (antenna == 2) {
		/* RX: Antenna A */
		reg_rx |= 0x00;
		/* TX: Antenna B */
		reg_tx |= 0x02;
		rt2x00_set_field16(&csr5_reg, PHY_CSR5_CCK, 2);
		rt2x00_set_field16(&csr6_reg, PHY_CSR6_OFDM, 2);
	}

	/*
	 * RT2525E and RT5222 need to flip TX I/Q
	 */
	if (rt2x00_rf(&rt2x00dev->chip, RF2525E) ||
	    rt2x00_rf(&rt2x00dev->chip, RF5222)) {
		reg_tx |= 0x04;
		rt2x00_set_field16(&csr5_reg, PHY_CSR5_CCK_FLIP, 1);
		rt2x00_set_field16(&csr6_reg, PHY_CSR6_OFDM_FLIP, 1);

		/*
		 * RT2525E does not need RX I/Q Flip.
		 */
		if (rt2x00_rf(&rt2x00dev->chip, RF2525E))
			reg_rx &= ~0x04;
	} else {
		rt2x00_set_field16(&csr5_reg, PHY_CSR5_CCK_FLIP, 0);
		rt2x00_set_field16(&csr6_reg, PHY_CSR6_OFDM_FLIP, 0);
	}

	rt2x00_bbp_write(rt2x00dev, 2, reg_tx);
	rt2x00_bbp_write(rt2x00dev, 14, reg_rx);
	rt2x00_register_write(rt2x00dev, PHY_CSR5, csr5_reg);
	rt2x00_register_write(rt2x00dev, PHY_CSR6, csr6_reg);

	/*
	 * Update active info for RX.
	 */
	rt2x00dev->rx_params.antenna = antenna;
}

static void rt2500usb_config_duration(struct rt2x00_dev *rt2x00dev,
	unsigned short short_slot_time)
{
	short_slot_time = short_slot_time ? SHORT_SLOT_TIME : SLOT_TIME;

	rt2x00_register_write(rt2x00dev, MAC_CSR10, short_slot_time);
}

static void rt2500usb_config_rate(struct rt2x00_dev *rt2x00dev, const int rate)
{
	struct ieee80211_conf *conf = &rt2x00dev->hw->conf;
	u16 reg;
	u16 value;
	u16 preamble;

	preamble = DEVICE_GET_RATE_FIELD(rate, PREAMBLE)
		? SHORT_PREAMBLE : PREAMBLE;

	reg = DEVICE_GET_RATE_FIELD(rate, RATEMASK) & DEV_BASIC_RATE;

	rt2x00_register_write(rt2x00dev, TXRX_CSR11, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR1, &reg);
	value = ((conf->flags & IEEE80211_CONF_SHORT_SLOT_TIME) ?
		 SHORT_DIFS :  DIFS) +
		PLCP + preamble + get_duration(ACK_SIZE, 10);
	rt2x00_set_field16(&reg, TXRX_CSR1_ACK_TIMEOUT, value);
	rt2x00_register_write(rt2x00dev, TXRX_CSR1, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR10, &reg);
	if (preamble == SHORT_PREAMBLE)
		rt2x00_set_field16(&reg, TXRX_CSR10_AUTORESPOND_PREAMBLE, 1);
	else
		rt2x00_set_field16(&reg, TXRX_CSR10_AUTORESPOND_PREAMBLE, 0);
	rt2x00_register_write(rt2x00dev, TXRX_CSR10, reg);
}

static void rt2500usb_config_phymode(struct rt2x00_dev *rt2x00dev,
	const int phymode)
{
	struct ieee80211_rate *rate;

	/*
	 * Only continue when there is something to be done.
	 */
	if (rt2x00dev->rx_params.phymode == phymode)
		return;

	if (phymode == MODE_IEEE80211A &&
	    rt2x00_rf(&rt2x00dev->chip, RF5222))
		rate = &rt2x00dev->hw->modes[2].rates[
			rt2x00dev->hw->modes[2].num_rates - 1];
	else if (phymode == MODE_IEEE80211B)
		rate = &rt2x00dev->hw->modes[1].rates[
			rt2x00dev->hw->modes[1].num_rates - 1];
	else
		rate = &rt2x00dev->hw->modes[0].rates[
			rt2x00dev->hw->modes[0].num_rates - 1];

	rt2500usb_config_rate(rt2x00dev, rate->val2);

	/*
	 * Update physical mode for rx ring.
	 */
	rt2x00dev->rx_params.phymode = phymode;
}

static void rt2500usb_config_mac_address(struct rt2x00_dev *rt2x00dev,
	void *addr)
{
	u16 reg[3] = { 0, 0, 0 };

	/*
	 * The MAC address is passed to us as an array of bytes,
	 * that array is little endian, so no need for byte ordering.
	 */
	memcpy(&reg, addr, ETH_ALEN);
	rt2x00_register_multiwrite(rt2x00dev, MAC_CSR2, &reg[0], sizeof(reg));
}

/*
 * Link tuning
 */
static void rt2500usb_link_tuner(void *data)
{
	struct rt2x00_dev *rt2x00dev = data;
	u16 reg;
	u32 rssi;
	u8 reg_r17;
	u8 up_bound;
	u8 low_bound;

	/*
	 * Don't perform any tuning when it is disabled
	 * in the EEPROM.
	 */
	if (GET_FLAG(rt2x00dev, CONFIG_DISABLE_LINK_TUNING))
		return;

	/*
	 * Retreive link quality.
	 */
	rssi = rt2x00_get_link_rssi(&rt2x00dev->link);
	if (!rssi)
		goto exit;

	low_bound = 0x32;
	if (rssi >= 43)
		up_bound = 0x40;
	else
		up_bound = 0x40 - (43 - rssi);
	if (up_bound < low_bound)
		up_bound = low_bound;

	if (rssi > 75) {
		rt2x00_bbp_write(rt2x00dev, 24, 0x70);
		rt2x00_bbp_write(rt2x00dev, 25, 0x40);
		rt2x00_bbp_write(rt2x00dev, 61, 0x6d);
	} else {
		rt2x00_bbp_write(rt2x00dev, 24, 0x80);
		rt2x00_bbp_write(rt2x00dev, 25, 0x50);
		rt2x00_bbp_write(rt2x00dev, 61, 0x60);
	}

	rt2x00_bbp_read(rt2x00dev, 17, &reg_r17);

	if (rssi > 80) {
		if (reg_r17 != 0x60)
			rt2x00_bbp_write(rt2x00dev, 17, 0x60);
		goto exit;
	} else if (rssi >= 62) {
		if (reg_r17 != 0x48)
			rt2x00_bbp_write(rt2x00dev, 17, 0x48);
		goto exit;
	} else if (rssi >= 46) {
		if (reg_r17 != 0x41)
			rt2x00_bbp_write(rt2x00dev, 17, 0x41);
		goto exit;
	} else if (reg_r17 > up_bound) {
		rt2x00_bbp_write(rt2x00dev, 17, up_bound);
		goto exit;
	}

	rt2x00_register_read(rt2x00dev, STA_CSR3, &reg);

	if (reg > 512 && reg_r17 < up_bound)
		rt2x00_bbp_write(rt2x00dev, 17, ++reg_r17);
	else if (reg < 100 && reg_r17 > low_bound)
		rt2x00_bbp_write(rt2x00dev, 17, --reg_r17);

exit:
	/*
	 * Update noise statistics.
	 */
	if (reg_r17)
		rt2x00_update_link_noise(&rt2x00dev->link, reg_r17);

	queue_delayed_work(rt2x00dev->workqueue, &rt2x00dev->link.work,
		LINK_TUNE_INTERVAL);
}

/*
 * LED functions.
 */
static void rt2500usb_enable_led(struct rt2x00_dev *rt2x00dev)
{
	u16 reg;

	rt2x00_register_read(rt2x00dev, MAC_CSR21, &reg);
	rt2x00_set_field16(&reg, MAC_CSR21_ON_PERIOD, 70);
	rt2x00_set_field16(&reg, MAC_CSR21_OFF_PERIOD, 30);
	rt2x00_register_write(rt2x00dev, MAC_CSR21, reg);

	rt2x00_register_read(rt2x00dev, MAC_CSR20, &reg);

	if (rt2x00dev->led_mode == LED_MODE_TXRX_ACTIVITY) {
		rt2x00_set_field16(&reg, MAC_CSR20_LINK, 1);
		rt2x00_set_field16(&reg, MAC_CSR20_ACTIVITY, 0);
	} else if (rt2x00dev->led_mode == LED_MODE_ASUS) {
		rt2x00_set_field16(&reg, MAC_CSR20_LINK, 0);
		rt2x00_set_field16(&reg, MAC_CSR20_ACTIVITY, 1);
	} else {
		rt2x00_set_field16(&reg, MAC_CSR20_LINK, 1);
		rt2x00_set_field16(&reg, MAC_CSR20_ACTIVITY, 1);
	}

	rt2x00_register_write(rt2x00dev, MAC_CSR20, reg);
}

static void rt2500usb_disable_led(struct rt2x00_dev *rt2x00dev)
{
	u16 reg;

	rt2x00_register_read(rt2x00dev, MAC_CSR20, &reg);
	rt2x00_set_field16(&reg, MAC_CSR20_LINK, 0);
	rt2x00_set_field16(&reg, MAC_CSR20_ACTIVITY, 0);
	rt2x00_register_write(rt2x00dev, MAC_CSR20, reg);
}

static void rt2500usb_activity_led(struct rt2x00_dev *rt2x00dev, char activity)
{
	u16 reg;

	if (rt2x00dev->led_mode != LED_MODE_TXRX_ACTIVITY)
		return;

	rt2x00_register_read(rt2x00dev, MAC_CSR20, &reg);
	rt2x00_set_field16(&reg, MAC_CSR20_ACTIVITY, activity);
	rt2x00_register_write(rt2x00dev, MAC_CSR20, reg);
}

/*
 * Device state switch.
 * This will put the device to sleep, or awake it.
 */
static int rt2500usb_set_state(struct rt2x00_dev *rt2x00dev,
	enum dev_state state)
{
	u16 reg;
	u16 reg2;
	unsigned int i;
	char put_to_sleep;
	char bbp_state;
	char rf_state;

	put_to_sleep = (state != STATE_AWAKE);

	reg = 0;
	rt2x00_set_field16(&reg, MAC_CSR17_BBP_DESIRE_STATE, state);
	rt2x00_set_field16(&reg, MAC_CSR17_RF_DESIRE_STATE, state);
	rt2x00_set_field16(&reg, MAC_CSR17_PUT_TO_SLEEP, put_to_sleep);
	rt2x00_register_write(rt2x00dev, MAC_CSR17, reg);
	rt2x00_set_field16(&reg, MAC_CSR17_SET_STATE, 1);
	rt2x00_register_write(rt2x00dev, MAC_CSR17, reg);

	/*
	 * Device is not guarenteed to be in the requested state yet.
	 * We must wait untill the register indicates that the
	 * device has entered the correct state.
	 */
	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_register_read(rt2x00dev, MAC_CSR17, &reg2);
		bbp_state = rt2x00_get_field16(reg2,
			MAC_CSR17_BBP_CURR_STATE);
		rf_state = rt2x00_get_field16(reg2,
			MAC_CSR17_RF_CURR_STATE);
		if (bbp_state == state && rf_state == state)
			return 0;
		rt2x00_register_write(rt2x00dev, MAC_CSR17, reg);
		msleep(30);
	}

	NOTICE("Device failed to enter state %d, "
		"current device state: bbp %d and rf %d.\n",
		state, bbp_state, rf_state);

	return -EBUSY;
}

/*
 * Initialization functions.
 */
static int rt2500usb_alloc_dma_ring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type, void (*handler)(void *),
	const u16 max_entries, const u16 data_size, const u16 desc_size)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	unsigned int i;
	int status;

	/*
	 * Initialize work structure for deferred work.
	 */
	INIT_WORK(&ring->irq_work, handler, ring);

	ring->stats.limit = max_entries;
	ring->data_size = data_size;
	ring->desc_size = desc_size;

	/*
	 * Allocate all ring entries.
	 */
	ring->entry = kzalloc(ring->stats.limit * sizeof(struct data_entry),
		GFP_KERNEL);
	if (!ring->entry)
		return -ENOMEM;

	/*
	 * Allocate DMA memory for descriptor and buffer.
	 */
	ring->data_addr = usb_buffer_alloc(
		interface_to_usbdev(rt2x00dev_usb(rt2x00dev)),
		rt2x00_get_ring_size(ring), GFP_KERNEL, &ring->data_dma);
	if (!ring->data_addr) {
		kfree(ring->entry);
		return -ENOMEM;
	}

	/*
	 * Initialize all ring entries to contain valid
	 * addresses.
	 */
	status = 0;
	for (i = 0; i < ring->stats.limit; i++) {
		ring->entry[i].flags = 0;
		ring->entry[i].ring = ring;
		ring->entry[i].priv =
			(!status) ? usb_alloc_urb(0, GFP_KERNEL) : NULL;
		if (!ring->entry[i].priv)
			status = -ENOMEM;
		ring->entry[i].skb = NULL;
		ring->entry[i].data_addr = ring->data_addr
			+ (i * ring->desc_size)
			+ (i * ring->data_size);
		ring->entry[i].data_dma = ring->data_dma
			+ (i * ring->desc_size)
			+ (i * ring->data_size);
	}

	return status;
}

static void rt2500usb_free_ring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	unsigned int i;

	if (!ring->entry)
		goto exit;

	for (i = 0; i < ring->stats.limit; i++) {
		usb_kill_urb(rt2x00_urb(&ring->entry[i]));
		usb_free_urb(rt2x00_urb(&ring->entry[i]));
	}

	kfree(ring->entry);
	ring->entry = NULL;

exit:
	if (ring->data_addr)
		usb_buffer_free(
			interface_to_usbdev(rt2x00dev_usb(rt2x00dev)),
			rt2x00_get_ring_size(ring), ring->data_addr,
			ring->data_dma);
	ring->data_addr = NULL;
}

static int rt2500usb_allocate_dma_rings(struct rt2x00_dev *rt2x00dev)
{
	if (rt2500usb_alloc_dma_ring(rt2x00dev, RING_RX,
		&rt2500usb_rxdone, RX_ENTRIES, DATA_FRAME_SIZE,
		RXD_DESC_SIZE) ||
	    rt2500usb_alloc_dma_ring(rt2x00dev, RING_TX,
		&rt2500usb_txdone, TX_ENTRIES, DATA_FRAME_SIZE,
		TXD_DESC_SIZE) ||
	    rt2500usb_alloc_dma_ring(rt2x00dev, RING_ATIM,
		&rt2500usb_txdone, ATIM_ENTRIES, DATA_FRAME_SIZE,
		TXD_DESC_SIZE) ||
	    rt2500usb_alloc_dma_ring(rt2x00dev, RING_PRIO,
		&rt2500usb_txdone, TX_ENTRIES, DATA_FRAME_SIZE,
		TXD_DESC_SIZE) ||
	    rt2500usb_alloc_dma_ring(rt2x00dev, RING_BEACON,
		&rt2500usb_beacondone, BEACON_ENTRIES, MGMT_FRAME_SIZE,
		TXD_DESC_SIZE)) {
		return -ENOMEM;
	}

	return 0;
}

static void rt2500usb_free_rings(struct rt2x00_dev *rt2x00dev)
{
	rt2500usb_free_ring(rt2x00dev, RING_RX);
	rt2500usb_free_ring(rt2x00dev, RING_TX);
	rt2500usb_free_ring(rt2x00dev, RING_ATIM);
	rt2500usb_free_ring(rt2x00dev, RING_PRIO);
	rt2500usb_free_ring(rt2x00dev, RING_BEACON);
}

static void rt2500usb_init_rxring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	struct usb_device *usb_dev =
		interface_to_usbdev(rt2x00dev_usb(rt2x00dev));
	struct urb *urb;
	unsigned int i;

	ring->type = ring_type;

	for (i = 0; i < ring->stats.limit; i++) {
		urb = rt2x00_urb(&ring->entry[i]);

		urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;
		urb->transfer_dma = ring->entry[i].data_dma;
		usb_fill_bulk_urb(
			urb,
			usb_dev,
			usb_rcvbulkpipe(usb_dev, 1),
			ring->entry[i].data_addr,
			ring->data_size + ring->desc_size,
			rt2500usb_interrupt,
			&ring->entry[i]);
	}

	rt2x00_ring_index_clear(ring);
}

static void rt2500usb_init_txring(struct rt2x00_dev *rt2x00dev,
	enum ring_index ring_type)
{
	struct data_ring *ring = &rt2x00dev->ring[ring_type];
	struct urb *urb;
	unsigned int i;

	ring->type = ring_type;

	for (i = 0; i < ring->stats.limit; i++) {
		urb = rt2x00_urb(&ring->entry[i]);

		urb->transfer_flags |= URB_NO_TRANSFER_DMA_MAP;
		urb->transfer_dma = ring->entry[i].data_dma;
		CLEAR_FLAGS(&ring->entry[i]);
	}

	rt2x00_ring_index_clear(ring);
}

static int rt2500usb_init_rings(struct rt2x00_dev *rt2x00dev)
{
	rt2500usb_init_rxring(rt2x00dev, RING_RX);
	rt2500usb_init_txring(rt2x00dev, RING_TX);
	rt2500usb_init_txring(rt2x00dev, RING_ATIM);
	rt2500usb_init_txring(rt2x00dev, RING_PRIO);
	rt2500usb_init_txring(rt2x00dev, RING_BEACON);

	return 0;
}

static int rt2500usb_init_registers(struct rt2x00_dev *rt2x00dev)
{
	u16 reg;

	if (rt2500usb_set_state(rt2x00dev, STATE_AWAKE))
		return -EBUSY;

	rt2x00_vendor_request(rt2x00dev, USB_DEVICE_MODE,
		USB_VENDOR_REQUEST_OUT, 0x00, USB_MODE_TEST, NULL, 0,
		REGISTER_TIMEOUT);
	rt2x00_vendor_request(rt2x00dev, USB_SINGLE_WRITE,
		USB_VENDOR_REQUEST_OUT, 0x0308, 0xf0, NULL, 0,
		REGISTER_TIMEOUT);

	rt2x00_register_write(rt2x00dev, MAC_CSR13, 0x1111);
	rt2x00_register_write(rt2x00dev, MAC_CSR14, 0x1e11);

	rt2x00_register_write(rt2x00dev, TXRX_CSR2, 0x0001);
	rt2x00_register_write(rt2x00dev, MAC_CSR1, 0x0003);
	rt2x00_register_write(rt2x00dev, MAC_CSR1, 0x0000);
	rt2x00_register_write(rt2x00dev, TXRX_CSR5, 0x8c8d);
	rt2x00_register_write(rt2x00dev, TXRX_CSR6, 0x8b8a);
	rt2x00_register_write(rt2x00dev, TXRX_CSR7, 0x8687);
	rt2x00_register_write(rt2x00dev, TXRX_CSR8, 0x0085);
	rt2x00_register_write(rt2x00dev, TXRX_CSR21, 0xe78f);
	rt2x00_register_write(rt2x00dev, MAC_CSR9, 0xff1d);
	rt2x00_register_write(rt2x00dev, MAC_CSR1, 0x0004);

	reg = 0;
	rt2x00_register_read(rt2x00dev, MAC_CSR0, &reg);
	if (reg >= 0x0003) {
		rt2x00_register_read(rt2x00dev, PHY_CSR2, &reg);
		reg &= ~0x0002;
	} else {
		reg = 0x3002;
	}
	rt2x00_register_write(rt2x00dev, PHY_CSR2, reg);

	rt2x00_register_write(rt2x00dev, MAC_CSR11, 0x0005);
	rt2x00_register_write(rt2x00dev, MAC_CSR22, 0x0053);
	rt2x00_register_write(rt2x00dev, MAC_CSR15, 0x01ee);
	rt2x00_register_write(rt2x00dev, MAC_CSR16, 0);

	rt2x00_register_read(rt2x00dev, TXRX_CSR0, &reg);
	rt2x00_set_field16(&reg, TXRX_CSR0_IV_OFFSET, IEEE80211_HEADER);
	rt2x00_set_field16(&reg, TXRX_CSR0_KEY_ID, 0xff);
	rt2x00_register_write(rt2x00dev, TXRX_CSR0, reg);

	rt2x00_register_read(rt2x00dev, MAC_CSR8, &reg);
	rt2x00_set_field16(&reg, MAC_CSR8_MAX_FRAME_UNIT,
		rt2x00dev->ring[RING_RX].data_size);
	rt2x00_register_write(rt2x00dev, MAC_CSR8, reg);

	rt2x00_register_read(rt2x00dev, MAC_CSR18, &reg);
	rt2x00_set_field16(&reg, MAC_CSR18_DELAY_AFTER_BEACON, 0x5a);
	rt2x00_register_write(rt2x00dev, MAC_CSR18, reg);

	rt2x00_register_read(rt2x00dev, TXRX_CSR1, &reg);
	rt2x00_set_field16(&reg, TXRX_CSR1_AUTO_SEQUENCE, 1);
	rt2x00_register_write(rt2x00dev, TXRX_CSR1, reg);

	rt2x00_register_read(rt2x00dev, PHY_CSR4, &reg);
	rt2x00_register_write(rt2x00dev, PHY_CSR4, reg | 0x0001);

	return 0;
}

static int rt2500usb_init_bbp(struct rt2x00_dev *rt2x00dev)
{
	unsigned int i;
	u16 eeprom;
	u8 reg_id;
	u8 value;

	for (i = 0; i < REGISTER_BUSY_COUNT; i++) {
		rt2x00_bbp_read(rt2x00dev, 0, &value);
		if ((value != 0xff) && (value != 0x00))
			goto continue_csr_init;
		NOTICE("Waiting for BBP register.\n");
	}

	ERROR("BBP register access failed, aborting.\n");
	return -EACCES;

continue_csr_init:
	rt2x00_bbp_write(rt2x00dev, 3, 0x02);
	rt2x00_bbp_write(rt2x00dev, 4, 0x19);
	rt2x00_bbp_write(rt2x00dev, 14, 0x1c);
	rt2x00_bbp_write(rt2x00dev, 15, 0x30);
	rt2x00_bbp_write(rt2x00dev, 16, 0xac);
	rt2x00_bbp_write(rt2x00dev, 17, 0x48);
	rt2x00_bbp_write(rt2x00dev, 18, 0x18);
	rt2x00_bbp_write(rt2x00dev, 19, 0xff);
	rt2x00_bbp_write(rt2x00dev, 20, 0x1e);
	rt2x00_bbp_write(rt2x00dev, 21, 0x08);
	rt2x00_bbp_write(rt2x00dev, 22, 0x08);
	rt2x00_bbp_write(rt2x00dev, 23, 0x08);
	rt2x00_bbp_write(rt2x00dev, 24, 0x80);
	rt2x00_bbp_write(rt2x00dev, 25, 0x50);
	rt2x00_bbp_write(rt2x00dev, 26, 0x08);
	rt2x00_bbp_write(rt2x00dev, 27, 0x23);
	rt2x00_bbp_write(rt2x00dev, 30, 0x10);
	rt2x00_bbp_write(rt2x00dev, 31, 0x2b);
	rt2x00_bbp_write(rt2x00dev, 32, 0xb9);
	rt2x00_bbp_write(rt2x00dev, 34, 0x12);
	rt2x00_bbp_write(rt2x00dev, 35, 0x50);
	rt2x00_bbp_write(rt2x00dev, 39, 0xc4);
	rt2x00_bbp_write(rt2x00dev, 40, 0x02);
	rt2x00_bbp_write(rt2x00dev, 41, 0x60);
	rt2x00_bbp_write(rt2x00dev, 53, 0x10);
	rt2x00_bbp_write(rt2x00dev, 54, 0x18);
	rt2x00_bbp_write(rt2x00dev, 56, 0x08);
	rt2x00_bbp_write(rt2x00dev, 57, 0x10);
	rt2x00_bbp_write(rt2x00dev, 58, 0x08);
	rt2x00_bbp_write(rt2x00dev, 61, 0x60);
	rt2x00_bbp_write(rt2x00dev, 62, 0x10);
	rt2x00_bbp_write(rt2x00dev, 75, 0xff);

	DEBUG("Start initialization from EEPROM...\n");
	for (i = 0; i < EEPROM_BBP_SIZE; i++) {
		rt2x00_eeprom_read(rt2x00dev, EEPROM_BBP_START + i, &eeprom);

		if (eeprom != 0xffff && eeprom != 0x0000) {
			reg_id = rt2x00_get_field16(eeprom, EEPROM_BBP_REG_ID);
			value = rt2x00_get_field16(eeprom, EEPROM_BBP_VALUE);
			DEBUG("BBP: 0x%02x, value: 0x%02x.\n", reg_id, value);
			rt2x00_bbp_write(rt2x00dev, reg_id, value);
		}
	}
	DEBUG("...End initialization from EEPROM.\n");

	return 0;
}

static int rt2500usb_init_channel_time(struct rt2x00_dev *rt2x00dev)
{
	unsigned long jiffies_start;
	unsigned long jiffies_end;

	/*
	 * Only initialize the channel_change_time
	 * if it has not been set previously.
	 */
	if (rt2x00dev->hw->channel_change_time)
		return 0;

	/*
	 * Invalidate the rx_params.channel value to make sure
	 * the config channel will be correctly executed.
	 */
	rt2x00dev->rx_params.channel = 0;

	/*
	 * Determine channel_change_time
	 * by measuring the time it takes
	 * to switch the channel.
	 */
	jiffies_start = jiffies;
	rt2500usb_config_channel(rt2x00dev,
		rt2x00dev->hw->modes[0].channels[0].val,
		rt2x00dev->hw->modes[0].channels[0].chan,
		rt2x00dev->hw->modes[0].channels[0].freq,
		rt2x00dev->hw->modes[0].channels[0].power_level);
	jiffies_end = jiffies;

	rt2x00dev->hw->channel_change_time =
		jiffies_to_usecs((long)jiffies_end - (long)jiffies_start);

	NOTICE("Channel change time has been set to %d.\n",
		rt2x00dev->hw->channel_change_time);

	return 0;
}

/*
 * Device initialization functions.
 */
static int rt2500usb_initialize(struct rt2x00_dev *rt2x00dev)
{
	if (GET_FLAG(rt2x00dev, DEVICE_INITIALIZED))
		return 0;

	/*
	 * Allocate all data rings.
	 */
	if (rt2500usb_allocate_dma_rings(rt2x00dev)) {
		ERROR("DMA allocation failed.\n");
		goto exit_fail;
	}

	/*
	 * Reset the channel_change_time value
	 * to make sure it will be correctly initialized
	 * after the radio has been enabled.
	 */
	rt2x00dev->hw->channel_change_time = 0;

	SET_FLAG(rt2x00dev, DEVICE_INITIALIZED);

	return 0;

exit_fail:
	rt2500usb_free_rings(rt2x00dev);

	return -EIO;
}

static void rt2500usb_uninitialize(struct rt2x00_dev *rt2x00dev)
{
	if (!GET_FLAG(rt2x00dev, DEVICE_INITIALIZED))
		return;

	/*
	 * Cancel scanning.
	 */
	if (rt2x00dev->scan)
		rt2x00_signal_scan(rt2x00dev->scan, SCANNING_CANCELLED);

	/*
	 * Flush out all pending work.
	 */
	flush_workqueue(rt2x00dev->workqueue);

	/*
	 * Free DMA rings.
	 */
	rt2500usb_free_rings(rt2x00dev);

	CLEAR_FLAG(rt2x00dev, DEVICE_INITIALIZED);
}

/*
 * Radio control functions.
 */
static int rt2500usb_enable_radio(struct rt2x00_dev *rt2x00dev)
{
	struct data_ring *ring;
	u16 reg;
	unsigned int i;

	/*
	 * Don't enable the radio twice.
	 */
	if (GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO))
		return 0;

	/*
	 * Check if the hardware has been initialized,
	 * if not then do it now.
	 */
	if (!GET_FLAG(rt2x00dev, DEVICE_INITIALIZED))
		if (rt2500usb_initialize(rt2x00dev))
			return -EIO;

	/*
	 * Initialize all registers.
	 */
	if (rt2500usb_init_rings(rt2x00dev) ||
	    rt2500usb_init_registers(rt2x00dev) ||
	    rt2500usb_init_bbp(rt2x00dev)) {
		ERROR("Register initialization failed.\n");
		goto exit_fail;
	}

	/*
	 * Determine channel change time.
	 */
	if (rt2500usb_init_channel_time(rt2x00dev))
		goto exit_fail;

	SET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO);

	/*
	 * Enable RX.
	 */
	rt2x00_register_read(rt2x00dev, TXRX_CSR2, &reg);
	rt2x00_set_field16(&reg, TXRX_CSR2_DISABLE_RX, 0);
	rt2x00_register_write(rt2x00dev, TXRX_CSR2, reg);

	ring = &rt2x00dev->ring[RING_RX];
	for (i = 0; i < ring->stats.limit; i++) {
		SET_FLAG(&ring->entry[i], ENTRY_OWNER_NIC);
		usb_submit_urb(rt2x00_urb(&ring->entry[i]), GFP_ATOMIC);
	}

	/*
	 * Enable LED
	 */
	rt2500usb_enable_led(rt2x00dev);

	ieee80211_start_queues(rt2x00dev->hw);
	ieee80211_netif_oper(rt2x00dev->hw, NETIF_WAKE);

	return 0;

exit_fail:
	rt2500usb_uninitialize(rt2x00dev);
	return -EIO;
}

static void rt2500usb_disable_radio(struct rt2x00_dev *rt2x00dev)
{
	struct data_ring *ring;
	u16 reg;
	unsigned int i;

	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO))
		return;

	ieee80211_netif_oper(rt2x00dev->hw, NETIF_STOP);
	ieee80211_stop_queues(rt2x00dev->hw);

	/*
	 * Disable LED
	 */
	rt2500usb_disable_led(rt2x00dev);

	CLEAR_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO);

	rt2x00_register_write(rt2x00dev, MAC_CSR13, 0x2121);
	rt2x00_register_write(rt2x00dev, MAC_CSR14, 0x2121);

	/*
	 * Disable synchronisation.
	 */
	rt2x00_register_write(rt2x00dev, TXRX_CSR19, 0);

	/*
	 * Cancel RX and TX.
	 */
	rt2x00_register_read(rt2x00dev, TXRX_CSR2, &reg);
	rt2x00_set_field16(&reg, TXRX_CSR2_DISABLE_RX, 1);
	rt2x00_register_write(rt2x00dev, TXRX_CSR2, reg);

	rt2x00_vendor_request(rt2x00dev, USB_RX_CONTROL,
		USB_VENDOR_REQUEST_OUT, 0x00, 0x00, NULL, 0, REGISTER_TIMEOUT);

	ring = &rt2x00dev->ring[RING_RX];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00_urb(&ring->entry[i]));

	ring = &rt2x00dev->ring[RING_TX];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00_urb(&ring->entry[i]));

	ring = &rt2x00dev->ring[RING_ATIM];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00_urb(&ring->entry[i]));

	ring = &rt2x00dev->ring[RING_PRIO];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00_urb(&ring->entry[i]));

	ring = &rt2x00dev->ring[RING_BEACON];
	for (i = 0; i < ring->stats.limit; i++)
		usb_kill_urb(rt2x00_urb(&ring->entry[i]));
}

/*
 * RTS frame creation.
 */
static struct sk_buff* rt2500usb_create_rts(struct rt2x00_dev *rt2x00dev,
	struct ieee80211_hdr *hdr, unsigned short duration)
{
	struct ieee80211_hdr *ieee80211hdr;
	struct sk_buff *skb;
	u16 frame_control;

	skb = dev_alloc_skb(IEEE80211_HEADER);
	if (!skb)
		return NULL;

	/*
	 * Copy the entire header over to RTS frame.
	 */
	memcpy(skb_put(skb, IEEE80211_HEADER), hdr, IEEE80211_HEADER);
	ieee80211hdr = (struct ieee80211_hdr*)skb->data;

	frame_control = IEEE80211_FTYPE_CTL | IEEE80211_STYPE_RTS;
	ieee80211hdr->frame_control = cpu_to_le16(frame_control);

	ieee80211hdr->duration_id = cpu_to_le16(duration);

	ieee80211hdr->seq_ctrl = 0;

	return skb;
}

/*
 * TX descriptor initialization
 */
static void rt2500usb_write_tx_desc(struct rt2x00_dev *rt2x00dev,
	struct data_desc *txd, struct sk_buff *skb,
	struct ieee80211_tx_control *control)
{
	struct ieee80211_hdr *ieee80211hdr = (struct ieee80211_hdr*)skb->data;
	struct data_ring *ring;
	int tx_rate;
	u32 word;
	u32 length;
	u32 residual;
	u16 length_high;
	u16 length_low;
	u16 frame_control;
	u16 seq_ctrl;
	char rts_frame;
	char ofdm_rate;
	char req_timestamp;
	char more_frag;
	char new_seq;
	char ifs;
	u8 signal;
	u8 service;
	u8 bitrate;

	/*
	 * We require the ring structure this packet is being send to.
	 */
	ring = rt2x00_get_ring(rt2x00dev, control->queue);
	if (unlikely(!ring))
		return;

	/*
	 * Read required fields from ieee80211 header.
	 */
	frame_control = le16_to_cpu(ieee80211hdr->frame_control);
	seq_ctrl = le16_to_cpu(ieee80211hdr->seq_ctrl);

	/*
	 * Check if this frame is a RTS frame.
	 */
	rts_frame = is_rts_frame(frame_control);

	/*
	 * Check which rate should be used for this frame.
	 */
	if (rts_frame && control->rts_cts_rate)
		tx_rate = control->rts_cts_rate;
	else
		tx_rate = control->tx_rate;

	/*
	 * Are we working with OFDM rates.
	 */
	ofdm_rate = !!(DEVICE_GET_RATE_FIELD(tx_rate, RATEMASK) &
		       DEV_OFDM_RATE);

	/*
	 * Check if more fragments will follow this frame.
	 */
	more_frag = !!(ieee80211_get_morefrag(ieee80211hdr));

	/*
	 * Check if this is the first frame in a new sequence.
	 */
	new_seq = !!((seq_ctrl & IEEE80211_SCTL_FRAG) == 0);

	/*
	 * Beacons and probe responses require the tsf timestamp
	 * to be inserted into the frame.
	 */
	req_timestamp = !!(control->queue == IEEE80211_TX_QUEUE_BEACON ||
		  	   control->pkt_type == PKT_PROBE_RESP);

	/*
	 * Determine with what IFS priority this frame should be send.
	 * Set ifs to IFS_SIFS when the this is not the first fragment,
	 * or this fragment came after RTS/CTS.
	 */
	if (((seq_ctrl & IEEE80211_SCTL_FRAG) > 0) || rts_frame)
		ifs = IFS_SIFS;
	else
		ifs = IFS_BACKOFF;

	/*
	 * Add 4 bytes for FCS.
	 */
	length = skb->len + FCS_LEN;

	/*
	 * How the length should be processed depends
	 * on if we are working with OFDM rates or not.
	 */
	if (ofdm_rate) {
		residual = 0;
		length_high = (length >> 6) & 0x3f;
		length_low = (length & 0x3f);

	} else {
		bitrate = DEVICE_GET_RATE_FIELD(tx_rate, RATE);

		/*
		 * Convert length to microseconds.
		 */
		residual = get_duration_res(length, bitrate);
		length = get_duration(length, bitrate);

		if (residual != 0)
			length++;

		length_high = length >> 8;
		length_low = length & 0xff;
	}

	/*
	 * Create the signal and service values.
	 */
	signal = DEVICE_GET_RATE_FIELD(tx_rate, PLCP);
	if (DEVICE_GET_RATE_FIELD(tx_rate, PREAMBLE))
		signal |= 0x08;

	service = 0x04;
	if (residual <= (8 % 11))
		service |= 0x80;

	/*
	 * Start writing the descriptor words.
	 */
	rt2x00_desc_read(txd, 1, &word);
	rt2x00_set_field32(&word, TXD_W1_IV_OFFSET, IEEE80211_HEADER);
	rt2x00_set_field32(&word, TXD_W1_AIFS, ring->tx_params.aifs);
	rt2x00_set_field32(&word, TXD_W1_CWMIN, ring->tx_params.cw_min);
	rt2x00_set_field32(&word, TXD_W1_CWMAX, ring->tx_params.cw_max);
	rt2x00_desc_write(txd, 1, word);

	rt2x00_desc_read(txd, 2, &word);
	rt2x00_set_field32(&word, TXD_W2_PLCP_SIGNAL, signal);
	rt2x00_set_field32(&word, TXD_W2_PLCP_SERVICE, service);
	rt2x00_set_field32(&word, TXD_W2_PLCP_LENGTH_LOW, length_low);
	rt2x00_set_field32(&word, TXD_W2_PLCP_LENGTH_HIGH, length_high);
	rt2x00_desc_write(txd, 2, word);

	rt2x00_desc_read(txd, 0, &word);
	rt2x00_set_field32(&word, TXD_W0_RETRY_LIMIT, control->retry_limit);
	rt2x00_set_field32(&word, TXD_W0_MORE_FRAG, more_frag);
	rt2x00_set_field32(&word, TXD_W0_ACK,
		!(control->flags & IEEE80211_TXCTL_NO_ACK));
	rt2x00_set_field32(&word, TXD_W0_TIMESTAMP, req_timestamp);
	rt2x00_set_field32(&word, TXD_W0_OFDM, ofdm_rate);
	rt2x00_set_field32(&word, TXD_W0_NEW_SEQ, new_seq);
	rt2x00_set_field32(&word, TXD_W0_IFS, ifs);
	rt2x00_set_field32(&word, TXD_W0_DATABYTE_COUNT, skb->len);
	rt2x00_set_field32(&word, TXD_W0_CIPHER, CIPHER_NONE);
	rt2x00_desc_write(txd, 0, word);
}

/*
 * Interrupt functions.
 */
static void rt2500usb_beacondone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_dev *rt2x00dev = ring->rt2x00dev;
	struct data_entry *entry = rt2x00_get_data_entry(
		&rt2x00dev->ring[RING_BEACON]);
	struct sk_buff *skb;

	skb = ieee80211_beacon_get(rt2x00dev->hw,
		rt2x00dev->interface.id, &entry->tx_status.control);
	if (!skb)
		return;

	rt2500usb_beacon_update(rt2x00dev->hw, skb, &entry->tx_status.control);

	dev_kfree_skb_any(skb);
}

static void rt2500usb_rxdone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_dev *rt2x00dev = ring->rt2x00dev;
	struct data_entry *entry;
	struct sk_buff *skb;
	struct data_desc *rxd;
	struct urb *urb;
	u32 word0;
	u32 word1;
	u16 size;

	while (1) {
		entry = rt2x00_get_data_entry(ring);
		rxd = rt2x00_rxdesc_addr(entry);
		urb = rt2x00_urb(entry);
		rt2x00_desc_read(rxd, 0, &word0);
		rt2x00_desc_read(rxd, 1, &word1);

		if (GET_FLAG(entry, ENTRY_OWNER_NIC))
			break;

		/*
		 * There has been a problem. Ignore packet.
		 */
		if (urb->status) {
			SET_FLAG(entry, ENTRY_OWNER_NIC);
			usb_submit_urb(urb, GFP_ATOMIC);
			rt2x00_ring_index_inc(ring);
			continue;
		}

		/*
		 * Received USB packets have 4 bytes of extra data.
		 */
		size = rt2x00_get_field32(word0, RXD_W0_DATABYTE_COUNT) - 4;

		/*
		 * TODO: Don't we need to keep statistics
		 * updated about events like CRC and physical errors?
		 */
		if (!rt2x00_get_field32(word0, RXD_W0_CRC) &&
		    !rt2x00_get_field32(word0, RXD_W0_PHYSICAL_ERROR)) {
			skb = dev_alloc_skb(size + NET_IP_ALIGN);
			if (!skb)
				break;

			skb_reserve(skb, NET_IP_ALIGN);

			memcpy(skb_put(skb, size),
				rt2x00_rxdata_addr(entry), size);

			rt2x00dev->rx_params.rate = device_signal_to_rate(
				&rt2x00dev->hw->modes[1],
				rt2x00_get_field32(word1, RXD_W1_SIGNAL),
				rt2x00_get_field32(word0, RXD_W0_OFDM));

			rt2x00dev->rx_params.ssi =
				rt2x00_get_field32(word1, RXD_W1_RSSI);

			/*
			 * The following set of assignments supply
			 * additional data for wireless statistics. The
			 * necessary quantities are (1) the "noise" value
			 * in dBm, and (2) the "signal" value in dBm.
			 */
			rt2x00dev->rx_params.noise =
				rt2x00_get_link_noise(&rt2x00dev->link);
			rt2x00dev->rx_params.signal = rt2x00dev->rx_params.ssi
				- rt2x00dev->hw->maxssi;

			__ieee80211_rx(rt2x00dev->hw,
				skb, &rt2x00dev->rx_params);

			/*
			 * Update link statistics
			 */
			rt2x00_update_link_rssi(&rt2x00dev->link,
				rt2x00dev->rx_params.ssi);
		}

		SET_FLAG(entry, ENTRY_OWNER_NIC);
		usb_submit_urb(urb, GFP_ATOMIC);

		rt2x00_ring_index_inc(ring);
	}

	/*
	 * Update LED.
	 */
	rt2500usb_activity_led(rt2x00dev, 0);
}

static void rt2500usb_txdone(void *data)
{
	struct data_ring *ring = data;
	struct rt2x00_dev *rt2x00dev = ring->rt2x00dev;
	struct data_entry *entry;
	struct data_desc *txd;
	struct urb *urb;
	u32 word;
	int tx_status;
	int ack;
	int rts;

	 while (!rt2x00_ring_empty(ring)) {
		entry = rt2x00_get_data_entry_done(ring);
		txd = rt2x00_txdesc_addr(entry);
		urb = rt2x00_urb(entry);
		rt2x00_desc_read(txd, 0, &word);

		if (GET_FLAG(entry, ENTRY_OWNER_NIC))
			break;

		entry->tx_status.flags = 0;
		entry->tx_status.queue_length = entry->ring->stats.limit;
		entry->tx_status.queue_number = entry->tx_status.control.queue;

		/*
		 * Check if we have received an
		 * ACK response when ACK was requested and status
		 * was succesfull.
		 */
		ack = rt2x00_get_field32(word, TXD_W0_ACK);
		rts = GET_FLAG(entry, ENTRY_RTS_FRAME);
		tx_status = !urb->status ? TX_SUCCESS : TX_FAIL_RETRY;
		rt2x00_update_tx_stats(rt2x00dev, &entry->tx_status, tx_status,
				       ack, rts);

		rt2x00_bbp_read(rt2x00dev, 0,
			(u8*)&entry->tx_status.ack_signal);

		/*
		 * If this is not an RTS frame send the tx_status to d80211,
		 * that method also cleans up the skb structure. When this
		 * is a RTS frame, that it is our job to clean this structure up.
		 */
		if (!rts)
			ieee80211_tx_status(rt2x00dev->hw,
				entry->skb, &entry->tx_status);
		else
			dev_kfree_skb(entry->skb);

		CLEAR_FLAG(entry, ENTRY_RTS_FRAME);
		entry->skb = NULL;

		rt2x00_ring_index_done_inc(entry->ring);
	}

	/*
	 * Check if we are waiting on an empty queue
	 * to start scanning.
	 */
	if (rt2x00dev->scan &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_TX]) &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_ATIM]) &&
	    rt2x00_ring_empty(&rt2x00dev->ring[RING_PRIO]))
		rt2x00_signal_scan(rt2x00dev->scan, SCANNING_READY);

	/*
	 * If the data ring was full before the txdone handler
	 * we must make sure the packet queue in the d80211 stack
	 * is reenabled when the txdone handler has finished.
	 */
	entry = ring->entry;
	if (!rt2x00_ring_full(ring))
		ieee80211_wake_queue(rt2x00dev->hw,
			entry->tx_status.control.queue);
}

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2, 6, 19)
static void rt2500usb_interrupt(struct urb *urb)
#else
static void rt2500usb_interrupt(struct urb *urb, struct pt_regs *regs)
#endif
{
	struct data_entry *entry = (struct data_entry*)urb->context;
	struct rt2x00_dev *rt2x00dev = entry->ring->rt2x00dev;

	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO))
		return;

	CLEAR_FLAG(entry, ENTRY_OWNER_NIC);

	if (urb->status)
		return;

	if (entry->ring->type == RING_RX)
		rt2500usb_activity_led(rt2x00dev, 1);

	queue_work(rt2x00dev->workqueue, &entry->ring->irq_work);
}

/*
 * IEEE80211 stack callback functions.
 */
static int rt2500usb_tx(struct ieee80211_hw *hw,
	struct sk_buff *skb, struct ieee80211_tx_control *control)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	struct ieee80211_hdr *ieee80211hdr = (struct ieee80211_hdr*)skb->data;
	struct usb_device *usb_dev =
		interface_to_usbdev(rt2x00dev_usb(rt2x00dev));
	struct data_ring *ring;
	struct data_entry *entry;
	struct data_desc *txd;
	struct sk_buff *skb_rts;
	u16 frame_control;
	int res;
	int length;

	/*
	 * Determine which ring to put packet on.
	 */
	ring = rt2x00_get_ring(rt2x00dev, control->queue);
	if (unlikely(!ring)) {
		ERROR("Attempt to send packet over invalid queue %d.\n"
			"Please file bug report to %s.\n",
			control->queue, DRV_PROJECT);
		dev_kfree_skb_any(skb);
		return NETDEV_TX_OK;
	}

	if (rt2x00_ring_full(ring)) {
		ieee80211_stop_queue(hw, control->queue);
		return NETDEV_TX_BUSY;
	}

	/*
	 * If RTS is required. and this frame is not RTS,
	 * create and queue that frame first.
	 */
	frame_control = le16_to_cpu(ieee80211hdr->frame_control);
	if ((control->flags & IEEE80211_TXCTL_USE_RTS_CTS) &&
	    !is_rts_frame(frame_control)) {
		skb_rts = rt2500usb_create_rts(rt2x00dev,
				ieee80211hdr, control->rts_cts_duration);
		if (!skb_rts) {
			WARNING("Failed to create RTS frame.\n");
			return NETDEV_TX_BUSY;
		}

		res = rt2500usb_tx(hw, skb_rts, control);
		if (res) {
			WARNING("Failed to send RTS frame.\n");
			return res;
		}
	}

	entry = rt2x00_get_data_entry(ring);
	txd = rt2x00_txdesc_addr(entry);

	if (GET_FLAG(entry, ENTRY_OWNER_NIC)) {
		ERROR("Arrived at non-free entry in the non-full queue %d.\n"
			"Please file bug report to %s.\n",
			control->queue, DRV_PROJECT);
		ieee80211_stop_queue(hw, control->queue);
		return NETDEV_TX_BUSY;
	}

	if (entry->flags)
		ERROR("Arrived at broken entry (flags 0x%08x) in queue %d.\n"
			"Please file bug report to %s.\n",
			control->queue, entry->flags, DRV_PROJECT);

	memcpy(rt2x00_txdata_addr(entry), skb->data, skb->len);
	rt2500usb_write_tx_desc(rt2x00dev, txd, skb, control);
	memcpy(&entry->tx_status.control, control, sizeof(*control));
	if (is_rts_frame(frame_control))
		SET_FLAG(entry, ENTRY_RTS_FRAME);
	entry->skb = skb;

	/*
	 * Length passed to usb_fill_urb cannot be an odd number,
	 * so add 1 byte to make it even.
	 */
	length = skb->len + ring->desc_size;
	if (length % 2)
		length++;

	SET_FLAG(entry, ENTRY_OWNER_NIC);
	usb_fill_bulk_urb(
		rt2x00_urb(entry),
		usb_dev,
		usb_sndbulkpipe(usb_dev, 1),
		entry->data_addr,
		length,
		rt2500usb_interrupt,
		entry);
	usb_submit_urb(rt2x00_urb(entry), GFP_ATOMIC);

	rt2x00_ring_index_inc(ring);

	if (rt2x00_ring_full(ring))
		ieee80211_stop_queue(hw, control->queue);

	return NETDEV_TX_OK;
}

static int rt2500usb_reset(struct ieee80211_hw *hw)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	rt2500usb_disable_radio(rt2x00dev);
	return rt2500usb_enable_radio(rt2x00dev);
}

static int rt2500usb_add_interface(struct ieee80211_hw *hw,
	struct ieee80211_if_init_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	int status;

	/*
	 * We only support 1 non-monitor interface.
	 */
	if (conf->type != IEEE80211_IF_TYPE_MNTR &&
	    GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
		return -ENOBUFS;

	/*
	 * Add the new interface.
	 */
	rt2x00_add_interface(&rt2x00dev->interface, conf);

	/*
	 * Initialize interface, and enable the radio when this
	 * is the first interface that is brought up.
	 */
	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO)) {
		/*
		 * Before doing anything else, the MAC address
		 * of this device should be initialized correctly.
		 */
		rt2500usb_config_mac_address(rt2x00dev, conf->mac_addr);

		/*
		 * Initialize the device.
		 */
		status = rt2500usb_initialize(rt2x00dev);
		if (status)
			return status;

		/*
		 * Enable radio.
		 */
		status = rt2500usb_enable_radio(rt2x00dev);
		if (status)
			return status;

		/*
		 * Enable periodic link tuning if this is a non-monitor
		 * interface. Also set the INTERFACE_INITIALIZED FLAG
		 * to prevent new non-monitor interfaces to be added.
		 */
		if (conf->type != IEEE80211_IF_TYPE_MNTR) {
			queue_delayed_work(rt2x00dev->workqueue,
				&rt2x00dev->link.work, LINK_TUNE_INTERVAL);
			SET_FLAG(rt2x00dev, INTERFACE_INITIALIZED);
		} else
			SET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR);
	}

	return 0;
}

static void rt2500usb_remove_interface(struct ieee80211_hw *hw,
	struct ieee80211_if_init_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	/*
	 * We only support 1 non-monitor interface.
	 */
	if (conf->type != IEEE80211_IF_TYPE_MNTR &&
	    !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
		return;

	/*
	 * Remove the interface.
	 */
	rt2x00_remove_interface(&rt2x00dev->interface, conf);

	/*
	 * When this is a non-monitor mode,
	 * stop the periodic link tuning,
	 * and clear the INTERFACE_INITIALIZED FLAG to allow
	 * new non-monitor interfaces to be added.
	 */
	if (conf->type != IEEE80211_IF_TYPE_MNTR) {
		cancel_rearming_delayed_workqueue(rt2x00dev->workqueue,
			&rt2x00dev->link.work);
		CLEAR_FLAG(rt2x00dev, INTERFACE_INITIALIZED);
	}

	/*
	 * Disable radio if this was the last interface
	 * that was working with this device.
	 */
	if (!GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) &&
	    !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
		rt2500usb_disable_radio(rt2x00dev);

	/*
	 * Check if we still have 1 non-monitor or a monitor
	 * interface enabled. In that case we should update the
	 * registers.
	 */
	if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR) ^
	    GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED)) {
		if (GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED))
			rt2500usb_config_type(rt2x00dev,
				rt2x00dev->interface.type);
		else
			rt2500usb_config_type(rt2x00dev,
				IEEE80211_IF_TYPE_MNTR);
	}
}

static int rt2500usb_config(struct ieee80211_hw *hw,
	struct ieee80211_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	u16 reg;

	/*
	 * Check if we need to disable the radio,
	 * if this is not the case, at least the RX must be disabled.
	 */
	if (GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO)) {
		if (!conf->radio_enabled)
			rt2500usb_disable_radio(rt2x00dev);
		else {
			rt2x00_register_read(rt2x00dev, TXRX_CSR2, &reg);
			rt2x00_set_field16(&reg, TXRX_CSR2_DISABLE_RX, 1);
			rt2x00_register_write(rt2x00dev, TXRX_CSR2, reg);
		}
	}

	rt2500usb_config_channel(rt2x00dev,
		conf->channel_val, conf->channel, conf->freq,
		conf->power_level);
	rt2500usb_config_txpower(rt2x00dev, conf->power_level);
	rt2500usb_config_antenna(rt2x00dev, conf->antenna_sel);
	rt2500usb_config_duration(rt2x00dev,
		(conf->flags & IEEE80211_CONF_SHORT_SLOT_TIME));
	rt2500usb_config_phymode(rt2x00dev, conf->phymode);

	/*
	 * Reenable RX only if the radio should be on.
	 */
	if (GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO)) {
		rt2x00_register_read(rt2x00dev, TXRX_CSR2, &reg);
		rt2x00_set_field16(&reg, TXRX_CSR2_DISABLE_RX, 0);
		rt2x00_register_write(rt2x00dev, TXRX_CSR2, reg);
	} else if (conf->radio_enabled)
		return rt2500usb_enable_radio(rt2x00dev);

	return 0;
}

static int rt2500usb_config_interface(struct ieee80211_hw *hw, int if_id,
	struct ieee80211_if_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	/*
	 * Monitor mode does not need configuring.
	 * If the given type does not match the configured type,
	 * there has been a problem.
	 */
	if (conf->type == IEEE80211_IF_TYPE_MNTR)
		return 0;
	else if (conf->type != rt2x00dev->interface.type)
		return -EINVAL;

	/*
	 * If the interface does not work in master mode,
	 * then the bssid value in the interface structure
	 * should now be set.
	 */
	if (conf->type != IEEE80211_IF_TYPE_AP)
		memcpy(&rt2x00dev->interface.bssid, conf->bssid, ETH_ALEN);

	/*
	 * Enable configuration.
	 */
	rt2500usb_config_type(rt2x00dev, rt2x00dev->interface.type);
	rt2500usb_config_bssid(rt2x00dev, &rt2x00dev->interface.bssid[0]);

	return 0;
}

static void rt2500usb_set_multicast_list(struct ieee80211_hw *hw,
	unsigned short flags, int mc_count)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	int update = 0;

	if (GET_FLAG(rt2x00dev, INTERFACE_ENABLED_PROMISC)) {
		if (!(flags & IFF_PROMISC)) {
			rt2x00dev->interface.promisc = 0;
			update = 1;
		}
	} else {
		if (flags & IFF_PROMISC) {
			rt2x00dev->interface.promisc = 1;
			update = 1;
		}
	}

	/*
	 * Monitor mode works with PROMISC mode forced on,
	 * so there is nothing to be done here.
	 */
	if (update && !GET_FLAG(rt2x00dev, INTERFACE_INITIALIZED_MONITOR)) {
		NOTICE("For the moment promisc mode is ignored");
	/*	queue_work(rt2x00dev->workqueue,
			&rt2x00dev->interface.work);*/
	}
}

static void rt2500usb_scan(void *data)
{
	struct rt2x00_dev *rt2x00dev = data;

	if (unlikely(!rt2x00dev->scan))
		return;

	/*
	 * Before we can start switch the channel for scanning
	 * we need to wait untill all TX rings are empty to
	 * guarentee that all frames are send on the correct channel.
	 */
	if (rt2x00_wait_scan(rt2x00dev->scan))
		goto exit;

	/*
	 * Switch channel and update active info for RX.
	 */
	if (rt2x00dev->scan->state == IEEE80211_SCAN_START) {
		rt2500usb_config_phymode(rt2x00dev,
			rt2x00dev->scan->conf.scan_phymode);

		rt2500usb_config_channel(rt2x00dev,
			rt2x00dev->scan->conf.scan_channel_val,
			rt2x00dev->scan->conf.scan_channel,
			rt2x00dev->scan->conf.scan_freq,
			rt2x00dev->scan->conf.scan_power_level);
	} else {
		rt2500usb_config_phymode(rt2x00dev,
			rt2x00dev->scan->conf.running_phymode);

		rt2500usb_config_channel(rt2x00dev,
			rt2x00dev->scan->conf.running_channel_val,
			rt2x00dev->scan->conf.running_channel,
			rt2x00dev->scan->conf.running_freq,
			rt2x00dev->scan->conf.scan_power_level);
	}

exit:
	kfree(rt2x00dev->scan);
	rt2x00dev->scan = NULL;
}

static int rt2500usb_passive_scan(struct ieee80211_hw *hw,
	int state, struct ieee80211_scan_conf *conf)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	/*
	 * Check if we are not busy with the previous
	 * passive scan request.
	 */
	if (rt2x00dev->scan)
		return -EBUSY;

	/*
	 * Check if the radio is enabled.
	 */
	if (!GET_FLAG(rt2x00dev, DEVICE_ENABLED_RADIO))
		return -EIO;

	/*
	 * Allocate scanning structure to store scanning info.
	 */
	rt2x00dev->scan = kzalloc(sizeof(struct scanning), GFP_ATOMIC);
	if (!rt2x00dev->scan)
		return -ENOMEM;

	/*
	 * Check if we have to send a packet before the
	 * channel switch.
	 */
	if (conf->skb) {
		if (rt2500usb_tx(hw, conf->skb, conf->tx_control))
			goto exit;
	}

	/*
	 * Initialize Scanning structure.
	 */
	rt2x00_start_scan(rt2x00dev->scan, conf, state);

	/*
	 * Queue work.
	 */
	INIT_WORK(&rt2x00dev->scan->work, rt2500usb_scan, rt2x00dev);
	if (!queue_work(rt2x00dev->workqueue, &rt2x00dev->scan->work))
		goto exit;

	return 0;

exit:
	kfree(rt2x00dev->scan);
	rt2x00dev->scan = NULL;

	return -EIO;
}

static int rt2500usb_get_stats(struct ieee80211_hw *hw,
	struct ieee80211_low_level_stats *stats)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	u16 reg;

	/*
	 * Update FCS error count from register.
	 * The dot11ACKFailureCount, dot11RTSFailureCount and
	 * dot11RTSSuccessCount are updated in interrupt time.
	 */
	rt2x00_register_read(rt2x00dev, STA_CSR0, &reg);
	rt2x00dev->low_level_stats.dot11FCSErrorCount +=
		rt2x00_get_field16(reg, STA_CSR0_FCS_ERROR);

	memcpy(stats, &rt2x00dev->low_level_stats, sizeof(*stats));

	return 0;
}

static int rt2500usb_conf_tx(struct ieee80211_hw *hw,
	int queue, const struct ieee80211_tx_queue_params *params)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	struct data_ring *ring;

	ring = rt2x00_get_ring(rt2x00dev, queue);
	if (unlikely(!ring))
		return -EINVAL;

	memcpy(&ring->tx_params, params, sizeof(*params));

	/*
	 * The passed variables are stored as real value ((2^n)-1).
	 * RT2400 registers require to know the bit number 'n'.
	 */
	if (params->cw_min)
		ring->tx_params.cw_min = HIGHEST_BIT16(params->cw_min) + 1;
	else
		ring->tx_params.cw_min = 5; /* cw_min: 2^5 = 32. */

	if (params->cw_max)
		ring->tx_params.cw_max = HIGHEST_BIT16(params->cw_max) + 1;
	else
		ring->tx_params.cw_max = 10; /* cw_min: 2^10 = 1024. */

	if (!params->aifs)
		ring->tx_params.aifs = 2;

	INFO("Configured TX ring %d - CWmin: %d, CWmax: %d, Aifs: %d.\n",
		queue, ring->tx_params.cw_min, ring->tx_params.cw_max,
		ring->tx_params.aifs);

	return 0;
}

static int rt2500usb_get_tx_stats(struct ieee80211_hw *hw,
	struct ieee80211_tx_queue_stats *stats)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA0],
		&rt2x00dev->ring[RING_PRIO].stats,
		sizeof(rt2x00dev->ring[RING_PRIO].stats));
	memcpy(&stats->data[IEEE80211_TX_QUEUE_DATA1],
		&rt2x00dev->ring[RING_TX].stats,
		sizeof(rt2x00dev->ring[RING_TX].stats));

	return 0;
}

static int rt2500usb_beacon_update(struct ieee80211_hw *hw,
	struct sk_buff *skb, struct ieee80211_tx_control *control)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;
	struct usb_device *usb_dev =
		interface_to_usbdev(rt2x00dev_usb(rt2x00dev));
	struct data_entry *entry;

	entry = rt2x00_get_data_entry(&rt2x00dev->ring[RING_BEACON]);

	/*
	 * Just in case the ieee80211 doesn't set this,
	 * but we need this queue set for the descriptor
	 * initialization.
	 */
	control->queue = IEEE80211_TX_QUEUE_BEACON;

	/*
	 * Update the beacon entry.
	 */
	memcpy(rt2x00_txdata_addr(entry), skb->data, skb->len);
	rt2500usb_write_tx_desc(rt2x00dev,
		rt2x00_txdesc_addr(entry), skb, control);

	SET_FLAG(entry, ENTRY_OWNER_NIC);
	usb_fill_bulk_urb(
		rt2x00_urb(entry),
		usb_dev,
		usb_sndbulkpipe(usb_dev, 1),
		entry->data_addr,
		skb->len + rt2x00dev->ring[RING_BEACON].desc_size,
		rt2500usb_interrupt,
		entry);
	usb_submit_urb(rt2x00_urb(entry), GFP_ATOMIC);

	return 0;
}

static struct ieee80211_ops rt2500usb_d80211_ops = {
	.tx			= rt2500usb_tx,
	.reset			= rt2500usb_reset,
	.add_interface		= rt2500usb_add_interface,
	.remove_interface	= rt2500usb_remove_interface,
	.config			= rt2500usb_config,
	.config_interface	= rt2500usb_config_interface,
	.set_multicast_list	= rt2500usb_set_multicast_list,
	.passive_scan		= rt2500usb_passive_scan,
	.get_stats		= rt2500usb_get_stats,
	.conf_tx		= rt2500usb_conf_tx,
	.get_tx_stats		= rt2500usb_get_tx_stats,
	.beacon_update		= rt2500usb_beacon_update,
	.ethtool		= &rt2500usb_ethtool_ops,
};

/*
 * Device initialization functions.
 */
static int rt2500usb_alloc_eeprom(struct rt2x00_dev *rt2x00dev)
{
	/*
	 * Allocate the eeprom memory, check the eeprom width
	 * and copy the entire eeprom into this allocated memory.
	 */
	rt2x00dev->eeprom = kzalloc(EEPROM_SIZE, GFP_KERNEL);
	if (!rt2x00dev->eeprom)
		return -ENOMEM;

	rt2x00_vendor_request(
		rt2x00dev, USB_EEPROM_READ, USB_VENDOR_REQUEST_IN,
		EEPROM_BASE * sizeof(u16), 0x00, rt2x00dev->eeprom,
		EEPROM_SIZE * sizeof(u16), REGISTER_TIMEOUT);

	return 0;
}

static int rt2500usb_alloc_rings(struct rt2x00_dev *rt2x00dev)
{
	unsigned int i;

	rt2x00dev->ring = kzalloc(
		sizeof(struct data_ring) * RING_NUM, GFP_KERNEL);
	if (!rt2x00dev->ring) {
		ERROR("Ring allocation failed.\n");
		return -ENOMEM;
	}

	SET_FLAG(rt2x00dev, DEVICE_SUPPORT_ATIM);

	for (i = 0; i < RING_NUM; i++) {
		rt2x00dev->ring[i].rt2x00dev = rt2x00dev;

		/*
		 * Initialize ring parameters.
		 * cw_min: 2^5 = 32.
		 * cw_max: 2^10 = 1024.
		 */
		rt2x00dev->ring[i].tx_params.aifs = 2;
		rt2x00dev->ring[i].tx_params.cw_min = 5;
		rt2x00dev->ring[i].tx_params.cw_max = 10;
	}

	return 0;
}

static int rt2500usb_init_eeprom(struct rt2x00_dev *rt2x00dev)
{
	u16 reg;
	u16 value;
	u16 eeprom;

	/*
	 * Read EEPROM word for configuration.
	 */
	rt2x00_eeprom_read(rt2x00dev, EEPROM_ANTENNA, &eeprom);

	/*
	 * Identify RF chipset.
	 */
	value = rt2x00_get_field16(eeprom, EEPROM_ANTENNA_RF_TYPE);
	rt2x00_register_read(rt2x00dev, MAC_CSR0, &reg);
	rt2x00_set_chip(&rt2x00dev->chip, RT2570, value, reg);

	if (!rt2x00_rf(&rt2x00dev->chip, RF2522) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF2523) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF2524) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF2525) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF2525E) &&
	    !rt2x00_rf(&rt2x00dev->chip, RF5222))
		return -ENODEV;

	/*
	 * Identify default antenna configuration.
	 * Ralink devices have have antenna options for both TX as RX.
	 * The ieee80211 stack currently only provide the user to set
	 * 1 antenna, by default this is considered to be the TX antenna.
	 */
	rt2x00dev->hw->conf.antenna_sel = rt2x00_get_field16(eeprom,
		EEPROM_ANTENNA_TX_DEFAULT);

	/*
	 * Store led mode, for correct led behaviour.
	 */
	rt2x00dev->led_mode = rt2x00_get_field16(eeprom,
		EEPROM_ANTENNA_LED_MODE);

	/*
	 * Check if the BBP tuning should be disabled.
	 */
	rt2x00_eeprom_read(rt2x00dev, EEPROM_NIC, &eeprom);
	if (eeprom == 0xffff)
		eeprom = 0;
	if (rt2x00_get_field16(eeprom, EEPROM_NIC_DYN_BBP_TUNE))
		SET_FLAG(rt2x00dev, CONFIG_DISABLE_LINK_TUNING);

	/*
	 * Read the RSSI <-> dBm offset information.
	 */
	rt2x00_eeprom_read(rt2x00dev, EEPROM_CALIBRATE_OFFSET, &eeprom);
	rt2x00dev->hw->maxssi =
		rt2x00_get_field16(eeprom, EEPROM_CALIBRATE_OFFSET_RSSI);
	if (rt2x00dev->hw->maxssi == 0x00 || rt2x00dev->hw->maxssi == 0xff)
		rt2x00dev->hw->maxssi = MAX_RX_SSI;

	return 0;
}

static int rt2500usb_init_hw_mac(struct rt2x00_dev *rt2x00dev)
{
	u8 *addr;

	/*
	 * Get the pointer to the MAC address in the EEPROM.
	 */
	addr = rt2x00_eeprom_addr(rt2x00dev, EEPROM_MAC_ADDR_0);

	/*
	 * Check if a valid MAC address is present.
	 */
	if (!is_valid_ether_addr(addr)) {
		ERROR("Invalid MAC address: " MAC_FMT ".\n", MAC_ARG(addr));
		return -EINVAL;
	}

	/*
	 * Write MAC address to register.
	 */
	rt2500usb_config_mac_address(rt2x00dev, addr);

	/*
	 * Copy MAC address to the hw structure.
	 */
	memcpy(&rt2x00dev->hw->perm_addr, addr, ETH_ALEN);

	return 0;
}

static void rt2500usb_init_hw_channels(struct rt2x00_dev *rt2x00dev,
	struct ieee80211_channel *channels)
{
	unsigned int i;
	u32 rf2_base;
	u16 eeprom;
	static const struct {
		unsigned int chip;
		u32 val[3];
	} rf[] = {
		{ RF2522,	{ 0x00002050, 0x00000101, 0x00000000 } },
		{ RF2523,	{ 0x00022010, 0x000e0111, 0x00000a1b } },
		{ RF2524,	{ 0x00032020, 0x00000101, 0x00000a1b } },
		{ RF2525,	{ 0x00022020, 0x00060111, 0x00000a1b } },
		{ RF2525E,	{ 0x00022010, 0x00060111, 0x00000000 } },
		{ RF5222,	{ 0x00000000, 0x00000101, 0x00000000 } }
	};

	/*
	 * Channel initialization.
	 * First we set the basic variables.
	 */
	for (i = 0; i < 13; i++) {
		channels[i].chan = i + 1;
		channels[i].freq = 2407 + ((i + 1) * 5);
		channels[i].flag = IEEE80211_CHAN_W_IBSS |
			IEEE80211_CHAN_W_ACTIVE_SCAN | IEEE80211_CHAN_W_SCAN;
		channels[i].antenna_max = 0xff;
	}

	channels[13].chan = 14;
	channels[13].freq = 2484;
	channels[13].flag = IEEE80211_CHAN_W_IBSS |
		IEEE80211_CHAN_W_ACTIVE_SCAN | IEEE80211_CHAN_W_SCAN;
	channels[13].antenna_max = 0xff;

	if (rt2x00_rf(&rt2x00dev->chip, RF5222)) {
		for (i = 14; i < 37; i++) {
			if (i < 22)
				channels[i].chan = 36;
			else if (i < 33)
				channels[i].chan = 100;
			else
				channels[i].chan = 149;
			channels[i].chan += ((i - 14) * 4);
			channels[i].freq = ((i - 13) + 1000) * 5;
			channels[i].flag = IEEE80211_CHAN_W_IBSS |
				IEEE80211_CHAN_W_ACTIVE_SCAN |
				IEEE80211_CHAN_W_SCAN;
			channels[i].power_level = DEFAULT_TXPOWER;
			channels[i].antenna_max = 0xff;
		}
	}

	/*
	 * Set device specific value.
	 */
	rf2_base = 0;
	if (rt2x00_rf(&rt2x00dev->chip, RF2525))
		rf2_base = 0x00080000;

	if (rt2x00_rf(&rt2x00dev->chip, RF2522)) {
		static const u32 vals[] = {
			0x000c1fda, 0x000c1fee, 0x000c2002, 0x000c2016,
			0x000c202a, 0x000c203e, 0x000c2052, 0x000c2066,
			0x000c207a, 0x000c208e, 0x000c20a2, 0x000c20b6,
			0x000c20ca, 0x000c20fa
		};

		for (i = 0; i < ARRAY_SIZE(vals); i++)
			channels[i].val = vals[i];
	} else if (rt2x00_rf(&rt2x00dev->chip, RF2523) ||
		   rt2x00_rf(&rt2x00dev->chip, RF2524) ||
		   rt2x00_rf(&rt2x00dev->chip, RF2525)) {
		static const u32 vals[] = {
			0x00000c9e, 0x00000ca2, 0x00000ca6, 0x00000caa,
			0x00000cae, 0x00000cb2, 0x00000cb6, 0x00000cba,
			0x00000cbe, 0x00000d02, 0x00000d06, 0x00000d0a,
			0x00000d0e, 0x00000d1a
		};

		for (i = 0; i < ARRAY_SIZE(vals); i++)
			channels[i].val = vals[i] | rf2_base;
	} else if (rt2x00_rf(&rt2x00dev->chip, RF2525E)) {
		static const u32 vals[] = {
			0x0000089a, 0x0000089e, 0x0000089e, 0x000008a2,
			0x000008a2, 0x000008a6, 0x000008a6, 0x000008aa,
			0x000008aa, 0x000008ae, 0x000008ae, 0x000008b2,
			0x000008b2, 0x000008b6
		};

		for (i = 0; i < ARRAY_SIZE(vals); i++)
			channels[i].val = vals[i];
	} else if (rt2x00_rf(&rt2x00dev->chip, RF5222)) {
		static const u32 vals[] = {
			0x00001136, 0x0000113a, 0x0000113e, 0x00001182,
			0x00001186, 0x0000118a, 0x0000118e, 0x00001192,
			0x00001196, 0x0000119a, 0x0000119e, 0x000011a2,
			0x000011a6, 0x000011ae, 0x0001889a, 0x0001889a,
			0x0001889e, 0x000188a2, 0x000188a6, 0x000188aa,
			0x000188ae, 0x000188b2, 0x00008802, 0x00008806,
			0x0000880a, 0x0000880e, 0x00008812, 0x00008816,
			0x0000881a, 0x0000881e, 0x00008822, 0x00008826,
			0x0000882a, 0x000090a6, 0x000090ae, 0x000090b6,
			0x000090be
		};

		for (i = 0; i < ARRAY_SIZE(vals); i++)
			channels[i].val = vals[i];
	}

	/*
	 * Set TX power, each EEPROM TXpower entry
	 * contains the TXpower value for 2 channels.
	 */
	for (i = 0; i < EEPROM_TXPOWER_SIZE; i++) {
		rt2x00_eeprom_read(rt2x00dev,
			EEPROM_TXPOWER_START + i, &eeprom);

		channels[(i * 2)].power_level = TXPOWER_FROM_DEV(
			rt2x00_get_field16(eeprom, EEPROM_TXPOWER_1));

		channels[(i * 2) + 1].power_level = TXPOWER_FROM_DEV(
			rt2x00_get_field16(eeprom, EEPROM_TXPOWER_2));
	}

	/*
	 * Set device specific, but channel independent RF values.
	 */
	for (i = 0; i < ARRAY_SIZE(rf); i++) {
		if (rt2x00_rf(&rt2x00dev->chip, rf[i].chip)) {
			rt2x00dev->rf1 = rf[i].val[0];
			rt2x00dev->rf3 = rf[i].val[1];
			rt2x00dev->rf4 = rf[i].val[2];
		}
	}
}

static void rt2500usb_init_hw_rates(struct rt2x00_dev *rt2x00dev,
	struct ieee80211_rate *rates)
{
	/*
	 * Rates initialization.
	 */
	device_rate_entry(&rates[0], 10, 0x001, 0x00, IEEE80211_RATE_CCK);
	device_rate_entry(&rates[1], 20, 0x003, 0x01, IEEE80211_RATE_CCK_2);
	device_rate_entry(&rates[2], 55, 0x007, 0x02, IEEE80211_RATE_CCK_2);
	device_rate_entry(&rates[3], 110, 0x00f, 0x03, IEEE80211_RATE_CCK_2);
	device_rate_entry(&rates[4], 60, 0x01f, 0x0b, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[5], 90, 0x03f, 0x0f, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[6], 120, 0x07f, 0x0a, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[7], 180, 0x0ff, 0x0e, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[8], 240, 0x1ff, 0x09, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[9], 360, 0x3ff, 0x0d, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[10], 480, 0x7ff, 0x08, IEEE80211_RATE_OFDM);
	device_rate_entry(&rates[11], 540, 0xfff, 0x0c, IEEE80211_RATE_OFDM);
}

static int rt2500usb_init_hw_modes(struct rt2x00_dev *rt2x00dev)
{
	struct ieee80211_hw *hw = rt2x00dev->hw;
	int num_modes;
	int num_channels;

	/*
	 * RT2500 only supports 802.11b & 802.11g,
	 * so we should allocate 14 OFDM channels, 4 CCK rates
	 * and 8 OFDM rates.
	 * RF5222 also supports 802.11a, so allocate an
	 * additional 23 5.2GHz channels.
	 */
	num_modes = 2;
	num_channels = 14;
	if (rt2x00_rf(&rt2x00dev->chip, RF5222)) {
		num_modes = 3;
		num_channels = 37;
	}

	hw->num_modes = num_modes;
	hw->modes =
		kzalloc((sizeof(struct ieee80211_hw_modes) * num_modes),
		GFP_KERNEL);
	if (!hw->modes)
		goto exit;

	hw->modes->channels =
		kzalloc((sizeof(struct ieee80211_channel) * num_channels),
		GFP_KERNEL);
	if (!hw->modes->channels)
		goto exit_free_modes;

	hw->modes->rates =
		kzalloc((sizeof(struct ieee80211_rate) * 12),
		GFP_KERNEL);
	if (!hw->modes->rates)
		goto exit_free_channels;

	/*
	 * Intitialize 802.11g
	 * Rates: CCK, OFDM.
	 * Channels: OFDM.
	 */
	hw->modes[0].mode = MODE_IEEE80211G;
	hw->modes[0].num_channels = 14;
	hw->modes[0].num_rates = 12;

	/*
	 * Intitialize 802.11b
	 * Rates: CCK.
	 * Channels: OFDM.
	 */
	hw->modes[1].mode = MODE_IEEE80211B;
	hw->modes[1].num_channels = 14;
	hw->modes[1].num_rates = 4;
	hw->modes[1].channels = hw->modes[0].channels;
	hw->modes[1].rates = hw->modes[0].rates;

	/*
	 * Intitialize 802.11a
	 * Rates: OFDM.
	 * Channels: OFDM, UNII, HiperLAN2.
	 */
	if (rt2x00_rf(&rt2x00dev->chip, RF5222)) {
		hw->modes[2].mode = MODE_IEEE80211A;
		hw->modes[2].num_channels = 37;
		hw->modes[2].num_rates = 8;
		hw->modes[2].channels = &hw->modes[0].channels[14];
		hw->modes[2].rates = &hw->modes[0].rates[4];
	}

	rt2500usb_init_hw_channels(rt2x00dev, hw->modes[0].channels);
	rt2500usb_init_hw_rates(rt2x00dev, hw->modes[0].rates);

	return ieee80211_update_hw(hw);

exit_free_channels:
	kfree(hw->modes->channels);
	hw->modes->channels = NULL;

exit_free_modes:
	kfree(hw->modes);
	hw->modes = NULL;

exit:
	ERROR("Allocation ieee80211 modes failed.\n");
	return -ENOMEM;
}

static int rt2500usb_init_hw(struct rt2x00_dev *rt2x00dev)
{
	int status;

	if (GET_FLAG(rt2x00dev, DEVICE_INITIALIZED_HW))
		return 0;

	rt2x00dev->hw->dev = &rt2x00dev_usb(rt2x00dev)->dev;

	/*
	 * Read MAC address from EEPROM.
	 */
	status = rt2500usb_init_hw_mac(rt2x00dev);
	if (status)
		return status;

	/*
	 * Initialize all hw fields.
	 */
	rt2x00dev->hw->flags = IEEE80211_HW_HOST_GEN_BEACON |
		IEEE80211_HW_HOST_BROADCAST_PS_BUFFERING |
		IEEE80211_HW_WEP_INCLUDE_IV |
		IEEE80211_HW_DATA_NULLFUNC_ACK |
		IEEE80211_HW_NO_TKIP_WMM_HWACCEL |
		IEEE80211_HW_MONITOR_DURING_OPER;
	rt2x00dev->hw->extra_tx_headroom = 0;
	rt2x00dev->hw->maxssi = MAX_RX_SSI;
	rt2x00dev->hw->queues = RING_NUM_TX;

	status = rt2500usb_init_hw_modes(rt2x00dev);
	if (status)
		return status;

	if (ieee80211_register_hw(rt2x00dev->hw))
		return -EIO;

	SET_FLAG(rt2x00dev, DEVICE_INITIALIZED_HW);

	return 0;
}

static void rt2500usb_free_dev(struct rt2x00_dev *rt2x00dev)
{
	/*
	 * Free ring structures.
	 */
	kfree(rt2x00dev->ring);
	rt2x00dev->ring = NULL;

	/*
	 * Free workqueue.
	 */
	if (likely(rt2x00dev->workqueue)) {
		destroy_workqueue(rt2x00dev->workqueue);
		rt2x00dev->workqueue = NULL;
	}

	/*
	 * Free EEPROM memory.
	 */
	kfree(rt2x00dev->eeprom);

	/*
	 * Free ieee80211_hw memory.
	 */
	if (likely(rt2x00dev->hw->modes)) {
		kfree(rt2x00dev->hw->modes->channels);
		kfree(rt2x00dev->hw->modes->rates);
		kfree(rt2x00dev->hw->modes);
		rt2x00dev->hw->modes = NULL;
	}
}

static int rt2500usb_alloc_dev(struct usb_interface *usb_intf,
	struct ieee80211_hw *hw)
{
	struct rt2x00_dev *rt2x00dev = hw->priv;

	rt2x00dev->dev = usb_intf;
	rt2x00dev->hw = hw;

	/*
	 * Allocate eeprom data.
	 */
	if (rt2500usb_alloc_eeprom(rt2x00dev))
		goto exit;

	/*
	 * Create workqueue.
	 */
	rt2x00dev->workqueue = create_singlethread_workqueue(DRV_NAME);
	if (!rt2x00dev->workqueue)
		return -ENODEV;

	/*
	 * Initialize configuration work.
	 */
	INIT_WORK(&rt2x00dev->link.work, rt2500usb_link_tuner, rt2x00dev);

	/*
	 * Reset current working type.
	 */
	rt2x00dev->interface.type = -EINVAL;

	/*
	 * Intialize scanning attributes.
	 */
	rt2x00dev->scan = NULL;

	/*
	 * Allocate ring array.
	 */
	if (rt2500usb_alloc_rings(rt2x00dev))
		goto exit;

	/*
	 * Initialize hardware.
	 */
	if (rt2500usb_init_eeprom(rt2x00dev) ||
	    rt2500usb_init_hw(rt2x00dev)) {
		ERROR("Failed to initialize device.\n");
		goto exit;
	}

	return 0;

exit:
	rt2500usb_free_dev(rt2x00dev);

	return -ENODEV;
}

/*
 * USB driver handlers.
 */
static int rt2500usb_probe(struct usb_interface *usb_intf,
	const struct usb_device_id *id)
{
	struct usb_device *usb_dev = interface_to_usbdev(usb_intf);
	struct ieee80211_hw *hw;
	int status;

	usb_dev = usb_get_dev(usb_dev);

	hw = ieee80211_alloc_hw(sizeof(struct rt2x00_dev),
		&rt2500usb_d80211_ops);
	if (!hw) {
		ERROR("Failed to allocate hardware.\n");
		status = -ENOMEM;
		goto exit_put_device;
	}

	usb_set_intfdata(usb_intf, hw);

	status = rt2500usb_alloc_dev(usb_intf, hw);
	if (status) {
		ERROR("Failed to allocate device.\n");
		goto exit_free_device;
	}

	ieee80211_netif_oper(hw, NETIF_ATTACH);

	return 0;

exit_free_device:
	ieee80211_free_hw(hw);

exit_put_device:
	usb_put_dev(usb_dev);

	return status;
}

static void rt2500usb_disconnect(struct usb_interface *usb_intf)
{
	struct ieee80211_hw *hw = usb_get_intfdata(usb_intf);
	struct rt2x00_dev *rt2x00dev = hw->priv;

	ieee80211_netif_oper(hw, NETIF_DETACH);

	/*
	 * Uninitialize and free the rt2500usb driver data.
	 */
	rt2500usb_disable_radio(rt2x00dev);
	rt2500usb_uninitialize(rt2x00dev);
	rt2500usb_free_dev(rt2x00dev);

	/*
	 * Uninitialize and free the 80211 stack data.
	 */
	ieee80211_unregister_hw(hw);
	ieee80211_free_hw(hw);

	/*
	 * Free the USB device data.
	 */
	usb_set_intfdata(usb_intf, NULL);
	usb_put_dev(interface_to_usbdev(usb_intf));
}

#ifdef CONFIG_PM
static int rt2500usb_suspend(struct usb_interface *usb_intf,
	pm_message_t state)
{
	struct ieee80211_hw *hw = usb_get_intfdata(usb_intf);
	struct rt2x00_dev *rt2x00dev = hw->priv;
	int status;

	NOTICE("Going to sleep.\n");

	ieee80211_netif_oper(hw, NETIF_DETACH);

	/*
	 * Disable the radio.
	 */
	rt2500usb_disable_radio(rt2x00dev);

	/*
	 * Set device mode to sleep for power management.
	 */
	status = rt2500usb_set_state(rt2x00dev, STATE_SLEEP);
	if (status)
		return status;

	/*
	 * Uninitialize device and hardware.
	 */
	rt2500usb_uninitialize(rt2x00dev);
	rt2500usb_free_dev(rt2x00dev);

	/*
	 * Decrease usbdev refcount.
	 */
	usb_put_dev(interface_to_usbdev(usb_intf));

	return 0;
}

static int rt2500usb_resume(struct usb_interface *usb_intf)
{
	struct ieee80211_hw *hw = usb_get_intfdata(usb_intf);
	struct rt2x00_dev *rt2x00dev = hw->priv;
	int status;

	NOTICE("Waking up.\n");

	/*
	 * Increase usbdev refcount.
	 */
	usb_get_dev(interface_to_usbdev(usb_intf));

	/*
	 * Initialize hardware.
	 */
	status = rt2500usb_alloc_dev(usb_intf, hw);
	if (status) {
		ERROR("Failed to allocate device.\n");
		return status;
	}

	/*
	 * Set device mode to awake for power management.
	 */
	status = rt2500usb_set_state(rt2x00dev, STATE_AWAKE);
	if (status)
		return status;

	ieee80211_netif_oper(hw, NETIF_ATTACH);

	return 0;
}
#endif /* CONFIG_PM */

/*
 * rt2500usb module information.
 */
static char version[] =
	DRV_NAME " - " DRV_VERSION " (" DRV_RELDATE ") by " DRV_PROJECT;

static struct usb_device_id rt2500usb_device_table[] = {
	/* ASUS */
	{ USB_DEVICE(0x0b05, 0x1706) },
	{ USB_DEVICE(0x0b05, 0x1707) },
	/* Belkin */
	{ USB_DEVICE(0x050d, 0x7050) },
	{ USB_DEVICE(0x050d, 0x7051) },
	{ USB_DEVICE(0x050d, 0x705a) },
	/* Cisco Systems */
	{ USB_DEVICE(0x13b1, 0x000d) },
	{ USB_DEVICE(0x13b1, 0x0011) },
	{ USB_DEVICE(0x13b1, 0x001a) },
	/* Conceptronic */
	{ USB_DEVICE(0x14b2, 0x3c02) },
	/* D-LINK */
	{ USB_DEVICE(0x2001, 0x3c00) },
	/* Gigabyte */
	{ USB_DEVICE(0x1044, 0x8001) },
	{ USB_DEVICE(0x1044, 0x8007) },
	/* Hercules */
	{ USB_DEVICE(0x06f8, 0xe000) },
	/* Melco */
	{ USB_DEVICE(0x0411, 0x0066) },
	{ USB_DEVICE(0x0411, 0x0067) },
	{ USB_DEVICE(0x0411, 0x008b) },
	/* MSI */
	{ USB_DEVICE(0x0db0, 0x6861) },
	{ USB_DEVICE(0x0db0, 0x6865) },
	{ USB_DEVICE(0x0db0, 0x6869) },
	/* Ralink */
	{ USB_DEVICE(0x148f, 0x1706) },
	{ USB_DEVICE(0x148f, 0x2570) },
	{ USB_DEVICE(0x148f, 0x2573) },
	{ USB_DEVICE(0x148f, 0x9020) },
	/* Siemens */
	{ USB_DEVICE(0x0681, 0x3c06) },
	/* SMC */
	{ USB_DEVICE(0x0707, 0xee13) },
	/* Spairon */
	{ USB_DEVICE(0x114b, 0x0110) },
	/* Trust */
	{ USB_DEVICE(0x0eb0, 0x9020) },
	/* Zinwell */
	{ USB_DEVICE(0x5a57, 0x0260) },
	{ 0, }
};

MODULE_AUTHOR(DRV_PROJECT);
MODULE_VERSION(DRV_VERSION);
MODULE_DESCRIPTION("Ralink RT2500 USB Wireless LAN driver.");
MODULE_SUPPORTED_DEVICE("Ralink RT2570 USB chipset based cards");
MODULE_DEVICE_TABLE(usb, rt2500usb_device_table);
MODULE_LICENSE("GPL");

#ifdef CONFIG_RT2500USB_DEBUG
module_param_named(debug, rt2x00_debug_level, bool, S_IWUSR | S_IRUGO);
MODULE_PARM_DESC(debug, "Set this parameter to 1 to enable debug output.");
#endif /* CONFIG_RT2500USB_DEBUG */

static struct usb_driver rt2500usb_driver = {
	.name		= DRV_NAME,
	.id_table	= rt2500usb_device_table,
	.probe		= rt2500usb_probe,
	.disconnect	= rt2500usb_disconnect,
#ifdef CONFIG_PM
	.suspend	= rt2500usb_suspend,
	.resume		= rt2500usb_resume,
#endif /* CONFIG_PM */
};

static int __init rt2500usb_init(void)
{
	printk(KERN_INFO "Loading module: %s.\n", version);
	return usb_register(&rt2500usb_driver);
}

static void __exit rt2500usb_exit(void)
{
	printk(KERN_INFO "Unloading module: %s.\n", version);
	usb_deregister(&rt2500usb_driver);
}

module_init(rt2500usb_init);
module_exit(rt2500usb_exit);
