/*
 *  $Id: rsplib.h,v 1.8 2002/08/22 09:37:25 dreibh Exp $
 *
 * RSerPool implementation.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * Acknowledgement
 * This work was partially funded by the Bundesministerium fr Bildung und
 * Forschung (BMBF) of the Federal Republic of Germany (Frderkennzeichen 01AK045).
 * The authors alone are responsible for the contents.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * There are two mailinglists available at http://www.sctp.de/rserpool.html
 * which should be used for any discussion related to this implementation.
 *
 * Contact: rsplib-discussion@sctp.de
 *          dreibh@exp-math.uni-essen.de
 *
 * Purpose: RSerPool Library
 *
 */


#ifndef RSPLIB_H
#define RSPLIB_H


#include "tagitem.h"

#include <sys/time.h>


#ifdef __cplusplus
extern "C" {
#endif


#define TAG_PoolPolicy_Type                          (TAG_USER + 1000)
#define TAGDATA_PoolPolicy_Type_RoundRobin           0x01
#define TAGDATA_PoolPolicy_Type_LeastUsed            0x02
#define TAGDATA_PoolPolicy_Type_LeastUsedDegradation 0x03
#define TAGDATA_PoolPolicy_Type_WeightedRoundRobin   0x04
#define TAGDATA_PoolPolicy_Type_Random               0xfe
#define TAGDATA_PoolPolicy_Type_WeightedRandom       0xff

#define TAG_PoolPolicy_Parameter_Weight              (TAG_USER + 1001)
#define TAG_PoolPolicy_Parameter_Load                (TAG_USER + 1002)


struct EndpointAddressInfo {
   int                         ai_family;
   int                         ai_socktype;
   int                         ai_protocol;
   size_t                      ai_addrlen;
   size_t                      ai_addrs;
   struct sockaddr_storage*    ai_addr;
   struct EndpointAddressInfo* ai_next;
   uint32_t                    ai_pe_id;
};



/**
  * Initialize rsplib.
  *
  * @param tags Additional parameters.
  * @return 0 in case of success, <>0 in case of error.
  */
int rspInitialize(struct TagItem* tags);

/**
  * Clean up rsplib.
  *
  * @return 0 in case of success, <>0 in case of error.
  */
void rspCleanUp();

/**
  * Get code of last error.
  *
  * @return Last error.
  */
unsigned int rspGetLastError();

/**
  * Get text description of last error.
  *
  * @return Last error.
  */
const char* rspGetLastErrorDescription();

/**
  * Register pool element. In case of success, the pool element's identifier
  * is returned. Otherwise, 0 is returned and the exact error can be obtained
  * using rspGetLastError() and rspGetLastErrorDescription().
  *
  * @param poolHandle Pool handle.
  * @param poolHandleSize Size of pool handle.
  * @param endpointAddressInfo Pool element's address information.
  * @param tags Additional parameters.
  * @return Pool element identifier or 0 in case of error.
  *
  * @see rspGetLastError
  * @see rspGetLastErrorDescription
  */
uint32_t rspRegister(const char*                       poolHandle,
                     const size_t                      poolHandleSize,
                     const struct EndpointAddressInfo* endpointAddressInfo,
                     struct TagItem*                   tags);

/**
  * Deregister pool element. In case of success, 0 is returned. Otherwise,
  * a nonzero value is returned and the exact error can be obtained
  * using rspGetLastError() and rspGetLastErrorDescription().
  *
  * @param poolHandle Pool handle.
  * @param poolHandleSize Size of pool handle.
  * @param identifier Pool element identifier.
  * @param tags Additional parameters.
  * @return Success.
  */
int rspDeregister(const char*     poolHandle,
                  const size_t    poolHandleSize,
                  const uint32_t  identifier,
                  struct TagItem* tags);

/**
  * Do name resolution for given pool handle and select one pool element
  * by policy. The resolved address structure is stored to a variable
  * given as reference parameter. This structure must be freed after usage
  * using rspFreeEndpointAddressArray().
  * In case of success, 0 is returned. Otherwise,
  * a nonzero value is returned and the exact error can be obtained
  * using rspGetLastError() and rspGetLastErrorDescription().
  *
  * @param poolHandle Pool handle.
  * @param poolHandleSize Size of pool handle.
  * @param endpointAddressInfo Pool element's address information.
  * @param tags Additional parameters.
  * @return Pool element identifier or 0 in case of error.
  *
  * @rspFreeEndpointAddressArray
  * @see rspGetLastError
  * @see rspGetLastErrorDescription
  */
int rspNameResolution(const char*                  poolHandle,
                      const size_t                 poolHandleSize,
                      struct EndpointAddressInfo** endpointAddressInfo,
                      struct TagItem*              tags);

/**
  * Free endpoint address structure created by rspNameResolution().
  *
  * @param endpointAddressInfo Endpoint address structure to be freed.
  *
  * @see rspNameResolution
  */
void rspFreeEndpointAddressArray(struct EndpointAddressInfo* endpointAddressInfo);

/**
  * Report pool element failure to rsplib.
  *
  * @param poolHandle Pool handle.
  * @param poolHandleSize Size of pool handle.
  * @param identifier Pool element identifier.
  * @param tags Additional parameters.
  * @return Success.
  */
void rspFailure(const char*     poolHandle,
                const size_t    poolHandleSize,
                const uint32_t  identifier,
                struct TagItem* tags);

/**
  * Wrapper to the system's select() function. This function handles
  * ASAP-internal management functions like keep alives. Always use
  * this function instead of select()!
  */
int rspSelect(int             n,
              fd_set*         readfds,
              fd_set*         writefds,
              fd_set*         exceptfds,
              struct timeval* timeout);


#ifdef __cplusplus
}
#endif

#endif
