/*
 *  $Id: examplepe.c,v 1.40 2002/08/22 10:02:22 dreibh Exp $
 *
 * RSerPool implementation.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * Acknowledgement
 * This work was partially funded by the Bundesministerium fr Bildung und
 * Forschung (BMBF) of the Federal Republic of Germany (Frderkennzeichen 01AK045).
 * The authors alone are responsible for the contents.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * There are two mailinglists available at http://www.sctp.de/rserpool.html
 * which should be used for any discussion related to this implementation.
 *
 * Contact: rsplib-discussion@sctp.de
 *          dreibh@exp-math.uni-essen.de
 *
 * Purpose: Example Pool Element
 *
 */


#include "tdtypes.h"
#include "utilities.h"
#include "netutilities.h"
#include "localaddresses.h"
#include "breakdetector.h"
#include "rsplib.h"

#include <ext_socket.h>


/* Exit immediately on Ctrl-C. No clean shutdown. */
// #define FAST_BREAK

/* Disable RSerPool */
// #define NO_RSP



static int      ListenSocketFamily   = AF_INET;
static int      ListenSocketType     = SOCK_STREAM;
static int      ListenSocketProtocol = IPPROTO_TCP;
static uint16_t ListenPort           = 0;
static int      ListenSocket         = -1;
static uint32_t Identifier           = 0x00000000;
static char*    PoolName             = NULL;
static GList*   ClientList           = NULL;


static unsigned int PolicyType            = TAGDATA_PoolPolicy_Type_RoundRobin;
static unsigned int PolicyParameterWeight = 1;
static unsigned int PolicyParameterLoad   = 0;



/* ###### Clean-up ####################################################### */
void cleanUp()
{
   GList* list;
   int    fd;

   if(Identifier) {
      rspDeregister(PoolName,strlen(PoolName),Identifier,NULL);
   }

   list = g_list_first(ClientList);
   while(list != NULL) {
      fd = (int)list->data;
      ClientList = g_list_remove(ClientList,(gpointer)fd);
      ext_close(fd);
      printTimeStamp(stdout);
      printf("Removed client #%d.\n",fd);
      list = g_list_first(ClientList);
   }

   if(ListenSocket >= 0) {
      ext_close(ListenSocket);
      ListenSocket = -1;
   }

   rspCleanUp();
}


/* ###### Initialize ##################################################### */
void initAll()
{
   struct TagItem              tags[16];
   struct sockaddr_storage     localAddress;
   struct sockaddr_storage*    localAddressArray;
   size_t                      localAddresses;
   socklen_t                   localAddressLength;
   struct EndpointAddressInfo* eai;
   struct EndpointAddressInfo* eai2 = NULL;
   struct EndpointAddressInfo* lasteai;
   uint16_t                    port;
   size_t                      i;

#ifndef NO_RSP
   if(rspInitialize(NULL) != 0) {
      puts("ERROR: Unable to initialize rsplib!");
      cleanUp();
      exit(1);
   }
#endif

#ifndef FAST_BREAK
   installBreakDetector();
#endif

   ListenSocket = ext_socket(ListenSocketFamily,
                             ListenSocketType,
                             ListenSocketProtocol);
   if(ListenSocket < 0) {
      perror("Unable to create application socket");
      cleanUp();
      exit(1);
   }
   setNonBlocking(ListenSocket);


   memset((void*)&localAddress,0,sizeof(localAddress));
   ((struct sockaddr*)&localAddress)->sa_family = ListenSocketFamily;
   setPort((struct sockaddr*)&localAddress,ListenPort);

   if(bindplus(ListenSocket,(struct sockaddr_storage*)&localAddress,1) == false) {
      perror("Unable to bind application socket");
      cleanUp();
      exit(1);
   }

   if(ListenSocketType != SOCK_DGRAM) {
      if(ext_listen(ListenSocket,5) < 0) {
         perror("Unable to set application socket to listen mode");
         cleanUp();
         exit(1);
      }
   }


   localAddressLength = sizeof(localAddress);
   if(ext_getsockname(ListenSocket,(struct sockaddr*)&localAddress,&localAddressLength) != 0) {
      perror("Unable to get socket's local address");
   }
   port = getPort((struct sockaddr*)&localAddress);

   eai = (struct EndpointAddressInfo*)malloc(sizeof(struct EndpointAddressInfo));
   if(eai == NULL) {
      puts("ERROR: Out of memory!");
      cleanUp();
      exit(1);
   }
   eai->ai_family   = ListenSocketFamily;
   eai->ai_socktype = ListenSocketType;
   eai->ai_protocol = ListenSocketProtocol;
   eai->ai_addrs    = 0;
   eai->ai_addr     = NULL;
   eai->ai_addrlen  = sizeof(struct sockaddr_storage);
   eai->ai_next     = NULL;
   eai->ai_pe_id    = 0;
   if(ListenSocketProtocol == IPPROTO_SCTP) {
      if(gatherLocalAddresses(&eai->ai_addr,&eai->ai_addrs,ASF_HideMulticast) == false) {
         puts("ERROR: Unable to obtain local addresses!");
         cleanUp();
         exit(1);
      }
      if(eai->ai_addrs < 1) {
         puts("ERROR: You seem to have 0 local addresses?!");
         cleanUp();
         exit(1);
      }
      for(i = 0;i < eai->ai_addrs;i++) {
         setPort((struct sockaddr*)&eai->ai_addr[i],port);
      }
   }
   else {
      if(gatherLocalAddresses(&localAddressArray,&localAddresses,ASF_HideMulticast) == false) {
         puts("ERROR: Unable to obtain local addresses!");
         cleanUp();
         exit(1);
      }
      if(localAddresses < 1) {
         puts("ERROR: You seem to have 0 local addresses?!");
         cleanUp();
         exit(1);
      }

      lasteai = NULL;
      for(i = 0;i < localAddresses;i++) {
         setPort((struct sockaddr*)&localAddressArray[i],port);

         eai2 = (struct EndpointAddressInfo*)malloc(sizeof(struct EndpointAddressInfo));
         if(eai2 == NULL) {
            puts("ERROR: Out of memory!");
            free(localAddressArray);
            rspFreeEndpointAddressArray(eai);
            cleanUp();
            exit(1);
         }
         *eai2 = *eai;
         eai2->ai_addrs = 1;
         eai2->ai_addr  = (struct sockaddr_storage*)memdup(&localAddressArray[i],sizeof(struct sockaddr_storage));
         if(eai2->ai_addr == NULL) {
            puts("ERROR: Out of memory!");
            free(localAddressArray);
            rspFreeEndpointAddressArray(eai);
            cleanUp();
            exit(1);
         }
         eai2->ai_next  = lasteai;
         lasteai        = eai2;
      }

      free(localAddressArray);
      rspFreeEndpointAddressArray(eai);
      eai = eai2;
   }

#ifndef NO_RSP
   tags[0].Tag  = TAG_PoolPolicy_Type;
   tags[0].Data = PolicyType;
   tags[1].Tag  = TAG_PoolPolicy_Parameter_Load;
   tags[1].Data = PolicyParameterLoad;
   tags[2].Tag  = TAG_PoolPolicy_Parameter_Weight;
   tags[2].Data = PolicyParameterWeight;
   tags[3].Tag  = TAG_END;

   Identifier = rspRegister(PoolName,strlen(PoolName),eai,(struct TagItem*)&tags);
   if(Identifier == 0x000000) {
      puts("ERROR: Unable to register pool element.");
      puts(rspGetLastErrorDescription());
      cleanUp();
      exit(1);
   }
#endif

   rspFreeEndpointAddressArray(eai);
}


/* ###### Handle socket events ########################################### */
void socketHandler(int fd)
{
   char                       buffer[16385];
   struct sockaddr_storage    address;
   socklen_t                  addressSize;
   ssize_t                    received;
   int                        sd;

   if(ListenSocketType != SOCK_DGRAM) {
      if(fd == ListenSocket) {
         sd = ext_accept(ListenSocket,NULL,0);
         if(sd >= 0) {
            ClientList = g_list_append(ClientList,(gpointer)sd);
            setNonBlocking(sd);

            printTimeStamp(stdout);
            printf("Added client #%d.\n",sd);
         }
      }
      else {
         if(g_list_find(ClientList,(gpointer)fd) != NULL) {
            received = recvfromplus(fd, (char*)&buffer, sizeof(buffer) - 1, 0,
                                    NULL, 0,
                                    NULL, NULL, NULL,
                                    0);
            do {
               if(received > 0) {

                  buffer[received] = 0x00;
                  printf("Echo> %s",buffer);

                  ext_send(fd,(char*)&buffer,received,0);
               }
               else {
                  ClientList = g_list_remove(ClientList,(gpointer)fd);
                  ext_close(fd);

                  printTimeStamp(stdout);
                  printf("Removed client #%d.\n",fd);
                  break;
               }
               received = recvfromplus(fd, (char*)&buffer, sizeof(buffer) - 1, 0,
                                       NULL, 0,
                                       NULL, NULL, NULL,
                                       0);
            } while(received >= 0);
         }
      }
   }

   else {
      received = ext_recvfrom(fd,
                              (char*)&buffer, sizeof(buffer) - 1, 0,
                              (struct sockaddr*)&address, &addressSize);
      do {
         if(received > 0) {
            buffer[received] = 0x00;
            printf("Echo> %s",buffer);
            received = ext_sendto(fd,
                       (char*)&buffer, received, 0,
                       (struct sockaddr*)&address, addressSize);
         }
         received = ext_recvfrom(fd,
                                 (char*)&buffer, sizeof(buffer) - 1, 0,
                                 (struct sockaddr*)&address, &addressSize);
      } while(received >= 0);
   }
}



/* ###### Main program ################################################### */
int main(int argc, char** argv)
{
   int            n;
   int            result;
   struct timeval timeout;
   fd_set         readfdset;
   card64         start;
   card64         stop;
   GList*         list;

   if(argc < 2) {
      printf("Usage: %s [Pool Name] {-sctp|-sctp-udplike|-sctp-tcplike|-tcp|-udp} {-port=local port} {-stop=seconds} {-logfile=file|-logappend=file|-logquiet} {-loglevel=level} {-logcolor=on|off} {-policy=roundrobin|rr|weightedroundrobin|wee|leastused|lu|leastuseddegradation|lud|random|rd|weightedrandom|wrd} {-load=load} {-weight=weight}\n",argv[0]);
      exit(1);
   }

   start    = getMicroTime();
   stop     = 0;
   PoolName = argv[1];
   if(argc > 2) {
      for(n = 2;n < argc;n++) {
         if(!(strcmp(argv[n],"-sctp-udplike"))) {
            ListenSocketFamily   = AF_INET6;
            ListenSocketProtocol = IPPROTO_SCTP;
            ListenSocketType     = SOCK_DGRAM;
         }
         else if(!(strcmp(argv[n],"-sctp-tcplike"))) {
            ListenSocketFamily   = AF_INET6;
            ListenSocketProtocol = IPPROTO_SCTP;
            ListenSocketType     = SOCK_STREAM;
         }
         else if(!(strcmp(argv[n],"-sctp"))) {
            ListenSocketFamily   = AF_INET6;
            ListenSocketProtocol = IPPROTO_SCTP;
            ListenSocketType     = SOCK_STREAM;
         }
         else if(!(strcmp(argv[n],"-udp"))) {
            ListenSocketFamily   = AF_INET6;
            ListenSocketProtocol = IPPROTO_UDP;
            ListenSocketType     = SOCK_DGRAM;
         }
         else if(!(strcmp(argv[n],"-tcp"))) {
            ListenSocketFamily   = AF_INET6;
            ListenSocketProtocol = IPPROTO_TCP;
            ListenSocketType     = SOCK_STREAM;
         }
         else if(!(strncmp(argv[n],"-stop=",6))) {
            stop = start + ((card64)1000000 * (card64)atol((char*)&argv[n][6]));
         }
         else if(!(strncmp(argv[n],"-port=",6))) {
            ListenPort = atol((char*)&argv[n][6]);
         }
         else if(!(strncmp(argv[n],"-load=",6))) {
            PolicyParameterLoad = atol((char*)&argv[n][6]);
            if(PolicyParameterLoad > 0xffffff) {
               PolicyParameterLoad = 0xffffff;
            }
         }
         else if(!(strncmp(argv[n],"-weight=",8))) {
            PolicyParameterWeight = atol((char*)&argv[n][8]);
            if(PolicyParameterWeight > 0xffffff) {
               PolicyParameterWeight = 0xffffff;
            }
         }
         else if(!(strncmp(argv[n],"-policy=",8))) {
            if((!(strcmp((char*)&argv[n][8],"roundrobin"))) || (!(strcmp((char*)&argv[n][8],"rr")))) {
               PolicyType = TAGDATA_PoolPolicy_Type_RoundRobin;
            }
            else if((!(strcmp((char*)&argv[n][8],"weightedroundrobin"))) || (!(strcmp((char*)&argv[n][8],"wrr")))) {
               PolicyType = TAGDATA_PoolPolicy_Type_WeightedRoundRobin;
            }
            else if((!(strcmp((char*)&argv[n][8],"leastused"))) || (!(strcmp((char*)&argv[n][8],"lu")))) {
               PolicyType = TAGDATA_PoolPolicy_Type_LeastUsed;
            }
            else if((!(strcmp((char*)&argv[n][8],"leastuseddegradation"))) || (!(strcmp((char*)&argv[n][8],"lud")))) {
               PolicyType = TAGDATA_PoolPolicy_Type_LeastUsedDegradation;
            }
            else if((!(strcmp((char*)&argv[n][8],"random"))) || (!(strcmp((char*)&argv[n][8],"rd")))) {
               PolicyType = TAGDATA_PoolPolicy_Type_Random;
            }
            else if((!(strcmp((char*)&argv[n][8],"weightedrandom"))) || (!(strcmp((char*)&argv[n][8],"wrd")))) {
               PolicyType = TAGDATA_PoolPolicy_Type_WeightedRandom;
            }
            else {
               printf("ERROR: Unknown policy type \"%s\"!\n",(char*)&argv[n][8]);
               exit(1);
            }
         }
         else if(!(strncmp(argv[n],"-log",4))) {
            if(initLogging(argv[n]) == false) {
               exit(1);
            }
         }
         else {
            printf("Bad argument \"%s\"!\n",argv[n]);
            exit(1);
         }
      }
   }

   beginLogging();
   initAll();

   puts("Example Pool Element - Version 1.0");
   puts("----------------------------------\n");

   while(!breakDetected()) {
      FD_ZERO(&readfdset);
      FD_SET(ListenSocket,&readfdset);
      n = ListenSocket;
      list = g_list_first(ClientList);
      while(list != NULL) {
         FD_SET((int)list->data,&readfdset);
         n = max(n,(int)list->data);
         list = g_list_next(list);
      }

      timeout.tv_sec  = 0;
      timeout.tv_usec = 500000;
#ifndef NO_RSP
      result = rspSelect(n + 1, &readfdset, NULL, NULL, &timeout);
#else
      result = ext_select(n + 1, &readfdset, NULL, NULL, &timeout);
#endif
      if(errno == EINTR) {
         break;
      }

      if(result > 0) {
         if(FD_ISSET(ListenSocket,&readfdset)) {
            socketHandler(ListenSocket);
         }
         list = g_list_first(ClientList);
         while(list != NULL) {
            if(FD_ISSET((int)list->data,&readfdset)) {
               socketHandler((int)list->data);
            }
            list = g_list_next(list);
         }
      }
      if((stop > 0) && (getMicroTime() >= stop)) {
         puts("Auto-stop time reached -> exiting!");
         break;
      }
      if(result < 0) {
         perror("select() failed");
         cleanUp();
         exit(1);
      }
   }

   cleanUp();
   finishLogging();
   puts("\nTerminated!");
   return(0);
}
