/*
 *  $Id: dispatcher.c,v 1.12 2002/08/22 09:37:22 dreibh Exp $
 *
 * RSerPool implementation.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * Acknowledgement
 * This work was partially funded by the Bundesministerium fr Bildung und
 * Forschung (BMBF) of the Federal Republic of Germany (Frderkennzeichen 01AK045).
 * The authors alone are responsible for the contents.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * There are two mailinglists available at http://www.sctp.de/rserpool.html
 * which should be used for any discussion related to this implementation.
 *
 * Contact: rsplib-discussion@sctp.de
 *          dreibh@exp-math.uni-essen.de
 *
 * Purpose: Dispatcher
 *
 */


#include "tdtypes.h"
#include "utilities.h"
#include "dispatcher.h"

#include <ext_socket.h>



struct FDCallback
{
   struct Dispatcher* Master;
   int                FD;
   int                EventMask;

   void               (*Callback)(struct Dispatcher* dispatcher,
                                  int                fd,
                                  int                eventMask,
                                  void*              userData);
   void*              UserData;
};



/* ###### Constructor #################################################### */
struct Dispatcher* dispatcherNew(void (*lock)(struct Dispatcher* dispatcher, void* userData),
                                 void (*unlock)(struct Dispatcher* dispatcher, void* userData),
                                 void* lockUserData)
{
   struct Dispatcher* dispatcher = (struct Dispatcher*)malloc(sizeof(struct Dispatcher));
   if(dispatcher != NULL) {
      if(lock != NULL) {
         dispatcher->Lock = lock;
      }
      else {
         dispatcher->Lock = dispatcherDefaultLock;
      }
      if(unlock != NULL) {
         dispatcher->Unlock = unlock;
      }
      else {
         dispatcher->Unlock = dispatcherDefaultUnlock;
      }
      dispatcher->LockUserData = lockUserData;
      dispatcher->TimerList    = NULL;
      dispatcher->SocketList   = NULL;
   }
   return(dispatcher);
}


/* ###### Destructor ##################################################### */
void dispatcherDelete(struct Dispatcher* dispatcher)
{
   if(dispatcher != NULL) {
      free(dispatcher);
   }
}


/* ###### Lock ########################################################### */
void dispatcherLock(struct Dispatcher* dispatcher)
{
   if(dispatcher != NULL) {
      dispatcher->Lock(dispatcher, dispatcher->LockUserData);
   }
}


/* ###### Unlock ######################################################### */
void dispatcherUnlock(struct Dispatcher* dispatcher)
{
   if(dispatcher != NULL) {
      dispatcher->Unlock(dispatcher, dispatcher->LockUserData);
   }
}


/* ###### Default locking function ####################################### */
void dispatcherDefaultLock(struct Dispatcher* dispatcher, void* userData)
{
}


/* ###### Default unlocking function ##################################### */
void dispatcherDefaultUnlock(struct Dispatcher* dispatcher, void* userData)
{
}


/* ###### Handle timer events ############################################ */
static void dispatcherHandleTimerEvents(struct Dispatcher* dispatcher)
{
   GList*        timerList;
   struct Timer* timer;
   card64        now;
   dispatcherLock(dispatcher);

   dispatcher->AddRemove = false;
   timerList = g_list_first(dispatcher->TimerList);
   while(timerList != NULL) {
      timer = (struct Timer*)timerList->data;
      now   = getMicroTime();

      if(now >= timer->Time) {
         timer->Time = 0;
         dispatcher->TimerList = g_list_remove(dispatcher->TimerList,timer);
         timerList = g_list_first(dispatcher->TimerList);

         if(timer->Callback != NULL) {
            timer->Callback(dispatcher,timer,timer->UserData);
            if(dispatcher->AddRemove == true) {
               dispatcher->AddRemove = false;
               timerList = g_list_first(dispatcher->TimerList);
            }
         }
      }
      else {
         break;
      }
   }

   dispatcherUnlock(dispatcher);
}


/* ###### Get select() parameters ######################################## */
void dispatcherGetSelectParameters(struct Dispatcher* dispatcher,
                                   int*               n,
                                   fd_set*            readfdset,
                                   fd_set*            writefdset,
                                   fd_set*            exceptfdset,
                                   struct timeval*    timeout)
{
   GList*             glist;
   struct FDCallback* fdCallback;
   struct Timer*      timer;
   card64             now;
   int64              timeToNextEvent;
   int                fds;

   if(dispatcher != NULL) {
      dispatcherLock(dispatcher);

      /*  ====== Create FD sets for select() ============================= */
      FD_ZERO(readfdset);
      FD_ZERO(writefdset);
      FD_ZERO(exceptfdset);
      fds = 0;
      glist = g_list_first(dispatcher->SocketList);
      while(glist != NULL) {
         fdCallback = (struct FDCallback*)glist->data;
         if(fdCallback->EventMask & (FDCE_Read|FDCE_Write|FDCE_Exception)) {
            fds = max(fds,fdCallback->FD);
            if(fdCallback->EventMask & FDCE_Read) {
               FD_SET(fdCallback->FD,readfdset);
            }
            if(fdCallback->EventMask & FDCE_Write) {
               FD_SET(fdCallback->FD,writefdset);
            }
            if(fdCallback->EventMask & FDCE_Exception) {
               FD_SET(fdCallback->FD,exceptfdset);
            }
         }
         else {
            LOG_WARNING
            fputs("Empty event mask?!\n",stdlog);
            LOG_END
         }
         glist = g_list_next(glist);
      }

      /*  ====== Get time to next timer event ============================ */
      now = getMicroTime();
      glist = g_list_first(dispatcher->TimerList);
      if(glist != NULL) {
         timer = (struct Timer*)glist->data;
         timeToNextEvent = max((int64)0,(int64)timer->Time - (int64)now);
      }
      else {
         timeToNextEvent = 10000000;
      }
      if(timeToNextEvent > 10000000) {
         timeToNextEvent = 10000000;
      }

      dispatcherUnlock(dispatcher);


      /* ====== Wait ===================================================== */
      timeout->tv_sec  = (timeToNextEvent / 1000000);
      timeout->tv_usec = (timeToNextEvent % 1000000);
      *n = fds + 1;
   }
   else {
      *n = 0;
   }
}


/* ###### Handle select() result ######################################### */
void dispatcherHandleSelectResult(struct Dispatcher* dispatcher,
                                  int                result,
                                  fd_set*            readfdset,
                                  fd_set*            writefdset,
                                  fd_set*            exceptfdset)
{
   GList*             glist;
   struct FDCallback* fdCallback;
   int                eventMask;

   if(dispatcher != NULL) {

      /* ====== Handle events ============================================ */
      dispatcherLock(dispatcher);

      dispatcherHandleTimerEvents(dispatcher);

      if(result > 0) {
         dispatcher->AddRemove = false;
         glist = g_list_first(dispatcher->SocketList);
         while(glist != NULL) {
            fdCallback = (struct FDCallback*)glist->data;

            eventMask = 0;
            if(FD_ISSET(fdCallback->FD,readfdset)) {
               eventMask |= FDCE_Read;
               FD_CLR(fdCallback->FD,readfdset);
            }
            if(FD_ISSET(fdCallback->FD,writefdset)) {
               eventMask |= FDCE_Write;
               FD_CLR(fdCallback->FD,writefdset);
            }
            if(FD_ISSET(fdCallback->FD,exceptfdset)) {
               eventMask |= FDCE_Exception;
               FD_CLR(fdCallback->FD,exceptfdset);
            }

            if(eventMask & fdCallback->EventMask) {
               LOG_VERBOSE3
               fprintf(stdlog,"Event $%04x (mask $%04x) for socket %d\n",
                       eventMask, fdCallback->EventMask, fdCallback->FD);
               LOG_END

               if(fdCallback->Callback != NULL) {
                  LOG_VERBOSE2
                  fprintf(stdlog,"Executing callback for event $%04x of socket %d\n",
                          eventMask, fdCallback->FD);
                  LOG_END

                  fdCallback->Callback(dispatcher,
                                       fdCallback->FD, eventMask,
                                       fdCallback->UserData);
                  if(dispatcher->AddRemove == true) {
                     dispatcher->AddRemove = false;
                     glist = g_list_first(dispatcher->SocketList);
                     continue;
                  }
               }

            }

            glist = g_list_next(glist);
         }
      }

      dispatcherUnlock(dispatcher);
   }
}


/* ###### Dispatcher event loop ########################################## */
void dispatcherEventLoop(struct Dispatcher* dispatcher)
{
   int            n;
   int            result;
   struct timeval timeout;
   fd_set         readfdset;
   fd_set         writefdset;
   fd_set         exceptfdset;

   if(dispatcher != NULL) {
      dispatcherGetSelectParameters(dispatcher, &n, &readfdset, &writefdset, &exceptfdset, &timeout);

      result = ext_select(n, &readfdset, &writefdset, &exceptfdset, &timeout);
      if(result < 0) {
         logerror("select() failed");
         exit(1);
      }

      dispatcherHandleSelectResult(dispatcher, result, &readfdset, &writefdset, &exceptfdset);
   }
}


/* ###### File descriptor callback comparision function ################## */
static gint fdCallbackCompareFunc(gconstpointer a,
                                  gconstpointer b)
{
   struct FDCallback* t1 = (struct FDCallback*)a;
   struct FDCallback* t2 = (struct FDCallback*)b;
   if(t1->FD < t2->FD) {
      return(-1);
   }
   else if(t1->FD > t2->FD) {
      return(1);
   }
   return(0);
}


/* ###### Add file descriptor callback ################################### */
bool dispatcherAddFDCallback(struct Dispatcher* dispatcher,
                             int                fd,
                             int                eventMask,
                             void               (*callback)(struct Dispatcher* dispatcher,
                                                            int                fd,
                                                            int                eventMask,
                                                            void*              userData),
                             void*              userData)
{
   struct FDCallback* fdCallback;
   if(dispatcher != NULL) {
      fdCallback = (struct FDCallback*)malloc(sizeof(struct FDCallback));
      if(fdCallback != NULL) {
         fdCallback->Master    = dispatcher;
         fdCallback->FD        = fd;
         fdCallback->EventMask = eventMask;
         fdCallback->Callback  = callback;
         fdCallback->UserData  = userData;
         dispatcherLock(dispatcher);
         dispatcher->SocketList = g_list_insert_sorted(dispatcher->SocketList,
                                                      (gpointer)fdCallback,
                                                      fdCallbackCompareFunc);
         dispatcher->AddRemove = true;
         dispatcherUnlock(dispatcher);
         return(true);
      }
   }
   return(false);
}


/* ###### Update file descriptor callback ################################ */
void dispatcherUpdateFDCallback(struct Dispatcher* dispatcher,
                                int                fd,
                                int                eventMask)
{
   GList*             found;
   struct FDCallback* fdCallback;
   struct FDCallback  example;

   example.FD = fd;

   dispatcherLock(dispatcher);
   found = g_list_find_custom(dispatcher->SocketList,(gpointer)&example,fdCallbackCompareFunc);
   if(found != NULL) {
      fdCallback = (struct FDCallback*)found->data;
      fdCallback->EventMask = eventMask;
   }
   dispatcherUnlock(dispatcher);
}


/* ###### Remove file descriptor callback ################################ */
void dispatcherRemoveFDCallback(struct Dispatcher* dispatcher,
                                int                fd)
{
   GList*           found;
   struct FDCallback* fdCallback;
   struct FDCallback  example;

   example.FD = fd;

   dispatcherLock(dispatcher);
   found = g_list_find_custom(dispatcher->SocketList,(gpointer)&example,fdCallbackCompareFunc);
   if(found != NULL) {
      fdCallback = (struct FDCallback*)found->data;
      dispatcher->SocketList = g_list_remove(dispatcher->SocketList,fdCallback);
      dispatcher->AddRemove  = true;
      free(fdCallback);
   }
   dispatcherUnlock(dispatcher);
}
