/*
 *  $Id: asapinstance.c,v 1.38 2002/08/22 09:37:22 dreibh Exp $
 *
 * RSerPool implementation.
 *
 * Realized in co-operation between Siemens AG
 * and University of Essen, Institute of Computer Networking Technology.
 *
 * Acknowledgement
 * This work was partially funded by the Bundesministerium fr Bildung und
 * Forschung (BMBF) of the Federal Republic of Germany (Frderkennzeichen 01AK045).
 * The authors alone are responsible for the contents.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * There are two mailinglists available at http://www.sctp.de/rserpool.html
 * which should be used for any discussion related to this implementation.
 *
 * Contact: rsplib-discussion@sctp.de
 *          dreibh@exp-math.uni-essen.de
 *
 * Purpose: ASAP Instance
 *
 */


#include "tdtypes.h"
#include "asapinstance.h"
#include "asapmessage.h"
#include "asapcreator.h"
#include "asapparser.h"
#include "asapcache.h"
#include "utilities.h"
#include "asaperror.h"
#include "asapcache.h"


#include <ext_socket.h>



static void asapNameServerConnectionHandler(struct Dispatcher* dispatcher,
                                            int                fd,
                                            int                eventMask,
                                            void*              userData);

static void asapDisconnectFromNameServer(struct ASAPInstance* asap);



/* ###### Get configuration from file #################################### */
static void asapConfigure(struct ASAPInstance* asap)
{
   const char* configFile = ASAP_DEFAULT_INSTANCE_CONFIG_FILE;
   FILE*  fh;
   char   buffer[512];
   char   command[64];
   char   parameter1[256];
   size_t position;
   size_t line;

   asap->AsapServerAnnounceConfigFile          = ASAP_DEFAULT_SERVER_ANNOUNCE_CONFIG_FILE;
   asap->AsapNameServersConfigFile             = ASAP_DEFAULT_NAMESERVERS_CONFIG_FILE;
   asap->NameServerAnnounceMaintenanceInterval = ASAP_DEFAULT_NAMESERVER_ANNOUNCE_MAINTENANCE_INTERVAL;
   asap->NameServerAnnounceTimeout             = ASAP_DEFAULT_NAMESERVER_ANNOUNCE_TIMEOUT;

   asap->NameServerConnectMaxTrials            = ASAP_DEFAULT_NAMESERVER_CONNECT_MAXTRIALS;
   asap->NameServerConnectTimeout              = ASAP_DEFAULT_NAMESERVER_CONNECT_TIMEOUT;
   asap->NameServerRequestMaxTrials            = ASAP_DEFAULT_NAMESERVER_REQUEST_MAXTRIALS;
   asap->NameServerRequestTimeout              = ASAP_DEFAULT_NAMESERVER_REQUEST_TIMEOUT;
   asap->NameServerResponseTimeout             = ASAP_DEFAULT_NAMESERVER_RESPONSE_TIMEOUT;

   asap->CacheMaintenanceInterval              = ASAP_DEFAULT_CACHE_MAINTENANCE_INTERVAL;
   asap->CacheElementTimeout                   = ASAP_DEFAULT_CACHE_ELEMENT_TIMEOUT;

   fh = fopen(configFile, "r");
   if(fh != NULL) {
      line = 1;
      while(!feof(fh)) {
         if( (fgets((char*)&buffer,sizeof(buffer),fh)) && (strlen(buffer) > 1) ) {
            buffer[strlen(buffer) - 1] = 0x00;
            position = 0;
            LOG_VERBOSE4
            fprintf(stdlog,"Read string <%s>\n",buffer);
            LOG_END
            if(getNextWord(buffer,(char*)&command,sizeof(command),&position)) {
               if(!(strcmp(command,"cache.elementtimeout"))) {
                  if(getNextWord(buffer,(char*)&parameter1,sizeof(parameter1),&position)) {
                     asap->CacheElementTimeout = atol(parameter1);
                  }
               }
               else if(!(strcmp(command,"cache.mtinterval"))) {
                  if(getNextWord(buffer,(char*)&parameter1,sizeof(parameter1),&position)) {
                     asap->CacheMaintenanceInterval = atol(parameter1);
                  }
               }
               else if(!(strcmp(command,"nameserver.connect.maxtrials"))) {
                  if(getNextWord(buffer,(char*)&parameter1,sizeof(parameter1),&position)) {
                     asap->NameServerConnectMaxTrials = atoi(parameter1);
                  }
               }
               else if(!(strcmp(command,"nameserver.connect.timeout"))) {
                  if(getNextWord(buffer,(char*)&parameter1,sizeof(parameter1),&position)) {
                     asap->NameServerConnectTimeout = atol(parameter1);
                  }
               }
               else if(!(strcmp(command,"nameserver.request.maxtrials"))) {
                  if(getNextWord(buffer,(char*)&parameter1,sizeof(parameter1),&position)) {
                     asap->NameServerRequestMaxTrials = atoi(parameter1);
                  }
               }
               else if(!(strcmp(command,"nameserver.request.timeout"))) {
                  if(getNextWord(buffer,(char*)&parameter1,sizeof(parameter1),&position)) {
                     asap->NameServerRequestTimeout = atol(parameter1);
                  }
               }
               else if(!(strcmp(command,"nameserver.response.timeout"))) {
                  if(getNextWord(buffer,(char*)&parameter1,sizeof(parameter1),&position)) {
                     asap->NameServerResponseTimeout = atol(parameter1);
                  }
               }
               else if(!(strcmp(command,"nameserver.announce.timeout"))) {
                  if(getNextWord(buffer,(char*)&parameter1,sizeof(parameter1),&position)) {
                     asap->NameServerAnnounceTimeout = atol(parameter1);
                  }
               }
               else if(!(strcmp(command,"nameserver.announce.mtinterval"))) {
                  if(getNextWord(buffer,(char*)&parameter1,sizeof(parameter1),&position)) {
                     asap->NameServerAnnounceMaintenanceInterval = atol(parameter1);
                     if(asap->NameServerAnnounceMaintenanceInterval < 100000) {
                        asap->NameServerAnnounceMaintenanceInterval = 100000;
                     }
                  }
               }
               else if(command[0] == '#') {
               }
               else {
                  LOG_WARNING
                  fprintf(stdlog,"File %s, line %u, column %u: Unknown command <%s>!\n",
                          configFile, (unsigned int)line, (unsigned int)position, command);
                  LOG_END
               }
            }
            else {
               LOG_WARNING
               fprintf(stdlog,"File %s, line %u: Line too long!\n",
                       configFile, (unsigned int)line);
               LOG_END
            }
         }
         line++;
      }
      fclose(fh);
   }
   else {
      LOG_VERBOSE2
      fputs("There is no ASAP instance configuration file. Using defaults.\n",stdlog);
      LOG_END
   }

   LOG_VERBOSE3
   fputs("New ASAP instance configuration:\n",stdlog);
   fprintf(stdlog,"cache.elementtimeout           = %Ld [s]\n", asap->CacheElementTimeout);
   fprintf(stdlog,"cache.mtinterval               = %Ld [s]\n", asap->CacheMaintenanceInterval);
   fprintf(stdlog,"nameserver.connect.maxtrials   = %d\n",       asap->NameServerConnectMaxTrials);
   fprintf(stdlog,"nameserver.connect.timeout     = %Ld [s]\n", asap->NameServerConnectTimeout);
   fprintf(stdlog,"nameserver.request.maxtrials   = %d\n",       asap->NameServerRequestMaxTrials);
   fprintf(stdlog,"nameserver.request.timeout     = %Ld [s]\n", asap->NameServerRequestTimeout);
   fprintf(stdlog,"nameserver.response.timeout    = %Ld [s]\n", asap->NameServerResponseTimeout);
   fprintf(stdlog,"nameserver.announce.timeout    = %Ld [s]\n", asap->NameServerAnnounceTimeout);
   fprintf(stdlog,"nameserver.announce.mtinterval = %Ld [s]\n", asap->NameServerAnnounceMaintenanceInterval);
   LOG_END
}


/* ###### Constructor #################################################### */
struct ASAPInstance* asapNew(struct Dispatcher* dispatcher)
{
   struct ASAPInstance* asap = NULL;
   if(dispatcher != NULL) {
      asap = (struct ASAPInstance*)malloc(sizeof(struct ASAPInstance));
      if(asap != NULL) {
         asap->StateMachine = dispatcher;

         asapConfigure(asap);

         asap->NameServerSocket = -1;
         asap->Cache            = asapCacheNew(asap->StateMachine,
                                               asap->CacheMaintenanceInterval,
                                               asap->CacheElementTimeout);
         asap->NameServerTable  = serverTableNew(asap->StateMachine,
                                                 asap->NameServerAnnounceMaintenanceInterval,
                                                 asap->NameServerAnnounceTimeout,
                                                 asap->AsapServerAnnounceConfigFile);
         if((asap->Cache == NULL) || (asap->NameServerTable == NULL)) {
            asapDelete(asap);
            asap = NULL;
         }
      }
   }
   return(asap);
}


/* ###### Destructor ##################################################### */
void asapDelete(struct ASAPInstance* asap)
{
   if(asap) {
      asapDisconnectFromNameServer(asap);
      if(asap->Cache) {
         asapCacheDelete(asap->Cache);
         asap->Cache = NULL;
      }
      if(asap->NameServerTable) {
         serverTableDelete(asap->NameServerTable);
         asap->NameServerTable = NULL;
      }
      free(asap);
   }
}


/* ###### Connect to name server ######################################### */
static bool asapConnectToNameServer(struct ASAPInstance* asap,
                                    const int            protocol)
{
   bool result = true;

   if(asap->NameServerSocket < 0) {
      asap->NameServerSocket = serverTableFindServer(asap->NameServerTable,
                                                     asap->NameServerConnectTimeout,
                                                     asap->NameServerConnectMaxTrials,
                                                     asap->AsapNameServersConfigFile,
                                                     protocol);
      if(asap->NameServerSocket >= 0) {
         dispatcherAddFDCallback(asap->StateMachine,
                                 asap->NameServerSocket,
                                 FDCE_Read|FDCE_Exception,
                                 asapNameServerConnectionHandler,
                                 (gpointer)asap);
         LOG_ACTION
         fputs("Connection to name server server successfully established\n",stdlog);
         LOG_END
      }
      else {
         result = false;
         LOG_ERROR
         fputs("Unable to connect to an name server server\n",stdlog);
         LOG_END
      }
   }

   return(result);
}


/* ###### Disconnect from name server #################################### */
static void asapDisconnectFromNameServer(struct ASAPInstance* asap)
{
   if(asap->NameServerSocket >= 0) {
      dispatcherRemoveFDCallback(asap->StateMachine,asap->NameServerSocket);
      ext_close(asap->NameServerSocket);
      LOG_ACTION
      fputs("Disconnected from nameserver\n",stdlog);
      LOG_END
      asap->NameServerSocket = -1;
   }
}


/* ###### Receive response from name server ############################## */
enum ASAPError asapReceiveResponse(struct ASAPInstance* asap, struct ASAPMessage** message)
{
   *message = asapMessageReceive(asap->NameServerSocket,
                                 asap->NameServerResponseTimeout,
                                 asap->NameServerResponseTimeout,
                                 65536,
                                 NULL, NULL);
   if(*message) {
      return(ASAP_Okay);
   }

   LOG_ERROR
   fputs("Receiving response failed\n",stdlog);
   LOG_END

   return(ASAP_ReadError);
}


/* ###### Send request to name server #################################### */
static enum ASAPError asapSendRequest(struct ASAPInstance* asap,
                                      struct ASAPMessage*  request)
{
   bool result;

   if(asapConnectToNameServer(asap,0) == false) {
      LOG_ERROR
      fputs("No name server server available\n",stdlog);
      LOG_END
      return(ASAP_NoNameServerFound);
   }

   result = asapMessageSend(asap->NameServerSocket,
                            asap->NameServerRequestTimeout,
                            request);
   if(result) {
      return(ASAP_Okay);
   }

   LOG_ERROR
   fputs("Sending request failed\n",stdlog);
   LOG_END
   return(ASAP_WriteError);
}


/* ###### Send request to name server and wait for response ############## */
static enum ASAPError asapIO(struct ASAPInstance* asap,
                             struct ASAPMessage*  message,
                             struct ASAPMessage** responsePtr,
                             uint16_t*            error)
{
   struct ASAPMessage* response;
   enum ASAPError      result = ASAP_Okay;
   cardinal            i;

   if(responsePtr != NULL) {
      *responsePtr = NULL;
   }
   *error = ASAP_Okay;


   for(i = 0;i < asap->NameServerRequestMaxTrials;i++) {
      LOG_VERBOSE2
      fprintf(stdlog,"Request trial #%d - sending request...\n",i + 1);
      LOG_END

      result = asapSendRequest(asap,message);
      if(result == ASAP_Okay) {
         LOG_VERBOSE2
         fprintf(stdlog,"Request trial #%d - waiting for response...\n",i + 1);
         LOG_END
         result = asapReceiveResponse(asap,&response);
         while(response != NULL) {
            *error = response->Error;

            if(response->Type == AHT_ENDPOINT_KEEP_ALIVE) {
            }
            else if(response->Type == AHT_ENDPOINT_KEEP_ALIVE_ACK) {
            }
            else if( ((response->Type == AHT_REGISTRATION_RESPONSE)    && (message->Type == AHT_REGISTRATION))   ||
                     ((response->Type == AHT_DEREGISTRATION_RESPONSE)  && (message->Type == AHT_DEREGISTRATION)) ||
                     ((response->Type == AHT_NAME_RESOLUTION_RESPONSE) && (message->Type == AHT_NAME_RESOLUTION)) ) {
               if(responsePtr != NULL) {
                  *responsePtr = response;
               }
               else {
                  asapMessageDelete(response);
               }
               return(ASAP_Okay);
            }
            else {
               LOG_WARNING
               fprintf(stdlog,"Bad request/response type pair: %02x/%02x\n",message->Type,response->Type);
               LOG_END
               asapMessageDelete(response);
               return(ASAP_InvalidValues);
            }

            asapMessageDelete(response);
            result = asapReceiveResponse(asap,&response);
            if(result == ASAP_Okay) {
               LOG_VERBOSE2
               fprintf(stdlog,"Request trial #%d - Success\n",i + 1);
               LOG_END
               break;
            }
         }
      }

      LOG_ERROR
      fprintf(stdlog,"Request trial #%d failed\n",i + 1);
      LOG_END
      asapDisconnectFromNameServer(asap);
   }

   return(result);
}


/* ###### Register pool element ########################################## */
enum ASAPError asapRegister(struct ASAPInstance* asap,
                            struct PoolHandle*   poolHandle,
                            struct PoolElement*  poolElement)
{
   struct ASAPMessage* message;
   struct PoolElement* newPoolElement;
   enum ASAPError      result;
   uint16_t            error;

   dispatcherLock(asap->StateMachine);

   LOG_ACTION
   fputs("Trying to register ",stdlog);
   poolElementPrint(poolElement,stdlog);
   fputs("to pool ",stdlog);
   poolHandlePrint(poolHandle,stdlog);
   fputs("\n",stdlog);
   LOG_END

   message = asapMessageNew(NULL,ASAP_BUFFER_SIZE);
   if(message != NULL) {
      message->Type           = AHT_REGISTRATION;
      message->PoolHandlePtr  = poolHandle;
      message->PoolElementPtr = poolElement;

      result = asapIO(asap,message,NULL,&error);
      if(result == ASAP_Okay) {
         if(error == ASAP_Okay) {
            newPoolElement = asapCacheUpdatePoolElement(asap->Cache,poolHandle,message->PoolElementPtr,false);
            if(newPoolElement != NULL) {
               /* asapCacheUpdatePoolElement() may not directly increment the user counter,
                  since this would also increment it in case of re-registration only. */
               if(newPoolElement->UserCounter == 0) {
                  newPoolElement->UserCounter++;
               }
               newPoolElement->UserData = (void*)asap;
            }
         }
         else {
            result = (enum ASAPError)error;
         }
      }

      asapMessageDelete(message);
   }
   else {
      result = ASAP_OutOfMemory;
   }

   LOG_ACTION
   fputs("Registration result is ",stdlog);
   asapErrorPrint(result,stdlog);
   fputs("\n",stdlog);
   LOG_END

   dispatcherUnlock(asap->StateMachine);
   return(result);
}


/* ###### Deregister pool element ######################################## */
enum ASAPError asapDeregister(struct ASAPInstance*        asap,
                              struct PoolHandle*          poolHandle,
                              const PoolElementIdentifier identifier)
{
   struct PoolElement* poolElement;
   struct ASAPMessage* message;
   enum ASAPError      result;
   uint16_t            error;

   dispatcherLock(asap->StateMachine);

   LOG_ACTION
   fprintf(stdlog,"Trying to deregister $%08x from pool ",identifier);
   poolHandlePrint(poolHandle,stdlog);
   fputs("\n",stdlog);
   LOG_END

   message = asapMessageNew(NULL,ASAP_BUFFER_SIZE);
   if(message != NULL) {
      message->Type          = AHT_DEREGISTRATION;
      message->PoolHandlePtr = poolHandle;
      message->Identifier    = identifier;

      result = asapIO(asap,message,NULL,&error);
      if((result == ASAP_Okay) && (error == ASAP_Okay)) {
         poolElement = asapCacheFindPoolElement(asap->Cache, poolHandle, identifier);
         if(poolElement) {
            asapCachePurgePoolElement(asap->Cache,poolHandle,poolElement,true);
         }
      }

      asapMessageDelete(message);
   }
   else {
      result = ASAP_OutOfMemory;
   }

   LOG_ACTION
   fputs("Deregistration result is ",stdlog);
   asapErrorPrint(result,stdlog);
   fputs("\n",stdlog);
   LOG_END

   dispatcherUnlock(asap->StateMachine);
   return(result);
}


/* ###### Do name lookup ################################################# */
static enum ASAPError asapDoNameLookup(struct ASAPInstance* asap,
                                       struct PoolHandle*   poolHandle)
{
   struct ASAPMessage* message;
   struct ASAPMessage* response;
   enum ASAPError      result;
   uint16_t            error;

   message = asapMessageNew(NULL,ASAP_BUFFER_SIZE);
   if(message != NULL) {
      message->Type          = AHT_NAME_RESOLUTION;
      message->PoolHandlePtr = poolHandle;

      result = asapIO(asap,message,&response,&error);

      if(result == ASAP_Okay) {
         result = (enum ASAPError)response->Error;
         if(result == ASAP_Okay) {
            if(response->PoolPtr != NULL) {
               asapCacheUpdatePool(asap->Cache,response->PoolPtr);
            }
            if(response->PoolElementPtr != NULL) {
               asapCacheUpdatePoolElement(asap->Cache,poolHandle,response->PoolElementPtr,false);
            }
         }
         asapMessageDelete(response);
      }

      asapMessageDelete(message);
   }
   else {
      result = ASAP_OutOfMemory;
   }

   return(result);
}


/* ###### Report pool element failure ####################################### */
enum ASAPError asapFailure(struct ASAPInstance*        asap,
                           struct PoolHandle*          poolHandle,
                           const PoolElementIdentifier identifier)
{
   struct PoolElement* found;

   dispatcherLock(asap->StateMachine);
   found = asapCacheFindPoolElement(asap->Cache, poolHandle, identifier);
   if(found != NULL) {
      found->Flags |= PEF_FAILED;
      asapCachePurgePoolElement(asap->Cache, poolHandle, found, false);
   }
   dispatcherUnlock(asap->StateMachine);
   return(ASAP_Okay);
}


/* ###### Do name lookup from cache ###################################### */
static enum ASAPError asapNameResolutionFromCache(struct ASAPInstance* asap,
                                                  struct PoolHandle*   poolHandle,
                                                  struct Pool**        poolPtr)
{
   struct Pool*   pool;
   enum ASAPError result = ASAP_NotFound;

   dispatcherLock(asap->StateMachine);
   pool = asapCacheFindPool(asap->Cache,poolHandle);

   LOG_ACTION
   fprintf(stdlog,"Name Resolution for handle ");
   poolHandlePrint(poolHandle,stdlog);
   fputs(":\n",stdlog);
   poolPrint(pool,stdlog);
   LOG_END

   if(pool != NULL) {
      if(poolPtr != NULL) {
         *poolPtr = poolDuplicate(pool);
      }
      result = ASAP_Okay;
   }
   else {
      if(poolPtr != NULL) {
         *poolPtr = NULL;
      }
   }

   dispatcherUnlock(asap->StateMachine);
   return(result);
}


/* ###### Do name lookup from name server ################################ */
static enum ASAPError asapNameResolutionFromServer(struct ASAPInstance* asap,
                                                   struct PoolHandle*   poolHandle,
                                                   struct Pool**        poolPtr)
{
   enum ASAPError result;

   dispatcherLock(asap->StateMachine);

   result = asapDoNameLookup(asap,poolHandle);
   if(result == ASAP_Okay) {
      result = asapNameResolutionFromCache(asap,poolHandle,poolPtr);
   }
   else {
      if(poolPtr != NULL) {
         *poolPtr = NULL;
      }
   }

   dispatcherUnlock(asap->StateMachine);

   return(result);
}


/* ###### Do name resolution for given pool handle ####################### */
enum ASAPError asapNameResolution(struct ASAPInstance* asap,
                                  struct PoolHandle*   poolHandle,
                                  struct Pool**        poolPtr)
{
   enum ASAPError result;

   LOG_VERBOSE
   fputs("Trying name resolution from cache...\n",stdlog);
   LOG_END

   result = asapNameResolutionFromCache(asap,poolHandle,poolPtr);
   if(result != ASAP_Okay) {
      if(poolPtr != NULL) {
         poolDelete(*poolPtr);
      }

      LOG_VERBOSE
      fputs("Trying name resolution from server...\n",stdlog);
      LOG_END
      result = asapNameResolutionFromServer(asap,poolHandle,poolPtr);

      if(result != ASAP_Okay) {
         LOG_VERBOSE
         fputs("Failed -> Getting old results from cache...\n",stdlog);
         LOG_END
         result = asapNameResolutionFromCache(asap,poolHandle,poolPtr);
      }
   }

   return(result);
}


/* ###### Select pool element by policy ################################## */
struct PoolElement* asapSelectPoolElement(struct ASAPInstance* asap,
                                          struct PoolHandle*   poolHandle,
                                          enum ASAPError*      error)
{
   struct Pool*        pool;
   struct PoolElement* poolElement;
   struct PoolElement* selected;
   enum ASAPError      result;
   card64              now;

   poolElement = NULL;
   dispatcherLock(asap->StateMachine);

   pool = asapCacheFindPool(asap->Cache,poolHandle);
   now  = getMicroTime();
   if(pool == NULL) {
      result = asapDoNameLookup(asap,poolHandle);
   }
   else {
      result = ASAP_Okay;
   }

   pool = asapCacheFindPool(asap->Cache,poolHandle);
   if(pool != NULL) {
      selected = poolSelectByPolicy(pool);
      if(selected != NULL) {
         poolElement = poolElementDuplicate(selected);
      }
      else if(result == ASAP_Okay) {
         result = ASAP_NotFound;
      }
   }
   else if(result == ASAP_Okay) {
      result = ASAP_NotFound;
   }

   dispatcherUnlock(asap->StateMachine);

   if(error) {
      *error = result;
   }
   return(poolElement);
}


/* ###### Handle endpoint keepalive ###################################### */
static void asapHandleEndpointKeepAlive(struct ASAPInstance* asap,
                                        struct ASAPMessage*  message)
{
   struct Pool*        pool;
   struct PoolElement* poolElement;
   GList*              list;

   LOG_VERBOSE
   fprintf(stdlog,"Endpoint KeepAlive for pool handle ");
   poolHandlePrint(message->PoolHandlePtr,stdlog);
   fputs("\n",stdlog);
   LOG_END

   pool = asapCacheFindPool(asap->Cache,message->PoolHandlePtr);
   if(pool != NULL) {
      list = g_list_first(pool->PoolElementList);
      while(list != NULL) {
         poolElement = (struct PoolElement*)list->data;
         if(poolElement->UserData == (void*)asap) {
            message->Type       = AHT_ENDPOINT_KEEP_ALIVE_ACK;
            message->Identifier = poolElement->Identifier;

            LOG_ACTION
            fprintf(stdlog,"Sending KeepAliveAck for pool element $%08x\n",message->Identifier);
            LOG_END
            asapSendRequest(asap,message);
         }
         list = g_list_next(list);
      }
   }
   else {
      LOG_WARNING
      fputs("Pool not found\n",stdlog);
      LOG_END
   }
}


/* ###### Handle incoming requests from name server (keepalives) ######### */
static void asapNameServerConnectionHandler(struct Dispatcher* dispatcher,
                                      int                fd,
                                      int                eventMask,
                                      void*              userData)
{
   struct ASAPInstance* asap = (struct ASAPInstance*)userData;
   struct ASAPMessage*  message;
   enum ASAPError       result;

   dispatcherLock(asap->StateMachine);

   LOG_ACTION
   fputs("Entering Connection Handler...\n",stdlog);
   LOG_END

   if(eventMask & (FDCE_Read|FDCE_Exception)) {
      result = asapReceiveResponse(asap,&message);
      if(result == ASAP_Okay) {
         if(message->Type == AHT_ENDPOINT_KEEP_ALIVE) {
            asapHandleEndpointKeepAlive(asap,message);
         }
         else {
            LOG_WARNING
            fprintf(stdlog,"Received unexpected message type #%d\n",
                    message->Type);
            LOG_END
         }
         asapMessageDelete(message);
      }
      else {
         LOG_ACTION
         fputs("Disconnecting from name server server due to failure\n",stdlog);
         LOG_END
         asapDisconnectFromNameServer(asap);
      }
   }

   LOG_ACTION
   fputs("Leaving Connection Handler\n",stdlog);
   LOG_END

   dispatcherUnlock(asap->StateMachine);
}
