#!/usr/bin/perl -w

# RRDWeather
# copyright (c) 2005 Sebastien Wains <seb@wains.be>
# released under the GNU General Public License
# http://www.wains.be/projects/rrdweather/

use RRDs;
use POSIX qw(uname);
use POSIX qw(strftime);

my $VERSION = "0.36";

############################################################
############## EDIT THESE VALUES BELOW #####################
############################################################

my $city = 'Mons, Belgium';
my $xpoints = 650;		# width of the graphs
my $ypoints = 250;		# height of the graphs
my $db_dir = '/usr/local/rrdweather/db';
my $tmp_dir = '/tmp/rrdweather';

############################################################
### YOU SHOULD NOT HAVE TO EDIT ANYTHING BELOW THIS LINE ###
############################################################

my $host = (POSIX::uname())[1];
my $scriptname = 'weather.cgi';
my $date = strftime "%a %b %e %Y %I.%M%p", localtime;
my $points_per_sample = 3;
my $ypoints_err = 96;
my $db_real = "$db_dir/real.rrd";
my $db_dew = "$db_dir/dew.rrd";
my $db_felt = "$db_dir/felt.rrd";
my $db_humidity = "$db_dir/humidity.rrd";
my $db_wind = "$db_dir/wind.rrd";
my $db_pressure = "$db_dir/pressure.rrd";
my $db_uv = "$db_dir/uv.rrd";

my @graphs = (
	{ title => 'Daily Graphs',   seconds => 3600*24,        },
	{ title => 'Weekly Graphs',  seconds => 3600*24*7,      },
	{ title => 'Monthly Graphs', seconds => 3600*24*31,     },
	{ title => 'Yearly Graphs',  seconds => 3600*24*365, },
);

sub graph_temperature($$$)
{
	my $range = shift;
	my $file = shift;
	my $title = shift;
	my $step = $range*$points_per_sample/$xpoints;

	my ($graphret,$xs,$ys) = RRDs::graph($file,
		'--imgformat', 'PNG',
		'--width', $xpoints,
		'--height', $ypoints,
		'--start', "-$range",
		"--title= Real & felt temperatures in $city",
		'--vertical-label', 'Real & felt temp.',
		'--units-exponent', 0,
		'--lazy',
		'-Y',
		
		"DEF:real_c=$db_real:real:AVERAGE",
		"DEF:felt_c=$db_felt:felt:AVERAGE",
		"DEF:dew_c=$db_dew:dew:AVERAGE",
		"COMMENT:           Min       Max       Ave       Last\\n",
		'LINE2:real_c#DD3F4A:Real', 'GPRINT:real_c:MIN: %5.2lf C',
		'GPRINT:real_c:MAX: %5.2lf C', 'GPRINT:real_c:AVERAGE: %5.2lf C',
		'GPRINT:real_c:LAST: %5.2lf C\\n',
		'LINE1:felt_c#3F4ADD:Felt', , 'GPRINT:felt_c:MIN: %5.2lf C',
                'GPRINT:felt_c:MAX: %5.2lf C', 'GPRINT:felt_c:AVERAGE: %5.2lf C',
                'GPRINT:felt_c:LAST: %5.2lf C\\n',	
		'LINE1:dew_c#4ADD3F:Dew ', , 'GPRINT:dew_c:MIN: %5.2lf C',
                'GPRINT:dew_c:MAX: %5.2lf C', 'GPRINT:dew_c:AVERAGE: %5.2lf C',
                'GPRINT:dew_c:LAST: %5.2lf C\\n',
		"COMMENT:                                                                    Generated by RRDWeather on $date",

	);
	my $ERR=RRDs::error;
	die "ERROR: $ERR\n" if $ERR;
}

sub graph_humidity($$$)
{
        my $range = shift;
        my $file = shift;
        my $title = shift;
        my $step = $range*$points_per_sample/$xpoints;

        my ($graphret,$xs,$ys) = RRDs::graph($file,
                '--imgformat', 'PNG',
                '--width', $xpoints,
                '--height', $ypoints,
                '--start', "-$range",
		"--title= Humidity in $city",
                '--vertical-label', 'Humidity',
                '--lower-limit', 0,
		'--upper-limit', 100,
                '--units-exponent', 0,
                '--lazy',
		'-Y',

                "DEF:humidity_pc=$db_humidity:humidity:AVERAGE",
                'COMMENT:        ', "COMMENT:     Min        Max        Ave        Last\\n",
                'LINE1:humidity_pc#4DD34A:Humidity', 'GPRINT:humidity_pc:MIN: %5.2lf pc',
                'GPRINT:humidity_pc:MAX: %5.2lf pc', 'GPRINT:humidity_pc:AVERAGE: %5.2lf pc',
                'GPRINT:humidity_pc:LAST: %5.2lf pc\\n',
                "COMMENT:                                                                    Generated by RRDWeather on $date",

	);
        my $ERR=RRDs::error;
        die "ERROR: $ERR\n" if $ERR;
}

sub graph_wind($$$)
{
        my $range = shift;
        my $file = shift;
        my $title = shift;
        my $step = $range*$points_per_sample/$xpoints;

        my ($graphret,$xs,$ys) = RRDs::graph($file,
                '--imgformat', 'PNG',
                '--width', $xpoints,
                '--height', $ypoints,
                '--start', "-$range",
                "--title= Wind in $city",
                '--vertical-label', 'Wind',
                '--units-exponent', 0,
                '--lazy',
                '-Y',

                "DEF:wind_s=$db_wind:wind:AVERAGE",
                "COMMENT:            Min        Max         Ave         Last\\n",
                'LINE2:wind_s#DD3F4A:Wind', 'GPRINT:wind_s:MIN: %5.2lf kph',
                'GPRINT:wind_s:MAX: %5.2lf kph', 'GPRINT:wind_s:AVERAGE: %5.2lf kph',
                'GPRINT:wind_s:LAST: %5.2lf kph\\n',
                "COMMENT:                                                                    Generated by RRDWeather on $date",

        );
        my $ERR=RRDs::error;
        die "ERROR: $ERR\n" if $ERR;
}


sub graph_pressure($$$)
{
        my $range = shift;
        my $file = shift;
        my $title = shift;
        my $step = $range*$points_per_sample/$xpoints;

        my ($graphret,$xs,$ys) = RRDs::graph($file,
                '--imgformat', 'PNG',
                '--width', $xpoints,
                '--height', $ypoints,
                '--start', "-$range",
                "--title= Pressure in $city",
                '--vertical-label', 'Pressure',
                '--lower-limit', 1000,
                '--upper-limit', 1040,
		'--units-exponent', 0,
                '--lazy',
		'-Y',

        	"DEF:pressure_hpa=$db_pressure:pressure:AVERAGE",
                'COMMENT:        ', "COMMENT:        Min           Max           Ave          Last\\n",
                'LINE1:pressure_hpa#DD3F4A:Pressure', 'GPRINT:pressure_hpa:MIN: %5.2lf hPa',
                'GPRINT:pressure_hpa:MAX: %5.2lf hPa', 'GPRINT:pressure_hpa:AVERAGE: %5.2lf hPa',
                'GPRINT:pressure_hpa:LAST: %5.2lf hPa\\n',
                "COMMENT:                                                                    Generated by RRDWeather on $date",

	);
        my $ERR=RRDs::error;
        die "ERROR: $ERR\n" if $ERR;
}

sub graph_uv($$$)
{
        my $range = shift;
        my $file = shift;
        my $title = shift;
        my $step = $range*$points_per_sample/$xpoints;

        my ($graphret,$xs,$ys) = RRDs::graph($file,
                '--imgformat', 'PNG',
                '--width', $xpoints,
                '--height', $ypoints,
                '--start', "-$range",
                "--title= UV index in $city",
                '--vertical-label', 'UV index',
                '--lower-limit', 0,
	        '--upper-limit', 10,        
		'--units-exponent', 0,
                '--lazy',
		'-y', '1:5',		

                "DEF:uv_index=$db_uv:uv:AVERAGE",
                'COMMENT:        ', "COMMENT:     Min     Max     Ave    Last\\n",
                'AREA:uv_index#DD3F4A:UV index', 'GPRINT:uv_index:MIN: %5.2lf',
                'GPRINT:uv_index:MAX: %5.2lf', 'GPRINT:uv_index:AVERAGE: %5.2lf',
		'GPRINT:uv_index:LAST: %5.2lf\\n',
                "COMMENT:                                                                    Generated by RRDWeather on $date",

	);
        my $ERR=RRDs::error;
        die "ERROR: $ERR\n" if $ERR;
}

sub print_html()
{
	print "Content-Type: text/html\n\n";

	print <<HEADER;
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.0 Transitional//EN" "http://www.w3.org/TR/REC-html40/loose.dtd">
<HTML>
<HEAD>
<TITLE>Weather in $city - Generated by RRDWeather</TITLE>
<META HTTP-EQUIV="Pragma" CONTENT="no-cache">

<style type="text/css">

body
{
	background-color: #ABABAB; 
	color: #737367;	
	font-size: 11px;
	margin: 0px;
	padding: 0px;
	text-align: left;
	font-family: arial, helvetica, sans-serif;
	background-color: #CC99;
}

a 
{
	text-decoration:none; color: #464646; 
}

a:visited 
{ 
	color: #59594F; 
}

a:hover 
{ 
	color: #FF6633; 
}

a:active 
{ 
	color: #444444; 
}

</style>

</HEAD>
<BODY>
<font>
HEADER

	print "<center><H1>Weather in $city</H1></center>\n";
	print "<center><H4>According to weather.com</H4></center>\n";
	for my $n (0..$#graphs) {
		print '<center><div align="center" style="background: #dddddd; width: 780px">';
		print "<H2>$graphs[$n]{title}</H2>\n";
		print "</div>\n";
		print "<P><IMG BORDER=\"0\" SRC=\"$scriptname?${n}-temperature\" ALT=\"RRDWeather\">\n";
		print "<P><IMG BORDER=\"0\" SRC=\"$scriptname?${n}-humidity\" ALT=\"RRDWeather\">\n";
		print "<P><IMG BORDER=\"0\" SRC=\"$scriptname?${n}-wind\" ALT=\"RRDWeather\">\n";
		print "<P><IMG BORDER=\"0\" SRC=\"$scriptname?${n}-pressure\" ALT=\"RRDWeather\">\n";
		print "<P><IMG BORDER=\"0\" SRC=\"$scriptname?${n}-uv\" ALT=\"RRDWeather\"></center>\n";
	}

	print <<FOOTER;
<hr width="780" align="center" size="1" noshade>
<table border="0" width="100%" cellpadding="0" cellspacing="0"><tr><td align="center">
<center><font><A target="_blank" href="http://www.wains.be/projects/rrdweather/">RRDWeather</A> $VERSION
by <A target="_blank" href="http://www.wains.be/">Sebastien Wains</A></font></center></td>
</tr></table>
</BODY></HTML></font>
FOOTER
}

sub send_image($)
{
	my $file = shift;
	-r $file or do {
		print "Content-type: text/plain\n\nERROR: can't find $file\n";
		exit 1;
	};

	print "Content-type: image/png\n";
	print "Content-length: ".((stat($file))[7])."\n";
	print "\n";
	open(IMG, $file) or die;
	my $data;
	print $data while read IMG, $data, 1024;
}

sub main()
{
	my $uri = "images";
	mkdir $tmp_dir, 0777 unless -d $tmp_dir;
	mkdir "$tmp_dir/$uri", 0777 unless -d "$tmp_dir/$uri";

	my $img = $ENV{QUERY_STRING};
	if(defined $img and $img =~ /\S/) {
		if($img =~ /^(\d+)-temperature$/) {
			my $file = "$tmp_dir/$uri/RRDWeather_$1_temperature.png";
			graph_temperature($graphs[$1]{seconds}, $file, $graphs[$1]{title});
			send_image($file);
		}
		elsif($img =~ /^(\d+)-humidity$/) {
			my $file = "$tmp_dir/$uri/RRDWeather_$1_humidity.png";
			graph_humidity($graphs[$1]{seconds}, $file, $graphs[$1]{title});
			send_image($file);
		}
		elsif($img =~ /^(\d+)-wind$/) {
                        my $file = "$tmp_dir/$uri/RRDWeather_$1_wind.png";
                        graph_wind($graphs[$1]{seconds}, $file, $graphs[$1]{title});
                        send_image($file);
                }
		elsif($img =~ /^(\d+)-pressure$/) {
                        my $file = "$tmp_dir/$uri/RRDWeather_$1_pressure.png";
                        graph_pressure($graphs[$1]{seconds}, $file, $graphs[$1]{title});
                        send_image($file);
                }
		elsif($img =~ /^(\d+)-uv$/) {
                        my $file = "$tmp_dir/$uri/RRDWeather_$1_uv.png";
                        graph_uv($graphs[$1]{seconds}, $file, $graphs[$1]{title});
                        send_image($file);
                }
                else {			
			die "ERROR: invalid argument\n";
			}
		}
		else {
			print_html;
		}
	}

main;
