/* rrdcollect.c - main(), command line, forking, main loop. */
/*
 *  RRDcollect  --  Round-Robin Database Collecting Daemon.
 *  Copyright (C) 2002, 2003  Dawid Kuroczko <qnex@knm.org.pl>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "rrdcollect.h"

#include <getopt.h>

#include <signal.h>
#include <sys/time.h>
#include <sys/types.h>

/* command line options */
const char *config_file = RRDCOLLECT_CONF;
enum working_mode {
#ifdef HAVE_LIBRRD
	LIBRRD_MODE,
#endif /* HAVE_LIBRRD */
	PIPE_MODE,
	FILE_MODE
} mode;
const char *output_pipe = RRDTOOL;
const char *output_file = NULL;
FILE *rrdtool = NULL;

int verbose = 0;
int no_fork = 0;

/* config options */
const char *workdir = NULL;
int step = 60;

/* signals */
volatile int action_request = 0;

void do_nothing(int signum) { }
void do_action(int signum) { action_request = signum; }
void do_abort(int signum) {
	fprintf(stderr, "\nSegmentation fault (core dumped), but The Memory Remains.\n\n");
	fflush(stderr);
	abort();
}

/* data delivery */
/* FIXME: this verbosity is ugly! */
int print_update(int argc, char **argv)
{
	int i;
	for (i = 0; i < argc && argv[i]; i++) {
		fprintf(rrdtool, (i) ? " %s" : "%s", argv[i]);
		if (verbose)
			fprintf(stderr, (i) ? " %s" : "%s", argv[i]);
	}
	putc('\n', rrdtool);
	if (verbose)
		putc('\n', stderr);

	return 0;
}

#ifdef HAVE_LIBRRD
int rrdlib_update(int argc, char **argv)
{
	optind = opterr = 0;
	
	rrd_update(argc, argv);

	if (rrd_test_error()) {
		fprintf(stderr, "rrd_update: %s\n", rrd_get_error());
		rrd_clear_error();
		
		return 1;
	}

	return 0;
}
#endif /* HAVE_LIBRRD */

int (*make_update)(int argc, char **argv);

int set_config_variable(const char *name, const char *value)
{
	switch (*name) {
		case 'd':
			if (!strcmp(name, "directory")) {
				workdir = strdup(value);
				return 0;
			}
			break;
		case 's':
			if (!strcmp(name, "step")) {
				step = atoi(value);
				if (!step) {
					fprintf(stderr, "Invalid step value: %s (%d)\n", value, step);
					exit(3);
				}
				return 0;
			}
			break;
		default:
			return 1;	
	}
	return 2;
}

/* some sugar */
void usage(int mode)
{
	/* usage */
	if (mode == -2) {
		printf("Usage: rrdcollect [-v] [--config=PATH] [--help] [--usage]\n");
		return exit(0);
	}
	/* version */
	if (mode == -1) {
		printf(PACKAGE_STRING
#if HAVE_LIBRRD || HAVE_LIBPCRE || !HAVE_GETOPT_LONG
				" (compiled with "
# ifdef HAVE_LIBRRD
				"librrd"
#  if HAVE_LIBPCRE || !HAVE_GETOPT_LONG
				" and "
#  endif /* HAVE_LIBPCRE || !HAVE_GETOPT_LONG */
# endif /* HAVE_LIBRRD */
# ifdef HAVE_LIBPCRE
				"libpcre"
#  ifndef HAVE_GETOPT_LONG
				" and "
#  endif /* !HAVE_GETOPT_LONG */
# endif /* HAVE_LIBPCRE */
# ifndef HAVE_GETOPT_LONG
				"built-in GNU getopt_long(3)"
# endif /* !HAVE_GETOPT_LONG */
				")"
#endif /* HAVE_LIBRRD || HAVE_LIBPCRE || !HAVE_GETOPT_LONG */
			"\n");
		printf("Copyright (C) 2002  Dawid Kuroczko <qnex@knm.org.pl>\n\n");

		printf("This program is distributed in the hope that it will be useful,\n");
		printf("but WITHOUT ANY WARRANTY; without even the implied warranty of\n");
		printf("MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
		printf("GNU General Public License for more details.\n\n");
		return exit(0);
	}

	printf(PACKAGE_STRING "  --  Round-Robin Database Collecting Daemon.\n");
	printf("Usage: rrdcollect [OPTION]...\n");

	/* error */
	if (mode > 0) {
		printf("\nTry `rrdcollect --help' for more options.\n");
		return exit(mode);
	}
	printf("\nGeneral options:\n");
	printf("  -c,  --config=FILE       specify alternative config file.\n");
	printf("  -n,  --no-fork           do not even think about forking.\n");
	printf("  -v,  --verbose           increase verbosity, implies -n.\n");

	printf("\nRRD update options:\n");
	printf("  -o,  --output=FILE       writes update commands to FILE.\n");
	printf("  -p,  --pipe[=RRDTOOL]    opens pipe to RRDTOOL and sends it commands.\n");
	printf("\nIf no RRD update option is given, " PACKAGE_NAME
#ifdef HAVE_LIBRRD
			" uses internal librrd routines.\n");
#else
			" opens pipe to " RRDTOOL " program.\n");
#endif /* HAVE_LIBRRD */
	
	/* help */
	printf("\nHelp options:\n");
	printf("  -V,  --version           display the version of " PACKAGE_NAME " and exit.\n");
	printf("  -h,  --help              print this help.\n");
	printf("       --usage             terse usage information.\n");

	printf("\nMail bug reports and suggestions to <bugs-rrdcollect@ssw.krakow.pl>\n");
	return exit(0);
};

int main(int argc, char **argv)
{
	int c;

	struct sigaction action;
	struct itimerval period;

	/* command line options */

#ifdef HAVE_LIBRRD
	mode = LIBRRD_MODE;
#else
	mode = PIPE_MODE;
#endif /* HAVE_LIBRRD */
	
	while (1) {
		int option_index = 0;
	
		static struct option long_options[] = {
			{"verbose", 0, 0, 'v'},
			{"no-fork", 0, 0, 'n'},
			{"config", 1, 0, 'c'},
			
			{"output", 1, 0, 'o'},
			{"pipe", 2, 0, 'p'},

			{"version", 0, 0, 'V'},
			{"help", 0, 0, 'h'},
			{"usage", 0, 0, 'U'},
			{0, 0, 0, 0}
		};

		c = getopt_long(argc, argv, "vnc:o:pVh",
				long_options, &option_index);
		if (c == -1)
			break;

		switch (c) {
			case 'c':
				config_file = optarg;
				break;
			case 'v':
				verbose++;
			case 'n':
				no_fork = 1;
				break;

			case 'o':
				mode = FILE_MODE;
				output_file = optarg;
				break;
			case 'p':
				mode = PIPE_MODE;
				if (optarg) {
					output_pipe = optarg;
				}
				break;
				
			case 'h':
				usage(0);
				break;
			case 'V':
				usage(-1);
				break;
			case 'U':	/* --usage */
				usage(-2);
				break;
		}
	}

	if (optind < argc) {
		usage(1);
		return 1;
	}

	if (parse_conf(config_file)) {
		fprintf(stderr, "Cannot parse config file %s: %s\n", config_file, strerror(errno));
		return 2;
	}

	if (!workdir) {
		fprintf(stderr, "You have to give working directory in config file\n");
		fprintf(stderr, "If you're desperate, you can use something like this:\n");
		fprintf(stderr, "    # current directory:\n");
		fprintf(stderr, "    directory = .\n");
		return 2;
	}
	
	/* Open Sesame... */
#ifdef HAVE_LIBRRD
	if (mode == LIBRRD_MODE) {
		make_update = &rrdlib_update;
	} else
#endif /* HAVE_LIBRRD */
	{
		make_update = &print_update;
		if (mode == FILE_MODE) {
			if (output_file[0] == '-' && output_file[1] == '\0') {
				no_fork = 1;
				rrdtool = stdout;
			} else { 
				rrdtool = fopen(output_file, "a");
			}
			if (!rrdtool) {
				fprintf(stderr, "Failed opening %s: %s\n", output_file, strerror(errno));
				return 3;
			}
		} else if (mode == PIPE_MODE) {
			chdir(workdir);
			rrdtool = popen(output_pipe, "w");
			if (!rrdtool) {
				fprintf(stderr, "Failed opening pipe %s: %s\n", output_pipe, strerror(errno));
				return 3;
			}
			make_update = &print_update;
		}
		
		/* implicit flushing on each line */
		setlinebuf(rrdtool);
	}

	if (chdir(workdir)) {
		fprintf(stderr, "Cannot chdir to %s: %s\n", workdir, strerror(errno));
		return 2;
	}

#ifdef HAVE_FORK
	if (!no_fork && fork())
		return 0;

# if 0
	/* FIXME: Seems not to work on BSD/Solaris, got to investigate... */
	if (!no_fork) {
		stdin = freopen("/dev/null", "r", stdin);
		stdout = freopen("/dev/null", "w", stdout);
		stderr = freopen("err.log", "w", stderr);
	}
# else
	/* I'm asking for trouble... */
	if (!no_fork) {
		fclose(stdin);
		fclose(stdout);
		fclose(stderr);
	}
# endif
#endif /* HAVE_FORK */

	/* being raised every "step" */
	action.sa_handler = &do_nothing;
	action.sa_flags = 0;
	sigaction(SIGALRM, &action, NULL);

	/* user requests, et. al. */
	action.sa_handler = &do_action;
	action.sa_flags = 0;
	sigaction(SIGINT, &action, NULL);
	sigaction(SIGQUIT, &action, NULL);
	sigaction(SIGTERM, &action, NULL);
	sigaction(SIGHUP, &action, NULL);
	sigaction(SIGUSR1, &action, NULL);
	sigaction(SIGUSR2, &action, NULL);

	/* deep trouble */
	action.sa_handler = &do_abort;
	action.sa_flags = 0;
	sigaction(SIGSEGV, &action, NULL);
	sigaction(SIGFPE, &action, NULL);
	sigaction(SIGBUS, &action, NULL);
	
	period.it_interval.tv_sec  = period.it_value.tv_sec  = step;
	period.it_interval.tv_usec = period.it_value.tv_usec = 0;
	setitimer(ITIMER_REAL, &period, NULL);

	for (;;) {
		if (action_request) {
			switch (action_request) {
				case SIGINT:
				case SIGQUIT:
				case SIGTERM:
					/* FIXME: better leave for(;;) loop... */
					exit(0);
					break;
					
				case SIGHUP:
				case SIGUSR1:
				case SIGUSR2:
				default:
					fflush(stdout);
					break;
			}
			
			action_request = 0;

		} else {
			perform_tests();
			print_results();
			clear_counters();
		}

		pause();
	}
}
